/*
    graphs.c  - produces graphs used by the Webalizer

    Copyright (C) 1997-2001  Bradford L. Barrett (brad@mrunix.net)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

    This software uses the gd graphics library, which is copyright by
    Quest Protein Database Center, Cold Spring Harbor Labs.  Please
    see the documentation supplied with the library for additional
    information and license terms, or visit www.boutell.com/gd/ for the
    most recent version of the library and supporting documentation.
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <gd.h>
#include <gdfontt.h>
#include <gdfonts.h>
#include <gdfontmb.h>

#include "webalizer.h"
#include "lang.h"
#include "graphs.h"

/* Some systems don't define this */
#ifndef PI
#define PI 3.14159265358979323846
#endif

#define HITCOLOR       hit_or_green        /* graph color - hits  */
#define FILECOLOR      file_or_blue        /* files               */
#define SITECOLOR      site_or_orange      /* sites               */
#define KBYTECOLOR     kbyte_or_red        /* KBytes              */
#define PAGECOLOR      page_or_cyan        /* Files               */
#define VISITCOLOR     visit_or_yellow     /* Visits              */
#define CHLEGENDCOLOR  chartlegend         /* chart legend color  */
#define CHSHADCOLOR1   chartshadow1        /* chart shadow color1 */
#define CHSHADCOLOR2   chartshadow2        /* chart shadow color2 */

#define CX 156                             /* center x (for pie)  */
#define CY 150                             /* center y  (chart)   */
#define XRAD 240                           /* X-axis radius       */
#define YRAD 200                           /* Y-axis radius       */

/* forward reference internal routines */

void    init_graph(char *, int, int);
struct  pie_data *calc_arc(float, float);

/* common public declarations */

char *numchar[] = { " 0"," 1"," 2"," 3"," 4"," 5"," 6"," 7"," 8"," 9","10",
                    "11","12","13","14","15","16","17","18","19","20",
                    "21","22","23","24","25","26","27","28","29","30","31"};

gdImagePtr	im;                        /* image buffer        */
FILE		*out;                      /* output file for PNG */
char		maxvaltxt[32];             /* graph values        */
float		percent;                   /* percent storage     */
u_long		julday;                    /* julday value        */

struct pie_data { int x; int y;            /* line x,y            */
                  int mx; int my; };       /* midpoint x,y        */
/* colors */
int		black, white, grey, dkgrey, kbyte_or_red, file_or_blue, blue,
                site_or_orange, hit_or_green, page_or_cyan, visit_or_yellow,
		chartlegend, chartshadow1, chartshadow2, chartbackground;

/****************************************************************/
/*                                                              */
/* ASHEX2INT - ASCII HEXA TO INT CONVERTER [ not strtoul() ]    */
/*                                                              */
/****************************************************************/

int ashex2int(char* twocharstr){
/* returns the base-10 integer value from a 2 ascii hex number */
   int val;
   switch (twocharstr[1])
   {
      case 'a': 
      case 'A': val=10; break;
      case 'b':
      case 'B': val=11; break;
      case 'c':
      case 'C': val=12; break;
      case 'd':
      case 'D': val=13; break;
      case 'e':
      case 'E': val=14; break;
      case 'f':
      case 'F': val=15; break;
      default: val=(int)twocharstr[1]-48;
   }
   switch (twocharstr[0])
   {
      case 'a':
      case 'A': val+=160; break;
      case 'b':
      case 'B': val+=176; break;
      case 'c': 
      case 'C': val+=192; break;
      case 'd':
      case 'D': val+=208; break;
      case 'e':
      case 'E': val+=224; break;
      case 'f':
      case 'F': val+=240; break;
      default: val+=(int)(twocharstr[0]-48)*16;
   }
   return val;
}

/* shortcuts to convert ascii hex color for gdImageColorAllocate() */

#define getred(s) (ashex2int((s[0] == '#')?s+1:s))
/* returns the red base-10 integer value from a html color */

#define getgreen(s) (ashex2int((s[0] == '#')?s+3:s+2))
/* returns the green base-10 integer value from a html color */

#define getblue(s) (ashex2int((s[0] == '#')?s+5:s+4))
/* returns the blue base-10 integer value from a html color */

/*****************************************************************/
/*                                                               */
/* YEAR_GRAPH6x  - Year graph with six data sets                 */
/*                                                               */
/*****************************************************************/

int year_graph6x(  char *fname,            /* file name use      */
                   char *title,            /* title for graph    */
                    int fmonth,            /* begin month number */
                 u_long data1[12],         /* data1 (hits)       */
                 u_long data2[12],         /* data2 (files)      */
                 u_long data3[12],         /* data3 (sites)      */
                 double data4[12],         /* data4 (kbytes)     */
		 double data5[12],         /* data5 (kbytes)     */
		 double data6[12],         /* data6 (kbytes)     */
		 u_long data7[12],         /* data7 (views)      */
		 u_long data8[12])         /* data8 (visits)     */
{

   /* local variables */
   int i,j,k,x1,y1,x2;
   int s_mth;

   u_long maxval=1;
   double fmaxval=0.0;

   /* initalize the graph */
   init_graph(title,512,256);              /* init as 512 x 256  */

   gdImageLine(im, 305,25,305,233,black);  /* draw section lines */
   gdImageLine(im, 304,25,304,233,white);
   gdImageLine(im, 305,130,490,130,black);
   gdImageLine(im, 305,129,490,129,white);

   /* index lines? */
   if (graph_lines)
   {
      y1=210/(graph_lines+1);
      for (i=0;i<graph_lines;i++)
       gdImageLine(im,21,((i+1)*y1)+25,303,((i+1)*y1)+25,dkgrey);
      y1=105/(graph_lines+1);
      for (i=0;i<graph_lines;i++)
       gdImageLine(im,306,((i+1)*y1)+25,489,((i+1)*y1)+25,dkgrey);
      for (i=0;i<graph_lines;i++)
       gdImageLine(im,306,((i+1)*y1)+130,489,((i+1)*y1)+130,dkgrey);
   }

   /* x-axis legend */
   s_mth = fmonth;
   for (i=0;i<12;i++)
   {
      gdImageString(im,gdFontSmall,28+(i*23),             /* use language   */
                    238,s_month[s_mth-1],CHLEGENDCOLOR);  /* specific array */
      s_mth++;
      if (s_mth > 12) s_mth = 1;
      if (data1[i] > maxval) maxval = data1[i];           /* get max val    */
      if (data2[i] > maxval) maxval = data2[i];
      if (data7[i] > maxval) maxval = data7[i];
   }
   if (maxval <= 0) maxval = 1;
   sprintf(maxvaltxt, "%lu", maxval);
   gdImageStringUp(im,gdFontSmall,8,26+(strlen(maxvaltxt)*6),maxvaltxt,CHLEGENDCOLOR);

   if (graph_legend)                          /* print color coded legends? */
   {
      /* Kbytes Legend */
      i = (strlen(msg_h_xfer)*6);
      j = (strlen(msg_h_ixfer)*6);
      k = (strlen(msg_h_oxfer)*6);
      gdImageString(im,gdFontSmall,491-i-j-k-24,239,msg_h_xfer,dkgrey);
      gdImageString(im,gdFontSmall,490-i-j-k-24,238,msg_h_xfer,KBYTECOLOR);
      gdImageString(im,gdFontSmall,491-k-j-21,239,"/",dkgrey);
      gdImageString(im,gdFontSmall,490-k-j-21,238,"/",black);
      gdImageString(im,gdFontSmall,491-k-j-12,239,msg_h_ixfer,dkgrey);
      gdImageString(im,gdFontSmall,490-k-j-12,238,msg_h_ixfer,FILECOLOR);
      gdImageString(im,gdFontSmall,491-k-9,239,"/",dkgrey);
      gdImageString(im,gdFontSmall,490-k-9,238,"/",black);
      gdImageString(im,gdFontSmall,491-k,239,msg_h_oxfer,dkgrey);
      gdImageString(im,gdFontSmall,490-k,238,msg_h_oxfer,HITCOLOR);

      /* Sites/Visits Legend */
      i = (strlen(msg_h_visits)*6);
      j = (strlen(msg_h_sites)*6);
      gdImageString(im,gdFontSmall,491-i-j-12,11,msg_h_visits,dkgrey);
      gdImageString(im,gdFontSmall,490-i-j-12,10,msg_h_visits,VISITCOLOR);
      gdImageString(im,gdFontSmall,491-j-9,11,"/",dkgrey);
      gdImageString(im,gdFontSmall,490-j-9,10,"/",black);
      gdImageString(im,gdFontSmall,491-j,11,msg_h_sites,dkgrey);
      gdImageString(im,gdFontSmall,490-j,10,msg_h_sites,SITECOLOR);

      /* Hits/Files/Pages Legend */
      i = (strlen(msg_h_pages)*6);
      j = (strlen(msg_h_files)*6);
      gdImageStringUp(im,gdFontSmall,4,231,msg_h_pages,dkgrey);
      gdImageStringUp(im,gdFontSmall,3,230,msg_h_pages,PAGECOLOR);
      gdImageStringUp(im,gdFontSmall,4,231-i-3,"/",dkgrey);
      gdImageStringUp(im,gdFontSmall,3,230-i-3,"/",black);
      gdImageStringUp(im,gdFontSmall,4,231-i-12,msg_h_files,dkgrey);
      gdImageStringUp(im,gdFontSmall,3,230-i-12,msg_h_files,FILECOLOR);
      gdImageStringUp(im,gdFontSmall,4,231-i-j-15,"/",dkgrey);
      gdImageStringUp(im,gdFontSmall,3,230-i-j-15,"/",black);
      gdImageStringUp(im,gdFontSmall,4,231-i-j-24,msg_h_hits,dkgrey);
      gdImageStringUp(im,gdFontSmall,3,230-i-j-24,msg_h_hits,HITCOLOR);
   }

   /* data1 */
   s_mth = fmonth;
   for (i=0; i<12; i++)
   {
      if (s_mth > 12) s_mth = 1;
      percent = ((float)data1[s_mth++ -1] / (float)maxval);
      if (percent <= 0.0) continue;
      x1 = 26 + (i*23);
      x2 = x1 + 13;
      y1 = 232 - (percent * 203);
      gdImageFilledRectangle(im, x1, y1, x2, 232, HITCOLOR);
      gdImageRectangle(im, x1, y1, x2, 232, black);
   }

   /* data2 */
   s_mth = fmonth;
   for (i=0; i<12; i++)
   {
      if (s_mth > 12) s_mth = 1;
      percent = ((float)data2[s_mth++ -1] / (float)maxval);
      if (percent <= 0.0) continue;
      x1 = 29 + (i*23);
      x2 = x1 + 13;
      y1 = 232 - (percent * 203);
      gdImageFilledRectangle(im, x1, y1, x2, 232, FILECOLOR);
      gdImageRectangle(im, x1, y1, x2, 232, black);
   }

   /* data7 */
   s_mth = fmonth;
   for (i=0; i<12; i++)
   {
      if (s_mth > 12) s_mth = 1;
      percent = ((float)data7[s_mth++ -1] / (float)maxval);
      if (percent <= 0.0) continue;
      x1 = 32 + (i*23);
      x2 = x1 + 13;
      y1 = 232 - (percent * 203);
      gdImageFilledRectangle(im, x1, y1, x2, 232, PAGECOLOR);
      gdImageRectangle(im, x1, y1, x2, 232, black);
   }

   maxval=0;
   for (i=0; i<12; i++)
   {
       if (data3[i] > maxval) maxval = data3[i];           /* get max val    */
       if (data8[i] > maxval) maxval = data8[i];
   }
   if (maxval <= 0) maxval = 1;
   sprintf(maxvaltxt, "%lu", maxval);
   gdImageStringUp(im, gdFontSmall,493,26+(strlen(maxvaltxt)*6),
                   maxvaltxt,CHLEGENDCOLOR);

   /* data8 */
   s_mth = fmonth;
   for (i=0; i<12; i++)
   {
      if (s_mth > 12) s_mth = 1;
      percent = ((float)data8[s_mth++ -1] / (float)maxval);
      if (percent <= 0.0) continue;
      x1 = 310 + (i*15);
      x2 = x1 + 8;
      y1 = 127 - (percent * 98);
      gdImageFilledRectangle(im, x1, y1, x2, 127, VISITCOLOR);
      gdImageRectangle(im, x1, y1, x2, 127, black);
   }

   /* data3 */
   s_mth = fmonth;
   for (i=0; i<12; i++)
   {
      if (s_mth > 12) s_mth = 1;
      percent = ((float)data3[s_mth++ -1] / (float)maxval);
      if (percent <= 0.0) continue;
      x1 = 314 + (i*15);
      x2 = x1 + 7;
      y1 = 127 - (percent * 98);
      gdImageFilledRectangle(im, x1, y1, x2, 127, SITECOLOR);
      gdImageRectangle(im, x1, y1, x2, 127, black);
   }

   fmaxval=0.0;
   for (i=0; i<12; i++){
       if (data4[i] > fmaxval) fmaxval = data4[i];         /* get max val    */
       if (data5[i] > fmaxval) fmaxval = data5[i];         /* get max val    */
       if (data6[i] > fmaxval) fmaxval = data6[i];         /* get max val    */
   }
   if (fmaxval <= 0.0) fmaxval = 1.0;
   sprintf(maxvaltxt, "%.0f", fmaxval);
   gdImageStringUp(im, gdFontSmall,493,130+(strlen(maxvaltxt)*6),
                   maxvaltxt,CHLEGENDCOLOR);

   /* data4 */
   s_mth = fmonth;
   for (i=0; i<12; i++)
   {
      if (s_mth > 12) s_mth = 1;
      percent = ((float)data4[s_mth++ -1] / (float)fmaxval);
      if (percent <= 0.0) continue;
      x1 = 309 + (i*15);
      x2 = x1 + 7;
      y1 = 232 - (percent * 98);
      gdImageFilledRectangle(im, x1, y1, x2, 232, KBYTECOLOR);
      gdImageRectangle(im, x1, y1, x2, 232, black);
   }

   /* data5 */
   s_mth = fmonth;
   for (i=0; i<12; i++)
   {
      if (s_mth > 12) s_mth = 1;
      percent = ((float)data5[s_mth++ -1] / (float)fmaxval);
      if (percent <= 0.0) continue;
      x1 = 311 + (i*15);
      x2 = x1 + 7;
      y1 = 232 - (percent * 98);
      gdImageFilledRectangle(im, x1, y1, x2, 232, FILECOLOR);
      gdImageRectangle(im, x1, y1, x2, 232, black);
   }

   /* data6 */
   s_mth = fmonth;
   for (i=0; i<12; i++)
   {
      if (s_mth > 12) s_mth = 1;
      percent = ((float)data6[s_mth++ -1] / (float)fmaxval);
      if (percent <= 0.0) continue;
      x1 = 313 + (i*15);
      x2 = x1 + 7;
      y1 = 232 - (percent * 98);
      gdImageFilledRectangle(im, x1, y1, x2, 232, HITCOLOR);
      gdImageRectangle(im, x1, y1, x2, 232, black);
   }

   /* save png image */
   if ((out = fopen(fname, "wb")) != NULL)
   {
      gdImagePng(im, out);
      fclose(out);
   }
   /* deallocate memory */
   gdImageDestroy(im);

   return (0);
}

/*****************************************************************/
/*                                                               */
/* MONTH_GRAPH6  - Month graph with six data sets                */
/*                                                               */
/*****************************************************************/

#define YSIZE 400

int month_graph6(  char *fname,            /* filename           */
                   char *title,            /* graph title        */
                    int month,             /* graph month        */
                    int year,              /* graph year         */
                 u_long data1[31],         /* data1 (hits)       */
                 u_long data2[31],         /* data2 (files)      */
                 u_long data3[31],         /* data3 (sites)      */
                 double data4[31],         /* data4 (kbytes)     */
                 double data5[31],         /* data5 (kbytes)     */
                 double data6[31],         /* data6 (kbytes)     */
                 u_long data7[31],         /* data7 (views)      */
                 u_long data8[31])         /* data8 (visits)     */
{

   /* local variables */
   int i,j,k,x1,y1,x2;
   u_long maxval=0;
   double fmaxval=0.0;

   /* calc julian date for month */
   julday = (jdate(1, month,year) % 7);

   /* initalize the graph */
   init_graph(title,512,500);

   gdImageLine(im, 21, 224, 490, 224, black); /* draw section lines */
   gdImageLine(im, 21, 223, 490, 223, white);
   gdImageLine(im, 21, 352, 490, 352, black);
   gdImageLine(im, 21, 351, 490, 351, white);

   /* index lines? */
   if (graph_lines)
   {
      y1=204/(graph_lines+1);
      for (i=0;i<graph_lines;i++)
       gdImageLine(im,21,((i+1)*y1)+25,489,((i+1)*y1)+25,dkgrey);
      y1=128/(graph_lines+1);
      for (i=0;i<graph_lines;i++)
       gdImageLine(im,21,((i+1)*y1)+224,489,((i+1)*y1)+224,dkgrey);
      for (i=0;i<graph_lines;i++)
       gdImageLine(im,21,((i+1)*y1)+352,489,((i+1)*y1)+352,dkgrey);
   }

   /* x-axis legend */
   for (i=0;i<31;i++)
   {
      if ((julday % 7 == 6) || (julday % 7 == 0))
       gdImageString(im,gdFontSmall,25+(i*15),482,numchar[i+1],HITCOLOR);
      else
       gdImageString(im,gdFontSmall,25+(i*15),482,numchar[i+1],CHLEGENDCOLOR);
      julday++;
   }

   /* y-axis legend */
   for (i=0; i<31; i++)
   {
       if (data1[i] > maxval) maxval = data1[i];           /* get max val    */
       if (data2[i] > maxval) maxval = data2[i];
       if (data7[i] > maxval) maxval = data7[i];
   }
   if (maxval <= 0) maxval = 1;
   sprintf(maxvaltxt, "%lu", maxval);
   gdImageStringUp(im, gdFontSmall,8,26+(strlen(maxvaltxt)*6),
                   maxvaltxt,CHLEGENDCOLOR);

   if (graph_legend)                           /* Print color coded legends? */
   {
      /* Kbytes Legend */
      i=(strlen(msg_h_xfer)*6);
      j=(strlen(msg_h_ixfer)*6);
      k=(strlen(msg_h_oxfer)*6);
      gdImageStringUp(im,gdFontSmall,494,476,msg_h_xfer,dkgrey);
      gdImageStringUp(im,gdFontSmall,493,475,msg_h_xfer,KBYTECOLOR);
      gdImageStringUp(im,gdFontSmall,494,476-i-3,"/",dkgrey);
      gdImageStringUp(im,gdFontSmall,493,475-i-3,"/",black);
      gdImageStringUp(im,gdFontSmall,494,476-i-12,msg_h_ixfer,dkgrey);
      gdImageStringUp(im,gdFontSmall,493,475-i-12,msg_h_ixfer,FILECOLOR);
      gdImageStringUp(im,gdFontSmall,494,476-i-j-15,"/",dkgrey);
      gdImageStringUp(im,gdFontSmall,493,475-i-j-15,"/",black);
      gdImageStringUp(im,gdFontSmall,494,476-i-j-24,msg_h_oxfer,dkgrey);
      gdImageStringUp(im,gdFontSmall,493,475-i-j-24,msg_h_oxfer,HITCOLOR);

      /* Sites/Visits Legend */
      i = 476-i-j-24-k; // latest position
      j = (strlen(msg_h_sites)*6);
      k = (strlen(msg_h_visits)*6);
      gdImageStringUp(im,gdFontSmall,494,i-3,"/",dkgrey);
      gdImageStringUp(im,gdFontSmall,493,i-4,"/",black);
      gdImageStringUp(im,gdFontSmall,494,i-12,msg_h_sites,dkgrey);
      gdImageStringUp(im,gdFontSmall,493,i-13,msg_h_sites,SITECOLOR);
      gdImageStringUp(im,gdFontSmall,494,i-j-15,"/",dkgrey);
      gdImageStringUp(im,gdFontSmall,493,i-j-16,"/",black);
      gdImageStringUp(im,gdFontSmall,494,i-j-24,msg_h_visits,dkgrey);
      gdImageStringUp(im,gdFontSmall,493,i-j-25,msg_h_visits,VISITCOLOR);

      /* Pages/Files/Hits Legend */
      i = i-j-24-k; // latest position
      j = (strlen(msg_h_pages)*6);
      k = (strlen(msg_h_files)*6);
      gdImageStringUp(im,gdFontSmall,494,i-3,"/",dkgrey);
      gdImageStringUp(im,gdFontSmall,493,i-4,"/",black);
      gdImageStringUp(im,gdFontSmall,494,i-12,msg_h_pages,dkgrey);
      gdImageStringUp(im,gdFontSmall,493,i-13,msg_h_pages,PAGECOLOR);
      gdImageStringUp(im,gdFontSmall,494,i-j-15,"/",dkgrey);
      gdImageStringUp(im,gdFontSmall,493,i-j-16,"/",black);
      gdImageStringUp(im,gdFontSmall,494,i-j-24,msg_h_files,dkgrey);
      gdImageStringUp(im,gdFontSmall,493,i-j-25,msg_h_files,FILECOLOR);
      gdImageStringUp(im,gdFontSmall,494,i-j-k-27,"/",dkgrey);
      gdImageStringUp(im,gdFontSmall,493,i-j-k-28,"/",black);
      gdImageStringUp(im,gdFontSmall,494,i-j-k-36,msg_h_hits,dkgrey);
      gdImageStringUp(im,gdFontSmall,493,i-j-k-37,msg_h_hits,HITCOLOR);
   }

   /* data1 */
   for (i=0; i<31; i++)
   {
      percent = ((float)data1[i] / (float)maxval);
      if (percent <= 0.0) continue;
      x1 = 25 + (i*15);
      x2 = x1 + 7;
      y1 = 220 - (percent * 191);
      gdImageFilledRectangle(im, x1, y1, x2, 220, HITCOLOR);
      gdImageRectangle(im, x1, y1, x2, 220, black);
   }

   /* data2 */
   for (i=0; i<31; i++)
   {
      percent = ((float)data2[i] / (float)maxval);
      if (percent <= 0.0) continue;
      x1 = 27 + (i*15);
      x2 = x1 + 7;
      y1 = 220 - (percent * 191);
      gdImageFilledRectangle(im, x1, y1, x2, 220, FILECOLOR);
      gdImageRectangle(im, x1, y1, x2, 220, black);
   }

   /* data7 */
   for (i=0; i<31; i++)
   {
      if (data7[i]==0) continue;
      percent = ((float)data7[i] / (float)maxval);
      if (percent <= 0.0) continue;
      x1 = 29 + (i*15);
      x2 = x1 + 7;
      y1 = 220 - (percent * 191);
      gdImageFilledRectangle(im, x1, y1, x2, 220, PAGECOLOR);
      gdImageRectangle(im, x1, y1, x2, 220, black);
   }

   /* sites / visits */
   maxval=0;
   for (i=0; i<31; i++)
   {
      if (data3[i]>maxval) maxval = data3[i];
      if (data8[i]>maxval) maxval = data8[i];
   }
   if (maxval <= 0) maxval = 1;
   sprintf(maxvaltxt, "%lu", maxval);
   gdImageStringUp(im, gdFontSmall,8,224+(strlen(maxvaltxt)*6),
                   maxvaltxt,CHLEGENDCOLOR);
   
   /* data 6 */
   for (i=0; i<31; i++)
   {
      percent = ((float)data8[i] / (float)maxval);
      if (percent <= 0.0) continue;
      x1 = 25 + (i*15);
      x2 = x1 + 8;
      y1 = 348 - (percent * 120);
      gdImageFilledRectangle(im, x1, y1, x2, 348, VISITCOLOR);
      gdImageRectangle(im, x1, y1, x2, 348, black);
   }

   /* data 3 */
   for (i=0; i<31; i++)
   {
      percent = ((float)data3[i] / (float)maxval);
      if (percent <= 0.0) continue;
      x1 = 29 + (i*15);
      x2 = x1 + 7;
      y1 = 348 - (percent * 120);
      gdImageFilledRectangle(im, x1, y1, x2, 348, SITECOLOR);
      gdImageRectangle(im, x1, y1, x2, 348, black);
   }

   fmaxval=0.0;
   for (i=0; i<31; i++){
      if (data4[i]>fmaxval) fmaxval = data4[i];
      if (data5[i]>fmaxval) fmaxval = data5[i];
      if (data6[i]>fmaxval) fmaxval = data6[i];
   }
   if (fmaxval <= 0.0) fmaxval = 1.0;
   sprintf(maxvaltxt, "%.0f", fmaxval/1024);
   gdImageStringUp(im, gdFontSmall,8,352+(strlen(maxvaltxt)*6),
                   maxvaltxt,CHLEGENDCOLOR);

   /* data4 */
   for (i=0; i<31; i++)
   {
      percent = data4[i] / fmaxval;
      if (percent <= 0.0) continue;
      x1 = 25 + (i*15);
      x2 = x1 + 7;
      y1 = 475 - ( percent * 119 );
      gdImageFilledRectangle(im, x1, y1, x2, 475, KBYTECOLOR);
      gdImageRectangle(im, x1, y1, x2, 475, black);
   }

   /* data5 */
   for (i=0; i<31; i++)
   {
      percent = data5[i] / fmaxval;
      if (percent <= 0.0) continue;
      x1 = 27 + (i*15);
      x2 = x1 + 7;
      y1 = 475 - ( percent * 119 );
      gdImageFilledRectangle(im, x1, y1, x2, 475, FILECOLOR);
      gdImageRectangle(im, x1, y1, x2, 475, black);
   }

   /* data6 */
   for (i=0; i<31; i++)
   {
      percent = data6[i] / fmaxval;
      if (percent <= 0.0) continue;
      x1 = 29 + (i*15);
      x2 = x1 + 7;
      y1 = 475 - ( percent * 119 );
      gdImageFilledRectangle(im, x1, y1, x2, 475, HITCOLOR);
      gdImageRectangle(im, x1, y1, x2, 475, black);
   }

   /* open file for writing */
   if ((out = fopen(fname, "wb")) != NULL)
   {
      gdImagePng(im, out);
      fclose(out);
   }
   /* deallocate memory */
   gdImageDestroy(im);

   return (0);
}

/*****************************************************************/
/*                                                               */
/* DAY_GRAPH3  - Day graph with three data sets                  */
/*                                                               */
/*****************************************************************/

int day_graph3(  char *fname,
                 char *title,
               u_long data1[24],
               u_long data2[24],
               u_long data3[24])
{

   /* local variables */
   int i,j,s,x1,y1,x2;
   u_long maxval=0;

   /* initalize the graph */
   init_graph(title,512,256);

   /* index lines? */
   if (graph_lines)
   {
      y1=210/(graph_lines+1);
      for (i=0;i<graph_lines;i++)
       gdImageLine(im,21,((i+1)*y1)+25,489,((i+1)*y1)+25,dkgrey);
   }

   /* x-axis legend */
   for (i=0;i<24;i++)
   {
      gdImageString(im,gdFontSmall,33+(i*19),238,numchar[i],CHLEGENDCOLOR);
      if (data1[i] > maxval) maxval = data1[i];           /* get max val    */
      if (data2[i] > maxval) maxval = data2[i];
      if (data3[i] > maxval) maxval = data3[i];
   }
   if (maxval <= 0) maxval = 1;
   sprintf(maxvaltxt, "%lu", maxval);
   gdImageStringUp(im, gdFontSmall, 8, 26+(strlen(maxvaltxt)*6),
                   maxvaltxt,CHLEGENDCOLOR);
   
   if (graph_legend)                          /* print color coded legends? */
   {
      /* Pages/Files/Hits Legend */
      s = ( i=(strlen(msg_h_pages)*6) )+
          ( j=(strlen(msg_h_files)*6) )+
          ( strlen(msg_h_hits)*6 )+ 52;
      gdImageStringUp(im,gdFontSmall,494,s,msg_h_pages,dkgrey);
      gdImageStringUp(im,gdFontSmall,493,s-1,msg_h_pages,PAGECOLOR);
      gdImageStringUp(im,gdFontSmall,494,s-i-3,"/",dkgrey);
      gdImageStringUp(im,gdFontSmall,493,s-i-4,"/",black);
      gdImageStringUp(im,gdFontSmall,494,s-i-12,msg_h_files,dkgrey);
      gdImageStringUp(im,gdFontSmall,493,s-i-13,msg_h_files,FILECOLOR);
      gdImageStringUp(im,gdFontSmall,494,s-i-j-15,"/",dkgrey);
      gdImageStringUp(im,gdFontSmall,493,s-i-j-16,"/",black);
      gdImageStringUp(im,gdFontSmall,494,s-i-j-24,msg_h_hits,dkgrey);
      gdImageStringUp(im,gdFontSmall,493,s-i-j-25,msg_h_hits,HITCOLOR);
   }

   /* data1 */
   for (i=0; i<24; i++)
   {
      percent = ((float)data1[i] / (float)maxval);  /* percent of 100% */
      if (percent <= 0.0) continue;
      x1 = 29 + (i*19);
      x2 = x1 + 10;
      y1 = 232 - (percent * 203);
      gdImageFilledRectangle(im, x1, y1, x2, 232, HITCOLOR);
      gdImageRectangle(im, x1, y1, x2, 232, black);
   }

   /* data2 */
   for (i=0; i<24; i++)
   {
      percent = ((float)data2[i] / (float)maxval);  /* percent of 100% */
      if (percent <= 0.0) continue;
      x1 = 32 + (i*19);
      x2 = x1 + 10;
      y1 = 232 - (percent * 203);
      gdImageFilledRectangle(im, x1, y1, x2, 232, FILECOLOR);
      gdImageRectangle(im, x1, y1, x2, 232, black);
   }

   /* data3 */
   for (i=0; i<24; i++)
   {
      percent = ((float)data3[i] / (float)maxval);  /* percent of 100% */
      if (percent <= 0.0) continue;
      x1 = 35 + (i*19);
      x2 = x1 + 10;
      y1 = 232 - (percent * 203);
      gdImageFilledRectangle(im, x1, y1, x2, 232, PAGECOLOR);
      gdImageRectangle(im, x1, y1, x2, 232, black);
   }

   /* save as png	file */
   if ( (out = fopen(fname, "wb")) != NULL)
   {
      gdImagePng(im, out);
      fclose(out);
   }
   /* deallocate memory */
   gdImageDestroy(im);

   return (0);
}

/*****************************************************************/
/*                                                               */
/* PIE_CHART  - draw a pie chart (10 data items max)             */
/*                                                               */
/*****************************************************************/

int pie_chart(char *fname, char *title, u_long t_val,
              u_long data1[], char *legend[])
{
   int i,x,percent,y=47;
   double s_arc=0.0;
   int purple_or_pie1, ltpurple_or_pie2, ltgreen_or_pie3, brown_or_pie4;
   int r, g, b;
   char buffer[128];

   struct pie_data gdata;

   /* init graph and colors */
   init_graph(title,512,300);
   r=getred(pie_color1); g=getgreen(pie_color1); b=getblue(pie_color1);
   purple_or_pie1  = gdImageColorAllocate(im, r, g, b);
   r=getred(pie_color2); g=getgreen(pie_color2); b=getblue(pie_color2);
   ltpurple_or_pie2= gdImageColorAllocate(im, r, g, b);
   r=getred(pie_color3); g=getgreen(pie_color3); b=getblue(pie_color3);
   ltgreen_or_pie3 = gdImageColorAllocate(im, r, g, b);
   r=getred(pie_color4); g=getgreen(pie_color4); b=getblue(pie_color4);
   brown_or_pie4 = gdImageColorAllocate(im, r, g, b);

   /* do the circle... */
   gdImageArc(im, CX, CY, XRAD, YRAD, 0, 360, black);
   gdImageArc(im, CX, CY+10, XRAD-2, YRAD-2, 2, 178, black);
   gdImageFillToBorder(im, CX, CY+(YRAD/2)+1, black, black);

   /* slice the pie */
   gdata=*calc_arc(0.0,0.0);
   gdImageLine(im,CX,CY,gdata.x,gdata.y,black);  /* inital line           */

   for (i=0;i<10;i++)                      /* run through data array      */
   {
      if ((data1[i]!=0)&&(s_arc<1.0))      /* make sure valid slice       */
      {
         percent=(((double)data1[i]/t_val)+0.005)*100.0;
         if (percent<1) break;

         if (s_arc+((double)percent/100.0)>=1.0)
         {
            gdata=*calc_arc(s_arc,1.0);
            s_arc=1.0;
         }
         else
         {
            gdata=*calc_arc(s_arc,s_arc+((double)percent/100.0));
            s_arc+=(double)percent/100.0;
         }

         gdImageLine(im, CX, CY, gdata.x, gdata.y, black);
         gdImageFill(im, gdata.mx, gdata.my, i+7);

         snprintf(buffer, sizeof(buffer),"%s (%d%%)",legend[i], percent);
         x=480-(strlen(buffer)*7);
         gdImageString(im,gdFontMediumBold, x+1, y+1, buffer, black);
         gdImageString(im,gdFontMediumBold, x, y, buffer, i+7);
         y+=20;
      }
   }

   if (s_arc < 1.0)                         /* anything left over?        */
   {
      gdata=*calc_arc(s_arc,1.0);

      gdImageFill(im, gdata.mx, gdata.my, white);
      snprintf(buffer, sizeof(buffer),"%s (%d%%)",msg_h_other,100-(int)(s_arc*100));
      x=480-(strlen(buffer)*7);
      gdImageString(im,gdFontMediumBold, x+1, y+1, buffer, black);
      gdImageString(im,gdFontMediumBold, x, y, buffer, white);
   }

   /* save png image */
   if ((out = fopen(fname, "wb")) != NULL)
   {
      gdImagePng(im, out);
      fclose(out);
   }
   /* deallocate memory */
   gdImageDestroy(im);

   return (0);
}

/*****************************************************************/
/*                                                               */
/* CALC_ARC  - generate x,y coordinates for pie chart            */
/*                                                               */
/*****************************************************************/

struct pie_data *calc_arc(float min, float max)
{
   static struct pie_data data;
   double d;

   /* Calculate max line */
   d=max;
   data.x=cos(d*(2*PI))*((XRAD-2)/2)+CX;
   data.y=sin(d*(2*PI))*((YRAD-2)/2)+CY;
   /* Now get mid-point  */
   d=((min+max)/2);
   data.mx=cos(d*(2*PI))*(XRAD/3)+CX;
   data.my=sin(d*(2*PI))*(YRAD/3)+CY;
   return &data;
}

/*****************************************************************/
/*                                                               */
/* INIT_GRAPH  - initalize graph and draw borders                */
/*                                                               */
/*****************************************************************/

void init_graph(char *title, int xsize, int ysize)
{
   int i, r, g, b;

   im = gdImageCreate(xsize,ysize);

   /* allocate color maps, background color first (grey) */
   r=getred(chbg_color); g=getgreen(chbg_color); b=getblue(chbg_color);
   chartbackground = gdImageColorAllocate(im, r, g, b);
   dkgrey     = gdImageColorAllocate(im, 128, 128, 128);
   black      = gdImageColorAllocate(im, 0, 0, 0);
   white      = gdImageColorAllocate(im, 255, 255, 255);
   r=getred(chlegend_color); g=getgreen(chlegend_color); b=getblue(chlegend_color);
   chartlegend = gdImageColorAllocate(im, r, g, b);
   r=getred(chshad_color1); g=getgreen(chshad_color1); b=getblue(chshad_color1);
   chartshadow1 = gdImageColorAllocate(im, r, g, b);
   r=getred(chshad_color2); g=getgreen(chshad_color2); b=getblue(chshad_color2);
   chartshadow2 = gdImageColorAllocate(im, r, g, b);
   r=getred(hit_color); g=getgreen(hit_color); b=getblue(hit_color);
   hit_or_green = gdImageColorAllocate(im, r, g, b);
   r=getred(site_color); g=getgreen(site_color); b=getblue(site_color);
   site_or_orange = gdImageColorAllocate(im, r, g, b);
   r=getred(file_color); g=getgreen(file_color); b=getblue(file_color);
   file_or_blue = gdImageColorAllocate(im, r, g, b);
   r=getred(kbyte_color); g=getgreen(kbyte_color); b=getblue(kbyte_color);
   kbyte_or_red = gdImageColorAllocate(im, r, g, b);
   r=getred(page_color); g=getgreen(page_color); b=getblue(page_color);
   page_or_cyan = gdImageColorAllocate(im, r, g, b);
   r=getred(visit_color); g=getgreen(visit_color); b=getblue(visit_color);
   visit_or_yellow = gdImageColorAllocate(im, r, g, b);

   /* make borders */

   for (i=0; i<(chart_border+1) ;i++)          /* do shadow effect */
   {
      gdImageLine(im, i, i, xsize-i, i, CHSHADCOLOR1);
      gdImageLine(im, i, i, i, ysize-i, CHSHADCOLOR1);
      gdImageLine(im, i, ysize-i-1, xsize-i-1, ysize-i-1, CHSHADCOLOR2);
      gdImageLine(im, xsize-i-1, i, xsize-i-1, ysize-i-1, CHSHADCOLOR2);
   }

   gdImageRectangle(im, 20, 25, xsize-21, ysize-21, black);
   gdImageRectangle(im, 19, 24, xsize-22, ysize-22, white);
   gdImageRectangle(im, 0, 0, xsize-1, ysize-1, black);

   /* display the graph title */
   blue=file_or_blue;           /* currently bound to blue or file color */
   gdImageString(im, gdFontMediumBold, 20, 8, title, blue);

   return;
}
