/*
 * Priority queue.
 */
#include <stdio.h>
#include <stdlib.h>
#include <gc.h>
#include "pq.h"

#ifndef NULL
#define NULL 0
#endif

PriorityQueue *
PQ_new(int init_size,int (*p_rout)(void*,void*))
{
  PriorityQueue *pq;

  pq = (PriorityQueue*)GC_malloc(sizeof(PriorityQueue));
  pq->data = (void*)GC_malloc(sizeof(void*)*init_size+1);
  pq->n_data = 0;
  pq->area_size = init_size+1;
  pq->priority = p_rout;
  return pq;
}

static void
siftdown(PriorityQueue *pq, int i)
{
  double v0,v1,v2;
  void *tmp;

  if (i*2+1 >= pq->n_data)
    return;

  v0 = pq->priority(pq->data[i],pq->data[i*2+1]);
  if (i*2+2 < pq->n_data) {
    v1 = pq->priority(pq->data[i],pq->data[i*2+2]);
    v2 = pq->priority(pq->data[i*2+1],pq->data[i*2+2]);
  }
  else {
    v1 = 1;
    v2 = 1;
  }
  if (v0 < 0 || v1 < 0) {
    if (v2 >= 0) {
      tmp = pq->data[i*2+1];
      pq->data[i*2+1] = pq->data[i];
      pq->data[i] = tmp;
      siftdown(pq,i*2+1);
    }
    else {
      tmp = pq->data[i*2+2];
      pq->data[i*2+2] = pq->data[i];
      pq->data[i] = tmp;
      siftdown(pq,i*2+2);
    }
  }
}

static void
siftup(PriorityQueue *pq, int i)
{
  double v0;
  void *tmp;

  while (i > 0) {
    v0 = pq->priority(pq->data[(i-1)/2],pq->data[i]);
    if (v0 >= 0) return;
    tmp = pq->data[i];
    pq->data[i] = pq->data[(i-1)/2];
    pq->data[(i-1)/2] = tmp;
    i = (i-1)/2;
  }
}

#ifdef PQ_DEBUG
static void
check(PriorityQueue *pq,char *mes)
{
  int i;
  double v0,v1,v2;
  for (i = 0; i < pq->n_data; i++) {
    if (i*2+1 >= pq->n_data)
      break;
    v0 = pq->priority(pq->data[i]);
    v1 = pq->priority(pq->data[i*2+1]);
    if (i*2+2 < pq->n_data)
      v2 = pq->priority(pq->data[i*2+2]);
    else
      v2 = v0;
    if (v0 < v1 || v0 < v2) {
      fprintf(stderr,"%s: Bad data at %d! parent is %f, children are %f and %f [size=%d area=%d]\n",
	      mes,i,v0,v1,v2,pq->n_data,pq->area_size);
    }
  }
}
#endif

void
PQ_push(PriorityQueue *pq, void *d)
{
  void **newarea;
  int i;

  if (pq->n_data == pq->area_size-1) {
    /* automatically expand queue size */
    newarea = (void**)GC_malloc(sizeof(void*)*pq->area_size*2);
    for (i = 0; i < pq->area_size; i++)
      newarea[i] = pq->data[i];
    pq->data = newarea;
    pq->area_size *= 2;
  }
  pq->data[pq->n_data++] = d;
  if (pq->n_data > 1)
    siftup(pq,pq->n_data-1);
#ifdef PQ_DEBUG
  check(pq,"PQ_push");
#endif
}

void*
PQ_pop(PriorityQueue *pq)
{
  void *res;

  if (pq->n_data == 0)
    return NULL;
  res = pq->data[0];
  pq->data[0] = pq->data[--pq->n_data];
  siftdown(pq,0);
#ifdef PQ_DEBUG
  check(pq,"PQ_pop");
#endif
  return res;
}

int string_alphabetical(void *s1, void *s2)
{
  return -strcmp(s1,s2);
}

int string_reverse_alphabetical(void *s1, void *s2)
{
  return strcmp(s1,s2);
}

