#include "ornamentedobj.h"
#include "texteditor.h"
#include "mapeditor.h"
#include "linkablemapobj.h"

extern TextEditor *textEditor;
extern FlagRowObj *systemFlagsDefault;
extern FlagRowObj *standardFlagsDefault;


/////////////////////////////////////////////////////////////////
// OrnamentedObj
/////////////////////////////////////////////////////////////////

OrnamentedObj::OrnamentedObj():LinkableMapObj()
{
  //  cout << "Const OrnamentedObj ()\n";
    init ();
}

OrnamentedObj::OrnamentedObj(QCanvas* c) :LinkableMapObj(c)
{
//    cout << "Const OrnamentedObj\n";
    init ();
}

OrnamentedObj::OrnamentedObj (OrnamentedObj* lmo) : LinkableMapObj (lmo->canvas)
{
    copy (lmo);
}

OrnamentedObj::~OrnamentedObj()
{
    delete (heading);
	delete (systemFlags);
	delete (standardFlags);

}


void OrnamentedObj::init ()
{
	heading = new HeadingObj(canvas);
	heading->move (absPos.x(), absPos.y());

	note.setNote("");
	note.setFontHint (textEditor->getFontHintDefault() );

	systemFlags=new FlagRowObj(canvas);
	systemFlags->clone(systemFlagsDefault);
	systemFlags->setName ("systemFlags");
	
	standardFlags=new FlagRowObj(canvas);
	standardFlags->clone(standardFlagsDefault);
	standardFlags->setName ("standardFlags");

	hideExport=false;
	hidden=false;

	url="";
	vymLink="";
	
}

void OrnamentedObj::copy (OrnamentedObj* other)
{
    LinkableMapObj::copy(other);
	heading->copy(other->heading);
    setColor   (other->heading->getColor());	

	note.copy (other->note);
	systemFlags->copy (other->systemFlags);
	standardFlags->copy (other->standardFlags);

	ornamentsBBox=other->ornamentsBBox;

	hideExport=other->hideExport;
	url=other->url;
	vymLink=other->vymLink;
}

QString OrnamentedObj::getHeading()
{
    return heading->text();
}

void OrnamentedObj::setLinkColor()
{
	if (mapEditor->getLinkColorHint()==HeadingColor)
		LinkableMapObj::setLinkColor (heading->getColor());
	else	
		LinkableMapObj::setLinkColor (mapEditor->getDefLinkColor());
}

void OrnamentedObj::setColor (QColor col)
{
    heading->setColor(col);
	setLinkColor();
}

QColor OrnamentedObj::getColor ()
{
    return heading->getColor();
}


void OrnamentedObj::positionContents()
{
	double d=frame->getBorder()/2;
	double x=absPos.x();
	double y=absPos.y();

	double ox,oy;	// Offset due to padding

	ox=leftPad + d;
	oy=topPad + d;
	
	systemFlags-> move (ox +x , oy + y );

	// vertical align heading to bottom
    heading->move (ox + x + systemFlags->getBBox().width(),
				   oy + y + ornamentsBBox.height() - heading->getHeight() 
					);
	standardFlags->move (ox +x + heading->getWidth() + systemFlags->getBBox().width() , oy + y );

	ornamentsBBox.moveTopLeft ( QPoint ((int)(ox+x),(int)(oy+y)));
	clickBox.moveTopLeft (QPoint ((int)(ox + x), (int)(oy + y)));
}

void OrnamentedObj::move (double x, double y)
{
	MapObj::move (x,y);
	positionContents();
	updateLink();
	requestReposition();
}

void OrnamentedObj::move (QPoint p)
{
	move (p.x(), p.y());
}	

void OrnamentedObj::moveBy (double x, double y)
{

	MapObj::moveBy (x,y);
    frame->moveBy (x,y);
    systemFlags->moveBy (x,y);
    standardFlags->moveBy (x,y);
    heading->moveBy (x,y);
	updateLink();
	requestReposition();
}

void OrnamentedObj::moveBy (QPoint p)
{
	moveBy (p.x(), p.y());
}	

void OrnamentedObj::move2RelPos(double x, double y)
{
	if (!parObj) return;
	move (parObj->getChildPos().x()+x, parObj->getChildPos().y()+y);
}

void OrnamentedObj::move2RelPos(QPoint p)
{
	move2RelPos (p.x(),p.y());
}

void OrnamentedObj::setNote(QString s)
{
	note.setNote(s);
	if (!note.isEmpty())
		systemFlags->activate("note");
	else		
		systemFlags->deactivate("note");
	calcBBoxSize();
	positionBBox();	
	move (absPos.x(), absPos.y() );
	forceReposition();
}

void OrnamentedObj::setNote(NoteObj n)
{
	note=n;
	if (!note.isEmpty())
		systemFlags->activate("note");
	else		
		systemFlags->deactivate("note");
	calcBBoxSize();
	positionBBox();	
	move (absPos.x(), absPos.y() );
	forceReposition();
	
}

QString OrnamentedObj::getNote()
{
    return note.getNote();
}

QString OrnamentedObj::getNoteASCII()
{
    return note.getNoteASCII();
}

QString OrnamentedObj::getNoteOpenDoc()
{
    return note.getNoteOpenDoc();
}

void OrnamentedObj::setURL(QString s)
{
	url=s;
	if (!url.isEmpty())
		systemFlags->activate("url");
	else	
		systemFlags->deactivate("url");
	calcBBoxSize();			// recalculate bbox
    positionBBox();			// rearrange contents
	forceReposition();
}

QString OrnamentedObj::getURL()
{
	return url;
}

void OrnamentedObj::setVymLink(QString s)
{
	if (!s.isEmpty())
	{
		// We need the relative (from loading) 
		// or absolute path (from User event)
		// and build the absolute path.
		// Note: If we have relative, use path of
		// current map to build absolute path
		QDir d(s);
		if (!d.path().startsWith ("/"))
		{
			QString p=mapEditor->getDestPath();
			int i=p.findRev("/",-1);
			d.setPath(p.left(i)+"/"+s);
			d.convertToAbs();
		}
		vymLink=d.path();
		systemFlags->activate("vymLink");
	}	
	else	
	{
		systemFlags->deactivate("vymLink");
		vymLink="";
	}	
	calcBBoxSize();			// recalculate bbox
    positionBBox();			// rearrange contents
	forceReposition();
}

QString OrnamentedObj::getVymLink()
{
	return vymLink;
}


void OrnamentedObj::toggleStandardFlag(QString f, bool exclusive)
{
	standardFlags->toggle(f,exclusive);
	calcBBoxSize();
	positionBBox();
	move (absPos.x(), absPos.y() );
	forceReposition();
}

void OrnamentedObj::activateStandardFlag(QString f)
{
	standardFlags->activate(f);
	calcBBoxSize();
	positionBBox();
	move (absPos.x(), absPos.y() );
	forceReposition();
}

bool OrnamentedObj::isSetStandardFlag (QString f)
{
	return standardFlags->isActive(f);
}

QString OrnamentedObj::getSystemFlagName(const QPoint &p)
{
	return systemFlags->getFlagName(p);	
}

bool OrnamentedObj::isActiveFlag (const QString & fname)
{
	if (standardFlags->isActive (fname) ) return true;
	return false;
}

void OrnamentedObj::updateNoteFlag()
{
	if (selected) 
	{
		// text in NoteEditor has changed, notify MapEditor 
		mapEditor->setChanged();

		// save text
		setNote( textEditor->getText() );
	
		// save font   
		note.setFontHint (textEditor->getFontHint() );
	}	
}

void OrnamentedObj::setHideInExport(bool b)
{
	if (parObj)
	{
		// Don't allow to MapCenter to be hidden
		hideExport=b;
		if (b)
			systemFlags->activate("hideInExport");
		else	
			systemFlags->deactivate("hideInExport");
		calcBBoxSize();
		positionBBox();
		requestReposition();	
	}
}	

bool OrnamentedObj::hideInExport()
{
	return hideExport;
}	

bool OrnamentedObj::isHidden()
{
	return hidden;
}	

QString OrnamentedObj::getOrnAttr()
{
	QString posAttr;

	if (useRelPos)
		posAttr=attribut("relPosX",QString().setNum(relPos.x(),10)) +
                attribut("relPosY",QString().setNum(relPos.y(),10)); 
	else
	{
		if (depth==0 || depth==1) posAttr=
			attribut("absPosX",QString().setNum(absPos.x(),10)) +
			attribut("absPosY",QString().setNum(absPos.y(),10)); 
		else
			posAttr="";
	}	

	QString urlAttr;
	if (!url.isEmpty())
		urlAttr=attribut ("url",url);

	QString vymLinkAttr;
	if (!vymLink.isEmpty())
		vymLinkAttr=attribut ("vymLink",convertToRel(mapEditor->getDestPath(),vymLink) );

	QString hideExpAttr;
	if (hideExport)
		hideExpAttr= attribut("hideInExport","true");
	else	
		hideExpAttr="";

	return posAttr +urlAttr +vymLinkAttr +getLinkAttr() +hideExpAttr;
}

