/*
 * pcaputil.c
 *
 * Copyright (c) 2000 Dug Song <dugsong@monkey.org>
 *
 * $Id: pcapu.c,v 1.1.1.1 2001/12/12 18:31:10 provos Exp $
 */

#include "config.h"

#include <sys/param.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/queue.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>
#include <pcap.h>
#ifdef BSD
#include <pcap-int.h>
#endif
#include <event.h>

#include "pcapu.h"
#include "voip.h"

struct event pcapev;
int filemode;

#ifdef BSD
static int
bpf_immediate(int fd, int on)
{
	return (ioctl(fd, BIOCIMMEDIATE, &on));
}
#endif

int
pcap_dloff(pcap_t *pd)
{
	int offset = -1;
	
	switch (pcap_datalink(pd)) {
	case DLT_EN10MB:
		offset = 14;
		break;
	case DLT_IEEE802:
		offset = 22;
		break;
	case DLT_FDDI:
		offset = 21;
		break;
#ifdef DLT_LOOP
	case DLT_LOOP:
#endif
	case DLT_NULL:
		offset = 4;
		break;
	default:
		warnx("unsupported datalink type");
		break;
	}
	return (offset);
}

void
pcap_cb(int fd, short event, void *arg)
{
	struct timeval tv;
	pcap_t *pd = (pcap_t *)arg;
	u_char *pkt;
	struct pcap_pkthdr pkthdr;

	timerclear (&tv);

	if ((pkt = (u_char *) pcap_next(pd, &pkthdr)) == NULL) {
		if (filemode) {
			fprintf(stderr, __FUNCTION__": ! add\n");
			return;
		}

		tv.tv_usec = 10 * 1000;	/* 10 ms */
		goto out;
	}

	voip_sniff((u_char *)pd, &pkthdr, pkt);

 out:
	timeout_add(&pcapev, &tv);
}

pcap_t *
pcap_init(char *intf, char *file, char *filter, int snaplen)
{
	pcap_t *pd = NULL;
	u_int net, mask;
	struct bpf_program fcode;
	struct timeval tv;
	char ebuf[PCAP_ERRBUF_SIZE];

	/* Cludge */
	if (intf == NULL && (intf = pcap_lookupdev(ebuf)) == NULL) {
		warnx("%s", ebuf);
		return (NULL);
	}

	if (file == NULL) {
		errx(1, "no dumpfile specified");
		/* NOT REACHED */
	} else {
		pd = pcap_open_offline(file, ebuf);
		filemode = 1;
	}
	
	if (pd == NULL) {
		warnx("%s", ebuf);
		return (NULL);
	}

	if (pcap_lookupnet(intf, &net, &mask, ebuf) == -1) {
		warnx("%s", ebuf);
		return (NULL);
	}  
	if (pcap_compile(pd, &fcode, filter, 1, mask) < 0) {
		pcap_perror(pd, "pcap_compile");
		return (NULL);
	}
	if (pcap_setfilter(pd, &fcode) == -1) {
		pcap_perror(pd, "pcap_compile");
		return (NULL);
	}
#ifdef BSD
	if (file == NULL && bpf_immediate(pd->fd, 1) < 0) {
		perror("ioctl");
		return (NULL);
	}
#endif

	timeout_set(&pcapev, pcap_cb, pd);
	timerclear(&tv);
	timeout_add(&pcapev, &tv);

	return (pd);
}

/* from tcpdump util.c. */
char *
copy_argv(char **argv)
{
	char **p, *buf, *src, *dst;
	u_int len = 0;
	
	p = argv;
	if (*p == 0)
		return (0);
	
	while (*p)
		len += strlen(*p++) + 1;
	
	if ((buf = (char *)malloc(len)) == NULL)
		err(1, "copy_argv: malloc");
	
	p = argv;
	dst = buf;
	
	while ((src = *p++) != NULL) {
		while ((*dst++ = *src++) != '\0')
			;
		dst[-1] = ' ';
	}
	dst[-1] = '\0';
	
	return (buf);
}
