/*
 * Usage:
 * $0 [ options... ] [ files ]

 * Filters the named files; if there are no files, stdin is used.

 * Options:
 *	-r: reverse short-, int-, and long-sized objects.
 *	    N.B.  General byte-swapping throughout the file should
 *	    be handled by something like dd.
 *	-f fmt: format output according to fmt (default "a").
 *	-F file: file contains format.
 *	-h: print help message, then exit.
 *	-l x: limit output to no more than x characters per line (default 80).
 *	-p: print offsets (lines begin <offset>:<tab> ).
 *	-o: like -p, but print using octal.
 *	-P len: print offsets (lines begin <len>x<n>+<offset>:<tab> ).
 *	-O: like -P, but print offset part using octal.
 *	-t: use \# T (T=F,I,...) to indicate data type; also
 *	    start newlines for each data type or copyout, and
 *	    terminate copyouts with tabs.  WARNING: inviz can't properly
 *		invert output unless -t is set.

 *	-L: print a human-readable version of the internal representation
 *		of the format.
 *	-D: turn on debug mode.

 * Formats:
 *	see grammar.y for structure.

 * Output formatting:
 * Ascii:
 * bell -> \a	(__STDC__ only)
 * backspace -> \b
 * formfeed -> \f
 * newline -> \n
 * return -> \r
 * tab -> \t
 * vertical tab -> \v	(__STDC__ only)
 * backslash -> \\
 * ^ -> \^
 * all other printing characters (octal 040 - 177): unchanged
 * all other non-printing characters in the range 0 - 039: ^@ ^A, etc
 * all others -> \nnn
 * In addition, after transforming newline to \n, a true newline is printed.

 */

#include <stdio.h>
#include <ctype.h>
#include "string.h"
#include "memory.h"
#include <errno.h>

#define VIZ_MAIN
#include "viz.h"

extern errno;
extern int sys_nerr;
extern char *sys_errlist[];


extern char *malloc();
extern optind, opterr;
extern char *optarg;

char *prog;
struct Format Fmt = {
	"a",		/* Default format */
	NULL,		/* input text */
	0,		/* format is not from a file */
	0,		/* current line number in format */
	NULL,		/* ptr to start of curr line */
	NULL};		/* ptr to 1 past last item */
int show_list = 0;		/* show the format list */
int debug = 0;			/* ==list + show the creation of the list */
int show_type = 0;		/* use \# T to show data type */
int process_to_eof = 1;		/* !0 means user gave a counted item list */
int print_offset_octal = 0;	/* print offsets using octal output */
int reverse_int = 0;		/* reverse ea integral type before evaluating */
int do_condense = 1;		/* bool: condense list if !0 */

int print_offset = 0;

/* Describes what's happening on the current line. */
struct {
    int maxlen;			/* Maximum line length */
    int avail;			/* Amount remaining on current line
				 * before running into maxlen.
				 */
    int avail_nl;		/* amount remaining after doing newline
				 * plus any offset labelling, etc.
				 */
    char mode;			/* Current output mode: ( a/c/C/S/.../\0 ) */
} line  =  { 80, 80, 80, '\0' };

int item_len;

/* For printing values in binary */
char binary[256][9];

/* Registers for storing counts */
long count_reg[256];

/* Amount of data (in chars) that have been processed so far.  */
long *data_offset = &count_reg[(int) '@'];

static struct {
    char *main;		/* main i/o buffer */
    char *secondary;	/* when needed, data is copied here for alignment */
    char *end;		/* marks 1 past last valid posn in main buf */
    char *next;		/* points to next char in main that is unprocessed */
} readbuf = { NULL, NULL, NULL, NULL };

int out_of_size = 0;	/* Set to 1 when read_data has returned all it can in
			 * requested size
			 */
int out_of_data = 0;	/* Set to 1 when read_data has returned all it can
			 * (note that EOF occurs before read_data finishes
			 * returning all the final data from its buffer)
			 */


typedef union {
	char *c;
	unsigned char *uc;
	short *s;
	unsigned short *us;
	int *i;
	unsigned int *ui;
	long *l;
	unsigned long *ul;
	float *f;
	double *d;
    } ALLTYPE;

#ifdef MIN
#undef MIN
#endif

#define MIN(a,b) ( (a) < (b) ? (a) : (b) )

/* Need should be used before outputting an item; it makes sure there's
 * enough space on the current line, and if not, it calls do_newline
 * to get the space.  It's a macro because it's called on a per-item basis.
 */
#define need(n) {			\
    register mode;			\
    if (line.avail < (n) ) {		\
	mode = line.mode;		\
	do_newline();			\
	set_mode(mode);			\
    }					\
}

/* OUTPUTSTR uses need() to check that there's room for a string; then
 * prints it, and adjusts the available-space counter.  It's a macro
 * because it's called on a per-item basis.
 */
#define OUTPUTSTR(s)			\
{					\
    register n=0;			\
    register char *q, *p;		\
    q = p = (s);			\
    while (*p) {			\
	n++;				\
	if (*p++ == '\t') n += 7;	\
    }					\
    need(n);				\
    fputs((q), stdout);			\
    line.avail -= n;			\
}

/* adj_count(x) is used to keep a counter in the valid range when
 * it's counting to EOF.  (Note that a file could be larger than
 * LONG_MAX, so counting bytes through EOF would not be good.)
 * Therefore adj_count(x) checks if x < 0 (which is the sign that
 * we are counting to EOF) and replaces it with UPTO_EOF, a number
 * that is less than 0 but not << 0.  If x > 0, we don't assign
 * anything to x.  adj_count() should not appear in any inner loop,
 * of course.
 */
#define adj_count(x) (x < 0 ? x = UPTO_EOF : UPTO_EOF)

main(argc, argv)
int argc;
char **argv;
{
    int n;
    void fillformat();
    void viz();
    void viz_text();
    void line_init();
    void binary_init();

    int i, nfiles;

    prog = argv[0];

    rootlist = NIL;

    binary_init();

    if  ( (i = get_options(argc, argv)) == 0)
	return 1;

    if (yyparse() != 0)
	return 1;

    if (process_to_eof) {
	/* Apply an <inf> multiplier to the user's list.  Potential
	 * problem: when the list includes sublists, the rootlist
	 * returned by the grammar points to the first sublist.
	 * Thus, we repackage the entire list as a sublist of a new root;
	 * this makes sure that the count applied to the rootlist is
	 * applied to the entire list.
	 */
	MEMBER *temp = newlist(rootlist);
	temp->count = UPTO_EOF;
	rootlist = temp;
    }

    if (debug || show_list) {
	(void) fprintf(stderr, "Before condensing:\n");
	printlist(0, rootlist);
	(void) fprintf(stderr, "--------\n");
    }

    if (do_condense) {
	n = condenselist(rootlist);

	if (debug || show_list) {
	    (void) fprintf(stderr, "After condensing:\n");
	    (void) fprintf(stderr, "%d change%s\n", n, (n!=1) ? "s" : "");
	    printlist(0, rootlist);
	    (void) fprintf(stderr, "--------\n");
	}
    } else {
	if (debug || show_list) {
	    (void) fprintf(stderr, "List condensing is turned off\n");
	}
    }

    fillformat(rootlist);


    nfiles = argc - i;

    if (i == argc) {
	if (read_data_init() == -1)
	    return 1;
	clearerr(stdin);
	out_of_size = 0;
	out_of_data = 0;

	line_init();

	if (print_offset)
	    (void) put_offset();

	viz(rootlist, 1);
    } else {
	do {
	    if (freopen(argv[i], "r", stdin) == (FILE *) NULL) {
		(void) fprintf(stderr, "%s: failed to open %s: ",prog, argv[i]);
		perror("");
	    } else {

		if (read_data_init() == -1)
		    return 1;
		clearerr(stdin);
		out_of_size = 0;
		out_of_data = 0;

		*data_offset = 0;

		if (nfiles > 1)
		    (void) printf("\n==> %s <==\t\n", argv[i]);

		line_init();

		if (print_offset)
		    (void) put_offset();

		viz(rootlist, 1);
	    }
		
	} while (++i < argc);
    }

    if ( !out_of_data  &&  process_to_eof  &&  !ferror(stdin) ) {
	/* There can be up to (sizeof(double) - 1) characters that remain
	 * unprocessed, if the final input elements are 'D', but only 1
	 * char remains on stdin.  Process the remainder in ascii.
	 */
	viz_text((long) sizeof(double), 1, 'C', NULL, 0);
    }

    if (line.avail < (line.maxlen-1) ) {	/* Note that line_init sets
						 * line.avail=line.maxlen-1
						 * for an empty line */
	(void) putchar('\n');
    }
    return 0;
}

void
binary_init()
{
    /* Initializes the array binary[256][9] with
     * the binary ascii representation of 0..255.
     */

    int i, j, n;
    static char *s16[16] = {
	"0000", "0001", "0010", "0011",
	"0100", "0101", "0110", "0111",

	"1000", "1001", "1010", "1011",
	"1100", "1101", "1110", "1111",
    };


    for (i=0; i<16; i++)
	for (n=i*16, j=0; j<16; j++) {
	    (void) strcpy(binary[n+j], s16[i]);
	    (void) strcpy(binary[n+j]+4, s16[j]);
	}
}

int get_options(argc, argv)
int argc;
char **argv;
{
    /* Sets options; returns argv index of first file argument.
     * If there are no file arguments, returns argc.
     */
    int c;
    char *loadfile();
    int fmt_done = 0;
    int i;

    while ( (c = getopt(argc, argv, "c:f:F:hl:oO:pP:tDL")) != -1) {
	switch (c) {
	case 'r':
	    reverse_int = 1;
	    break;
	case 'f':
	    if (fmt_done) {
		(void) fprintf(stderr,
			"%s: Only one instance of -f and/or -F may be used.\n",
			prog);
		exit(1);
	    }
	    Fmt.text = optarg;
	    fmt_done = 1;
	    break;
	case 'F':
	    if (fmt_done) {
		(void) fprintf(stderr,
			"%s: Only one instance of -f and/or -F may be used.\n",
			prog);
		exit(1);
	    }
	    Fmt.file = 1;
	    Fmt.text = loadfile(optarg);
	    fmt_done = 1;
	    if (Fmt.text == NULL) {
		(void) fprintf(stderr,
			"%s: Failed to load format from `%s' -- ",
			prog, optarg);
		if (errno > 0 && errno <= sys_nerr)
		    (void) fprintf(stderr, "%s\n", sys_errlist[errno]);
		else
		    (void) fprintf(stderr, "errno = %d\n", errno);
		exit(1);
	    }
	    break;
	case 'h':
	    (void) fprintf(stderr, "\
Usage:\n%s [-f fmt | -F file] \n\
\t[-r] [-h] [-l x] [-p|o] [-P|O len] [-t] [-L] [-D] [file ...]\n",
		prog);
	    (void) fprintf(stderr, "\
Options:\n");
	    (void) fprintf(stderr, "\
-f fmt:  format output according to fmt (default \"a\").\n\
-F file: format is contained in file.\n\
\n\
-h:      print this help message on stderr, then exit.\n\
-l x:    limit output to no more than x characters per line (default 80).\n\
-p:      print offsets (each line is begun <offset>:<tab> ).\n\
-P len:  print offsets using format <len>*<n>+<offset>:<tab> ).\n\
-o, -O len: like -p and -P, but using octal for offset.\n\
-r:      reverse bytes in each integer-type value before interpreting\n\
         (this applies to short, int, and long-sized data).\n\
-t:      use \\# <T> to show data type (so inviz can invert result).\n\
-L:      print internal version of the format (not generally useful).\n\
-D:      turn on debug mode (not generally useful).\n\n");

	    (void) fprintf(stderr, "\
Formats:\n\
If fmt doesn't begin with a count, it's repeated until EOF is reached.\n\
Basic format: [count] input_size output_fmt [ >reg_char ].\n\
    + count:      decimal or $reg_char or $$.\n\
    + input_size: one of C/Z/S/I/L/F/D.\n\
    + output_fmt: one of a/b/c/d/u/o/h/x/f/g/~/\"printf-fmt\".\n\
    + Default pairings: [abc~] <-> C; [duohx] <-> I; [fg] <-> F\n\
			[CZ]   <-> a; [SIL]   <-> d; [FD] <-> g\n\
    + Illegal pairings: [CSIL] & [fg]; [SILFD] & [ac]; [FD] & [abcduohx]\n");
	    (void) fprintf(stderr, "\
Special formats:\n\
    + n or \\n  output a newline; e.g. 17n means 17 newlines\n\
    + [xxx]    output comment xxx; e.g. 2[hello] means print `hello' twice\n\
Compound formats:\n\
    + The above simple formats can be concatenated:\n\
	    24a5f      means 24 ascii chars, then 5 floats\n\
    + Parentheses group and nest:\n\
	    I3(24a5f)  means 1 int, then 3 groups of 24 chars & 5 floats\n");
	    exit(0);
	    break;
	case 'l':
	    line.maxlen = atoi(optarg);
	    break;
	case 'o':
	    print_offset_octal = 1;
	    /* FALLTHROUGH */
	case 'p':
	    if (print_offset) {
		fprintf(stderr,
		 	"Options -p|o and -P|O are mutually exclusive\n");
		exit(1);
	    }
	    print_offset = 1;
	    break;
	case 'O':
	    print_offset_octal = 1;
	    /* FALLTHROUGH */
	case 'P':
	    if (print_offset) {
		fprintf(stderr, "Options -p and -P are mutually exclusive\n");
		exit(1);
	    }
	    print_offset = atoi(optarg);
	    if (print_offset < 2) {
		fprintf(stderr,
		"%s: record length for offsets must be > 1; you used `%s'\n",
		prog, optarg);
		exit(1);
	    }
	    break;
	case 't':
	    show_type = 1;
	    break;
	case 'L':
	    show_list = 1;
	    break;
	case 'D':
	    debug = 1;
	    break;
	case '?':
	    return 0;
	    /* NOTREACHED */
	    break;
	default:
	    (void) fprintf(stderr,
		"%s: unexpected option `%c' returned to get_options\n",
		prog, c);
	    return 0;
	    /* NOTREACHED */
	    break;
	}
    }
    return optind;
}

void fillformat(list)
MEMBER *list;
{
    MEMBER *p;

#define IC (p->u.iospec.ichar)
#define OC (p->u.iospec.ochar)

    for (p = list; p != NIL; p = p->next) {
	if (p->type == T_LISTHEAD) {
	    fillformat(p->u.sublist);
	} else if (p->type == T_IOSPEC) {
	    if (p->u.iospec.fmt != NULL)
		;	/* Format already assigned by user */

	    else if (OC == 'a' || OC == 'c' || OC == '~')
		;	/* Format isn't used for 'a', 'c', '~' output */

	    else if (IC == 'C' && OC == 'b')
		p->u.iospec.fmt = Cb_fmt;
	    else if (IC == 'C' && OC == 'o')
		p->u.iospec.fmt = Co_fmt;
	    else if (IC == 'C' && OC == 'd')
		p->u.iospec.fmt = Cd_fmt;
	    else if (IC == 'C' && OC == 'u')
		p->u.iospec.fmt = Cu_fmt;
	    else if (IC == 'C' && OC == 'x')
		p->u.iospec.fmt = Cx_fmt;

	    else if (IC == 'Z' && OC == 'b')
		p->u.iospec.fmt = Zb_fmt;
	    else if (IC == 'Z' && OC == 'o')
		p->u.iospec.fmt = Zo_fmt;
	    else if (IC == 'Z' && OC == 'd')
		p->u.iospec.fmt = Zd_fmt;
	    else if (IC == 'Z' && OC == 'u')
		p->u.iospec.fmt = Zu_fmt;
	    else if (IC == 'Z' && OC == 'x')
		p->u.iospec.fmt = Zx_fmt;

	    else if (IC == 'S' && OC == 'b')
		p->u.iospec.fmt = Sb_fmt;
	    else if (IC == 'S' && OC == 'o')
		p->u.iospec.fmt = So_fmt;
	    else if (IC == 'S' && OC == 'd')
		p->u.iospec.fmt = Sd_fmt;
	    else if (IC == 'S' && OC == 'u')
		p->u.iospec.fmt = Su_fmt;
	    else if (IC == 'S' && OC == 'x')
		p->u.iospec.fmt = Sx_fmt;

	    else if (IC == 'I' && OC == 'b')
		p->u.iospec.fmt = Ib_fmt;
	    else if (IC == 'I' && OC == 'o')
		p->u.iospec.fmt = Io_fmt;
	    else if (IC == 'I' && OC == 'd')
		p->u.iospec.fmt = Id_fmt;
	    else if (IC == 'I' && OC == 'u')
		p->u.iospec.fmt = Iu_fmt;
	    else if (IC == 'I' && OC == 'x')
		p->u.iospec.fmt = Ix_fmt;

	    else if (IC == 'L' && OC == 'b')
		p->u.iospec.fmt = Lb_fmt;
	    else if (IC == 'L' && OC == 'o')
		p->u.iospec.fmt = Lo_fmt;
	    else if (IC == 'L' && OC == 'd')
		p->u.iospec.fmt = Ld_fmt;
	    else if (IC == 'L' && OC == 'u')
		p->u.iospec.fmt = Lu_fmt;
	    else if (IC == 'L' && OC == 'x')
		p->u.iospec.fmt = Lx_fmt;

	    else if (IC == 'F' && OC == 'f')
		p->u.iospec.fmt = Ff_fmt;
	    else if (IC == 'F' && OC == 'g')
		p->u.iospec.fmt = Fg_fmt;

	    else if (IC == 'D' && OC == 'f')
		p->u.iospec.fmt = Df_fmt;
	    else if (IC == 'D' && OC == 'g')
		p->u.iospec.fmt = Dg_fmt;

	    else {
		(void) fprintf(stderr,
		    "%s: ichar = `%c', ochar = `%c' in fillformat()!\n",
		    prog, IC, OC);
		break;
	    }
	}
    }
#undef IC
#undef OC
}

void line_init()
{
    /* Allow 1 position at end of line for the newline that is bound to
     * eventually be printed.
     */
    line.avail_nl = line.avail = line.maxlen - 1;
    line.mode = '\0';
}

do_newline()
{
    /* Starts a new line of output; the output mode is set to '\0' */

    putchar('\n');
    line_init();
    if (print_offset)
	(void) put_offset();
    line.avail_nl = line.avail;	/* amount available after doing newline */
}

void set_mode(newmode)
int newmode;
{
    /* Set_mode changes modes.  If the mode is currently the desired mode,
     * no action is taken.  Otherwise, if the mode is currently '\0',
     * it means that nothing has yet been written to the current line
     * (excepting a possible offset), and a newline need not be done
     * before output setting mode.  Otherwise, if the mode is switching into
     * or out of 'a' and show_type is set, do_newline is called to set up
     * a new line; the mode is then set.

     * Special case: if the old mode is '~', and we are printing offsets,
     * force a newline to begin so that skipped data don't cause a
     * nonsensical offset on the line following the skipped data

     * Set_mode is a function on the assumption that mode changes are
     * relatively rare, compared to outputting many values at a single mode.
     */

#if 0
    fprintf(stderr,
    "set_mode: line.mode = %c  newmode = %c print_offset = %d\n",
    line.mode, newmode, print_offset);
#endif

    if (newmode == line.mode)
	return;

    if (show_type) {
	if (line.avail != line.avail_nl) {
	    /* have printed something; may need a newline */
	    if (line.mode == 'a' || newmode == 'a'
		    || (print_offset && line.mode == '~'))
	    do_newline();
	}

	if (newmode != '\0'  &&  newmode != 'a') {
	    printf("\\# %c ", newmode);
	    line.avail -= 6;
	}
    } else if (print_offset && line.mode == '~') {
	do_newline();
    }
    line.mode = newmode;
}

read_data_init()
{
    if ( readbuf.main == NULL) {
	if ( (readbuf.main = malloc((unsigned) BUFFERSIZE)) == NULL ||
	     (readbuf.secondary = malloc((unsigned) BUFFERSIZE)) == NULL ) {
	    (void) fprintf(stderr,
		    "%s: malloc failed in read_data_init: ", prog);
	    perror("");
	    return -1;
	}
    }
    readbuf.end = readbuf.main + BUFFERSIZE;
    readbuf.next = readbuf.end;
    return 0;
}

void
seek_data(dsize, offset, whence)
int dsize;	/* Size of one datum */
long offset;	/* How much to seek, in units of dsize */
int whence;	/* 0, 1, 2 mean seek w.r.t. start, cur, eof */
{
    /* If seeking relative to current location, convert to an absolute
     * seek from origin (because we've probably read past user's notion
     * of current location.
     */
    long my_offset = dsize*offset;
    int my_whence = whence;
    if (whence == 1) {
	my_offset += *data_offset;
	my_whence = 0;
    }
    if (fseek(stdin, my_offset, my_whence) != 0) {
	fprintf(stderr, "%s: fseek(stdin, %ld, %d) failed:\n",
			prog, offset*dsize, whence);
	perror("");
	if (my_whence != whence) {
	    fprintf(stderr, "N.B. Actual call was fseek(stdin, %ld, %d), \
to correct for %s's readahead.\n", my_offset, my_whence, prog);
	}
	exit(1);
    }
    /* Invalidate the current contents of the input stream */
    readbuf.next = readbuf.end;
}

read_data(dsize, type, n, p)
int dsize;	/* Size of one datum; must be < length of an io buffer */
int type;	/* one of 'C' 'S', 'I', 'L', 'F', 'D', 'Z' */
long n;		/* number of dsize elt's requested; ignored for type==Z;
		 * if < 0, LONG_MAX is substituted.
		 */
char **p;	/* On return, *p pts to beginning of first datum */

    /* Return: 0 on EOF or fread error; else number available, 0 .. n
     * (A 0 return, if not EOF or error, means that there are fewer than dsize
     * chars available).
     */
{
    register int m, nc;

    char *s;
    int nread;

    if (n < 0)
	n = LONG_MAX;

    /* Check if we need to get more data:
     * we get more whenever there is less than dsize data
     * remaining in our buffer (or when the type is 'Z' and there isn't
     * a null somewhere in the input buffer); otherwise, we return whatever
     * we have already available (dsize is size of one datum) */

    nc = readbuf.end - readbuf.next;	/* number of remaining chars */

    /* fprintf(stderr, "readdata request is %d nc = %d... ", n, nc); */

    if (nc < dsize ||
	(type == 'Z' && (s = memchr(readbuf.next, '\0', nc)) == NULL) ) {
	/* need more */

	if (nc > 0)
	    (void) memcpy(readbuf.main, readbuf.next, nc);

	readbuf.next = readbuf.main;

	/* Read more data and recompute nc */
	nread = fread(readbuf.main + nc, 1, BUFFERSIZE - nc, stdin);
	readbuf.end = readbuf.main + nc + nread;
	nc = readbuf.end - readbuf.next;

	if (nc < dsize)
	    out_of_size = 1;
	if (nc == 0)
	    out_of_data = 1;
    }
    /* fprintf(stderr, "%d\n", nc); */

    /* Compute m = number of dsize elements available in buffer; m is the
     * return value from read_data.  We assume that all m elements
     * be consumed by caller.

     * Note that we've tested, above, that the number of available
     * characters is at least dsize; m will be > 0.
     */
    if (type == 'Z') {
	if ((s = memchr(readbuf.next, '\0', nc)) == NULL)
	    m = nc; /* Didn't reach a null; oh well. */
	else
	    m = s + 1 - readbuf.next;
    } else {
	m = MIN( nc / dsize, n);
    }


    /* Now let nc be the number of characters in m elements: */
    nc = m * dsize;
    if (aligned(type, readbuf.next)) {
	*p = readbuf.next;
    } else {
	(void) memcpy(readbuf.secondary, readbuf.next, nc);
	*p = readbuf.secondary;
    }
    readbuf.next += nc;

    if (reverse_int) {
	switch (type) {
	case 'S':
	    {
	    register i;
	    register short *sp;
	    for (sp = (short *) *p, i=m+1; --i; ++sp) {
		REVERSE_SHORT(*sp, short);
	    }
	    }
	    break;

	case 'I':
	    {
	    register int i;
	    register int *ip;
	    for (ip = (int *) *p, i=m+1; --i; ++ip) {
		REVERSE_INT(*ip, int);
	    }
	    }
	    break;

	case 'L':
	    {
	    register int i;
	    register long *lp;
	    for (lp = (long *) *p, i=m+1; --i; ++lp) {
		REVERSE_LONG(*lp, long);
	    }
	    }
	    break;
	default:
	    /* No action is default */
	    break;
	}
    }
    /* fprintf(stderr, "readdata returning %d\n", m); */
    return m;
}

void
unread_data(n)
long n;		/* number of chars to unread */

    /* Can cause terrible problems if n > number_points gotten on last
     * read_data.  Silently tries to protect itself against
     * invalid arguments.  (There didn't seem to be much point
     * in telling caller about invalid arguments, as caller should
     * know exactly how much data it got on last call to read_data().)
     */
{



    if (readbuf.next - readbuf.main < n)
	return;		/* Can't push back n chars */

    readbuf.next -= n;

    return;
}

put_offset()
{
    char offset_buf[60];
    int i;

    if (print_offset == 1) {
	if (print_offset_octal)
	    (void) sprintf(offset_buf, "%#6o:\t", *data_offset);
	else
	    (void) sprintf(offset_buf, "%6d:\t", *data_offset);

    } else {
	register int l = *data_offset / print_offset;
	register int m = *data_offset % print_offset;
	if (print_offset_octal)
	    (void) sprintf(offset_buf, "%d*%d+%#3o:\t", print_offset, l, m);
	else
	    (void) sprintf(offset_buf, "%d*%d+%3d:\t", print_offset, l, m);
    }
    /* Compute amount on line.  Allow 8 for tab -- but strlen() returns
     * 1 for tab, so add 7 more.  The line.avail value is increased by
     * 1 to allow for fact that it's decremented by line_init.
     */
    i = (line.maxlen - (line.avail + 1)) + strlen(offset_buf) + 7;
    /* Take result modulo 8 by masking off lowest 3 bits. */
    i &= ~7;
    line.avail = line.maxlen - i;
    (void) fputs(offset_buf, stdout);

    return 0;
}

void
viz(list, iteration)
MEMBER *list;
int iteration;	/* which iteration of the list this is */
{

    register i;
    MEMBER *p;

    long count;
    int size;
    char ichar;

    int n;
    long lastv;

    void viz_text(), viz_numeric();

    ALLTYPE u;

    count_reg[(int) '('] = iteration;

    for (p = list; p != NIL && !out_of_size ; p = p->next) {

	count = (p->count >= 0) ? p->count :
		    (p->count == UPTO_EOF) ? LONG_MAX : count_reg[ - p->count];
	if (count <= 0)
	    continue;

	switch (p->type) {
	case T_IOSPEC:
	    switch (p->u.iospec.ochar) {
	    case 'a':
		viz_text(count, p->u.iospec.size,
			p->u.iospec.ichar, p->u.iospec.fmt,
			    (int) p->u.iospec.reg_no);
		break;
	    case '~':
		/* Output nothing, but read at least the last value so that
		 * we can store it in a register if need be. ...ok, we should
		 * really do a seek here, but I don't think it matters that
		 * much and it would be a pain to incorporate seeking when
		 * not all input sources might support it.
		 */
		set_mode('~');
		size = p->u.iospec.size;
		ichar = p->u.iospec.ichar;
		lastv = 0;
		/* save the last value.  This is a kludgy way
		 * of doing it, but what the hey.
		 */
		switch (ichar) {
		case 'C':
		    for (; count && (n = read_data(size, ichar, count, &u.c));
				count -= n) {
			*data_offset += n * size;
			lastv = (long) *(u.c + n - 1);
		    }
		    break;
		case 'Z':
		    for (; count &&
			(n = read_data(size, ichar, count, &u.c)); count--) {
			*data_offset += n * size;
			lastv = n;
		    }
		    break;
		case 'S':
		    for (; count &&
			(n = read_data(size, ichar, count, (char **) &u.s));
				count -= n) {
			*data_offset += n * size;
			lastv = (long) *(u.s + n - 1);
		    }
		    break;
		case 'I':
		    for (; count &&
			    (n = read_data(size, ichar, count, (char **) &u.i));
				count -= n) {
			*data_offset += n * size;
			lastv = (long) *(u.i + n - 1);
		    }
		    break;
		case 'L':
		    for (; count &&
			(n = read_data(size, ichar, count, (char **) &u.l));
				count -= n) {
			*data_offset += n * size;
			lastv = *(u.l + n - 1);
		    }
		    break;
		default:
		    /* Don't need to save values of float or double data;
		     * just read it
		     */
		    for (; count && (n = read_data(size, ichar, count, &u.c));
				count -= n) {
			*data_offset += n * size;
		    }
		    break;
		}
		if (p->u.iospec.reg_no)
		    count_reg[p->u.iospec.reg_no] = lastv;
		else
		    count_reg['#'] = lastv;
		break;
	    default:
		    viz_numeric(count, p->u.iospec.size,
			    p->u.iospec.ichar, p->u.iospec.ochar,
			    p->u.iospec.fmt, (int) p->u.iospec.reg_no);
		    break;
		}
		break;
	case T_LISTHEAD:
	    for (i=0; i < count  &&  !out_of_size; i++)
		viz(p->u.sublist, i+1);
	    /* Re-load $( with iteration count for this level; was overwritten
	     * by recursive call to viz().
	     */
	    count_reg[(int) '('] = iteration;
	    break;
	case T_COPYOUT:
	    {

		if (show_type)
		    do_newline();
		for (i=0; i < count; i++) {
		    char s[1000];
		    viz_decode(p->u.copyout, 'a', s);
		    OUTPUTSTR(s);
		    if (show_type)
			OUTPUTSTR("\t");
		}
	    }
	    break;
	case T_SEEK:
	    {
	    int whence;
	    if (p->u.seek.direction < 0 && p->u.seek.direction >= -255)
		whence = count_reg[-p->u.seek.direction];
	    else
		whence = p->u.seek.direction;
	    seek_data(1, p->u.seek.count, whence);
	    }
	    break;
	case T_NEWLINE:
	    /* Put out n-1 newlines; then use do_newline to get the
	     * final newline out.
	     */
	    for (i=1; i < count; i++)
		putchar('\n');
	    do_newline();
	    break;
	case T_MATH:
	    /* Do math on register */
	    n = p->u.math.operand;
	    if (n == UPTO_EOF)
		n = iteration;
	    else if (n < 0)
		n = count_reg[-n];

	    for (i=0; i < count; i++) {
		switch (p->u.math.operator) {
		case '+':
		    count_reg[p->u.math.reg_no] += n;
		    break;
		case '-':
		    count_reg[p->u.math.reg_no] -= n;
		    break;
		case '*':
		    count_reg[p->u.math.reg_no] *= n;
		    break;
		case '/':
		    count_reg[p->u.math.reg_no] /= n;
		    break;
		case '%':
		    count_reg[p->u.math.reg_no] %= n;
		    break;
		case '=':
		    count_reg[p->u.math.reg_no] = n;
		    break;
		case 'P':
		    /* Special case: null operator, w/ side effect of
		     * printing register.
		     */
		    {
		    char buf[100];
		    (void) sprintf(buf, "%ld ", count_reg[p->u.math.reg_no]);
		    OUTPUTSTR(buf);
		    }
		    break;
		default:
		    (void) fprintf(stderr,
			    "%s: unknown math operator `%c' in viz().\n",
			    prog, p->u.math.operator);
		    exit(1);
		}
	    }
	    /***
		(void) fprintf(stderr, "data_offset = %d Register %c = %d\n",
		*data_offset, p->u.math.reg_no, count_reg[p->u.math.reg_no]);
	     ***/
	    break;
	default:
	    (void) fprintf(stderr, "%s: unknown member type %d in viz().\n",
							prog, p->type);
	    exit(1);
	    break;
	}
    }
}

void
viz_text(count, size, ichar, fmt, reg_no)
long count;	/* negative counts process data forever */
int size;
int ichar;
char *fmt;
int reg_no;
{
    int n;
    register int i;
    register unsigned char c;
    register ALLTYPE u;
    register l;
    register last_was_backslash_0=0;
    ALLTYPE t;

    char item_buf[1000];	/* output buffer for each non-ascii item */

    set_mode('a');

    if (fmt != NULL) {
	/* User-assigned format */
	for (; count && adj_count(count) &&
			(n = read_data(size, ichar, count, (char **) &t.uc));
				    ichar == 'Z' ? count-- : (count -= n)) {
	    if (ichar == 'Z') {
		/* Treat it as a string */
		(void) sprintf(item_buf, fmt, t.c);
		l = strlen(item_buf);
		need(l);
		line.avail -= l;
		(void) puts(item_buf);
		*data_offset += n;
	    } else {
		/* Treat it as a bunch of characters */
		u.uc = t.uc;
		for (i = n; i != 0; --i, (*data_offset)++ ) {
		    (void) sprintf(item_buf, fmt, *u.uc++);
		    l = strlen(item_buf);
		    need(l);
		    line.avail -= l;
		    (void) puts(item_buf);
		}
	    }
	}

    } else {
    /* Use built-in rules for outputting data */

    for (; count && adj_count(count) &&
		(n = read_data(size, ichar, count, (char **) &t.uc));
				    ichar == 'Z' ? count-- : (count -= n)) {
	u.uc = t.uc;
	for (i = n; i != 0; --i, (*data_offset)++ ) {
	    c = *u.uc++;
	    if (isascii(c) &&  isprint(c)) {
		if (c == '\\' || c == '^') {
		    need(2);
		    (void) putchar('\\');
		    (void) putchar(c);
		    line.avail -= 2;
		    last_was_backslash_0 = 0;
		} else {
		    if (last_was_backslash_0 && isdigit(c)) {
			/* Start newline to distinguish "\03" from "\0" "3" */
			do_newline();
		    }
		    need(1);
		    (void) putchar(c);
		    line.avail--;
		    last_was_backslash_0 = 0;
		}
	    } else {
		/* Non-printing.  N.B.  Can't take the common putchar('\\')
		 * lines out to the top, because if we print a '\\', and then
		 * find we really need four characters, it's too late to
		 * take back the already-printed backslash.
		 */
		need(2);
		last_was_backslash_0 = 0;
		switch (c) {
		case '\0':
		    (void) putchar('\\');
		    (void) putchar('0');
		    last_was_backslash_0 = 1;
		    line.avail -= 2;
		    break;
#ifdef __STDC__
		case '\a':
		    (void) putchar('\\');
		    (void) putchar('a');
		    line.avail -= 2;
		    break;
#endif
		case '\b':
		    (void) putchar('\\');
		    (void) putchar('b');
		    line.avail -= 2;
		    break;
		case '\f':
		    (void) putchar('\\');
		    (void) putchar('f');
		    line.avail -= 2;
		    break;
		case '\n':
		    (void) putchar('\\');
		    (void) putchar('n');
		    line.avail -= 2;
		    (*data_offset)++;
		    do_newline();
		    set_mode('a');
		    (*data_offset)--;
		    break;
		case '\r':
		    (void) putchar('\\');
		    (void) putchar('r');
		    line.avail -= 2;
		    break;
		case '\t':
		    (void) putchar('\\');
		    (void) putchar('t');
		    line.avail -= 2;
		    break;
#ifdef __STDC__
		case '\v':
		    (void) putchar('\\');
		    (void) putchar('v');
		    line.avail -= 2;
		    break;
#endif
		default:
#ifndef EBCDIC
		    if (c <= ' ') {
			(void) putchar('^');
			(void) putchar(c + 'A' - 1);
			line.avail -= 2;
			break;
		    } else
#endif
		    {
			need(4);
			(void) sprintf(item_buf, "\\%3.3o", (int) c);
			(void) fputs(item_buf, stdout);
			line.avail -= 4;
		    }
		    break;
		}
	    }
	}
    }
    } /* endif (fmt == NULL) */

    /* Store the count away. */
    count_reg[reg_no ? reg_no : '#'] = (ichar != 'Z') ? (long) c : n;
}

void
viz_numeric(count, size, ichar, ochar, fmt, reg_no)
long count;	/* negative counts process data forever */
int size;
int ichar;
int ochar;
char *fmt;
int reg_no;
{
    register int i;
    int n;
    ALLTYPE u;
    long lastv;

    char item_buf[100];	/* output buffer for each item */

    if (ochar == 'c')
	set_mode('c');
    else
	set_mode(ichar);

    switch (ichar) {
    case 'C':
    case 'Z':
	/* chars */
	if (ochar == 'c') {
	    for (; count && adj_count(count) &&
			(n=read_data(size, ichar, count, (char **) &u.uc));
		    ichar == 'Z' ? count-- : (count -= n)) {

		for (i = n; i != 0; i--, (*data_offset)++) {
		    if (isprint(*u.uc) && !isspace(*u.uc)) {
			(void) sprintf(item_buf, "   %c ", (int) *u.uc++);
		    } else if (*u.uc == '\0') {
			(void) sprintf(item_buf, "  \\0 ", (int) *u.uc++);
		    } else {
			(void) sprintf(item_buf, "\\%3.3o ", (int) *u.uc++);
		    }
		    OUTPUTSTR(item_buf);
		}
	    }
	    if (ichar != 'Z')
		lastv = (long) *(u.uc - 1);
	    else
		lastv = n;

	} else if (ochar == 'b') {
	    /* print in binary */
	    for (; count && adj_count(count) &&
			(n = read_data(size, ichar, count, &u.uc));
		    ichar == 'Z' ? count-- : (count -= n)) {
		for (i = n; i != 0; i--, (*data_offset)++) {
		    /* register unsigned uc = *u.uc++;
		    fprintf(stderr, "uc = %d\n", uc);
		    */
		    (void) sprintf(item_buf, fmt, binary[*u.uc++]);
		    OUTPUTSTR(item_buf);
		}
	    }
	    if (ichar != 'Z')
		lastv = (long) *(u.uc - 1);
	    else
		lastv = n;

	} else if (ochar == 'd') {
	    /* plain char */
	    for (; count && adj_count(count) &&
			(n = read_data(size, ichar, count, &u.c));
		    ichar == 'Z' ? count-- : (count -= n)) {
		for (i = n; i != 0; i--, (*data_offset)++) {
		    (void) sprintf(item_buf, fmt, (int) *u.c++);
		    OUTPUTSTR(item_buf);
		}
	    }
	    if (ichar != 'Z')
		lastv = (long) *(u.c - 1);
	    else
		lastv = n;

	} else {
	    /* unsigned char */
	    for (;count && adj_count(count) &&
			(n = read_data(size, ichar, count, (char **) &u.uc));
		    ichar == 'Z' ? count-- : (count -= n)) {
		for (i = n; i != 0; i--, (*data_offset)++) {
		    (void) sprintf(item_buf, fmt, (unsigned int) *u.uc++);
		    OUTPUTSTR(item_buf);
		}
	    }
	    if (ichar != 'Z')
		lastv = (long) *(u.uc - 1);
	    else
		lastv = n;
	}
	break;
    case 'S':
	/* shorts */
	if (ochar == 'd') {
	    /* signed short */
	    for (; count && adj_count(count) &&
			(n = read_data(size, ichar, count, (char **) &u.s));
		    count -= n) {
		for (i = n; i != 0; i--, *data_offset += sizeof(short)) {
		    (void) sprintf(item_buf, fmt, (int) *u.s++);
		    OUTPUTSTR(item_buf);
		}
	    }
	    lastv = (long) *(u.s - 1);

	} else if (ochar == 'b') {
	    /* output in binary representation */
	    for (; count && adj_count(count) &&
			(n = read_data(size, ichar, count, &u.c));
		    count -= n) {
		for (i = n; i != 0; i--, *data_offset += sizeof(short)) {
		    register unsigned short us = *u.us++;
		    (void) sprintf(item_buf, fmt
#if (L_SHORT > 7)
						, binary[(us >> 56) & 0377]
#endif
#if (L_SHORT > 6)
						, binary[(us >> 48) & 0377]
#endif
#if (L_SHORT > 5)
						, binary[(us >> 40) & 0377]
#endif
#if (L_SHORT > 4)
						, binary[(us >> 32) & 0377]
#endif
#if (L_SHORT > 3)
						, binary[(us >> 24) & 0377]
#endif
#if (L_SHORT > 2)
						, binary[(us >> 16) & 0377]
#endif
						, binary[(us >> 8) & 0377]
						, binary[us & 0377]
					);
		    OUTPUTSTR(item_buf);
		}
	    }
	    lastv = (long) *(u.us - 1);

	} else {
	    /* unsigned short */
	    for (;count && adj_count(count) &&
			(n = read_data(size, ichar, count, (char **) &u.us));
		    count -= n) {
		for (i=n; i != 0; i--, *data_offset += sizeof(unsigned short)) {
		    (void) sprintf(item_buf, fmt, (unsigned int) *u.us++);
		    OUTPUTSTR(item_buf);
		}
	    }
	    lastv = (long) *(u.us - 1);
	}
	break;
    case 'I':
	/* ints */
	if (ochar == 'd') {
	    /* signed int */
	    for (;count && adj_count(count) &&
			(n = read_data(size, ichar, count, (char **) &u.i));
		    count -= n) {
		for (i = n; i != 0; i--, *data_offset += sizeof(int)) {
		    (void) sprintf(item_buf, fmt, *u.i++);
		    OUTPUTSTR(item_buf);
		}
	    }
	    lastv = (long) *(u.i - 1);

	} else if (ochar == 'b') {
	    /* output in binary representation */
	    for (; count && adj_count(count) &&
			(n = read_data(size, ichar, count, &u.c));
		    count -= n) {
		for (i = n; i != 0; i--, *data_offset += sizeof(int)) {
		    register unsigned int ui = *u.ui++;
		    (void) sprintf(item_buf, fmt
#if (L_INT > 7)
						, binary[(ui >> 56) & 0377]
#endif
#if (L_INT > 6)
						, binary[(ui >> 48) & 0377]
#endif
#if (L_INT > 5)
						, binary[(ui >> 40) & 0377]
#endif
#if (L_INT > 4)
						, binary[(ui >> 32) & 0377]
#endif
#if (L_INT > 3)
						, binary[(ui >> 24) & 0377]
#endif
#if (L_INT > 2)
						, binary[(ui >> 16) & 0377]
#endif
						, binary[(ui >> 8) & 0377]
		    				, binary[ui & 0377]
					);
		    OUTPUTSTR(item_buf);
		}
	    }
	    lastv = (long) *(u.i - 1);

	} else {
	    /* unsigned int */
	    for (;count && adj_count(count) &&
			(n = read_data(size, ichar, count, (char **) &u.ui));
		    count -= n) {
		for (i = n; i != 0; i--, *data_offset += sizeof(unsigned int)) {
		    (void) sprintf(item_buf, fmt, *u.ui++);
		    OUTPUTSTR(item_buf);
		}
	    }
	    lastv = (long) *(u.ui - 1);
	}
	break;
    case 'L':
	/* longs */
	if (ochar == 'd') {
	    /* signed long */
	    for (;count && adj_count(count) &&
			(n = read_data(size, ichar, count, (char **) &u.l));
		    count -= n) {
		for (i = n; i != 0; i--, *data_offset += sizeof(long)) {
		    (void) sprintf(item_buf, fmt, *u.l++);
		    OUTPUTSTR(item_buf);
		}
	    }
	    lastv = *(u.l - 1);

	} else if (ochar == 'b') {
	    /* output in binary representation */
	    for (; count && adj_count(count) &&
			(n = read_data(size, ichar, count, &u.c));
		    count -= n) {
		for (i = n; i != 0; i--, *data_offset += sizeof(long)) {
		    register unsigned int ul = *u.ul++;
		    (void) sprintf(item_buf, fmt
#if (L_LONG > 7)
						, binary[(ul >> 56) & 0377]
#endif
#if (L_LONG > 6)
						, binary[(ul >> 48) & 0377]
#endif
#if (L_LONG > 5)
						, binary[(ul >> 40) & 0377]
#endif
#if (L_LONG > 4)
						, binary[(ul >> 32) & 0377]
#endif
#if (L_LONG > 3)
						, binary[(ul >> 24) & 0377]
#endif
#if (L_LONG > 2)
						, binary[(ul >> 16) & 0377]
#endif
						, binary[(ul >> 8) & 0377]
		    				, binary[ul & 0377]
					);
		    OUTPUTSTR(item_buf);
		}
	    }
	    lastv = (long) *(u.ul - 1);

	} else {
	    /* unsigned long */
	    for (;count && adj_count(count) &&
			(n = read_data(size, ichar, count, (char **) &u.ul));
		    count -= n) {
		for (i = n; i != 0; i--, *data_offset += sizeof(unsigned long)) {
		    (void) sprintf(item_buf, fmt, *u.ul++);
		    OUTPUTSTR(item_buf);
		}
	    }
	    lastv = (long) *(u.ul - 1);
	}
	break;
	/* NOTREACHED */
    break;
    case 'F':
	/* floats */
	for (; count && adj_count(count) &&
			(n = read_data(size, ichar, count, (char **) &u.f));
		count -= n) {
	    for (i = n; i != 0; i--, *data_offset += sizeof(float)) {
		(void) sprintf(item_buf, fmt, *u.f++);
		OUTPUTSTR(item_buf);
	    }
	}
	lastv = 0;
	break;
    case 'D':
	/* doubles */
	for (; count && adj_count(count) &&
			(n = read_data(size, ichar, count, (char **) &u.d));
		count -= n) {
	    for (i = n; i != 0; i--, *data_offset += sizeof(double)) {
		(void) sprintf(item_buf, fmt, *u.d++);
		OUTPUTSTR(item_buf);
	    }
	}
	lastv = 0;
	break;
    default:
	(void) fprintf(stderr,
	    "%s: ichar = `%c', ochar = `%c' in viz_numeric()!\n",
	    prog, ichar, ochar);
	exit(1);
	break;
    }
    count_reg[reg_no ? reg_no : '#'] = lastv;
}
