/* valaassignment.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaExpressionPrivate ValaExpressionPrivate;

#define VALA_TYPE_ASSIGNMENT (vala_assignment_get_type ())
#define VALA_ASSIGNMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ASSIGNMENT, ValaAssignment))
#define VALA_ASSIGNMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ASSIGNMENT, ValaAssignmentClass))
#define VALA_IS_ASSIGNMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ASSIGNMENT))
#define VALA_IS_ASSIGNMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ASSIGNMENT))
#define VALA_ASSIGNMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ASSIGNMENT, ValaAssignmentClass))

typedef struct _ValaAssignment ValaAssignment;
typedef struct _ValaAssignmentClass ValaAssignmentClass;
typedef struct _ValaAssignmentPrivate ValaAssignmentPrivate;

#define VALA_TYPE_ASSIGNMENT_OPERATOR (vala_assignment_operator_get_type ())

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_MEMBER_ACCESS (vala_member_access_get_type ())
#define VALA_MEMBER_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccess))
#define VALA_MEMBER_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessClass))
#define VALA_IS_MEMBER_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER_ACCESS))
#define VALA_IS_MEMBER_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER_ACCESS))
#define VALA_MEMBER_ACCESS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessClass))

typedef struct _ValaMemberAccess ValaMemberAccess;
typedef struct _ValaMemberAccessClass ValaMemberAccessClass;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;

#define VALA_TYPE_SIGNAL (vala_signal_get_type ())
#define VALA_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SIGNAL, ValaSignal))
#define VALA_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SIGNAL, ValaSignalClass))
#define VALA_IS_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SIGNAL))
#define VALA_IS_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SIGNAL))
#define VALA_SIGNAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SIGNAL, ValaSignalClass))

typedef struct _ValaSignal ValaSignal;
typedef struct _ValaSignalClass ValaSignalClass;

#define VALA_TYPE_PROPERTY (vala_property_get_type ())
#define VALA_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_PROPERTY, ValaProperty))
#define VALA_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_PROPERTY, ValaPropertyClass))
#define VALA_IS_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_PROPERTY))
#define VALA_IS_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_PROPERTY))
#define VALA_PROPERTY_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_PROPERTY, ValaPropertyClass))

typedef struct _ValaProperty ValaProperty;
typedef struct _ValaPropertyClass ValaPropertyClass;

#define VALA_TYPE_DYNAMIC_PROPERTY (vala_dynamic_property_get_type ())
#define VALA_DYNAMIC_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DYNAMIC_PROPERTY, ValaDynamicProperty))
#define VALA_DYNAMIC_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DYNAMIC_PROPERTY, ValaDynamicPropertyClass))
#define VALA_IS_DYNAMIC_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DYNAMIC_PROPERTY))
#define VALA_IS_DYNAMIC_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DYNAMIC_PROPERTY))
#define VALA_DYNAMIC_PROPERTY_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DYNAMIC_PROPERTY, ValaDynamicPropertyClass))

typedef struct _ValaDynamicProperty ValaDynamicProperty;
typedef struct _ValaDynamicPropertyClass ValaDynamicPropertyClass;

#define VALA_TYPE_DYNAMIC_SIGNAL (vala_dynamic_signal_get_type ())
#define VALA_DYNAMIC_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DYNAMIC_SIGNAL, ValaDynamicSignal))
#define VALA_DYNAMIC_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DYNAMIC_SIGNAL, ValaDynamicSignalClass))
#define VALA_IS_DYNAMIC_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DYNAMIC_SIGNAL))
#define VALA_IS_DYNAMIC_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DYNAMIC_SIGNAL))
#define VALA_DYNAMIC_SIGNAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DYNAMIC_SIGNAL, ValaDynamicSignalClass))

typedef struct _ValaDynamicSignal ValaDynamicSignal;
typedef struct _ValaDynamicSignalClass ValaDynamicSignalClass;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_DELEGATE (vala_delegate_get_type ())
#define VALA_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE, ValaDelegate))
#define VALA_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE, ValaDelegateClass))
#define VALA_IS_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE))
#define VALA_IS_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE))
#define VALA_DELEGATE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE, ValaDelegateClass))

typedef struct _ValaDelegate ValaDelegate;
typedef struct _ValaDelegateClass ValaDelegateClass;

#define VALA_TYPE_DELEGATE_TYPE (vala_delegate_type_get_type ())
#define VALA_DELEGATE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE_TYPE, ValaDelegateType))
#define VALA_DELEGATE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE_TYPE, ValaDelegateTypeClass))
#define VALA_IS_DELEGATE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE_TYPE))
#define VALA_IS_DELEGATE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE_TYPE))
#define VALA_DELEGATE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE_TYPE, ValaDelegateTypeClass))

typedef struct _ValaDelegateType ValaDelegateType;
typedef struct _ValaDelegateTypeClass ValaDelegateTypeClass;

#define VALA_TYPE_ELEMENT_ACCESS (vala_element_access_get_type ())
#define VALA_ELEMENT_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ELEMENT_ACCESS, ValaElementAccess))
#define VALA_ELEMENT_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ELEMENT_ACCESS, ValaElementAccessClass))
#define VALA_IS_ELEMENT_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ELEMENT_ACCESS))
#define VALA_IS_ELEMENT_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ELEMENT_ACCESS))
#define VALA_ELEMENT_ACCESS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ELEMENT_ACCESS, ValaElementAccessClass))

typedef struct _ValaElementAccess ValaElementAccess;
typedef struct _ValaElementAccessClass ValaElementAccessClass;

#define VALA_TYPE_METHOD (vala_method_get_type ())
#define VALA_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_METHOD, ValaMethod))
#define VALA_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_METHOD, ValaMethodClass))
#define VALA_IS_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_METHOD))
#define VALA_IS_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_METHOD))
#define VALA_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_METHOD, ValaMethodClass))

typedef struct _ValaMethod ValaMethod;
typedef struct _ValaMethodClass ValaMethodClass;

#define VALA_TYPE_METHOD_CALL (vala_method_call_get_type ())
#define VALA_METHOD_CALL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_METHOD_CALL, ValaMethodCall))
#define VALA_METHOD_CALL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_METHOD_CALL, ValaMethodCallClass))
#define VALA_IS_METHOD_CALL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_METHOD_CALL))
#define VALA_IS_METHOD_CALL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_METHOD_CALL))
#define VALA_METHOD_CALL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_METHOD_CALL, ValaMethodCallClass))

typedef struct _ValaMethodCall ValaMethodCall;
typedef struct _ValaMethodCallClass ValaMethodCallClass;

#define VALA_TYPE_POINTER_INDIRECTION (vala_pointer_indirection_get_type ())
#define VALA_POINTER_INDIRECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_POINTER_INDIRECTION, ValaPointerIndirection))
#define VALA_POINTER_INDIRECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_POINTER_INDIRECTION, ValaPointerIndirectionClass))
#define VALA_IS_POINTER_INDIRECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_POINTER_INDIRECTION))
#define VALA_IS_POINTER_INDIRECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_POINTER_INDIRECTION))
#define VALA_POINTER_INDIRECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_POINTER_INDIRECTION, ValaPointerIndirectionClass))

typedef struct _ValaPointerIndirection ValaPointerIndirection;
typedef struct _ValaPointerIndirectionClass ValaPointerIndirectionClass;

#define VALA_TYPE_BINARY_OPERATOR (vala_binary_operator_get_type ())

#define VALA_TYPE_BINARY_EXPRESSION (vala_binary_expression_get_type ())
#define VALA_BINARY_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_BINARY_EXPRESSION, ValaBinaryExpression))
#define VALA_BINARY_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_BINARY_EXPRESSION, ValaBinaryExpressionClass))
#define VALA_IS_BINARY_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_BINARY_EXPRESSION))
#define VALA_IS_BINARY_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_BINARY_EXPRESSION))
#define VALA_BINARY_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_BINARY_EXPRESSION, ValaBinaryExpressionClass))

typedef struct _ValaBinaryExpression ValaBinaryExpression;
typedef struct _ValaBinaryExpressionClass ValaBinaryExpressionClass;

#define VALA_TYPE_FORMAL_PARAMETER (vala_formal_parameter_get_type ())
#define VALA_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameter))
#define VALA_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))
#define VALA_IS_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_IS_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_FORMAL_PARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))

typedef struct _ValaFormalParameter ValaFormalParameter;
typedef struct _ValaFormalParameterClass ValaFormalParameterClass;

#define VALA_TYPE_OBJECT_TYPE_SYMBOL (vala_object_type_symbol_get_type ())
#define VALA_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbol))
#define VALA_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))
#define VALA_IS_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_IS_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_OBJECT_TYPE_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))

typedef struct _ValaObjectTypeSymbol ValaObjectTypeSymbol;
typedef struct _ValaObjectTypeSymbolClass ValaObjectTypeSymbolClass;

#define VALA_TYPE_REFERENCE_TYPE (vala_reference_type_get_type ())
#define VALA_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceType))
#define VALA_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))
#define VALA_IS_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_REFERENCE_TYPE))
#define VALA_IS_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_REFERENCE_TYPE))
#define VALA_REFERENCE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))

typedef struct _ValaReferenceType ValaReferenceType;
typedef struct _ValaReferenceTypeClass ValaReferenceTypeClass;

#define VALA_TYPE_OBJECT_TYPE (vala_object_type_get_type ())
#define VALA_OBJECT_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_OBJECT_TYPE, ValaObjectType))
#define VALA_OBJECT_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_OBJECT_TYPE, ValaObjectTypeClass))
#define VALA_IS_OBJECT_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_OBJECT_TYPE))
#define VALA_IS_OBJECT_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_OBJECT_TYPE))
#define VALA_OBJECT_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_OBJECT_TYPE, ValaObjectTypeClass))

typedef struct _ValaObjectType ValaObjectType;
typedef struct _ValaObjectTypeClass ValaObjectTypeClass;

#define VALA_TYPE_PROPERTY_ACCESSOR (vala_property_accessor_get_type ())
#define VALA_PROPERTY_ACCESSOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessor))
#define VALA_PROPERTY_ACCESSOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessorClass))
#define VALA_IS_PROPERTY_ACCESSOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_PROPERTY_ACCESSOR))
#define VALA_IS_PROPERTY_ACCESSOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_PROPERTY_ACCESSOR))
#define VALA_PROPERTY_ACCESSOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessorClass))

typedef struct _ValaPropertyAccessor ValaPropertyAccessor;
typedef struct _ValaPropertyAccessorClass ValaPropertyAccessorClass;

#define VALA_TYPE_CREATION_METHOD (vala_creation_method_get_type ())
#define VALA_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethod))
#define VALA_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))
#define VALA_IS_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CREATION_METHOD))
#define VALA_IS_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CREATION_METHOD))
#define VALA_CREATION_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))

typedef struct _ValaCreationMethod ValaCreationMethod;
typedef struct _ValaCreationMethodClass ValaCreationMethodClass;

#define VALA_TYPE_FIELD (vala_field_get_type ())
#define VALA_FIELD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FIELD, ValaField))
#define VALA_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FIELD, ValaFieldClass))
#define VALA_IS_FIELD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FIELD))
#define VALA_IS_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FIELD))
#define VALA_FIELD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FIELD, ValaFieldClass))

typedef struct _ValaField ValaField;
typedef struct _ValaFieldClass ValaFieldClass;

#define VALA_TYPE_POINTER_TYPE (vala_pointer_type_get_type ())
#define VALA_POINTER_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_POINTER_TYPE, ValaPointerType))
#define VALA_POINTER_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_POINTER_TYPE, ValaPointerTypeClass))
#define VALA_IS_POINTER_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_POINTER_TYPE))
#define VALA_IS_POINTER_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_POINTER_TYPE))
#define VALA_POINTER_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_POINTER_TYPE, ValaPointerTypeClass))

typedef struct _ValaPointerType ValaPointerType;
typedef struct _ValaPointerTypeClass ValaPointerTypeClass;

#define VALA_TYPE_ARRAY_TYPE (vala_array_type_get_type ())
#define VALA_ARRAY_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ARRAY_TYPE, ValaArrayType))
#define VALA_ARRAY_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ARRAY_TYPE, ValaArrayTypeClass))
#define VALA_IS_ARRAY_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ARRAY_TYPE))
#define VALA_IS_ARRAY_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ARRAY_TYPE))
#define VALA_ARRAY_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ARRAY_TYPE, ValaArrayTypeClass))

typedef struct _ValaArrayType ValaArrayType;
typedef struct _ValaArrayTypeClass ValaArrayTypeClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Base class for all code nodes that might be used as an expression.
 */
struct _ValaExpression {
	ValaCodeNode parent_instance;
	ValaExpressionPrivate * priv;
	GeeArrayList* temp_vars;
};

struct _ValaExpressionClass {
	ValaCodeNodeClass parent_class;
	gboolean (*is_constant) (ValaExpression* self);
	gboolean (*is_pure) (ValaExpression* self);
	gboolean (*is_non_null) (ValaExpression* self);
};

/**
 * Represents an assignment expression in the source code.
 *
 * Supports =, |=, &=, ^=, +=, -=, *=, /=, %=, <<=, >>=.
 */
struct _ValaAssignment {
	ValaExpression parent_instance;
	ValaAssignmentPrivate * priv;
};

struct _ValaAssignmentClass {
	ValaExpressionClass parent_class;
};

typedef enum  {
	VALA_ASSIGNMENT_OPERATOR_NONE,
	VALA_ASSIGNMENT_OPERATOR_SIMPLE,
	VALA_ASSIGNMENT_OPERATOR_BITWISE_OR,
	VALA_ASSIGNMENT_OPERATOR_BITWISE_AND,
	VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR,
	VALA_ASSIGNMENT_OPERATOR_ADD,
	VALA_ASSIGNMENT_OPERATOR_SUB,
	VALA_ASSIGNMENT_OPERATOR_MUL,
	VALA_ASSIGNMENT_OPERATOR_DIV,
	VALA_ASSIGNMENT_OPERATOR_PERCENT,
	VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT,
	VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT
} ValaAssignmentOperator;

struct _ValaAssignmentPrivate {
	ValaAssignmentOperator _operator;
	ValaExpression* _left;
	ValaExpression* _right;
};

typedef enum  {
	VALA_BINARY_OPERATOR_NONE,
	VALA_BINARY_OPERATOR_PLUS,
	VALA_BINARY_OPERATOR_MINUS,
	VALA_BINARY_OPERATOR_MUL,
	VALA_BINARY_OPERATOR_DIV,
	VALA_BINARY_OPERATOR_MOD,
	VALA_BINARY_OPERATOR_SHIFT_LEFT,
	VALA_BINARY_OPERATOR_SHIFT_RIGHT,
	VALA_BINARY_OPERATOR_LESS_THAN,
	VALA_BINARY_OPERATOR_GREATER_THAN,
	VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL,
	VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL,
	VALA_BINARY_OPERATOR_EQUALITY,
	VALA_BINARY_OPERATOR_INEQUALITY,
	VALA_BINARY_OPERATOR_BITWISE_AND,
	VALA_BINARY_OPERATOR_BITWISE_OR,
	VALA_BINARY_OPERATOR_BITWISE_XOR,
	VALA_BINARY_OPERATOR_AND,
	VALA_BINARY_OPERATOR_OR,
	VALA_BINARY_OPERATOR_IN
} ValaBinaryOperator;



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_assignment_get_type (void);
GType vala_assignment_operator_get_type (void);
#define VALA_ASSIGNMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ASSIGNMENT, ValaAssignmentPrivate))
enum  {
	VALA_ASSIGNMENT_DUMMY_PROPERTY
};
void vala_assignment_set_right (ValaAssignment* self, ValaExpression* value);
void vala_assignment_set_operator (ValaAssignment* self, ValaAssignmentOperator value);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
void vala_code_node_set_source_reference (ValaCodeNode* self, ValaSourceReference* value);
void vala_assignment_set_left (ValaAssignment* self, ValaExpression* value);
ValaAssignment* vala_assignment_new (ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference);
ValaAssignment* vala_assignment_construct (GType object_type, ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference);
ValaAssignment* vala_assignment_new (ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference);
void vala_code_visitor_visit_assignment (ValaCodeVisitor* self, ValaAssignment* a);
void vala_code_visitor_visit_expression (ValaCodeVisitor* self, ValaExpression* expr);
static void vala_assignment_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaExpression* vala_assignment_get_left (ValaAssignment* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
ValaExpression* vala_assignment_get_right (ValaAssignment* self);
static void vala_assignment_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_assignment_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_assignment_real_is_pure (ValaExpression* base);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
void vala_expression_set_lvalue (ValaExpression* self, gboolean value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
GType vala_member_access_get_type (void);
ValaSymbol* vala_expression_get_symbol_reference (ValaExpression* self);
GType vala_member_get_type (void);
GType vala_signal_get_type (void);
GType vala_property_get_type (void);
GType vala_dynamic_property_get_type (void);
ValaDataType* vala_expression_get_value_type (ValaExpression* self);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
gboolean vala_member_access_get_prototype_access (ValaMemberAccess* self);
char* vala_symbol_get_full_name (ValaSymbol* self);
GType vala_dynamic_signal_get_type (void);
GType vala_typesymbol_get_type (void);
GType vala_delegate_get_type (void);
ValaDelegate* vala_signal_get_delegate (ValaSignal* self, ValaDataType* sender_type, ValaCodeNode* node_reference);
ValaExpression* vala_member_access_get_inner (ValaMemberAccess* self);
ValaDelegateType* vala_delegate_type_new (ValaDelegate* delegate_symbol);
ValaDelegateType* vala_delegate_type_construct (GType object_type, ValaDelegate* delegate_symbol);
GType vala_delegate_type_get_type (void);
void vala_expression_set_target_type (ValaExpression* self, ValaDataType* value);
ValaDataType* vala_expression_get_formal_value_type (ValaExpression* self);
void vala_expression_set_formal_target_type (ValaExpression* self, ValaDataType* value);
GType vala_element_access_get_type (void);
ValaExpression* vala_element_access_get_container (ValaElementAccess* self);
ValaSymbol* vala_data_type_get_member (ValaDataType* self, const char* member_name);
GType vala_method_get_type (void);
ValaMemberAccess* vala_member_access_new (ValaExpression* inner, const char* member_name, ValaSourceReference* source_reference);
ValaMemberAccess* vala_member_access_construct (GType object_type, ValaExpression* inner, const char* member_name, ValaSourceReference* source_reference);
ValaMethodCall* vala_method_call_new (ValaExpression* call, ValaSourceReference* source_reference);
ValaMethodCall* vala_method_call_construct (GType object_type, ValaExpression* call, ValaSourceReference* source_reference);
GType vala_method_call_get_type (void);
GeeList* vala_element_access_get_indices (ValaElementAccess* self);
void vala_method_call_add_argument (ValaMethodCall* self, ValaExpression* arg);
ValaCodeNode* vala_code_node_get_parent_node (ValaCodeNode* self);
void vala_code_node_replace_expression (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
GType vala_pointer_indirection_get_type (void);
ValaAssignmentOperator vala_assignment_get_operator (ValaAssignment* self);
const char* vala_member_access_get_member_name (ValaMemberAccess* self);
GType vala_binary_operator_get_type (void);
ValaBinaryExpression* vala_binary_expression_new (ValaBinaryOperator op, ValaExpression* _left, ValaExpression* _right, ValaSourceReference* source);
ValaBinaryExpression* vala_binary_expression_construct (GType object_type, ValaBinaryOperator op, ValaExpression* _left, ValaExpression* _right, ValaSourceReference* source);
GType vala_binary_expression_get_type (void);
ValaDataType* vala_expression_get_target_type (ValaExpression* self);
ValaDataType* vala_data_type_copy (ValaDataType* self);
void vala_data_type_set_value_owned (ValaDataType* self, gboolean value);
void vala_binary_expression_set_operator (ValaBinaryExpression* self, ValaBinaryOperator value);
ValaExpression* vala_dynamic_signal_get_handler (ValaDynamicSignal* self);
GType vala_formal_parameter_get_type (void);
GeeList* vala_data_type_get_parameters (ValaDataType* self);
void vala_signal_add_parameter (ValaSignal* self, ValaFormalParameter* param);
ValaFormalParameter* vala_formal_parameter_copy (ValaFormalParameter* self);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
GType vala_object_type_symbol_get_type (void);
ValaObjectType* vala_object_type_new (ValaObjectTypeSymbol* type_symbol);
ValaObjectType* vala_object_type_construct (GType object_type, ValaObjectTypeSymbol* type_symbol);
GType vala_reference_type_get_type (void);
GType vala_object_type_get_type (void);
gboolean vala_data_type_compatible (ValaDataType* self, ValaDataType* target_type);
char* vala_code_node_to_string (ValaCodeNode* self);
ValaDelegate* vala_delegate_type_get_delegate_symbol (ValaDelegateType* self);
char* vala_delegate_get_prototype_string (ValaDelegate* self, const char* name);
const char* vala_symbol_get_name (ValaSymbol* self);
void vala_property_set_property_type (ValaProperty* self, ValaDataType* value);
ValaDataType* vala_property_get_property_type (ValaProperty* self);
void vala_expression_set_value_type (ValaExpression* self, ValaDataType* value);
GType vala_property_accessor_get_type (void);
ValaPropertyAccessor* vala_property_get_set_accessor (ValaProperty* self);
gboolean vala_property_accessor_get_writable (ValaPropertyAccessor* self);
ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self);
GType vala_creation_method_get_type (void);
gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self);
ValaDataType* vala_local_variable_get_variable_type (ValaLocalVariable* self);
gboolean vala_delegate_matches_method (ValaDelegate* self, ValaMethod* m);
GType vala_field_get_type (void);
ValaDataType* vala_field_get_field_type (ValaField* self);
gboolean vala_data_type_is_disposable (ValaDataType* self);
GType vala_pointer_type_get_type (void);
gboolean vala_data_type_get_value_owned (ValaDataType* self);
GType vala_array_type_get_type (void);
ValaDataType* vala_array_type_get_element_type (ValaArrayType* self);
GeeList* vala_data_type_get_type_arguments (ValaDataType* self);
void vala_code_node_add_error_types (ValaCodeNode* self, GeeList* error_types);
GeeList* vala_code_node_get_error_types (ValaCodeNode* self);
static gboolean vala_assignment_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
void vala_code_node_get_defined_variables (ValaCodeNode* self, GeeCollection* collection);
static void vala_assignment_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
void vala_code_node_get_used_variables (ValaCodeNode* self, GeeCollection* collection);
static void vala_assignment_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
static gpointer vala_assignment_parent_class = NULL;
static void vala_assignment_finalize (ValaCodeNode* obj);



/**
 * Creates a new assignment.
 *
 * @param left             left hand side
 * @param operator         assignment operator
 * @param right            right hand side
 * @param source_reference reference to source code
 * @return                 newly created assignment
 */
ValaAssignment* vala_assignment_construct (GType object_type, ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference) {
	ValaAssignment* self;
	g_return_val_if_fail (left != NULL, NULL);
	g_return_val_if_fail (right != NULL, NULL);
	self = (ValaAssignment*) g_type_create_instance (object_type);
	vala_assignment_set_right (self, right);
	vala_assignment_set_operator (self, operator);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_assignment_set_left (self, left);
	return self;
}


ValaAssignment* vala_assignment_new (ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference) {
	return vala_assignment_construct (VALA_TYPE_ASSIGNMENT, left, right, operator, source_reference);
}


static void vala_assignment_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_assignment (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_assignment_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_assignment_get_left (self), visitor);
	vala_code_node_accept ((ValaCodeNode*) vala_assignment_get_right (self), visitor);
}


static void vala_assignment_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_assignment_get_left (self) == old_node) {
		vala_assignment_set_left (self, new_node);
	}
	if (vala_assignment_get_right (self) == old_node) {
		vala_assignment_set_right (self, new_node);
	}
}


static gboolean vala_assignment_real_is_pure (ValaExpression* base) {
	ValaAssignment * self;
	self = (ValaAssignment*) base;
	return FALSE;
}


static gboolean vala_assignment_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaAssignment * self;
	gboolean _tmp29;
	GeeList* _tmp111;
	GeeList* _tmp112;
	self = (ValaAssignment*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_expression_set_lvalue (vala_assignment_get_left (self), TRUE);
	if (!vala_code_node_check ((ValaCodeNode*) vala_assignment_get_left (self), analyzer)) {
		/* skip on error in inner expression*/
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self))) {
		ValaMemberAccess* _tmp2;
		ValaMemberAccess* ma;
		gboolean _tmp3;
		gboolean _tmp4;
		gboolean _tmp9;
		_tmp2 = NULL;
		ma = (_tmp2 = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
		_tmp3 = FALSE;
		_tmp4 = FALSE;
		if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
			_tmp4 = TRUE;
		} else {
			_tmp4 = VALA_IS_DYNAMIC_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma));
		}
		if (!_tmp4) {
			_tmp3 = vala_expression_get_value_type ((ValaExpression*) ma) == NULL;
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			gboolean _tmp5;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "unsupported lvalue in assignment");
			return (_tmp5 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp5);
		}
		if (vala_member_access_get_prototype_access (ma)) {
			char* _tmp7;
			char* _tmp6;
			gboolean _tmp8;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp7 = NULL;
			_tmp6 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp7 = g_strdup_printf ("Access to instance member `%s' denied", _tmp6 = vala_symbol_get_full_name (vala_expression_get_symbol_reference ((ValaExpression*) ma))));
			_tmp7 = (g_free (_tmp7), NULL);
			_tmp6 = (g_free (_tmp6), NULL);
			return (_tmp8 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp8);
		}
		_tmp9 = FALSE;
		if (vala_code_node_get_error ((ValaCodeNode*) ma)) {
			_tmp9 = TRUE;
		} else {
			_tmp9 = vala_expression_get_symbol_reference ((ValaExpression*) ma) == NULL;
		}
		if (_tmp9) {
			gboolean _tmp10;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			/* if no symbol found, skip this check */
			return (_tmp10 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp10);
		}
		if (VALA_IS_DYNAMIC_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
		} else {
			if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
				ValaSignal* _tmp11;
				ValaSignal* sig;
				ValaDelegateType* _tmp13;
				ValaDelegate* _tmp12;
				_tmp11 = NULL;
				sig = (_tmp11 = VALA_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp11 == NULL) ? NULL : vala_code_node_ref (_tmp11));
				_tmp13 = NULL;
				_tmp12 = NULL;
				vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp13 = vala_delegate_type_new (_tmp12 = vala_signal_get_delegate (sig, vala_expression_get_value_type (vala_member_access_get_inner (ma)), (ValaCodeNode*) self))));
				(_tmp13 == NULL) ? NULL : (_tmp13 = (vala_code_node_unref (_tmp13), NULL));
				(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL));
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			} else {
				vala_expression_set_formal_target_type (vala_assignment_get_right (self), vala_expression_get_formal_value_type ((ValaExpression*) ma));
				vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type ((ValaExpression*) ma));
			}
		}
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	} else {
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (self))) {
			ValaElementAccess* _tmp14;
			ValaElementAccess* ea;
			gboolean _tmp15;
			_tmp14 = NULL;
			ea = (_tmp14 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (self)), (_tmp14 == NULL) ? NULL : vala_code_node_ref (_tmp14));
			_tmp15 = FALSE;
			if (VALA_IS_MEMBER_ACCESS (vala_element_access_get_container (ea))) {
				_tmp15 = VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_element_access_get_container (ea)));
			} else {
				_tmp15 = FALSE;
			}
			if (_tmp15) {
				ValaMemberAccess* _tmp16;
				ValaMemberAccess* ma;
				ValaSignal* _tmp17;
				ValaSignal* sig;
				ValaDelegateType* _tmp19;
				ValaDelegate* _tmp18;
				_tmp16 = NULL;
				ma = (_tmp16 = VALA_MEMBER_ACCESS (vala_element_access_get_container (ea)), (_tmp16 == NULL) ? NULL : vala_code_node_ref (_tmp16));
				_tmp17 = NULL;
				sig = (_tmp17 = VALA_SIGNAL (vala_expression_get_symbol_reference (vala_element_access_get_container (ea))), (_tmp17 == NULL) ? NULL : vala_code_node_ref (_tmp17));
				_tmp19 = NULL;
				_tmp18 = NULL;
				vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp19 = vala_delegate_type_new (_tmp18 = vala_signal_get_delegate (sig, vala_expression_get_value_type (vala_member_access_get_inner (ma)), (ValaCodeNode*) self))));
				(_tmp19 == NULL) ? NULL : (_tmp19 = (vala_code_node_unref (_tmp19), NULL));
				(_tmp18 == NULL) ? NULL : (_tmp18 = (vala_code_node_unref (_tmp18), NULL));
				(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			} else {
				ValaSymbol* _tmp20;
				gboolean _tmp21;
				_tmp20 = NULL;
				if ((_tmp21 = VALA_IS_METHOD (_tmp20 = vala_data_type_get_member (vala_expression_get_value_type (vala_element_access_get_container (ea)), "set")), (_tmp20 == NULL) ? NULL : (_tmp20 = (vala_code_node_unref (_tmp20), NULL)), _tmp21)) {
					ValaMemberAccess* _tmp22;
					ValaMethodCall* _tmp23;
					ValaMethodCall* set_call;
					gboolean _tmp26;
					_tmp22 = NULL;
					_tmp23 = NULL;
					set_call = (_tmp23 = vala_method_call_new ((ValaExpression*) (_tmp22 = vala_member_access_new (vala_element_access_get_container (ea), "set", NULL)), NULL), (_tmp22 == NULL) ? NULL : (_tmp22 = (vala_code_node_unref (_tmp22), NULL)), _tmp23);
					{
						GeeList* _tmp24;
						GeeIterator* _tmp25;
						GeeIterator* _e_it;
						_tmp24 = NULL;
						_tmp25 = NULL;
						_e_it = (_tmp25 = gee_iterable_iterator ((GeeIterable*) (_tmp24 = vala_element_access_get_indices (ea))), (_tmp24 == NULL) ? NULL : (_tmp24 = (gee_collection_object_unref (_tmp24), NULL)), _tmp25);
						while (gee_iterator_next (_e_it)) {
							ValaExpression* e;
							e = (ValaExpression*) gee_iterator_get (_e_it);
							vala_method_call_add_argument (set_call, e);
							(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
						}
						(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
					}
					vala_method_call_add_argument (set_call, vala_assignment_get_right (self));
					vala_code_node_replace_expression (vala_code_node_get_parent_node ((ValaCodeNode*) self), (ValaExpression*) self, (ValaExpression*) set_call);
					return (_tmp26 = vala_code_node_check ((ValaCodeNode*) set_call, analyzer), (set_call == NULL) ? NULL : (set_call = (vala_code_node_unref (set_call), NULL)), (ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL)), _tmp26);
				} else {
					vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type (vala_assignment_get_left (self)));
				}
			}
			(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
		} else {
			if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (self))) {
				vala_expression_set_target_type (vala_assignment_get_right (self), vala_expression_get_value_type (vala_assignment_get_left (self)));
			} else {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "unsupported lvalue in assignment");
				return FALSE;
			}
		}
	}
	if (!vala_code_node_check ((ValaCodeNode*) vala_assignment_get_right (self), analyzer)) {
		/* skip on error in inner expression*/
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	_tmp29 = FALSE;
	if (self->priv->_operator != VALA_ASSIGNMENT_OPERATOR_SIMPLE) {
		_tmp29 = VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self));
	} else {
		_tmp29 = FALSE;
	}
	if (_tmp29) {
		ValaMemberAccess* _tmp30;
		ValaMemberAccess* ma;
		/* transform into simple assignment
		 FIXME: only do this if the backend doesn't support
		 the assignment natively*/
		_tmp30 = NULL;
		ma = (_tmp30 = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp30 == NULL) ? NULL : vala_code_node_ref (_tmp30));
		if (!VALA_IS_SIGNAL (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
			ValaMemberAccess* old_value;
			ValaBinaryExpression* bin;
			ValaDataType* _tmp31;
			old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
			bin = vala_binary_expression_new (VALA_BINARY_OPERATOR_PLUS, (ValaExpression*) old_value, vala_assignment_get_right (self), NULL);
			vala_expression_set_target_type ((ValaExpression*) bin, vala_expression_get_target_type (vala_assignment_get_right (self)));
			_tmp31 = NULL;
			vala_expression_set_target_type (vala_assignment_get_right (self), _tmp31 = vala_data_type_copy (vala_expression_get_target_type (vala_assignment_get_right (self))));
			(_tmp31 == NULL) ? NULL : (_tmp31 = (vala_code_node_unref (_tmp31), NULL));
			vala_data_type_set_value_owned (vala_expression_get_target_type (vala_assignment_get_right (self)), FALSE);
			if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_OR) {
				vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_OR);
			} else {
				if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_AND) {
					vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_AND);
				} else {
					if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR) {
						vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_XOR);
					} else {
						if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_ADD) {
							vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_PLUS);
						} else {
							if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SUB) {
								vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MINUS);
							} else {
								if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_MUL) {
									vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MUL);
								} else {
									if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_DIV) {
										vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_DIV);
									} else {
										if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_PERCENT) {
											vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MOD);
										} else {
											if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT) {
												vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_LEFT);
											} else {
												if (self->priv->_operator == VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT) {
													vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_RIGHT);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			vala_assignment_set_right (self, (ValaExpression*) bin);
			vala_code_node_check ((ValaCodeNode*) vala_assignment_get_right (self), analyzer);
			vala_assignment_set_operator (self, VALA_ASSIGNMENT_OPERATOR_SIMPLE);
			(old_value == NULL) ? NULL : (old_value = (vala_code_node_unref (old_value), NULL));
			(bin == NULL) ? NULL : (bin = (vala_code_node_unref (bin), NULL));
		}
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	}
	if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_assignment_get_left (self)))) {
		ValaSignal* _tmp32;
		ValaSignal* sig;
		ValaMethod* _tmp34;
		ValaSymbol* _tmp33;
		ValaMethod* m;
		ValaDynamicSignal* _tmp37;
		ValaSignal* _tmp36;
		ValaDynamicSignal* dynamic_sig;
		ValaMemberAccess* _tmp39;
		ValaExpression* _tmp38;
		ValaMemberAccess* right_ma;
		_tmp32 = NULL;
		sig = (_tmp32 = VALA_SIGNAL (vala_expression_get_symbol_reference (vala_assignment_get_left (self))), (_tmp32 == NULL) ? NULL : vala_code_node_ref (_tmp32));
		_tmp34 = NULL;
		_tmp33 = NULL;
		m = (_tmp34 = (_tmp33 = vala_expression_get_symbol_reference (vala_assignment_get_right (self)), VALA_IS_METHOD (_tmp33) ? ((ValaMethod*) _tmp33) : NULL), (_tmp34 == NULL) ? NULL : vala_code_node_ref (_tmp34));
		if (m == NULL) {
			gboolean _tmp35;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), "unsupported expression for signal handler");
			return (_tmp35 = FALSE, (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), _tmp35);
		}
		_tmp37 = NULL;
		_tmp36 = NULL;
		dynamic_sig = (_tmp37 = (_tmp36 = sig, VALA_IS_DYNAMIC_SIGNAL (_tmp36) ? ((ValaDynamicSignal*) _tmp36) : NULL), (_tmp37 == NULL) ? NULL : vala_code_node_ref (_tmp37));
		_tmp39 = NULL;
		_tmp38 = NULL;
		right_ma = (_tmp39 = (_tmp38 = vala_assignment_get_right (self), VALA_IS_MEMBER_ACCESS (_tmp38) ? ((ValaMemberAccess*) _tmp38) : NULL), (_tmp39 == NULL) ? NULL : vala_code_node_ref (_tmp39));
		if (dynamic_sig != NULL) {
			gboolean first;
			ValaDelegateType* _tmp45;
			ValaDelegate* _tmp44;
			ValaObjectType* _tmp43;
			first = TRUE;
			{
				GeeList* _tmp40;
				GeeIterator* _tmp41;
				GeeIterator* _param_it;
				_tmp40 = NULL;
				_tmp41 = NULL;
				_param_it = (_tmp41 = gee_iterable_iterator ((GeeIterable*) (_tmp40 = vala_data_type_get_parameters (vala_expression_get_value_type (vala_dynamic_signal_get_handler (dynamic_sig))))), (_tmp40 == NULL) ? NULL : (_tmp40 = (gee_collection_object_unref (_tmp40), NULL)), _tmp41);
				while (gee_iterator_next (_param_it)) {
					ValaFormalParameter* param;
					param = (ValaFormalParameter*) gee_iterator_get (_param_it);
					if (first) {
						/* skip sender parameter*/
						first = FALSE;
					} else {
						ValaFormalParameter* _tmp42;
						_tmp42 = NULL;
						vala_signal_add_parameter ((ValaSignal*) dynamic_sig, _tmp42 = vala_formal_parameter_copy (param));
						(_tmp42 == NULL) ? NULL : (_tmp42 = (vala_code_node_unref (_tmp42), NULL));
					}
					(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				}
				(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
			}
			_tmp45 = NULL;
			_tmp44 = NULL;
			_tmp43 = NULL;
			vala_expression_set_target_type (vala_assignment_get_right (self), (ValaDataType*) (_tmp45 = vala_delegate_type_new (_tmp44 = vala_signal_get_delegate (sig, (ValaDataType*) (_tmp43 = vala_object_type_new (VALA_OBJECT_TYPE_SYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) sig)))), (ValaCodeNode*) self))));
			(_tmp45 == NULL) ? NULL : (_tmp45 = (vala_code_node_unref (_tmp45), NULL));
			(_tmp44 == NULL) ? NULL : (_tmp44 = (vala_code_node_unref (_tmp44), NULL));
			(_tmp43 == NULL) ? NULL : (_tmp43 = (vala_code_node_unref (_tmp43), NULL));
		} else {
			if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_target_type (vala_assignment_get_right (self)))) {
				ValaDelegateType* _tmp46;
				ValaDelegateType* delegate_type;
				char* _tmp50;
				char* _tmp49;
				char* _tmp48;
				char* _tmp47;
				gboolean _tmp51;
				_tmp46 = NULL;
				delegate_type = (_tmp46 = VALA_DELEGATE_TYPE (vala_expression_get_target_type (vala_assignment_get_right (self))), (_tmp46 == NULL) ? NULL : vala_code_node_ref (_tmp46));
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp50 = NULL;
				_tmp49 = NULL;
				_tmp48 = NULL;
				_tmp47 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), _tmp50 = g_strdup_printf ("method `%s' is incompatible with signal `%s', expected `%s'", _tmp47 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp48 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_target_type (vala_assignment_get_right (self))), _tmp49 = vala_delegate_get_prototype_string (vala_delegate_type_get_delegate_symbol (delegate_type), vala_symbol_get_name ((ValaSymbol*) m))));
				_tmp50 = (g_free (_tmp50), NULL);
				_tmp49 = (g_free (_tmp49), NULL);
				_tmp48 = (g_free (_tmp48), NULL);
				_tmp47 = (g_free (_tmp47), NULL);
				return (_tmp51 = FALSE, (delegate_type == NULL) ? NULL : (delegate_type = (vala_code_node_unref (delegate_type), NULL)), (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL)), (right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL)), _tmp51);
			} else {
				gboolean _tmp52;
				_tmp52 = FALSE;
				if (right_ma != NULL) {
					_tmp52 = vala_member_access_get_prototype_access (right_ma);
				} else {
					_tmp52 = FALSE;
				}
				if (_tmp52) {
					char* _tmp54;
					char* _tmp53;
					gboolean _tmp55;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp54 = NULL;
					_tmp53 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_assignment_get_right (self)), _tmp54 = g_strdup_printf ("Access to instance member `%s' denied", _tmp53 = vala_symbol_get_full_name ((ValaSymbol*) m)));
					_tmp54 = (g_free (_tmp54), NULL);
					_tmp53 = (g_free (_tmp53), NULL);
					return (_tmp55 = FALSE, (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL)), (right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL)), _tmp55);
				}
			}
		}
		(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		(dynamic_sig == NULL) ? NULL : (dynamic_sig = (vala_code_node_unref (dynamic_sig), NULL));
		(right_ma == NULL) ? NULL : (right_ma = (vala_code_node_unref (right_ma), NULL));
	} else {
		if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (self))) {
			ValaMemberAccess* _tmp56;
			ValaMemberAccess* ma;
			gboolean _tmp92;
			_tmp56 = NULL;
			ma = (_tmp56 = VALA_MEMBER_ACCESS (vala_assignment_get_left (self)), (_tmp56 == NULL) ? NULL : vala_code_node_ref (_tmp56));
			if (VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
				ValaProperty* _tmp57;
				ValaProperty* prop;
				ValaDynamicProperty* _tmp59;
				ValaProperty* _tmp58;
				ValaDynamicProperty* dynamic_prop;
				gboolean _tmp62;
				_tmp57 = NULL;
				prop = (_tmp57 = VALA_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp57 == NULL) ? NULL : vala_code_node_ref (_tmp57));
				_tmp59 = NULL;
				_tmp58 = NULL;
				dynamic_prop = (_tmp59 = (_tmp58 = prop, VALA_IS_DYNAMIC_PROPERTY (_tmp58) ? ((ValaDynamicProperty*) _tmp58) : NULL), (_tmp59 == NULL) ? NULL : vala_code_node_ref (_tmp59));
				if (dynamic_prop != NULL) {
					ValaDataType* _tmp60;
					ValaDataType* _tmp61;
					_tmp60 = NULL;
					vala_property_set_property_type ((ValaProperty*) dynamic_prop, _tmp60 = vala_data_type_copy (vala_expression_get_value_type (vala_assignment_get_right (self))));
					(_tmp60 == NULL) ? NULL : (_tmp60 = (vala_code_node_unref (_tmp60), NULL));
					_tmp61 = NULL;
					vala_expression_set_value_type (vala_assignment_get_left (self), _tmp61 = vala_data_type_copy (vala_property_get_property_type ((ValaProperty*) dynamic_prop)));
					(_tmp61 == NULL) ? NULL : (_tmp61 = (vala_code_node_unref (_tmp61), NULL));
				}
				_tmp62 = FALSE;
				if (vala_property_get_set_accessor (prop) == NULL) {
					_tmp62 = TRUE;
				} else {
					gboolean _tmp63;
					_tmp63 = FALSE;
					if (!vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
						gboolean _tmp64;
						ValaMethod* _tmp65;
						gboolean _tmp66;
						_tmp64 = FALSE;
						_tmp65 = NULL;
						if ((_tmp66 = VALA_IS_CREATION_METHOD (_tmp65 = vala_semantic_analyzer_find_current_method (analyzer)), (_tmp65 == NULL) ? NULL : (_tmp65 = (vala_code_node_unref (_tmp65), NULL)), _tmp66)) {
							_tmp64 = TRUE;
						} else {
							_tmp64 = vala_semantic_analyzer_is_in_constructor (analyzer);
						}
						_tmp63 = !_tmp64;
					} else {
						_tmp63 = FALSE;
					}
					_tmp62 = _tmp63;
				}
				if (_tmp62) {
					char* _tmp68;
					char* _tmp67;
					gboolean _tmp69;
					vala_code_node_set_error ((ValaCodeNode*) ma, TRUE);
					_tmp68 = NULL;
					_tmp67 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) ma), _tmp68 = g_strdup_printf ("Property `%s' is read-only", _tmp67 = vala_symbol_get_full_name ((ValaSymbol*) prop)));
					_tmp68 = (g_free (_tmp68), NULL);
					_tmp67 = (g_free (_tmp67), NULL);
					return (_tmp69 = FALSE, (prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL)), (dynamic_prop == NULL) ? NULL : (dynamic_prop = (vala_code_node_unref (dynamic_prop), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp69);
				}
				(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
				(dynamic_prop == NULL) ? NULL : (dynamic_prop = (vala_code_node_unref (dynamic_prop), NULL));
			} else {
				gboolean _tmp70;
				_tmp70 = FALSE;
				if (VALA_IS_LOCAL_VARIABLE (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
					_tmp70 = vala_expression_get_value_type (vala_assignment_get_right (self)) == NULL;
				} else {
					_tmp70 = FALSE;
				}
				if (_tmp70) {
					ValaLocalVariable* _tmp71;
					ValaLocalVariable* local;
					gboolean _tmp72;
					_tmp71 = NULL;
					local = (_tmp71 = VALA_LOCAL_VARIABLE (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp71 == NULL) ? NULL : vala_code_node_ref (_tmp71));
					_tmp72 = FALSE;
					if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self)))) {
						_tmp72 = VALA_IS_DELEGATE_TYPE (vala_local_variable_get_variable_type (local));
					} else {
						_tmp72 = FALSE;
					}
					if (_tmp72) {
						ValaMethod* _tmp73;
						ValaMethod* m;
						ValaDelegateType* _tmp74;
						ValaDelegateType* dt;
						ValaDelegate* _tmp75;
						ValaDelegate* cb;
						_tmp73 = NULL;
						m = (_tmp73 = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self))), (_tmp73 == NULL) ? NULL : vala_code_node_ref (_tmp73));
						_tmp74 = NULL;
						dt = (_tmp74 = VALA_DELEGATE_TYPE (vala_local_variable_get_variable_type (local)), (_tmp74 == NULL) ? NULL : vala_code_node_ref (_tmp74));
						_tmp75 = NULL;
						cb = (_tmp75 = vala_delegate_type_get_delegate_symbol (dt), (_tmp75 == NULL) ? NULL : vala_code_node_ref (_tmp75));
						/* check whether method matches callback type */
						if (!vala_delegate_matches_method (cb, m)) {
							char* _tmp78;
							char* _tmp77;
							char* _tmp76;
							gboolean _tmp79;
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							_tmp78 = NULL;
							_tmp77 = NULL;
							_tmp76 = NULL;
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp78 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", _tmp76 = vala_symbol_get_full_name ((ValaSymbol*) m), _tmp77 = vala_symbol_get_full_name ((ValaSymbol*) cb)));
							_tmp78 = (g_free (_tmp78), NULL);
							_tmp77 = (g_free (_tmp77), NULL);
							_tmp76 = (g_free (_tmp76), NULL);
							return (_tmp79 = FALSE, (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp79);
						}
						vala_expression_set_value_type (vala_assignment_get_right (self), vala_local_variable_get_variable_type (local));
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
						(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
						(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
					} else {
						gboolean _tmp80;
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Assignment: Invalid callback assignment attempt");
						return (_tmp80 = FALSE, (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp80);
					}
					(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
				} else {
					gboolean _tmp81;
					_tmp81 = FALSE;
					if (VALA_IS_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
						_tmp81 = vala_expression_get_value_type (vala_assignment_get_right (self)) == NULL;
					} else {
						_tmp81 = FALSE;
					}
					if (_tmp81) {
						ValaField* _tmp82;
						ValaField* f;
						gboolean _tmp83;
						_tmp82 = NULL;
						f = (_tmp82 = VALA_FIELD (vala_expression_get_symbol_reference ((ValaExpression*) ma)), (_tmp82 == NULL) ? NULL : vala_code_node_ref (_tmp82));
						_tmp83 = FALSE;
						if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self)))) {
							_tmp83 = VALA_IS_DELEGATE_TYPE (vala_field_get_field_type (f));
						} else {
							_tmp83 = FALSE;
						}
						if (_tmp83) {
							ValaMethod* _tmp84;
							ValaMethod* m;
							ValaDelegateType* _tmp85;
							ValaDelegateType* dt;
							ValaDelegate* _tmp86;
							ValaDelegate* cb;
							_tmp84 = NULL;
							m = (_tmp84 = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (self))), (_tmp84 == NULL) ? NULL : vala_code_node_ref (_tmp84));
							_tmp85 = NULL;
							dt = (_tmp85 = VALA_DELEGATE_TYPE (vala_field_get_field_type (f)), (_tmp85 == NULL) ? NULL : vala_code_node_ref (_tmp85));
							_tmp86 = NULL;
							cb = (_tmp86 = vala_delegate_type_get_delegate_symbol (dt), (_tmp86 == NULL) ? NULL : vala_code_node_ref (_tmp86));
							/* check whether method matches callback type */
							if (!vala_delegate_matches_method (cb, m)) {
								char* _tmp89;
								char* _tmp88;
								char* _tmp87;
								gboolean _tmp90;
								vala_code_node_set_error ((ValaCodeNode*) f, TRUE);
								_tmp89 = NULL;
								_tmp88 = NULL;
								_tmp87 = NULL;
								vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp89 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", _tmp87 = vala_symbol_get_full_name ((ValaSymbol*) m), _tmp88 = vala_symbol_get_full_name ((ValaSymbol*) cb)));
								_tmp89 = (g_free (_tmp89), NULL);
								_tmp88 = (g_free (_tmp88), NULL);
								_tmp87 = (g_free (_tmp87), NULL);
								return (_tmp90 = FALSE, (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), (f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp90);
							}
							vala_expression_set_value_type (vala_assignment_get_right (self), vala_field_get_field_type (f));
							(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
							(dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL));
							(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
						} else {
							gboolean _tmp91;
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Assignment: Invalid callback assignment attempt");
							return (_tmp91 = FALSE, (f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL)), (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp91);
						}
						(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
					}
				}
			}
			_tmp92 = FALSE;
			if (vala_expression_get_value_type (vala_assignment_get_left (self)) != NULL) {
				_tmp92 = vala_expression_get_value_type (vala_assignment_get_right (self)) != NULL;
			} else {
				_tmp92 = FALSE;
			}
			if (_tmp92) {
				/* if there was an error on either side,
				 * i.e. {left|right}.value_type == null, skip type check */
				if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					char* _tmp95;
					char* _tmp94;
					char* _tmp93;
					gboolean _tmp96;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp95 = NULL;
					_tmp94 = NULL;
					_tmp93 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp95 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", _tmp93 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp94 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_left (self)))));
					_tmp95 = (g_free (_tmp95), NULL);
					_tmp94 = (g_free (_tmp94), NULL);
					_tmp93 = (g_free (_tmp93), NULL);
					return (_tmp96 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp96);
				}
				if (!VALA_IS_PROPERTY (vala_expression_get_symbol_reference ((ValaExpression*) ma))) {
					if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_assignment_get_right (self)))) {
						gboolean _tmp97;
						_tmp97 = FALSE;
						if (!VALA_IS_POINTER_TYPE (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
							_tmp97 = !vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)));
						} else {
							_tmp97 = FALSE;
						}
						/* rhs transfers ownership of the expression */
						if (_tmp97) {
							/* lhs doesn't own the value */
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid assignment from owned expression to unowned variable");
						}
					} else {
						if (vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
						}
					}
				}
			}
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
		} else {
			if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (self))) {
				ValaElementAccess* _tmp98;
				ValaElementAccess* ea;
				_tmp98 = NULL;
				ea = (_tmp98 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (self)), (_tmp98 == NULL) ? NULL : vala_code_node_ref (_tmp98));
				if (!vala_data_type_compatible (vala_expression_get_value_type (vala_assignment_get_right (self)), vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					char* _tmp101;
					char* _tmp100;
					char* _tmp99;
					gboolean _tmp102;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp101 = NULL;
					_tmp100 = NULL;
					_tmp99 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp101 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", _tmp99 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_right (self))), _tmp100 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_assignment_get_left (self)))));
					_tmp101 = (g_free (_tmp101), NULL);
					_tmp100 = (g_free (_tmp100), NULL);
					_tmp99 = (g_free (_tmp99), NULL);
					return (_tmp102 = FALSE, (ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL)), _tmp102);
				}
				if (vala_data_type_is_disposable (vala_expression_get_value_type (vala_assignment_get_right (self)))) {
					ValaDataType* element_type;
					gboolean _tmp107;
					/* rhs transfers ownership of the expression */
					element_type = NULL;
					if (VALA_IS_ARRAY_TYPE (vala_expression_get_value_type (vala_element_access_get_container (ea)))) {
						ValaArrayType* _tmp103;
						ValaArrayType* array_type;
						ValaDataType* _tmp105;
						ValaDataType* _tmp104;
						_tmp103 = NULL;
						array_type = (_tmp103 = VALA_ARRAY_TYPE (vala_expression_get_value_type (vala_element_access_get_container (ea))), (_tmp103 == NULL) ? NULL : vala_code_node_ref (_tmp103));
						_tmp105 = NULL;
						_tmp104 = NULL;
						element_type = (_tmp105 = (_tmp104 = vala_array_type_get_element_type (array_type), (_tmp104 == NULL) ? NULL : vala_code_node_ref (_tmp104)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp105);
						(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
					} else {
						GeeList* args;
						ValaDataType* _tmp106;
						args = vala_data_type_get_type_arguments (vala_expression_get_value_type (vala_element_access_get_container (ea)));
						g_assert (gee_collection_get_size ((GeeCollection*) args) == 1);
						_tmp106 = NULL;
						element_type = (_tmp106 = (ValaDataType*) gee_list_get (args, 0), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp106);
						(args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL));
					}
					_tmp107 = FALSE;
					if (!VALA_IS_POINTER_TYPE (element_type)) {
						_tmp107 = !vala_data_type_get_value_owned (element_type);
					} else {
						_tmp107 = FALSE;
					}
					if (_tmp107) {
						gboolean _tmp108;
						/* lhs doesn't own the value */
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Invalid assignment from owned expression to unowned variable");
						return (_tmp108 = FALSE, (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), (ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL)), _tmp108);
					}
					(element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL));
				} else {
					if (vala_data_type_get_value_owned (vala_expression_get_value_type (vala_assignment_get_left (self)))) {
					}
				}
				(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
			} else {
				/* lhs wants to own the value
				 * rhs doesn't transfer the ownership
				 * code generator needs to add reference
				 * increment calls */
				return TRUE;
			}
		}
	}
	if (vala_expression_get_value_type (vala_assignment_get_left (self)) != NULL) {
		ValaDataType* _tmp110;
		_tmp110 = NULL;
		vala_expression_set_value_type ((ValaExpression*) self, _tmp110 = vala_data_type_copy (vala_expression_get_value_type (vala_assignment_get_left (self))));
		(_tmp110 == NULL) ? NULL : (_tmp110 = (vala_code_node_unref (_tmp110), NULL));
		vala_data_type_set_value_owned (vala_expression_get_value_type ((ValaExpression*) self), FALSE);
	} else {
		vala_expression_set_value_type ((ValaExpression*) self, NULL);
	}
	_tmp111 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp111 = vala_code_node_get_error_types ((ValaCodeNode*) vala_assignment_get_left (self)));
	(_tmp111 == NULL) ? NULL : (_tmp111 = (gee_collection_object_unref (_tmp111), NULL));
	_tmp112 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp112 = vala_code_node_get_error_types ((ValaCodeNode*) vala_assignment_get_right (self)));
	(_tmp112 == NULL) ? NULL : (_tmp112 = (gee_collection_object_unref (_tmp112), NULL));
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


static void vala_assignment_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaAssignment * self;
	ValaLocalVariable* _tmp1;
	ValaSymbol* _tmp0;
	ValaLocalVariable* local;
	self = (ValaAssignment*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_assignment_get_right (self), collection);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_assignment_get_left (self), collection);
	_tmp1 = NULL;
	_tmp0 = NULL;
	local = (_tmp1 = (_tmp0 = vala_expression_get_symbol_reference (vala_assignment_get_left (self)), VALA_IS_LOCAL_VARIABLE (_tmp0) ? ((ValaLocalVariable*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	if (local != NULL) {
		gee_collection_add (collection, local);
	}
	(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
}


static void vala_assignment_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaAssignment * self;
	ValaMemberAccess* _tmp1;
	ValaExpression* _tmp0;
	ValaMemberAccess* ma;
	ValaElementAccess* _tmp3;
	ValaExpression* _tmp2;
	ValaElementAccess* ea;
	gboolean _tmp4;
	self = (ValaAssignment*) base;
	g_return_if_fail (collection != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	ma = (_tmp1 = (_tmp0 = vala_assignment_get_left (self), VALA_IS_MEMBER_ACCESS (_tmp0) ? ((ValaMemberAccess*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	_tmp3 = NULL;
	_tmp2 = NULL;
	ea = (_tmp3 = (_tmp2 = vala_assignment_get_left (self), VALA_IS_ELEMENT_ACCESS (_tmp2) ? ((ValaElementAccess*) _tmp2) : NULL), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
	_tmp4 = FALSE;
	if (ma != NULL) {
		_tmp4 = vala_member_access_get_inner (ma) != NULL;
	} else {
		_tmp4 = FALSE;
	}
	if (_tmp4) {
		vala_code_node_get_used_variables ((ValaCodeNode*) vala_member_access_get_inner (ma), collection);
	} else {
		if (ea != NULL) {
			vala_code_node_get_used_variables ((ValaCodeNode*) ea, collection);
		}
	}
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_assignment_get_right (self), collection);
	(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	(ea == NULL) ? NULL : (ea = (vala_code_node_unref (ea), NULL));
}


ValaExpression* vala_assignment_get_left (ValaAssignment* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_left;
}


void vala_assignment_set_left (ValaAssignment* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_left = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_code_node_unref (self->priv->_left), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_left, (ValaCodeNode*) self);
}


ValaAssignmentOperator vala_assignment_get_operator (ValaAssignment* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_operator;
}


void vala_assignment_set_operator (ValaAssignment* self, ValaAssignmentOperator value) {
	g_return_if_fail (self != NULL);
	self->priv->_operator = value;
}


ValaExpression* vala_assignment_get_right (ValaAssignment* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_right;
}


void vala_assignment_set_right (ValaAssignment* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_right = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_code_node_unref (self->priv->_right), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_right, (ValaCodeNode*) self);
}


static void vala_assignment_class_init (ValaAssignmentClass * klass) {
	vala_assignment_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_assignment_finalize;
	g_type_class_add_private (klass, sizeof (ValaAssignmentPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_assignment_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_assignment_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_assignment_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_assignment_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_assignment_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_assignment_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_assignment_real_get_used_variables;
}


static void vala_assignment_instance_init (ValaAssignment * self) {
	self->priv = VALA_ASSIGNMENT_GET_PRIVATE (self);
}


static void vala_assignment_finalize (ValaCodeNode* obj) {
	ValaAssignment * self;
	self = VALA_ASSIGNMENT (obj);
	(self->priv->_left == NULL) ? NULL : (self->priv->_left = (vala_code_node_unref (self->priv->_left), NULL));
	(self->priv->_right == NULL) ? NULL : (self->priv->_right = (vala_code_node_unref (self->priv->_right), NULL));
	VALA_CODE_NODE_CLASS (vala_assignment_parent_class)->finalize (obj);
}


GType vala_assignment_get_type (void) {
	static GType vala_assignment_type_id = 0;
	if (vala_assignment_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaAssignmentClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_assignment_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaAssignment), 0, (GInstanceInitFunc) vala_assignment_instance_init, NULL };
		vala_assignment_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaAssignment", &g_define_type_info, 0);
	}
	return vala_assignment_type_id;
}



GType vala_assignment_operator_get_type (void) {
	static GType vala_assignment_operator_type_id = 0;
	if (G_UNLIKELY (vala_assignment_operator_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_ASSIGNMENT_OPERATOR_NONE, "VALA_ASSIGNMENT_OPERATOR_NONE", "none"}, {VALA_ASSIGNMENT_OPERATOR_SIMPLE, "VALA_ASSIGNMENT_OPERATOR_SIMPLE", "simple"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_OR, "VALA_ASSIGNMENT_OPERATOR_BITWISE_OR", "bitwise-or"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_AND, "VALA_ASSIGNMENT_OPERATOR_BITWISE_AND", "bitwise-and"}, {VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR, "VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR", "bitwise-xor"}, {VALA_ASSIGNMENT_OPERATOR_ADD, "VALA_ASSIGNMENT_OPERATOR_ADD", "add"}, {VALA_ASSIGNMENT_OPERATOR_SUB, "VALA_ASSIGNMENT_OPERATOR_SUB", "sub"}, {VALA_ASSIGNMENT_OPERATOR_MUL, "VALA_ASSIGNMENT_OPERATOR_MUL", "mul"}, {VALA_ASSIGNMENT_OPERATOR_DIV, "VALA_ASSIGNMENT_OPERATOR_DIV", "div"}, {VALA_ASSIGNMENT_OPERATOR_PERCENT, "VALA_ASSIGNMENT_OPERATOR_PERCENT", "percent"}, {VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT, "VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT", "shift-left"}, {VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT, "VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT", "shift-right"}, {0, NULL, NULL}};
		vala_assignment_operator_type_id = g_enum_register_static ("ValaAssignmentOperator", values);
	}
	return vala_assignment_operator_type_id;
}




