/* valagirparser.vala
 *
 * Copyright (C) 2008-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vapigen/valagirparser.h>
#include <vala/valanamespace.h>
#include <vala/valasourcelocation.h>
#include <gee/hashmap.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valacodenode.h>
#include <vala/valastruct.h>
#include <vala/valadatatype.h>
#include <vala/valavoidtype.h>
#include <vala/valapointertype.h>
#include <vala/valaarraytype.h>
#include <vala/valaunresolvedtype.h>
#include <vala/valaunresolvedsymbol.h>
#include <vala/valaenum.h>
#include <vala/valaenumvalue.h>
#include <vala/valamethod.h>
#include <vala/valamember.h>
#include <gee/arraylist.h>
#include <vala/valaformalparameter.h>
#include <gee/collection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valaerrortype.h>
#include <vala/valaerrordomain.h>
#include <vala/valaerrorcode.h>
#include <vala/valadelegate.h>
#include <vala/valafield.h>
#include <vala/valaexpression.h>
#include <vala/valacreationmethod.h>
#include <vala/valaclass.h>
#include <vala/valasignal.h>
#include <vala/valaproperty.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valablock.h>
#include <vala/valainterface.h>
#include <vala/valaconstant.h>
#include <glib/gstdio.h>
#include <gee/map.h>
#include "vapigen/valamarkupreader.h"




static char* string_substring (const char* self, glong offset, glong len);
static glong string_get_length (const char* self);
struct _ValaGirParserPrivate {
	char* _package_name;
	ValaMarkupReader* reader;
	ValaCodeContext* context;
	ValaNamespace* glib_ns;
	ValaSourceFile* current_source_file;
	ValaSourceLocation begin;
	ValaSourceLocation end;
	ValaMarkupTokenType current_token;
	char** cheader_filenames;
	gint cheader_filenames_length1;
	gint cheader_filenames_size;
	GeeHashMap* attributes_map;
};

#define VALA_GIR_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIR_PARSER, ValaGirParserPrivate))
enum  {
	VALA_GIR_PARSER_DUMMY_PROPERTY
};
static void vala_gir_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void vala_gir_parser_next (ValaGirParser* self);
static void vala_gir_parser_start_element (ValaGirParser* self, const char* name);
static void vala_gir_parser_end_element (ValaGirParser* self, const char* name);
static ValaSourceReference* vala_gir_parser_get_current_src (ValaGirParser* self);
static void vala_gir_parser_parse_repository (ValaGirParser* self);
static void vala_gir_parser_parse_include (ValaGirParser* self);
static void vala_gir_parser_parse_package (ValaGirParser* self);
static void _vala_array_add1 (char*** array, int* length, int* size, char* value);
static void vala_gir_parser_parse_c_include (ValaGirParser* self);
static void _g_list_free_vala_code_node_unref (GList* self);
static ValaNamespace* vala_gir_parser_parse_namespace (ValaGirParser* self);
static ValaStruct* vala_gir_parser_parse_alias (ValaGirParser* self);
static ValaEnum* vala_gir_parser_parse_enumeration (ValaGirParser* self);
static ValaEnum* vala_gir_parser_parse_bitfield (ValaGirParser* self);
static ValaEnumValue* vala_gir_parser_parse_enumeration_member (ValaGirParser* self);
static ValaDataType* vala_gir_parser_parse_return_value (ValaGirParser* self);
static ValaFormalParameter* vala_gir_parser_parse_parameter (ValaGirParser* self, gint* array_length_idx, gint* closure_idx, gint* destroy_idx);
static ValaDataType* vala_gir_parser_parse_type (ValaGirParser* self, gint* array_length_index);
static ValaDataType* vala_gir_parser_parse_type_from_name (ValaGirParser* self, const char* type_name);
static char* vala_gir_parser_transform_namespace_name (ValaGirParser* self, const char* gir_module_name);
static ValaStruct* vala_gir_parser_parse_record (ValaGirParser* self);
static ValaClass* vala_gir_parser_parse_class (ValaGirParser* self);
static ValaInterface* vala_gir_parser_parse_interface (ValaGirParser* self);
static ValaField* vala_gir_parser_parse_field (ValaGirParser* self);
static ValaProperty* vala_gir_parser_parse_property (ValaGirParser* self);
static ValaDelegate* vala_gir_parser_parse_callback (ValaGirParser* self);
static ValaMethod* vala_gir_parser_parse_constructor (ValaGirParser* self);
static ValaMethod* vala_gir_parser_parse_method (ValaGirParser* self, const char* element_name);
static ValaSignal* vala_gir_parser_parse_signal (ValaGirParser* self);
static ValaStruct* vala_gir_parser_parse_boxed (ValaGirParser* self);
static ValaStruct* vala_gir_parser_parse_union (ValaGirParser* self);
static ValaConstant* vala_gir_parser_parse_constant (ValaGirParser* self);
static char* vala_gir_parser_get_attribute (ValaGirParser* self, const char* node, const char* key);
static void vala_gir_parser_set_package_name (ValaGirParser* self, const char* value);
static gpointer vala_gir_parser_parent_class = NULL;
static void vala_gir_parser_finalize (ValaCodeVisitor* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


static glong string_get_length (const char* self) {
	g_return_val_if_fail (self != NULL, 0L);
	return g_utf8_strlen (self, -1);
}


/**
 * Parses all .gir source files in the specified code
 * context and builds a code tree.
 *
 * @param context a code context
 */
void vala_gir_parser_parse (ValaGirParser* self, ValaCodeContext* context) {
	ValaCodeContext* _tmp1;
	ValaCodeContext* _tmp0;
	ValaNamespace* _tmp3;
	ValaSymbol* _tmp2;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL) ? NULL : vala_code_context_ref (_tmp0)), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp1);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->glib_ns = (_tmp3 = (_tmp2 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) vala_code_context_get_root (context)), "GLib"), VALA_IS_NAMESPACE (_tmp2) ? ((ValaNamespace*) _tmp2) : NULL), (self->priv->glib_ns == NULL) ? NULL : (self->priv->glib_ns = (vala_code_node_unref (self->priv->glib_ns), NULL)), _tmp3);
	vala_code_context_accept (context, (ValaCodeVisitor*) self);
}


static void vala_gir_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaGirParser * self;
	self = (ValaGirParser*) base;
	g_return_if_fail (source_file != NULL);
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".gir")) {
		vala_gir_parser_parse_file (self, source_file);
	}
}


void vala_gir_parser_parse_file (ValaGirParser* self, ValaSourceFile* source_file) {
	ValaSourceFile* _tmp1;
	ValaSourceFile* _tmp0;
	ValaMarkupReader* _tmp2;
	GeeArrayList* remove_queue;
	ValaMarkupReader* _tmp17;
	ValaSourceFile* _tmp18;
	g_return_if_fail (self != NULL);
	g_return_if_fail (source_file != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->current_source_file = (_tmp1 = (_tmp0 = source_file, (_tmp0 == NULL) ? NULL : vala_source_file_ref (_tmp0)), (self->priv->current_source_file == NULL) ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL)), _tmp1);
	_tmp2 = NULL;
	self->priv->reader = (_tmp2 = vala_markup_reader_new (vala_source_file_get_filename (source_file)), (self->priv->reader == NULL) ? NULL : (self->priv->reader = (g_object_unref (self->priv->reader), NULL)), _tmp2);
	/* xml prolog*/
	vala_gir_parser_next (self);
	vala_gir_parser_next (self);
	vala_gir_parser_next (self);
	vala_gir_parser_parse_repository (self);
	remove_queue = gee_array_list_new (VALA_TYPE_CODE_NODE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	{
		GeeList* _tmp3;
		GeeIterator* _tmp4;
		GeeIterator* _node_it;
		_tmp3 = NULL;
		_tmp4 = NULL;
		_node_it = (_tmp4 = gee_iterable_iterator ((GeeIterable*) (_tmp3 = vala_source_file_get_nodes (source_file))), (_tmp3 == NULL) ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL)), _tmp4);
		while (gee_iterator_next (_node_it)) {
			ValaCodeNode* node;
			node = (ValaCodeNode*) gee_iterator_get (_node_it);
			if (VALA_IS_CLASS (node)) {
				ValaClass* _tmp5;
				ValaClass* cl;
				ValaNamespace* _tmp7;
				ValaSymbol* _tmp6;
				ValaNamespace* ns;
				char* _tmp8;
				ValaSymbol* _tmp9;
				ValaStruct* _tmp10;
				ValaStruct* class_struct;
				_tmp5 = NULL;
				cl = (_tmp5 = VALA_CLASS (node), (_tmp5 == NULL) ? NULL : vala_code_node_ref (_tmp5));
				_tmp7 = NULL;
				_tmp6 = NULL;
				ns = (_tmp7 = (_tmp6 = vala_symbol_get_parent_symbol ((ValaSymbol*) cl), VALA_IS_NAMESPACE (_tmp6) ? ((ValaNamespace*) _tmp6) : NULL), (_tmp7 == NULL) ? NULL : vala_code_node_ref (_tmp7));
				/* remove Class records*/
				_tmp8 = NULL;
				_tmp9 = NULL;
				_tmp10 = NULL;
				class_struct = (_tmp10 = (_tmp9 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), _tmp8 = g_strconcat (vala_symbol_get_name ((ValaSymbol*) cl), "Class", NULL)), VALA_IS_STRUCT (_tmp9) ? ((ValaStruct*) _tmp9) : NULL), _tmp8 = (g_free (_tmp8), NULL), _tmp10);
				if (class_struct != NULL) {
					vala_namespace_remove_struct (ns, VALA_STRUCT (class_struct));
					gee_collection_add ((GeeCollection*) remove_queue, (ValaCodeNode*) class_struct);
				}
				(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
				(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
				(class_struct == NULL) ? NULL : (class_struct = (vala_code_node_unref (class_struct), NULL));
			} else {
				if (VALA_IS_INTERFACE (node)) {
					ValaInterface* _tmp11;
					ValaInterface* iface;
					ValaNamespace* _tmp13;
					ValaSymbol* _tmp12;
					ValaNamespace* ns;
					char* _tmp14;
					ValaSymbol* _tmp15;
					ValaStruct* _tmp16;
					ValaStruct* iface_struct;
					_tmp11 = NULL;
					iface = (_tmp11 = VALA_INTERFACE (node), (_tmp11 == NULL) ? NULL : vala_code_node_ref (_tmp11));
					_tmp13 = NULL;
					_tmp12 = NULL;
					ns = (_tmp13 = (_tmp12 = vala_symbol_get_parent_symbol ((ValaSymbol*) iface), VALA_IS_NAMESPACE (_tmp12) ? ((ValaNamespace*) _tmp12) : NULL), (_tmp13 == NULL) ? NULL : vala_code_node_ref (_tmp13));
					/* remove Iface records*/
					_tmp14 = NULL;
					_tmp15 = NULL;
					_tmp16 = NULL;
					iface_struct = (_tmp16 = (_tmp15 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), _tmp14 = g_strconcat (vala_symbol_get_name ((ValaSymbol*) iface), "Iface", NULL)), VALA_IS_STRUCT (_tmp15) ? ((ValaStruct*) _tmp15) : NULL), _tmp14 = (g_free (_tmp14), NULL), _tmp16);
					if (iface_struct != NULL) {
						vala_namespace_remove_struct (ns, VALA_STRUCT (iface_struct));
						gee_collection_add ((GeeCollection*) remove_queue, (ValaCodeNode*) iface_struct);
					}
					(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
					(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
					(iface_struct == NULL) ? NULL : (iface_struct = (vala_code_node_unref (iface_struct), NULL));
				}
			}
			(node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL));
		}
		(_node_it == NULL) ? NULL : (_node_it = (gee_collection_object_unref (_node_it), NULL));
	}
	{
		GeeIterator* _node_it;
		_node_it = gee_iterable_iterator ((GeeIterable*) remove_queue);
		while (gee_iterator_next (_node_it)) {
			ValaCodeNode* node;
			node = (ValaCodeNode*) gee_iterator_get (_node_it);
			vala_source_file_remove_node (source_file, node);
			(node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL));
		}
		(_node_it == NULL) ? NULL : (_node_it = (gee_collection_object_unref (_node_it), NULL));
	}
	_tmp17 = NULL;
	self->priv->reader = (_tmp17 = NULL, (self->priv->reader == NULL) ? NULL : (self->priv->reader = (g_object_unref (self->priv->reader), NULL)), _tmp17);
	_tmp18 = NULL;
	self->priv->current_source_file = (_tmp18 = NULL, (self->priv->current_source_file == NULL) ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL)), _tmp18);
	(remove_queue == NULL) ? NULL : (remove_queue = (gee_collection_object_unref (remove_queue), NULL));
}


static void vala_gir_parser_next (ValaGirParser* self) {
	g_return_if_fail (self != NULL);
	self->priv->current_token = vala_markup_reader_read_token (self->priv->reader, &self->priv->begin, &self->priv->end);
}


static void vala_gir_parser_start_element (ValaGirParser* self, const char* name) {
	gboolean _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	_tmp0 = FALSE;
	if (self->priv->current_token != VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), name) != 0;
	}
	if (_tmp0) {
		char* _tmp2;
		ValaSourceReference* _tmp1;
		/* error*/
		_tmp2 = NULL;
		_tmp1 = NULL;
		vala_report_error (_tmp1 = vala_gir_parser_get_current_src (self), _tmp2 = g_strdup_printf ("expected start element of `%s'", name));
		_tmp2 = (g_free (_tmp2), NULL);
		(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL));
	}
}


static void vala_gir_parser_end_element (ValaGirParser* self, const char* name) {
	gboolean _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	_tmp0 = FALSE;
	if (self->priv->current_token != VALA_MARKUP_TOKEN_TYPE_END_ELEMENT) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), name) != 0;
	}
	if (_tmp0) {
		char* _tmp2;
		ValaSourceReference* _tmp1;
		/* error*/
		_tmp2 = NULL;
		_tmp1 = NULL;
		vala_report_error (_tmp1 = vala_gir_parser_get_current_src (self), _tmp2 = g_strdup_printf ("expected end element of `%s'", name));
		_tmp2 = (g_free (_tmp2), NULL);
		(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL));
	}
	vala_gir_parser_next (self);
}


static ValaSourceReference* vala_gir_parser_get_current_src (ValaGirParser* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return vala_source_reference_new (self->priv->current_source_file, self->priv->begin.line, self->priv->begin.column, self->priv->end.line, self->priv->end.column);
}


static void vala_gir_parser_parse_repository (ValaGirParser* self) {
	g_return_if_fail (self != NULL);
	vala_gir_parser_start_element (self, "repository");
	vala_gir_parser_next (self);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "namespace") == 0) {
			ValaNamespace* ns;
			ns = vala_gir_parser_parse_namespace (self);
			if (ns != NULL) {
				vala_namespace_add_namespace (vala_code_context_get_root (self->priv->context), ns);
			}
			(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "include") == 0) {
				vala_gir_parser_parse_include (self);
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "package") == 0) {
					vala_gir_parser_parse_package (self);
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "c:include") == 0) {
						vala_gir_parser_parse_c_include (self);
					} else {
						char* _tmp1;
						ValaSourceReference* _tmp0;
						/* error*/
						_tmp1 = NULL;
						_tmp0 = NULL;
						vala_report_error (_tmp0 = vala_gir_parser_get_current_src (self), _tmp1 = g_strdup_printf ("unknown child element `%s' in `repository'", vala_markup_reader_get_name (self->priv->reader)));
						_tmp1 = (g_free (_tmp1), NULL);
						(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL));
						break;
					}
				}
			}
		}
	}
	vala_gir_parser_end_element (self, "repository");
}


static void vala_gir_parser_parse_include (ValaGirParser* self) {
	g_return_if_fail (self != NULL);
	vala_gir_parser_start_element (self, "include");
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "include");
}


static void vala_gir_parser_parse_package (ValaGirParser* self) {
	char* _tmp0;
	g_return_if_fail (self != NULL);
	vala_gir_parser_start_element (self, "package");
	_tmp0 = NULL;
	vala_gir_parser_set_package_name (self, _tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name"));
	_tmp0 = (g_free (_tmp0), NULL);
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "package");
}


static void _vala_array_add1 (char*** array, int* length, int* size, char* value) {
	if ((*length) == (*size)) {
		*size = (*size) ? (2 * (*size)) : 4;
		*array = g_renew (char*, *array, (*size) + 1);
	}
	(*array)[(*length)++] = value;
	(*array)[*length] = NULL;
}


static void vala_gir_parser_parse_c_include (ValaGirParser* self) {
	g_return_if_fail (self != NULL);
	vala_gir_parser_start_element (self, "c:include");
	_vala_array_add1 (&self->priv->cheader_filenames, &self->priv->cheader_filenames_length1, &self->priv->cheader_filenames_size, vala_markup_reader_get_attribute (self->priv->reader, "name"));
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "c:include");
}


static void _g_list_free_vala_code_node_unref (GList* self) {
	g_list_foreach (self, (GFunc) vala_code_node_unref, NULL);
	g_list_free (self);
}


static ValaNamespace* vala_gir_parser_parse_namespace (ValaGirParser* self) {
	gboolean new_namespace;
	char* _tmp0;
	char* _tmp1;
	char* namespace_name;
	ValaSymbol* _tmp2;
	ValaNamespace* ns;
	ValaNamespace* _tmp21;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "namespace");
	new_namespace = FALSE;
	_tmp0 = NULL;
	_tmp1 = NULL;
	namespace_name = (_tmp1 = vala_gir_parser_transform_namespace_name (self, _tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name")), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
	_tmp2 = NULL;
	ns = (_tmp2 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) vala_code_context_get_root (self->priv->context)), namespace_name), VALA_IS_NAMESPACE (_tmp2) ? ((ValaNamespace*) _tmp2) : NULL);
	if (ns == NULL) {
		ValaNamespace* _tmp3;
		_tmp3 = NULL;
		ns = (_tmp3 = vala_namespace_new (namespace_name, NULL), (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), _tmp3);
		new_namespace = TRUE;
	} else {
		if (vala_symbol_get_external_package ((ValaSymbol*) ns)) {
			GList* _tmp4;
			ValaSourceReference* _tmp5;
			_tmp4 = NULL;
			((ValaCodeNode*) ns)->attributes = (_tmp4 = NULL, (((ValaCodeNode*) ns)->attributes == NULL) ? NULL : (((ValaCodeNode*) ns)->attributes = (_g_list_free_vala_code_node_unref (((ValaCodeNode*) ns)->attributes), NULL)), _tmp4);
			_tmp5 = NULL;
			vala_code_node_set_source_reference ((ValaCodeNode*) ns, _tmp5 = vala_gir_parser_get_current_src (self));
			(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL));
		}
	}
	{
		char** c_header_collection;
		int c_header_collection_length1;
		int c_header_it;
		c_header_collection = self->priv->cheader_filenames;
		c_header_collection_length1 = self->priv->cheader_filenames_length1;
		for (c_header_it = 0; c_header_it < self->priv->cheader_filenames_length1; c_header_it = c_header_it + 1) {
			const char* _tmp6;
			char* c_header;
			_tmp6 = NULL;
			c_header = (_tmp6 = c_header_collection[c_header_it], (_tmp6 == NULL) ? NULL : g_strdup (_tmp6));
			{
				vala_namespace_add_cheader_filename (ns, c_header);
				c_header = (g_free (c_header), NULL);
			}
		}
	}
	vala_gir_parser_next (self);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		ValaSymbol* sym;
		sym = NULL;
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "alias") == 0) {
			ValaSymbol* _tmp7;
			_tmp7 = NULL;
			sym = (_tmp7 = (ValaSymbol*) vala_gir_parser_parse_alias (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp7);
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "enumeration") == 0) {
				ValaSymbol* _tmp8;
				_tmp8 = NULL;
				sym = (_tmp8 = (ValaSymbol*) vala_gir_parser_parse_enumeration (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp8);
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "bitfield") == 0) {
					ValaSymbol* _tmp9;
					_tmp9 = NULL;
					sym = (_tmp9 = (ValaSymbol*) vala_gir_parser_parse_bitfield (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp9);
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "function") == 0) {
						ValaSymbol* _tmp10;
						_tmp10 = NULL;
						sym = (_tmp10 = (ValaSymbol*) vala_gir_parser_parse_method (self, "function"), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp10);
					} else {
						if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "callback") == 0) {
							ValaSymbol* _tmp11;
							_tmp11 = NULL;
							sym = (_tmp11 = (ValaSymbol*) vala_gir_parser_parse_callback (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp11);
						} else {
							if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "record") == 0) {
								ValaSymbol* _tmp12;
								_tmp12 = NULL;
								sym = (_tmp12 = (ValaSymbol*) vala_gir_parser_parse_record (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp12);
							} else {
								if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "class") == 0) {
									ValaSymbol* _tmp13;
									_tmp13 = NULL;
									sym = (_tmp13 = (ValaSymbol*) vala_gir_parser_parse_class (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp13);
								} else {
									if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "interface") == 0) {
										ValaSymbol* _tmp14;
										_tmp14 = NULL;
										sym = (_tmp14 = (ValaSymbol*) vala_gir_parser_parse_interface (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp14);
									} else {
										if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "glib:boxed") == 0) {
											ValaSymbol* _tmp15;
											_tmp15 = NULL;
											sym = (_tmp15 = (ValaSymbol*) vala_gir_parser_parse_boxed (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp15);
										} else {
											if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "union") == 0) {
												ValaSymbol* _tmp16;
												_tmp16 = NULL;
												sym = (_tmp16 = (ValaSymbol*) vala_gir_parser_parse_union (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp16);
											} else {
												if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constant") == 0) {
													ValaSymbol* _tmp17;
													_tmp17 = NULL;
													sym = (_tmp17 = (ValaSymbol*) vala_gir_parser_parse_constant (self), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp17);
												} else {
													char* _tmp19;
													ValaSourceReference* _tmp18;
													/* error*/
													_tmp19 = NULL;
													_tmp18 = NULL;
													vala_report_error (_tmp18 = vala_gir_parser_get_current_src (self), _tmp19 = g_strdup_printf ("unknown child element `%s' in `namespace'", vala_markup_reader_get_name (self->priv->reader)));
													_tmp19 = (g_free (_tmp19), NULL);
													(_tmp18 == NULL) ? NULL : (_tmp18 = (vala_source_reference_unref (_tmp18), NULL));
													(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
													break;
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
		if (VALA_IS_CLASS (sym)) {
			vala_namespace_add_class (ns, VALA_CLASS (sym));
		} else {
			if (VALA_IS_INTERFACE (sym)) {
				vala_namespace_add_interface (ns, VALA_INTERFACE (sym));
			} else {
				if (VALA_IS_STRUCT (sym)) {
					vala_namespace_add_struct (ns, VALA_STRUCT (sym));
				} else {
					if (VALA_IS_ENUM (sym)) {
						vala_namespace_add_enum (ns, VALA_ENUM (sym));
					} else {
						if (VALA_IS_DELEGATE (sym)) {
							vala_namespace_add_delegate (ns, VALA_DELEGATE (sym));
						} else {
							if (VALA_IS_METHOD (sym)) {
								vala_namespace_add_method (ns, VALA_METHOD (sym));
							} else {
								if (VALA_IS_CONSTANT (sym)) {
									vala_namespace_add_constant (ns, VALA_CONSTANT (sym));
								} else {
									if (sym == NULL) {
										(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
										continue;
									}
								}
							}
						}
					}
				}
			}
		}
		vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) sym);
		(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
	}
	vala_gir_parser_end_element (self, "namespace");
	if (!new_namespace) {
		ValaNamespace* _tmp20;
		_tmp20 = NULL;
		ns = (_tmp20 = NULL, (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), _tmp20);
	}
	_tmp21 = NULL;
	return (_tmp21 = ns, namespace_name = (g_free (namespace_name), NULL), _tmp21);
}


static ValaStruct* vala_gir_parser_parse_alias (ValaGirParser* self) {
	ValaSourceReference* _tmp1;
	char* _tmp0;
	ValaStruct* _tmp2;
	ValaStruct* st;
	ValaDataType* _tmp4;
	char* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "alias");
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	st = (_tmp2 = vala_struct_new (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name"), _tmp1 = vala_gir_parser_get_current_src (self)), (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
	vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp4 = NULL;
	_tmp3 = NULL;
	vala_struct_set_base_type (st, _tmp4 = vala_gir_parser_parse_type_from_name (self, _tmp3 = vala_markup_reader_get_attribute (self->priv->reader, "target")));
	(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
	_tmp3 = (g_free (_tmp3), NULL);
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "alias");
	return st;
}


static ValaEnum* vala_gir_parser_parse_enumeration (ValaGirParser* self) {
	ValaSourceReference* _tmp1;
	char* _tmp0;
	ValaEnum* _tmp2;
	ValaEnum* en;
	char* enum_cname;
	char* common_prefix;
	ValaEnum* _tmp12;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "enumeration");
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	en = (_tmp2 = vala_enum_new (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name"), _tmp1 = vala_gir_parser_get_current_src (self)), (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
	vala_symbol_set_access ((ValaSymbol*) en, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	enum_cname = vala_markup_reader_get_attribute (self->priv->reader, "c:type");
	if (enum_cname != NULL) {
		vala_enum_set_cname (en, enum_cname);
	}
	vala_gir_parser_next (self);
	common_prefix = NULL;
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "member") == 0) {
			ValaEnumValue* ev;
			char* cname;
			ev = vala_gir_parser_parse_enumeration_member (self);
			vala_enum_add_value (en, ev);
			cname = vala_enum_value_get_cname (ev);
			if (common_prefix == NULL) {
				char* _tmp4;
				const char* _tmp3;
				_tmp4 = NULL;
				_tmp3 = NULL;
				common_prefix = (_tmp4 = (_tmp3 = cname, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3)), common_prefix = (g_free (common_prefix), NULL), _tmp4);
				while (TRUE) {
					gboolean _tmp5;
					char* _tmp6;
					_tmp5 = FALSE;
					if (g_utf8_strlen (common_prefix, -1) > 0) {
						_tmp5 = !g_str_has_suffix (common_prefix, "_");
					} else {
						_tmp5 = FALSE;
					}
					if (!_tmp5) {
						break;
					}
					/* FIXME: could easily be made faster*/
					_tmp6 = NULL;
					common_prefix = (_tmp6 = g_strndup (common_prefix, (gsize) (strlen (common_prefix) - 1)), common_prefix = (g_free (common_prefix), NULL), _tmp6);
				}
			} else {
				while (!g_str_has_prefix (cname, common_prefix)) {
					char* _tmp7;
					_tmp7 = NULL;
					common_prefix = (_tmp7 = g_strndup (common_prefix, (gsize) (strlen (common_prefix) - 1)), common_prefix = (g_free (common_prefix), NULL), _tmp7);
				}
			}
			while (TRUE) {
				gboolean _tmp8;
				char* _tmp11;
				_tmp8 = FALSE;
				if (g_utf8_strlen (common_prefix, -1) > 0) {
					gboolean _tmp9;
					_tmp9 = FALSE;
					if (!g_str_has_suffix (common_prefix, "_")) {
						_tmp9 = TRUE;
					} else {
						gboolean _tmp10;
						_tmp10 = FALSE;
						if (g_unichar_isdigit (g_utf8_get_char (g_utf8_offset_to_pointer (cname, strlen (common_prefix))))) {
							_tmp10 = (g_utf8_strlen (cname, -1) - g_utf8_strlen (common_prefix, -1)) <= 1;
						} else {
							_tmp10 = FALSE;
						}
						_tmp9 = _tmp10;
					}
					_tmp8 = _tmp9;
				} else {
					_tmp8 = FALSE;
				}
				if (!_tmp8) {
					break;
				}
				/* enum values may not consist solely of digits*/
				_tmp11 = NULL;
				common_prefix = (_tmp11 = g_strndup (common_prefix, (gsize) (strlen (common_prefix) - 1)), common_prefix = (g_free (common_prefix), NULL), _tmp11);
			}
			(ev == NULL) ? NULL : (ev = (vala_code_node_unref (ev), NULL));
			cname = (g_free (cname), NULL);
		} else {
			/* error*/
			break;
		}
	}
	vala_enum_set_cprefix (en, common_prefix);
	vala_gir_parser_end_element (self, "enumeration");
	_tmp12 = NULL;
	return (_tmp12 = en, enum_cname = (g_free (enum_cname), NULL), common_prefix = (g_free (common_prefix), NULL), _tmp12);
}


static ValaEnum* vala_gir_parser_parse_bitfield (ValaGirParser* self) {
	ValaSourceReference* _tmp1;
	char* _tmp0;
	ValaEnum* _tmp2;
	ValaEnum* en;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "bitfield");
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	en = (_tmp2 = vala_enum_new (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name"), _tmp1 = vala_gir_parser_get_current_src (self)), (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
	vala_symbol_set_access ((ValaSymbol*) en, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_next (self);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "member") == 0) {
			ValaEnumValue* _tmp3;
			_tmp3 = NULL;
			vala_enum_add_value (en, _tmp3 = vala_gir_parser_parse_enumeration_member (self));
			(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL));
		} else {
			/* error*/
			break;
		}
	}
	vala_gir_parser_end_element (self, "bitfield");
	return en;
}


static ValaEnumValue* vala_gir_parser_parse_enumeration_member (ValaGirParser* self) {
	char* _tmp4;
	gint _tmp3_length1;
	char** _tmp3;
	char** _tmp2;
	char* _tmp1;
	char* _tmp0;
	ValaEnumValue* _tmp5;
	ValaEnumValue* ev;
	char* _tmp6;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "member");
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp5 = NULL;
	ev = (_tmp5 = vala_enum_value_new (_tmp4 = g_strjoinv ("_", (_tmp3 = _tmp2 = g_strsplit (_tmp1 = g_utf8_strup (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name"), -1), "-", 0), _tmp3_length1 = _vala_array_length (_tmp2), _tmp3)), NULL), _tmp4 = (g_free (_tmp4), NULL), _tmp3 = (_vala_array_free (_tmp3, _tmp3_length1, (GDestroyNotify) g_free), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp5);
	_tmp6 = NULL;
	vala_enum_value_set_cname (ev, _tmp6 = vala_markup_reader_get_attribute (self->priv->reader, "c:identifier"));
	_tmp6 = (g_free (_tmp6), NULL);
	vala_gir_parser_next (self);
	vala_gir_parser_end_element (self, "member");
	return ev;
}


static ValaDataType* vala_gir_parser_parse_return_value (ValaGirParser* self) {
	char* transfer;
	ValaDataType* type;
	ValaDataType* _tmp0;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "return-value");
	transfer = vala_markup_reader_get_attribute (self->priv->reader, "transfer-ownership");
	vala_gir_parser_next (self);
	type = vala_gir_parser_parse_type (self, NULL);
	if (_vala_strcmp0 (transfer, "full") == 0) {
		vala_data_type_set_value_owned (type, TRUE);
	}
	vala_gir_parser_end_element (self, "return-value");
	_tmp0 = NULL;
	return (_tmp0 = type, transfer = (g_free (transfer), NULL), _tmp0);
}


static ValaFormalParameter* vala_gir_parser_parse_parameter (ValaGirParser* self, gint* array_length_idx, gint* closure_idx, gint* destroy_idx) {
	ValaFormalParameter* param;
	char* name;
	char* direction;
	char* transfer;
	char* allow_none;
	char* closure;
	char* destroy;
	gboolean _tmp0;
	gboolean _tmp1;
	ValaFormalParameter* _tmp6;
	g_return_val_if_fail (self != NULL, NULL);
	param = NULL;
	if ((&(*array_length_idx)) != NULL) {
		(*array_length_idx) = -1;
	}
	if ((&(*closure_idx)) != NULL) {
		(*closure_idx) = -1;
	}
	if ((&(*destroy_idx)) != NULL) {
		(*destroy_idx) = -1;
	}
	vala_gir_parser_start_element (self, "parameter");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	direction = vala_markup_reader_get_attribute (self->priv->reader, "direction");
	transfer = vala_markup_reader_get_attribute (self->priv->reader, "transfer-ownership");
	allow_none = vala_markup_reader_get_attribute (self->priv->reader, "allow-none");
	closure = vala_markup_reader_get_attribute (self->priv->reader, "closure");
	destroy = vala_markup_reader_get_attribute (self->priv->reader, "destroy");
	_tmp0 = FALSE;
	if (closure != NULL) {
		_tmp0 = (&(*closure_idx)) != NULL;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		(*closure_idx) = atoi (closure);
	}
	_tmp1 = FALSE;
	if (destroy != NULL) {
		_tmp1 = (&(*destroy_idx)) != NULL;
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		(*destroy_idx) = atoi (destroy);
	}
	vala_gir_parser_next (self);
	if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "varargs") == 0) {
		ValaFormalParameter* _tmp3;
		ValaSourceReference* _tmp2;
		vala_gir_parser_start_element (self, "varargs");
		vala_gir_parser_next (self);
		_tmp3 = NULL;
		_tmp2 = NULL;
		param = (_tmp3 = vala_formal_parameter_new_with_ellipsis (_tmp2 = vala_gir_parser_get_current_src (self)), (param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL)), _tmp3);
		(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL));
		vala_gir_parser_end_element (self, "varargs");
	} else {
		ValaDataType* type;
		ValaFormalParameter* _tmp5;
		ValaSourceReference* _tmp4;
		type = vala_gir_parser_parse_type (self, &(*array_length_idx));
		if (_vala_strcmp0 (transfer, "full") == 0) {
			vala_data_type_set_value_owned (type, TRUE);
		}
		if (_vala_strcmp0 (allow_none, "1") == 0) {
			vala_data_type_set_nullable (type, TRUE);
		}
		_tmp5 = NULL;
		_tmp4 = NULL;
		param = (_tmp5 = vala_formal_parameter_new (name, type, _tmp4 = vala_gir_parser_get_current_src (self)), (param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL)), _tmp5);
		(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL));
		if (_vala_strcmp0 (direction, "out") == 0) {
			vala_formal_parameter_set_direction (param, VALA_PARAMETER_DIRECTION_OUT);
		} else {
			if (_vala_strcmp0 (direction, "inout") == 0) {
				vala_formal_parameter_set_direction (param, VALA_PARAMETER_DIRECTION_REF);
			}
		}
		(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
	}
	vala_gir_parser_end_element (self, "parameter");
	_tmp6 = NULL;
	return (_tmp6 = param, name = (g_free (name), NULL), direction = (g_free (direction), NULL), transfer = (g_free (transfer), NULL), allow_none = (g_free (allow_none), NULL), closure = (g_free (closure), NULL), destroy = (g_free (destroy), NULL), _tmp6);
}


static ValaDataType* vala_gir_parser_parse_type (ValaGirParser* self, gint* array_length_index) {
	g_return_val_if_fail (self != NULL, NULL);
	if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "array") == 0) {
		gboolean _tmp0;
		char* _tmp1;
		gboolean _tmp2;
		ValaDataType* element_type;
		ValaDataType* _tmp4;
		vala_gir_parser_start_element (self, "array");
		_tmp0 = FALSE;
		_tmp1 = NULL;
		if ((_tmp2 = (_tmp1 = vala_markup_reader_get_attribute (self->priv->reader, "length")) != NULL, _tmp1 = (g_free (_tmp1), NULL), _tmp2)) {
			_tmp0 = (&(*array_length_index)) != NULL;
		} else {
			_tmp0 = FALSE;
		}
		if (_tmp0) {
			char* _tmp3;
			_tmp3 = NULL;
			(*array_length_index) = atoi (_tmp3 = vala_markup_reader_get_attribute (self->priv->reader, "length"));
			_tmp3 = (g_free (_tmp3), NULL);
		}
		vala_gir_parser_next (self);
		element_type = vala_gir_parser_parse_type (self, NULL);
		vala_gir_parser_end_element (self, "array");
		_tmp4 = NULL;
		return (_tmp4 = (ValaDataType*) vala_array_type_new (element_type, 1, NULL), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp4);
	} else {
		char* _tmp5;
		ValaDataType* _tmp6;
		ValaDataType* type;
		vala_gir_parser_start_element (self, "type");
		_tmp5 = NULL;
		_tmp6 = NULL;
		type = (_tmp6 = vala_gir_parser_parse_type_from_name (self, _tmp5 = vala_markup_reader_get_attribute (self->priv->reader, "name")), _tmp5 = (g_free (_tmp5), NULL), _tmp6);
		vala_gir_parser_next (self);
		while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
			ValaDataType* _tmp7;
			_tmp7 = NULL;
			_tmp7 = vala_gir_parser_parse_type (self, NULL);
			(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_code_node_unref (_tmp7), NULL));
		}
		vala_gir_parser_end_element (self, "type");
		return type;
	}
}


static ValaDataType* vala_gir_parser_parse_type_from_name (ValaGirParser* self, const char* type_name) {
	ValaDataType* type;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type_name != NULL, NULL);
	type = NULL;
	if (_vala_strcmp0 (type_name, "none") == 0) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		type = (_tmp0 = (ValaDataType*) vala_void_type_new (NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp0);
	} else {
		if (_vala_strcmp0 (type_name, "any") == 0) {
			ValaDataType* _tmp2;
			ValaVoidType* _tmp1;
			_tmp2 = NULL;
			_tmp1 = NULL;
			type = (_tmp2 = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp1 = vala_void_type_new (NULL)), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp2);
			(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
		} else {
			if (_vala_strcmp0 (type_name, "GObject.Strv") == 0) {
				ValaDataType* _tmp5;
				ValaUnresolvedType* _tmp4;
				ValaUnresolvedSymbol* _tmp3;
				_tmp5 = NULL;
				_tmp4 = NULL;
				_tmp3 = NULL;
				type = (_tmp5 = (ValaDataType*) vala_array_type_new ((ValaDataType*) (_tmp4 = vala_unresolved_type_new_from_symbol (_tmp3 = vala_unresolved_symbol_new (NULL, "string", NULL), NULL)), 1, NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp5);
				(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
				(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL));
			} else {
				char** _tmp7;
				gint type_components_size;
				gint type_components_length1;
				char** _tmp6;
				char** type_components;
				if (_vala_strcmp0 (type_name, "utf8") == 0) {
					type_name = "string";
				} else {
					if (_vala_strcmp0 (type_name, "boolean") == 0) {
						type_name = "bool";
					} else {
						if (_vala_strcmp0 (type_name, "GLib.offset") == 0) {
							type_name = "int64";
						} else {
							if (_vala_strcmp0 (type_name, "GType") == 0) {
								type_name = "GLib.Type";
							} else {
								if (_vala_strcmp0 (type_name, "GObject.String") == 0) {
									type_name = "GLib.StringBuilder";
								} else {
									if (_vala_strcmp0 (type_name, "GObject.Class") == 0) {
										type_name = "GLib.ObjectClass";
									} else {
										if (_vala_strcmp0 (type_name, "GLib.unichar") == 0) {
											type_name = "unichar";
										} else {
											if (_vala_strcmp0 (type_name, "GLib.Data") == 0) {
												type_name = "GLib.Datalist";
											}
										}
									}
								}
							}
						}
					}
				}
				_tmp7 = NULL;
				_tmp6 = NULL;
				type_components = (_tmp7 = _tmp6 = g_strsplit (type_name, ".", 0), type_components_length1 = _vala_array_length (_tmp6), type_components_size = type_components_length1, _tmp7);
				if (type_components[1] != NULL) {
					char* namespace_name;
					const char* _tmp8;
					char* transformed_type_name;
					ValaDataType* _tmp11;
					ValaUnresolvedSymbol* _tmp10;
					ValaUnresolvedSymbol* _tmp9;
					/* namespaced name*/
					namespace_name = vala_gir_parser_transform_namespace_name (self, type_components[0]);
					_tmp8 = NULL;
					transformed_type_name = (_tmp8 = type_components[1], (_tmp8 == NULL) ? NULL : g_strdup (_tmp8));
					_tmp11 = NULL;
					_tmp10 = NULL;
					_tmp9 = NULL;
					type = (_tmp11 = (ValaDataType*) vala_unresolved_type_new_from_symbol (_tmp10 = vala_unresolved_symbol_new (_tmp9 = vala_unresolved_symbol_new (NULL, namespace_name, NULL), transformed_type_name, NULL), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp11);
					(_tmp10 == NULL) ? NULL : (_tmp10 = (vala_code_node_unref (_tmp10), NULL));
					(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
					namespace_name = (g_free (namespace_name), NULL);
					transformed_type_name = (g_free (transformed_type_name), NULL);
				} else {
					ValaDataType* _tmp13;
					ValaUnresolvedSymbol* _tmp12;
					_tmp13 = NULL;
					_tmp12 = NULL;
					type = (_tmp13 = (ValaDataType*) vala_unresolved_type_new_from_symbol (_tmp12 = vala_unresolved_symbol_new (NULL, type_name, NULL), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp13);
					(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL));
				}
				type_components = (_vala_array_free (type_components, type_components_length1, (GDestroyNotify) g_free), NULL);
			}
		}
	}
	return type;
}


static char* vala_gir_parser_transform_namespace_name (ValaGirParser* self, const char* gir_module_name) {
	const char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (gir_module_name != NULL, NULL);
	if (_vala_strcmp0 (gir_module_name, "GObject") == 0) {
		return g_strdup ("GLib");
	} else {
		if (_vala_strcmp0 (gir_module_name, "Gio") == 0) {
			return g_strdup ("GLib");
		}
	}
	_tmp2 = NULL;
	return (_tmp2 = gir_module_name, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


static ValaStruct* vala_gir_parser_parse_record (ValaGirParser* self) {
	ValaSourceReference* _tmp1;
	char* _tmp0;
	ValaStruct* _tmp2;
	ValaStruct* st;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "record");
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	st = (_tmp2 = vala_struct_new (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name"), _tmp1 = vala_gir_parser_get_current_src (self)), (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
	vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_next (self);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
			ValaField* _tmp3;
			_tmp3 = NULL;
			vala_struct_add_field (st, _tmp3 = vala_gir_parser_parse_field (self));
			(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "callback") == 0) {
				ValaDelegate* _tmp4;
				_tmp4 = NULL;
				_tmp4 = vala_gir_parser_parse_callback (self);
				(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constructor") == 0) {
					ValaMethod* _tmp5;
					_tmp5 = NULL;
					_tmp5 = vala_gir_parser_parse_constructor (self);
					(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
						ValaMethod* _tmp6;
						_tmp6 = NULL;
						vala_struct_add_method (st, _tmp6 = vala_gir_parser_parse_method (self, "method"));
						(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
					} else {
						char* _tmp8;
						ValaSourceReference* _tmp7;
						/* error*/
						_tmp8 = NULL;
						_tmp7 = NULL;
						vala_report_error (_tmp7 = vala_gir_parser_get_current_src (self), _tmp8 = g_strdup_printf ("unknown child element `%s' in `record'", vala_markup_reader_get_name (self->priv->reader)));
						_tmp8 = (g_free (_tmp8), NULL);
						(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL));
						break;
					}
				}
			}
		}
	}
	vala_gir_parser_end_element (self, "record");
	return st;
}


static ValaClass* vala_gir_parser_parse_class (ValaGirParser* self) {
	ValaSourceReference* _tmp1;
	char* _tmp0;
	ValaClass* _tmp2;
	ValaClass* cl;
	char* cname;
	char* parent;
	GeeArrayList* signals;
	GeeArrayList* methods;
	GeeArrayList* vmethods;
	GeeArrayList* fields;
	ValaClass* _tmp25;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "class");
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	cl = (_tmp2 = vala_class_new (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name"), _tmp1 = vala_gir_parser_get_current_src (self)), (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
	vala_symbol_set_access ((ValaSymbol*) cl, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	cname = vala_markup_reader_get_attribute (self->priv->reader, "c:type");
	if (cname != NULL) {
		vala_class_set_cname (cl, cname);
	}
	parent = vala_markup_reader_get_attribute (self->priv->reader, "parent");
	if (parent != NULL) {
		ValaDataType* _tmp3;
		_tmp3 = NULL;
		vala_class_add_base_type (cl, _tmp3 = vala_gir_parser_parse_type_from_name (self, parent));
		(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL));
	}
	vala_gir_parser_next (self);
	signals = gee_array_list_new (VALA_TYPE_SIGNAL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	methods = gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	vmethods = gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	fields = gee_array_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "implements") == 0) {
			ValaDataType* _tmp5;
			char* _tmp4;
			vala_gir_parser_start_element (self, "implements");
			_tmp5 = NULL;
			_tmp4 = NULL;
			vala_class_add_base_type (cl, _tmp5 = vala_gir_parser_parse_type_from_name (self, _tmp4 = vala_markup_reader_get_attribute (self->priv->reader, "name")));
			(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
			_tmp4 = (g_free (_tmp4), NULL);
			vala_gir_parser_next (self);
			vala_gir_parser_end_element (self, "implements");
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
				ValaField* _tmp6;
				_tmp6 = NULL;
				gee_collection_add ((GeeCollection*) fields, _tmp6 = vala_gir_parser_parse_field (self));
				(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "property") == 0) {
					ValaProperty* _tmp7;
					_tmp7 = NULL;
					vala_class_add_property (cl, _tmp7 = vala_gir_parser_parse_property (self));
					(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_code_node_unref (_tmp7), NULL));
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constructor") == 0) {
						ValaMethod* _tmp8;
						_tmp8 = NULL;
						vala_class_add_method (cl, _tmp8 = vala_gir_parser_parse_constructor (self));
						(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL));
					} else {
						if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "function") == 0) {
							ValaMethod* _tmp9;
							_tmp9 = NULL;
							gee_collection_add ((GeeCollection*) methods, _tmp9 = vala_gir_parser_parse_method (self, "function"));
							(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
						} else {
							if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
								ValaMethod* _tmp10;
								_tmp10 = NULL;
								gee_collection_add ((GeeCollection*) methods, _tmp10 = vala_gir_parser_parse_method (self, "method"));
								(_tmp10 == NULL) ? NULL : (_tmp10 = (vala_code_node_unref (_tmp10), NULL));
							} else {
								if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "callback") == 0) {
									ValaMethod* _tmp11;
									_tmp11 = NULL;
									gee_collection_add ((GeeCollection*) vmethods, _tmp11 = vala_gir_parser_parse_method (self, "callback"));
									(_tmp11 == NULL) ? NULL : (_tmp11 = (vala_code_node_unref (_tmp11), NULL));
								} else {
									if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "glib:signal") == 0) {
										ValaSignal* _tmp12;
										_tmp12 = NULL;
										gee_collection_add ((GeeCollection*) signals, _tmp12 = vala_gir_parser_parse_signal (self));
										(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL));
									} else {
										char* _tmp14;
										ValaSourceReference* _tmp13;
										/* error*/
										_tmp14 = NULL;
										_tmp13 = NULL;
										vala_report_error (_tmp13 = vala_gir_parser_get_current_src (self), _tmp14 = g_strdup_printf ("unknown child element `%s' in `class'", vala_markup_reader_get_name (self->priv->reader)));
										_tmp14 = (g_free (_tmp14), NULL);
										(_tmp13 == NULL) ? NULL : (_tmp13 = (vala_source_reference_unref (_tmp13), NULL));
										break;
									}
								}
							}
						}
					}
				}
			}
		}
	}
	/* signal merging*/
	{
		GeeIterator* _sig_it;
		/* signal merging*/
		_sig_it = gee_iterable_iterator ((GeeIterable*) signals);
		/* signal merging*/
		while (gee_iterator_next (_sig_it)) {
			ValaSignal* sig;
			ValaSymbol* symbol;
			/* signal merging*/
			sig = (ValaSignal*) gee_iterator_get (_sig_it);
			symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) cl), vala_symbol_get_name ((ValaSymbol*) sig));
			if (symbol == NULL) {
				vala_class_add_signal (cl, sig);
			} else {
				if (VALA_IS_PROPERTY (symbol)) {
				} else {
					char* _tmp16;
					ValaSourceReference* _tmp15;
					/* properties take precedence*/
					_tmp16 = NULL;
					_tmp15 = NULL;
					vala_report_error (_tmp15 = vala_gir_parser_get_current_src (self), _tmp16 = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name ((ValaSymbol*) sig), vala_symbol_get_name ((ValaSymbol*) cl)));
					_tmp16 = (g_free (_tmp16), NULL);
					(_tmp15 == NULL) ? NULL : (_tmp15 = (vala_source_reference_unref (_tmp15), NULL));
				}
			}
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_sig_it == NULL) ? NULL : (_sig_it = (gee_collection_object_unref (_sig_it), NULL));
	}
	/* virtual method merging*/
	{
		GeeIterator* _m_it;
		/* virtual method merging*/
		_m_it = gee_iterable_iterator ((GeeIterable*) vmethods);
		/* virtual method merging*/
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			ValaSymbol* symbol;
			/* virtual method merging*/
			m = (ValaMethod*) gee_iterator_get (_m_it);
			symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) cl), vala_symbol_get_name ((ValaSymbol*) m));
			if (symbol == NULL) {
				vala_class_add_method (cl, m);
			} else {
				if (VALA_IS_SIGNAL (symbol)) {
					ValaSignal* _tmp17;
					ValaSignal* sig;
					_tmp17 = NULL;
					sig = (_tmp17 = VALA_SIGNAL (symbol), (_tmp17 == NULL) ? NULL : vala_code_node_ref (_tmp17));
					vala_signal_set_is_virtual (sig, TRUE);
					(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				} else {
					gboolean _tmp18;
					_tmp18 = FALSE;
					if (VALA_IS_PROPERTY (symbol)) {
						_tmp18 = TRUE;
					} else {
						_tmp18 = VALA_IS_FIELD (symbol);
					}
					if (_tmp18) {
					} else {
						char* _tmp20;
						ValaSourceReference* _tmp19;
						/* assume method is getter for property/field ignore method*/
						_tmp20 = NULL;
						_tmp19 = NULL;
						vala_report_error (_tmp19 = vala_gir_parser_get_current_src (self), _tmp20 = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) cl)));
						_tmp20 = (g_free (_tmp20), NULL);
						(_tmp19 == NULL) ? NULL : (_tmp19 = (vala_source_reference_unref (_tmp19), NULL));
					}
				}
			}
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	/* method merging*/
	{
		GeeIterator* _m_it;
		/* method merging*/
		_m_it = gee_iterable_iterator ((GeeIterable*) methods);
		/* method merging*/
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			ValaSymbol* symbol;
			/* method merging*/
			m = (ValaMethod*) gee_iterator_get (_m_it);
			symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) cl), vala_symbol_get_name ((ValaSymbol*) m));
			if (symbol == NULL) {
				vala_class_add_method (cl, m);
			} else {
				if (VALA_IS_SIGNAL (symbol)) {
					ValaSignal* _tmp21;
					ValaSignal* sig;
					_tmp21 = NULL;
					sig = (_tmp21 = VALA_SIGNAL (symbol), (_tmp21 == NULL) ? NULL : vala_code_node_ref (_tmp21));
					vala_signal_set_has_emitter (sig, TRUE);
					(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				} else {
					gboolean _tmp22;
					_tmp22 = FALSE;
					if (VALA_IS_PROPERTY (symbol)) {
						_tmp22 = TRUE;
					} else {
						_tmp22 = VALA_IS_FIELD (symbol);
					}
					if (_tmp22) {
					} else {
						if (VALA_IS_METHOD (symbol)) {
						} else {
							char* _tmp24;
							ValaSourceReference* _tmp23;
							/* assume method is wrapper for virtual method*/
							_tmp24 = NULL;
							_tmp23 = NULL;
							vala_report_error (_tmp23 = vala_gir_parser_get_current_src (self), _tmp24 = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) cl)));
							_tmp24 = (g_free (_tmp24), NULL);
							(_tmp23 == NULL) ? NULL : (_tmp23 = (vala_source_reference_unref (_tmp23), NULL));
						}
					}
				}
			}
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	/* fields have lowest priority*/
	{
		GeeIterator* _f_it;
		/* fields have lowest priority*/
		_f_it = gee_iterable_iterator ((GeeIterable*) fields);
		/* fields have lowest priority*/
		while (gee_iterator_next (_f_it)) {
			ValaField* f;
			ValaSymbol* symbol;
			/* fields have lowest priority*/
			f = (ValaField*) gee_iterator_get (_f_it);
			symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) cl), vala_symbol_get_name ((ValaSymbol*) f));
			if (symbol == NULL) {
				vala_class_add_field (cl, f);
			}
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
	}
	vala_gir_parser_end_element (self, "class");
	_tmp25 = NULL;
	return (_tmp25 = cl, cname = (g_free (cname), NULL), parent = (g_free (parent), NULL), (signals == NULL) ? NULL : (signals = (gee_collection_object_unref (signals), NULL)), (methods == NULL) ? NULL : (methods = (gee_collection_object_unref (methods), NULL)), (vmethods == NULL) ? NULL : (vmethods = (gee_collection_object_unref (vmethods), NULL)), (fields == NULL) ? NULL : (fields = (gee_collection_object_unref (fields), NULL)), _tmp25);
}


static ValaInterface* vala_gir_parser_parse_interface (ValaGirParser* self) {
	ValaSourceReference* _tmp1;
	char* _tmp0;
	ValaInterface* _tmp2;
	ValaInterface* iface;
	char* cname;
	GeeArrayList* methods;
	GeeArrayList* vmethods;
	ValaInterface* _tmp19;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "interface");
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	iface = (_tmp2 = vala_interface_new (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name"), _tmp1 = vala_gir_parser_get_current_src (self)), (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
	vala_symbol_set_access ((ValaSymbol*) iface, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	cname = vala_markup_reader_get_attribute (self->priv->reader, "c:type");
	if (cname != NULL) {
		vala_interface_set_cname (iface, cname);
	}
	vala_gir_parser_next (self);
	methods = gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	vmethods = gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "prerequisite") == 0) {
			ValaDataType* _tmp4;
			char* _tmp3;
			vala_gir_parser_start_element (self, "prerequisite");
			_tmp4 = NULL;
			_tmp3 = NULL;
			vala_interface_add_prerequisite (iface, _tmp4 = vala_gir_parser_parse_type_from_name (self, _tmp3 = vala_markup_reader_get_attribute (self->priv->reader, "name")));
			(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
			_tmp3 = (g_free (_tmp3), NULL);
			vala_gir_parser_next (self);
			vala_gir_parser_end_element (self, "prerequisite");
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
				ValaField* _tmp5;
				_tmp5 = NULL;
				_tmp5 = vala_gir_parser_parse_field (self);
				(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "property") == 0) {
					ValaProperty* _tmp6;
					_tmp6 = NULL;
					vala_interface_add_property (iface, _tmp6 = vala_gir_parser_parse_property (self));
					(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
				} else {
					if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "callback") == 0) {
						ValaMethod* _tmp7;
						_tmp7 = NULL;
						gee_collection_add ((GeeCollection*) vmethods, _tmp7 = vala_gir_parser_parse_method (self, "callback"));
						(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_code_node_unref (_tmp7), NULL));
					} else {
						if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "function") == 0) {
							ValaMethod* _tmp8;
							_tmp8 = NULL;
							gee_collection_add ((GeeCollection*) methods, _tmp8 = vala_gir_parser_parse_method (self, "function"));
							(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL));
						} else {
							if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
								ValaMethod* _tmp9;
								_tmp9 = NULL;
								gee_collection_add ((GeeCollection*) methods, _tmp9 = vala_gir_parser_parse_method (self, "method"));
								(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
							} else {
								if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "glib:signal") == 0) {
									ValaSignal* _tmp10;
									_tmp10 = NULL;
									vala_interface_add_signal (iface, _tmp10 = vala_gir_parser_parse_signal (self));
									(_tmp10 == NULL) ? NULL : (_tmp10 = (vala_code_node_unref (_tmp10), NULL));
								} else {
									char* _tmp12;
									ValaSourceReference* _tmp11;
									/* error*/
									_tmp12 = NULL;
									_tmp11 = NULL;
									vala_report_error (_tmp11 = vala_gir_parser_get_current_src (self), _tmp12 = g_strdup_printf ("unknown child element `%s' in `interface'", vala_markup_reader_get_name (self->priv->reader)));
									_tmp12 = (g_free (_tmp12), NULL);
									(_tmp11 == NULL) ? NULL : (_tmp11 = (vala_source_reference_unref (_tmp11), NULL));
									break;
								}
							}
						}
					}
				}
			}
		}
	}
	/* virtual method merging*/
	{
		GeeIterator* _m_it;
		/* virtual method merging*/
		_m_it = gee_iterable_iterator ((GeeIterable*) vmethods);
		/* virtual method merging*/
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			ValaSymbol* symbol;
			/* virtual method merging*/
			m = (ValaMethod*) gee_iterator_get (_m_it);
			symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) iface), vala_symbol_get_name ((ValaSymbol*) m));
			if (symbol == NULL) {
				vala_interface_add_method (iface, m);
			} else {
				if (VALA_IS_SIGNAL (symbol)) {
					ValaSignal* _tmp13;
					ValaSignal* sig;
					_tmp13 = NULL;
					sig = (_tmp13 = VALA_SIGNAL (symbol), (_tmp13 == NULL) ? NULL : vala_code_node_ref (_tmp13));
					vala_signal_set_is_virtual (sig, TRUE);
					(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				} else {
					char* _tmp15;
					ValaSourceReference* _tmp14;
					_tmp15 = NULL;
					_tmp14 = NULL;
					vala_report_error (_tmp14 = vala_gir_parser_get_current_src (self), _tmp15 = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) iface)));
					_tmp15 = (g_free (_tmp15), NULL);
					(_tmp14 == NULL) ? NULL : (_tmp14 = (vala_source_reference_unref (_tmp14), NULL));
				}
			}
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	/* method merging*/
	{
		GeeIterator* _m_it;
		/* method merging*/
		_m_it = gee_iterable_iterator ((GeeIterable*) methods);
		/* method merging*/
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			ValaSymbol* symbol;
			/* method merging*/
			m = (ValaMethod*) gee_iterator_get (_m_it);
			symbol = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) iface), vala_symbol_get_name ((ValaSymbol*) m));
			if (symbol == NULL) {
				vala_interface_add_method (iface, m);
			} else {
				if (VALA_IS_SIGNAL (symbol)) {
					ValaSignal* _tmp16;
					ValaSignal* sig;
					_tmp16 = NULL;
					sig = (_tmp16 = VALA_SIGNAL (symbol), (_tmp16 == NULL) ? NULL : vala_code_node_ref (_tmp16));
					vala_signal_set_has_emitter (sig, TRUE);
					(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				} else {
					if (VALA_IS_METHOD (symbol)) {
					} else {
						char* _tmp18;
						ValaSourceReference* _tmp17;
						/* assume method is wrapper for virtual method*/
						_tmp18 = NULL;
						_tmp17 = NULL;
						vala_report_error (_tmp17 = vala_gir_parser_get_current_src (self), _tmp18 = g_strdup_printf ("duplicate member `%s' in `%s'", vala_symbol_get_name ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) iface)));
						_tmp18 = (g_free (_tmp18), NULL);
						(_tmp17 == NULL) ? NULL : (_tmp17 = (vala_source_reference_unref (_tmp17), NULL));
					}
				}
			}
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			(symbol == NULL) ? NULL : (symbol = (vala_code_node_unref (symbol), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	vala_gir_parser_end_element (self, "interface");
	_tmp19 = NULL;
	return (_tmp19 = iface, cname = (g_free (cname), NULL), (methods == NULL) ? NULL : (methods = (gee_collection_object_unref (methods), NULL)), (vmethods == NULL) ? NULL : (vmethods = (gee_collection_object_unref (vmethods), NULL)), _tmp19);
}


static ValaField* vala_gir_parser_parse_field (ValaGirParser* self) {
	char* name;
	ValaDataType* type;
	ValaSourceReference* _tmp0;
	ValaField* _tmp1;
	ValaField* field;
	ValaField* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "field");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	vala_gir_parser_next (self);
	type = vala_gir_parser_parse_type (self, NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	field = (_tmp1 = vala_field_new (name, type, NULL, _tmp0 = vala_gir_parser_get_current_src (self)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access ((ValaSymbol*) field, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_end_element (self, "field");
	_tmp2 = NULL;
	return (_tmp2 = field, name = (g_free (name), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp2);
}


static ValaProperty* vala_gir_parser_parse_property (ValaGirParser* self) {
	gint _tmp2_length1;
	char** _tmp2;
	char** _tmp1;
	char* _tmp0;
	char* _tmp3;
	char* name;
	char* readable;
	char* writable;
	char* construct_;
	ValaDataType* type;
	ValaSourceReference* _tmp4;
	ValaProperty* _tmp5;
	ValaProperty* prop;
	gboolean _tmp8;
	ValaProperty* _tmp11;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "property");
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp3 = NULL;
	name = (_tmp3 = g_strjoinv ("_", (_tmp2 = _tmp1 = g_strsplit (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name"), "-", 0), _tmp2_length1 = _vala_array_length (_tmp1), _tmp2)), _tmp2 = (_vala_array_free (_tmp2, _tmp2_length1, (GDestroyNotify) g_free), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp3);
	readable = vala_markup_reader_get_attribute (self->priv->reader, "readable");
	writable = vala_markup_reader_get_attribute (self->priv->reader, "writable");
	construct_ = vala_markup_reader_get_attribute (self->priv->reader, "construct");
	vala_gir_parser_next (self);
	type = vala_gir_parser_parse_type (self, NULL);
	_tmp4 = NULL;
	_tmp5 = NULL;
	prop = (_tmp5 = vala_property_new (name, type, NULL, NULL, _tmp4 = vala_gir_parser_get_current_src (self)), (_tmp4 == NULL) ? NULL : (_tmp4 = (vala_source_reference_unref (_tmp4), NULL)), _tmp5);
	vala_symbol_set_access ((ValaSymbol*) prop, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	if (_vala_strcmp0 (readable, "0") != 0) {
		ValaPropertyAccessor* _tmp7;
		ValaDataType* _tmp6;
		_tmp7 = NULL;
		_tmp6 = NULL;
		vala_property_set_get_accessor (prop, _tmp7 = vala_property_accessor_new (TRUE, FALSE, FALSE, _tmp6 = vala_data_type_copy (vala_property_get_property_type (prop)), NULL, NULL));
		(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_code_node_unref (_tmp7), NULL));
		(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
	}
	_tmp8 = FALSE;
	if (_vala_strcmp0 (writable, "1") == 0) {
		_tmp8 = TRUE;
	} else {
		_tmp8 = _vala_strcmp0 (construct_, "1") == 0;
	}
	if (_tmp8) {
		ValaPropertyAccessor* _tmp10;
		ValaDataType* _tmp9;
		_tmp10 = NULL;
		_tmp9 = NULL;
		vala_property_set_set_accessor (prop, _tmp10 = vala_property_accessor_new (FALSE, _vala_strcmp0 (writable, "1") == 0, _vala_strcmp0 (construct_, "1") == 0, _tmp9 = vala_data_type_copy (vala_property_get_property_type (prop)), NULL, NULL));
		(_tmp10 == NULL) ? NULL : (_tmp10 = (vala_code_node_unref (_tmp10), NULL));
		(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
	}
	vala_gir_parser_end_element (self, "property");
	_tmp11 = NULL;
	return (_tmp11 = prop, name = (g_free (name), NULL), readable = (g_free (readable), NULL), writable = (g_free (writable), NULL), construct_ = (g_free (construct_), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp11);
}


static ValaDelegate* vala_gir_parser_parse_callback (ValaGirParser* self) {
	char* name;
	ValaDataType* return_type;
	gboolean _tmp0;
	ValaSourceReference* _tmp3;
	ValaDelegate* _tmp4;
	ValaDelegate* d;
	gboolean _tmp5;
	ValaDelegate* _tmp7;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "callback");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	vala_gir_parser_next (self);
	return_type = NULL;
	_tmp0 = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp0 = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "return-value") == 0;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		ValaDataType* _tmp1;
		_tmp1 = NULL;
		return_type = (_tmp1 = vala_gir_parser_parse_return_value (self), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp1);
	} else {
		ValaDataType* _tmp2;
		_tmp2 = NULL;
		return_type = (_tmp2 = (ValaDataType*) vala_void_type_new (NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp2);
	}
	_tmp3 = NULL;
	_tmp4 = NULL;
	d = (_tmp4 = vala_delegate_new (name, return_type, _tmp3 = vala_gir_parser_get_current_src (self)), (_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)), _tmp4);
	vala_symbol_set_access ((ValaSymbol*) d, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp5 = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp5 = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0;
	} else {
		_tmp5 = FALSE;
	}
	if (_tmp5) {
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
			ValaFormalParameter* _tmp6;
			_tmp6 = NULL;
			vala_delegate_add_parameter (d, _tmp6 = vala_gir_parser_parse_parameter (self, NULL, NULL, NULL));
			(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	vala_gir_parser_end_element (self, "callback");
	_tmp7 = NULL;
	return (_tmp7 = d, name = (g_free (name), NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp7);
}


static ValaMethod* vala_gir_parser_parse_constructor (ValaGirParser* self) {
	char* name;
	ValaDataType* _tmp0;
	ValaSourceReference* _tmp1;
	ValaCreationMethod* _tmp2;
	ValaCreationMethod* m;
	gboolean _tmp3;
	ValaMethod* _tmp5;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "constructor");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	vala_gir_parser_next (self);
	_tmp0 = NULL;
	_tmp0 = vala_gir_parser_parse_return_value (self);
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL));
	_tmp1 = NULL;
	_tmp2 = NULL;
	m = (_tmp2 = vala_creation_method_new (NULL, name, _tmp1 = vala_gir_parser_get_current_src (self)), (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_source_reference_unref (_tmp1), NULL)), _tmp2);
	vala_symbol_set_access ((ValaSymbol*) m, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_method_set_has_construct_function ((ValaMethod*) m, FALSE);
	if (g_str_has_prefix (vala_symbol_get_name ((ValaSymbol*) m), "new_")) {
		vala_symbol_set_name ((ValaSymbol*) m, g_utf8_offset_to_pointer (vala_symbol_get_name ((ValaSymbol*) m), g_utf8_strlen ("new_", -1)));
	}
	_tmp3 = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp3 = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0;
	} else {
		_tmp3 = FALSE;
	}
	if (_tmp3) {
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
			ValaFormalParameter* _tmp4;
			_tmp4 = NULL;
			vala_method_add_parameter ((ValaMethod*) m, _tmp4 = vala_gir_parser_parse_parameter (self, NULL, NULL, NULL));
			(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	vala_gir_parser_end_element (self, "constructor");
	_tmp5 = NULL;
	return (_tmp5 = (ValaMethod*) m, name = (g_free (name), NULL), _tmp5);
}


static ValaMethod* vala_gir_parser_parse_method (ValaGirParser* self, const char* element_name) {
	char* name;
	char* throws_string;
	ValaDataType* return_type;
	gboolean _tmp0;
	ValaSourceReference* _tmp3;
	ValaMethod* _tmp4;
	ValaMethod* m;
	GeeArrayList* parameters;
	GeeArrayList* array_length_parameters;
	GeeArrayList* closure_parameters;
	GeeArrayList* destroy_parameters;
	gboolean _tmp5;
	gint i;
	ValaMethod* _tmp10;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (element_name != NULL, NULL);
	vala_gir_parser_start_element (self, element_name);
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	throws_string = vala_markup_reader_get_attribute (self->priv->reader, "throws");
	vala_gir_parser_next (self);
	return_type = NULL;
	_tmp0 = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp0 = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "return-value") == 0;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		ValaDataType* _tmp1;
		_tmp1 = NULL;
		return_type = (_tmp1 = vala_gir_parser_parse_return_value (self), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp1);
	} else {
		ValaDataType* _tmp2;
		_tmp2 = NULL;
		return_type = (_tmp2 = (ValaDataType*) vala_void_type_new (NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp2);
	}
	_tmp3 = NULL;
	_tmp4 = NULL;
	m = (_tmp4 = vala_method_new (name, return_type, _tmp3 = vala_gir_parser_get_current_src (self)), (_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL)), _tmp4);
	vala_symbol_set_access ((ValaSymbol*) m, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	if (_vala_strcmp0 (element_name, "callback") == 0) {
		vala_method_set_is_virtual (m, TRUE);
	} else {
		if (_vala_strcmp0 (element_name, "function") == 0) {
			vala_method_set_binding (m, MEMBER_BINDING_STATIC);
		}
	}
	parameters = gee_array_list_new (VALA_TYPE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	array_length_parameters = gee_array_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	closure_parameters = gee_array_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	destroy_parameters = gee_array_list_new (G_TYPE_INT, NULL, NULL, g_direct_equal);
	_tmp5 = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp5 = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0;
	} else {
		_tmp5 = FALSE;
	}
	if (_tmp5) {
		gboolean first;
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		first = TRUE;
		while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
			gint array_length_idx;
			gint closure_idx;
			gint destroy_idx;
			ValaFormalParameter* param;
			gboolean _tmp6;
			array_length_idx = 0;
			closure_idx = 0;
			destroy_idx = 0;
			param = vala_gir_parser_parse_parameter (self, &array_length_idx, &closure_idx, &destroy_idx);
			if (array_length_idx != (-1)) {
				gee_collection_add ((GeeCollection*) array_length_parameters, GINT_TO_POINTER (array_length_idx));
			}
			if (closure_idx != (-1)) {
				gee_collection_add ((GeeCollection*) closure_parameters, GINT_TO_POINTER (closure_idx));
			}
			if (destroy_idx != (-1)) {
				gee_collection_add ((GeeCollection*) destroy_parameters, GINT_TO_POINTER (destroy_idx));
			}
			_tmp6 = FALSE;
			if (_vala_strcmp0 (element_name, "callback") != 0) {
				_tmp6 = TRUE;
			} else {
				_tmp6 = !first;
			}
			/* first parameter is instance pointer in virtual methods, ignore*/
			if (_tmp6) {
				gee_collection_add ((GeeCollection*) parameters, param);
			} else {
				first = FALSE;
			}
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	i = 0;
	if (_vala_strcmp0 (element_name, "method") == 0) {
		/* implicit instance parameter*/
		i++;
	}
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) parameters);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			gboolean _tmp7;
			gboolean _tmp8;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			_tmp7 = FALSE;
			_tmp8 = FALSE;
			if (!gee_collection_contains ((GeeCollection*) array_length_parameters, GINT_TO_POINTER (i))) {
				_tmp8 = !gee_collection_contains ((GeeCollection*) closure_parameters, GINT_TO_POINTER (i));
			} else {
				_tmp8 = FALSE;
			}
			if (_tmp8) {
				_tmp7 = !gee_collection_contains ((GeeCollection*) destroy_parameters, GINT_TO_POINTER (i));
			} else {
				_tmp7 = FALSE;
			}
			if (_tmp7) {
				vala_method_add_parameter (m, param);
			}
			i++;
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	if (_vala_strcmp0 (throws_string, "1") == 0) {
		ValaErrorType* _tmp9;
		_tmp9 = NULL;
		vala_code_node_add_error_type ((ValaCodeNode*) m, (ValaDataType*) (_tmp9 = vala_error_type_new (NULL, NULL, NULL)));
		(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
	}
	vala_gir_parser_end_element (self, element_name);
	_tmp10 = NULL;
	return (_tmp10 = m, name = (g_free (name), NULL), throws_string = (g_free (throws_string), NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), (parameters == NULL) ? NULL : (parameters = (gee_collection_object_unref (parameters), NULL)), (array_length_parameters == NULL) ? NULL : (array_length_parameters = (gee_collection_object_unref (array_length_parameters), NULL)), (closure_parameters == NULL) ? NULL : (closure_parameters = (gee_collection_object_unref (closure_parameters), NULL)), (destroy_parameters == NULL) ? NULL : (destroy_parameters = (gee_collection_object_unref (destroy_parameters), NULL)), _tmp10);
}


static ValaSignal* vala_gir_parser_parse_signal (ValaGirParser* self) {
	gint _tmp2_length1;
	char** _tmp2;
	char** _tmp1;
	char* _tmp0;
	char* _tmp3;
	char* name;
	ValaDataType* return_type;
	gboolean _tmp4;
	ValaSignal* sig;
	gboolean _tmp7;
	ValaSignal* _tmp9;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "glib:signal");
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp3 = NULL;
	name = (_tmp3 = g_strjoinv ("_", (_tmp2 = _tmp1 = g_strsplit (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name"), "-", 0), _tmp2_length1 = _vala_array_length (_tmp1), _tmp2)), _tmp2 = (_vala_array_free (_tmp2, _tmp2_length1, (GDestroyNotify) g_free), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp3);
	vala_gir_parser_next (self);
	return_type = NULL;
	_tmp4 = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp4 = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "return-value") == 0;
	} else {
		_tmp4 = FALSE;
	}
	if (_tmp4) {
		ValaDataType* _tmp5;
		_tmp5 = NULL;
		return_type = (_tmp5 = vala_gir_parser_parse_return_value (self), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp5);
	} else {
		ValaDataType* _tmp6;
		_tmp6 = NULL;
		return_type = (_tmp6 = (ValaDataType*) vala_void_type_new (NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp6);
	}
	sig = vala_signal_new (name, return_type, NULL);
	vala_symbol_set_access ((ValaSymbol*) sig, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp7 = FALSE;
	if (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		_tmp7 = _vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "parameters") == 0;
	} else {
		_tmp7 = FALSE;
	}
	if (_tmp7) {
		vala_gir_parser_start_element (self, "parameters");
		vala_gir_parser_next (self);
		while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
			ValaFormalParameter* _tmp8;
			_tmp8 = NULL;
			vala_signal_add_parameter (sig, _tmp8 = vala_gir_parser_parse_parameter (self, NULL, NULL, NULL));
			(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL));
		}
		vala_gir_parser_end_element (self, "parameters");
	}
	vala_gir_parser_end_element (self, "glib:signal");
	_tmp9 = NULL;
	return (_tmp9 = sig, name = (g_free (name), NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp9);
}


static ValaStruct* vala_gir_parser_parse_boxed (ValaGirParser* self) {
	char* _tmp0;
	ValaStruct* _tmp1;
	ValaStruct* st;
	char* cname;
	ValaStruct* _tmp7;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "glib:boxed");
	_tmp0 = NULL;
	_tmp1 = NULL;
	st = (_tmp1 = vala_struct_new (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "glib:name"), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
	vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	cname = vala_markup_reader_get_attribute (self->priv->reader, "c:type");
	if (cname != NULL) {
		vala_struct_set_cname (st, cname);
	}
	vala_gir_parser_next (self);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
			ValaField* _tmp2;
			_tmp2 = NULL;
			vala_struct_add_field (st, _tmp2 = vala_gir_parser_parse_field (self));
			(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constructor") == 0) {
				ValaMethod* _tmp3;
				_tmp3 = NULL;
				_tmp3 = vala_gir_parser_parse_constructor (self);
				(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
					ValaMethod* _tmp4;
					_tmp4 = NULL;
					vala_struct_add_method (st, _tmp4 = vala_gir_parser_parse_method (self, "method"));
					(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
				} else {
					char* _tmp6;
					ValaSourceReference* _tmp5;
					/* error*/
					_tmp6 = NULL;
					_tmp5 = NULL;
					vala_report_error (_tmp5 = vala_gir_parser_get_current_src (self), _tmp6 = g_strdup_printf ("unknown child element `%s' in `class'", vala_markup_reader_get_name (self->priv->reader)));
					_tmp6 = (g_free (_tmp6), NULL);
					(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL));
					break;
				}
			}
		}
	}
	vala_gir_parser_end_element (self, "glib:boxed");
	_tmp7 = NULL;
	return (_tmp7 = st, cname = (g_free (cname), NULL), _tmp7);
}


static ValaStruct* vala_gir_parser_parse_union (ValaGirParser* self) {
	char* _tmp0;
	ValaStruct* _tmp1;
	ValaStruct* st;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "union");
	_tmp0 = NULL;
	_tmp1 = NULL;
	st = (_tmp1 = vala_struct_new (_tmp0 = vala_markup_reader_get_attribute (self->priv->reader, "name"), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
	vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_next (self);
	while (self->priv->current_token == VALA_MARKUP_TOKEN_TYPE_START_ELEMENT) {
		if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "field") == 0) {
			ValaField* _tmp2;
			_tmp2 = NULL;
			vala_struct_add_field (st, _tmp2 = vala_gir_parser_parse_field (self));
			(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL));
		} else {
			if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "constructor") == 0) {
				ValaMethod* _tmp3;
				_tmp3 = NULL;
				_tmp3 = vala_gir_parser_parse_constructor (self);
				(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL));
			} else {
				if (_vala_strcmp0 (vala_markup_reader_get_name (self->priv->reader), "method") == 0) {
					ValaMethod* _tmp4;
					_tmp4 = NULL;
					vala_struct_add_method (st, _tmp4 = vala_gir_parser_parse_method (self, "method"));
					(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
				} else {
					char* _tmp6;
					ValaSourceReference* _tmp5;
					/* error*/
					_tmp6 = NULL;
					_tmp5 = NULL;
					vala_report_error (_tmp5 = vala_gir_parser_get_current_src (self), _tmp6 = g_strdup_printf ("unknown child element `%s' in `union'", vala_markup_reader_get_name (self->priv->reader)));
					_tmp6 = (g_free (_tmp6), NULL);
					(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL));
					break;
				}
			}
		}
	}
	vala_gir_parser_end_element (self, "union");
	return st;
}


static ValaConstant* vala_gir_parser_parse_constant (ValaGirParser* self) {
	char* name;
	ValaDataType* type;
	ValaSourceReference* _tmp0;
	ValaConstant* _tmp1;
	ValaConstant* c;
	ValaConstant* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	vala_gir_parser_start_element (self, "constant");
	name = vala_markup_reader_get_attribute (self->priv->reader, "name");
	vala_gir_parser_next (self);
	type = vala_gir_parser_parse_type (self, NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	c = (_tmp1 = vala_constant_new (name, type, NULL, _tmp0 = vala_gir_parser_get_current_src (self)), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access ((ValaSymbol*) c, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_gir_parser_end_element (self, "constant");
	_tmp2 = NULL;
	return (_tmp2 = c, name = (g_free (name), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp2);
}


void vala_gir_parser_parse_metadata (ValaGirParser* self, const char* metadata_filename) {
	GError * inner_error;
	g_return_if_fail (self != NULL);
	g_return_if_fail (metadata_filename != NULL);
	inner_error = NULL;
	if (g_file_test (metadata_filename, G_FILE_TEST_EXISTS)) {
		{
			char* metadata;
			char* _tmp2;
			gboolean _tmp1;
			char* _tmp0;
			metadata = NULL;
			_tmp2 = NULL;
			_tmp0 = NULL;
			_tmp1 = g_file_get_contents (metadata_filename, &_tmp0, NULL, &inner_error);
			metadata = (_tmp2 = _tmp0, metadata = (g_free (metadata), NULL), _tmp2);
			_tmp1;
			if (inner_error != NULL) {
				metadata = (g_free (metadata), NULL);
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch2_g_file_error;
				}
				goto __finally2;
			}
			{
				char** _tmp3;
				char** line_collection;
				int line_collection_length1;
				int line_it;
				_tmp3 = NULL;
				line_collection = _tmp3 = g_strsplit (metadata, "\n", 0);
				line_collection_length1 = _vala_array_length (_tmp3);
				for (line_it = 0; line_it < _vala_array_length (_tmp3); line_it = line_it + 1) {
					const char* _tmp12;
					char* line;
					_tmp12 = NULL;
					line = (_tmp12 = line_collection[line_it], (_tmp12 == NULL) ? NULL : g_strdup (_tmp12));
					{
						char** _tmp5;
						gint tokens_size;
						gint tokens_length1;
						char** _tmp4;
						char** tokens;
						if (g_str_has_prefix (line, "#")) {
							/* ignore comment lines*/
							line = (g_free (line), NULL);
							continue;
						}
						_tmp5 = NULL;
						_tmp4 = NULL;
						tokens = (_tmp5 = _tmp4 = g_strsplit (line, " ", 2), tokens_length1 = _vala_array_length (_tmp4), tokens_size = tokens_length1, _tmp5);
						if (NULL == tokens[0]) {
							line = (g_free (line), NULL);
							tokens = (_vala_array_free (tokens, tokens_length1, (GDestroyNotify) g_free), NULL);
							continue;
						}
						{
							char** _tmp6;
							char** attribute_collection;
							int attribute_collection_length1;
							int attribute_it;
							_tmp6 = NULL;
							attribute_collection = _tmp6 = g_strsplit (tokens[1], " ", 0);
							attribute_collection_length1 = _vala_array_length (_tmp6);
							for (attribute_it = 0; attribute_it < _vala_array_length (_tmp6); attribute_it = attribute_it + 1) {
								const char* _tmp11;
								char* attribute;
								_tmp11 = NULL;
								attribute = (_tmp11 = attribute_collection[attribute_it], (_tmp11 == NULL) ? NULL : g_strdup (_tmp11));
								{
									char** _tmp8;
									gint pair_size;
									gint pair_length1;
									char** _tmp7;
									char** pair;
									gboolean _tmp9;
									char* key;
									char* _tmp10;
									_tmp8 = NULL;
									_tmp7 = NULL;
									pair = (_tmp8 = _tmp7 = g_strsplit (attribute, "=", 2), pair_length1 = _vala_array_length (_tmp7), pair_size = pair_length1, _tmp8);
									_tmp9 = FALSE;
									if (pair[0] == NULL) {
										_tmp9 = TRUE;
									} else {
										_tmp9 = pair[1] == NULL;
									}
									if (_tmp9) {
										attribute = (g_free (attribute), NULL);
										pair = (_vala_array_free (pair, pair_length1, (GDestroyNotify) g_free), NULL);
										continue;
									}
									key = g_strdup_printf ("%s/@%s", tokens[0], pair[0]);
									_tmp10 = NULL;
									gee_map_set ((GeeMap*) self->priv->attributes_map, key, _tmp10 = string_substring (pair[1], (glong) 1, string_get_length (pair[1]) - 2));
									_tmp10 = (g_free (_tmp10), NULL);
									attribute = (g_free (attribute), NULL);
									pair = (_vala_array_free (pair, pair_length1, (GDestroyNotify) g_free), NULL);
									key = (g_free (key), NULL);
								}
							}
							attribute_collection = (_vala_array_free (attribute_collection, attribute_collection_length1, (GDestroyNotify) g_free), NULL);
						}
						line = (g_free (line), NULL);
						tokens = (_vala_array_free (tokens, tokens_length1, (GDestroyNotify) g_free), NULL);
					}
				}
				line_collection = (_vala_array_free (line_collection, line_collection_length1, (GDestroyNotify) g_free), NULL);
			}
			metadata = (g_free (metadata), NULL);
		}
		goto __finally2;
		__catch2_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* _tmp13;
				_tmp13 = NULL;
				vala_report_error (NULL, _tmp13 = g_strdup_printf ("Unable to read metadata file: %s", e->message));
				_tmp13 = (g_free (_tmp13), NULL);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally2:
		if (inner_error != NULL) {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return;
		}
	} else {
		char* _tmp14;
		_tmp14 = NULL;
		vala_report_error (NULL, _tmp14 = g_strdup_printf ("Metadata file `%s' not found", metadata_filename));
		_tmp14 = (g_free (_tmp14), NULL);
	}
}


static char* vala_gir_parser_get_attribute (ValaGirParser* self, const char* node, const char* key) {
	char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (node != NULL, NULL);
	g_return_val_if_fail (key != NULL, NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = (char*) gee_map_get ((GeeMap*) self->priv->attributes_map, _tmp0 = g_strdup_printf ("%s/@%s", node, key)), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
}


/**
 * Code visitor parsing all Vala source files.
 */
ValaGirParser* vala_gir_parser_construct (GType object_type) {
	ValaGirParser* self;
	self = (ValaGirParser*) g_type_create_instance (object_type);
	return self;
}


ValaGirParser* vala_gir_parser_new (void) {
	return vala_gir_parser_construct (VALA_TYPE_GIR_PARSER);
}


const char* vala_gir_parser_get_package_name (ValaGirParser* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_package_name;
}


static void vala_gir_parser_set_package_name (ValaGirParser* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_package_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_package_name = (g_free (self->priv->_package_name), NULL), _tmp2);
}


static void vala_gir_parser_class_init (ValaGirParserClass * klass) {
	vala_gir_parser_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_gir_parser_finalize;
	g_type_class_add_private (klass, sizeof (ValaGirParserPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_gir_parser_real_visit_source_file;
}


static void vala_gir_parser_instance_init (ValaGirParser * self) {
	self->priv = VALA_GIR_PARSER_GET_PRIVATE (self);
	self->priv->attributes_map = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
}


static void vala_gir_parser_finalize (ValaCodeVisitor* obj) {
	ValaGirParser * self;
	self = VALA_GIR_PARSER (obj);
	self->priv->_package_name = (g_free (self->priv->_package_name), NULL);
	(self->priv->reader == NULL) ? NULL : (self->priv->reader = (g_object_unref (self->priv->reader), NULL));
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
	(self->priv->glib_ns == NULL) ? NULL : (self->priv->glib_ns = (vala_code_node_unref (self->priv->glib_ns), NULL));
	(self->priv->current_source_file == NULL) ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL));
	self->priv->cheader_filenames = (_vala_array_free (self->priv->cheader_filenames, self->priv->cheader_filenames_length1, (GDestroyNotify) g_free), NULL);
	(self->priv->attributes_map == NULL) ? NULL : (self->priv->attributes_map = (gee_collection_object_unref (self->priv->attributes_map), NULL));
	VALA_CODE_VISITOR_CLASS (vala_gir_parser_parent_class)->finalize (obj);
}


GType vala_gir_parser_get_type (void) {
	static GType vala_gir_parser_type_id = 0;
	if (vala_gir_parser_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGirParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gir_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGirParser), 0, (GInstanceInitFunc) vala_gir_parser_instance_init, NULL };
		vala_gir_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGirParser", &g_define_type_info, 0);
	}
	return vala_gir_parser_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




