/* valageniescanner.vala
 *
 * Copyright (C) 2008  Jamie McCracken, Jürg Billeter
 * Based on code by Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jamie McCracken jamiemcc gnome org
 */

#include <vala/valageniescanner.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <gobject/gvaluecollector.h>




struct _ValaGenieScannerPrivate {
	ValaSourceFile* _source_file;
	gint _indent_spaces;
	gchar* begin;
	gchar* current;
	gchar* end;
	gint line;
	gint column;
	gint current_indent_level;
	gint indent_level;
	gint pending_dedents;
	ValaGenieTokenType last_token;
	gboolean parse_started;
	char* _comment;
};

#define VALA_GENIE_SCANNER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_GENIE_TYPE_SCANNER, ValaGenieScannerPrivate))
enum  {
	VALA_GENIE_SCANNER_DUMMY_PROPERTY
};
static gboolean vala_genie_scanner_is_ident_char (ValaGenieScanner* self, gchar c);
static ValaGenieTokenType vala_genie_scanner_get_identifier_or_keyword (ValaGenieScanner* self, gchar* begin, gint len);
static gint vala_genie_scanner_count_tabs (ValaGenieScanner* self);
static gboolean vala_genie_scanner_matches (ValaGenieScanner* self, gchar* begin, const char* keyword);
static gboolean vala_genie_scanner_whitespace (ValaGenieScanner* self);
static inline gboolean vala_genie_scanner_newline (ValaGenieScanner* self);
static gboolean vala_genie_scanner_skip_newlines (ValaGenieScanner* self);
static gboolean vala_genie_scanner_comment (ValaGenieScanner* self);
static gboolean vala_genie_scanner_skip_tabs (ValaGenieScanner* self);
static void vala_genie_scanner_skip_space_tabs (ValaGenieScanner* self);
static void vala_genie_scanner_space (ValaGenieScanner* self);
static void vala_genie_scanner_push_comment (ValaGenieScanner* self, const char* comment_item, gboolean file_comment);
static void vala_genie_scanner_set_source_file (ValaGenieScanner* self, ValaSourceFile* value);
static gpointer vala_genie_scanner_parent_class = NULL;
static void vala_genie_scanner_finalize (ValaGenieScanner* obj);



ValaGenieScanner* vala_genie_scanner_construct (GType object_type, ValaSourceFile* source_file) {
	ValaGenieScanner* self;
	g_return_val_if_fail (source_file != NULL, NULL);
	self = (ValaGenieScanner*) g_type_create_instance (object_type);
	vala_genie_scanner_set_source_file (self, source_file);
	self->priv->begin = vala_source_file_get_mapped_contents (source_file);
	self->priv->end = self->priv->begin + vala_source_file_get_mapped_length (source_file);
	self->priv->current = self->priv->begin;
	self->priv->_indent_spaces = 0;
	self->priv->line = 1;
	self->priv->column = 1;
	self->priv->current_indent_level = 0;
	self->priv->indent_level = 0;
	self->priv->pending_dedents = 0;
	self->priv->parse_started = FALSE;
	self->priv->last_token = VALA_GENIE_TOKEN_TYPE_NONE;
	return self;
}


ValaGenieScanner* vala_genie_scanner_new (ValaSourceFile* source_file) {
	return vala_genie_scanner_construct (VALA_GENIE_TYPE_SCANNER, source_file);
}


static gboolean vala_genie_scanner_is_ident_char (ValaGenieScanner* self, gchar c) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0 = FALSE;
	if (g_ascii_isalnum (c)) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = c == '_';
	}
	return _tmp0;
}


static ValaGenieTokenType vala_genie_scanner_get_identifier_or_keyword (ValaGenieScanner* self, gchar* begin, gint len) {
	g_return_val_if_fail (self != NULL, 0);
	switch (len) {
		case 2:
		{
			switch (begin[0]) {
				case 'a':
				{
					if (vala_genie_scanner_matches (self, begin, "as")) {
						return VALA_GENIE_TOKEN_TYPE_AS;
					}
					break;
				}
				case 'd':
				{
					if (vala_genie_scanner_matches (self, begin, "do")) {
						return VALA_GENIE_TOKEN_TYPE_DO;
					}
					break;
				}
				case 'i':
				{
					switch (begin[1]) {
						case 'f':
						{
							return VALA_GENIE_TOKEN_TYPE_IF;
						}
						case 'n':
						{
							return VALA_GENIE_TOKEN_TYPE_IN;
						}
						case 's':
						{
							return VALA_GENIE_TOKEN_TYPE_IS;
						}
					}
					break;
				}
				case 'o':
				{
					if (vala_genie_scanner_matches (self, begin, "of")) {
						return VALA_GENIE_TOKEN_TYPE_OF;
					}
					if (vala_genie_scanner_matches (self, begin, "or")) {
						return VALA_GENIE_TOKEN_TYPE_OP_OR;
					}
					break;
				}
				case 't':
				{
					if (vala_genie_scanner_matches (self, begin, "to")) {
						return VALA_GENIE_TOKEN_TYPE_TO;
					}
					break;
				}
			}
			break;
		}
		case 3:
		{
			switch (begin[0]) {
				case 'a':
				{
					if (vala_genie_scanner_matches (self, begin, "and")) {
						return VALA_GENIE_TOKEN_TYPE_OP_AND;
					}
					break;
				}
				case 'd':
				{
					if (vala_genie_scanner_matches (self, begin, "def")) {
						return VALA_GENIE_TOKEN_TYPE_DEF;
					}
					break;
				}
				case 'f':
				{
					if (vala_genie_scanner_matches (self, begin, "for")) {
						return VALA_GENIE_TOKEN_TYPE_FOR;
					}
					break;
				}
				case 'g':
				{
					if (vala_genie_scanner_matches (self, begin, "get")) {
						return VALA_GENIE_TOKEN_TYPE_GET;
					}
					break;
				}
				case 'i':
				{
					if (vala_genie_scanner_matches (self, begin, "isa")) {
						return VALA_GENIE_TOKEN_TYPE_ISA;
					}
					break;
				}
				case 'n':
				{
					switch (begin[1]) {
						case 'e':
						{
							if (vala_genie_scanner_matches (self, begin, "new")) {
								return VALA_GENIE_TOKEN_TYPE_NEW;
							}
							break;
						}
						case 'o':
						{
							if (vala_genie_scanner_matches (self, begin, "not")) {
								return VALA_GENIE_TOKEN_TYPE_OP_NEG;
							}
							break;
						}
					}
					break;
				}
				case 'o':
				{
					if (vala_genie_scanner_matches (self, begin, "out")) {
						return VALA_GENIE_TOKEN_TYPE_OUT;
					}
					break;
				}
				case 'r':
				{
					if (vala_genie_scanner_matches (self, begin, "ref")) {
						return VALA_GENIE_TOKEN_TYPE_REF;
					}
					break;
				}
				case 's':
				{
					if (vala_genie_scanner_matches (self, begin, "set")) {
						return VALA_GENIE_TOKEN_TYPE_SET;
					}
					break;
				}
				case 't':
				{
					if (vala_genie_scanner_matches (self, begin, "try")) {
						return VALA_GENIE_TOKEN_TYPE_TRY;
					}
					break;
				}
				case 'v':
				{
					if (vala_genie_scanner_matches (self, begin, "var")) {
						return VALA_GENIE_TOKEN_TYPE_VAR;
					}
					break;
				}
			}
			break;
		}
		case 4:
		{
			switch (begin[0]) {
				case 'c':
				{
					if (vala_genie_scanner_matches (self, begin, "case")) {
						return VALA_GENIE_TOKEN_TYPE_CASE;
					}
					break;
				}
				case 'd':
				{
					if (vala_genie_scanner_matches (self, begin, "dict")) {
						return VALA_GENIE_TOKEN_TYPE_DICT;
					}
					break;
				}
				case 'e':
				{
					switch (begin[1]) {
						case 'l':
						{
							if (vala_genie_scanner_matches (self, begin, "else")) {
								return VALA_GENIE_TOKEN_TYPE_ELSE;
							}
							break;
						}
						case 'n':
						{
							if (vala_genie_scanner_matches (self, begin, "enum")) {
								return VALA_GENIE_TOKEN_TYPE_ENUM;
							}
							break;
						}
					}
					break;
				}
				case 'i':
				{
					if (vala_genie_scanner_matches (self, begin, "init")) {
						return VALA_GENIE_TOKEN_TYPE_INIT;
					}
					break;
				}
				case 'l':
				{
					switch (begin[1]) {
						case 'i':
						{
							if (vala_genie_scanner_matches (self, begin, "list")) {
								return VALA_GENIE_TOKEN_TYPE_LIST;
							}
							break;
						}
						case 'o':
						{
							if (vala_genie_scanner_matches (self, begin, "lock")) {
								return VALA_GENIE_TOKEN_TYPE_LOCK;
							}
							break;
						}
					}
					break;
				}
				case 'n':
				{
					if (vala_genie_scanner_matches (self, begin, "null")) {
						return VALA_GENIE_TOKEN_TYPE_NULL;
					}
					break;
				}
				case 'p':
				{
					switch (begin[1]) {
						case 'a':
						{
							if (vala_genie_scanner_matches (self, begin, "pass")) {
								return VALA_GENIE_TOKEN_TYPE_PASS;
							}
							break;
						}
						case 'r':
						{
							if (vala_genie_scanner_matches (self, begin, "prop")) {
								return VALA_GENIE_TOKEN_TYPE_PROP;
							}
							break;
						}
					}
					break;
				}
				case 's':
				{
					if (vala_genie_scanner_matches (self, begin, "self")) {
						return VALA_GENIE_TOKEN_TYPE_THIS;
					}
					break;
				}
				case 't':
				{
					if (vala_genie_scanner_matches (self, begin, "true")) {
						return VALA_GENIE_TOKEN_TYPE_TRUE;
					}
					break;
				}
				case 'u':
				{
					if (vala_genie_scanner_matches (self, begin, "uses")) {
						return VALA_GENIE_TOKEN_TYPE_USES;
					}
					break;
				}
				case 'v':
				{
					if (vala_genie_scanner_matches (self, begin, "void")) {
						return VALA_GENIE_TOKEN_TYPE_VOID;
					}
					break;
				}
				case 'w':
				{
					switch (begin[1]) {
						case 'e':
						{
							if (vala_genie_scanner_matches (self, begin, "weak")) {
								return VALA_GENIE_TOKEN_TYPE_WEAK;
							}
							break;
						}
						case 'h':
						{
							if (vala_genie_scanner_matches (self, begin, "when")) {
								return VALA_GENIE_TOKEN_TYPE_WHEN;
							}
							break;
						}
					}
					break;
				}
			}
			break;
		}
		case 5:
		{
			switch (begin[0]) {
				case 'a':
				{
					if (vala_genie_scanner_matches (self, begin, "array")) {
						return VALA_GENIE_TOKEN_TYPE_ARRAY;
					}
					break;
				}
				case 'b':
				{
					if (vala_genie_scanner_matches (self, begin, "break")) {
						return VALA_GENIE_TOKEN_TYPE_BREAK;
					}
					break;
				}
				case 'c':
				{
					switch (begin[1]) {
						case 'l':
						{
							if (vala_genie_scanner_matches (self, begin, "class")) {
								return VALA_GENIE_TOKEN_TYPE_CLASS;
							}
							break;
						}
						case 'o':
						{
							if (vala_genie_scanner_matches (self, begin, "const")) {
								return VALA_GENIE_TOKEN_TYPE_CONST;
							}
							break;
						}
					}
					break;
				}
				case 'e':
				{
					if (vala_genie_scanner_matches (self, begin, "event")) {
						return VALA_GENIE_TOKEN_TYPE_EVENT;
					}
					break;
				}
				case 'f':
				{
					switch (begin[1]) {
						case 'a':
						{
							if (vala_genie_scanner_matches (self, begin, "false")) {
								return VALA_GENIE_TOKEN_TYPE_FALSE;
							}
							break;
						}
						case 'i':
						{
							if (vala_genie_scanner_matches (self, begin, "final")) {
								return VALA_GENIE_TOKEN_TYPE_FINAL;
							}
							break;
						}
					}
					break;
				}
				case 'p':
				{
					if (vala_genie_scanner_matches (self, begin, "print")) {
						return VALA_GENIE_TOKEN_TYPE_PRINT;
					}
					break;
				}
				case 's':
				{
					if (vala_genie_scanner_matches (self, begin, "super")) {
						return VALA_GENIE_TOKEN_TYPE_SUPER;
					}
					break;
				}
				case 'r':
				{
					if (vala_genie_scanner_matches (self, begin, "raise")) {
						return VALA_GENIE_TOKEN_TYPE_RAISE;
					}
					break;
				}
				case 'w':
				{
					if (vala_genie_scanner_matches (self, begin, "while")) {
						return VALA_GENIE_TOKEN_TYPE_WHILE;
					}
					break;
				}
			}
			break;
		}
		case 6:
		{
			switch (begin[0]) {
				case 'a':
				{
					if (vala_genie_scanner_matches (self, begin, "assert")) {
						return VALA_GENIE_TOKEN_TYPE_ASSERT;
					}
					break;
				}
				case 'd':
				{
					switch (begin[1]) {
						case 'e':
						{
							if (vala_genie_scanner_matches (self, begin, "delete")) {
								return VALA_GENIE_TOKEN_TYPE_DELETE;
							}
							break;
						}
						case 'o':
						{
							if (vala_genie_scanner_matches (self, begin, "downto")) {
								return VALA_GENIE_TOKEN_TYPE_DOWNTO;
							}
							break;
						}
					}
					break;
				}
				case 'e':
				{
					switch (begin[1]) {
						case 'x':
						{
							switch (begin[2]) {
								case 'c':
								{
									if (vala_genie_scanner_matches (self, begin, "except")) {
										return VALA_GENIE_TOKEN_TYPE_EXCEPT;
									}
									break;
								}
								case 't':
								{
									if (vala_genie_scanner_matches (self, begin, "extern")) {
										return VALA_GENIE_TOKEN_TYPE_EXTERN;
									}
									break;
								}
							}
							break;
						}
					}
					break;
				}
				case 'i':
				{
					if (vala_genie_scanner_matches (self, begin, "inline")) {
						return VALA_GENIE_TOKEN_TYPE_INLINE;
					}
					break;
				}
				case 'p':
				{
					if (vala_genie_scanner_matches (self, begin, "public")) {
						return VALA_GENIE_TOKEN_TYPE_PUBLIC;
					}
					break;
				}
				case 'r':
				{
					switch (begin[1]) {
						case 'a':
						{
							if (vala_genie_scanner_matches (self, begin, "raises")) {
								return VALA_GENIE_TOKEN_TYPE_RAISES;
							}
							break;
						}
						case 'e':
						{
							if (vala_genie_scanner_matches (self, begin, "return")) {
								return VALA_GENIE_TOKEN_TYPE_RETURN;
							}
							break;
						}
					}
					break;
				}
				case 's':
				{
					switch (begin[1]) {
						case 'i':
						{
							if (vala_genie_scanner_matches (self, begin, "sizeof")) {
								return VALA_GENIE_TOKEN_TYPE_SIZEOF;
							}
							break;
						}
						case 't':
						{
							switch (begin[2]) {
								case 'a':
								{
									if (vala_genie_scanner_matches (self, begin, "static")) {
										return VALA_GENIE_TOKEN_TYPE_STATIC;
									}
									break;
								}
								case 'r':
								{
									if (vala_genie_scanner_matches (self, begin, "struct")) {
										return VALA_GENIE_TOKEN_TYPE_STRUCT;
									}
									break;
								}
							}
							break;
						}
					}
					break;
				}
				case 't':
				{
					if (vala_genie_scanner_matches (self, begin, "typeof")) {
						return VALA_GENIE_TOKEN_TYPE_TYPEOF;
					}
					break;
				}
			}
			break;
		}
		case 7:
		{
			switch (begin[0]) {
				case 'd':
				{
					switch (begin[1]) {
						case 'e':
						{
							if (vala_genie_scanner_matches (self, begin, "default")) {
								return VALA_GENIE_TOKEN_TYPE_DEFAULT;
							}
							break;
						}
						case 'y':
						{
							if (vala_genie_scanner_matches (self, begin, "dynamic")) {
								return VALA_GENIE_TOKEN_TYPE_DYNAMIC;
							}
							break;
						}
					}
					break;
				}
				case 'e':
				{
					if (vala_genie_scanner_matches (self, begin, "ensures")) {
						return VALA_GENIE_TOKEN_TYPE_ENSURES;
					}
					break;
				}
				case 'f':
				{
					switch (begin[1]) {
						case 'i':
						{
							if (vala_genie_scanner_matches (self, begin, "finally")) {
								return VALA_GENIE_TOKEN_TYPE_FINALLY;
							}
							break;
						}
						case 'o':
						{
							if (vala_genie_scanner_matches (self, begin, "foreach")) {
								return VALA_GENIE_TOKEN_TYPE_FOREACH;
							}
							break;
						}
					}
					break;
				}
				case 'p':
				{
					if (vala_genie_scanner_matches (self, begin, "private")) {
						return VALA_GENIE_TOKEN_TYPE_PRIVATE;
					}
					break;
				}
				case 'v':
				{
					if (vala_genie_scanner_matches (self, begin, "virtual")) {
						return VALA_GENIE_TOKEN_TYPE_VIRTUAL;
					}
					break;
				}
			}
			break;
		}
		case 8:
		{
			switch (begin[0]) {
				case 'a':
				{
					if (vala_genie_scanner_matches (self, begin, "abstract")) {
						return VALA_GENIE_TOKEN_TYPE_ABSTRACT;
					}
					break;
				}
				case 'c':
				{
					if (vala_genie_scanner_matches (self, begin, "continue")) {
						return VALA_GENIE_TOKEN_TYPE_CONTINUE;
					}
					break;
				}
				case 'd':
				{
					if (vala_genie_scanner_matches (self, begin, "delegate")) {
						return VALA_GENIE_TOKEN_TYPE_DELEGATE;
					}
					break;
				}
				case 'o':
				{
					if (vala_genie_scanner_matches (self, begin, "override")) {
						return VALA_GENIE_TOKEN_TYPE_OVERRIDE;
					}
					break;
				}
				case 'r':
				{
					switch (begin[2]) {
						case 'a':
						{
							if (vala_genie_scanner_matches (self, begin, "readonly")) {
								return VALA_GENIE_TOKEN_TYPE_READONLY;
							}
							break;
						}
						case 'q':
						{
							if (vala_genie_scanner_matches (self, begin, "requires")) {
								return VALA_GENIE_TOKEN_TYPE_REQUIRES;
							}
							break;
						}
					}
					break;
				}
				case 'v':
				{
					if (vala_genie_scanner_matches (self, begin, "volatile")) {
						return VALA_GENIE_TOKEN_TYPE_VOLATILE;
					}
					break;
				}
			}
			break;
		}
		case 9:
		{
			switch (begin[0]) {
				case 'c':
				{
					if (vala_genie_scanner_matches (self, begin, "construct")) {
						return VALA_GENIE_TOKEN_TYPE_CONSTRUCT;
					}
					break;
				}
				case 'e':
				{
					if (vala_genie_scanner_matches (self, begin, "exception")) {
						return VALA_GENIE_TOKEN_TYPE_ERRORDOMAIN;
					}
					break;
				}
				case 'i':
				{
					if (vala_genie_scanner_matches (self, begin, "interface")) {
						return VALA_GENIE_TOKEN_TYPE_INTERFACE;
					}
					break;
				}
				case 'n':
				{
					if (vala_genie_scanner_matches (self, begin, "namespace")) {
						return VALA_GENIE_TOKEN_TYPE_NAMESPACE;
					}
					break;
				}
				case 'p':
				{
					if (vala_genie_scanner_matches (self, begin, "protected")) {
						return VALA_GENIE_TOKEN_TYPE_PROTECTED;
					}
					break;
				}
				case 'w':
				{
					if (vala_genie_scanner_matches (self, begin, "writeonly")) {
						return VALA_GENIE_TOKEN_TYPE_WRITEONLY;
					}
					break;
				}
			}
			break;
		}
		case 10:
		{
			switch (begin[0]) {
				case 'i':
				{
					if (vala_genie_scanner_matches (self, begin, "implements")) {
						return VALA_GENIE_TOKEN_TYPE_IMPLEMENTS;
					}
					break;
				}
			}
			break;
		}
	}
	return VALA_GENIE_TOKEN_TYPE_IDENTIFIER;
}


ValaGenieTokenType vala_genie_scanner_read_token (ValaGenieScanner* self, ValaSourceLocation* token_begin, ValaSourceLocation* token_end) {
	gboolean _tmp1;
	gboolean _tmp4;
	gboolean _tmp5;
	gboolean _tmp6;
	ValaGenieTokenType type;
	gchar* begin;
	gint token_length_in_chars;
	g_return_val_if_fail (self != NULL, 0);
	/* emit dedents if outstanding before checking any other chars */
	if (self->priv->pending_dedents > 0) {
		self->priv->pending_dedents--;
		self->priv->indent_level--;
		(*token_begin).pos = self->priv->current;
		(*token_begin).line = self->priv->line;
		(*token_begin).column = self->priv->column;
		(*token_end).pos = self->priv->current;
		(*token_end).line = self->priv->line;
		(*token_end).column = self->priv->column;
		self->priv->last_token = VALA_GENIE_TOKEN_TYPE_DEDENT;
		return VALA_GENIE_TOKEN_TYPE_DEDENT;
	}
	_tmp1 = FALSE;
	if (self->priv->_indent_spaces == 0) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = self->priv->last_token != VALA_GENIE_TOKEN_TYPE_EOL;
	}
	if (_tmp1) {
		/* scrub whitespace (excluding newlines) and comments */
		vala_genie_scanner_space (self);
	}
	while (TRUE) {
		gboolean _tmp2;
		gboolean _tmp3;
		_tmp2 = FALSE;
		_tmp3 = FALSE;
		if (self->priv->current < self->priv->end) {
			_tmp3 = self->priv->current[0] == '\\';
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			_tmp2 = self->priv->current[1] == '\n';
		} else {
			_tmp2 = FALSE;
		}
		if (!_tmp2) {
			break;
		}
		self->priv->current = self->priv->current + 2;
		self->priv->line++;
		vala_genie_scanner_skip_space_tabs (self);
	}
	_tmp4 = FALSE;
	_tmp5 = FALSE;
	_tmp6 = FALSE;
	if (vala_genie_scanner_newline (self)) {
		_tmp6 = self->priv->parse_started;
	} else {
		_tmp6 = FALSE;
	}
	if (_tmp6) {
		_tmp5 = self->priv->last_token != VALA_GENIE_TOKEN_TYPE_EOL;
	} else {
		_tmp5 = FALSE;
	}
	if (_tmp5) {
		_tmp4 = self->priv->last_token != VALA_GENIE_TOKEN_TYPE_SEMICOLON;
	} else {
		_tmp4 = FALSE;
	}
	/* handle non-consecutive new line once parsing is underway - EOL */
	if (_tmp4) {
		(*token_begin).pos = self->priv->current;
		(*token_begin).line = self->priv->line;
		(*token_begin).column = self->priv->column;
		(*token_end).pos = self->priv->current;
		(*token_end).line = self->priv->line;
		(*token_end).column = self->priv->column;
		self->priv->last_token = VALA_GENIE_TOKEN_TYPE_EOL;
		return VALA_GENIE_TOKEN_TYPE_EOL;
	}
	while (vala_genie_scanner_skip_newlines (self)) {
		(*token_begin).pos = self->priv->current;
		(*token_begin).line = self->priv->line;
		(*token_begin).column = self->priv->column;
		self->priv->current_indent_level = vala_genie_scanner_count_tabs (self);
		/* if its an empty new line then ignore */
		if (self->priv->current_indent_level == (-1)) {
			continue;
		}
		if (self->priv->current_indent_level > self->priv->indent_level) {
			self->priv->indent_level = self->priv->current_indent_level;
			(*token_end).pos = self->priv->current;
			(*token_end).line = self->priv->line;
			(*token_end).column = self->priv->column;
			self->priv->last_token = VALA_GENIE_TOKEN_TYPE_INDENT;
			return VALA_GENIE_TOKEN_TYPE_INDENT;
		} else {
			if (self->priv->current_indent_level < self->priv->indent_level) {
				self->priv->indent_level--;
				self->priv->pending_dedents = self->priv->indent_level - self->priv->current_indent_level;
				(*token_end).pos = self->priv->current;
				(*token_end).line = self->priv->line;
				(*token_end).column = self->priv->column;
				self->priv->last_token = VALA_GENIE_TOKEN_TYPE_DEDENT;
				return VALA_GENIE_TOKEN_TYPE_DEDENT;
			}
		}
	}
	type = 0;
	begin = self->priv->current;
	(*token_begin).pos = begin;
	(*token_begin).line = self->priv->line;
	(*token_begin).column = self->priv->column;
	token_length_in_chars = -1;
	self->priv->parse_started = TRUE;
	if (self->priv->current >= self->priv->end) {
		if (self->priv->indent_level > 0) {
			self->priv->indent_level--;
			self->priv->pending_dedents = self->priv->indent_level;
			type = VALA_GENIE_TOKEN_TYPE_DEDENT;
		} else {
			type = VALA_GENIE_TOKEN_TYPE_EOF;
		}
	} else {
		gboolean _tmp10;
		_tmp10 = FALSE;
		if (g_ascii_isalpha (self->priv->current[0])) {
			_tmp10 = TRUE;
		} else {
			_tmp10 = self->priv->current[0] == '_';
		}
		if (_tmp10) {
			gint len;
			len = 0;
			while (TRUE) {
				gboolean _tmp11;
				_tmp11 = FALSE;
				if (self->priv->current < self->priv->end) {
					_tmp11 = vala_genie_scanner_is_ident_char (self, self->priv->current[0]);
				} else {
					_tmp11 = FALSE;
				}
				if (!_tmp11) {
					break;
				}
				self->priv->current++;
				len++;
			}
			type = vala_genie_scanner_get_identifier_or_keyword (self, begin, len);
		} else {
			if (self->priv->current[0] == '@') {
				gint len;
				len = 0;
				if (self->priv->current[1] == '@') {
					(*token_begin).pos = (*token_begin).pos + 2;
					/* @@ is not part of the identifier*/
					self->priv->current = self->priv->current + 2;
				} else {
					self->priv->current++;
					len = 1;
				}
				while (TRUE) {
					gboolean _tmp12;
					_tmp12 = FALSE;
					if (self->priv->current < self->priv->end) {
						_tmp12 = vala_genie_scanner_is_ident_char (self, self->priv->current[0]);
					} else {
						_tmp12 = FALSE;
					}
					if (!_tmp12) {
						break;
					}
					self->priv->current++;
					len++;
				}
				type = VALA_GENIE_TOKEN_TYPE_IDENTIFIER;
			} else {
				if (g_ascii_isdigit (self->priv->current[0])) {
					gboolean _tmp14;
					while (TRUE) {
						gboolean _tmp13;
						_tmp13 = FALSE;
						if (self->priv->current < self->priv->end) {
							_tmp13 = g_ascii_isdigit (self->priv->current[0]);
						} else {
							_tmp13 = FALSE;
						}
						if (!_tmp13) {
							break;
						}
						self->priv->current++;
					}
					type = VALA_GENIE_TOKEN_TYPE_INTEGER_LITERAL;
					_tmp14 = FALSE;
					if (self->priv->current < self->priv->end) {
						_tmp14 = g_ascii_tolower (self->priv->current[0]) == 'l';
					} else {
						_tmp14 = FALSE;
					}
					if (_tmp14) {
						gboolean _tmp15;
						self->priv->current++;
						_tmp15 = FALSE;
						if (self->priv->current < self->priv->end) {
							_tmp15 = g_ascii_tolower (self->priv->current[0]) == 'l';
						} else {
							_tmp15 = FALSE;
						}
						if (_tmp15) {
							self->priv->current++;
						}
					} else {
						gboolean _tmp16;
						_tmp16 = FALSE;
						if (self->priv->current < self->priv->end) {
							_tmp16 = g_ascii_tolower (self->priv->current[0]) == 'u';
						} else {
							_tmp16 = FALSE;
						}
						if (_tmp16) {
							gboolean _tmp17;
							self->priv->current++;
							_tmp17 = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp17 = g_ascii_tolower (self->priv->current[0]) == 'l';
							} else {
								_tmp17 = FALSE;
							}
							if (_tmp17) {
								gboolean _tmp18;
								self->priv->current++;
								_tmp18 = FALSE;
								if (self->priv->current < self->priv->end) {
									_tmp18 = g_ascii_tolower (self->priv->current[0]) == 'l';
								} else {
									_tmp18 = FALSE;
								}
								if (_tmp18) {
									self->priv->current++;
								}
							}
						} else {
							gboolean _tmp19;
							gboolean _tmp20;
							_tmp19 = FALSE;
							_tmp20 = FALSE;
							if (self->priv->current < (self->priv->end - 1)) {
								_tmp20 = self->priv->current[0] == '.';
							} else {
								_tmp20 = FALSE;
							}
							if (_tmp20) {
								_tmp19 = g_ascii_isdigit (self->priv->current[1]);
							} else {
								_tmp19 = FALSE;
							}
							if (_tmp19) {
								gboolean _tmp22;
								gboolean _tmp26;
								self->priv->current++;
								while (TRUE) {
									gboolean _tmp21;
									_tmp21 = FALSE;
									if (self->priv->current < self->priv->end) {
										_tmp21 = g_ascii_isdigit (self->priv->current[0]);
									} else {
										_tmp21 = FALSE;
									}
									if (!_tmp21) {
										break;
									}
									self->priv->current++;
								}
								_tmp22 = FALSE;
								if (self->priv->current < self->priv->end) {
									_tmp22 = g_ascii_tolower (self->priv->current[0]) == 'e';
								} else {
									_tmp22 = FALSE;
								}
								if (_tmp22) {
									gboolean _tmp23;
									self->priv->current++;
									_tmp23 = FALSE;
									if (self->priv->current < self->priv->end) {
										gboolean _tmp24;
										_tmp24 = FALSE;
										if (self->priv->current[0] == '+') {
											_tmp24 = TRUE;
										} else {
											_tmp24 = self->priv->current[0] == '-';
										}
										_tmp23 = _tmp24;
									} else {
										_tmp23 = FALSE;
									}
									if (_tmp23) {
										self->priv->current++;
									}
									while (TRUE) {
										gboolean _tmp25;
										_tmp25 = FALSE;
										if (self->priv->current < self->priv->end) {
											_tmp25 = g_ascii_isdigit (self->priv->current[0]);
										} else {
											_tmp25 = FALSE;
										}
										if (!_tmp25) {
											break;
										}
										self->priv->current++;
									}
								}
								_tmp26 = FALSE;
								if (self->priv->current < self->priv->end) {
									_tmp26 = g_ascii_tolower (self->priv->current[0]) == 'f';
								} else {
									_tmp26 = FALSE;
								}
								if (_tmp26) {
									self->priv->current++;
								}
								type = VALA_GENIE_TOKEN_TYPE_REAL_LITERAL;
							} else {
								gboolean _tmp27;
								gboolean _tmp28;
								gboolean _tmp29;
								gboolean _tmp30;
								_tmp27 = FALSE;
								_tmp28 = FALSE;
								_tmp29 = FALSE;
								_tmp30 = FALSE;
								if (self->priv->current < self->priv->end) {
									_tmp30 = self->priv->current == (begin + 1);
								} else {
									_tmp30 = FALSE;
								}
								if (_tmp30) {
									_tmp29 = begin[0] == '0';
								} else {
									_tmp29 = FALSE;
								}
								if (_tmp29) {
									_tmp28 = begin[1] == 'x';
								} else {
									_tmp28 = FALSE;
								}
								if (_tmp28) {
									_tmp27 = g_ascii_isxdigit (begin[2]);
								} else {
									_tmp27 = FALSE;
								}
								if (_tmp27) {
									/* hexadecimal integer literal*/
									self->priv->current++;
									while (TRUE) {
										gboolean _tmp31;
										_tmp31 = FALSE;
										if (self->priv->current < self->priv->end) {
											_tmp31 = g_ascii_isxdigit (self->priv->current[0]);
										} else {
											_tmp31 = FALSE;
										}
										if (!_tmp31) {
											break;
										}
										self->priv->current++;
									}
								} else {
									gboolean _tmp32;
									_tmp32 = FALSE;
									if (self->priv->current < self->priv->end) {
										_tmp32 = vala_genie_scanner_is_ident_char (self, self->priv->current[0]);
									} else {
										_tmp32 = FALSE;
									}
									if (_tmp32) {
										while (TRUE) {
											gboolean _tmp33;
											_tmp33 = FALSE;
											if (self->priv->current < self->priv->end) {
												_tmp33 = vala_genie_scanner_is_ident_char (self, self->priv->current[0]);
											} else {
												_tmp33 = FALSE;
											}
											if (!_tmp33) {
												break;
											}
											self->priv->current++;
										}
										type = VALA_GENIE_TOKEN_TYPE_IDENTIFIER;
									}
								}
							}
						}
					}
				} else {
					switch (self->priv->current[0]) {
						case '{':
						{
							type = VALA_GENIE_TOKEN_TYPE_OPEN_BRACE;
							self->priv->current++;
							break;
						}
						case '}':
						{
							type = VALA_GENIE_TOKEN_TYPE_CLOSE_BRACE;
							self->priv->current++;
							break;
						}
						case '(':
						{
							type = VALA_GENIE_TOKEN_TYPE_OPEN_PARENS;
							self->priv->current++;
							break;
						}
						case ')':
						{
							type = VALA_GENIE_TOKEN_TYPE_CLOSE_PARENS;
							self->priv->current++;
							break;
						}
						case '[':
						{
							type = VALA_GENIE_TOKEN_TYPE_OPEN_BRACKET;
							self->priv->current++;
							break;
						}
						case ']':
						{
							type = VALA_GENIE_TOKEN_TYPE_CLOSE_BRACKET;
							self->priv->current++;
							break;
						}
						case '.':
						{
							type = VALA_GENIE_TOKEN_TYPE_DOT;
							self->priv->current++;
							if (self->priv->current < (self->priv->end - 1)) {
								gboolean _tmp34;
								_tmp34 = FALSE;
								if (self->priv->current[0] == '.') {
									_tmp34 = self->priv->current[1] == '.';
								} else {
									_tmp34 = FALSE;
								}
								if (_tmp34) {
									type = VALA_GENIE_TOKEN_TYPE_ELLIPSIS;
									self->priv->current = self->priv->current + 2;
								}
							}
							break;
						}
						case ':':
						{
							type = VALA_GENIE_TOKEN_TYPE_COLON;
							self->priv->current++;
							break;
						}
						case ',':
						{
							type = VALA_GENIE_TOKEN_TYPE_COMMA;
							self->priv->current++;
							break;
						}
						case ';':
						{
							type = VALA_GENIE_TOKEN_TYPE_SEMICOLON;
							self->priv->current++;
							break;
						}
						case '#':
						{
							type = VALA_GENIE_TOKEN_TYPE_HASH;
							self->priv->current++;
							break;
						}
						case '?':
						{
							type = VALA_GENIE_TOKEN_TYPE_INTERR;
							self->priv->current++;
							break;
						}
						case '|':
						{
							type = VALA_GENIE_TOKEN_TYPE_BITWISE_OR;
							self->priv->current++;
							if (self->priv->current < self->priv->end) {
								switch (self->priv->current[0]) {
									case '=':
									{
										type = VALA_GENIE_TOKEN_TYPE_ASSIGN_BITWISE_OR;
										self->priv->current++;
										break;
									}
									case '|':
									{
										type = VALA_GENIE_TOKEN_TYPE_OP_OR;
										self->priv->current++;
										break;
									}
								}
							}
							break;
						}
						case '&':
						{
							type = VALA_GENIE_TOKEN_TYPE_BITWISE_AND;
							self->priv->current++;
							if (self->priv->current < self->priv->end) {
								switch (self->priv->current[0]) {
									case '=':
									{
										type = VALA_GENIE_TOKEN_TYPE_ASSIGN_BITWISE_AND;
										self->priv->current++;
										break;
									}
									case '&':
									{
										type = VALA_GENIE_TOKEN_TYPE_OP_AND;
										self->priv->current++;
										break;
									}
								}
							}
							break;
						}
						case '^':
						{
							gboolean _tmp35;
							type = VALA_GENIE_TOKEN_TYPE_CARRET;
							self->priv->current++;
							_tmp35 = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp35 = self->priv->current[0] == '=';
							} else {
								_tmp35 = FALSE;
							}
							if (_tmp35) {
								type = VALA_GENIE_TOKEN_TYPE_ASSIGN_BITWISE_XOR;
								self->priv->current++;
							}
							break;
						}
						case '~':
						{
							type = VALA_GENIE_TOKEN_TYPE_TILDE;
							self->priv->current++;
							break;
						}
						case '=':
						{
							type = VALA_GENIE_TOKEN_TYPE_ASSIGN;
							self->priv->current++;
							if (self->priv->current < self->priv->end) {
								switch (self->priv->current[0]) {
									case '=':
									{
										type = VALA_GENIE_TOKEN_TYPE_OP_EQ;
										self->priv->current++;
										break;
									}
									case '>':
									{
										type = VALA_GENIE_TOKEN_TYPE_LAMBDA;
										self->priv->current++;
										break;
									}
								}
							}
							break;
						}
						case '<':
						{
							type = VALA_GENIE_TOKEN_TYPE_OP_LT;
							self->priv->current++;
							if (self->priv->current < self->priv->end) {
								switch (self->priv->current[0]) {
									case '=':
									{
										type = VALA_GENIE_TOKEN_TYPE_OP_LE;
										self->priv->current++;
										break;
									}
									case '<':
									{
										gboolean _tmp36;
										type = VALA_GENIE_TOKEN_TYPE_OP_SHIFT_LEFT;
										self->priv->current++;
										_tmp36 = FALSE;
										if (self->priv->current < self->priv->end) {
											_tmp36 = self->priv->current[0] == '=';
										} else {
											_tmp36 = FALSE;
										}
										if (_tmp36) {
											type = VALA_GENIE_TOKEN_TYPE_ASSIGN_SHIFT_LEFT;
											self->priv->current++;
										}
										break;
									}
								}
							}
							break;
						}
						case '>':
						{
							gboolean _tmp37;
							type = VALA_GENIE_TOKEN_TYPE_OP_GT;
							self->priv->current++;
							_tmp37 = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp37 = self->priv->current[0] == '=';
							} else {
								_tmp37 = FALSE;
							}
							if (_tmp37) {
								type = VALA_GENIE_TOKEN_TYPE_OP_GE;
								self->priv->current++;
							}
							break;
						}
						case '!':
						{
							gboolean _tmp38;
							type = VALA_GENIE_TOKEN_TYPE_OP_NEG;
							self->priv->current++;
							_tmp38 = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp38 = self->priv->current[0] == '=';
							} else {
								_tmp38 = FALSE;
							}
							if (_tmp38) {
								type = VALA_GENIE_TOKEN_TYPE_OP_NE;
								self->priv->current++;
							}
							break;
						}
						case '+':
						{
							type = VALA_GENIE_TOKEN_TYPE_PLUS;
							self->priv->current++;
							if (self->priv->current < self->priv->end) {
								switch (self->priv->current[0]) {
									case '=':
									{
										type = VALA_GENIE_TOKEN_TYPE_ASSIGN_ADD;
										self->priv->current++;
										break;
									}
									case '+':
									{
										type = VALA_GENIE_TOKEN_TYPE_OP_INC;
										self->priv->current++;
										break;
									}
								}
							}
							break;
						}
						case '-':
						{
							type = VALA_GENIE_TOKEN_TYPE_MINUS;
							self->priv->current++;
							if (self->priv->current < self->priv->end) {
								switch (self->priv->current[0]) {
									case '=':
									{
										type = VALA_GENIE_TOKEN_TYPE_ASSIGN_SUB;
										self->priv->current++;
										break;
									}
									case '-':
									{
										type = VALA_GENIE_TOKEN_TYPE_OP_DEC;
										self->priv->current++;
										break;
									}
									case '>':
									{
										type = VALA_GENIE_TOKEN_TYPE_OP_PTR;
										self->priv->current++;
										break;
									}
								}
							}
							break;
						}
						case '*':
						{
							gboolean _tmp39;
							type = VALA_GENIE_TOKEN_TYPE_STAR;
							self->priv->current++;
							_tmp39 = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp39 = self->priv->current[0] == '=';
							} else {
								_tmp39 = FALSE;
							}
							if (_tmp39) {
								type = VALA_GENIE_TOKEN_TYPE_ASSIGN_MUL;
								self->priv->current++;
							}
							break;
						}
						case '/':
						{
							gboolean _tmp40;
							type = VALA_GENIE_TOKEN_TYPE_DIV;
							self->priv->current++;
							_tmp40 = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp40 = self->priv->current[0] == '=';
							} else {
								_tmp40 = FALSE;
							}
							if (_tmp40) {
								type = VALA_GENIE_TOKEN_TYPE_ASSIGN_DIV;
								self->priv->current++;
							}
							break;
						}
						case '%':
						{
							gboolean _tmp41;
							type = VALA_GENIE_TOKEN_TYPE_PERCENT;
							self->priv->current++;
							_tmp41 = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp41 = self->priv->current[0] == '=';
							} else {
								_tmp41 = FALSE;
							}
							if (_tmp41) {
								type = VALA_GENIE_TOKEN_TYPE_ASSIGN_PERCENT;
								self->priv->current++;
							}
							break;
						}
						case '\'':
						case '"':
						{
							gboolean _tmp46;
							if (begin[0] == '\'') {
								type = VALA_GENIE_TOKEN_TYPE_CHARACTER_LITERAL;
							} else {
								type = VALA_GENIE_TOKEN_TYPE_STRING_LITERAL;
							}
							token_length_in_chars = 2;
							self->priv->current++;
							while (TRUE) {
								gboolean _tmp42;
								_tmp42 = FALSE;
								if (self->priv->current < self->priv->end) {
									_tmp42 = self->priv->current[0] != begin[0];
								} else {
									_tmp42 = FALSE;
								}
								if (!_tmp42) {
									break;
								}
								if (self->priv->current[0] == '\\') {
									gboolean _tmp43;
									self->priv->current++;
									token_length_in_chars++;
									_tmp43 = FALSE;
									if (self->priv->current < self->priv->end) {
										_tmp43 = self->priv->current[0] == 'x';
									} else {
										_tmp43 = FALSE;
									}
									if (_tmp43) {
										/* hexadecimal escape character*/
										self->priv->current++;
										token_length_in_chars++;
										while (TRUE) {
											gboolean _tmp44;
											_tmp44 = FALSE;
											if (self->priv->current < self->priv->end) {
												_tmp44 = g_ascii_isxdigit (self->priv->current[0]);
											} else {
												_tmp44 = FALSE;
											}
											if (!_tmp44) {
												break;
											}
											self->priv->current++;
											token_length_in_chars++;
										}
									} else {
										self->priv->current++;
										token_length_in_chars++;
									}
								} else {
									if (self->priv->current[0] == '\n') {
										break;
									} else {
										gunichar u;
										u = g_utf8_get_char_validated ((const char*) self->priv->current, (glong) (self->priv->end - self->priv->current));
										if (u != ((gunichar) (-1))) {
											self->priv->current = self->priv->current + g_unichar_to_utf8 (u, NULL);
											token_length_in_chars++;
										} else {
											ValaSourceReference* _tmp45;
											_tmp45 = NULL;
											vala_report_error (_tmp45 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars), "invalid UTF-8 character");
											(_tmp45 == NULL) ? NULL : (_tmp45 = (vala_source_reference_unref (_tmp45), NULL));
										}
									}
								}
							}
							_tmp46 = FALSE;
							if (self->priv->current < self->priv->end) {
								_tmp46 = self->priv->current[0] != '\n';
							} else {
								_tmp46 = FALSE;
							}
							if (_tmp46) {
								self->priv->current++;
							} else {
								char* _tmp48;
								ValaSourceReference* _tmp47;
								_tmp48 = NULL;
								_tmp47 = NULL;
								vala_report_error (_tmp47 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column + token_length_in_chars, self->priv->line, self->priv->column + token_length_in_chars), _tmp48 = g_strdup_printf ("syntax error, expected %c", (gint) begin[0]));
								_tmp48 = (g_free (_tmp48), NULL);
								(_tmp47 == NULL) ? NULL : (_tmp47 = (vala_source_reference_unref (_tmp47), NULL));
							}
							break;
						}
						default:
						{
							gunichar u;
							u = g_utf8_get_char_validated ((const char*) self->priv->current, (glong) (self->priv->end - self->priv->current));
							if (u != ((gunichar) (-1))) {
								ValaSourceReference* _tmp49;
								self->priv->current = self->priv->current + g_unichar_to_utf8 (u, NULL);
								_tmp49 = NULL;
								vala_report_error (_tmp49 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "syntax error, unexpected character");
								(_tmp49 == NULL) ? NULL : (_tmp49 = (vala_source_reference_unref (_tmp49), NULL));
							} else {
								ValaSourceReference* _tmp50;
								self->priv->current++;
								_tmp50 = NULL;
								vala_report_error (_tmp50 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "invalid UTF-8 character");
								(_tmp50 == NULL) ? NULL : (_tmp50 = (vala_source_reference_unref (_tmp50), NULL));
							}
							self->priv->column++;
							self->priv->last_token = VALA_GENIE_TOKEN_TYPE_STRING_LITERAL;
							return vala_genie_scanner_read_token (self, &(*token_begin), &(*token_end));
						}
					}
				}
			}
		}
	}
	if (token_length_in_chars < 0) {
		self->priv->column = self->priv->column + ((gint) (self->priv->current - begin));
	} else {
		self->priv->column = self->priv->column + token_length_in_chars;
	}
	(*token_end).pos = self->priv->current;
	(*token_end).line = self->priv->line;
	(*token_end).column = self->priv->column - 1;
	self->priv->last_token = type;
	return type;
}


static gint vala_genie_scanner_count_tabs (ValaGenieScanner* self) {
	gint tab_count;
	gboolean _tmp2;
	g_return_val_if_fail (self != NULL, 0);
	tab_count = 0;
	if (self->priv->_indent_spaces == 0) {
		while (TRUE) {
			gboolean _tmp0;
			_tmp0 = FALSE;
			if (self->priv->current < self->priv->end) {
				_tmp0 = self->priv->current[0] == '\t';
			} else {
				_tmp0 = FALSE;
			}
			if (!_tmp0) {
				break;
			}
			self->priv->current++;
			self->priv->column++;
			tab_count++;
		}
	} else {
		gint space_count;
		space_count = 0;
		while (TRUE) {
			gboolean _tmp1;
			_tmp1 = FALSE;
			if (self->priv->current < self->priv->end) {
				_tmp1 = self->priv->current[0] == ' ';
			} else {
				_tmp1 = FALSE;
			}
			if (!_tmp1) {
				break;
			}
			self->priv->current++;
			self->priv->column++;
			space_count++;
		}
		tab_count = space_count / self->priv->_indent_spaces;
	}
	/* ignore comments and whitspace and other lines that contain no code */
	vala_genie_scanner_space (self);
	_tmp2 = FALSE;
	if (self->priv->current < self->priv->end) {
		_tmp2 = self->priv->current[0] == '\n';
	} else {
		_tmp2 = FALSE;
	}
	if (_tmp2) {
		return -1;
	}
	return tab_count;
}


static gboolean vala_genie_scanner_matches (ValaGenieScanner* self, gchar* begin, const char* keyword) {
	gchar* keyword_array;
	glong len;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (keyword != NULL, FALSE);
	keyword_array = keyword;
	len = g_utf8_strlen (keyword, -1);
	{
		gint i;
		i = 0;
		for (; i < len; i++) {
			if (begin[i] != keyword_array[i]) {
				return FALSE;
			}
		}
	}
	return TRUE;
}


static gboolean vala_genie_scanner_whitespace (ValaGenieScanner* self) {
	gboolean found;
	g_return_val_if_fail (self != NULL, FALSE);
	found = FALSE;
	while (TRUE) {
		gboolean _tmp0;
		gboolean _tmp1;
		_tmp0 = FALSE;
		_tmp1 = FALSE;
		if (self->priv->current < self->priv->end) {
			_tmp1 = g_ascii_isspace (self->priv->current[0]);
		} else {
			_tmp1 = FALSE;
		}
		if (_tmp1) {
			_tmp0 = self->priv->current[0] != '\n';
		} else {
			_tmp0 = FALSE;
		}
		if (!_tmp0) {
			break;
		}
		found = TRUE;
		self->priv->current++;
		self->priv->column++;
	}
	return found;
}


static inline gboolean vala_genie_scanner_newline (ValaGenieScanner* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (self->priv->current[0] == '\n') {
		return TRUE;
	}
	return FALSE;
}


static gboolean vala_genie_scanner_skip_newlines (ValaGenieScanner* self) {
	gboolean new_lines;
	g_return_val_if_fail (self != NULL, FALSE);
	new_lines = FALSE;
	while (vala_genie_scanner_newline (self)) {
		self->priv->current++;
		self->priv->line++;
		self->priv->column = 1;
		self->priv->current_indent_level = 0;
		new_lines = TRUE;
	}
	return new_lines;
}


static gboolean vala_genie_scanner_comment (ValaGenieScanner* self) {
	gboolean _tmp0;
	gboolean _tmp1;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (self->priv->current > (self->priv->end - 2)) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = self->priv->current[0] != '/';
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		gboolean _tmp2;
		_tmp2 = FALSE;
		if (self->priv->current[1] != '/') {
			_tmp2 = self->priv->current[1] != '*';
		} else {
			_tmp2 = FALSE;
		}
		_tmp0 = _tmp2;
	}
	if (_tmp0) {
		return FALSE;
	}
	if (self->priv->current[1] == '/') {
		gchar* begin;
		char* _tmp5;
		/* single-line comment*/
		self->priv->current = self->priv->current + 2;
		begin = self->priv->current;
		while (TRUE) {
			gboolean _tmp4;
			_tmp4 = FALSE;
			if (self->priv->current < self->priv->end) {
				_tmp4 = self->priv->current[0] != '\n';
			} else {
				_tmp4 = FALSE;
			}
			if (!_tmp4) {
				break;
			}
			self->priv->current++;
		}
		_tmp5 = NULL;
		vala_genie_scanner_push_comment (self, _tmp5 = g_strndup ((const char*) begin, (gsize) ((glong) (self->priv->current - begin))), self->priv->line == 1);
		_tmp5 = (g_free (_tmp5), NULL);
		if (self->priv->current[0] == '\n') {
			self->priv->current++;
			self->priv->line++;
			self->priv->column = 1;
			self->priv->current_indent_level = 0;
		}
	} else {
		gchar* begin;
		gint begin_line;
		char* _tmp10;
		/* delimited comment*/
		self->priv->current = self->priv->current + 2;
		begin = self->priv->current;
		begin_line = self->priv->line;
		while (TRUE) {
			gboolean _tmp6;
			_tmp6 = FALSE;
			if (self->priv->current < (self->priv->end - 1)) {
				gboolean _tmp7;
				_tmp7 = FALSE;
				if (self->priv->current[0] != '*') {
					_tmp7 = TRUE;
				} else {
					_tmp7 = self->priv->current[1] != '/';
				}
				_tmp6 = _tmp7;
			} else {
				_tmp6 = FALSE;
			}
			if (!_tmp6) {
				break;
			}
			if (self->priv->current[0] == '\n') {
				self->priv->line++;
				self->priv->column = 0;
			}
			self->priv->current++;
			self->priv->column++;
		}
		if (self->priv->current == (self->priv->end - 1)) {
			ValaSourceReference* _tmp8;
			_tmp8 = NULL;
			vala_report_error (_tmp8 = vala_source_reference_new (self->priv->_source_file, self->priv->line, self->priv->column, self->priv->line, self->priv->column), "syntax error, expected */");
			(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_source_reference_unref (_tmp8), NULL));
			return TRUE;
		}
		_tmp10 = NULL;
		vala_genie_scanner_push_comment (self, _tmp10 = g_strndup ((const char*) begin, (gsize) ((glong) (self->priv->current - begin))), begin_line == 1);
		_tmp10 = (g_free (_tmp10), NULL);
		self->priv->current = self->priv->current + 2;
		self->priv->column = self->priv->column + 2;
	}
	return TRUE;
}


static gboolean vala_genie_scanner_skip_tabs (ValaGenieScanner* self) {
	gboolean found;
	g_return_val_if_fail (self != NULL, FALSE);
	found = FALSE;
	while (TRUE) {
		gboolean _tmp0;
		_tmp0 = FALSE;
		if (self->priv->current < self->priv->end) {
			_tmp0 = self->priv->current[0] == '\t';
		} else {
			_tmp0 = FALSE;
		}
		if (!_tmp0) {
			break;
		}
		self->priv->current++;
		self->priv->column++;
		found = TRUE;
	}
	return found;
}


static void vala_genie_scanner_skip_space_tabs (ValaGenieScanner* self) {
	g_return_if_fail (self != NULL);
	while (TRUE) {
		gboolean _tmp0;
		gboolean _tmp1;
		_tmp0 = FALSE;
		_tmp1 = FALSE;
		if (vala_genie_scanner_whitespace (self)) {
			_tmp1 = TRUE;
		} else {
			_tmp1 = vala_genie_scanner_skip_tabs (self);
		}
		if (_tmp1) {
			_tmp0 = TRUE;
		} else {
			_tmp0 = vala_genie_scanner_comment (self);
		}
		if (!_tmp0) {
			break;
		}
	}
}


static void vala_genie_scanner_space (ValaGenieScanner* self) {
	g_return_if_fail (self != NULL);
	while (TRUE) {
		gboolean _tmp0;
		_tmp0 = FALSE;
		if (vala_genie_scanner_whitespace (self)) {
			_tmp0 = TRUE;
		} else {
			_tmp0 = vala_genie_scanner_comment (self);
		}
		if (!_tmp0) {
			break;
		}
	}
}


static void vala_genie_scanner_push_comment (ValaGenieScanner* self, const char* comment_item, gboolean file_comment) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (comment_item != NULL);
	if (self->priv->_comment == NULL) {
		char* _tmp1;
		const char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->_comment = (_tmp1 = (_tmp0 = comment_item, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->_comment = (g_free (self->priv->_comment), NULL), _tmp1);
	} else {
		char* _tmp2;
		_tmp2 = NULL;
		self->priv->_comment = (_tmp2 = g_strdup_printf ("%s\n%s", self->priv->_comment, comment_item), self->priv->_comment = (g_free (self->priv->_comment), NULL), _tmp2);
	}
	if (file_comment) {
		char* _tmp3;
		vala_source_file_set_comment (self->priv->_source_file, self->priv->_comment);
		_tmp3 = NULL;
		self->priv->_comment = (_tmp3 = NULL, self->priv->_comment = (g_free (self->priv->_comment), NULL), _tmp3);
	}
}


/**
 * Clears and returns the content of the comment stack.
 *
 * @return saved comment
 */
char* vala_genie_scanner_pop_comment (ValaGenieScanner* self) {
	GString* result_builder;
	char* _tmp1;
	const char* index;
	const char* _tmp2;
	char* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_comment == NULL) {
		return NULL;
	}
	result_builder = g_string_new (self->priv->_comment);
	_tmp1 = NULL;
	self->priv->_comment = (_tmp1 = NULL, self->priv->_comment = (g_free (self->priv->_comment), NULL), _tmp1);
	index = NULL;
	while ((index = g_utf8_strchr (result_builder->str, (glong) (-1), (gunichar) '\t')) != NULL) {
		g_string_erase (result_builder, g_utf8_pointer_to_offset (result_builder->str, index), (glong) 1);
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = (_tmp2 = result_builder->str, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), (result_builder == NULL) ? NULL : (result_builder = (g_string_free (result_builder, TRUE), NULL)), _tmp3);
}


ValaSourceFile* vala_genie_scanner_get_source_file (ValaGenieScanner* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_source_file;
}


static void vala_genie_scanner_set_source_file (ValaGenieScanner* self, ValaSourceFile* value) {
	ValaSourceFile* _tmp2;
	ValaSourceFile* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_source_file = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_source_file_ref (_tmp1)), (self->priv->_source_file == NULL) ? NULL : (self->priv->_source_file = (vala_source_file_unref (self->priv->_source_file), NULL)), _tmp2);
}


gint vala_genie_scanner_get_indent_spaces (ValaGenieScanner* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_indent_spaces;
}


void vala_genie_scanner_set_indent_spaces (ValaGenieScanner* self, gint value) {
	g_return_if_fail (self != NULL);
	self->priv->_indent_spaces = value;
}


static void vala_genie_value_scanner_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_genie_value_scanner_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_genie_scanner_unref (value->data[0].v_pointer);
	}
}


static void vala_genie_value_scanner_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_genie_scanner_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_genie_value_scanner_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_genie_value_scanner_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaGenieScanner* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_genie_scanner_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_genie_value_scanner_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaGenieScanner** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_genie_scanner_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_genie_param_spec_scanner (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaGenieParamSpecScanner* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_GENIE_TYPE_SCANNER), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_genie_value_get_scanner (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_GENIE_TYPE_SCANNER), NULL);
	return value->data[0].v_pointer;
}


void vala_genie_value_set_scanner (GValue* value, gpointer v_object) {
	ValaGenieScanner* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_GENIE_TYPE_SCANNER));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_GENIE_TYPE_SCANNER));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_genie_scanner_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_genie_scanner_unref (old);
	}
}


static void vala_genie_scanner_class_init (ValaGenieScannerClass * klass) {
	vala_genie_scanner_parent_class = g_type_class_peek_parent (klass);
	VALA_GENIE_SCANNER_CLASS (klass)->finalize = vala_genie_scanner_finalize;
	g_type_class_add_private (klass, sizeof (ValaGenieScannerPrivate));
}


static void vala_genie_scanner_instance_init (ValaGenieScanner * self) {
	self->priv = VALA_GENIE_SCANNER_GET_PRIVATE (self);
	self->ref_count = 1;
}


static void vala_genie_scanner_finalize (ValaGenieScanner* obj) {
	ValaGenieScanner * self;
	self = VALA_GENIE_SCANNER (obj);
	(self->priv->_source_file == NULL) ? NULL : (self->priv->_source_file = (vala_source_file_unref (self->priv->_source_file), NULL));
	self->priv->_comment = (g_free (self->priv->_comment), NULL);
}


GType vala_genie_scanner_get_type (void) {
	static GType vala_genie_scanner_type_id = 0;
	if (vala_genie_scanner_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_genie_value_scanner_init, vala_genie_value_scanner_free_value, vala_genie_value_scanner_copy_value, vala_genie_value_scanner_peek_pointer, "p", vala_genie_value_scanner_collect_value, "p", vala_genie_value_scanner_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaGenieScannerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_genie_scanner_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGenieScanner), 0, (GInstanceInitFunc) vala_genie_scanner_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_genie_scanner_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaGenieScanner", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_genie_scanner_type_id;
}


gpointer vala_genie_scanner_ref (gpointer instance) {
	ValaGenieScanner* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_genie_scanner_unref (gpointer instance) {
	ValaGenieScanner* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_GENIE_SCANNER_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}




