/* valaforstatement.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaforstatement.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valaexpression.h>
#include <vala/valablock.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valadatatype.h>
#include <vala/valareport.h>
#include <vala/valaunaryexpression.h>
#include <vala/valabreakstatement.h>
#include <vala/valaifstatement.h>
#include <vala/valabooleanliteral.h>




struct _ValaForStatementPrivate {
	GeeList* initializer;
	GeeList* iterator;
	ValaExpression* _condition;
	ValaBlock* _body;
};

#define VALA_FOR_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_FOR_STATEMENT, ValaForStatementPrivate))
enum  {
	VALA_FOR_STATEMENT_DUMMY_PROPERTY
};
static void vala_for_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_for_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_for_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_for_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_for_statement_parent_class = NULL;
static ValaStatementIface* vala_for_statement_vala_statement_parent_iface = NULL;
static void vala_for_statement_finalize (ValaCodeNode* obj);



/**
 * Creates a new for statement.
 *
 * @param cond             loop condition
 * @param body             loop body
 * @param source_reference reference to source code
 * @return                 newly created for statement
 */
ValaForStatement* vala_for_statement_construct (GType object_type, ValaExpression* condition, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaForStatement* self;
	g_return_val_if_fail (body != NULL, NULL);
	self = (ValaForStatement*) g_type_create_instance (object_type);
	vala_for_statement_set_condition (self, condition);
	vala_for_statement_set_body (self, body);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaForStatement* vala_for_statement_new (ValaExpression* condition, ValaBlock* body, ValaSourceReference* source_reference) {
	return vala_for_statement_construct (VALA_TYPE_FOR_STATEMENT, condition, body, source_reference);
}


/**
 * Appends the specified expression to the list of initializers.
 *
 * @param init an initializer expression
 */
void vala_for_statement_add_initializer (ValaForStatement* self, ValaExpression* init) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (init != NULL);
	vala_code_node_set_parent_node ((ValaCodeNode*) init, (ValaCodeNode*) self);
	gee_collection_add ((GeeCollection*) self->priv->initializer, init);
}


/**
 * Returns a copy of the list of initializers.
 *
 * @return initializer list
 */
GeeList* vala_for_statement_get_initializer (ValaForStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->initializer);
}


/**
 * Appends the specified expression to the iterator.
 *
 * @param iter an iterator expression
 */
void vala_for_statement_add_iterator (ValaForStatement* self, ValaExpression* iter) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (iter != NULL);
	vala_code_node_set_parent_node ((ValaCodeNode*) iter, (ValaCodeNode*) self);
	gee_collection_add ((GeeCollection*) self->priv->iterator, iter);
}


/**
 * Returns a copy of the iterator.
 *
 * @return iterator
 */
GeeList* vala_for_statement_get_iterator (ValaForStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->iterator);
}


static void vala_for_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaForStatement * self;
	self = (ValaForStatement*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_for_statement (visitor, self);
}


static void vala_for_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaForStatement * self;
	self = (ValaForStatement*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _init_expr_it;
		_init_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->initializer);
		while (gee_iterator_next (_init_expr_it)) {
			ValaExpression* init_expr;
			init_expr = (ValaExpression*) gee_iterator_get (_init_expr_it);
			vala_code_node_accept ((ValaCodeNode*) init_expr, visitor);
			vala_code_visitor_visit_end_full_expression (visitor, init_expr);
			(init_expr == NULL) ? NULL : (init_expr = (vala_code_node_unref (init_expr), NULL));
		}
		(_init_expr_it == NULL) ? NULL : (_init_expr_it = (gee_collection_object_unref (_init_expr_it), NULL));
	}
	if (vala_for_statement_get_condition (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_for_statement_get_condition (self), visitor);
		vala_code_visitor_visit_end_full_expression (visitor, vala_for_statement_get_condition (self));
	}
	{
		GeeIterator* _it_expr_it;
		_it_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->iterator);
		while (gee_iterator_next (_it_expr_it)) {
			ValaExpression* it_expr;
			it_expr = (ValaExpression*) gee_iterator_get (_it_expr_it);
			vala_code_node_accept ((ValaCodeNode*) it_expr, visitor);
			vala_code_visitor_visit_end_full_expression (visitor, it_expr);
			(it_expr == NULL) ? NULL : (it_expr = (vala_code_node_unref (it_expr), NULL));
		}
		(_it_expr_it == NULL) ? NULL : (_it_expr_it = (gee_collection_object_unref (_it_expr_it), NULL));
	}
	vala_code_node_accept ((ValaCodeNode*) vala_for_statement_get_body (self), visitor);
}


static void vala_for_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaForStatement * self;
	self = (ValaForStatement*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_for_statement_get_condition (self) == old_node) {
		vala_for_statement_set_condition (self, new_node);
		return;
	}
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) self->priv->initializer); i++) {
			ValaExpression* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = (ValaExpression*) gee_list_get (self->priv->initializer, i)) == old_node, (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1)) {
				gee_list_set (self->priv->initializer, i, new_node);
				return;
			}
		}
	}
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) self->priv->iterator); i++) {
			ValaExpression* _tmp2;
			gboolean _tmp3;
			_tmp2 = NULL;
			if ((_tmp3 = (_tmp2 = (ValaExpression*) gee_list_get (self->priv->iterator, i)) == old_node, (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)), _tmp3)) {
				gee_list_set (self->priv->iterator, i, new_node);
				return;
			}
		}
	}
}


static gboolean vala_for_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaForStatement * self;
	gboolean _tmp1;
	gboolean _tmp3;
	GeeList* _tmp6;
	self = (ValaForStatement*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	{
		GeeIterator* _init_expr_it;
		_init_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->initializer);
		while (gee_iterator_next (_init_expr_it)) {
			ValaExpression* init_expr;
			init_expr = (ValaExpression*) gee_iterator_get (_init_expr_it);
			vala_code_node_check ((ValaCodeNode*) init_expr, analyzer);
			(init_expr == NULL) ? NULL : (init_expr = (vala_code_node_unref (init_expr), NULL));
		}
		(_init_expr_it == NULL) ? NULL : (_init_expr_it = (gee_collection_object_unref (_init_expr_it), NULL));
	}
	if (vala_for_statement_get_condition (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_for_statement_get_condition (self), analyzer);
	}
	{
		GeeIterator* _it_expr_it;
		_it_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->iterator);
		while (gee_iterator_next (_it_expr_it)) {
			ValaExpression* it_expr;
			it_expr = (ValaExpression*) gee_iterator_get (_it_expr_it);
			vala_code_node_check ((ValaCodeNode*) it_expr, analyzer);
			(it_expr == NULL) ? NULL : (it_expr = (vala_code_node_unref (it_expr), NULL));
		}
		(_it_expr_it == NULL) ? NULL : (_it_expr_it = (gee_collection_object_unref (_it_expr_it), NULL));
	}
	vala_code_node_check ((ValaCodeNode*) vala_for_statement_get_body (self), analyzer);
	_tmp1 = FALSE;
	if (vala_for_statement_get_condition (self) != NULL) {
		_tmp1 = vala_code_node_get_error ((ValaCodeNode*) vala_for_statement_get_condition (self));
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		/* if there was an error in the condition, skip this check */
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	_tmp3 = FALSE;
	if (vala_for_statement_get_condition (self) != NULL) {
		_tmp3 = !vala_data_type_compatible (vala_expression_get_value_type (vala_for_statement_get_condition (self)), analyzer->bool_type);
	} else {
		_tmp3 = FALSE;
	}
	if (_tmp3) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_for_statement_get_condition (self)), "Condition must be boolean");
		return FALSE;
	}
	if (vala_for_statement_get_condition (self) != NULL) {
		GeeList* _tmp5;
		_tmp5 = NULL;
		vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp5 = vala_code_node_get_error_types ((ValaCodeNode*) vala_for_statement_get_condition (self)));
		(_tmp5 == NULL) ? NULL : (_tmp5 = (gee_collection_object_unref (_tmp5), NULL));
	}
	_tmp6 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp6 = vala_code_node_get_error_types ((ValaCodeNode*) vala_for_statement_get_body (self)));
	(_tmp6 == NULL) ? NULL : (_tmp6 = (gee_collection_object_unref (_tmp6), NULL));
	{
		GeeList* _tmp7;
		GeeIterator* _tmp8;
		GeeIterator* _exp_it;
		_tmp7 = NULL;
		_tmp8 = NULL;
		_exp_it = (_tmp8 = gee_iterable_iterator ((GeeIterable*) (_tmp7 = vala_for_statement_get_initializer (self))), (_tmp7 == NULL) ? NULL : (_tmp7 = (gee_collection_object_unref (_tmp7), NULL)), _tmp8);
		while (gee_iterator_next (_exp_it)) {
			ValaExpression* exp;
			GeeList* _tmp9;
			exp = (ValaExpression*) gee_iterator_get (_exp_it);
			_tmp9 = NULL;
			vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp9 = vala_code_node_get_error_types ((ValaCodeNode*) exp));
			(_tmp9 == NULL) ? NULL : (_tmp9 = (gee_collection_object_unref (_tmp9), NULL));
			(exp == NULL) ? NULL : (exp = (vala_code_node_unref (exp), NULL));
		}
		(_exp_it == NULL) ? NULL : (_exp_it = (gee_collection_object_unref (_exp_it), NULL));
	}
	{
		GeeList* _tmp10;
		GeeIterator* _tmp11;
		GeeIterator* _exp_it;
		_tmp10 = NULL;
		_tmp11 = NULL;
		_exp_it = (_tmp11 = gee_iterable_iterator ((GeeIterable*) (_tmp10 = vala_for_statement_get_iterator (self))), (_tmp10 == NULL) ? NULL : (_tmp10 = (gee_collection_object_unref (_tmp10), NULL)), _tmp11);
		while (gee_iterator_next (_exp_it)) {
			ValaExpression* exp;
			GeeList* _tmp12;
			exp = (ValaExpression*) gee_iterator_get (_exp_it);
			_tmp12 = NULL;
			vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp12 = vala_code_node_get_error_types ((ValaCodeNode*) exp));
			(_tmp12 == NULL) ? NULL : (_tmp12 = (gee_collection_object_unref (_tmp12), NULL));
			(exp == NULL) ? NULL : (exp = (vala_code_node_unref (exp), NULL));
		}
		(_exp_it == NULL) ? NULL : (_exp_it = (gee_collection_object_unref (_exp_it), NULL));
	}
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


ValaBlock* vala_for_statement_prepare_condition_split (ValaForStatement* self, ValaSemanticAnalyzer* analyzer) {
	ValaUnaryExpression* if_condition;
	ValaBlock* true_block;
	ValaBreakStatement* _tmp0;
	ValaIfStatement* if_stmt;
	ValaBooleanLiteral* _tmp1;
	ValaBlock* _tmp2;
	ValaBlock* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (analyzer != NULL, NULL);
	/* move condition into the loop body to allow split
	 in multiple statements*/
	if_condition = vala_unary_expression_new (VALA_UNARY_OPERATOR_LOGICAL_NEGATION, vala_for_statement_get_condition (self), vala_code_node_get_source_reference ((ValaCodeNode*) vala_for_statement_get_condition (self)));
	true_block = vala_block_new (vala_code_node_get_source_reference ((ValaCodeNode*) vala_for_statement_get_condition (self)));
	_tmp0 = NULL;
	vala_block_add_statement (true_block, (ValaStatement*) (_tmp0 = vala_break_statement_new (vala_code_node_get_source_reference ((ValaCodeNode*) vala_for_statement_get_condition (self)))));
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL));
	if_stmt = vala_if_statement_new ((ValaExpression*) if_condition, true_block, NULL, vala_code_node_get_source_reference ((ValaCodeNode*) vala_for_statement_get_condition (self)));
	vala_block_insert_statement (vala_for_statement_get_body (self), 0, (ValaStatement*) if_stmt);
	_tmp1 = NULL;
	vala_for_statement_set_condition (self, (ValaExpression*) (_tmp1 = vala_boolean_literal_new (TRUE, vala_code_node_get_source_reference ((ValaCodeNode*) self))));
	(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
	vala_code_node_check ((ValaCodeNode*) vala_for_statement_get_condition (self), analyzer);
	_tmp2 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = (_tmp2 = vala_for_statement_get_body (self), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2)), (if_condition == NULL) ? NULL : (if_condition = (vala_code_node_unref (if_condition), NULL)), (true_block == NULL) ? NULL : (true_block = (vala_code_node_unref (true_block), NULL)), (if_stmt == NULL) ? NULL : (if_stmt = (vala_code_node_unref (if_stmt), NULL)), _tmp3);
}


ValaExpression* vala_for_statement_get_condition (ValaForStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_condition;
}


void vala_for_statement_set_condition (ValaForStatement* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_condition = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_condition == NULL) ? NULL : (self->priv->_condition = (vala_code_node_unref (self->priv->_condition), NULL)), _tmp2);
	if (self->priv->_condition != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_condition, (ValaCodeNode*) self);
	}
}


ValaBlock* vala_for_statement_get_body (ValaForStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_body;
}


void vala_for_statement_set_body (ValaForStatement* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_body, (ValaCodeNode*) self);
}


static void vala_for_statement_class_init (ValaForStatementClass * klass) {
	vala_for_statement_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_for_statement_finalize;
	g_type_class_add_private (klass, sizeof (ValaForStatementPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_for_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_for_statement_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_for_statement_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->check = vala_for_statement_real_check;
}


static void vala_for_statement_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_for_statement_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_for_statement_instance_init (ValaForStatement * self) {
	self->priv = VALA_FOR_STATEMENT_GET_PRIVATE (self);
	self->priv->initializer = (GeeList*) gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->iterator = (GeeList*) gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_for_statement_finalize (ValaCodeNode* obj) {
	ValaForStatement * self;
	self = VALA_FOR_STATEMENT (obj);
	(self->priv->initializer == NULL) ? NULL : (self->priv->initializer = (gee_collection_object_unref (self->priv->initializer), NULL));
	(self->priv->iterator == NULL) ? NULL : (self->priv->iterator = (gee_collection_object_unref (self->priv->iterator), NULL));
	(self->priv->_condition == NULL) ? NULL : (self->priv->_condition = (vala_code_node_unref (self->priv->_condition), NULL));
	(self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL));
	VALA_CODE_NODE_CLASS (vala_for_statement_parent_class)->finalize (obj);
}


GType vala_for_statement_get_type (void) {
	static GType vala_for_statement_type_id = 0;
	if (vala_for_statement_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaForStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_for_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaForStatement), 0, (GInstanceInitFunc) vala_for_statement_instance_init, NULL };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_for_statement_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_for_statement_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaForStatement", &g_define_type_info, 0);
		g_type_add_interface_static (vala_for_statement_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_for_statement_type_id;
}




