/* valastringliteral.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valastringliteral.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaexpression.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valadatatype.h>




static char* string_substring (const char* self, glong offset, glong len);
struct _ValaStringLiteralPrivate {
	char* _value;
};

#define VALA_STRING_LITERAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_STRING_LITERAL, ValaStringLiteralPrivate))
enum  {
	VALA_STRING_LITERAL_DUMMY_PROPERTY
};
static void vala_string_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_string_literal_real_is_pure (ValaExpression* base);
static gboolean vala_string_literal_real_is_non_null (ValaExpression* base);
static char* vala_string_literal_real_to_string (ValaCodeNode* base);
static gboolean vala_string_literal_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_string_literal_parent_class = NULL;
static void vala_string_literal_finalize (ValaCodeNode* obj);



static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_warn_if_fail (offset >= 0);
	} else {
		g_warn_if_fail (offset <= string_length);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_warn_if_fail ((offset + len) <= string_length);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


/**
 * Creates a new string literal.
 *
 * @param s      the literal value
 * @param source reference to source code
 * @return       newly created string literal
 */
ValaStringLiteral* vala_string_literal_construct (GType object_type, const char* value, ValaSourceReference* source_reference) {
	ValaStringLiteral* self;
	g_return_val_if_fail (value != NULL, NULL);
	self = (ValaStringLiteral*) g_type_create_instance (object_type);
	vala_string_literal_set_value (self, value);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaStringLiteral* vala_string_literal_new (const char* value, ValaSourceReference* source_reference) {
	return vala_string_literal_construct (VALA_TYPE_STRING_LITERAL, value, source_reference);
}


/**
 * Evaluates the literal string value.
 *
 * @return the unescaped string
 */
char* vala_string_literal_eval (ValaStringLiteral* self) {
	char* noquotes;
	char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_value == NULL) {
		return NULL;
	}
	/* remove quotes */
	noquotes = string_substring (self->priv->_value, (glong) 1, (glong) ((guint) (g_utf8_strlen (self->priv->_value, -1) - 2)));
	/* unescape string */
	_tmp1 = NULL;
	return (_tmp1 = g_strcompress (noquotes), noquotes = (g_free (noquotes), NULL), _tmp1);
}


static void vala_string_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaStringLiteral * self;
	self = (ValaStringLiteral*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_string_literal (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static gboolean vala_string_literal_real_is_pure (ValaExpression* base) {
	ValaStringLiteral * self;
	self = (ValaStringLiteral*) base;
	return TRUE;
}


static gboolean vala_string_literal_real_is_non_null (ValaExpression* base) {
	ValaStringLiteral * self;
	self = (ValaStringLiteral*) base;
	return TRUE;
}


static char* vala_string_literal_real_to_string (ValaCodeNode* base) {
	ValaStringLiteral * self;
	const char* _tmp0;
	self = (ValaStringLiteral*) base;
	_tmp0 = NULL;
	return (_tmp0 = self->priv->_value, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
}


static gboolean vala_string_literal_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaStringLiteral * self;
	ValaDataType* _tmp1;
	self = (ValaStringLiteral*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	_tmp1 = NULL;
	vala_expression_set_value_type ((ValaExpression*) self, _tmp1 = vala_data_type_copy (analyzer->string_type));
	(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


const char* vala_string_literal_get_value (ValaStringLiteral* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_value;
}


void vala_string_literal_set_value (ValaStringLiteral* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_value = (g_free (self->priv->_value), NULL), _tmp2);
}


static void vala_string_literal_class_init (ValaStringLiteralClass * klass) {
	vala_string_literal_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_string_literal_finalize;
	g_type_class_add_private (klass, sizeof (ValaStringLiteralPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_string_literal_real_accept;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_string_literal_real_is_pure;
	VALA_EXPRESSION_CLASS (klass)->is_non_null = vala_string_literal_real_is_non_null;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_string_literal_real_to_string;
	VALA_CODE_NODE_CLASS (klass)->check = vala_string_literal_real_check;
}


static void vala_string_literal_instance_init (ValaStringLiteral * self) {
	self->priv = VALA_STRING_LITERAL_GET_PRIVATE (self);
}


static void vala_string_literal_finalize (ValaCodeNode* obj) {
	ValaStringLiteral * self;
	self = VALA_STRING_LITERAL (obj);
	self->priv->_value = (g_free (self->priv->_value), NULL);
	VALA_CODE_NODE_CLASS (vala_string_literal_parent_class)->finalize (obj);
}


GType vala_string_literal_get_type (void) {
	static GType vala_string_literal_type_id = 0;
	if (vala_string_literal_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaStringLiteralClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_string_literal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaStringLiteral), 0, (GInstanceInitFunc) vala_string_literal_instance_init, NULL };
		vala_string_literal_type_id = g_type_register_static (VALA_TYPE_LITERAL, "ValaStringLiteral", &g_define_type_info, 0);
	}
	return vala_string_literal_type_id;
}




