/* valabaseaccess.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valabaseaccess.h>
#include <gee/collection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valareport.h>
#include <vala/valastruct.h>
#include <vala/valaclass.h>
#include <vala/valaobjecttype.h>
#include <vala/valaobjecttypesymbol.h>
#include <vala/valadatatype.h>




enum  {
	VALA_BASE_ACCESS_DUMMY_PROPERTY
};
static void vala_base_access_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_base_access_real_to_string (ValaCodeNode* base);
static gboolean vala_base_access_real_is_pure (ValaExpression* base);
static gboolean vala_base_access_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_base_access_parent_class = NULL;



/**
 * Creates a new base access expression.
 *
 * @param source reference to source code
 * @return       newly created base access expression
 */
ValaBaseAccess* vala_base_access_construct (GType object_type, ValaSourceReference* source) {
	ValaBaseAccess* self;
	self = (ValaBaseAccess*) g_type_create_instance (object_type);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source);
	return self;
}


ValaBaseAccess* vala_base_access_new (ValaSourceReference* source) {
	return vala_base_access_construct (VALA_TYPE_BASE_ACCESS, source);
}


static void vala_base_access_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaBaseAccess * self;
	self = (ValaBaseAccess*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_base_access (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static char* vala_base_access_real_to_string (ValaCodeNode* base) {
	ValaBaseAccess * self;
	self = (ValaBaseAccess*) base;
	return g_strdup ("base");
}


static gboolean vala_base_access_real_is_pure (ValaExpression* base) {
	ValaBaseAccess * self;
	self = (ValaBaseAccess*) base;
	return TRUE;
}


static gboolean vala_base_access_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaBaseAccess * self;
	self = (ValaBaseAccess*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (!vala_semantic_analyzer_is_in_instance_method (analyzer)) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Base access invalid outside of instance methods");
		return FALSE;
	}
	if (analyzer->current_class == NULL) {
		GeeList* _tmp8;
		GeeIterator* _tmp9;
		GeeIterator* base_type_it;
		ValaDataType* _tmp10;
		if (analyzer->current_struct == NULL) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Base access invalid outside of class and struct");
			return FALSE;
		} else {
			GeeList* _tmp3;
			gboolean _tmp4;
			_tmp3 = NULL;
			if ((_tmp4 = gee_collection_get_size ((GeeCollection*) (_tmp3 = vala_struct_get_base_types (analyzer->current_struct))) != 1, (_tmp3 == NULL) ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL)), _tmp4)) {
				char* _tmp6;
				GeeList* _tmp5;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp6 = NULL;
				_tmp5 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp6 = g_strdup_printf ("Base access invalid without base type %d", gee_collection_get_size ((GeeCollection*) (_tmp5 = vala_struct_get_base_types (analyzer->current_struct)))));
				_tmp6 = (g_free (_tmp6), NULL);
				(_tmp5 == NULL) ? NULL : (_tmp5 = (gee_collection_object_unref (_tmp5), NULL));
				return FALSE;
			}
		}
		_tmp8 = NULL;
		_tmp9 = NULL;
		base_type_it = (_tmp9 = gee_iterable_iterator ((GeeIterable*) (_tmp8 = vala_struct_get_base_types (analyzer->current_struct))), (_tmp8 == NULL) ? NULL : (_tmp8 = (gee_collection_object_unref (_tmp8), NULL)), _tmp9);
		gee_iterator_next (base_type_it);
		_tmp10 = NULL;
		vala_expression_set_value_type ((ValaExpression*) self, _tmp10 = (ValaDataType*) gee_iterator_get (base_type_it));
		(_tmp10 == NULL) ? NULL : (_tmp10 = (vala_code_node_unref (_tmp10), NULL));
		(base_type_it == NULL) ? NULL : (base_type_it = (gee_collection_object_unref (base_type_it), NULL));
	} else {
		if (vala_class_get_base_class (analyzer->current_class) == NULL) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Base access invalid without base class");
			return FALSE;
		} else {
			ValaObjectType* _tmp12;
			_tmp12 = NULL;
			vala_expression_set_value_type ((ValaExpression*) self, (ValaDataType*) (_tmp12 = vala_object_type_new ((ValaObjectTypeSymbol*) vala_class_get_base_class (analyzer->current_class))));
			(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL));
		}
	}
	vala_expression_set_symbol_reference ((ValaExpression*) self, (ValaSymbol*) vala_data_type_get_data_type (vala_expression_get_value_type ((ValaExpression*) self)));
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


static void vala_base_access_class_init (ValaBaseAccessClass * klass) {
	vala_base_access_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->accept = vala_base_access_real_accept;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_base_access_real_to_string;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_base_access_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_base_access_real_check;
}


static void vala_base_access_instance_init (ValaBaseAccess * self) {
}


GType vala_base_access_get_type (void) {
	static GType vala_base_access_type_id = 0;
	if (vala_base_access_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaBaseAccessClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_base_access_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaBaseAccess), 0, (GInstanceInitFunc) vala_base_access_instance_init, NULL };
		vala_base_access_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaBaseAccess", &g_define_type_info, 0);
	}
	return vala_base_access_type_id;
}




