/* valaenumvalue.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaenumvalue.h>
#include <vala/valacodevisitor.h>
#include <vala/valaexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valaattribute.h>
#include <vala/valaenum.h>




struct _ValaEnumValuePrivate {
	ValaExpression* _value;
	char* cname;
};

#define VALA_ENUM_VALUE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ENUM_VALUE, ValaEnumValuePrivate))
enum  {
	VALA_ENUM_VALUE_DUMMY_PROPERTY
};
static void vala_enum_value_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_enum_value_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_enum_value_parent_class = NULL;
static void vala_enum_value_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new enum value.
 *
 * @param name enum value name
 * @return     newly created enum value
 */
ValaEnumValue* vala_enum_value_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaEnumValue* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = ((ValaEnumValue*) (g_type_create_instance (object_type)));
	self = ((ValaEnumValue*) (vala_symbol_construct (object_type, name, source_reference)));
	return self;
}


ValaEnumValue* vala_enum_value_new (const char* name, ValaSourceReference* source_reference) {
	return vala_enum_value_construct (VALA_TYPE_ENUM_VALUE, name, source_reference);
}


/**
 * Creates a new enum value with the specified numerical representation.
 *
 * @param name  enum value name
 * @param value numerical representation
 * @return      newly created enum value
 */
ValaEnumValue* vala_enum_value_construct_with_value (GType object_type, const char* name, ValaExpression* value, ValaSourceReference* source_reference) {
	ValaEnumValue* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (value != NULL, NULL);
	self = ((ValaEnumValue*) (g_type_create_instance (object_type)));
	self = ((ValaEnumValue*) (vala_enum_value_construct (object_type, name, source_reference)));
	vala_enum_value_set_value (self, value);
	return self;
}


ValaEnumValue* vala_enum_value_new_with_value (const char* name, ValaExpression* value, ValaSourceReference* source_reference) {
	return vala_enum_value_construct_with_value (VALA_TYPE_ENUM_VALUE, name, value, source_reference);
}


/**
 * Returns the string literal of this signal to be used in C code.
 * (FIXME: from vlaasignal.vala)
 *
 * @return string literal to be used in C code
 */
ValaCCodeConstant* vala_enum_value_get_canonical_cconstant (ValaEnumValue* self) {
	GString* str;
	const char* _tmp0;
	char* i;
	ValaCCodeConstant* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	str = g_string_new ("\"");
	_tmp0 = NULL;
	i = (_tmp0 = vala_symbol_get_name (((ValaSymbol*) (self))), (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* _tmp2;
		const char* _tmp1;
		c = g_utf8_get_char (i);
		if (c == '_') {
			g_string_append_c (str, '-');
		} else {
			g_string_append_unichar (str, g_unichar_tolower (c));
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		i = (_tmp2 = (_tmp1 = g_utf8_next_char (i), (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (i = (g_free (i), NULL)), _tmp2);
	}
	g_string_append_c (str, '"');
	_tmp3 = NULL;
	return (_tmp3 = vala_ccode_constant_new (str->str), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), _tmp3);
}


static void vala_enum_value_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaEnumValue * self;
	self = ((ValaEnumValue*) (base));
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_enum_value (visitor, self);
}


static void vala_enum_value_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaEnumValue * self;
	self = ((ValaEnumValue*) (base));
	g_return_if_fail (visitor != NULL);
	if (self->priv->_value != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (self->priv->_value)), visitor);
	}
}


/**
 * Process all associated attributes.
 */
void vala_enum_value_process_attributes (ValaEnumValue* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) (self))->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp1;
			ValaAttribute* a;
			_tmp1 = NULL;
			a = (_tmp1 = ((ValaAttribute*) (a_it->data)), (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0 && vala_attribute_has_argument (a, "cname")) {
					char* _tmp0;
					_tmp0 = NULL;
					self->priv->cname = (_tmp0 = vala_attribute_get_string (a, "cname"), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
				}
				(a == NULL ? NULL : (a = (vala_code_node_unref (a), NULL)));
			}
		}
	}
}


/**
 * Returns the name of this enum value as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_enum_value_get_cname (ValaEnumValue* self) {
	const char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->cname = (_tmp0 = vala_enum_value_get_default_cname (self), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->cname, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


char* vala_enum_value_get_default_cname (ValaEnumValue* self) {
	ValaEnum* _tmp0;
	ValaEnum* en;
	char* _tmp1;
	char* _tmp2;
	char* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = NULL;
	en = (_tmp0 = VALA_ENUM (vala_symbol_get_parent_symbol (((ValaSymbol*) (self)))), (_tmp0 == NULL ? NULL : vala_code_node_ref (_tmp0)));
	_tmp1 = NULL;
	_tmp2 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = (_tmp2 = g_strdup_printf ("%s%s", (_tmp1 = vala_enum_get_cprefix (en)), vala_symbol_get_name (((ValaSymbol*) (self)))), (_tmp1 = (g_free (_tmp1), NULL)), _tmp2), (en == NULL ? NULL : (en = (vala_code_node_unref (en), NULL))), _tmp3);
}


/**
 * Sets the name of this enum value to be used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_enum_value_set_cname (ValaEnumValue* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
}


ValaExpression* vala_enum_value_get_value (ValaEnumValue* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_value;
}


void vala_enum_value_set_value (ValaEnumValue* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_value == NULL ? NULL : (self->priv->_value = (vala_code_node_unref (self->priv->_value), NULL))), _tmp2);
}


static void vala_enum_value_class_init (ValaEnumValueClass * klass) {
	vala_enum_value_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_enum_value_finalize;
	g_type_class_add_private (klass, sizeof (ValaEnumValuePrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_enum_value_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_enum_value_real_accept_children;
}


static void vala_enum_value_instance_init (ValaEnumValue * self) {
	self->priv = VALA_ENUM_VALUE_GET_PRIVATE (self);
}


static void vala_enum_value_finalize (ValaCodeNode* obj) {
	ValaEnumValue * self;
	self = VALA_ENUM_VALUE (obj);
	(self->priv->_value == NULL ? NULL : (self->priv->_value = (vala_code_node_unref (self->priv->_value), NULL)));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	VALA_CODE_NODE_CLASS (vala_enum_value_parent_class)->finalize (obj);
}


GType vala_enum_value_get_type (void) {
	static GType vala_enum_value_type_id = 0;
	if (vala_enum_value_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaEnumValueClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_enum_value_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaEnumValue), 0, (GInstanceInitFunc) vala_enum_value_instance_init, NULL };
		vala_enum_value_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaEnumValue", &g_define_type_info, 0);
	}
	return vala_enum_value_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




