/* valaerrordomain.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaerrordomain.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valaerrorcode.h>
#include <vala/valamethod.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacreationmethod.h>
#include <vala/valareport.h>
#include <vala/valamember.h>
#include <vala/valaformalparameter.h>
#include <vala/valavaluetype.h>
#include <vala/valadatatype.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>




struct _ValaErrorDomainPrivate {
	GeeList* codes;
	GeeList* methods;
	char* cname;
	char* cprefix;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
};

#define VALA_ERROR_DOMAIN_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomainPrivate))
enum  {
	VALA_ERROR_DOMAIN_DUMMY_PROPERTY
};
static void vala_error_domain_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_error_domain_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_error_domain_real_get_cname (ValaTypeSymbol* base, gboolean const_type);
static char* vala_error_domain_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_error_domain_get_lower_case_csuffix (ValaErrorDomain* self);
static char* vala_error_domain_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_error_domain_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix);
static gboolean vala_error_domain_real_is_reference_type (ValaTypeSymbol* base);
static void vala_error_domain_set_cname (ValaErrorDomain* self, const char* cname);
static void vala_error_domain_process_ccode_attribute (ValaErrorDomain* self, ValaAttribute* a);
static char* vala_error_domain_real_get_type_id (ValaTypeSymbol* base);
static char* vala_error_domain_real_get_marshaller_type_name (ValaTypeSymbol* base);
static char* vala_error_domain_real_get_get_value_function (ValaTypeSymbol* base);
static char* vala_error_domain_real_get_set_value_function (ValaTypeSymbol* base);
static gpointer vala_error_domain_parent_class = NULL;
static void vala_error_domain_finalize (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new error domain.
 *
 * @param name             type name
 * @param source_reference reference to source code
 * @return                 newly created error domain
 */
ValaErrorDomain* vala_error_domain_new (const char* name, ValaSourceReference* source_reference) {
	ValaErrorDomain * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_ERROR_DOMAIN, 0, NULL);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	return self;
}


/**
 * Appends the specified code to the list of error codes.
 *
 * @param ecode an error code
 */
void vala_error_domain_add_code (ValaErrorDomain* self, ValaErrorCode* ecode) {
	g_return_if_fail (VALA_IS_ERROR_DOMAIN (self));
	g_return_if_fail (VALA_IS_ERROR_CODE (ecode));
	gee_collection_add (GEE_COLLECTION (self->priv->codes), ecode);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (ecode)), VALA_SYMBOL (ecode));
}


/**
 * Adds the specified method as a member to this error domain.
 *
 * @param m a method
 */
void vala_error_domain_add_method (ValaErrorDomain* self, ValaMethod* m) {
	g_return_if_fail (VALA_IS_ERROR_DOMAIN (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), "construction methods may only be declared within classes and structs");
		vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
		return;
	}
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		ValaFormalParameter* _tmp1;
		ValaValueType* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_method_set_this_parameter (m, (_tmp1 = vala_formal_parameter_new ("this", VALA_DATA_TYPE ((_tmp0 = vala_value_type_new (VALA_TYPESYMBOL (self)))), NULL)));
		(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
		(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
		vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (m)), vala_symbol_get_name (VALA_SYMBOL (vala_method_get_this_parameter (m))), VALA_SYMBOL (vala_method_get_this_parameter (m)));
	}
	gee_collection_add (GEE_COLLECTION (self->priv->methods), m);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (m)), VALA_SYMBOL (m));
}


/**
 * Returns a copy of the list of error codes.
 *
 * @return list of error codes
 */
GeeList* vala_error_domain_get_codes (ValaErrorDomain* self) {
	g_return_val_if_fail (VALA_IS_ERROR_DOMAIN (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_ERROR_CODE, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, self->priv->codes));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GeeList* vala_error_domain_get_methods (ValaErrorDomain* self) {
	g_return_val_if_fail (VALA_IS_ERROR_DOMAIN (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, self->priv->methods));
}


static void vala_error_domain_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaErrorDomain * self;
	self = VALA_ERROR_DOMAIN (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_error_domain (visitor, self);
}


static void vala_error_domain_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaErrorDomain * self;
	self = VALA_ERROR_DOMAIN (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* ecode_collection;
		int ecode_it;
		ecode_collection = self->priv->codes;
		for (ecode_it = 0; ecode_it < gee_collection_get_size (GEE_COLLECTION (ecode_collection)); ecode_it = ecode_it + 1) {
			ValaErrorCode* ecode;
			ecode = ((ValaErrorCode*) (gee_list_get (GEE_LIST (ecode_collection), ecode_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (ecode), visitor);
				(ecode == NULL ? NULL : (ecode = (g_object_unref (ecode), NULL)));
			}
		}
	}
	{
		GeeList* m_collection;
		int m_it;
		m_collection = self->priv->methods;
		for (m_it = 0; m_it < gee_collection_get_size (GEE_COLLECTION (m_collection)); m_it = m_it + 1) {
			ValaMethod* m;
			m = ((ValaMethod*) (gee_list_get (GEE_LIST (m_collection), m_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			}
		}
	}
}


static char* vala_error_domain_real_get_cname (ValaTypeSymbol* base, gboolean const_type) {
	ValaErrorDomain * self;
	const char* _tmp2;
	self = VALA_ERROR_DOMAIN (base);
	if (self->priv->cname == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cname = (_tmp1 = g_strdup_printf ("%s%s", (_tmp0 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), vala_symbol_get_name (VALA_SYMBOL (self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->cname, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static char* vala_error_domain_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaErrorDomain * self;
	const char* _tmp2;
	self = VALA_ERROR_DOMAIN (base);
	if (self->priv->lower_case_cprefix == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->lower_case_cprefix = (_tmp1 = g_strdup_printf ("%s_", (_tmp0 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), NULL))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->lower_case_cprefix, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


static char* vala_error_domain_get_lower_case_csuffix (ValaErrorDomain* self) {
	const char* _tmp1;
	g_return_val_if_fail (VALA_IS_ERROR_DOMAIN (self), NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->lower_case_csuffix = (_tmp0 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name (VALA_SYMBOL (self))), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->lower_case_csuffix, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


static char* vala_error_domain_real_get_lower_case_cname (ValaSymbol* base, const char* infix) {
	ValaErrorDomain * self;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	self = VALA_ERROR_DOMAIN (base);
	if (infix == NULL) {
		infix = "";
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s%s%s", (_tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), infix, (_tmp1 = vala_error_domain_get_lower_case_csuffix (self))), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
}


static char* vala_error_domain_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix) {
	ValaErrorDomain * self;
	char* _tmp0;
	char* _tmp1;
	self = VALA_ERROR_DOMAIN (base);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_utf8_strup ((_tmp0 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), NULL)), -1), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


static gboolean vala_error_domain_real_is_reference_type (ValaTypeSymbol* base) {
	ValaErrorDomain * self;
	self = VALA_ERROR_DOMAIN (base);
	return FALSE;
}


static void vala_error_domain_set_cname (ValaErrorDomain* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_ERROR_DOMAIN (self));
	g_return_if_fail (cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
}


/**
 * Returns the string to be prepended to the name of members of this
 * error domain when used in C code.
 *
 * @return the prefix to be used in C code
 */
char* vala_error_domain_get_cprefix (ValaErrorDomain* self) {
	const char* _tmp2;
	g_return_val_if_fail (VALA_IS_ERROR_DOMAIN (self), NULL);
	if (self->priv->cprefix == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cprefix = (_tmp1 = g_strdup_printf ("%s_", (_tmp0 = vala_typesymbol_get_upper_case_cname (VALA_TYPESYMBOL (self), NULL))), (self->priv->cprefix = (g_free (self->priv->cprefix), NULL)), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->cprefix, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


/**
 * Sets the string to be prepended to the name of members of this error
 * domain when used in C code.
 *
 * @param cprefix the prefix to be used in C code
 */
void vala_error_domain_set_cprefix (ValaErrorDomain* self, const char* cprefix) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_ERROR_DOMAIN (self));
	g_return_if_fail (cprefix != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cprefix = (_tmp1 = (_tmp0 = cprefix, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cprefix = (g_free (self->priv->cprefix), NULL)), _tmp1);
}


static void vala_error_domain_process_ccode_attribute (ValaErrorDomain* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_ERROR_DOMAIN (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_error_domain_set_cname (self, (_tmp0 = vala_attribute_get_string (a, "cname")));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* _tmp1;
		_tmp1 = NULL;
		vala_error_domain_set_cprefix (self, (_tmp1 = vala_attribute_get_string (a, "cprefix")));
		_tmp1 = (g_free (_tmp1), NULL);
	}
	if (vala_attribute_has_argument (a, "lower_case_csuffix")) {
		char* _tmp2;
		_tmp2 = NULL;
		self->priv->lower_case_csuffix = (_tmp2 = vala_attribute_get_string (a, "lower_case_csuffix"), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), _tmp2);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			int filename_collection_length1;
			char** filename_it;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp3;
				char* filename;
				_tmp3 = NULL;
				filename = (_tmp3 = *filename_it, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
				{
					vala_typesymbol_add_cheader_filename (VALA_TYPESYMBOL (self), filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) (g_free))), NULL);
		}
		val = (g_free (val), NULL);
	}
}


/**
 * Process all associated attributes.
 */
void vala_error_domain_process_attributes (ValaErrorDomain* self) {
	g_return_if_fail (VALA_IS_ERROR_DOMAIN (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = ((ValaAttribute*) (a_it->data)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_error_domain_process_ccode_attribute (self, a);
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static char* vala_error_domain_real_get_type_id (ValaTypeSymbol* base) {
	ValaErrorDomain * self;
	self = VALA_ERROR_DOMAIN (base);
	return g_strdup ("G_TYPE_POINTER");
}


static char* vala_error_domain_real_get_marshaller_type_name (ValaTypeSymbol* base) {
	ValaErrorDomain * self;
	self = VALA_ERROR_DOMAIN (base);
	return g_strdup ("POINTER");
}


static char* vala_error_domain_real_get_get_value_function (ValaTypeSymbol* base) {
	ValaErrorDomain * self;
	self = VALA_ERROR_DOMAIN (base);
	return g_strdup ("g_value_get_pointer");
}


static char* vala_error_domain_real_get_set_value_function (ValaTypeSymbol* base) {
	ValaErrorDomain * self;
	self = VALA_ERROR_DOMAIN (base);
	return g_strdup ("g_value_set_pointer");
}


static void vala_error_domain_class_init (ValaErrorDomainClass * klass) {
	vala_error_domain_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaErrorDomainPrivate));
	G_OBJECT_CLASS (klass)->finalize = vala_error_domain_finalize;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_error_domain_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_error_domain_real_accept_children;
	VALA_TYPESYMBOL_CLASS (klass)->get_cname = vala_error_domain_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_error_domain_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_error_domain_real_get_lower_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->get_upper_case_cname = vala_error_domain_real_get_upper_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_type = vala_error_domain_real_is_reference_type;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_error_domain_real_get_type_id;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_error_domain_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_error_domain_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_error_domain_real_get_set_value_function;
}


static void vala_error_domain_instance_init (ValaErrorDomain * self) {
	self->priv = VALA_ERROR_DOMAIN_GET_PRIVATE (self);
	self->priv->codes = GEE_LIST (gee_array_list_new (VALA_TYPE_ERROR_CODE, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_equal));
	self->priv->methods = GEE_LIST (gee_array_list_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_equal));
}


static void vala_error_domain_finalize (GObject * obj) {
	ValaErrorDomain * self;
	self = VALA_ERROR_DOMAIN (obj);
	(self->priv->codes == NULL ? NULL : (self->priv->codes = (g_object_unref (self->priv->codes), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_object_unref (self->priv->methods), NULL)));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	self->priv->cprefix = (g_free (self->priv->cprefix), NULL);
	self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL);
	self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL);
	G_OBJECT_CLASS (vala_error_domain_parent_class)->finalize (obj);
}


GType vala_error_domain_get_type (void) {
	static GType vala_error_domain_type_id = 0;
	if (vala_error_domain_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaErrorDomainClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_error_domain_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaErrorDomain), 0, (GInstanceInitFunc) vala_error_domain_instance_init };
		vala_error_domain_type_id = g_type_register_static (VALA_TYPE_TYPESYMBOL, "ValaErrorDomain", &g_define_type_info, 0);
	}
	return vala_error_domain_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) (array))[i] != NULL)
			destroy_func (((gpointer*) (array))[i]);
		}
		else
		for (i = 0; ((gpointer*) (array))[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) (array))[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




