/* valaenumvalue.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaenumvalue.h>
#include <vala/valaexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>
#include <vala/valaenum.h>




struct _ValaEnumValuePrivate {
	ValaExpression* _value;
	char* cname;
};

#define VALA_ENUM_VALUE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ENUM_VALUE, ValaEnumValuePrivate))
enum  {
	VALA_ENUM_VALUE_DUMMY_PROPERTY,
	VALA_ENUM_VALUE_VALUE
};
static void vala_enum_value_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_enum_value_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_enum_value_parent_class = NULL;
static void vala_enum_value_dispose (GObject * obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new enum value.
 *
 * @param name enum value name
 * @return     newly created enum value
 */
ValaEnumValue* vala_enum_value_new (const char* name, ValaSourceReference* source_reference) {
	ValaEnumValue * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_ENUM_VALUE, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Creates a new enum value with the specified numerical representation.
 *
 * @param name  enum value name
 * @param value numerical representation
 * @return      newly created enum value
 */
ValaEnumValue* vala_enum_value_new_with_value (const char* name, ValaExpression* value, ValaSourceReference* source_reference) {
	ValaEnumValue * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (value), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_ENUM_VALUE, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_enum_value_set_value (self, value);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Returns the string literal of this signal to be used in C code.
 * (FIXME: from vlaasignal.vala)
 *
 * @return string literal to be used in C code
 */
ValaCCodeConstant* vala_enum_value_get_canonical_cconstant (ValaEnumValue* self) {
	GString* str;
	const char* _tmp0;
	char* i;
	ValaCCodeConstant* _tmp3;
	g_return_val_if_fail (VALA_IS_ENUM_VALUE (self), NULL);
	str = g_string_new ("\"");
	_tmp0 = NULL;
	i = (_tmp0 = vala_symbol_get_name (VALA_SYMBOL (self)), (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* _tmp2;
		const char* _tmp1;
		c = g_utf8_get_char (i);
		if (c == '_') {
			g_string_append_c (str, '-');
		} else {
			g_string_append_unichar (str, g_unichar_tolower (c));
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		i = (_tmp2 = (_tmp1 = g_utf8_next_char (i), (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (i = (g_free (i), NULL)), _tmp2);
	}
	g_string_append_c (str, '"');
	_tmp3 = NULL;
	return (_tmp3 = vala_ccode_constant_new (str->str), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), _tmp3);
}


static void vala_enum_value_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaEnumValue * self;
	self = VALA_ENUM_VALUE (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_enum_value (visitor, self);
}


static void vala_enum_value_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaEnumValue * self;
	self = VALA_ENUM_VALUE (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (self->priv->_value != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (self->priv->_value), visitor);
	}
}


/**
 * Process all associated attributes.
 */
void vala_enum_value_process_attributes (ValaEnumValue* self) {
	g_return_if_fail (VALA_IS_ENUM_VALUE (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp1;
			ValaAttribute* a;
			_tmp1 = NULL;
			a = (_tmp1 = ((ValaAttribute*) (a_it->data)), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0 && vala_attribute_has_argument (a, "cname")) {
					char* _tmp0;
					_tmp0 = NULL;
					self->priv->cname = (_tmp0 = vala_attribute_get_string (a, "cname"), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


/**
 * Returns the name of this enum value as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_enum_value_get_cname (ValaEnumValue* self) {
	const char* _tmp3;
	g_return_val_if_fail (VALA_IS_ENUM_VALUE (self), NULL);
	if (self->priv->cname == NULL) {
		ValaEnum* _tmp0;
		ValaEnum* en;
		char* _tmp2;
		char* _tmp1;
		_tmp0 = NULL;
		en = (_tmp0 = VALA_ENUM (vala_symbol_get_parent_symbol (VALA_SYMBOL (self))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		_tmp2 = NULL;
		_tmp1 = NULL;
		self->priv->cname = (_tmp2 = g_strdup_printf ("%s%s", (_tmp1 = vala_enum_get_cprefix (en)), vala_symbol_get_name (VALA_SYMBOL (self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp2);
		_tmp1 = (g_free (_tmp1), NULL);
		(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->cname, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
}


ValaExpression* vala_enum_value_get_value (ValaEnumValue* self) {
	g_return_val_if_fail (VALA_IS_ENUM_VALUE (self), NULL);
	return self->priv->_value;
}


void vala_enum_value_set_value (ValaEnumValue* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_ENUM_VALUE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_value == NULL ? NULL : (self->priv->_value = (g_object_unref (self->priv->_value), NULL))), _tmp2);
}


static void vala_enum_value_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaEnumValue * self;
	self = VALA_ENUM_VALUE (object);
	switch (property_id) {
		case VALA_ENUM_VALUE_VALUE:
		g_value_set_object (value, vala_enum_value_get_value (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_enum_value_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaEnumValue * self;
	self = VALA_ENUM_VALUE (object);
	switch (property_id) {
		case VALA_ENUM_VALUE_VALUE:
		vala_enum_value_set_value (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_enum_value_class_init (ValaEnumValueClass * klass) {
	vala_enum_value_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaEnumValuePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_enum_value_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_enum_value_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_enum_value_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_enum_value_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_enum_value_real_accept_children;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_ENUM_VALUE_VALUE, g_param_spec_object ("value", "value", "value", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_enum_value_instance_init (ValaEnumValue * self) {
	self->priv = VALA_ENUM_VALUE_GET_PRIVATE (self);
}


static void vala_enum_value_dispose (GObject * obj) {
	ValaEnumValue * self;
	self = VALA_ENUM_VALUE (obj);
	(self->priv->_value == NULL ? NULL : (self->priv->_value = (g_object_unref (self->priv->_value), NULL)));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	G_OBJECT_CLASS (vala_enum_value_parent_class)->dispose (obj);
}


GType vala_enum_value_get_type (void) {
	static GType vala_enum_value_type_id = 0;
	if (G_UNLIKELY (vala_enum_value_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaEnumValueClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_enum_value_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaEnumValue), 0, (GInstanceInitFunc) vala_enum_value_instance_init };
		vala_enum_value_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaEnumValue", &g_define_type_info, 0);
	}
	return vala_enum_value_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




