/* valaformalparameter.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valaformalparameter.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>




struct _ValaFormalParameterPrivate {
	ValaParameterDirection _direction;
	gboolean _ellipsis;
	ValaExpression* _default_expression;
	gboolean _no_array_length;
	gboolean _construct_parameter;
	double _cparameter_position;
	double _carray_length_parameter_position;
	double _cdelegate_target_parameter_position;
	ValaDataType* _data_type;
};

#define VALA_FORMAL_PARAMETER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterPrivate))
enum  {
	VALA_FORMAL_PARAMETER_DUMMY_PROPERTY,
	VALA_FORMAL_PARAMETER_TYPE_REFERENCE,
	VALA_FORMAL_PARAMETER_DIRECTION,
	VALA_FORMAL_PARAMETER_ELLIPSIS,
	VALA_FORMAL_PARAMETER_DEFAULT_EXPRESSION,
	VALA_FORMAL_PARAMETER_NO_ARRAY_LENGTH,
	VALA_FORMAL_PARAMETER_CONSTRUCT_PARAMETER,
	VALA_FORMAL_PARAMETER_CPARAMETER_POSITION,
	VALA_FORMAL_PARAMETER_CARRAY_LENGTH_PARAMETER_POSITION,
	VALA_FORMAL_PARAMETER_CDELEGATE_TARGET_PARAMETER_POSITION
};
static void vala_formal_parameter_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_formal_parameter_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_formal_parameter_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static void vala_formal_parameter_process_ccode_attribute (ValaFormalParameter* self, ValaAttribute* a);
static GObject * vala_formal_parameter_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer vala_formal_parameter_parent_class = NULL;
static void vala_formal_parameter_dispose (GObject * obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new formal parameter.
 *
 * @param name   parameter name
 * @param type   parameter type
 * @param source reference to source code
 * @return       newly created formal parameter
 */
ValaFormalParameter* vala_formal_parameter_new (const char* _name, ValaDataType* type, ValaSourceReference* source) {
	ValaFormalParameter * self;
	g_return_val_if_fail (_name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (type), NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	self = g_object_newv (VALA_TYPE_FORMAL_PARAMETER, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), _name);
	vala_formal_parameter_set_type_reference (self, type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	return self;
}


/**
 * Creates a new ellipsis parameter representing an indefinite number of
 * parameters.
 */
ValaFormalParameter* vala_formal_parameter_new_with_ellipsis (ValaSourceReference* source) {
	ValaFormalParameter * self;
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	self = g_object_newv (VALA_TYPE_FORMAL_PARAMETER, 0, NULL);
	vala_formal_parameter_set_ellipsis (self, TRUE);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	return self;
}


static void vala_formal_parameter_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaFormalParameter * self;
	self = VALA_FORMAL_PARAMETER (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_formal_parameter (visitor, self);
}


static void vala_formal_parameter_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaFormalParameter * self;
	self = VALA_FORMAL_PARAMETER (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (!self->priv->_ellipsis) {
		vala_code_node_accept (VALA_CODE_NODE (vala_formal_parameter_get_type_reference (self)), visitor);
		if (self->priv->_default_expression != NULL) {
			vala_code_node_accept (VALA_CODE_NODE (self->priv->_default_expression), visitor);
		}
	}
}


static void vala_formal_parameter_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaFormalParameter * self;
	self = VALA_FORMAL_PARAMETER (base);
	g_return_if_fail (VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (VALA_IS_DATA_TYPE (new_type));
	if (vala_formal_parameter_get_type_reference (self) == old_type) {
		vala_formal_parameter_set_type_reference (self, new_type);
	}
}


static void vala_formal_parameter_process_ccode_attribute (ValaFormalParameter* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "array_length_pos")) {
		vala_formal_parameter_set_carray_length_parameter_position (self, vala_attribute_get_double (a, "array_length_pos"));
	}
	if (vala_attribute_has_argument (a, "delegate_target_pos")) {
		vala_formal_parameter_set_cdelegate_target_parameter_position (self, vala_attribute_get_double (a, "delegate_target_pos"));
	}
}


/**
 * Process all associated attributes.
 */
void vala_formal_parameter_process_attributes (ValaFormalParameter* self) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = ((ValaAttribute*) a_it->data), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_formal_parameter_process_ccode_attribute (self, a);
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


ValaFormalParameter* vala_formal_parameter_copy (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), NULL);
	if (!self->priv->_ellipsis) {
		ValaFormalParameter* result;
		result = vala_formal_parameter_new (vala_symbol_get_name (VALA_SYMBOL (self)), vala_formal_parameter_get_type_reference (self), vala_code_node_get_source_reference (VALA_CODE_NODE (self)));
		return result;
	} else {
		return vala_formal_parameter_new_with_ellipsis (NULL);
	}
}


ValaDataType* vala_formal_parameter_get_type_reference (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), NULL);
	return self->priv->_data_type;
}


void vala_formal_parameter_set_type_reference (ValaFormalParameter* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_data_type), VALA_CODE_NODE (self));
}


ValaParameterDirection vala_formal_parameter_get_direction (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), 0);
	return self->priv->_direction;
}


void vala_formal_parameter_set_direction (ValaFormalParameter* self, ValaParameterDirection value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_direction = value;
}


gboolean vala_formal_parameter_get_ellipsis (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), FALSE);
	return self->priv->_ellipsis;
}


void vala_formal_parameter_set_ellipsis (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_ellipsis = value;
}


ValaExpression* vala_formal_parameter_get_default_expression (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), NULL);
	return self->priv->_default_expression;
}


void vala_formal_parameter_set_default_expression (ValaFormalParameter* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_default_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_default_expression == NULL ? NULL : (self->priv->_default_expression = (g_object_unref (self->priv->_default_expression), NULL))), _tmp2);
}


gboolean vala_formal_parameter_get_no_array_length (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), FALSE);
	return self->priv->_no_array_length;
}


void vala_formal_parameter_set_no_array_length (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_no_array_length = value;
}


gboolean vala_formal_parameter_get_construct_parameter (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), FALSE);
	return self->priv->_construct_parameter;
}


void vala_formal_parameter_set_construct_parameter (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_construct_parameter = value;
}


double vala_formal_parameter_get_cparameter_position (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), 0.0);
	return self->priv->_cparameter_position;
}


void vala_formal_parameter_set_cparameter_position (ValaFormalParameter* self, double value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_cparameter_position = value;
}


double vala_formal_parameter_get_carray_length_parameter_position (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), 0.0);
	return self->priv->_carray_length_parameter_position;
}


void vala_formal_parameter_set_carray_length_parameter_position (ValaFormalParameter* self, double value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_carray_length_parameter_position = value;
}


double vala_formal_parameter_get_cdelegate_target_parameter_position (ValaFormalParameter* self) {
	g_return_val_if_fail (VALA_IS_FORMAL_PARAMETER (self), 0.0);
	return self->priv->_cdelegate_target_parameter_position;
}


void vala_formal_parameter_set_cdelegate_target_parameter_position (ValaFormalParameter* self, double value) {
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (self));
	self->priv->_cdelegate_target_parameter_position = value;
}


static GObject * vala_formal_parameter_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	ValaFormalParameterClass * klass;
	GObjectClass * parent_class;
	ValaFormalParameter * self;
	klass = VALA_FORMAL_PARAMETER_CLASS (g_type_class_peek (VALA_TYPE_FORMAL_PARAMETER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = VALA_FORMAL_PARAMETER (obj);
	{
		vala_symbol_set_access (VALA_SYMBOL (self), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	}
	return obj;
}


static void vala_formal_parameter_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaFormalParameter * self;
	self = VALA_FORMAL_PARAMETER (object);
	switch (property_id) {
		case VALA_FORMAL_PARAMETER_TYPE_REFERENCE:
		g_value_set_object (value, vala_formal_parameter_get_type_reference (self));
		break;
		case VALA_FORMAL_PARAMETER_DIRECTION:
		g_value_set_enum (value, vala_formal_parameter_get_direction (self));
		break;
		case VALA_FORMAL_PARAMETER_ELLIPSIS:
		g_value_set_boolean (value, vala_formal_parameter_get_ellipsis (self));
		break;
		case VALA_FORMAL_PARAMETER_DEFAULT_EXPRESSION:
		g_value_set_object (value, vala_formal_parameter_get_default_expression (self));
		break;
		case VALA_FORMAL_PARAMETER_NO_ARRAY_LENGTH:
		g_value_set_boolean (value, vala_formal_parameter_get_no_array_length (self));
		break;
		case VALA_FORMAL_PARAMETER_CONSTRUCT_PARAMETER:
		g_value_set_boolean (value, vala_formal_parameter_get_construct_parameter (self));
		break;
		case VALA_FORMAL_PARAMETER_CPARAMETER_POSITION:
		g_value_set_double (value, vala_formal_parameter_get_cparameter_position (self));
		break;
		case VALA_FORMAL_PARAMETER_CARRAY_LENGTH_PARAMETER_POSITION:
		g_value_set_double (value, vala_formal_parameter_get_carray_length_parameter_position (self));
		break;
		case VALA_FORMAL_PARAMETER_CDELEGATE_TARGET_PARAMETER_POSITION:
		g_value_set_double (value, vala_formal_parameter_get_cdelegate_target_parameter_position (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_formal_parameter_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaFormalParameter * self;
	self = VALA_FORMAL_PARAMETER (object);
	switch (property_id) {
		case VALA_FORMAL_PARAMETER_TYPE_REFERENCE:
		vala_formal_parameter_set_type_reference (self, g_value_get_object (value));
		break;
		case VALA_FORMAL_PARAMETER_DIRECTION:
		vala_formal_parameter_set_direction (self, g_value_get_enum (value));
		break;
		case VALA_FORMAL_PARAMETER_ELLIPSIS:
		vala_formal_parameter_set_ellipsis (self, g_value_get_boolean (value));
		break;
		case VALA_FORMAL_PARAMETER_DEFAULT_EXPRESSION:
		vala_formal_parameter_set_default_expression (self, g_value_get_object (value));
		break;
		case VALA_FORMAL_PARAMETER_NO_ARRAY_LENGTH:
		vala_formal_parameter_set_no_array_length (self, g_value_get_boolean (value));
		break;
		case VALA_FORMAL_PARAMETER_CONSTRUCT_PARAMETER:
		vala_formal_parameter_set_construct_parameter (self, g_value_get_boolean (value));
		break;
		case VALA_FORMAL_PARAMETER_CPARAMETER_POSITION:
		vala_formal_parameter_set_cparameter_position (self, g_value_get_double (value));
		break;
		case VALA_FORMAL_PARAMETER_CARRAY_LENGTH_PARAMETER_POSITION:
		vala_formal_parameter_set_carray_length_parameter_position (self, g_value_get_double (value));
		break;
		case VALA_FORMAL_PARAMETER_CDELEGATE_TARGET_PARAMETER_POSITION:
		vala_formal_parameter_set_cdelegate_target_parameter_position (self, g_value_get_double (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_formal_parameter_class_init (ValaFormalParameterClass * klass) {
	vala_formal_parameter_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaFormalParameterPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_formal_parameter_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_formal_parameter_set_property;
	G_OBJECT_CLASS (klass)->constructor = vala_formal_parameter_constructor;
	G_OBJECT_CLASS (klass)->dispose = vala_formal_parameter_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_formal_parameter_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_formal_parameter_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_formal_parameter_real_replace_type;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FORMAL_PARAMETER_TYPE_REFERENCE, g_param_spec_object ("type-reference", "type-reference", "type-reference", VALA_TYPE_DATA_TYPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FORMAL_PARAMETER_DIRECTION, g_param_spec_enum ("direction", "direction", "direction", VALA_TYPE_PARAMETER_DIRECTION, VALA_PARAMETER_DIRECTION_IN, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FORMAL_PARAMETER_ELLIPSIS, g_param_spec_boolean ("ellipsis", "ellipsis", "ellipsis", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FORMAL_PARAMETER_DEFAULT_EXPRESSION, g_param_spec_object ("default-expression", "default-expression", "default-expression", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FORMAL_PARAMETER_NO_ARRAY_LENGTH, g_param_spec_boolean ("no-array-length", "no-array-length", "no-array-length", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FORMAL_PARAMETER_CONSTRUCT_PARAMETER, g_param_spec_boolean ("construct-parameter", "construct-parameter", "construct-parameter", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FORMAL_PARAMETER_CPARAMETER_POSITION, g_param_spec_double ("cparameter-position", "cparameter-position", "cparameter-position", -G_MAXDOUBLE, G_MAXDOUBLE, 0.0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FORMAL_PARAMETER_CARRAY_LENGTH_PARAMETER_POSITION, g_param_spec_double ("carray-length-parameter-position", "carray-length-parameter-position", "carray-length-parameter-position", -G_MAXDOUBLE, G_MAXDOUBLE, 0.0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FORMAL_PARAMETER_CDELEGATE_TARGET_PARAMETER_POSITION, g_param_spec_double ("cdelegate-target-parameter-position", "cdelegate-target-parameter-position", "cdelegate-target-parameter-position", -G_MAXDOUBLE, G_MAXDOUBLE, 0.0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_formal_parameter_init (ValaFormalParameter * self) {
	self->priv = VALA_FORMAL_PARAMETER_GET_PRIVATE (self);
	self->priv->_direction = VALA_PARAMETER_DIRECTION_IN;
}


static void vala_formal_parameter_dispose (GObject * obj) {
	ValaFormalParameter * self;
	self = VALA_FORMAL_PARAMETER (obj);
	(self->priv->_default_expression == NULL ? NULL : (self->priv->_default_expression = (g_object_unref (self->priv->_default_expression), NULL)));
	(self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL)));
	G_OBJECT_CLASS (vala_formal_parameter_parent_class)->dispose (obj);
}


GType vala_formal_parameter_get_type (void) {
	static GType vala_formal_parameter_type_id = 0;
	if (G_UNLIKELY (vala_formal_parameter_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaFormalParameterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_formal_parameter_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaFormalParameter), 0, (GInstanceInitFunc) vala_formal_parameter_init };
		vala_formal_parameter_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaFormalParameter", &g_define_type_info, 0);
	}
	return vala_formal_parameter_type_id;
}



GType vala_parameter_direction_get_type (void) {
	static GType vala_parameter_direction_type_id = 0;
	if (G_UNLIKELY (vala_parameter_direction_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_PARAMETER_DIRECTION_IN, "VALA_PARAMETER_DIRECTION_IN", "in"}, {VALA_PARAMETER_DIRECTION_OUT, "VALA_PARAMETER_DIRECTION_OUT", "out"}, {VALA_PARAMETER_DIRECTION_REF, "VALA_PARAMETER_DIRECTION_REF", "ref"}, {0, NULL, NULL}};
		vala_parameter_direction_type_id = g_enum_register_static ("ValaParameterDirection", values);
	}
	return vala_parameter_direction_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




