// TTY-Grin
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "unix.h"
#include "tgws.h"
#include "tempfile.h"
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <fstream>

#define NUM_HEADERS (sizeof(headers)/sizeof(char *))
#define REAL_HEADERS 6
#define isword(x) (isalpha(x)||x=='\'')

void
tty_grin_composition_spellcheck(tgws::edit *body) {
  const char *command;
  try {
    command=config_get_string("ISPELL");
  }
  catch(config_error e) {
    return;
  }

  int in[2], out[3];

  if(pipe(in)<0) {
    tgws::error_message(strerror(errno));
    return;
  }
  if(pipe(out)<0) {
    close(in[0]);
    close(in[1]);
    tgws::error_message(strerror(errno));
    return;
  }

  switch(fork()) {
  case -1:
    close(in[0]);
    close(in[1]);
    close(out[0]);
    close(out[1]);
    tgws::error_message(strerror(errno));
    return;
  case 0:
    close(0);
    close(1);
    close(in[1]);
    close(out[0]);
    dup2(in[0], 0);
    dup2(out[1], 1);
    execl("/bin/sh", "sh", "-c", command, 0);
    exit(-1);
  }

  close(in[0]);
  close(out[1]);
  ofstream spin(in[1]);
  ifstream spout(out[0]);

  char buf[1024];

  spout.getline(buf, sizeof(buf));
  if(spout.fail()) {
    spin.close();
    spout.close();
    return;
  }

  crope text(body->get_text());
  unsigned int i=0;
  string word, context;

  tgws::label contextbuf;
  contextbuf.size(1, COLS);
  contextbuf.position(LINES-3, 0);
  contextbuf.show();

  tgws::entry wordbuf;
  wordbuf.size(1, COLS);
  wordbuf.position(LINES-2, 0);
  wordbuf.show();

  tgws::keyblock keys;
  keys.add("^G", _("Suggest"));
  keys.add("^P", _("Skip all"));
  keys.add("^A", _("Add"));
  keys.add("^C", _("Cancel"));
  keys.size(1, COLS);
  keys.position(LINES-1, 0);
  keys.show();

  while(i<text.size()) {
    while(i<text.size()&&!isword(text[i])) i++;
    while(i<text.size()&&isword(text[i])) word+=text[i++];
    if(word.size()) {
      spin<<word<<endl;
      
      spout.getline(buf, sizeof(buf));
      while(spout.get()!='\n') if(spout.fail()) {
	spin.close();
	spout.close();
	return;
      }

      if(buf[0]=='&'||buf[0]=='#') {
	string context;
	unsigned int j;

	// Not quite right, but I'm too lazy to fix it.
	for(j=0;j<i&&j<(unsigned)COLS&&text[i-j]!='\n';j++);
	if(text[i-j]=='\n') j--;
	for(j=i-j;j<text.size()&&text[j]!='\n'&&
	      context.size()<(unsigned)COLS;j++) context+=text[j];

	contextbuf.set_text(context.c_str());
	wordbuf.set_text(word.c_str());

	int key;
	do {
	  switch(key=wordbuf.run()) {
	  case 3:
	    if(tgws::yes_or_no(_("Apply changes so far (yes or no)? "))) {
	      body->set_text(text.c_str());
	      body->draw();
	    }
	    cerr.flush();
	    spin.close();
	    spout.close();
	    return;
	  case 7:
	    if(buf[0]=='#')
	      tgws::error_message(_("No suggestions available."));
	    else {
	      char *x=strchr(buf, ':');
	      if(x) {
		tgws::menu m;
		m.position(2, 4);
		x=strtok(x+2, ", ");
		while(x) {
		  m.add_item(x);
		  x=strtok(0, ", ");
		}
		m.size(m.count()+2, COLS/3);
		m.show();
		while((key=m.run())!='\n'&&key!=3);
		if(key=='\n') wordbuf.set_text(m.get_cursor_name());
		key=0;
	      }
	    }
	    break;
	  case 1:
	    spin<<'*'<<word<<endl;
	    break;
	  case 16:
	    spin<<'@'<<word<<endl;
	    key='\n';
	    break;
	  }
	} while(key!='\n');

	i-=word.size();
	text.erase(i, word.size());
	text.insert(i, wordbuf.get_text());
	i+=strlen(wordbuf.get_text());
      }
      
      word="";
    }
  }

  spin.close();
  spout.close();
  body->set_text(text.c_str());
  body->draw();
}

void
tty_grin_composition(composition& comp, const char *comment) {
  tgws::active_screen->push();

  static char *headers[]={N_("To"), N_("Subject"), N_("Newsgroups"),
			  N_("Cc"), N_("Bcc"), N_("References"),
			  N_("Posting server"), N_("Forwarded")};
  tgws::entry *header_widgets[NUM_HEADERS];

  {
    tgws::title *t=new tgws::title(comment);
    t->size(1, COLS);
    t->position(0, 0);
    t->show();

    tgws::keyblock *k=new tgws::keyblock;
    k->add("^C", _("Cancel"));
    k->add("^N", _("Attachments"));
    k->add("^B", _("Body"));
    k->add("^T", _("Attach..."));
    k->add("^K", _("Cut/remove"));
    k->add("^F", _("Send"));
    k->add("^P", _("Postpone"));
    k->add("^X", _("External"));
    k->add("^G", _("Spelling"));
    k->size(2, COLS);
    k->position(LINES-2, 0);
    k->show();

    for(unsigned int i=0;i<NUM_HEADERS;i++) {
      char buf[32];
      sprintf(buf, "%s:", gettext(headers[i]));
      tgws::label *l=new tgws::label(buf);
      l->size(1, 15);
      l->position(i+2, 0);
      l->show();
      header_widgets[i]=new tgws::entry;
      header_widgets[i]->set_text(comp.get_header(headers[i]));
      header_widgets[i]->size(1, COLS-40);
      header_widgets[i]->position(i+2, 15);
      header_widgets[i]->show();
    }
  }

  tgws::edit *body=new tgws::messageedit;
  body->set_text(comp.get_body());
  body->size(LINES-6-NUM_HEADERS, COLS);
  body->position(NUM_HEADERS+3, 0);
  body->show();

  tgws::clist *attachments=new tgws::clist(2);
  attachments->set_headings(true);
  attachments->set_title(0, _("Attachments"));
  attachments->set_width(0, 25);
  attachments->size(NUM_HEADERS, 25);
  attachments->position(2, COLS-25);
  attachments->show();

  // FIXME: tab to select newsgroups, server
  int key=0;
  unsigned int state=0, state2=0;
  do {
    switch(state2) {
    case 0: key=header_widgets[state]->run(); break;
    case 1: key=body->run(); break;
    case 2: key=attachments->run(); break;
    }
    if(isupper(key)) key+=32;
    switch(key) {
    case 7: tty_grin_composition_spellcheck(body); break;
    case 9:
      if(state==0||state==3||state==4) {
	string r;
	if(tty_grin_addresses_choose(r)) {
	  if(header_widgets[state]->length())
	    header_widgets[state]->append_text(" ");
	  header_widgets[state]->append_text(r.c_str());
	}
      }
      break;
    case 12: tgws::active_screen->redraw(); break;
    case 14:
      if(state2==2) state2=0;
      else if(!attachments->empty()) state2=2;
      break;
    case 2: state2=(state2==1)?0:1; break;
    case KEY_UP: if(state) state--; break;
    case KEY_DOWN: if(state<NUM_HEADERS-1) state++;
    case 11:
      if(state2==2) {
	delete attachments->get_cursor();
	if(attachments->empty()) state2=0;
      }
      break;
    case 20:
      {
	tgws::pilot *p=new tgws::pilot;
	p->size(LINES, COLS);
	p->position(0, 0);
	p->show();
	int x;
	do {
	  while((x=p->run())!='\n'&&x!=3);
	  if(x=='\n') {
	    if(access(p->get_filename(), R_OK))
	      tgws::error_message(strerror(errno));
	    else {
	      tgws::clist_node *n=attachments->insert(0);
	      const char *b=p->get_filename()+strlen(p->get_filename());
	      while(b>=p->get_filename()&&*b!='/') b--;
	      n->set_text(0, ++b);
	      n->set_text(1, p->get_filename());
	    }
	  }
	} while(x!='\n'&&x!=3);
	delete p;
      }
      break;
    case 24:
      try {
	string ed=config_get_string("EDITOR");
	tempfile t;
	ofstream out(t.filename());
	if(out.good()) {
	  out<<body->get_text();
	  out.close();
	  ed+=' ';
	  ed+=t.filename();
	  tgws::suspend();
	  system(ed.c_str());
	  tgws::unsuspend();
	  ifstream in(t.filename());
	  if(in.good()) {
	    body->set_text("");
	    body->append(in);
	    body->draw();
	    in.close();
	  }
	}
      }
      catch(config_error e) { }
      break;
    case 3:
      if(!tgws::yes_or_no(_("Abandon this message (yes or no)?"))) key=0;
      break;
    case 16:
    case 6:
      try {
	comp.clear();
	comp.set_body(body->get_text());
	for(unsigned int i=0;i<REAL_HEADERS;i++) 
	  if(header_widgets[i]->length())
	    comp.set_header(headers[i], header_widgets[i]->get_text());
	if(key==6) {
	  for(tgws::clist_node *n=attachments->get_first();n;n=n->get_next())
	    comp.attach(n->get_text(1),
			config_filename_to_mimetype(n->get_text(1)),
			n->get_text(0));
	  if(*header_widgets[7]->get_text())
	    comp.forward(header_widgets[7]->get_text());

	  char tmp[1024];
	  snprintf(tmp, sizeof(tmp), "%s <%s>",
		   config_get_string("REAL-NAME"), config_get_string("FROM"));
	  comp.set_header("From", tmp);
	  try {
	    comp.set_header("Reply-To", config_get_string("REPLY-TO"));
	    comp.set_header("Organization", config_get_string("ORGANIZATION"));
	  }
	  catch(config_error e) { }
	  comp.set_header("X-Mailer", "TTY-Grin version " VERSION
			  " -- Copyright (C) 2001 Daniel Beer");
	  if(!(header_widgets[0]->length()||header_widgets[2]->length()))
	    tgws::error_message(_("No recipients specified."));
	  else if(!(header_widgets[0]->length()||header_widgets[3]->length()||
		    header_widgets[4]->length())||
		  comp.sendmail(config_get_string("SENDMAIL"))) {
	    if(!header_widgets[2]->length()||
	       comp.news(header_widgets[6]->get_text())) key=3;
	    else tgws::error_message(_("Unable to send via news server."));
	  } else {
	    string error("sendmail: ");
	    error+=strerror(errno);
	    tgws::error_message(error.c_str());
	  }
	  ofstream outbox(config_get_string("OUTBOX"), ios::app);
	  if(outbox.good()) {
	    comp.outbox(outbox);
	    outbox.close();
	  }
	} else if(key==16) {
	  string save_as;
	  if(tgws::get_line(_("Save as:"), save_as)&&
	     (access(save_as.c_str(), F_OK)||
	      tgws::yes_or_no(_("Overwrite existing file (yes or no)?")))) {
	    ofstream out(save_as.c_str());
	    if(out.bad()) tgws::error_message(strerror(errno));
	    else {
	      for(unsigned int i=REAL_HEADERS;i<NUM_HEADERS;i++)
		out<<headers[i]<<": "<<header_widgets[i]->get_text()<<endl;
	      comp.postpone(out);
	      out.close();
	      key=3;
	    }
	  }
	}
      }
      catch(config_error e) { }
      break;
    }
  } while(key!=3);

  tgws::active_screen->pop();
}
