// Grin LISP
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <math.h>
#include <strstream>
#include "lisp.h"

lisp::std_library::std_library(void) {
  set_variable("PI", object(new real(M_PI)));

  static struct {
    const char *name;
    system_function_t code;
  } functions[]={
    {"+", add},
    {"-", subtract},
    {"*", multiply},
    {"/", divide},
    {"=", num_eq},
    {"/=", num_noteq},
    {"<", num_lt},
    {">", num_gt},
    {"<=", num_lteq},
    {">=", num_gteq},
    {"PRINT", print},
    {"CONS", make_cons},
    {"CAR", car},
    {"SETF CAR", setf_car},
    {"CDR", cdr},
    {"SETF CDR", setf_cdr},
    {"MAKE-ARRAY", make_array},
    {"AREF", aref},
    {"SETF AREF", setf_aref},
    {"MACROEXPAND", macroexpand},
    {"EQ", eq},
    {"EQUAL", equal},
    {"EQL", eql},
    {"MAKE-HASH-TABLE", make_hash_table},
    {"GETHASH", gethash},
    {"SETF GETHASH", setf_gethash},
    {"EXECUTE", execute},
    {"CONCATENATE", concatenate},
    {"LOAD", loadf},
    {"APPLY", apply},
    {"MAPCAR", mapcar},
    {"EVAL", eval},
    {"READ-FROM-STRING", read_from_string},
    {"REVERSE", reverse_sequence},
    {"NTH", nth},
    {"SETF NTH", setf_nth},
    {"LENGTH", length},
    {"SUBSEQ", subseq},
    {"TYPE-OF", type_of},
    {"TYPEP", typep},
    {"CHARACTER", make_character},
    {"CHAR-CODE", char_code},
    {"CHAR", string_char},
    {"SETF CHAR", setf_string_char},
    {"ERROR", user_error},
    {0, 0}
  };

  for(int i=0;functions[i].name;i++)
    set_function(functions[i].name,
		 object(new system_function(functions[i].code)));

  // Trivial functions, macros and structures

  static char *real_code="
(defun symbolp (x) (eq (type-of x) 'symbol))
(defun integerp (x) (eq (type-of x) 'integer))
(defun realp (x) (eq (type-of x) 'real))
(defun consp (x) (eq (type-of x) 'cons))
(defun functionp (x) (eq (type-of x) 'function))
(defun arrayp (x) (eq (type-of x) 'array))
(defun stringp (x) (eq (type-of x) 'string))
(defun hash-table-p (x) (eq (type-of x) 'hash-table))
(defun characterp (x) (eq (type-of x) 'char))

(defun listp (x) (if x (consp x) t))
(defun numberp (x) (or (realp x) (integerp x)))

(defun lower-case-p (x) (< 96 (char-code x) 123))
(defun upper-case-p (x) (< 64 (char-code x) 91))
(defun digit-char-p (x) (< 47 (char-code x) 58))
(defun alpha-char-p (x) (or (lower-case-p x) (upper-case-p x)))

(defun 1+ (x) (+ x 1))
(defun 1- (x) (- x 1))
(defun not (x) (if x nil t))
(defun list (&rest x) x)
(defun funcall (func &rest parm) (apply func parm))

(defmacro return (&optional x) (list 'return-from nil x))
(defmacro incf (x) (list 'setf x (list '+ x 1)))
(defmacro decf (x) (list 'setf x (list '- x 1)))
(defmacro push (value list) (list 'setf list (list 'cons value list)))
(defmacro pop (x) (list 'let (list (list 'temp (list 'car x)))
                        (list 'setf x (list 'cdr x)) 'temp))

(defmacro when (condition &rest sequence)
  (list 'if condition (cons 'progn sequence)))
(defmacro unless (condition &rest sequence)
  (list 'if condition nil (cons 'progn sequence)))

(defmacro or (&rest conditions)
  (unless conditions (return-from or nil))
  (let* ((con (reverse conditions)) (final (pop con)))
    (loop
     (unless con (return final))
     (let ((x (pop con))) (setq final (list 'if x x final))))))

(defmacro and (&rest conditions)
  (unless conditions (return-from and t))
  (let* ((con (reverse conditions)) (final (pop con)))
    (loop
     (unless con (return final))
     (let ((x (pop con)))
       (setq final (list 'if (list 'not x) nil final))))))

(defmacro cond (&rest conditions)
  (let ((con (reverse conditions)) final)
    (loop
     (unless con (return final))
     (let ((x (pop con)))
       (setq final (list 'if (car x) (cons 'progn (cdr x)) final))))))

(defmacro case (variable &rest conditions)
  (let ((con (reverse conditions)) final)
    (unless con (return (list 'let (list (list 'temp variable)) final)))
    (let ((x (pop con)))
      (setq final (list 'if (list 'eql 'temp (list 'quote (car x)))
			(cons 'progn (cdr x)) final)))))

(defmacro do (init terminate &rest iterate)
  (let (final-init final)
    (dolist (x (reverse init))
      (push (if (consp x) (if (cdr x) (list (car x) (car (cdr x))) x) x)
	    final-init))
    (dolist (x (reverse init))
      (if (consp x) (if (nth 2 x)
			(push (list 'setq (car x) (nth 2 x)) final))))
    (dolist (x (reverse iterate)) (push x final))
    (push (list 'when (car terminate) (list 'return (nth 1 terminate))) final)
    (list 'let final-init (cons 'loop final))))

(defmacro do* (&rest iterate)
  (cons 'let* (cdr (macroexpand (cons 'do iterate)))))

(defmacro dolist (clause &rest iterate)
  (list 'let (list (list (car clause) (nth 1 clause)))
	(list 'loop
	      (list 'unless (car clause) (list 'return (nth 2 clause)))
	      (cons 'let (cons (list (list (car clause)
					   (list 'car (car clause))))
			       iterate))
	      (list 'pop (car clause)))))
";

  istrstream code(real_code);
  object r;
  domain d(*this);

  code>>r;
  while(!code.eof()) {
    evaluate(d, r);
    code>>r;
  }
}

lisp::object
lisp::std_library::type_of(library& lib, int argc, object *argv) {
  if(argc<1) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  return object(new atom(argv[0].get_data()->get_typeof()));
}

lisp::object
lisp::std_library::typep(library& lib, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));

  string name(as_atom(argv[1]).get_text());
  name+="P";
  object func=lib.get_function(name.c_str());
  return assume_function(func).funcall(lib, 1, argv);
}

lisp::object
lisp::std_library::make_character(library& lib, int argc, object *argv) {
  if(argc<1) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));

  if(is_integer(argv[0]))
    return object(new character(assume_integer(argv[0]).get_value()));

  const char *s;

  if(is_str(argv[0])) s=assume_str(argv[0]).get_text();
  else if(is_atom(argv[0])) s=assume_atom(argv[0]).get_text();
  else throw error(_("Cannot be coerced to character"));

  if(!(*s&&!s[1])) throw error(_("Cannot be coerced to character"));
  return object(new character(*s));
}

lisp::object
lisp::std_library::char_code(library& lib, int argc, object *argv) {
  if(argc<1) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  return object(new integer(as_character(argv[0]).get_char()));
}

lisp::object
lisp::std_library::string_char(library& lib, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));

  int index=as_integer(argv[1]).get_value();
  const char *s=as_str(argv[0]).get_text();
  if(index<0||index>=(signed)strlen(s)) throw error(_("Out of bounds"));

  return object(new character(s[index]));
}

lisp::object
lisp::std_library::setf_string_char(library& lib, int argc, object *argv) {
  if(argc<3) throw error(_("Insufficient arguments"));
  if(argc>3) throw error(_("Excess arguments"));

  int index=as_integer(argv[2]).get_value();
  string s=as_str(argv[1]).get_text();
  if(index<0||index>=(signed)s.size()) throw error(_("Out of bounds"));
  s[index]=as_character(argv[0]).get_char();
  assume_str(argv[1]).set_text(s.c_str());

  return argv[0];
}

lisp::object
lisp::std_library::user_error(library& lib, int argc, object *argv) {
  if(argc<1) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  throw error(as_str(argv[0]).get_text());
}
