// TTY-Grin
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fstream>
#include <map>
#include "newsgroup.h"

class lt_string {
public:
  bool operator()(const string& a, const string& b) const { return a<b; }
};

bool
newsgroup_message::extract(ostream& extract_to) const {
  int fd=socket(PF_INET, SOCK_STREAM, 0);

  if(fd<0) return false;
  if(connect(fd, (struct sockaddr *)&parent->addr, sizeof(parent->addr))<0) {
    close(fd);
    return false;
  }
  char buf[4096];
  ifstream in(fd);
  ofstream out(fd);

  in.getline(buf, sizeof(buf));
  if(atoi(buf)!=200) {
    close(fd);
    return false;
  }
  out<<"GROUP "<<parent->get_group_name()<<"\r\n";
  out.flush();
  in.getline(buf, sizeof(buf));
  if(atoi(buf)!=211) {
    close(fd);
    return false;
  }
  out<<"ARTICLE "<<xref<<"\r\n";
  out.flush();
  in.getline(buf, sizeof(buf));
  if(atoi(buf)!=220) {
    close(fd);
    return false;
  }

  in.getline(buf, sizeof(buf));
  while(strcmp(buf, ".")&&strcmp(buf, ".\r")) {
    int i=strlen(buf);
    if(buf[i-1]=='\r') buf[i-1]=0;
    extract_to<<buf<<endl;
    in.getline(buf, sizeof(buf));
  }

  out<<"QUIT\r\n";
  out.flush();
  close(fd);
  return true;
}

newsgroup::newsgroup(const char *hostname, const char *group) :
  server_name(hostname), group_name(group) {
  struct hostent *info;
  int port;

  {
    const char *div=strchr(hostname, ':');
    if(div) {
      info=gethostbyname(string(hostname, 0, div-hostname).c_str());
      port=atoi(div+1);
    } else {
      info=gethostbyname(hostname);
      port=119;
    }
    if(!info) return;
  }
  
  addr.sin_family=AF_INET;
  addr.sin_addr=*(struct in_addr *)info->h_addr;
  addr.sin_port=htons(port);
  fetch_index();
}

newsgroup::~newsgroup(void) {
  if(size()) {
    struct group_flags f;
    vector<newsgroup_message>::const_iterator i;
    
    f.first=messages[0].get_xref();
    f.last=messages.back().get_xref();
    memset(f.flags, message_descriptor::EXPUNGE, MAX_ARTICLES);
    for(i=messages.begin();i!=messages.end();i++)
      f.flags[(*i).get_xref()-f.first]=(*i).get_flags();

    char outpath[1024];
    snprintf(outpath, sizeof(outpath), "%s/.ttygrin/news:%s/%s",
	     getenv("HOME"), server_name.c_str(), group_name.c_str());
    
    ofstream out(outpath);
    if(!out.fail()) {
      out.write(&f, sizeof(f));
      out.close();
    }
  }
}

void
newsgroup::fetch_index(void) {
  int fd=socket(PF_INET, SOCK_STREAM, 0);

  if(fd<0) return;
  if(connect(fd, (struct sockaddr *)&addr, sizeof(addr))<0) {
    close(fd);
    return;
  }

  ifstream in(fd);
  ofstream out(fd);
  char buf[10240];

  in.getline(buf, sizeof(buf));
  if(atoi(buf)!=200) {
    close(fd);
    return;
  }
  out<<"GROUP "<<group_name<<"\r\n";
  out.flush();
  in.getline(buf, sizeof(buf));
  if(atoi(buf)!=211) {
    close(fd);
    return;
  }
  int first, last;
  sscanf(buf, "%*d%*d%d%d", &first, &last);
  if(last-first>=MAX_ARTICLES) first=last+1-MAX_ARTICLES;

  out<<"XOVER "<<first<<'-'<<last<<"\r\n";
  out.flush();
  in.getline(buf, sizeof(buf)); 
  if(atoi(buf)!=224) {
    close(fd);
    return;
  }

  struct group_flags f;
  f.first=-1;
  f.last=-1;
  {
    char inpath[1024];

    snprintf(inpath, sizeof(inpath), "%s/.ttygrin/news:%s/%s",
	     getenv("HOME"), server_name.c_str(), group_name.c_str());
    ifstream in(inpath);
    if(in.fail()) f.last=0;
    else {
      in.read(&f, sizeof(f));
      in.close();
    }
  }

  map<string, newsgroup_message *, lt_string> midmap;

  in.getline(buf, sizeof(buf));
  while(buf[0]!='.'&&!in.fail()) {
    newsgroup_message msg(this);
    msg.xref=atoi(strtok(buf, "\t"));
    if(msg.xref>=f.first&&msg.xref<=f.last)
      msg.set_flags(f.flags[msg.xref-f.first]);
    if(!(msg.get_flags()&message_descriptor::EXPUNGE)) {
      msg.set_subject(strtok(NULL, "\t"));
      msg.set_from(strtok(NULL, "\t"));
      msg.set_date(strtok(NULL, "\t"));
      string mid=strtok(NULL, "\t");
      const char *r=strtok(NULL, "\t"), *rr;
      for(rr=r+strlen(r);rr>=r&&*rr!=' ';rr--);
      string ref(++rr);
      if(ref[0]=='<') {
	map<string, newsgroup_message *, lt_string>::iterator parent=
	  midmap.find(ref);
	if(parent!=midmap.end()) msg.in_reply_to=(*parent).second;
      }
      messages.push_back(msg);
      midmap[mid]=&messages.back();
    }
    in.getline(buf, sizeof(buf));
  }

  out<<"QUIT\r\n";
  out.flush();
  close(fd);
}

void
newsgroup::expunge(void) {
  vector<newsgroup_message>::iterator i;

  for(i=messages.begin();i!=messages.end();i++)
    if((*i).get_flags()&message_descriptor::DELETED)
      (*i).set_flags(message_descriptor::EXPUNGE);
}
