// TTY-Grin
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include "groupsearch.h"
#include "tgws.h"
#include "unix.h"

const char *
tty_grin_news_find_group(groupsearch& groups) {
  tgws::active_screen->push();
  {
    tgws::title *t=new tgws::title(_("Newsgroups"));
    t->size(1, COLS);
    t->position(0, 0);
    t->show();

    tgws::keyblock *k=new tgws::keyblock;
    k->add("^C", _("Cancel"));
    k->size(1, COLS);
    k->position(LINES-1, 0);
    k->show();
  }

  tgws::ctree *gr=new tgws::ctree(2);
  gr->set_width(0, COLS/3);
  gr->set_width(1, COLS*2/3);
  gr->size(LINES-4, COLS);
  gr->position(2, 0);
  gr->show();

  for(groupsearch::iterator i=groups.begin();i!=groups.end();i++) {
    tgws::ctree_node *n=0;
    const char *name=(*i).get_name();
    int j=0;

    while(name[j]!='.'&&name[j]) j++;
    while(name[j]) {
      j++;
      string sub(name, 0, j);

      tgws::ctree_node *testparent=n?n->get_last():gr->get_last();
      if(testparent&&sub==testparent->get_text(0)) n=testparent;
      else n=gr->insert(n, 0);
      n->set_text(0, sub.c_str());

      while(name[j]!='.'&&name[j]) j++;
    }

    n=gr->insert(n, 0);
    n->set_text(0, name);
    n->set_text(1, (*i).get_description());
    n->set_user_data(0, &(*i));
  }

  int key;
  while((key=gr->run())!=3&&!(key=='\n'&&!gr->get_cursor()->get_first()));

  const char *ret=key=='\n'?((groupsearch_group *)gr->get_cursor()->
			     get_user_data(0))->get_name():0;
  tgws::active_screen->pop();
  return ret;
}

void
tty_grin_news_groups(const char *server, const char *path) {
  DIR *d=opendir(path);

  if(!d) {
    tgws::error_message(strerror(errno));
    return;
  }

  tgws::active_screen->push();
  tgws::clist *groups=new tgws::clist(1);

  {
    struct dirent *ent;
    while((ent=readdir(d)))
      if(ent->d_name[0]!='.')
	groups->insert(0)->set_text(0, ent->d_name);
    closedir(d);

    groups->set_width(0, COLS);
    groups->size(LINES-4, COLS);
    groups->position(2, 0);
    groups->show();

    tgws::title *t=new tgws::title(server);
    t->size(1, COLS);
    t->position(0, 0);
    t->show();

    tgws::keyblock *k=new tgws::keyblock;
    k->add("<", _("Back"));
    k->add("^K", _("Delete"));
    k->add("^N", _("New"));
    k->add("^F", _("Find group"));
    k->size(1, COLS);
    k->position(LINES-1, 0);
    k->show();
  }

  int key;
  do {
    key=groups->run();
    switch(key) {
    case '\n':
    case ' ':
    case KEY_RIGHT:
      if(groups->get_cursor()) {
	newsgroup g(server, groups->get_cursor()->get_text(0));
	tty_grin_newsgroup(g);
      }
      break;
    case 11:
      if(groups->get_cursor()&&
	 tgws::yes_or_no(_("Delete this group (yes or no)?"))) {
	string full_name(path);
	full_name+='/';
	full_name+=groups->get_cursor()->get_text(0);
	unlink(full_name.c_str());
	delete groups->get_cursor();
      }
      break;
    case 14:
      {
	string name;
	if(tgws::get_line(_("Newsgroup name:"), name)) {
	  string full_name(path);
	  full_name+='/';
	  full_name+=name;
	  int fd=open(full_name.c_str(), O_CREAT|O_WRONLY|O_EXCL, 0600);
	  if(fd<0) tgws::error_message(strerror(errno));
	  else {
	    close(fd);
	    groups->insert(0)->set_text(0, name.c_str());
	  }
	}
      }
      break;
    case 6:
      {
	tgws::label l(_("Mask:"));
	l.size(1, 6);
	l.position(LINES-1, 0);
	l.show();

	tgws::entry ent;
	ent.set_text("*");
	ent.size(1, COLS-6);
	ent.position(LINES-1, 6);
	ent.show();

	int key;
	while((key=ent.run())!='\n'&&key!=3);

	if(key=='\n') {
	  groupsearch g(server, ent.get_text());
	  if(g.empty()) tgws::error_message(_("No groups could be found."));
	  else {
	    const char *name=tty_grin_news_find_group(g);
	    if(name) {
	      string full_name(path);
	      
	      full_name+='/';
	      full_name+=name;
	      int fd=open(full_name.c_str(), O_CREAT|O_WRONLY|O_EXCL, 0600);
	      if(fd<0) tgws::error_message(strerror(errno));
	      else {
		close(fd);
		groups->insert(0)->set_text(0, name);
	      }
	    }
	  }
	}
      }
      break;
    }
  } while(key!='<'&&key!=','&&key!=KEY_LEFT);

  tgws::active_screen->pop();
}

void
tty_grin_news(void) {
  tgws::active_screen->push();

  {
    tgws::title *t=new tgws::title(_("News servers"));
    t->size(1, COLS);
    t->position(0, 0);
    t->show();

    tgws::keyblock *k=new tgws::keyblock;
    k->add("<", _("Back"));
    k->add("^K", _("Delete"));
    k->add("^N", _("New"));
    k->size(1, COLS);
    k->position(LINES-1, 0);
    k->show();
  }

  tgws::clist *servers=new tgws::clist(2);
  {
    string config_dir(getenv("HOME"));
    config_dir+="/.ttygrin";
    DIR *d=opendir(config_dir.c_str());
    if(d) {
      dirent *ent;
      while((ent=readdir(d))) {
	struct stat s;
	string full_name(config_dir);
	full_name+='/';
	full_name+=ent->d_name;
	if(!stat(full_name.c_str(), &s)&&s.st_mode&S_IFDIR&&
	   !strncmp(ent->d_name, "news:", 5)) {
	  tgws::clist_node *n=servers->insert(0);
	  n->set_text(0, ent->d_name+5);
	  n->set_text(1, full_name.c_str());
	}
      }
      closedir(d);
    }
  }

  servers->set_width(0, COLS);
  servers->size(LINES-4, COLS);
  servers->position(2, 0);
  servers->show();

  int key;
  do {
    key=servers->run();
    switch(key) {
    case '\n':
    case ' ':
    case KEY_RIGHT:
      if(servers->get_cursor())
	tty_grin_news_groups(servers->get_cursor()->get_text(0),
			     servers->get_cursor()->get_text(1));
      break;
    case 11:
      if(servers->get_cursor()&&
	 tgws::yes_or_no(_("Delete this server (yes or no)?"))) {
	const char *path=servers->get_cursor()->get_text(1);
	DIR *d=opendir(path);
	if(d) {
	  struct dirent *ent;
	  while((ent=readdir(d))) {
	    string full(path);
	    full+='/';
	    full+=ent->d_name;
	    unlink(full.c_str());
	  }
	  closedir(d);
	}
	rmdir(path);
	delete servers->get_cursor();
      }
      break;
    case 14:
      {
	string name;
	if(tgws::get_line(_("Hostname:"), name)) {
	  string full_name(getenv("HOME"));
	  full_name+="/.ttygrin/news:";
	  full_name+=name;
	  if(mkdir(full_name.c_str(), 0700))
	    tgws::error_message(strerror(errno));
	  else {
	    tgws::clist_node *n=servers->insert(0);
	    n->set_text(0, name.c_str());
	    n->set_text(1, full_name.c_str());
	  }
	}
      }
      break;
    }
  } while(key!='<'&&key!=','&&key!=KEY_LEFT);

  tgws::active_screen->pop();
}
