// TTY-Grin
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <fstream>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>
#include "tgws.h"
#include "tempfile.h"
#include "unix.h"

void
tty_grin_message_attachments(message& msg) {
  tgws::active_screen->push();

  {
    tgws::title *t=new tgws::title(_("Attachments"));
    t->size(1, COLS);
    t->position(0, 0);
    t->show();

    tgws::keyblock *k=new tgws::keyblock;
    k->add("<", _("Back"));
    k->size(1, COLS);
    k->position(LINES-1, 0);
    k->show();
  }

  tgws::clist *parts=new tgws::clist(2);
  parts->size(LINES-4, COLS);
  parts->position(2, 0);
  parts->set_width(0, COLS/2);
  parts->set_width(1, COLS/2);
  parts->show();

  for(int i=0;i<msg.size();i++)
    if(!msg[i].is_container()) {
      tgws::clist_node *n=parts->insert(0);
      n->set_text(0, msg[i]["content-type/name"]);
      n->set_text(1, msg[i]["content-type"]);
      n->set_user_data(0, (void *)i);
    }

  int key(0);
  do {
    key=parts->run();
    if(isupper(key)) key+=32;
    switch(key) {
    case ' ':
    case KEY_RIGHT:
    case '.':
    case '>':
    case '\n':
      if(parts->get_cursor()) {
	tempfile t;
	ofstream f(t.filename());
	if(f.good()) {
	  msg.extract((int)parts->get_cursor()->get_user_data(0), f);
	  f.close();
	  try {
	    if(config_handle_attachment(t.filename(),
					parts->get_cursor()->get_text(1),
					parts->get_cursor()->get_text(0)))
	      break;
	  }
	  catch(config_error e) {
	    break;
	  }
	}
      }
    case 's':
      {
	string filename;
	if(parts->get_cursor()&&
	   tgws::get_line(_("Save as:"), filename)&&
	   (access(filename.c_str(), F_OK)||
	    tgws::yes_or_no(_("Overwrite existing file (yes or no)? ")))) {
	  ofstream out(filename.c_str());
	  if(out.good()) {
	    msg.extract((int)parts->get_cursor()->get_user_data(0), out);
	    out.close();
	  } else tgws::error_message(strerror(errno));
	}
      }
      break;
    }
  } while(key!='<'&&key!=','&&key!=KEY_LEFT);
  
  tgws::active_screen->pop();
}

void
tty_grin_message(message& msg, message_descriptor& descriptor) {
  tgws::active_screen->push();

  {
    int count(0);
    for(int i=0;i<msg.size();i++)
      if(!msg[i].is_container()&&
	 strcasecmp(msg[i]["content-type"], "text/plain")) count++;
    tgws::title *t;
    if(count) {
      char tmp[COLS];
      snprintf(tmp, sizeof(tmp),
	       _("[%d attachment(s) (press 'A' to view)] %s"),
	       count, descriptor.get_subject());
      t=new tgws::title(tmp);
    } else t=new tgws::title(descriptor.get_subject());
    t->size(1, COLS);
    t->position(0, 0);
    t->show();

    tgws::keyblock *k=new tgws::keyblock;
    k->add("<", _("Back"));
    k->add("D", _("Delete"));
    k->add("S", _("Save to folder"));
    k->add("A", _("Attachments"));
    k->add("E", _("Export"));
    k->add("R", _("Reply"));
    k->add("F", _("Forward"));
    k->add("W", _("Raw"));
    k->size(2, COLS);
    k->position(LINES-2, 0);
    k->show();
  }

  static char *to_show[]={"date", "from", "to", "subject", "newsgroups",
			  "cc", 0};
  static char *to_display[]={N_("Date:"), N_("From:"), N_("To:"),
			     N_("Subject:"), N_("Newsgroups:"), N_("Cc:"), 0};
  
  tgws::pager *p=new tgws::messagepager;

  {
    tempfile t;
    ofstream o(t.filename());
    bool hrule(false);
    for(int i=0;i<msg.size();i++) {
      if(hrule) {
	for(int j=0;to_show[j];j++) {
	  const char *v=msg[i][to_show[j]];
	  if(*v) {
	    if(hrule) o<<endl<<'\x01'<<endl<<endl;
	    hrule=false;
	    const char *d=gettext(to_display[j]);
	    o<<d;
	    for(int k=strlen(d);k<15;k++) o<<' ';
	    o<<v<<endl;
	  }
	}
	if(!hrule) o<<endl;
      }

      if(!strcasecmp(msg[i]["content-type"], "text/plain")) {
	if(hrule) o<<endl<<'\x01'<<endl<<endl;
	msg.extract(i, o);
	hrule=true;
      }
    }
    o.close();
    ifstream i(t.filename());
    p->read_file(i);
    i.close();
  }

  {
    int numheaders=0;
    for(int i=0;to_show[i];i++) {
      const char *value=msg[0][to_show[i]];
      if(*value) {
	tgws::label *name=new tgws::label(gettext(to_display[i]));
	tgws::label *v=new tgws::label(value);

	name->size(1, 15);
	name->position(numheaders+2, 0);
	name->show();
	v->size(1, COLS-15);
	v->position(numheaders+2, 15);
	v->show();
	numheaders++;
      }
    }
    p->size(LINES-6-numheaders, COLS);
    p->position(numheaders+3, 0);
  }

  p->show();

  int key(0);
  do {
    if(isupper(key=p->run())) key+=32;
    switch(key) {
    case 'd':
      descriptor.set_flag(message_descriptor::DELETED);
      key='<';
      break;
    case 's':
      {
	mailbox *m=tty_grin_mail_choose();
	if(m) {
	  if(!m->append(msg, descriptor.get_flags()))
	    tgws::error_message(_("Unable to save message."));
	  delete m;
	}
      }
      break;
    case 'e':
      {
	string filename;
	if(tgws::get_line(_("Save as:"), filename)) {
	  if(access(filename.c_str(), F_OK)||
	     tgws::yes_or_no(_("Overwrite existing file?"))) {
	    ofstream o(filename.c_str());
	    if(o.good()) {
	      msg.extract(o);
	      o.close();
	    } else tgws::error_message(strerror(errno));
	  }
	}
      }
      break;
    case 'f':
      {
	composition forward;
	string subject(msg[0]["subject"]);
	subject+=" (fwd)";
	forward.set_header(_("Subject"), subject.c_str());
	forward.set_header(_("Forwarded"), msg.get_filename());
	try {
	  ifstream sig(config_get_string("SIGNATURE"));
	  if(sig.good()) {
	    forward.append("\n\n--\n");
	    forward.append(sig);
	    sig.close();
	  }
	}
	catch(config_error e) { }
	tty_grin_composition(forward, subject.c_str());
      }
      break;
    case 'r':
      {
	composition reply;
	string subject(msg[0]["subject"]);
	if(strncasecmp(subject.c_str(), "re:", 3))
	  subject.insert(0, "Re: ");
	reply.set_header("Subject", subject.c_str());
	{
	  const char *newsgroups=msg[0]["newsgroups"];
	  const char *replyto=msg[0]["reply-to"];
	  string references(msg[0]["references"]);
	  if(references.size()) references+=' ';
	  references+=msg[0]["message-id"];
	  reply.set_header("References", references.c_str());
	  if(!*replyto) replyto=msg[0]["from"];
	  if(*newsgroups) {
	    const char *x=msg[0]["xref"], *y=strchr(x, ' ');
	    string xref;
	    if(y) xref=string(x, 0, y-x);
	    else xref=x;
	    switch(tgws::choice(_("Reply via (N)ews, (M)ail or (B)oth?"),
				"NMB")) {
	    case 0:
	      reply.set_header("Posting server", xref.c_str());
	      reply.set_header("Newsgroups", newsgroups); break;
	    case 2:
	      reply.set_header("Posting server", xref.c_str());
	      reply.set_header("Newsgroups", newsgroups);
	    case 1: reply.set_header("To", replyto); break;
	    }
	  } else reply.set_header("To", replyto);
	  char tmp[1024];
	  snprintf(tmp, sizeof(tmp), _("On %s, %s wrote:\n"),
		   msg[0]["date"], msg[0]["from"]);
	  reply.append(tmp);
	  const char *body=p->get_text();
	  do {
	    string line(">");
	    while(*body&&*body!='\n') line+=*(body++);
	    if(*body=='\n') body++;
	    line+='\n';
	    reply.append(line.c_str());
	  } while(*body&&*body!=1);
	  try {
	    ifstream sig(config_get_string("SIGNATURE"));
	    if(sig.good()) {
	      reply.append("\n\n--\n");
	      reply.append(sig);
	      sig.close();
	    }
	  }
	  catch(config_error e) { }
	}
	tty_grin_composition(reply, subject.c_str());
	descriptor.set_flag(message_descriptor::ANSWERED);
      }
      break;
    case 'a': tty_grin_message_attachments(msg); break;
    case 'w':
      {
	ifstream in(msg.get_filename());
	if(in.good()) {
	  tgws::active_screen->push();
	  tgws::pager *raw=new tgws::pager;
	  raw->read_file(in);
	  raw->size(LINES-4, COLS);
	  raw->position(2, 0);
	  raw->show();
	  in.close();
	  tgws::title *t=new tgws::title(_("Raw message data"));
	  t->size(1, COLS);
	  t->position(0, 0);
	  t->show();
	  tgws::keyblock *k=new tgws::keyblock;
	  k->add("<", _("Back"));
	  k->size(1, COLS);
	  k->position(LINES-1, 0);
	  k->show();
	  int key;
	  do {
	    key=raw->run();
	    if(key==12) tgws::active_screen->redraw();
	  } while(key!='<'&&key!=','&&key!=KEY_LEFT);
	  tgws::active_screen->pop();
	}
      }
      break;
    }
  } while(key!=','&&key!='<'&&key!=KEY_LEFT);

  descriptor.set_flag(message_descriptor::READ);
  tgws::active_screen->pop();
}
