// TTY-Grin Widget Set
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <ctype.h>
#include "tgws.h"

void
tgws::edit::size(int h, int w) {
  input::size(h, w);
  rewrap();
}

void
tgws::edit::draw(bool running) {
  wbkgdset(win, look->get_attr(EDIT));
  werase(win);

  if(scr_line_number>cursor_line_number||
     scr_line_number+height-1<cursor_line_number) {
    scr_line_number=cursor_line_number;
    scr_line=cursor_line;
    scr=cursor-cursor_column;
    for(unsigned int i=0;i<(unsigned)height/2&&scr_line_number;i++) {
      --scr_line_number;
      scr-=*(--scr_line);
    }
  }

  unsigned int begin=scr, i, lines;
  wrap_t::iterator line=scr_line;

  for(lines=0;lines<(unsigned)height&&line!=wrap.end();lines++) {
    i=*line-1;
    if(begin+i<buffer.size()&&!isspace(buffer[begin+i])&&
       buffer[begin+i]!='\n') ++i;
    draw_line(begin, i);
    begin+=*(line++);
  }

  mvwaddstr(win, cursor_line_number-scr_line_number, cursor_column, "");
  input::draw();
}

void
tgws::edit::draw_line(unsigned int begin, unsigned int len) {
  for(unsigned int i=0;i<len;i++) waddch(win, (unsigned char)buffer[begin+i]);
  waddch(win, '\n');
}

int
tgws::edit::run(void) {
  for(;;) {
    draw();
    doupdate();
    switch(int key=wgetch(win)) {
    case KEY_UP: cursor_up(); break;
    case KEY_DOWN: cursor_down(); break;
    case KEY_LEFT: cursor_left(); break;
    case KEY_RIGHT: cursor_right(); break;
    case KEY_PPAGE: for(int i=0;i<height;i++) cursor_up(); break;
    case KEY_NPAGE: for(int i=0;i<height;i++) cursor_down(); break;
    case KEY_HOME: home(); break;
    case KEY_LL:
    case KEY_END: end(); break;
    case 8:
    case 127: delete_back(); break;
    case 4: delete_forward(); break;
    case 1: sol(); break;
    case 5: eol(); break;
    case 21: cut_back(); break;
    case 11: cut_forward(); break;
    case 25: yank(); break;
    default:
      if(key=='\n'||(key>=' '&&key<256)) insert(key);
      else return key;
      break;
    }
  }
}

void
tgws::edit::rewrap(void) {
  cut.erase(0, cut.size());
  wrap.clear();
  if(buffer.size()) {
    unsigned int begin=0, len;

    for(begin=0;begin<buffer.size();begin+=len)
      wrap.push_back(len=where_to_break(begin));
  } else wrap.push_back(1);
  scr=cursor=cursor_line_number=scr_line_number=cursor_column=
    saved_cursor_column=lastcut=0;
  scr_line=cursor_line=wrap.begin();
}

void
tgws::edit::cursor_up(void) {
  if(cursor_line_number) {
    cursor_line_number--;
    cursor_line--;
    cursor=cursor-cursor_column-*cursor_line;
    cursor+=(cursor_column=saved_cursor_column<*cursor_line-1?
	     saved_cursor_column:*cursor_line-1);
  }
}

void
tgws::edit::cursor_down(void) {
  wrap_t::iterator next_line=cursor_line;
  if(++next_line!=wrap.end()) {
    cursor_line_number++;
    cursor=cursor-cursor_column+*cursor_line;
    ++cursor_line;
    cursor+=(cursor_column=saved_cursor_column<*cursor_line-1?
	     saved_cursor_column:*cursor_line-1);
  }
}

void
tgws::edit::cursor_left(void) {
  if(cursor) {
    cursor--;
    if(cursor_column) cursor_column--;
    else {
      cursor_line_number--;
      cursor_column=*(--cursor_line)-1;
    }
    saved_cursor_column=cursor_column;
  }
}

void
tgws::edit::cursor_right(void) {
  if(cursor<buffer.size()) {
    cursor++;
    if(cursor_column+1<*cursor_line) cursor_column++;
    else {
      cursor_line_number++;
      ++cursor_line;
      cursor_column=0;
    }
    saved_cursor_column=cursor_column;
  }
}

void
tgws::edit::insert(char c) {
  buffer.insert(cursor, c);
  cursor++;
  cursor_column++;
  saved_cursor_column++;
  (*cursor_line)++;
  bust();
}

void
tgws::edit::delete_back(void) {
  if(!cursor) return;
  cursor_left();
  delete_forward();
}

void
tgws::edit::delete_forward(void) {
  if(cursor>=buffer.size()) return;
  buffer.erase(cursor, 1);
  (*cursor_line)--;
  bust();
  if(*cursor_line<=cursor_column) cursor_left();
}

void
tgws::edit::yank(void) {
  // I don't think the efficiency of this could get very much worse.
  // I'll fix it up later, perhaps.  I was thinking that the cut buffer
  // could be sliced into chunks width-1 in size and inserted without
  // disrupting the word wrap.
  for(unsigned int i=0;i<cut.size();i++) insert(cut[i]);
}

void
tgws::edit::sol(void) {
  cursor-=cursor_column;
  cursor_column=saved_cursor_column=0;
}

void
tgws::edit::eol(void) {
  cursor+=*cursor_line-1-cursor_column;
  cursor_column=saved_cursor_column=*cursor_line-1;
}

void
tgws::edit::home(void) {
  cursor_column=cursor=cursor_line_number=0;
  cursor_line=wrap.begin();
}

void
tgws::edit::end(void) {
  --(cursor_line=wrap.end());
  cursor_line_number=wrap.size()-1;
  saved_cursor_column=cursor_column=*cursor_line-1;
  cursor=buffer.size();
}

void
tgws::edit::cut_back(void) {
  if(lastcut!=cursor) cut.erase(0, cut.size());
  cut.append(buffer.substr(cursor-cursor_column, cursor_column));
  buffer.erase(cursor-cursor_column, cursor_column);
  cursor-=cursor_column;
  *cursor_line-=cursor_column;
  saved_cursor_column=cursor_column=0;
  bust();
  lastcut=cursor;
}

void
tgws::edit::cut_forward(void) {
  if(lastcut!=cursor) cut.erase(0, cut.size());
  if(cursor_column+1<*cursor_line) {
    cut.append(buffer.substr(cursor, *cursor_line-cursor_column-1));
    buffer.erase(cursor, *cursor_line-cursor_column-1);
    *cursor_line=cursor_column+1;
    bust();
  } else if(cursor<buffer.size()) {
    cut.append('\n');
    delete_forward();
  }
  lastcut=cursor;
}

// The purpose of this function is to correct the wrapping of screen lines
// in the buffer after it's been modified by insert or delete_forward.

void
tgws::edit::bust(void) {
  if(cursor_line_number) {
    wrap_t::iterator prev=cursor_line;
    --prev;
    bust_line(cursor-cursor_column-*prev, prev);
  }
  bust_line(cursor-cursor_column, cursor_line);
}

void
tgws::edit::bust_line(unsigned int begin, tgws::edit::wrap_t::iterator line) {
  unsigned int len=where_to_break(begin);
  if(len<*line) {
    wrap_t::iterator next_line=line;
    ++next_line;

    if(buffer[begin+*line]=='\n'||begin+*line+1>=buffer.size()) {
      wrap.insert(next_line, *line-len);
      *line=len;
    } else {
      if(cursor_line==next_line) {
	cursor_column+=*line-len;
	saved_cursor_column=cursor_column;
      }
      *next_line+=*line-len;
      *line=len;
      // This could be done by looping rather than recursion.
      bust_line(begin+len, next_line);
    }

    if(line==cursor_line&&cursor_column>=len) {
      cursor_column-=len;
      saved_cursor_column=cursor_column;
      cursor_line_number++;
      ++cursor_line;
    }
  } else if(len>*line) {
    wrap_t::iterator next_line=line;
    ++next_line;

    if(next_line==cursor_line) {
      if(cursor_column<len-*line) {
	--cursor_line;
	cursor_line_number--;
	cursor_column+=*line;
      } else {
	cursor_column-=len-*line;
      }
      saved_cursor_column=cursor_column;
    }

    if(*next_line+*line<=len) {
      *line+=*next_line;
      wrap.erase(next_line);
    } else {
      *next_line-=len-*line;
      *line=len;
      bust_line(begin+len, next_line);
    }
  }
}

unsigned int
tgws::edit::where_to_break(unsigned int begin) {
  unsigned int end, max=(unsigned)width-1;

  for(end=begin;end<buffer.size()&&buffer[end]!='\n'&&end-begin<max;end++);
  if(end<buffer.size()&&buffer[end]!='\n') {
    while(isspace(buffer[end])&&end>begin) end--;
    while(!isspace(buffer[end])&&end>begin) end--;
    if(begin==end) end=begin+max-1;
    else end++;
  } else end++;
  return end-begin;
}

void
tgws::edit::append(istream& in) {
  int i;

  while((i=in.get())>=0) buffer+=(char)i;
  rewrap();
}
