// Grin LISP
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <math.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <string.h>
#include <strstream>
#include <algorithm>
#include "lisp.h"

lisp::object
lisp::std_library::print(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  cout<<argv[0]<<endl;
  return argv[0];
}

lisp::object
lisp::std_library::make_cons(library& lib, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));
  return object(new cons(argv[0], argv[1]));
}

lisp::object
lisp::std_library::car(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  return as_cons(argv[0]).car;
}

lisp::object
lisp::std_library::setf_car(library& lib, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));
  return as_cons(argv[1]).car=argv[0];
}

lisp::object
lisp::std_library::cdr(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  return as_cons(argv[0]).cdr;
}

lisp::object
lisp::std_library::setf_cdr(library& lib, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));
  return as_cons(argv[1]).cdr=argv[0];
}

lisp::object
lisp::std_library::add(library& lib, int argc, object *argv) {
  object n(new integer(0));
  for(int i=0;i<argc;i++)
    n=assume_number(n).binary(number::ADD, argv[i]);
  return n;
}

lisp::object
lisp::std_library::subtract(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  object n(argv[0]);
  as_number(n);
  for(int i=1;i<argc;i++)
    n=assume_number(n).binary(number::SUBTRACT, argv[i]);
  return n;
}

lisp::object
lisp::std_library::multiply(library& lib, int argc, object *argv) {
  object n(new integer(1));
  for(int i=0;i<argc;i++)
    n=assume_number(n).binary(number::MULTIPLY, argv[i]);
  return n;
}

lisp::object
lisp::std_library::divide(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  object n(argv[0]);
  as_number(n);
  for(int i=1;i<argc;i++)
    n=assume_number(n).binary(number::DIVIDE, argv[i]);
  return n;
}

lisp::object
lisp::std_library::num_lt(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  as_number(argv[0]);
  for(int i=1;i<argc;i++)
    if(assume_number(argv[i-1]).binary(number::LT, argv[i]).eq(nil))
      return nil;
  return t;
}

lisp::object
lisp::std_library::num_eq(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  as_number(argv[0]);
  for(int i=1;i<argc;i++)
    if(assume_number(argv[i-1]).binary(number::EQ, argv[i]).eq(nil))
      return nil;
  return t;
}

lisp::object
lisp::std_library::num_noteq(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  as_number(argv[0]);
  for(int i=1;i<argc;i++)
    if(!assume_number(argv[i-1]).binary(number::EQ, argv[i]).eq(nil))
      return nil;
  return t;
}

lisp::object
lisp::std_library::num_gt(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  as_number(argv[0]);
  for(int i=1;i<argc;i++)
    if(assume_number(argv[i-1]).binary(number::GT, argv[i]).eq(nil))
      return nil;
  return t;
}

lisp::object
lisp::std_library::num_lteq(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  as_number(argv[0]);
  for(int i=1;i<argc;i++)
    if(assume_number(argv[i-1]).binary(number::LTEQ, argv[i]).eq(nil))
      return nil;
  return t;
}

lisp::object
lisp::std_library::num_gteq(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  as_number(argv[0]);
  for(int i=1;i<argc;i++)
    if(assume_number(argv[i-1]).binary(number::GTEQ, argv[i]).eq(nil))
      return nil;
  return t;
}

lisp::object
lisp::std_library::make_array(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  if(is_cons(argv[0])) {
    int i=0;
    for(object x=argv[0];is_cons(x);x=assume_cons(x).cdr) i++;
    object a(new array(i));
    i=0;
    for(object x=argv[0];is_cons(x);x=assume_cons(x).cdr)
      assume_array(a).set_dim(i++,
			      as_integer(assume_cons(x).car).get_value());
    assume_array(a).allocate();
    return a;
  } else {
    object a(new array(1));
    assume_array(a).set_dim(0, as_integer(argv[0]).get_value());
    assume_array(a).allocate();
    return a;
  }
}

lisp::object
lisp::std_library::aref(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  int num=as_array(argv[0]).num_dim();
  if(argc-1<num) throw error(_("Insufficient arguments"));
  if(argc-1>num) throw error(_("Excess arguments"));
  if(!num) return nil;
  int ref=as_integer(argv[1]).get_value();
  if(ref<0||ref>=assume_array(argv[0]).get_dim(0))
    throw error(_("Out of bounds"));
  for(int i=1;i<num;i++) {
    ref*=assume_array(argv[0]).get_dim(i);
    int x=as_integer(argv[i+1]).get_value();
    if(x<0||x>=assume_array(argv[0]).get_dim(i))
      throw error(_("Out of bounds"));
    ref+=x;
  }
  return assume_array(argv[0])[ref];
}

lisp::object
lisp::std_library::setf_aref(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  int num=as_array(argv[1]).num_dim();
  if(argc-2<num) throw error(_("Insufficient arguments"));
  if(argc-2>num) throw error(_("Excess arguments"));
  if(!num) return argv[0];
  int ref=as_integer(argv[2]).get_value();
  if(ref<0||ref>=assume_array(argv[1]).get_dim(0))
    throw error(_("Out of bounds"));
  for(int i=1;i<num;i++) {
    ref*=assume_array(argv[1]).get_dim(i);
    int x=as_integer(argv[i+2]).get_value();
    if(x<0||x>=assume_array(argv[1]).get_dim(i))
      throw error(_("Out of bounds"));
    ref+=x;
  }
  return assume_array(argv[1])[ref]=argv[0];
}

lisp::object
lisp::std_library::macroexpand(library& lib, int argc, object *argv) {
  if(!argc) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  return lib.expand_macro(argv[0]);
}

lisp::object
lisp::std_library::eq(library& lib, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));
  if(is_atom(argv[0])||is_number(argv[0])||is_character(argv[0]))
    return argv[0].equal(argv[1])?t:nil;
  return argv[0].eq(argv[1])?t:nil;
}

lisp::object
lisp::std_library::equal(library& lib, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));
  return argv[0].equal(argv[1])?t:nil;
}

lisp::object
lisp::std_library::eql(library& lib, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));
  if(is_number(argv[0])&&is_number(argv[1]))
    return assume_number(argv[0]).binary(number::EQ, argv[1]);
  return argv[0].eq(argv[1])?t:nil;
}

lisp::object
lisp::std_library::make_hash_table(library& lib, int argc, object *argv) {
  if(!argc) return object(new hash_table(lib.get_function("EQL")));
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));
  if(strcmp(as_atom(argv[0]).get_text(), ":TEST"))
    throw error(_("Syntax error"));
  return object(new hash_table(argv[1]));
}

lisp::object
lisp::std_library::gethash(library& lib, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));
  return as_hash_table(argv[1]).readhash(lib, argv[0]);
}

lisp::object
lisp::std_library::setf_gethash(library& lib, int argc, object *argv) {
  if(argc<3) throw error(_("Insufficient arguments"));
  if(argc>3) throw error(_("Excess arguments"));
  return *as_hash_table(argv[2]).gethash(lib, argv[1])=argv[0];
}

lisp::object
lisp::std_library::execute(library& lib, int argc, object *argv) {
  const char *var[argc];

  for(int i=0;i<argc;i++) var[i]=as_str(argv[i]).get_text();

  switch(fork()) {
  case -1: return lisp::nil;
  case 0:
    {
      char *varcopy[argc+1];
      for(int i=0;i<argc;i++) {
	varcopy[i]=new char[strlen(var[i])+1];
	strcpy(varcopy[i], var[i]);
      }
      varcopy[argc]=0;
      execv(var[0], varcopy);
      exit(-1);
    }
  }

  int value;
  wait(&value);
  return object(new integer(WEXITSTATUS(value)));
}

lisp::object
lisp::std_library::loadf(library& lib, int argc, object *argv) {
  if(argc<1) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  lib.load(as_str(argv[0]).get_text());
  return t;
}

lisp::object
lisp::std_library::apply(library& lib, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  int num=0;
  for(object x=argv[argc-1];is_cons(x);x=assume_cons(x).cdr) num++;
  object arg[num+argc-2];
  for(num=0;num<argc-2;num++) arg[num]=argv[num+1];
  for(object x=argv[argc-1];is_cons(x);x=assume_cons(x).cdr)
    arg[num++]=assume_cons(x).car;
  return as_function(argv[0]).funcall(lib, num, arg);
}

lisp::object
lisp::std_library::mapcar(library& lib, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));
  object r;
  as_function(argv[0]);
  for(object x=argv[1];is_cons(x);x=assume_cons(x).cdr) {
    object car=assume_cons(x).car;
    car=assume_function(argv[0]).funcall(lib, 1, &car);
    r=object(new cons(car, r));
  }
  object rev;
  for(object x=r;is_cons(x);x=assume_cons(x).cdr)
    rev=object(new cons(assume_cons(x).car, rev));
  return rev;
}

lisp::object
lisp::std_library::eval(library& lib, int argc, object *argv) {
  if(argc<1) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  domain d(lib);
  return lib.evaluate(d, argv[0]);
}

lisp::object
lisp::std_library::read_from_string(library& lib, int argc, object *argv) {
  if(argc<1) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  istrstream in(as_str(argv[0]).get_text());
  object r;
  in>>r;
  return r;
}

lisp::object
lisp::std_library::concatenate(library& lib, int argc, object *argv) {
  // SEQUENCES
  if(argc<1) throw error(_("Insufficient arguments"));
  const char *seq=as_atom(argv[0]).get_text();
  if(!strcmp(seq, "STRING")) {
    string data;
    for(int i=1;i<argc;i++) data+=as_str(argv[i]).get_text();
    return object(new str(data.c_str()));
  } else if(!strcmp(seq, "LIST")) {
    object stub;
    for(int i=1;i<argc;i++) {
      for(object x=argv[i];is_cons(x);x=assume_cons(x).cdr)
	stub=object(new cons(assume_cons(x).car, stub));
    }
    object rev;
    for(object x=stub;is_cons(x);x=assume_cons(x).cdr)
      rev=object(new cons(assume_cons(x).car, rev));
    return rev;
  } else if(!strcmp(seq, "ARRAY")) {
    int total=0;
    for(int i=1;i<argc;i++) {
      if(as_array(argv[i]).num_dim()!=1) throw error(_("Not a sequence"));
      total+=assume_array(argv[i]).get_dim(0);
    }
    array *a=new array(1);
    a->set_dim(0, total);
    a->allocate();
    total=0;
    for(int i=1;i<argc;i++) {
      for(int j=0;j<assume_array(argv[i]).get_dim(0);j++)
	(*a)[total++]=assume_array(argv[i])[j];
    }
    return object(a);
  }
  throw error(_("Unknown sequence type"));
}

lisp::object
lisp::std_library::reverse_sequence(library& lib, int argc, object *argv) {
  // SEQUENCES
  if(argc<1) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  if(argv[0].eq(nil)) return nil;
  if(is_str(argv[0])) {
    string s=assume_str(argv[0]).get_text();
    reverse(s.begin(), s.end());
    return object(new str(s.c_str()));
  } else if(is_cons(argv[0])) {
    object x;
    for(object y=argv[0];is_cons(y);y=assume_cons(y).cdr)
      x=object(new cons(assume_cons(y).car, x));
    return x;
  } else if(is_array(argv[0])) {
    if(assume_array(argv[0]).num_dim()!=1) throw error(_("Not a sequence"));
    int size=assume_array(argv[0]).get_dim(0);
    array *a=new array(1);
    a->set_dim(0, size);
    a->allocate();
    for(int i=0;i<size;i++) (*a)[i]=assume_array(argv[0])[size-i-1];
    return object(a);
  }
  throw error(_("Not a sequence"));
}

lisp::object
lisp::std_library::nth(library&, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));
  int num=as_integer(argv[0]).get_value();
  if(num<0) throw error(_("Out of bounds"));
  for(object x=argv[1];is_cons(x);x=assume_cons(x).cdr)
    if(!(num--)) return assume_cons(x).car;
  return nil;
}

lisp::object
lisp::std_library::setf_nth(library&, int argc, object *argv) {
  if(argc<3) throw error(_("Insufficient arguments"));
  if(argc>3) throw error(_("Excess arguments"));
  int num=as_integer(argv[1]).get_value();
  if(num<0) throw error(_("Out of bounds"));
  for(object x=argv[2];is_cons(x);x=assume_cons(x).cdr)
    if(!(num--)) return assume_cons(x).car=argv[0];
  throw error(_("Out of bounds"));
}

lisp::object
lisp::std_library::length(library&, int argc, object *argv) {
  // SEQUENCES
  if(argc<1) throw error(_("Insufficient arguments"));
  if(argc>1) throw error(_("Excess arguments"));
  if(argv[0].eq(nil)) return object(new integer(0));
  if(is_str(argv[0]))
    return object(new integer(strlen(assume_str(argv[0]).get_text())));
  else if(is_cons(argv[0])) {
    int size=0;
    for(object x=argv[0];is_cons(x);x=assume_cons(x).cdr) size++;
    return object(new integer(size));
  } else if(is_array(argv[0])) {
    if(assume_array(argv[0]).num_dim()!=1) throw error(_("Not a sequence"));
    return object(new integer(assume_array(argv[0]).get_dim(0)));
  }
  throw error(_("Not a sequence"));
}

lisp::object
lisp::std_library::subseq(library&, int argc, object *argv) {
  // SEQUENCES
  if(argc<3) throw error(_("Insufficient arguments"));
  if(argc>3) throw error(_("Excess arguments"));
  int begin=as_integer(argv[1]).get_value(),
    end=as_integer(argv[2]).get_value();
  if(begin<0||end<begin) throw error(_("Out of bounds"));

  if(is_str(argv[0])) {
    const char *text=assume_str(argv[0]).get_text();
    if(end>(signed)strlen(text)) throw error(_("Out of bounds"));
    string s(text, begin, end-begin);
    return object(new str(s.c_str()));
  } else if(is_cons(argv[0])) {
    object stub;
    int count=0;
    for(object x=argv[0];is_cons(x);x=assume_cons(x).cdr) {
      if(count>=end) {
	object rev;
	for(object x=stub;is_cons(x);x=assume_cons(x).cdr)
	  rev=object(new cons(assume_cons(x).car, rev));
	return rev;
      }
      if(count>=begin) stub=object(new cons(assume_cons(x).car, stub));
      count++;
    }
    throw error(_("Out of bounds"));
  } else if(is_array(argv[0])) {
    if(assume_array(argv[0]).num_dim()!=1) throw error(_("Not a sequence"));
    if(end>assume_array(argv[0]).get_dim(0)) throw error(_("Out of bounds"));
    array *a=new array(1);
    a->set_dim(0, end-begin);
    a->allocate();
    for(int i=begin;i<end;i++) (*a)[i-begin]=assume_array(argv[0])[i];
    return object(a);
  }
  throw error(_("Not a sequence"));
}
