// Grin LISP
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <errno.h>
#include <string.h>
#include <fstream>
#include "lisp.h"

lisp::library::library(void) {
  set_function("structread", object(new system_function(structread)));
  set_function("structwrite", object(new system_function(structwrite)));
  set_function("structbuild", object(new system_function(structbuild)));
  set_function("structp", object(new system_function(structp)));
}

lisp::library::~library(void) {
}

void
lisp::library::set_function(const char *name, object data) {
  string s(name);
  if(is_real_function(data))
    assume_real_function(data).set_name(object(new atom(name)));
  functions[s]=data;
}

lisp::object
lisp::library::get_function(const char *name) const {
  map<string, object, lt_string>::const_iterator i;
  string s(name);

  if((i=functions.find(s))==functions.end()) {
    string f(_("Unknown function: "));
    f+=name;
    throw error(nil, f.c_str());
  }
  return (*i).second;
}

void
lisp::library::set_macro(const char *name, object data) {
  string s(name);
  if(is_real_function(data))
    assume_real_function(data).set_name(object(new atom(name)));
  macros[s]=data;
}

lisp::object
lisp::library::expand_macro(object f) {
  for(;;) {
    if(is_cons(f)&&is_atom(assume_cons(f).car)) {
      string s(assume_atom(assume_cons(f).car).get_text());
      map<string, object, lt_string>::iterator i;
      
      if((i=macros.find(s))!=macros.end()) {
	int argc=0;
	object x;
	
	for(x=assume_cons(f).cdr;is_cons(x);x=assume_cons(x).cdr) argc++;
	object argv[argc];
	argc=0;
	for(x=assume_cons(f).cdr;is_cons(x);x=assume_cons(x).cdr)
	  argv[argc++]=assume_cons(x).car;
	f=assume_real_function((*i).second).funcall(*this, argc, argv);
      } else return f;
    } else return f;
  }
  return f;
}

lisp::object
lisp::library::get_variable(const char *name) {
  map<string, object, lt_string>::const_iterator i;
  string s(name);

  if((i=variables.find(s))==variables.end()) {
    if(!strcmp(name, "T")) return t;
    if(!strcmp(name, "NIL")) return nil;
    string s(_("Undefined symbol: "));
    s+=name;
    throw error(nil, s.c_str());
  }
  return (*i).second;
}

lisp::object *
lisp::library::get_variable_reference(const char *name) {
  if(!(strcmp(name, "T")&&strcmp(name, "NIL"))) {
    string s(_("Symbol is constant: "));
    s+=name;
    throw error(nil, s.c_str());
  }

  string s(name);
  return &variables[s];
}

void
lisp::library::set_variable(const char *name, object what) {
  if(!(strcmp(name, "T")&&strcmp(name, "NIL"))) {
    string s(_("Symbol is constant: "));
    s+=name;
    throw error(nil, s.c_str());
  }

  string s(name);
  variables[s]=what;
}

void
lisp::library::defstruct(const char *name, structdef& s) {
  // Avoid using too much memory while compiling
  defstruct_1(name, s);
  defstruct_2(name, s);
  defstruct_3(name, s);
}

void
lisp::library::defstruct_1(const char *name, structdef& s) {
  string n(name);
  structures[n]=s;

  { // (defun struct-p (var) (structp var 'struct)
    real_function *f=new real_function;
    f->add_form(object(new cons(object(new atom("var")), nil)));
    { // (structp var 'struct)
      object c;
      { // 'struct
	object d;
	d=object(new cons(object(new atom(name)), d));
	d=object(new cons(object(new atom("QUOTE")), d));
	c=object(new cons(d, c));
      }
      c=object(new cons(object(new atom("var")), c));
      c=object(new cons(object(new atom("structp")), c));
      f->add_form(c);
    }
    n=name;
    n+="-P";
    set_function(n.c_str(), object(f));
  }
}

void
lisp::library::defstruct_2(const char *name, structdef& s) {
  // (defun make-struct (&rest elements) (structbuild 'struct elements))
  real_function *f=new real_function;
  { // (&rest elements)
    object c;
    c=object(new cons(object(new atom("elements")), c));
    c=object(new cons(object(new atom("&REST")), c));
    f->add_form(c);
  }
  { // (structbuild 'struct elements)
    object c;
    c=object(new cons(object(new atom("elements")), c));
    { // 'struct
      object d;
      d=object(new cons(object(new atom(name)), d));
      d=object(new cons(object(new atom("QUOTE")), d));
      c=object(new cons(d, c));
    }
    c=object(new cons(object(new atom("structbuild")), c));
    f->add_form(c);
  }
  string n("MAKE-");
  n+=name;
  set_function(n.c_str(), object(f));
}

void
lisp::library::defstruct_3(const char *name, structdef& s) {
  for(int i=0;i<(signed)s.size();i++) {
    // (defun struct-element (struct) (structread struct i))
    real_function *f=new real_function;
    f->add_form(object(new cons(object(new atom("struct")), nil)));
    { // (structread struct i)
      object c;
      c=object(new cons(object(new integer(i)), c));
      c=object(new cons(object(new atom("struct")), c));
      c=object(new cons(object(new atom("structread")), c));
      f->add_form(c);
    }
    string n(name);
    n+='-';
    n+=s[i];
    set_function(n.c_str(), object(f));

    // (defun SETF struct-element (value struct) (structwrite value struct i))
    f=new real_function;
    { // (value struct)
      object c;
      c=object(new cons(object(new atom("struct")), c));
      c=object(new cons(object(new atom("value")), c));
      f->add_form(c);
    }
    { // (structread struct i)
      object c;
      c=object(new cons(object(new integer(i)), c));
      c=object(new cons(object(new atom("struct")), c));
      c=object(new cons(object(new atom("value")), c));
      c=object(new cons(object(new atom("structwrite")), c));
      f->add_form(c);
    }
    n="SETF ";
    n+=name;
    n+='-';
    n+=s[i];
    set_function(n.c_str(), object(f));
  }
}

lisp::object
lisp::library::structread(library& lib, int argc, object *argv) {
  return as_structure(argv[0])[assume_integer(argv[1]).get_value()];
}

lisp::object
lisp::library::structwrite(library& lib, int argc, object *argv) {
  return as_structure(argv[1])
    [assume_integer(argv[2]).get_value()]=argv[0];
}

lisp::object
lisp::library::structbuild(library& lib, int argc, object *argv) {
  structdef& d=lib.structures[assume_atom(argv[0]).get_text()];
  object s(new structure(assume_atom(argv[0]).get_text(), &d));

  for(object x=argv[1];is_cons(x);x=assume_cons(x).cdr) {
    const char *text=as_atom(assume_cons(x).car).get_text();
    int idx;
    if(*(text++)!=':') throw error(nil, _("Syntax error"));
    for(idx=0;idx<(signed)d.size()&&d[idx]!=text;idx++);
    if(idx==(signed)d.size()) {
      string s(_("Undefined symbol: "));
      s+=text;
      throw error(nil, s.c_str());
    }
    x=assume_cons(x).cdr;
    assume_structure(s)[idx]=as_cons(x).car;
  }
  return s;
}

lisp::object
lisp::library::structp(library& lib, int argc, object *argv) {
  return (is_structure(argv[0])&&
	  !strcmp(assume_structure(argv[0]).get_type(),
		  assume_atom(argv[1]).get_text()))?t:nil;
}

bool
lisp::library::eql(object a, object b) {
  if(is_number(a)&&is_number(b))
    return !as_number(a).binary(number::EQ, b).eq(nil);
  if(is_atom(a)) return a.equal(b);
  return a.eq(b);
}

lisp::object
lisp::library::evaluate(domain& d, object l) {
  if(is_cons(l)) l=expand_macro(l);

  if(is_atom(l))
    return d.get(assume_atom(l).get_text());

  if(!is_cons(l)) return l;

  int argc=0;
  object x;
  for(x=l;is_cons(x);x=assume_cons(x).cdr) argc++;
  object argv[argc];
  argc=0;
  for(x=l;is_cons(x);x=assume_cons(x).cdr) argv[argc++]=assume_cons(x).car;
  if(is_cons(argv[0])) {
    object f=evaluate(d, argv[0]);
    return as_function(f).funcall(*this, argc-1, &argv[1]);
  }
  if(!(argc&&is_atom(argv[0]))) throw error(l, _("Function expected"));
  const char *fn=assume_atom(argv[0]).get_text();

  static struct {
    const char *name;
    object (*code)(library&, domain&, const char *, int, object *);
  } special_forms[]={
    {"BLOCK", form_block},
    {"DEFMACRO", form_defun},
    {"DEFSTRUCT", form_defstruct},
    {"DEFUN", form_defun},
    {"DEFVAR", form_defvar},
    {"FUNCTION", form_function},
    {"IF", form_if},
    {"LAMBDA", form_lambda},
    {"LET", form_let},
    {"LET*", form_let},
    {"LOOP", form_loop},
    {"PROGN", form_progn},
    {"QUOTE", form_quote},
    {"RETURN-FROM", form_return_from},
    {"SETF", form_setf},
    {"SETQ", form_setq},
    {0, 0}
  };

  try {
    for(int i=0;special_forms[i].name;i++)
      if(!strcmp(fn, special_forms[i].name))
	return special_forms[i].code(*this, d, fn, argc, argv);
    for(int i=1;i<argc;i++) argv[i]=evaluate(d, argv[i]);
    return ((function *)get_function(fn).get_data())->
      funcall(*this, argc-1, &argv[1]);
  }
  catch(error e) {
    if(e.get_cause().eq(nil)) e.set_cause(l);
    throw e;
  }
}

lisp::object
lisp::library::form_block(library& lib, domain& d,
			  const char *fn, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  try {
    for(int i=2;i<argc-1;i++) lib.evaluate(d, argv[i]);
    if(argc>2) return lib.evaluate(d, argv[argc-1]);
    return nil;
  }
  catch(return_from r) {
    if(r.get_block_name().equal(argv[1])) return r.get_value();
    throw r;
  }
}

lisp::object
lisp::library::form_defun(library& lib, domain& d,
			  const char *fn, int argc, object *argv) {
  if(argc<3) throw error(_("Insufficient arguments"));
  real_function *f=new real_function;
  for(int i=2;i<argc;i++) f->add_form(argv[i]);
  object o(f);
  if(fn[3]=='U') lib.set_function(as_atom(argv[1]).get_text(), o);
  else lib.set_macro(as_atom(argv[1]).get_text(), o);
  return argv[1];
}

lisp::object
lisp::library::form_defstruct(library& lib, domain& d,
			      const char *fn, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  structdef def;
  for(int i=2;i<argc;i++) def.push_back(as_atom(argv[i]).get_text());
  lib.defstruct(as_atom(argv[1]).get_text(), def);
  return argv[1];
}

lisp::object
lisp::library::form_defvar(library& lib, domain& d,
			   const char *fn, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>3) throw error(_("Excess arguments"));
  lib.defvar(as_atom(argv[1]).get_text());
  if(argc==3) lib.set_variable(assume_atom(argv[1]).get_text(),
			       lib.evaluate(d, argv[2]));
  else lib.get_variable_reference(assume_atom(argv[1]).get_text());
  return argv[1];
}

lisp::object
lisp::library::form_function(library& lib, domain& d,
			     const char *fn, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));
  if(is_atom(argv[1]))
    return lib.get_function(assume_atom(argv[1]).get_text());
  return lib.evaluate(d, argv[1]);
}

lisp::object
lisp::library::form_if(library& lib, domain& d,
		       const char *fn, int argc, object *argv) {
  if(argc<3) throw error(_("Insufficient arguments"));
  if(argc>4) throw error(_("Excess arguments"));
  if(!lib.evaluate(d, argv[1]).eq(nil))
    return lib.evaluate(d, argv[2]);
  if(argc<4) return nil;
  return lib.evaluate(d, argv[3]);
}

lisp::object
lisp::library::form_loop(library& lib, domain& d,
			 const char *fn, int argc, object *argv) {
  for(;;) {
    try {
      for(int i=1;i<argc;i++) lib.evaluate(d, argv[i]);
    }
    catch(return_from r) {
      if(r.get_block_name().eq(nil)) return r.get_value();
      throw r;
    }
  }
}

lisp::object
lisp::library::form_lambda(library& lib, domain& d,
			   const char *fn, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  real_function *f=new real_function;
  for(int i=1;i<argc;i++) f->add_form(argv[i]);
  return object(f);
}

lisp::object
lisp::library::form_let(library& lib, domain& d,
			const char *fn, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  domain local(d);
  for(object o=argv[1];is_cons(o);o=assume_cons(o).cdr) {
    object x=assume_cons(o).car;
    if(is_cons(x)) {
      object y=assume_cons(x).cdr;
      if(is_cons(y))
	local.lex(as_atom(assume_cons(x).car).get_text(),
		  lib.evaluate(fn[3]?local:d, assume_cons(y).car));
      else local.lex(as_atom(assume_cons(x).car).get_text());
    } else local.lex(as_atom(x).get_text());
  }
  for(int i=2;i<argc-1;i++) lib.evaluate(local, argv[i]);
  if(argc>2) return lib.evaluate(local, argv[argc-1]);
  return nil;
}

lisp::object
lisp::library::form_progn(library& lib, domain& d,
			  const char *fn, int argc, object *argv) {
  for(int i=1;i<argc-1;i++) lib.evaluate(d, argv[i]);
  if(argc>1) return lib.evaluate(d, argv[argc-1]);
  return nil;
}

lisp::object
lisp::library::form_quote(library& lib, domain& d,
			  const char *fn, int argc, object *argv) {
  if(argc<2) throw error(_("Insufficient arguments"));
  if(argc>2) throw error(_("Excess arguments"));
  return argv[1];
}

lisp::object
lisp::library::form_return_from(library& lib, domain& d,
				const char *fn, int argc, object *argv) {
  if(argc<3) throw error(_("Insufficient arguments"));
  if(argc>3) throw error(_("Excess arguments"));
  throw return_from(argv[1], lib.evaluate(d, argv[2]));
}

lisp::object
lisp::library::form_setq(library& lib, domain& d,
			 const char *fn, int argc, object *argv) {
  if(argc<3) throw error(_("Insufficient arguments"));
  if(argc>3) throw error(_("Excess arguments"));
  object x=lib.evaluate(d, argv[2]);
  d.set(as_atom(argv[1]).get_text(), x);
  return x;
}

lisp::object
lisp::library::form_setf(library& lib, domain& d,
			 const char *fn, int argc, object *argv) {
  if(argc<3) throw error(_("Insufficient arguments"));
  if(argc>3) throw error(_("Excess arguments"));
  if(is_atom(argv[1])) {
    object x=lib.evaluate(d, argv[2]);
    d.set(assume_atom(argv[1]).get_text(), x);
    return x;
  } else if(is_cons(argv[1])) {
    string s("SETF ");
    s+=as_atom(assume_cons(argv[1]).car).get_text();
    object n(new cons(argv[2], assume_cons(argv[1]).cdr));
    n=object(new cons(object(new atom(s.c_str())), n));
    return lib.evaluate(d, n);
  } else throw error(_("Type mismatch"));
}

void
lisp::library::load(const char *filename) {
  ifstream in(filename);

  if(in.bad()) throw error(nil, strerror(errno));
  try {
    domain d(*this);
    object o;
    in>>o;
    while(!(in.fail()||in.eof())) {
      evaluate(d, o);
      in>>o;
    }
  }
  catch(error e) {
    in.close();
    throw e;
  }
  in.close();
}
