// Grin LISP
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#define _GNU_SOURCE
#undef _POSIX_SOURCE

#include <fnmatch.h>
#include "grinlisp.h"

#ifndef FNM_CASEFOLD // Last resort
extern "C" {
  int fnmatch(const char *pattern, const char *string, int flags);
}
#define FNM_CASEFOLD 0x10
#endif

grinlisp::grinlisp(void) {
  set_variable("DEFAULT-MIME-TYPE",
	       lisp::object(new
			    lisp::str("application/octet-stream")));

  lisp::structdef def;
  def.push_back("MASK");
  def.push_back("HANDLER");
  defstruct("MIME-HANDLER", def);
  set_variable("MIME-HANDLERS", lisp::nil);

  lisp::object mime_types=lisp::object(new lisp::hash_table
				       (get_function("EQUAL")));

  static const char *default_types[]={
    "application/postscript", "ai", "eps", "ps", 0,
    "application/rtf", "rtf", 0,
    "application/x-tex", "tex", 0,
    "application/x-texinfo", "texinfo", "texi", 0,
    "application/x-troff", "t", "tr", "roff", 0,
    "audio/basic", "au", "snd", 0,
    "audio/x-aiff", "aif", "aiff", "aifc", 0,
    "audio/x-wav", "wav", 0,
    "image/gif", "gif", 0,
    "image/ief", "ief", 0,
    "image/jpeg", "jpeg", "jpg", "jpe", 0,
    "image/tiff", "tiff", "tif", 0,
    "image/x-xwindowdump", "xwd", 0,
    "text/html", "html", 0,
    "text/plain", "txt", "c", "cc", "h", "", "diff", "patch", 0,
    "video/mpeg", "mpeg", "mpg", "mpe", 0,
    "video/quicktime", "qt", "mov", 0,
    "video/x-msvideo", "avi", 0,
    "video/x-sgi-movie", "movie", 0,
    0
  };

  for(const char **t=default_types;*t;t++) {
    lisp::object to(new lisp::str(*t));
    const char **e=t;
    for(e++;*e;e++)
      *assume_hash_table(mime_types).gethash
	(*this, lisp::object(new lisp::str(*e)))=to;
    t=e;
  }

  set_variable("MIME-TYPES", mime_types);
}

const char *
grinlisp::filename_to_mimetype(const char *filename) {
  const char *ext=filename+strlen(filename);
  
  while(ext>filename&&*ext!='/'&&*ext!='.') ext--;
  if(*ext=='.') ext++;
  else ext="";

  lisp::object result(get_variable("MIME-TYPES"));
  result=lisp::as_hash_table(result)
    .readhash(*this, lisp::object(new lisp::str(ext)));
  if(result.eq(lisp::nil)) result=get_variable("DEFAULT-MIME-TYPE");
  return lisp::as_str(result).get_text();
}

bool
grinlisp::handle_attachment(const char *filename, const char *type,
			    const char *name) {
  lisp::object handlers=get_variable("MIME-HANDLERS");
  while(lisp::is_cons(handlers)) {
    if(!fnmatch(lisp::as_str
		(lisp::as_structure
		 (lisp::assume_cons
		  (handlers).car)[0]).get_text(), type, FNM_CASEFOLD)) {
      lisp::object arg[]={lisp::object(new lisp::str(filename)),
			  lisp::object(new lisp::str(type)),
			  lisp::object(new lisp::str(name))};
      lisp::as_function(lisp::assume_structure
			(lisp::assume_cons
			 (handlers).car)[1]).funcall(*this, 3, arg);
      return true;
    }
    handlers=assume_cons(handlers).cdr;
  }
  return false;
}
