// TTY-Grin
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <fstream>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "tempfile.h"
#include "localmailbox.h"

message_descriptor&
localmailbox::operator[](int which) {
  return messages[which];
}

int
localmailbox::size(void) const {
  return messages.size();
}

bool
localmailbox_message::extract(ostream& out) const {
  return extract_real(out, false);
}

bool
localmailbox_message::extract_real(ostream& out, bool include_flags) const {
  ifstream in(parent->source_filename.c_str());
  char buf[1024];
  unsigned char flags=get_flags();

  if(in.fail()) return false;
  in.seekg(begin);
  in.getline(buf, sizeof(buf));
  if(!include_flags) in.getline(buf, sizeof(buf));
  while(in.tellg()<=end&&!in.fail()&&buf[0]) {
    if(strncasecmp(buf, "status: ", 8)) out<<buf<<endl;
    in.getline(buf, sizeof(buf));
  }
  if(flags&&include_flags) {
    int i;

    out<<"Status: ";
    for(i=0;localmailbox::status_flags[i];i+=2)
      if(flags&localmailbox::status_flags[i])
	out<<localmailbox::status_flags[i+1];
    out<<endl;
  }
  while(in.tellg()<=end&&!in.fail()) {
    out<<buf<<endl;
    in.getline(buf, sizeof(buf));
  }
  in.close();
  return true;
}

void
localmailbox_message::set_flags(unsigned char x) {
  if(x!=get_flags()) {
    parent->commit=true;
    message_descriptor::set_flags(x);
  }
}

unsigned char localmailbox::status_flags[]={
  message_descriptor::READ,     'R',
  message_descriptor::ANSWERED, 'A',
  message_descriptor::MARKED,   'F',
  message_descriptor::DELETED,  'D',
  0
};

localmailbox::localmailbox(const char *filename) : source_filename(filename) {
  ifstream in(filename);
  char buf[1024];

  in.getline(buf, sizeof(buf));
  while(!in.fail()&&!is_from_line(buf)) in.getline(buf, sizeof(buf));
  while(!in.fail()) {
    localmailbox_message msg(this);

    if(messages.empty()) msg.begin=0;
    else msg.begin=messages.back().end;
    in.getline(buf, sizeof(buf));
    while(!in.fail()&&buf[0]) {
      if(!strncasecmp(buf, "from: ", 6)) msg.set_from(buf+6);
      else if(!strncasecmp(buf, "subject: ", 9)) msg.set_subject(buf+9);
      else if(!strncasecmp(buf, "date: ", 6)) msg.set_date(buf+6);
      else if(!strncasecmp(buf, "status: ", 8)) {
	const char *s=buf+8;
	unsigned char flags=0;
	while(*s) {
	  int i;
	  for(i=0;status_flags[i];i++)
	    if(status_flags[i+1]==*s) flags|=status_flags[i];
	  s++;
	}
	msg.set_flags(flags);
      }
      in.getline(buf, sizeof(buf));
    }
    while(!in.fail()&&!is_from_line(buf)) {
      msg.end=in.tellg();
      in.getline(buf, sizeof(buf));
    }
    messages.push_back(msg);
  }
  in.close();
  commit=false;
}

mailbox::~mailbox(void) {
}

localmailbox::~localmailbox(void) {
  if(commit) {
    tempfile t;
    ofstream out(t.filename());
    ifstream in(source_filename.c_str());
    int i;

    if(out.fail()) return;
    for(i=0;i<size();i++) {
      if(i) in.seekg(messages[i-1].end);
      int c;
      while(in.tellg()<messages[i].begin&&
	    (c=in.get())>=0) out<<(char)c;
      if(!(messages[i].get_flags()&message_descriptor::EXPUNGE))
	messages[i].extract_real(out, true);
    }
    if(i) in.seekg(messages[i-1].end);
    {
      int c;
      while((c=in.get())>=0) out<<(char)c;
    }
    out.close();
    if(rename(t.filename(), source_filename.c_str())) {
      ifstream from(t.filename());
      ofstream to(source_filename.c_str());
      char c;

      if(from.fail()||to.fail()) return;
      from.get(c);
      while(!from.fail()) {
	to.put(c);
	from.get(c);
      }
      from.close();
      to.close();
    }
  }
}

bool
localmailbox::append(const message_descriptor& from) {
  ofstream out(source_filename.c_str(), ios::app);
  localmailbox_message addme(this);

  if(out.fail()) return false;
  {
    struct stat s;
    stat(source_filename.c_str(), &s);
    addme.begin=s.st_size;
  }
  time_t now=time(NULL);
  out<<"From "<<getenv("USER")<<' '<<ctime(&now);
  if(from.extract(out)) {
    addme.end=out.tellp();
    out.close();
    addme.set_date(from.get_date());
    addme.set_from(from.get_from());
    addme.set_subject(from.get_subject());
    addme.set_flags(from.get_flags());
    messages.push_back(addme);
    return true;
  } else {
    out.close();
    return false;
  }
}

bool
localmailbox::append(const message& from, unsigned char flags) {
  ofstream out(source_filename.c_str(), ios::app);
  ifstream in(from.get_filename());
  localmailbox_message addme(this);

  if(out.fail()||in.fail()) {
    out.close();
    in.close();
    return false;
  }
  {
    struct stat s;
    stat(source_filename.c_str(), &s);
    addme.begin=s.st_size;
  }
  time_t now=time(NULL);
  out<<"From "<<getenv("USER")<<' '<<ctime(&now);
  {
    char c;
    in.get(c);
    while(!in.fail()) {
      out.put(c);
      in.get(c);
    }
  }
  addme.end=out.tellp();
  out.close();
  in.close();
  int s=from.size()-1;
  addme.set_date(from[s]["date"]);
  addme.set_from(from[s]["from"]);
  addme.set_subject(from[s]["subject"]);
  addme.set_flags(flags);
  commit=true;
  messages.push_back(addme);
  return true;
}

bool
localmailbox::is_from_line(const char *line) const {
  const char *t;

  if(strncmp(line, "From ", 5)) return false;
  if((t=strchr(line+5, ' '))==NULL) return false;
  while(*t==' ') t++;
  if((t=strchr(t, ' '))==NULL) return false;
  while(*t==' ') t++;
  if((t=strchr(t, ' '))==NULL) return false;
  while(*t==' ') t++;
  if(!atoi(t)) return false;
  if((t=strchr(t, ' '))==NULL) return false;
  while(*t==' ') t++;
  if((t=strchr(t, ' '))==NULL) return false;
  while(*t==' ') t++;
  if(!atoi(t)) return false;
  return true;
}

void
localmailbox::expunge(void) {
  vector<localmailbox_message>::iterator i;

  for(i=messages.begin();i!=messages.end();i++)
    if((*i).get_flags()&message_descriptor::DELETED)
      (*i).set_flags(message_descriptor::EXPUNGE);
}
