/*
 *	Copyright (c) 1999 RISS-Telecom Networking Center
 *
 *	Copyright (c) 1993 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	BPFT $Id: main.c,v 1.4 1994/01/14 11:46:41 bob Exp $
 *
 *	$Log: main.c,v $
 * Revision 1.4  1994/01/14  11:46:41  bob
 * Any file extention maybe used, extra needed for FreeBSD, thanx vv@sonet.kemerovo.su
 *
 * Revision 1.3  1994/01/06  14:59:17  bob
 * Cosmetic changes
 *
 * Revision 1.2  1993/11/01  14:57:07  bob
 * Improve signal handlers, added SIGALRM and SIGCHLD
 *
 * Revision 1.1  1993/10/20  16:04:41  bob
 * Initial revision
 *
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 * Redistribution in binary form may occur without any restrictions.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

/*	main.c - tcp/udp data traffic collector daemon	*/

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#include <paths.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <net/bpf.h>
#include <pcap.h>

#include "../include/interface.h"
#include "../include/pathnames.h"
#include "trafd.h"

/*
 * Command line switches.
 * Left - default value, right - action, if switch has been used.
 */
int dflag;	/* dump to compiled packet-matching code to stdout and stop */
int Oflag = 1;	/* don't run the packet-matching code optimizer if have bug */
int pflag;	/* don't put the interface into promiscuous mode */
int rflag;	/* attempt to resume data from safe file if exist */
int nflag;
int Nflag;
int fflag;

/* Global interrupts flags */
int flag_hup;	/* SIGHUP - drop collected data to tempfile */
int flag_int;	/* SIGINT - append collected data to savefile */
int flag_usr1;

/* Global variables */
char *program_name;	/* myself */
char *device_name = 0;	/* interface name */
char file_pid[40];	/* pid file */
char file_out[40];	/* temp file */
char file_backup[40];	/* save file */
char file_fifo[40];	/* fifo file */

/* Block of the interrupt drivers */
static void cleanup();
static void onhup();
static void onint();
static void onusr1();
static void onusr2();
static void onalarm();
static void onchld();

/* Length of saved portion of packet */
int snaplen = DEFAULT_SNAPLEN;

static pcap_t *pd;

void
main(argc, argv)
	int argc;
	char **argv;
{
	struct bpf_program *parse();
	void bpf_dump(), usage();

	register int op, cnt = -1, i;
	struct bpf_program fcode;
	register char *cp, *infile = 0, *cmdbuf, *DevFileName = 0;
	FILE *fd;
	pcap_handler printer;
	u_char *pcap_userdata;
	char ebuf[PCAP_ERRBUF_SIZE];
	extern char *optarg;
	extern int optind, opterr;
	bpf_u_int32 localnet, netmask;
   
	if ((cp = strrchr(argv[0], '/')) != NULL)
			program_name = cp + 1;
	else
			program_name = argv[0];

	opterr = 0;
	while ((op = getopt(argc, argv, "c:df:F:i:Opr")) != EOF)
		switch (op) {
		case 'c':
			cnt = atoi(optarg);
			break;
		case 'd':
			++dflag;
			break;
		case 'f':
			DevFileName = optarg;
			break;
		case 'F':
			infile = optarg;
			break;
		case 'i':
			device_name = optarg;
			break;
		case 'O':
			Oflag = 0;
			break;
		case 'p':
			++pflag;
			break;
		case 'r':
			++rflag;
			break;
		default:
			usage();
		}

	if (access(_PATH_VARRUN, R_OK|W_OK|X_OK) < 0) {
		sprintf(file_pid, "%s: access to %s", program_name,
			_PATH_VARRUN);
		perror(file_pid);
		exit(1);
	}
	if (access(PATH_TOSAVE, R_OK|W_OK|X_OK) < 0) {
		sprintf(file_backup, "%s: access to %s", program_name,
			PATH_TOSAVE);
		perror(file_backup);
		exit(1);
	}
	if (device_name == 0)
		if ((device_name = getenv("IFF_LISTEN")) == NULL)
			if ((device_name = pcap_lookupdev(ebuf)) == 0)
				error("%s", ebuf);
	if (DevFileName == 0)
		DevFileName = device_name;

	/* Attach bpf interface to network interface */
	pd = pcap_open_live(device_name, snaplen, !pflag, 1000, ebuf);
	if (pd == NULL)
			error("%s", ebuf);

	i = pcap_snapshot(pd);
	if (snaplen < i) {
		warning("snaplen raised from %d to %d", snaplen, i);
		snaplen = i;
	}
	
	if (pcap_lookupnet(device_name, &localnet, &netmask, ebuf) < 0) {
		localnet = 0;
		netmask = 0;
		warning("%s", ebuf);
	}
							
	if (infile) 
		cmdbuf = read_infile(infile);
	else
		cmdbuf = copy_argv(&argv[optind]);

	if(pcap_compile(pd, &fcode, cmdbuf, Oflag, netmask) < 0)
			error("%s", pcap_geterr(pd));
	
	if (dflag) {
		bpf_dump(&fcode, dflag);
		exit(0);
	}
	init_addrtoname(localnet, netmask);

	/* Catch signals */
	flag_int = flag_hup = flag_usr1 = 0;
	(void)signal(SIGTERM, cleanup);
	(void)signal(SIGPIPE, SIG_IGN);
	(void)signal(SIGINT,  onint);
	(void)signal(SIGHUP,  onhup);
	(void)signal(SIGUSR1, onusr1);
	(void)signal(SIGUSR2, onusr2);
	(void)signal(SIGALRM, onalarm);
	(void)signal(SIGCHLD, onchld);

	/* Initialize file names */
	sprintf(file_pid, "%s%s.%s",  _PATH_VARRUN, program_name, device_name);
	sprintf(file_out, "%s%s.%s",  _PATH_VARTMP, program_name, DevFileName);
	sprintf(file_backup,"%s%s.%s", PATH_TOSAVE, program_name, DevFileName);
	sprintf(file_fifo, "%s%s.%s", _PATH_TMP,    program_name, device_name);

	if (access(file_pid, F_OK) == 0)
		error("unexpected pidfile, probable interface '%s' already listen",
		      device_name);

	/* Jump to background */
	daemon(1, 0);
	if ((fd = fopen(file_pid, "w")) == NULL)
		exit(1);

	fprintf(fd,"%d\n", getpid());
	fclose(fd);

	openlog(program_name, LOG_PID|LOG_CONS, LOG_DAEMON);
	syslog(LOG_NOTICE, "(%s) traffic collector started", device_name);

	if (pcap_setfilter(pd, &fcode) < 0)
		error("%s", pcap_geterr(pd));
	
	printer = lookup_printer(pcap_datalink(pd));
	pcap_userdata = 0;
	/* Start read from bpf */
	if (!traf_init(rflag)) {
		if (pcap_loop(pd, cnt, printer, pcap_userdata))
			/* Clear exit */
			cleanup();
	} else
		errno = ENOMEM;

	pcap_close(pd);
	/* Abnormal termination */
	(void)syslog(LOG_ERR, "(%s) traffic collector aborted: %m",
		     device_name);
	exit(1);
}

/* make a clean exit on interrupts */
static void
cleanup()
{
	
	struct pcap_stat stat;
	
	if (!pcap_stats(pd, &stat))
		(void)syslog(LOG_INFO, "(%s) packets: received %d, droped %d",
			     device_name, stat.ps_recv, stat.ps_drop);
	pcap_close(pd);
	traf_save(file_out, "w");
	(void)syslog(LOG_NOTICE, "(%s) traffic collector stoped", device_name);
	unlink(file_pid);
	exit(0);
}

#define	WAIT_ACTION	5

static void
onhup()
{
	alarm(WAIT_ACTION);
	flag_hup++;
}

static void
onint()
{
	alarm(WAIT_ACTION);
	flag_int++;
}

static void
onusr1()
{
	flag_usr1++;
}

static void
onusr2()
{
	flag_usr1 = 0;
	traf_pipe();
}

static void
onalarm()
{
	alarm(0);
	if (flag_hup) {
		flag_hup = 0;
		traf_save(file_out, "w");
	}
	if (flag_int) {
		flag_int = 0;
		traf_save(file_backup, "a");
		traf_clear();
	}
}

static void
onchld()
{
	if (wait(0) == -1)
		(void)syslog(LOG_WARNING, "(%s) wait: %m", device_name);
}

void
usage()
{
	extern char version[];

	fprintf(stderr, "trafd v%s - tcp/udp data traffic collector daemon\n",
		version);
	fprintf(stderr,
"Usage: %s [-dOpr] [-c count] [-i iface] [-f ext] [-F file | expr]\n", program_name);
	exit(-1);
}
