/****************************************************************************
 * Copyright (C) 1998 WIDE Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by WIDE Project and
 *    and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 ****************************************************************************/

/****************************************************************************
 * Copyright (C) 1999 University of Tromso.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by the University of Tromso
 *    and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 ****************************************************************************/

/*
 * <$Id: ne_mesg.c,v 3.15 2001/03/13 11:14:43 dillema Exp $>
 */


#include "tot.h"

COPYRIGHT1(
"@(#) Copyright (C) 1998 WIDE Project.  All rights reserved.\n");

COPYRIGHT2(
"@(#) Copyright (C) 1999 The University of Tromso.  All rights reserved.\n");

COPYRIGHT3("Written by: F. W. Dillema. feico@pasta.cs.uit.no.\n
based on code by: Yusuke DOI, Keio Univ. Murai Lab.");

CVSID("$Id: ne_mesg.c,v 3.15 2001/03/13 11:14:43 dillema Exp $");


u_int16_t mesg_id (void) {
	static u_int16_t id = 0;

	if (!id) {
		srandom (time (NULL));
		id = random ();
	}
	id++;

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "mesg_id() = %d", id));
	}
	return id;
}

int mesg_make_query (u_char *qname, u_int16_t qtype, u_int16_t qclass, u_int32_t id,
		         int rd, u_char *buf, int buflen) {
	DX (char *fn = "mesg_make_query()";)
	u_char *ucp;
	int i, written_len;
	Mesg_Hdr *hdr;

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: (qtype: %s, id: %d): start", fn, string_rtype (qtype), id));
	}

	hdr = (Mesg_Hdr *) buf;

	/* write header */
	hdr->id = id;
	hdr->qr = 0;
	hdr->opcode = OP_QUERY;
	hdr->aa = 0;
	hdr->tc = 0;
	hdr->rd = rd;
	hdr->ra = 0;
	hdr->zero = 0;
	hdr->rcode = RC_OK;
	hdr->qdcnt = ntohs (1);
	hdr->ancnt = ntohs (0);
	hdr->nscnt = ntohs (0);
	hdr->arcnt = ntohs (0);
	written_len = sizeof (Mesg_Hdr);
	ucp = (u_char *) (hdr + 1);

	/* write qname */
	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: qname offset = %d\n", fn, ucp - buf));
	}

	i = mesg_dname_copy (qname, ucp, buflen - written_len);
	if (i < 0)
		return -1;

	written_len += i;
	ucp += i;

	/* write qtype / qclass */
	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: qtype/qclass offset = %d\n", fn, ucp - buf));
	}
	written_len += sizeof (u_int16_t) * 2;
	if (written_len > buflen)
		return -1;

	PUTSHORT (qtype, ucp);
	PUTSHORT (qclass, ucp);

	return written_len;
}


int mesg_dname_copy (u_char *from, u_char *to, int tolen) {
	int skip, written_len;

	written_len = 0;
	while (*from) {
		skip = mesg_labellen(from) + 1;
		written_len += skip;
		if (written_len >= tolen)
			return -1;

		memcpy (to, from, skip);
		from += skip;
		to += skip;
	}
	*to = '\0';
	written_len++;

	return written_len;
}

u_char *mesg_dname_redirect (u_char * label, u_char * mesg_head) {
	u_int16_t us_tmp;

	if (mesg_head && (*label & DNCMP_MASK) == DNCMP_MASK) {
		GETSHORT (us_tmp, label);
		us_tmp = us_tmp & (~DNCMP_MASK_INT16T);
		label = mesg_head + us_tmp;
	}
	return label;
}

int mesg_write_dname (u_char * buf_head, u_char * buf_tail,
		          u_int16_t * offset, int offset_len,
		          u_char * dname, u_char * wp) {
	DX (char *fn = "mesg_write_dname()";)
	int written_len, match_len;
	u_char *rp_dname, *cp_dname;
	u_char *rp_mesg, *cp_mesg;
	int i, bestmatch_len = 0;
	u_char *bestmatch_dnamep;
	u_char *bestmatch_mesgp;
	u_int16_t us_tmp;

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: start\n", fn));
	}

	/** check if some (part of) dname has already appeared in message */
	rp_dname = dname;
	while(*rp_dname != '\0') {
		int dlen = mesg_labellen(rp_dname);
 
		for (i = 0; offset[i] != 0 && i < offset_len; i++) {
			rp_mesg = buf_head + offset[i];
			while(*rp_mesg != '\0') {
				int rplen;

				cp_dname = rp_dname;
				/* check redirection */
				cp_mesg = rp_mesg = mesg_dname_redirect (rp_mesg, buf_head);
				rplen = mesg_labellen(rp_mesg);

				if (cp_mesg < buf_head || buf_tail < cp_mesg) {
					if (T.debug > 3) {
						DX (syslog (LOG_DEBUG, "out of bounds(1): buf_head = 0x%p buf_tail = 0x%p cp_mesg = 0x%p\n", 
						            (void *) buf_head, (void *) buf_tail, (void *) cp_mesg));
					}
					return -1;
				}

				/* now cp_dname is not compressed. */
				match_len = 0;
				/* start compare */
				while (*cp_mesg != '\0' &&
				       *cp_mesg == *cp_dname) {
					int mlen = mesg_labellen(cp_mesg);
					int dlen0 = mesg_labellen(cp_dname);

					if (dlen0 != mlen)
						break;
					if (*cp_mesg == EDNS0_ELT_BITLABEL) {
						if (memcmp(cp_mesg + 1,
							   cp_dname + 1,
							   mlen) != 0)
							break;
					} else if (strncasecmp (cp_mesg + 1,
								cp_dname + 1,
								*cp_mesg) != 0) {
						break;
					}
					
					/* a label match */
					cp_mesg += mlen + 1;
					cp_dname += mlen + 1;
					/* check redirection */
					cp_mesg = mesg_dname_redirect (cp_mesg, buf_head);
					if (cp_mesg < buf_head || buf_tail < cp_mesg) {
						/* out of bounds */
						if (T.debug > 3) {
							DX (syslog (LOG_DEBUG, "out of bounds(2): buf_head = 0x%p buf_tail = 0x%p cp_mesg = 0x%p\n", 
								    (void *) buf_head, (void *) buf_tail, (void *) cp_mesg));
						}
						return -1;
					}
					match_len++;
				}
				if (*cp_mesg == '\0' && *cp_dname == '\0') {
					/* complete match  */
					if (match_len > bestmatch_len) {
						bestmatch_dnamep = rp_dname;
						bestmatch_mesgp = rp_mesg;
						bestmatch_len = match_len;
					}
				}

				rp_mesg += (rplen + 1);
			}
		}

		if (bestmatch_len != 0) {
			/* don't check *shorter* value */
			break;
		}

		rp_dname += (dlen + 1);
	}

	/** register this name if not complete match */
	for (i = 0; offset[i] != 0; i++) {
		;		/* just skip */
	}

	if (bestmatch_dnamep != dname && i < offset_len - 1
	    && ((u_int16_t) (bestmatch_mesgp - buf_head) < DNCMP_REDIRECT_LIMIT)) {
		offset[i] = (u_int16_t) (wp - buf_head);
		offset[i + 1] = 0;
		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "%s: study offset[%d] = %d\n", fn, i, offset[i]));
		}
	}
	/** write dname */
	written_len = 0;
	cp_dname = dname;
	while (*cp_dname && cp_dname != bestmatch_dnamep) {
		i = mesg_labellen(cp_dname) + 1;
		if (wp + i > buf_tail) {
			/* overflow! */
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s: overflow!\n", fn));
			}
			return -1;
		}
		memcpy (wp, cp_dname, i);
		written_len += i;
		cp_dname += i;
		wp += i;
	}
	if (cp_dname == bestmatch_dnamep) {
		/* write redirection pointer */
		if (wp + sizeof (u_int16_t) > buf_tail) {
			/* overflow! */
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s: overflow!\n", fn));
			}
			return -1;
		}
		us_tmp = (u_int16_t) (bestmatch_mesgp - buf_head) | DNCMP_MASK_INT16T;
		PUTSHORT (us_tmp, wp);
		written_len += sizeof (u_int16_t);
	} else {
		/* null terminator */
		*wp = '\0';
		written_len++;
	}

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: return (written_len = %d)\n", fn, written_len));
	}
	return written_len;
}

int
mesg_labellen(const u_char *cp)
{
	register u_int i;

	i = *cp;
	if ((i & DNCMP_MASK) == EDNS0_MASK) {
		int bitlen;

		if (i != EDNS0_ELT_BITLABEL)
			return(-1);
		if ((bitlen = *(cp + 1)) == 0)
			bitlen = 256;
		return(((bitlen + 7) / 8) + 1);
	} else
		return(i);
}

u_char *mesg_skip_dname (u_char * dname, u_char *end) {
	int l;

	if (dname >= end)
		return(NULL);

	while(*dname) {
		if ((*dname & DNCMP_MASK) == DNCMP_MASK) {
			dname += 2;	/* redirection */
			return dname;
		}
		if (dname + 2 > end) /* we need at least 2 bytes */
			return(NULL);

		if ((l = mesg_labellen(dname)) < 0)
			return(NULL);
		dname += l + 1;

		if (dname >= end)
			return(NULL);
	}
	dname++;		/* push away null terminator */
	return dname;
}

int mesg_dname_cmp (u_char * mesg_head, u_char * dname_mesg, u_char * dname) {
	dname_mesg = mesg_dname_redirect (dname_mesg, mesg_head);

	/* compare label by label */
	while (*dname_mesg != '\0' && (*dname == *dname_mesg)) {
		int len0 = mesg_labellen(dname_mesg);
		int len1 = mesg_labellen(dname);

		if (len0 != len1)
			return(-1);

		if (*dname == EDNS0_ELT_BITLABEL) {
			if (memcmp(dname_mesg + 1, dname + 1,  len0) != 0)
				return -1;
		}
		else {
			if (strncasecmp (dname_mesg + 1, dname + 1, len0) != 0)
				return -1;
		}

		dname += len0 + 1; /* note that len0 == len1 */
		dname_mesg += len0 + 1;	/* <- plus one is for length byte
					 * (see RFC1035) */
		dname_mesg = mesg_dname_redirect (dname_mesg, mesg_head);
	}

	if (*dname != *dname_mesg)
		return -1;
	else
		return 0;
}

int mesg_write_rrset_list (G_List * rrls, u_char * buf_head,
			       u_char * buf_tail, u_int16_t * offset,
			       int offset_len, u_char ** wp, u_int16_t * cnt) {
	DX (char *fn = "mesg_write_rrset_list()";)
	u_char *wp_start, *wp_period;
	RRset *rrsp;
	RR *rrp;
	int i, ret;
	Mesg_Hdr *hdr;
	u_int16_t us_tmp;
	u_int32_t ul_tmp;

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: start.\n", fn));
	}

	hdr = (Mesg_Hdr *) buf_head;
	wp_start = wp_period = *wp;

	for (rrls = rrls->next; rrls->list_data; rrls = rrls->next) {
		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "%s: write a record.\n", fn));
		}
		for (i = 0; i < ((RRset *) (rrls->list_data))->data.d->data_cnt; i++) {
			rrsp = (RRset *) rrls->list_data;
			wp_period = *wp;

			/* write the owner name -- with compression */
			ret = mesg_write_dname (buf_head, buf_tail, offset, offset_len,
					  rrset_key_info_owner (rrsp), *wp);
			if (ret < 0) {
				if (T.debug > 3) {
					DX (syslog (LOG_DEBUG, "%s: fail to write owner name.\n", fn));
				}
				*wp = wp_period;
				return wp_period - wp_start;
			}
			*wp += ret;

			/* write RR field and data */
			rrp = (RR *) (rrsp->data.p + data_offset (i, rrsp->data.p));
			if (*wp + sizeof (u_int16_t) * 3 + sizeof (u_int32_t) + rrp->rd_len > buf_tail) {
				if (T.debug > 3) {
					DX (syslog (LOG_DEBUG, "%s: can't write Rdata; drop owners etc\n", fn));
					DX (syslog (LOG_DEBUG, "%s: return %d\n", fn, wp_period - wp_start));
				}
				*wp = wp_period;
				return wp_period - wp_start;
			}
			PUTSHORT (rrsp->key.info->r_type, *wp);
			PUTSHORT (rrsp->key.info->r_class, *wp);
			ul_tmp = rrp->ttl;
			PUTLONG (ul_tmp, *wp);
			PUTSHORT (rrp->rd_len, *wp);
			memcpy (*wp, rr_rdata (rrp), rrp->rd_len);
			/*
			 * XXX #warning "mesg_write_rrset_list(): RDATA
			 * COMPRESSION NOT YET"
			 */
			*wp += rrp->rd_len;
			/* update header */
			us_tmp = ntohs (*cnt) + 1;
			*cnt = htons (us_tmp);

			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s: now counter = %d", fn, us_tmp));
			}
		}
	}

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: return %d\n", fn, *wp - wp_start));
	}
	return (*wp - wp_start);
}

int mesg_assemble (Mesg_Hdr * hdr, G_List * an_list, G_List * ns_list,
		       G_List * ar_list, u_char * buf, u_int16_t buflen,
		       int mesg_len) {

	int written_len = 0;
	int ret;
#define MESG_ASSEMBLE_OFFSET_LEN 64
	u_int16_t offset[MESG_ASSEMBLE_OFFSET_LEN];
	u_char *ucp;
	u_char *ucp_tmp;

	/* check if header is preseant */
	if (hdr)
		memcpy (buf, (u_char *) hdr, mesg_len);

	/* now, header is written in the buffer -- refer it */
	hdr = (Mesg_Hdr *) buf;

	/* check and reset header */
	hdr->qr = 1;
	hdr->ancnt = 0;
	hdr->nscnt = 0;
	hdr->arcnt = 0;

	if (hdr->qdcnt) {
		int qdcnt = ntohs(hdr->qdcnt);

		/* register question name to compression name list */
		/*
		 * XXX -- ignore multiple question :p -- I think minor
		 * problem
		 */
		offset[0] = (u_int16_t) (sizeof (Mesg_Hdr));
		offset[1] = 0;	/* <- terminator */

		/*
		 * already the question written in buffer. Skip the question
		 * section.
		 */
		ucp = buf + sizeof(Mesg_Hdr);
		while(qdcnt--) {
			/* skip QNAME */
			ucp = mesg_skip_dname(ucp, buf + mesg_len);
			if (ucp == NULL ||
			    ucp + 2 * sizeof(u_int16_t) > buf + mesg_len) {
				syslog (LOG_NOTICE,
					"original query message overrun\n");
				return(-1);
			}
			/* skip QTYPE and QCLASS */
			ucp += (2 * sizeof(u_int16_t));
		}
		written_len = ucp - buf;
	} else {
		/* there mustnot be any question */
		written_len = sizeof (Mesg_Hdr);
		ucp = buf + written_len;
		offset[0] = 0;
	}

	/* write answers */
	ucp_tmp = ucp;
	ret = mesg_write_rrset_list (an_list, buf, buf + buflen, offset,
				     MESG_ASSEMBLE_OFFSET_LEN, &ucp,
				     &(hdr->ancnt));
	if (ret < 0) {
		/* truncated message */
		hdr->tc = 1;
		return ucp_tmp - buf;
	}
	written_len += ret;

	/* write ns */
	ucp_tmp = ucp;
	ret = mesg_write_rrset_list (ns_list, buf, buf + buflen, offset,
				     MESG_ASSEMBLE_OFFSET_LEN, &ucp,
				     &(hdr->nscnt));
	if (ret < 0) {
		/* truncated message */
		hdr->tc = 1;
		return ucp_tmp - buf;
	}
	written_len += ret;

	/* write additonal records */
	ucp_tmp = ucp;
	ret = mesg_write_rrset_list (ar_list, buf, buf + buflen, offset,
				     MESG_ASSEMBLE_OFFSET_LEN, &ucp,
				     &(hdr->arcnt));
	if (ret < 0) {
		/* ignore this error -- no matter for additional records */
		return ucp_tmp - buf;
	}
	written_len += ret;

	return written_len;
}

void rrset_couple_free (RRset_Couple * rc) {

	if (rc->rrs)
		rrset_free (rc->rrs);
	if (rc->rrl)
		rr_list_free (rc->rrl);

	free (rc);
}

void rrset_couple_freev (void *rcv) {
	rrset_couple_free ((RRset_Couple *) rcv);
}

/* XXX: unused? */
int mesg_rdata_cmp (u_int16_t r_type, u_int16_t rd_len, u_char * mesg_head,
		        u_char * rd_mesg, u_char * rd) {
	char *fn = "mesg_rdata_cmp()";
	u_char *rp, *rp_mesg;

	rp_mesg = rd_mesg;
	rp = rd;

	switch (r_type) {
	case RT_NS:
	case RT_CNAME:
	case RT_PTR:
		/* one domain name */
		return mesg_dname_cmp (mesg_head, rd_mesg, rd);
	case RT_SOA:
		/* two domain names and five numbers */
		if (mesg_dname_cmp (mesg_head, rp_mesg, rp))
			return -1;

		SKIP_DNAME (rp_mesg);
		SKIP_DNAME (rp);
		if (mesg_dname_cmp (mesg_head, rp_mesg, rp))
			return -1;

		SKIP_DNAME (rp_mesg);
		SKIP_DNAME (rp);

		if (memcmp (rp_mesg, rp, sizeof (u_int32_t) * 5))
			return -1;
		return 0;
	case RT_MX:
		if (*((u_int16_t *) rd_mesg) != *((u_int16_t *) rd))
			return -1;

		rd_mesg += sizeof (u_int16_t);
		rd += sizeof (u_int16_t);

		return mesg_dname_cmp (mesg_head, rd_mesg, rd);
	case RT_RP:
		if (mesg_dname_cmp (mesg_head, rp_mesg, rp))
			return -1;

		SKIP_DNAME (rp_mesg);
		SKIP_DNAME (rp);
		if (mesg_dname_cmp (mesg_head, rp_mesg, rp))
			return -1;

		return 0;
	default:
		syslog (LOG_INFO, "%s: unknow resource type. treat as binary data.\n", fn);
		/* fall through */
	case RT_A:
	case RT_HINFO:
	case RT_AAAA:
	case RT_SRV:
	case RT_TXT:
		if (memcmp (rd_mesg, rd, rd_len))
			return -1;
		else
			return 0;
	}

	syslog (LOG_ERR, "never come here! %s, line %d.\n", __FILE__, __LINE__);
	abort ();
	exit (-1);
}

int mesg_extract_rr (u_char * begin, u_char * end, u_int16_t r_type,
		         u_int16_t r_class, u_char * rrp, u_char * buf,
		         int buflen) {
	u_char *rp, *wp;
	int i, written_len = 0;

	switch (r_type) {
	case RT_NS:
	case RT_CNAME:
	case RT_PTR:
		/* just extract domain name */
		if (!conv_dname_decompress (buf, buflen, rrp, begin, end, &written_len)) {
			syslog (LOG_INFO, "resource record invalid -- %s.", string_rtype (r_type));
			return -1;
		}
		break;
	case RT_SOA:
		rp = rrp;
		wp = buf;
		rp = conv_dname_decompress (wp, buflen, rp, begin, end, &i);
		if (!rp) {
			syslog (LOG_INFO, "resource record invalid -- SOA MNAME.");
			return -1;
		}
		wp += i;
		rp = conv_dname_decompress (wp, buflen - (wp - buf), rp, begin, end, &i);
		if (!rp) {
			syslog (LOG_INFO, "resource record invalid -- SOA RNAME.");
			return -1;
		}
		wp += i;
		memcpy (wp, rp, (i = sizeof (u_int32_t) * 5));
		wp += i;
		written_len = wp - buf;
		break;
	case RT_MX:
		rp = rrp;
		wp = buf;
		memcpy (wp, rp, (i = sizeof (u_int16_t) * 1));	/* PREFERENCE */
		wp += i;
		rp += i;
		if (!conv_dname_decompress (wp, buflen - (wp - buf), rp, begin, end, &i)) {
			syslog (LOG_INFO, "resource record invalid -- MX EXCHANGE.");
			return -1;
		}
		wp += i;
		written_len = wp - buf;
		break;
	case RT_RP:
		/* two domain names */
		rp = rrp;
		wp = buf;
		rp = conv_dname_decompress (wp, buflen, rp, begin, end, &i);
		if (!rp) {
			syslog (LOG_INFO, "resource record invalid -- RP MBOX-DNAME\n");
			return -1;
		}
		wp += i;
		rp = conv_dname_decompress (wp, buflen - (wp - buf), rp, begin, end, &i);
		if (!rp) {
			syslog (LOG_INFO, "resource record invalid -- RP TXT-DNAME\n");
			return -1;
		}
		wp += i;
		written_len = wp - buf;
		break;
	case RT_A:
	case RT_HINFO:
	case RT_AAAA:
	case RT_A6:
	case RT_SRV:
	case RT_TXT:
		/* no modification */
		return 0;
		break;
	default:
		syslog (LOG_INFO, "unknown resource type %d / fall back to no modification.\n", r_type);
		return 0;
		break;
	}

	return written_len;
}

u_char *mesg_read_sec (int cnt, u_char * sec, u_char * begin, u_char * end, G_List * ls) {
	char *fn = "mesg_read_sec()";
	G_List *rc_head, *gl_tmp;
	RRset_Couple *rc_tmp;
	RRset *rrs_tmp = NULL;
	RR_List *rrl_tmp;

	int i, ret, dname_len;
	u_char *rp, *rp_ex, *rname;
	u_char buf[MAX_PACKET];
	u_int16_t r_type, r_class, rdlen, rdlen_ex;
	u_int32_t r_ttl;
	u_char *rdp;

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: start", fn));
	}

	/* initialize */
	rc_head = list_init ();
	if (!rc_head)
		return NULL;
	cnt = ntohs (cnt);

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: read and decompress RDATA", fn));
	}

	rp = sec;
	for (i = 0; i < cnt; i++) {
		/* parse header */
		rname = rp;
		if ((rp = mesg_skip_dname (rp, end)) == NULL)
			goto bad;
		if (rp + sizeof(u_int16_t) * 3 + sizeof(u_int32_t) > end)
			goto bad;
		GETSHORT (r_type, rp);
		GETSHORT (r_class, rp);
		GETLONG (r_ttl, rp);
		GETSHORT (rdlen, rp);
		rdp = rp;
		rp += rdlen;
		if (rp > end) {
		  bad:
			/* overrun!! */
			syslog (LOG_WARNING, "%s: message overrun", fn);
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s: i = %d, begin = 0x%p, end-begin = 0x%p, rdp - begin = 0x%p, rdlen = 0x%d, rname - begin = 0x%p",
				    fn, i, begin, (void *) (end - begin), (void *) (rdp - begin),
				    rdlen, (void *) (rname - begin)));
			}
			goto error;
		}
		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "%s: seek for matching RRset_Couple", fn));
		}
		rc_head->list_data = NULL;
		for (gl_tmp = rc_head->next; gl_tmp->list_data; gl_tmp = gl_tmp->next) {
			rc_tmp = (RRset_Couple *) (gl_tmp->list_data);
			if ((rc_tmp->rrs->key.info->r_type == r_type) &&
			    (rc_tmp->rrs->key.info->r_class == r_class) &&
			    (mesg_dname_cmp (begin, rname, rrset_key_info_owner (rc_tmp->rrs)) == 0)) {
				if (T.debug > 3) {
					DX (syslog (LOG_DEBUG, "%s: matching record find. rrs->dname = %s / rname = %s",
					            fn, rrset_key_info_owner (rc_tmp->rrs), rname));
				}
				break;
			}
		}
		if (!gl_tmp->list_data) {
			/* extract domain name */
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s: extract domain name", fn));
			}
			if (!conv_dname_decompress (buf, sizeof (buf), rname,
						  begin, end, &dname_len)) {
				syslog (LOG_WARNING, "%s: dname decompression error", fn);
				goto error;
			}
			/* create new one */
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s: create new entry.", fn));
			}

			if (((rc_tmp = malloc (sizeof (RRset_Couple))) == NULL)
			    || ((rc_tmp->rrl = rr_list_alloc ()) == NULL)
			    || ((rc_tmp->rrs = rrset_create (r_type, r_class, dname_len,
						     buf, NULL)) == NULL)) {
				syslog (LOG_ERR, "%s: out of memory", fn);
				goto error;
			}
			/* add it to list */
			if (list_add (rc_head, (void *) rc_tmp) < 0) {
				syslog (LOG_ERR, "%s: can't handle list", fn);
				free (rc_tmp);
				rc_tmp = NULL;
				goto error;
			}
		} else {
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s: re-use old entry (RRset_Couple).", fn));
			}
		}

		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "%s:  extract RR.", fn));
		}

		if (rdlen) {
			ret = mesg_extract_rr (begin, end, r_type, r_class, rdp, buf, sizeof (buf));
			if (ret < 0) {
				syslog (LOG_NOTICE, "%s: RR extraction failed.", fn);
				goto error;
			} else if (!ret) {
				rp_ex = rdp;
				rdlen_ex = rdlen;
			} else {
				rp_ex = buf;
				rdlen_ex = ret;
			}
		} else {
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s:  rdlen == 0 -- update?", fn));
			}
			rp_ex = NULL;
			rdlen_ex = 0;
		}

		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "%s: add the extracted RR to matching RR_List.", fn));
		}
		rrl_tmp = rr_list_add (rc_tmp->rrl, r_ttl, rdlen_ex, rp_ex);
		if (!rrl_tmp) {
			syslog (LOG_NOTICE, "%s: rr_list_add failed.", fn);
			goto error;
		}
		rc_tmp->rrl = rrl_tmp;
	}

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: make each RRset from list.", fn));
	}
	rc_head->list_data = NULL;
	for (gl_tmp = rc_head->next; gl_tmp->list_data; gl_tmp = gl_tmp->next) {
		/* create complete RRset */
		rc_tmp = (RRset_Couple *) (gl_tmp->list_data);
		rrs_tmp = rrset_create (rc_tmp->rrs->key.info->r_type,
					rc_tmp->rrs->key.info->r_class,
					rc_tmp->rrs->key.info->owner_len,
					rrset_key_info_owner (rc_tmp->rrs),
					rc_tmp->rrl);
		if (!rrs_tmp) {
			syslog (LOG_WARNING, "%s: failed to create RRset", fn);
			goto error;
		}
		if (ls) {
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s: add it to R_List* ls.", fn));
			}
			if (list_add (ls, rrset_copy (rrs_tmp)) < 0) {
				syslog (LOG_ERR, "%s: list_add failed.", fn);
				goto error;
			}
		}
		/* free rrs_tmp, we dont need it now */
		rrset_free (rrs_tmp);
		rrs_tmp = NULL;
	}

	/* free unused resources */
	list_destroy (rc_head, rrset_couple_freev);

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: end.", fn));
	}
	return rp;		/* <- next byte of this section */
	/* NOTREACHED */

error:
	syslog (LOG_INFO, "%s: message extraction failed.", fn);
	list_destroy (rc_head, rrset_couple_freev);
	return NULL;
}

/* XXX: unused? */
u_char *mesg_read_sec_in_order (int cnt, u_char * sec, u_char * begin,
				    u_char * end, G_List * ls) {

	u_char *rp;		/* read pointer */
	int i;

	u_char *rname;		/* reading resource name */
	u_char *dname;		/* extracted domain name */
	int dname_len;		/* extracted domain name length */
	u_int16_t r_type;
	u_int16_t r_class;
	u_int32_t r_ttl;
	u_int16_t rdlen;
	u_char *rdp;		/* resource data pointer(points in packet) */
	u_char *rdp_ex;		/* extracted resource data */
	u_int16_t rdlen_ex;	/* extracted resource data length */
	u_char dname_buf[MAX_DNAME];
	u_char rd_buf[MAX_PACKET];

	int ret;		/* return value holder */

	RRset *rrs_new = NULL;	/* ALLOC'ED */

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "mesg_read_sec_in_order(): start\n"));
	}

	cnt = ntohs (cnt);

	/* read and decompress RDATA */
	rp = sec;
	for (i = 0; i < cnt; i++) {
		/* parse header */
		rname = rp;
		if ((rp = mesg_skip_dname (rp, end)) == NULL)
			goto bad;
		GETSHORT (r_type, rp);
		GETSHORT (r_class, rp);
		GETLONG (r_ttl, rp);
		GETSHORT (rdlen, rp);
		rdp = rp;
		rp += rdlen;

		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "mesg_read_sec_in_order(): packet info: type = %s, class = %s, rd_length = %d\n", 
					string_rtype (r_type), string_rclass (r_class), rdlen));
		}

		if (rp > end) {
		  bad:
			/* overrun!! */
			syslog (LOG_WARNING, "mesg_read_sec_in_order(): message overrun\n");
			goto error;
		}
		/* extract domain name */
		if (conv_dname_decompress (dname_buf, sizeof (dname_buf), rname,
					   begin, end, &dname_len) == NULL) {
			syslog (LOG_WARNING, "mesg_read_sec_in_order(): dname decompression error.\n");
			goto error;
		}
		dname = dname_buf;

		/* extract RR */
		if (rdlen != 0) {
			/* #warning XXX -- unknown resource type treatment. */
			if ((ret = mesg_extract_rr (begin, end, r_type, r_class, rdp, rd_buf,
						    sizeof (rd_buf))) < 0) {
				syslog (LOG_NOTICE, "mesg_read_sec_in_order(): RR extraction failed.\n");
				goto error;
			} else if (ret == 0) {
				/* data was not compressed */
				rdp_ex = rdp;
				rdlen_ex = rdlen;
			} else {
				/* data was compressed */
				rdp_ex = rd_buf;
				rdlen_ex = ret;
			}
		} else {
			rdp_ex = NULL;
			rdlen_ex = 0;
		}		/* if(rdlen != 0) */

		/* make new RRset */
		rrs_new = rrset_create_single (dname, dname_len, r_type, r_class,
					       r_ttl, rdlen_ex, rdp_ex);
		if (!rrs_new) {
			syslog (LOG_ERR, "mesg_read_sec_in_order(): can't allocate memory for new RRset\n");
			goto error;
		}
		/* concat it to the tail of list */
		if (list_add_tail (ls, rrs_new) < 0) {
			/* add failed! */
			rrset_free (rrs_new);
			rrs_new = NULL;
		} else {
			/* added -- prevent destruction of rrs_new */
			rrs_new = NULL;
		}
	}

	/* failsafe */
	if (rrs_new) {
		rrset_free (rrs_new);
	}

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "mesg_read_sec_in_order(): finish.\n"));
	}
	return rp;		/* next byte of this section */
	/* NOTREACHED */

error:
	syslog (LOG_INFO, "mesg_read_sec_in_order(): message extraction failed.\n");
	if (rrs_new) {
		rrset_free (rrs_new);
	}
	return NULL;
}


int mesg_parse(u_char * mesg, int mesg_len, G_List * an_list,
	       G_List * ns_list, G_List * ar_list) {

	Mesg_Hdr *hdr;
	u_char *rp;

	if (mesg_len < sizeof(*hdr)) { /* for safety. */
		syslog(LOG_NOTICE, "message overrun\n");
		return(-1);
	}

	hdr = (Mesg_Hdr *) mesg;
	rp = (u_char *) (hdr + 1);	/* read next of the header */

	if (hdr->qdcnt) {
		/* skip question section */
		rp = mesg_skip_dname (rp, mesg + mesg_len);
		rp += 4;	/* sizeof qtype and qclass */
		if (rp > mesg +mesg_len) {
			syslog (LOG_NOTICE, "message overrun\n");
			return -1;
		}
	}
	rp = mesg_read_sec (hdr->ancnt, rp, mesg, mesg +mesg_len, an_list);
	if (!rp) {
		syslog (LOG_INFO, "mesg_parse(): can't read answer section.");
		return -1;
	}
	rp = mesg_read_sec (hdr->nscnt, rp, mesg, mesg +mesg_len, ns_list);
	if (!rp) {
		syslog (LOG_INFO, "mesg_parse(): can't read authority section.");
		return -1;
	}
	rp = mesg_read_sec (hdr->arcnt, rp, mesg, mesg +mesg_len, ar_list);
	if (!rp) {
		syslog (LOG_INFO, "mesg_parse(): can't read additional record section.");
		return -1;
	}
	return 0;
}
