/****************************************************************************
 * Copyright (C) 1998 WIDE Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by WIDE Project and
 *    and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 ****************************************************************************/

/****************************************************************************
 * Copyright (C) 1999 University of Tromso.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by the University of Tromso
 *    and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 ****************************************************************************/

/*
 * <$Id: ne_io.c,v 3.26 2001/03/13 11:16:50 dillema Exp $>
 */

#include "tot.h"

COPYRIGHT1(
"@(#) Copyright (C) 1998 WIDE Project.  All rights reserved.\n");

COPYRIGHT2(
"@(#) Copyright (C) 1999 The University of Tromso.  All rights reserved.\n");

COPYRIGHT3("Written by: F. W. Dillema. feico@pasta.cs.uit.no.\n
based on code by: Yusuke DOI, Keio Univ. Murai Lab.");

CVSID("$Id: ne_io.c,v 3.26 2001/03/13 11:16:50 dillema Exp $");

struct sockaddr *net_parse_alloc_addr (char *caddr, int port, int *sa_len_ret) {

	const char *fn = "net_parse_alloc_addr()";
	struct sockaddr *sa_p = NULL;
#ifdef USE_INET4
	struct sockaddr_in *sin_p;
#endif
#ifdef USE_INET6
	struct sockaddr_in6 *sin6_p;
#endif
	int sa_len = 0;

	if (strchr (caddr, ':')) {
#ifdef USE_INET6
		sa_len = sizeof (struct sockaddr_in6);

		sa_p = malloc (sa_len);
		if (!sa_p) {
			syslog (EM_F (EM_F_MEMEX), fn);
			goto error;
		}
		memset (sa_p, 0, sa_len);
		sin6_p = (struct sockaddr_in6 *) sa_p;
		memset ((void *) sin6_p, 0, sa_len);
		sin6_p->sin6_len = sa_len;
		sin6_p->sin6_family = AF_INET6;
		sin6_p->sin6_port = htons (port);
		if (inet_pton (AF_INET6, caddr, &sin6_p->sin6_addr) != 1)
			goto error;
#else
		goto error;
#endif
	} else {
#ifdef USE_INET4
		sa_len = sizeof (struct sockaddr_in);
		sa_p = malloc (sa_len);
		if (!sa_p) {
			syslog (EM_F (EM_F_MEMEX), fn);
			goto error;
		}
		memset ((void *) sa_p, 0, sa_len);
		sa_p->sa_len = sa_len;
		sin_p = (struct sockaddr_in *) sa_p;
		sin_p->sin_family = AF_INET;
		sin_p->sin_port = htons (port);
		sin_p->sin_addr.s_addr = inet_addr (caddr);
#else
		goto error;
#endif
	}

	if (sa_len_ret)
		*sa_len_ret = sa_len;

	return sa_p;
error:
	if (sa_p)
		free (sa_p);

	return NULL;
}

int net_sockaddr_cmp (struct sockaddr * sa_p_1, int sa_len_1,
		          struct sockaddr * sa_p_2, int sa_len_2, int ignore_flag) {
#ifdef USE_INET4
	struct sockaddr_in *sin_p_1;
	struct sockaddr_in *sin_p_2;
#endif
#ifdef USE_INET6
	struct sockaddr_in6 *sin6_p_1;
	struct sockaddr_in6 *sin6_p_2;
#endif

	if (sa_p_1->sa_family == sa_p_2->sa_family
	    && sa_len_1 == sa_len_2) {
		switch (sa_p_1->sa_family) {
#ifdef USE_INET4
		case AF_INET:
			sin_p_1 = (struct sockaddr_in *) sa_p_1;
			sin_p_2 = (struct sockaddr_in *) sa_p_2;
			if (((ignore_flag & NET_SOCKADDR_CMP_IGNORE_PORT)
			     || (sin_p_1->sin_port == sin_p_2->sin_port))
			    && (memcmp (&(sin_p_1->sin_addr), &(sin_p_2->sin_addr),
					IPV4_ADDR_LEN) == 0)) {
				/* perfect match :) */
				return 0;
			}
			break;
#endif
#ifdef USE_INET6
		case AF_INET6:
			sin6_p_1 = (struct sockaddr_in6 *) sa_p_1;
			sin6_p_2 = (struct sockaddr_in6 *) sa_p_2;
			if (((ignore_flag & NET_SOCKADDR_CMP_IGNORE_PORT)
			     || (sin6_p_1->sin6_port == sin6_p_2->sin6_port))
			    && (memcmp (&(sin6_p_1->sin6_addr), &(sin6_p_2->sin6_addr),
					IPV6_ADDR_LEN) == 0)) {
				return 0;
			}
			break;
#endif
		default:
			syslog (LOG_ERR, "net_sockaddr_cmp(): unknown address family %d\n",
				sa_p_1->sa_family);
			break;
		}		/* switch(address family) */
	}			/* if(address family and length match) */
	return -1;		/* not same */
}

int net_ifc_cmp (u_char * buf1, u_char * buf2, int buf_len) {
	struct ifreq *ifr_p1;
	struct ifreq *ifr_p2;
	u_char *cp1;
	u_char *cp2;

	cp1 = buf1;
	cp2 = buf2;

	while (*cp1 && *cp2) {
		ifr_p1 = (struct ifreq *) cp1;
		ifr_p2 = (struct ifreq *) cp2;

		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "net_ifc_cmp(): if1 %s, if2 %s\n", ifr_p1->ifr_name, ifr_p2->ifr_name));
		}

		/* comparison only AF_INET and AF_INET6 */
		if (ifr_p1->ifr_addr.sa_family != ifr_p2->ifr_addr.sa_family)
			return -1;

		switch (ifr_p1->ifr_addr.sa_family) {
#ifdef USE_INET4
		case AF_INET:
#endif
#ifdef USE_INET6
		case AF_INET6:
#endif
			if (strcmp (ifr_p1->ifr_name, ifr_p2->ifr_name)) {
				return -1;
			}
			if (memcmp (&(ifr_p1->ifr_addr), &(ifr_p2->ifr_addr),
				    SOCKADDR_SIZEOF (ifr_p1->ifr_addr))) {
				return -1;
			}
			break;

		default:
			/* do nothing */
			break;
		}

		cp1 = cp1 + IFNAMSIZ + SOCKADDR_SIZEOF (ifr_p1->ifr_addr);
		cp2 = cp2 + IFNAMSIZ + SOCKADDR_SIZEOF (ifr_p2->ifr_addr);
	}			/* while(*cp1 && *cp2) */

	return 0;
}

int net_addr_len (int af) {
#ifdef USE_INET6
	if (af == AF_INET6) {
		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "%s: addrlen: %d\n", "net_addr_len()", IPV6_ADDR_LEN));
		}
		return IPV6_ADDR_LEN;
	}
#endif
#ifdef USE_INET6
	if (af == AF_INET) {
		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "%s: addrlen: %d\n", "net_addr_len()", IPV4_ADDR_LEN));
		}
		return IPV4_ADDR_LEN;
	}
#endif
	return -1;
}

struct sockaddr *net_sockaddr_alloc (int af, u_char * addr, int addr_len, int port) {
	struct sockaddr *sa;
#ifdef USE_INET4
	struct sockaddr_in *sin;
#endif
#ifdef USE_INET6
	struct sockaddr_in6 *sin6;
#endif
	int sa_len = 0;

#ifdef USE_INET4
	if (af == AF_INET)
		sa_len = sizeof (struct sockaddr_in);
	else
#endif
#ifdef USE_INET6
	if (af == AF_INET6)
		sa_len = sizeof (struct sockaddr_in6);
#endif

	if (sa_len == 0) {
		syslog (LOG_ERR, "unknown address family(%d) specified to allocate sockaddr structure.\n", af);
		return NULL;
	}
	sa = (struct sockaddr *) malloc (sa_len);
	if (!sa)
		return NULL;

	memset ((void *) sa, 0, sa_len);
	sa->sa_len = sa_len;
	sa->sa_family = af;

#ifdef USE_INET4
	if (af == AF_INET) {
		sin = (struct sockaddr_in *) sa;
		sin->sin_port = htons (port);
		memcpy (&(sin->sin_addr.s_addr), addr, addr_len);
	} else
#endif
#ifdef USE_INET6
	if (af == AF_INET6) {
		sin6 = (struct sockaddr_in6 *) sa;
		sin6->sin6_port = htons (port);
		sin6->sin6_flowinfo = htonl (0);
		memcpy (sin6->sin6_addr.s6_addr, addr, addr_len);
	}
#endif
	{			/* ugly */
	};

	return sa;
}

/*-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* Private?                                                        */
/* ni_list_alloc      : allocate ni_list                          */
/* ni_list_destroy    : free all of members on the chain          */
/* ni_find_by_sock    : find NI_List by socket id                 */
/* ni_find_by_addr    : find NI_List by sockaddr                  */
/* ni_fds_set         : set all bits in the fd_set for all I/F    */
/* ni_fds_isset       : check NI_List available for fd_set        */
/* ni_set_wildsock    : set wildcard socket                       */
/* ni_default         : return default interface                  */
/*-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/

static NI_List *NI_head = NULL;	/* lists are to be allocated */
static NI_List *NI_tail = NULL;
static NI_List *NI_wildcard = NULL;	/* linked element of wildcard socket */
static NI_List *NI_wildcard6 = NULL;	/* wildcard 6 */

NI_List *ni_list_alloc (struct sockaddr * sa_p, int sock) {
	const char *fn = "ni_list_alloc()";
	NI_List *ni_tmp;

	ni_tmp = (NI_List *) malloc (sizeof (NI_List));
	if (!ni_tmp) {
		syslog (EM_F (EM_F_MEMEX), fn);
		return NULL;
	}
	if (sa_p) {
		ni_tmp->sa_p = (struct sockaddr *) malloc (SOCKADDR_SIZEOF (*sa_p));
		if (!ni_tmp->sa_p) {
			free (ni_tmp);
			syslog (LOG_WARNING, "%s: memory exhausted.", fn);
			return NULL;
		}
		memset (ni_tmp->sa_p, 0, SOCKADDR_SIZEOF (*sa_p));
		memcpy (ni_tmp->sa_p, sa_p, SOCKADDR_SIZEOF (*sa_p));
	} else
		ni_tmp->sa_p = NULL;

	ni_tmp->sock = sock;
	ni_tmp->next = NULL;

	return ni_tmp;
}

void ni_free (NI_List * ni_p, int close_flag) {
	if (ni_p->sa_p)
		free (ni_p->sa_p);

	if (ni_p->sock >= 0 && close_flag) {
		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "ni_free(): socket close(fd = %d)\n", ni_p->sock));
		}
		close (ni_p->sock);
	}
	free (ni_p);
}

void ni_list_destroy (NI_List * list_head) {
	NI_List *ni_tmp;

	while (list_head) {
		ni_tmp = list_head->next;
		if (list_head == NI_wildcard) {
			NI_wildcard = NULL;
		} else if (list_head == NI_wildcard6) {
			NI_wildcard6 = NULL;
		}
		ni_free (list_head, 1);

		list_head = ni_tmp;
	}

	return;
}

void ni_list_print (int fd) {
	char buf[512];
	NI_List *ni_tmp;
#ifdef USE_INET4
	struct sockaddr_in *sin_p;
#endif
#ifdef USE_INET6
	struct sockaddr_in6 *sin6_p;
	char addrbuf[80];
#endif

	for (ni_tmp = NI_head; ni_tmp->next; ni_tmp = ni_tmp->next) {
		if (ni_tmp->sa_p == NULL)
			sprintf (buf, "if: NULL\n");
		else {
			switch (ni_tmp->sa_p->sa_family) {
#ifdef USE_INET4
			case AF_INET:
				sin_p = (struct sockaddr_in *) (ni_tmp->sa_p);
				sprintf (buf, "IPv4: %s\tport = %d\tsocket = %d\n",
					 inet_ntoa (sin_p->sin_addr), ntohs (sin_p->sin_port),
					 ni_tmp->sock);
				break;
#endif
#ifdef USE_INET6
			case AF_INET6:
				sin6_p = (struct sockaddr_in6 *) (ni_tmp->sa_p);
				inet_ntop (AF_INET6, sin6_p->sin6_addr.s6_addr, addrbuf, sizeof (addrbuf));

				sprintf (buf, "IPv6: %s\tport = %d\tsocket = %d\n",
					 addrbuf, ntohs (sin6_p->sin6_port), ni_tmp->sock);

				break;
#endif
			default:
				sprintf (buf, "if: unknown(id = %d)\tsocket = %d\n",
				     ni_tmp->sa_p->sa_family, ni_tmp->sock);
				break;
			}	/* switch(ni_tmp->sa_p->sa_family) */
		}		/* if(ni_tmp->sa_p == NULL) */
		write (fd, buf, strlen (buf));
	}			/* for(ni_tmp = .... ) */

	return;
}


NI_List *ni_find_by_sock (int sock_id) {
	NI_List *ni_tmp;

	NI_tail->sock = sock_id;
	for (ni_tmp = NI_head; ni_tmp->sock != sock_id; ni_tmp = ni_tmp->next) {
		;		/* just skip it */
	}

	if (ni_tmp == NI_tail)
		return NULL;

	return ni_tmp;
}

NI_List *ni_find_by_addr (struct sockaddr * sa) {
	NI_List *ni_tmp;
#ifdef USE_INET4
	struct sockaddr_in *sin_p;
	struct sockaddr_in *sin_p2;
#endif
#ifdef USE_INET6
	struct sockaddr_in6 *sin6_p;
	struct sockaddr_in6 *sin6_p2;
#endif

	NI_tail->sa_p = sa;

	ni_tmp = NI_head;
	while (ni_tmp) {
		/* compare the address */
		if (ni_tmp->sa_p->sa_family == sa->sa_family) {
#ifdef USE_INET4
			if (sa->sa_family == AF_INET) {
				sin_p = (struct sockaddr_in *) sa;
				sin_p2 = (struct sockaddr_in *) ni_tmp->sa_p;
				if (!memcmp (&(sin_p->sin_addr), &(sin_p2->sin_addr), 4))
					break;

			} else
#endif
#ifdef USE_INET6
			if (sa->sa_family == AF_INET6) {
				sin6_p = (struct sockaddr_in6 *) sa;
				sin6_p2 = (struct sockaddr_in6 *) ni_tmp->sa_p;
				if (!memcmp (sin6_p->sin6_addr.s6_addr, sin6_p2->sin6_addr.s6_addr, 16))
					break;

			}
#endif
			{	/* ugly */
			};
		}
		/* goto next */
		ni_tmp = ni_tmp->next;
	}

	/* clean up */
	NI_tail->sa_p = NULL;


	if (ni_tmp == NI_tail)
		return NULL;

	return ni_tmp;
}

void ni_fds_set (fd_set * fds, int *max_fd) {
	NI_List *ni_tmp;

	if (NI_head == NULL)
		return;

	for (ni_tmp = NI_head; ni_tmp->next != NULL; ni_tmp = ni_tmp->next) {
		if (ni_tmp->sa_p != NULL && ni_tmp->sock >= 0) {
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "ni_fds_set(): FD_SET(%d)", ni_tmp->sock));
			}
			FD_SET (ni_tmp->sock, fds);
			if (max_fd)
				*max_fd = MAXNUM (*max_fd, ni_tmp->sock);

		}
	}

	return;
}

NI_List *ni_fds_isset (fd_set * fds) {
	NI_List *ni_tmp;

	if (!NI_head)
		return NULL;

	for (ni_tmp = NI_head; ni_tmp->next != NULL; ni_tmp = ni_tmp->next) {
		if (ni_tmp->sock >= 0 && ni_tmp->sa_p && FD_ISSET (ni_tmp->sock, fds)) {
			FD_CLR (ni_tmp->sock, fds);
			return ni_tmp;
		}
	}

	return NULL;
}

int ni_set_wildsock (void) {
        DX(const char *fn = "ni_set_wildsock()";)
#ifdef USE_INET4
	struct sockaddr_in *sin_p;
#endif
#ifdef USE_INET6
	struct sockaddr_in6 *sin6_p;
#ifdef IPV6_USE_MIN_MTU
	const int on = 1;
#endif
#endif

#ifdef USE_INET4
	if (!NI_wildcard) {
		NI_wildcard = (NI_List *) malloc (sizeof (NI_List));
		if (!NI_wildcard)
			return -1;

		/* make IPv4 default socket */
		NI_wildcard->sock = socket (PF_INET, SOCK_DGRAM, 0);
		if (NI_wildcard->sock < 0) {
			free (NI_wildcard);
			return -1;
		}
		NI_wildcard->sa_p = (struct sockaddr *) malloc (sizeof (struct sockaddr_in));
		if (!NI_wildcard->sa_p) {
			free (NI_wildcard);
			return -1;
		}
		sin_p = (struct sockaddr_in *) NI_wildcard->sa_p;
		memset (sin_p, 0, sizeof (struct sockaddr_in));
		sin_p->sin_len = sizeof (struct sockaddr_in);
		sin_p->sin_family = AF_INET;

		if (bind (NI_wildcard->sock, NI_wildcard->sa_p,
			  SOCKADDR_SIZEOF (*(NI_wildcard->sa_p))) < 0) {
			free (NI_wildcard->sa_p);
			free (NI_wildcard);
			return -1;
		}

		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "%s: socket(IPV4) open(wildcard fd = %d)\n", fn, NI_wildcard->sock));
		}

		/* concat to the list */
		NI_wildcard->next = NI_head;
		NI_head = NI_wildcard;

	}
#endif
#ifdef USE_INET6
	if (!NI_wildcard6) {
		NI_wildcard6 = (NI_List *) malloc (sizeof (NI_List));
		if (!NI_wildcard6)
			return -1;

		/* make IPv6 default socket */
		NI_wildcard6->sock = socket (PF_INET6, SOCK_DGRAM, 0);
		if (NI_wildcard6->sock < 0) {
			free (NI_wildcard6);
			return -1;
		}
		NI_wildcard6->sa_p = (struct sockaddr *) malloc (sizeof (struct sockaddr_in6));
		if (!NI_wildcard6->sa_p) {
			free (NI_wildcard6);
			return -1;
		}
		sin6_p = (struct sockaddr_in6 *) NI_wildcard6->sa_p;
		memset (sin6_p, 0, sizeof (struct sockaddr_in6));
		sin6_p->sin6_len = sizeof (struct sockaddr_in6);
		sin6_p->sin6_family = AF_INET6;
		sin6_p->sin6_port = 0;
		sin6_p->sin6_flowinfo = 0;
		sin6_p->sin6_addr = in6addr_any;

		if (bind (NI_wildcard6->sock, NI_wildcard6->sa_p,
			  SOCKADDR_SIZEOF (*(NI_wildcard6->sa_p))) < 0) {
			free (NI_wildcard6->sa_p);
			free (NI_wildcard6);
			return -1;
		}

		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "%s: socket(IPv6) open(wildcard fd = %d)\n", fn, NI_wildcard6->sock));
		}

		/* concat to the list */
		NI_wildcard6->next = NI_head;
		NI_head = NI_wildcard6;

#ifdef IPV6_USE_MIN_MTU
		/* ignore error */
		(void)setsockopt(NI_wildcard6->sock, IPPROTO_IPV6,
			IPV6_USE_MIN_MTU, &on, sizeof(on));
#endif
	}
#endif

	return 0;
}

NI_List *ni_default (int af) {
	switch (af) {
#ifdef USE_INET4
		case AF_INET:
		return NI_wildcard;
#endif
#ifdef USE_INET6
	case AF_INET6:
		return NI_wildcard6;
#endif
	}

	return NULL;		/* no match */

}

int ni_default_sock (int af) {
        DX(const char *fn = "ni_default_sock()";)
	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: start\n", fn));
	}

	switch (af) {
#ifdef USE_INET4
	case AF_INET:
		if (NI_wildcard) {
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s: return AF_INET.default = %d.\n", fn, NI_wildcard->sock));
			}
			return NI_wildcard->sock;
		}
		break;
#endif
#ifdef USE_INET4
	case AF_INET6:
		if (NI_wildcard6) {
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s: return AF_INET6.default = %d.\n", fn, NI_wildcard6->sock));
			}
			return NI_wildcard6->sock;
		}
		break;
#endif
	}

	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: return ERROR.\n", fn));
	}
	return -1;
}

NI_List *ni_copy (NI_List * ni_p) {
	if (ni_p)
		return ni_list_alloc (ni_p->sa_p, ni_p->sock);
	else
		return NULL;

}

/*-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* NOTE: TCP connection is based on socket and request is based on */
/* the address.                                              */
/*-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* TCP                                                             */
/* net_stream_socket  : open and bind socket for stream           */
/* net_stream_connect : connect client socket to specified dest   */
/* net_stream_accept  : (not implemented) accept connection       */
/* net_stream_read    : (not implemented) read data from stream   */
/* net_stream_write   : (not implemented) write data to stream    */
/*-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
int net_stream_socket (u_int16_t port, int af) {
	const char *fn = "net_stream_socket()";
	int sock;
	const int on = 1;
#ifdef USE_INET6
	struct sockaddr_in6 myaddr6;
#endif
#ifdef USE_INET4
	struct sockaddr_in myaddr;
#endif
	struct sockaddr *sa_p;

	if (af == AF_INET6) {
#ifdef USE_INET6
		sock = socket (AF_INET6, SOCK_STREAM, 0);
		if (sock < 0) {
			syslog (LOG_ERR, "%s: socket6 open failed: %m", fn);
			return -1;
		}
		memset (&myaddr6, 0, sizeof (myaddr6));
		myaddr6.sin6_len = sizeof (myaddr6);
		myaddr6.sin6_family = AF_INET6;
		myaddr6.sin6_flowinfo = 0;
		myaddr6.sin6_port = htons (port);
		myaddr6.sin6_addr = in6addr_any;
		sa_p = (struct sockaddr *) & myaddr6;
#else
		syslog (LOG_ERR, "%s: stream socket for AF_INET6 is not supported.\n", fn);
		return -1;
#endif
	} else if (af == AF_INET) {
#ifdef USE_INET4
		sock = socket (AF_INET, SOCK_STREAM, 0);
		if (sock < 0) {
			syslog (LOG_ERR, "%s: socket open failed: %m", fn);
			return -1;
		}
		memset (&myaddr, 0, sizeof (myaddr));
		myaddr.sin_len = sizeof (myaddr);
		myaddr.sin_family = AF_INET;
		myaddr.sin_port = htons (port);
		myaddr.sin_addr.s_addr = INADDR_ANY;
		sa_p = (struct sockaddr *) & myaddr;
#else
		syslog (LOG_ERR, "%s: stream socket for AF_INET6 is not supported.\n", fn);
		return -1;
#endif
	} else {
		syslog (LOG_ERR, "%s: address family %d is not available.\n", fn, af);
		return -1;
	}

	if (setsockopt (sock, SOL_SOCKET, SO_REUSEADDR, (char *) &on, sizeof (on))) {
		syslog (LOG_WARNING, "setsockopt(...): %m");
		/* anyway go ahead */
	}
	if (bind (sock, sa_p, sa_p->sa_len) != 0) {
		syslog (LOG_ERR, "%s: can't bind stream socket: %m", fn);
		close (sock);
		return -1;
	}
	if (ioctl (sock, FIONBIO, (char *) &on) < 0) {
		syslog (LOG_ERR, "%s: can't ioctl on service socket: %m", fn);
		return -1;
	}
	if (listen (sock, 5) != 0) {
		syslog (LOG_ERR, "%s: listen failed: %m", fn);
		close (sock);
		return -1;
	}
	return sock;
}

int net_stream_connect (u_int16_t af, u_char * addr, int addr_len, u_int16_t port) {
	const char *fn = "net_stream_connect()";
	u_char buf[128];
	const int on = 1;
	struct sockaddr *sa_p;
	int namelen;
	int socki;

	sa_p = (struct sockaddr *) buf;

	/* set sa_p->sa_len if available */
	if (sizeof (buf) - 2 < addr_len) {
		syslog (LOG_WARNING, "%s: too long network address.", fn);
		return -1;
	}
#ifdef USE_INET4
	if (af == AF_INET) {
		namelen = sa_p->sa_len = MAXNUM (sizeof (struct sockaddr_in),
				  addr_len + 2 /* sa_len and sa_family */ );
	} else
#endif
#ifdef USE_INET6
	if (af == AF_INET6) {
		namelen = sa_p->sa_len = sizeof (struct sockaddr_in6);
	} else
#endif
	{
		syslog (LOG_ERR, "%s: unknown address family to connect\n", fn);
		return -1;
	}

#ifdef USE_INET4
	if (af == AF_INET && addr_len == 4) {
		struct sockaddr_in *sin_p;
		sa_p->sa_family = af;
		sin_p = (struct sockaddr_in *) sa_p;
		sin_p->sin_port = htons (port);
		memcpy (&(sin_p->sin_addr), addr, addr_len);
		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "%s: connect to %s(v4)", fn, inet_ntoa (sin_p->sin_addr)));
		}
	} else
#endif
#ifdef USE_INET6
	if (af == AF_INET6 && addr_len == 16) {
		struct sockaddr_in6 *sin6_p;

		sa_p->sa_family = af;
		sin6_p = (struct sockaddr_in6 *) sa_p;
		sin6_p->sin6_flowinfo = 0;
		sin6_p->sin6_port = htons (port);
		memcpy (&sin6_p->sin6_addr, addr, addr_len);
	} else
#endif
	{
		/* unknown address length */
		syslog (LOG_WARNING, "%s: network address invalid.", fn);
		return -1;
	}

	/* create socket */
	socki = socket (af, SOCK_STREAM, 0);
	if (socki < 0) {
		syslog (LOG_WARNING, "%s: can't open socket: %m", fn);
		return -1;
	}
	if (ioctl (socki, FIONBIO, (char *) &on) < 0) {
		syslog (LOG_WARNING, "%s: can't ioctl on service socket: %m", fn);
		return -1;
	}
	/* start connection */
	if (connect (socki, sa_p, namelen)) {
		if (errno == EINPROGRESS)
			return socki;
		else {
			syslog (LOG_WARNING, "%s: can't connect: %m", fn);
			return -1;
		}
	}
	/* anyway return */
	return socki;
}

/*-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* NOTE: UDP datagram is based on interface and request is based   */
/* on the address.                                                 */
/*-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
/* UDP                                                             */
/* net_mesg_getif : get interface list from TCP socket             */
/* net_mesg_socket: open dgram binded socket on a interface        */
/* net_mesg_send  : send message from specific interface           */
/* net_mesg_recv  : (not implemented) recv mesg with input IF info */
/*-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
NI_List *net_mesg_getif (int tcp_sock, int port_srv) {
	const char *fn = "net_mesg_getif()";
	static int tcp_sock_old = -1;	/* old tcp socket if available */
	static u_char *old_bufp = NULL;	/* allocated */
	static int old_bufp_len = -1;

	NI_List *ni_tmp;
	struct ifconf ifc;
	struct ifreq *ifr_p;
	u_char buf[8192];	/* <- XXX -- is this value appropriate? */

	u_char *cp;

	memset (buf, 0, sizeof (buf));
	ifc.ifc_len = sizeof (buf);
	ifc.ifc_buf = buf;

	if (tcp_sock < 0) {
		if (tcp_sock_old < 0)
			return NULL;

		tcp_sock = tcp_sock_old;
	} else
		tcp_sock_old = tcp_sock;

	if (ioctl (tcp_sock, SIOCGIFCONF, (char *) &ifc) < 0) {
		syslog (LOG_ERR, "%s: can't get network interface information: %m.", fn);
		return NULL;
	}
	if (old_bufp) {
		if (T.debug > 3) {
			DX (syslog (LOG_DEBUG, "%s: check old data\n", fn));
		}

		if (old_bufp_len == ifc.ifc_len && net_ifc_cmp (old_bufp, buf, old_bufp_len) == 0) {
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s: interface list unchanged.\n", fn));
			}
			return NI_head;
		} else {
			syslog (LOG_INFO, "%s: interface list changed.. let's see...\n", fn);
			if (T.debug > 3) {
				DX (syslog (LOG_DEBUG, "%s: old_bufp_len = %d and ifc.ifc_len = %d\n", fn, old_bufp_len, ifc.ifc_len));
			}
		}
	}
	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: going to check data. old_bufp = %p\n", fn, old_bufp));
	}

	/* initialize NI_head */
	ni_tmp = NULL;
	if (NI_head) {
		ni_list_destroy (NI_head);
		NI_head = NI_tail = NI_wildcard = NULL;
	}

	NI_tail = NI_head = ni_list_alloc (NULL, -1);
	if (!NI_tail ) {
		syslog (LOG_ERR, "%s: can't initialize interface list.", fn);
		return NULL;
	}
	ni_set_wildsock ();	/* -- open wildcard interface as default. */

	/** search interfaces */
	cp = buf;
	while (*cp) {
		ifr_p = (struct ifreq *) cp;
		switch (ifr_p->ifr_addr.sa_family) {
#ifdef USE_INET4
		case AF_INET:
#endif
#ifdef USE_INET6
		case AF_INET6:
			if (ifr_p->ifr_addr.sa_family == AF_INET6 && !T.ip6) {
				if (T.debug > 3) {
					DX(syslog (LOG_DEBUG, "%s: ignoring interface %s for IPv6\n", fn, ifr_p->ifr_name));
				}
				break;
			}
#endif
#ifdef USE_INET4
			if (ifr_p->ifr_addr.sa_family == AF_INET && !T.ip4) {
				if (T.debug > 3) {
					DX(syslog (LOG_DEBUG, "%s: ignoring interface %s for IPv4\n", fn, ifr_p->ifr_name));
				}
				break;
			}
#endif
			if (T.debug > 3) {
				DX(syslog (LOG_DEBUG, "%s: try to make IPv%c socket for interface %s\n", fn,
					     	((ifr_p->ifr_addr.sa_family == AF_INET) ? '4' :
						(ifr_p->ifr_addr.sa_family == AF_INET6) ? '6' : '?'), ifr_p->ifr_name));
			}

			/* allocate new member of network interface list */
			ni_tmp = ni_list_alloc (&ifr_p->ifr_addr, -1);
			if (!ni_tmp) {
				syslog (LOG_ERR, "%s: can't allocate memory for interface structure.\n", fn);
				ni_list_destroy (NI_head);
				return NULL;
			}
			/* bind network interface and socket */
			if (net_mesg_socket (ni_tmp, port_srv) < 0) {
				syslog (LOG_ERR, "%s: can't bind socket for UDP(ifname = %s).\n", fn,
					ifr_p->ifr_name);
				ni_free (ni_tmp, 1);
			} else {
				/* concat to the list */
				ni_tmp->next = NI_head;
				NI_head = ni_tmp;
			}

			break;

		default:
			/** just ignore */
			break;
		}

		cp = cp + IFNAMSIZ + SOCKADDR_SIZEOF (ifr_p->ifr_addr);
	}

	/* preserve old buffer */
	cp++;			/* include terminating NULL */
	if (old_bufp)
		free (old_bufp);

	old_bufp = malloc (ifc.ifc_len);
	if (!old_bufp) {
		syslog (LOG_ERR, "%s: can't allocate memory.\n", fn);
		old_bufp_len = -1;
	} else {
		memcpy (old_bufp, buf, ifc.ifc_len);
		old_bufp_len = ifc.ifc_len;
	}

	return NI_head;
}

void net_mesg_getif_finish (void) {
	if (NI_head) {
		ni_list_destroy (NI_head);
		NI_head = NULL;
	}
}


int net_mesg_socket (NI_List * ni_p, int port) {
	const char *fn = "net_mesg_socket()";
	int sock;
#ifdef USE_INET4
	struct sockaddr_in *sinp;
#endif
#ifdef USE_INET6
	struct sockaddr_in6 *sin6p;
#endif

#ifdef USE_INET4
	if (ni_p->sa_p->sa_family == AF_INET) {
		sinp = (struct sockaddr_in *) ni_p->sa_p;
		sinp->sin_port = htons (port);
	} else
#endif
#ifdef USE_INET6
	if (ni_p->sa_p->sa_family == AF_INET6) {
		sin6p = (struct sockaddr_in6 *) ni_p->sa_p;
		sin6p->sin6_port = htons (port);
		sin6p->sin6_flowinfo = 0;
	} else
#endif
	{
		syslog (LOG_WARNING, "%s: unknown address family specified.", fn);
		return -1;
	}

	sock = socket (ni_p->sa_p->sa_family, SOCK_DGRAM, 0);
	if (sock < 0) {
		syslog (LOG_ERR, "%s: socket open failed: %m", fn);
		return -1;
	}
	if (T.debug > 3) {
		DX (syslog (LOG_DEBUG, "%s: socket open(fd = %d)\n", fn, sock));
	}

	if (bind (sock, ni_p->sa_p, SOCKADDR_SIZEOF (*ni_p->sa_p)) < 0) {
		syslog (LOG_ERR, "%s: can't bind dgram socket: %m", fn);
		close (sock);
		return -1;
	}
	ni_p->sock = sock;

#ifdef USE_INET6
#ifdef IPV6_USE_MIN_MTU
	if (ni_p->sa_p->sa_family == PF_INET6) {
		const int on = 1;

		/* ignore error */
		(void)setsockopt(NI_wildcard6->sock, IPPROTO_IPV6,
			IPV6_USE_MIN_MTU, &on, sizeof(on));
	}
#endif
#endif

	return sock;
}

int net_mesg_send (NI_List * ni_send, u_char * mesg, int mesg_len, struct sockaddr *sa_p) {
	const char *fn = "net_mesg_send()";

	if (!ni_send)
		ni_send = ni_default (sa_p->sa_family);

	if (!ni_send || ni_send->sock < 0) 
		syslog (LOG_WARNING, "%s: attempts to send message from interface which have no socket.", fn);

#ifdef USE_INET4
	if (ni_send->sa_p->sa_family == AF_INET) {
		if (sa_p->sa_family == AF_INET)
			return sendto (ni_send->sock, mesg, mesg_len, 0, sa_p, SOCKADDR_SIZEOF (*sa_p));
		else
			syslog (LOG_NOTICE, "%s: can't send to IPv6 address with IPv4 socket.\n", fn);
	} else
#endif
#ifdef USE_INET6
	if (ni_send->sa_p->sa_family == AF_INET6) {
		if (sa_p->sa_family == AF_INET6) 
			return sendto (ni_send->sock, mesg, mesg_len, 0, sa_p, SOCKADDR_SIZEOF (*sa_p));
		else
			syslog (LOG_ERR, "%s: can't send to IPv4 address with IPv6 socket.\n", fn);
	} else
#endif
	{
		syslog (LOG_ERR, "%s: unsupported address family %d\n", fn,
			ni_send->sa_p->sa_family);
	}

	return -1;
}
