/*
 *   toscsi.c
 *
 *   SCSI access functions for tosha.
 *
 *   Oliver Fromme  <olli@fromme.com>
 *
 *   Copyright (C) 1997,1998,1999
 *        Oliver Fromme.  All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. Neither the name of the author nor the names of any co-contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY OLIVER FROMME AND CONTRIBUTORS ``AS IS'' AND
 *   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *   ARE DISCLAIMED.  IN NO EVENT SHALL OLIVER FROMME OR CONTRIBUTORS BE LIABLE
 *   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *   SUCH DAMAGE.
 *
 *   @(#)$Id: toscsi.c,v 1.3 1999/01/01 23:32:01 olli Exp $
 */

static const char cvsid[]
    = "@(#)$Id: toscsi.c,v 1.3 1999/01/01 23:32:01 olli Exp $";

#include "utils.h"
#include "global.h"
#include "toscsi.h"

ulong readsectors = 0;	/* total number of sectors we've already read */

/*
 *   Send a "simple" request to the SCSI subsystem.
 *   See the scsi(3) manual page for more information.
 */

int
toscsi_request (toscsi_handle *hdl, ulong size, ulong flags, char *cmd)
{
	int result;

#ifdef CAM
	bzero (&(&hdl->ccb->ccb_h)[1], sizeof(struct ccb_scsiio));
	csio_build (&hdl->ccb->csio,
		    /* data_ptr */	(u_int8_t *) buf,
		    /* dxfer_len */	size,
		    /* flags */		flags | CAM_PASS_ERR_RECOVER,
		    /* retry_count */	3,
		    /* timeout */	10000,
		    /* cmd_spec */	cmd);
	if ((result = cam_send_ccb(hdl->cam_dev, hdl->ccb)) < 0) {
		perror ("error sending SCSI command");
		cam_freeccb (hdl->ccb);
		cam_close_device (hdl->cam_dev);
		exit (1);
	}
	if ((hdl->ccb->ccb_h.status & CAM_STATUS_MASK) != CAM_REQ_CMP) {
		fprintf (stderr, "error returned from SCSI command:\n");
		if ((hdl->ccb->ccb_h.status & CAM_STATUS_MASK) ==
		     CAM_SCSI_STATUS_ERROR)
			scsi_sense_print (hdl->cam_dev,
			    &hdl->ccb->csio, stderr);
		else
			fprintf (stderr, "ccb->ccb_h.status == %d\n",
			    hdl->ccb->ccb_h.status);
		exit (1);
	}
#else /* !CAM */
	scsireq_reset (hdl->sreq);
	hdl->sreq->timeout = 10000;
	scsireq_build (hdl->sreq, size, (char *) buf, flags, cmd);
	result = scsireq_enter(hdl->scsifd, hdl->sreq);
#    ifndef DEBUG
	if (SCSIREQ_ERROR (hdl->sreq))
#    endif
		scsi_debug (stderr, result, hdl->sreq);
#endif /* !CAM */

	return result;
}

static void
print_progress (long long trackstart, long long thistime, int num, int done)
{
	int trackdone, totaldone;
	long long duration;
	ulong trackleft, totalleft;

	totaldone = (readsectors * 100) / totalsectors;
	duration = (long long) thistime - starttime;
	totalleft = ((duration * totalsectors) / readsectors
	    - duration + 500000) / 1000000;
	if (tracklistsize > 1) {
		trackdone = ((ulong) done * 100) / num;
		duration = (long long) thistime - trackstart;
		trackleft = ((duration * num) / done - duration + 500000)
		    / 1000000;
		fprintf (stderr, "  track:%3d%% done,%2d:%02d to go,",
		    trackdone, (int) trackleft / 60, (int) trackleft % 60);
	}
	fprintf (stderr, "  total:%3d%% done,%3d:%02d to go\r",
	    totaldone, (int) totalleft / 60, (int) totalleft % 60);
}

/*
 *   Read sectors from "start" to "endpp" - 1, and output
 *   them to the file descriptor "outfd".
 */

int
toscsi_readsectors (toscsi_handle *hdl, int start, int endpp, int outfd)
{
	long long trackstarttime, lasttime, thistime;
	int sec, secread, result;
	int fcount = 0;

	get_time (&trackstarttime);
	lasttime = trackstarttime;
	for (sec = start; sec < endpp; sec += secread) {
		if ((secread = sectorsperbuf) > endpp - sec)
			secread = endpp - sec;

#ifdef CAM
		if (readcmd == 0xd8)
			csio_build (&hdl->ccb->csio,
			    /* data_ptr */	(u_int8_t *) buf,
			    /* dxfer_len */	SECTORSIZE * secread,
			    /* flags */		CAM_DIR_IN |
			    			CAM_PASS_ERR_RECOVER,
			    /* retries */	3,
			    /* timeout */	10000,
			    /* cmd_spec */	"v 0 0 v:i3 0 0 0 v 0 0",
			    readcmd, sec, secread);
		else
			csio_build (&hdl->ccb->csio,
			    /* data_ptr */	(u_int8_t *) buf,
			    /* dxfer_len */	SECTORSIZE * secread,
			    /* flags */		CAM_DIR_IN |
			    			CAM_PASS_ERR_RECOVER,
			    /* retries */	3,
			    /* timeout */	10000,
			    /* cmd_spec */	"v 0 0 v:i3 0 0 v 0",
			    readcmd, sec, secread);
		if ((result = cam_send_ccb(hdl->cam_dev, hdl->ccb)) < 0) {
			perror ("error sending CD-DA read command");
			cam_freeccb (hdl->ccb);
			cam_close_device (hdl->cam_dev);
			exit (1);
		}
		if ((hdl->ccb->ccb_h.status & CAM_STATUS_MASK) != CAM_REQ_CMP) {
			fprintf(stderr,
			    "error returned from CD-DA read command:\n");
			if ((hdl->ccb->ccb_h.status & CAM_STATUS_MASK) ==
			     CAM_SCSI_STATUS_ERROR)
				scsi_sense_print (hdl->cam_dev,
				    &hdl->ccb->csio, stderr);
			else
				fprintf (stderr, "ccb->ccb_h.status == %d\n",
					hdl->ccb->ccb_h.status);
			exit (1);
		}
#else /* !CAM */
		scsireq_reset (hdl->sreq);
		hdl->sreq->timeout = 10000;
		if (readcmd == 0xd8)
			scsireq_build (hdl->sreq, SECTORSIZE * secread,
				(char *) buf, SCCMD_READ,
				"v 0 0 v:i3 0 0 0 v 0 0",
				readcmd, sec, secread);
		else
			scsireq_build (hdl->sreq, SECTORSIZE * secread,
				(char *) buf, SCCMD_READ,
				"v 0 0 v:i3 0 0 v 0", readcmd,
				sec, secread);
		result = scsireq_enter (hdl->scsifd, hdl->sreq);
#    ifndef DEBUG
		if (SCSIREQ_ERROR (hdl->sreq))
#    endif
			scsi_debug (stderr, result, hdl->sreq);
#endif /* !CAM */
		if (byteswap) {
			int j, num = SECTORSIZE * secread;
			byte t;
			for (j = 0; j < num; j += 2) {
				t = buf[j];
				buf[j] = buf[j + 1];
				buf[j + 1] = t;
			}
		}
		write (outfd, buf, SECTORSIZE * secread);
		readsectors += secread;
		if (verbose && (fcount += secread) > 99) {
			fcount = 0;
			get_time (&thistime);
			if (lasttime + 2000000 < thistime) {
				print_progress (trackstarttime, thistime,
				    endpp - start, sec - start);
				lasttime = thistime;
			}
		}
	}
	return TRUE;
}

/*
 *   Read one data sector and return the mode byte.
 */

int
toscsi_readmode (toscsi_handle *hdl, int sector)
{
	int result;

#ifdef CAM
	csio_build (&hdl->ccb->csio,
	    /* data_ptr */	(u_int8_t *) buf,
	    /* dxfer_len */	SECTORSIZE * 1,
	    /* flags */		CAM_DIR_IN |
	    			CAM_PASS_ERR_RECOVER,
	    /* retries */	3,
	    /* timeout */	10000,
	    /* cmd_spec */	"v 0 0 v:i3 0 0 v 0",
	    0x28, sector, 1);
	if ((result = cam_send_ccb(hdl->cam_dev, hdl->ccb)) < 0) {
		perror ("error sending CD-ROM read command");
		cam_freeccb (hdl->ccb);
		cam_close_device (hdl->cam_dev);
		exit (1);
	}
	if ((hdl->ccb->ccb_h.status & CAM_STATUS_MASK) != CAM_REQ_CMP) {
		fprintf(stderr,
		    "error returned from CD-ROM read command:\n");
		if ((hdl->ccb->ccb_h.status & CAM_STATUS_MASK) ==
		     CAM_SCSI_STATUS_ERROR)
			scsi_sense_print (hdl->cam_dev,
			    &hdl->ccb->csio, stderr);
		else
			fprintf (stderr, "ccb->ccb_h.status == %d\n",
				hdl->ccb->ccb_h.status);
		exit (1);
	}
#else /* !CAM */
	scsireq_reset (hdl->sreq);
	hdl->sreq->timeout = 10000;
	scsireq_build (hdl->sreq, SECTORSIZE * 1,
		(char *) buf, SCCMD_READ,
		"v 0 0 v:i3 0 0 v 0", 0x28, sector, 1);
	result = scsireq_enter (hdl->scsifd, hdl->sreq);
#    ifndef DEBUG
	if (SCSIREQ_ERROR (hdl->sreq))
#    endif
		scsi_debug (stderr, result, hdl->sreq);
#endif /* !CAM */
	return buf[15];
}

toscsi_handle *
toscsi_open (const char *devname)
{
	toscsi_handle *hdl;

	if (!(hdl = (toscsi_handle *) malloc(sizeof(toscsi_handle))))
		out_of_memory();

#ifdef CAM
	if (!(hdl->cam_dev = cam_open_device(devname, O_RDWR))) {
		fprintf (stderr, "%s: %s\n", me, cam_errbuf);
		exit (1);
	}
	if (!(hdl->ccb = cam_getccb(hdl->cam_dev))) {
		fprintf (stderr, "%s: couldn't allocate CCB\n", me);
		cam_close_device (hdl->cam_dev);
		exit (1);
	}
#else /* !CAM */
	if ((hdl->scsifd = scsi_open(devname, O_RDWR)) < 0)
		die (devname);
	if (!(hdl->sreq = scsireq_new())) {
		fprintf (stderr, "%s: scsireq_new(): Out of memory.\n", me);
		exit (1);
	}
#endif /* !CAM */

	return hdl;
}

void
toscsi_close (toscsi_handle *hdl)
{
#ifdef CAM 
	cam_close_device(hdl->cam_dev);
#else /* !CAM */
	close (hdl->scsifd);
#endif /* !CAM */
}

/* EOF */
