/* 
 *   pcmplay.c
 *
 *   Oliver Fromme  <olli@fromme.com>
 *
 *   Copyright (C) 1997,1998,1999
 *        Oliver Fromme.  All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. Neither the name of the author nor the names of any co-contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY OLIVER FROMME AND CONTRIBUTORS ``AS IS'' AND
 *   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *   ARE DISCLAIMED.  IN NO EVENT SHALL OLIVER FROMME OR CONTRIBUTORS BE LIABLE
 *   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *   SUCH DAMAGE.
 *
 *   @(#)$Id: pcmplay.c,v 1.4 1999/01/01 23:31:55 olli Exp $
 */

static const char cvsid[]
    = "@(#)$Id: pcmplay.c,v 1.4 1999/01/01 23:31:55 olli Exp $";

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <machine/soundcard.h>

#include "utils.h"
#include "getlopt.h"

char *audevice = "/dev/dsp";
int buffersize = 32;
int dsp_samplesize = 16;
int dsp_channels = 2;
int dsp_speed = 44100;
int dsp_unsigned = FALSE;
int dsp_network = FALSE;
int mix_gain = -1;

int
usage (char *notused)
{
	fprintf (stderr, "Usage:  %s [options]\n", me);
	fprintf (stderr, "Options [defaults] (remarks):\n");
	fprintf (stderr, "   -d audio_device      [%s]\n", audevice);
	fprintf (stderr, "   -b buffer_size       [%d]   (in Kbytes)\n", buffersize);
	fprintf (stderr, "   -s bits_per_sample   [%d]   (must be 8 or 16)\n", dsp_samplesize);
	fprintf (stderr, "   -f frequency         [%d]\n", dsp_speed);
	fprintf (stderr, "   -c channels          [%d]   (must be 1 or 2)\n", dsp_channels);
	fprintf (stderr, "   -g gain              [don't change]   (0 - 255)\n");
	fprintf (stderr, "   -u                   (data is unsigned; default is signed)\n");
	fprintf (stderr, "   -n                   (data in network byte order; default is intel b.o.)\n");
	exit (1);
	return 0; /* make compiler happy */
}

topt opts[] = {
	{'d', "device",    GLO_CHAR, 0,     &audevice,       0},
	{'b', "buffer",    GLO_NUM,  0,     &buffersize,     0},
	{'s', "bits",      GLO_NUM,  0,     &dsp_samplesize, 0},
	{'f', "frequency", GLO_NUM,  0,     &dsp_speed,      0},
	{'c', "channels",  GLO_NUM,  0,     &dsp_channels,   0},
	{'g', "gain",      GLO_NUM,  0,     &mix_gain,       0},
	{'u', "unsigned",  0,        0,     &dsp_unsigned,   TRUE},
	{'n', "network",   0,        0,     &dsp_network,    TRUE},
	{'h', "help",      0,        usage, 0,               0},
	{0, NULL, 0, 0, 0, 0}
};

int audio_open ()
{
	int fmts;
	int stereo = (dsp_channels == 2 ? TRUE : FALSE);
	int fd;

	if (dsp_samplesize == 8)
		if (dsp_unsigned)
			fmts = AFMT_U8;
		else
			fmts = AFMT_S8;
	else
		if (dsp_network)
			if (dsp_unsigned)
				fmts = AFMT_U16_BE;
			else
				fmts = AFMT_S16_BE;
		else
			if (dsp_unsigned)
				fmts = AFMT_U16_LE;
			else
				fmts = AFMT_S16_LE;
	if ((fd = open(audevice, O_WRONLY)) < 0)
		return (-1);
	ioctl (fd, SNDCTL_DSP_SAMPLESIZE, &dsp_samplesize);
	ioctl (fd, SNDCTL_DSP_STEREO, &stereo);
	ioctl (fd, SNDCTL_DSP_SPEED, &dsp_speed);
	ioctl (fd, SNDCTL_DSP_SETFMT, &fmts);
	if (mix_gain >= 0) {
		int e, mask;
		e = ioctl(fd, SOUND_MIXER_READ_DEVMASK, &mask);
		if (e < 0)
			fprintf (stderr, "audio/gain: Can't get audio device features list.\n");
		else if (mask & SOUND_MASK_PCM) {
			int gain = (mix_gain << 8) | (mix_gain);
			e = ioctl(fd, SOUND_MIXER_WRITE_PCM, &gain);
		}
		else if (!(mask & SOUND_MASK_VOLUME))
			fprintf (stderr, "audio/gain: setable Volume/PCM-Level not supported by your audio device: %#04x\n", mask);
		else { 
			int gain = (mix_gain << 8) | (mix_gain);
			e = ioctl(fd, SOUND_MIXER_WRITE_VOLUME, &gain);
		}
	}
	return (fd);
}

int audio_play (int fd, char *buf, int len)
{
	return (write(fd, buf, len));
}

int audio_close (int fd)
{
  return (close (fd));
}

int main (int argc, char *argv[])
{
	int infd = 0;
	int outfd;
	int result, bufbytes, numbytes;
	char *buf;

	utils_init (argv[0]);
	while ((result = getlopt(argc, argv, opts)))
		switch (result) {
			case GLO_UNKNOWN:
				fprintf (stderr, "%s: Unknown option \"%s\".\n",
					me, loptarg);
				exit (1);
			case GLO_NOARG:
				fprintf (stderr, "%s: Missing argument for option \"%s\".\n",
					me, loptarg);
				exit (1);
		}
	if (loptind >= argc)
		usage (NULL);

	bufbytes = buffersize * 1024;
	if (!(buf = malloc(bufbytes)))
		die ("malloc()");
	if ((outfd = audio_open()) < 0)
		die (audevice);
	do {
		if (!strcmp(argv[loptind], "-"))
			infd = 0;
		else if ((infd = open(argv[loptind], O_RDONLY, 0)) < 0)
			die (argv[loptind]);
		do {
			numbytes = read(infd, buf, bufbytes);
			if (numbytes > 0)
				audio_play (outfd, buf, numbytes);
		} while (numbytes > 0);
		if (infd)
			close (infd);
	} while (++loptind < argc);
	audio_close (outfd);
	exit (0);
}

/* EOF */
