/*
 *   toform.c
 *
 *   Support for various output formats for tosha.
 *
 *   Oliver Fromme  <olli@fromme.com>
 *
 *   Copyright (C) 1997,1998,1999
 *        Oliver Fromme.  All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. Neither the name of the author nor the names of any co-contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY OLIVER FROMME AND CONTRIBUTORS ``AS IS'' AND
 *   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *   ARE DISCLAIMED.  IN NO EVENT SHALL OLIVER FROMME OR CONTRIBUTORS BE LIABLE
 *   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *   SUCH DAMAGE.
 *
 *   @(#)$Id: toform.c,v 1.3 1999/01/01 23:31:58 olli Exp $
 */

static const char cvsid[]
    = "@(#)$Id: toform.c,v 1.3 1999/01/01 23:31:58 olli Exp $";

#include <string.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>

#include "utils.h"
#include "global.h"

#include "toform.h"

#define CD_DA_SAMPLERATE  44100

/*   Note: the 1st line in the following list is the default.   */

const formspec format[] = {
	{"pcm",  "pcm",  "raw",  raw_writeheader,  RAW_SWAPBYTES},
	{"mcp",  "mcp",  "war",  war_writeheader,  WAR_SWAPBYTES},
	{"aiff", "iff",  "aiff", aiff_writeheader, AIFF_SWAPBYTES},
	{"wav",  "riff", "wav",  wav_writeheader,  WAV_SWAPBYTES},
	{"au",   "sun",  "au",   au_writeheader,   AU_SWAPBYTES},
	{NULL, NULL, NULL, NULL, 0}
};

void
copy_le2 (unsigned char *buf, unsigned int x)
{
	buf[0] = x & 0xff;
	buf[1] = x >> 8;
}

void
copy_le4 (unsigned char *buf, unsigned int x)
{
	copy_le2 (buf + 0, x & 0xffff);
	copy_le2 (buf + 2, x >> 16);
}

void
copy_be2 (unsigned char *buf, unsigned int x)
{
	buf[0] = x >> 8;
	buf[1] = x & 0xff;
}

void
copy_be4 (unsigned char *buf, unsigned int x)
{
	copy_be2 (buf + 0, x >> 16);
	copy_be2 (buf + 2, x & 0xffff);
}

/*
 *   ===================
 *         RAW/PCM
 *   ===================
 */

int
raw_writeheader (ulong tracksize, int fd)
{
	return 0;
}

/*
 *   ===================
 *         WAR/MCP
 *   ===================
 */

int
war_writeheader (ulong tracksize, int fd)
{
	return 0;
}

/*
 *   ================
 *         AIFF
 *   ================
 */

unsigned char aiffhead[54] =
	"FORM....AIFFCOMM....ccssssbbrrrrrrrrrrSSND....ooookkkk";

#define AIFF_O_FORMSIZE  4	/* [4] track size (ts) in bytes + 46 */
#define AIFF_O_COMMSIZE  16	/* [4] 18 */
#define AIFF_O_NUMCHAN   20	/* [2] number of channels (2) */
#define AIFF_O_NUMSAMP   22	/* [4] number of stereo samples (ts / 4) */
#define AIFF_O_BPS       26	/* [2] bits per channel sample (16) */
#define AIFF_O_RATE      28	/* [10] IEEE sample rate spec */
#define AIFF_O_SSNDSIZE  42	/* [4] track size (ts) in bytes + 8 */
#define AIFF_O_OFFSET    46	/* [4] sound offset (0) */
#define AIFF_O_BSIZE     50	/* [4] sound block size (0) */

const unsigned char ieee44100[10] = {
	0x40, 0x0e,
	CD_DA_SAMPLERATE >> 8, CD_DA_SAMPLERATE & 0xff,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

int
aiff_writeheader (ulong tracksize, int fd)
{
	int headsize = sizeof(aiffhead);

	copy_be4 (aiffhead + AIFF_O_FORMSIZE, tracksize + headsize - 8);
	copy_be4 (aiffhead + AIFF_O_COMMSIZE, 18);
	copy_be2 (aiffhead + AIFF_O_NUMCHAN,  2);
	copy_be4 (aiffhead + AIFF_O_NUMSAMP,  tracksize >> 2);
	copy_be2 (aiffhead + AIFF_O_BPS,      16);
	memcpy   (aiffhead + AIFF_O_RATE,     ieee44100, sizeof(ieee44100));
	copy_be4 (aiffhead + AIFF_O_SSNDSIZE, tracksize + 8);
	copy_be4 (aiffhead + AIFF_O_OFFSET,   0);
	copy_be4 (aiffhead + AIFF_O_BSIZE,    0);
	return write (fd, aiffhead, headsize);
}

/*
 *   ====================
 *         WAV/RIFF
 *   ====================
 */

unsigned char wavhead[44] =
	"RIFF....WAVEfmt ....ffccrrrryyyyaabbdata....";

#define WAV_O_RIFFSIZE  4	/* [4] track size in bytes + 36 */
#define WAV_O_FMTSIZE   16	/* [4] 16 */
#define WAV_O_FORMAT    20	/* [2] 1 (PCM) */
#define WAV_O_NUMCHAN   22	/* [2] number of channels (2) */
#define WAV_O_RATE      24	/* [4] sample rate (44100) */
#define WAV_O_BYTERATE  28	/* [4] bytes per second (44100 * 4) */
#define WAV_O_ALIGN     32	/* [2] sample alignment (4 bytes) */
#define WAV_O_BPS       34	/* [2] bits per channel sample (16) */
#define WAV_O_DATASIZE  40	/* [4] track size in bytes */

int
wav_writeheader (ulong tracksize, int fd)
{
	int headsize = sizeof(wavhead);

	copy_le4 (wavhead + WAV_O_RIFFSIZE, tracksize + headsize - 8);
	copy_le4 (wavhead + WAV_O_FMTSIZE,  16);
	copy_le2 (wavhead + WAV_O_FORMAT,   1);
	copy_le2 (wavhead + WAV_O_NUMCHAN,  2);
	copy_le4 (wavhead + WAV_O_RATE,     CD_DA_SAMPLERATE);
	copy_le4 (wavhead + WAV_O_BYTERATE, CD_DA_SAMPLERATE << 2);
	copy_le2 (wavhead + WAV_O_ALIGN,    4);
	copy_le2 (wavhead + WAV_O_BPS,      16);
	copy_le4 (wavhead + WAV_O_DATASIZE, tracksize);
	return write (fd, wavhead, headsize);
}

/*
 *   ==================
 *         SUN/AU
 *   ==================
 */

unsigned char auhead[28] =
	".sndoooo....ffffrrrrcccctosh";

#define AU_O_OFFSET    4	/* [4] file offset of sound data (28) */
#define AU_O_DATASIZE  8	/* [4] track size in bytes */
#define AU_O_FORMAT    12	/* [4] 3 (linear-16) */
#define AU_O_RATE      16	/* [4] sample rate (44100) */
#define AU_O_NUMCHAN   20	/* [4] number of channels (2) */

int
au_writeheader (ulong tracksize, int fd)
{
	int headsize = sizeof(auhead);

	copy_be4 (auhead + AU_O_OFFSET,   headsize);
	copy_be4 (auhead + AU_O_DATASIZE, tracksize);
	copy_be4 (auhead + AU_O_FORMAT,   3);
	copy_be4 (auhead + AU_O_RATE,     CD_DA_SAMPLERATE);
	copy_be4 (auhead + AU_O_NUMCHAN,  2);
	return write (fd, auhead, headsize);
}

/* EOF */
