/* 
 * ratSender.c --
 *
 *	This is the subprocess which handles the actual sending of messages.
 *
 *
 * TkRat software and its included text is Copyright 1996-1999 by
 * Martin Forssn
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "ratFolder.h"
#include "ratPGP.h"

/*
 * This is a list of outstanding commands
 */
typedef struct CmdList {
    char *cmd;
    struct CmdList *next;
} CmdList;
static CmdList *cmdList = NULL;

static SMTPChannel smtpChannel = NULL;

/*
 * Create bodypart s procedure
 */
static BODY *RatCreateBody(Tcl_Interp *interp, char *handler, ENVELOPE *env,
	int *errorFlag, Tcl_DString *files);
static int RatSenderSend(Tcl_Interp *interp, char *prefix,
	Tcl_DString *usedVariables, Tcl_DString *files, int *hardError);
static void RatSenderStanddown(Tcl_Interp *interp);
static int RatParseParameter(Tcl_Interp *interp, const char *src,
	PARAMETER **dstPtrPtr);


/*
 *----------------------------------------------------------------------
 *
 * RatSender --
 *
 *	This routine runs the sending process.
 *
 * Results:
 *	None, this routine never returns
 *
 * Side effects:
 *      Messages may be sent
 *
 *
 *----------------------------------------------------------------------
 */

void
RatSender(Tcl_Interp *interp)
{
    Tcl_DString usedVariables, files, result;
    char *buf, **argv, **sendlistArgv;
    CmdList *cmdPtr;
    int sendlistArgc, argc, i, s, buflen, hardError = 0;

    /*
     * Clear cached passwords
     */
    ClearPGPPass(NULL);

    Tcl_DStringInit(&usedVariables);
    Tcl_DStringInit(&files);
    Tcl_DStringInit(&result);
    buflen = 1024;
    buf = (char*)ckalloc(buflen);

    while (1) {
	if (cmdList) {
	    cmdPtr = cmdList;
	    RatStrNCpy(buf, cmdList->cmd, sizeof(buf));
	    cmdList = cmdList->next;
	    ckfree(cmdPtr->cmd);
	    ckfree(cmdPtr);
	} else {
	    i = 0;
	    while (buf[buflen-2] = '\0',
		    fgets(buf+i, buflen-i, stdin)
		    && buflen-i-1 == strlen(buf+i)
		    && buf[buflen-2] != '\n') {
		i = buflen-1;
		buflen += 1024;
		buf = ckrealloc(buf, buflen);
	    }
	    if (feof(stdin)) {
		exit(0);
	    }
	}
	
	if (!strncmp(buf, "SEND", 4)) {
	    (void)Tcl_SplitList(interp, buf, &sendlistArgc, &sendlistArgv);
	    for (s=1; s<sendlistArgc && !hardError; s++) {
		(void)Tcl_SplitList(interp, sendlistArgv[s], &argc, &argv);
		if (TCL_OK == RatSenderSend(interp, argv[1], &usedVariables,
			&files, &hardError)){
		    Tcl_DStringAppendElement(&result, "SENT");
		    Tcl_DStringAppendElement(&result, argv[0]);
		    ckfree(argv);
		    Tcl_SplitList(interp, Tcl_DStringValue(&files),&argc,&argv);
		    for (i=0; i<argc; i++) {
			(void)unlink(argv[i]);
		    }
		} else {
		    Tcl_DStringAppendElement(&result, "FAILED");
		    Tcl_DStringAppendElement(&result, argv[0]);
		    Tcl_DStringAppendElement(&result, argv[1]);
		    Tcl_DStringAppendElement(&result,
			    Tcl_GetStringResult(interp));
		    sprintf(buf, "%d", hardError);
		    Tcl_DStringAppendElement(&result, buf);
		}
		ckfree(argv);
		(void)Tcl_SplitList(interp, Tcl_DStringValue(&usedVariables),
				    &argc, &argv);
		for (i=0; i<argc; i++) {
		    (void)Tcl_UnsetVar(interp, argv[i], TCL_GLOBAL_ONLY);
		}
		ckfree(argv);
		for (i=Tcl_DStringLength(&result)-1; i>=0; i--) {
		    if ('\n' == Tcl_DStringValue(&result)[i]) {
			Tcl_DStringValue(&result)[i] = ' ';
		    }
		}
		fwrite(Tcl_DStringValue(&result), Tcl_DStringLength(&result)+1,
		       1, stdout);
		fflush(stdout);
		Tcl_DStringSetLength(&result, 0);
		Tcl_DStringSetLength(&usedVariables, 0);
		Tcl_DStringSetLength(&files, 0);
	    }
	    ckfree(sendlistArgv);
	    RatSenderStanddown(interp);
	} else if (!strncmp(buf, "RSET", 4)) {
	    hardError = 0;
	} else {
	    exit(0);
	}
    }
    /* Notreached */
    exit(0);
}


/*
 *----------------------------------------------------------------------
 *
 * RatSenderSend --
 *
 *	Send a specified message
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *      A message is sent.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatSenderSend(Tcl_Interp *interp, char *prefix, Tcl_DString *usedVariablesPtr,
	Tcl_DString *filesPtr, int *hardError)
{
    int listArgc, i, errorFlag = 0, requestDSN, useFrom, verbose;
    char *tmp, buf[1024], **listArgv, *handler, *saveTo, *header = NULL;
    Tcl_Obj *oPtr;
    ENVELOPE *env;
    BODY *body;

    /*
     * Extract the message
     */
    if (TCL_OK != RatHoldExtract(interp, prefix, usedVariablesPtr, filesPtr)) {
	return TCL_ERROR;
    }
    handler = cpystr(Tcl_GetStringResult(interp));

    /*
     * Construct the headers
     */
    if (!(tmp = Tcl_GetVar2(interp, handler, "request_dsn", TCL_GLOBAL_ONLY))
	    || TCL_OK != Tcl_GetBoolean(interp, tmp, &requestDSN)) {
	requestDSN = 0;
    }
    tmp = Tcl_GetVar2(interp, handler, "to", TCL_GLOBAL_ONLY);
    env = mail_newenvelope ();
    rfc822_parse_adrlist(&env->to, tmp, currentHost);
    RatEncodeAddresses(interp, env->to);
    env->remail = cpystr(Tcl_GetVar2(interp,handler,"remail",TCL_GLOBAL_ONLY));
    if ((tmp = Tcl_GetVar2(interp, handler, "return_path", TCL_GLOBAL_ONLY)) 
	    && !RatIsEmpty(tmp)) {
	rfc822_parse_adrlist(&env->return_path, tmp, currentHost);
    } else {
	env->return_path = mail_newaddr();
	env->return_path->personal =
		currentPersonalName ? cpystr(currentPersonalName) : NULL;
	env->return_path->mailbox = cpystr(currentMailboxName);
	env->return_path->host = cpystr(currentHost);
    }
    RatEncodeAddresses(interp, env->return_path);
    if (NULL==(env->date=Tcl_GetVar2(interp,handler,"date",TCL_GLOBAL_ONLY))) {
	rfc822_date(buf);
	env->date = buf;
    }
    env->date = cpystr(env->date);
    if (TCL_OK != Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option",
	    "use_from", TCL_GLOBAL_ONLY), &useFrom)) {
	useFrom = 0;
    }
    if (useFrom && (tmp = Tcl_GetVar2(interp, handler, "from", TCL_GLOBAL_ONLY))
	    && !RatIsEmpty(tmp)) {
	rfc822_parse_adrlist(&env->from, tmp, currentHost);
    }
    if (env->from) {
	ADDRESS *adrPtr;
	int cs;

	for (adrPtr = env->from; adrPtr; adrPtr = adrPtr->next) {
	    if (RatAddressIsMe(interp, adrPtr, 0)) {
		break;
	    }
	}
	oPtr = Tcl_GetVar2Ex(interp, "option","create_sender",TCL_GLOBAL_ONLY);
	Tcl_GetBooleanFromObj(interp, oPtr, &cs);
	if (!adrPtr && cs) {
	    env->sender = mail_newaddr();
	    env->sender->personal =
		    currentPersonalName ? cpystr(currentPersonalName) : NULL;
	    env->sender->mailbox = cpystr(currentMailboxName);
	    env->sender->host = cpystr(currentHost);
	    RatEncodeAddresses(interp, env->sender);
	} else {
	    env->sender = NULL;
	}
    } else {
	env->from = mail_newaddr();
	env->from->personal =
		currentPersonalName ? cpystr(currentPersonalName) : NULL;
	env->from->mailbox = cpystr(currentMailboxName);
	env->from->host = cpystr(currentHost);
    }
    RatEncodeAddresses(interp, env->from);
    if ((tmp = Tcl_GetVar2(interp, handler, "reply_to", TCL_GLOBAL_ONLY))
	    && !RatIsEmpty(tmp)) {
	rfc822_parse_adrlist(&env->reply_to, tmp, currentHost);
	RatEncodeAddresses(interp, env->reply_to);
    }
    oPtr = Tcl_GetVar2Ex(interp, handler, "subject", TCL_GLOBAL_ONLY);
    if (oPtr) {
	env->subject = (char*)RatEncodeHeaderLine(interp, oPtr, 9);
    }
    if ((tmp = Tcl_GetVar2(interp, handler, "cc", TCL_GLOBAL_ONLY))
	    && !RatIsEmpty(tmp)) {
	rfc822_parse_adrlist(&env->cc, tmp, currentHost);
	RatEncodeAddresses(interp, env->cc);
    }
    if ((tmp = Tcl_GetVar2(interp, handler, "bcc", TCL_GLOBAL_ONLY))
	    && !RatIsEmpty(tmp)) {
	rfc822_parse_adrlist(&env->bcc, tmp, currentHost);
	RatEncodeAddresses(interp, env->bcc);
    }
    env->in_reply_to = cpystr(Tcl_GetVar2(interp, handler, "in_reply_to",
	    TCL_GLOBAL_ONLY));
    env->message_id = cpystr(Tcl_GetVar2(interp, handler, "message_id",
	    TCL_GLOBAL_ONLY));
    env->newsgroups = NULL;

    /*
     * Construct the body
     */
    if ((tmp = Tcl_GetVar2(interp, handler, "body", TCL_GLOBAL_ONLY))) {
	body = RatCreateBody(interp, tmp, env, &errorFlag, filesPtr);
    } else {
	body = mail_newbody ();
    }
    if (errorFlag) {
	goto error;
    }

    /*
     * Send the message
     */
    header = (char*)ckalloc(RatHeaderSize(env, body));
    Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "smtp_verbose",
	    TCL_GLOBAL_ONLY), &verbose);
    if (1 == verbose) {
	RatLogF(interp, RAT_PARSE, "sending_message", RATLOG_EXPLICIT);
    }
    tmp = Tcl_GetVar2(interp, "option", "sendprot", TCL_GLOBAL_ONLY);
    if (tmp && !strcmp(tmp, "smtp")) {
	int reuseChannel;

	Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option", "smtp_reuse",
		TCL_GLOBAL_ONLY), &reuseChannel);
	listArgc = 0;
	if ((tmp = Tcl_GetVar2(interp,"option","smtp_hosts",TCL_GLOBAL_ONLY))) {
	    Tcl_SplitList(interp, tmp, &listArgc, &listArgv);
	}
	if (0 == listArgc) {
	    Tcl_SetResult(interp, 
			  "Configuration error; no valid SMTP hosts given",
		          TCL_STATIC);
	    *hardError = 1;
	    goto error;
	}
	for (i=0; !smtpChannel && i<listArgc; i++) {
	    smtpChannel = RatSMTPOpen(interp, listArgv[i], verbose);
	}
	ckfree(listArgv);
	if (!smtpChannel) {
	    Tcl_SetResult(interp, "No valid SMTP hosts", TCL_STATIC);
	    *hardError = 1;
	    goto error;
	}
	if (TCL_OK != RatSMTPSend(interp, smtpChannel, env, body, requestDSN,
		verbose)) {
	    Tcl_DString ds;
	    Tcl_DStringInit(&ds);
	    Tcl_DStringGetResult(interp, &ds);
	    RatSMTPClose(interp, smtpChannel, verbose);
	    smtpChannel = NULL;
	    Tcl_DStringResult(interp, &ds);
	    goto error;
	}
	if (!reuseChannel) {
	    RatSMTPClose(interp, smtpChannel, verbose);
	    smtpChannel = NULL;
	}

    } else if (tmp && !strcmp(tmp, "prog")) {
	Tcl_Channel channel;
	Tcl_DString ds;
	ADDRESS *adrPtr;

	Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option", "sendprog_8bit",
		TCL_GLOBAL_ONLY), &i);
	if (NULL == (tmp = Tcl_GetVar2(interp,"option", "sendprog",
		     TCL_GLOBAL_ONLY))) {
	    Tcl_SetResult(interp, "Invalid send program", TCL_STATIC);
	    *hardError = 1;
	    goto error;
	}
	Tcl_DStringInit(&ds);
	Tcl_DStringAppend(&ds, tmp, -1);
	for (adrPtr = env->to; adrPtr; adrPtr = adrPtr->next) {
	    if (RatAddressSize(adrPtr, 0) > sizeof(buf)-1) {
		Tcl_SetResult(interp, "Ridiculously long address", TCL_STATIC);
		*hardError = 1;
		goto error;
	    } else {
		buf[0] = '\0';
		rfc822_address(buf, adrPtr);
		Tcl_DStringAppendElement(&ds, buf);
	    }
	}
	for (adrPtr = env->cc; adrPtr; adrPtr = adrPtr->next) {
	    if (RatAddressSize(adrPtr, 0) > sizeof(buf)-1) {
		Tcl_SetResult(interp, "Ridiculously long address", TCL_STATIC);
		*hardError = 1;
		goto error;
	    } else {
		buf[0] = '\0';
		rfc822_address(buf, adrPtr);
		Tcl_DStringAppendElement(&ds, buf);
	    }
	}
	for (adrPtr = env->bcc; adrPtr; adrPtr = adrPtr->next) {
	    if (RatAddressSize(adrPtr, 0) > sizeof(buf)-1) {
		Tcl_SetResult(interp, "Ridiculously long address", TCL_STATIC);
		*hardError = 1;
		goto error;
	    } else {
		buf[0] = '\0';
		rfc822_address(buf, adrPtr);
		Tcl_DStringAppendElement(&ds, buf);
	    }
	}

	Tcl_ResetResult(interp);
	if (TCL_OK != Tcl_SplitList(interp, Tcl_DStringValue(&ds),
		    &listArgc, &listArgv)
		|| (NULL == (channel = Tcl_OpenCommandChannel(interp, listArgc,
		    listArgv, TCL_STDIN|TCL_STDOUT|TCL_STDERR)))) {
	    Tcl_DStringSetLength(&ds, 0);
	    Tcl_DStringAppend(&ds, "Failed to run send program: ", -1);
	    Tcl_DStringAppend(&ds, Tcl_GetStringResult(interp), -1);
	    Tcl_DStringResult(interp, &ds);
	    goto error;
	} else {
	    Tcl_DStringFree(&ds);
	    ckfree(listArgv);
	    rfc822_output(header, env, body, RatTclPutsSMTP, channel, i);
	    Tcl_Close(interp, channel);
	}
    } else {
	snprintf(buf, sizeof(buf), "Invalid send protocol '%s'",
		tmp ? tmp : "<NULL>");
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	*hardError = 1;
	goto error;
    }
    if (verbose) {
	RatLog(interp, RAT_PARSE, "", RATLOG_EXPLICIT);
    }

    saveTo = Tcl_GetVar2(interp, handler, "save_to", TCL_GLOBAL_ONLY);
    if (saveTo && *saveTo) {
	MESSAGECACHE elt;
	Tcl_DString saveArg, ds;
	Tcl_Channel channel;
	char saveFile[1024];
	struct tm *tmPtr;
	time_t now;
	int perm;

	Tcl_DStringInit(&saveArg);
	Tcl_DStringInit(&ds);
	tmp = Tcl_GetVar2(interp, "option", "tmp", TCL_GLOBAL_ONLY);
	RatGenId(NULL, interp, 0, NULL);
	snprintf(saveFile, sizeof(saveFile), "%s/rat.%s",
		tmp, Tcl_GetStringResult(interp));

	Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "permissions",
		TCL_GLOBAL_ONLY), &perm);
	if (NULL == (channel=Tcl_OpenFileChannel(interp,saveFile,"a",perm))){
	    Tcl_AppendResult(interp, "Failed to save copy of message: ",
		    Tcl_PosixError(interp), (char*) NULL);
	    goto error;
	}
	rfc822_output(header, env, body, RatTclPuts, channel, 1);
	Tcl_Close(interp, channel);

	Tcl_DStringAppendElement(&saveArg, saveFile);
	Tcl_DStringAppendElement(&saveArg, saveTo);
	Tcl_DStringSetLength(&ds, RatAddressSize(env->to, 1));
	Tcl_DStringValue(&ds)[0] = '\0';
	rfc822_write_address(Tcl_DStringValue(&ds), env->to);
	Tcl_DStringSetLength(&ds, strlen(Tcl_DStringValue(&ds)));
	Tcl_DStringAppendElement(&saveArg,
		RatDecodeHeader(interp, Tcl_DStringValue(&ds), 1));
	if (env->from) {
	    Tcl_DStringSetLength(&ds, RatAddressSize(env->from, 1));
	    Tcl_DStringValue(&ds)[0] = '\0';
	    rfc822_write_address(Tcl_DStringValue(&ds), env->from);
	    Tcl_DStringSetLength(&ds, strlen(Tcl_DStringValue(&ds)));
	} else {
	    Tcl_DStringSetLength(&ds,
		    strlen(currentMailboxName) + strlen(currentHost)+2);
	    sprintf(Tcl_DStringValue(&ds), "%s@%s", currentMailboxName,
		    currentHost);
	}
	Tcl_DStringAppendElement(&saveArg,
		RatDecodeHeader(interp, Tcl_DStringValue(&ds), 1));
	Tcl_DStringSetLength(&ds, RatAddressSize(env->cc, 1));
	Tcl_DStringValue(&ds)[0] = '\0';
	rfc822_write_address(Tcl_DStringValue(&ds), env->cc);
	Tcl_DStringSetLength(&ds, strlen(Tcl_DStringValue(&ds)));
	Tcl_DStringAppendElement(&saveArg,
		RatDecodeHeader(interp, Tcl_DStringValue(&ds), 1));
	Tcl_DStringAppendElement(&saveArg, env->message_id);
	Tcl_DStringAppendElement(&saveArg,
		(env->in_reply_to ? env->in_reply_to : env->references));
	Tcl_DStringAppendElement(&saveArg,
		RatDecodeHeader(interp, env->subject, 0));
	Tcl_DStringAppendElement(&saveArg, RAT_SEEN_STR);
	now = time(NULL);
	tmPtr = gmtime(&now);
	elt.day = tmPtr->tm_mday;
	elt.month = tmPtr->tm_mon+1;
	elt.year = tmPtr->tm_year+1900-BASEYEAR;
	elt.hours = tmPtr->tm_hour;
	elt.minutes = tmPtr->tm_min;
	elt.seconds = tmPtr->tm_sec;
	elt.zoccident = 0;
	elt.zhours = 0;
	elt.zminutes = 0;
	Tcl_DStringAppendElement(&saveArg, mail_date(buf, &elt));
	fprintf(stdout, "SAVE %s", Tcl_DStringValue(&saveArg));
	fputc('\0', stdout);
	fflush(stdout);
	Tcl_DStringFree(&saveArg);
    }

    mail_free_envelope(&env);
    mail_free_body(&body);
    ckfree(header);
    return TCL_OK;

error:
    if (verbose) {
	RatLog(interp, RAT_PARSE, "", RATLOG_EXPLICIT);
    }
    if (header) {
	ckfree(header);
    }
    mail_free_envelope(&env);
    mail_free_body(&body);
    return TCL_ERROR;
}


/*
 *----------------------------------------------------------------------
 *
 * RatSenderStanddown --
 *
 *	Closes the open SMTP channel (if open)
 *
 * Results:
 *	None
 *
 * Side effects:
 *      smtpChannel is closed.
 *
 *
 *----------------------------------------------------------------------
 */

static void
RatSenderStanddown(Tcl_Interp *interp)
{
    int verbose;

    if (smtpChannel) {
	Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "smtp_verbose",
		TCL_GLOBAL_ONLY), &verbose);
	RatSMTPClose(interp, smtpChannel, verbose);
	smtpChannel = NULL;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * RatCreateBody --
 *
 *	See ../doc/interface
 *
 * Results:
 *      The return value is normally TCL_OK and the result can be found
 *      in the result area. If something goes wrong TCL_ERROR is returned
 *      and an error message will be left in the result area.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */
static BODY*
RatCreateBody(Tcl_Interp *interp, char *handler, ENVELOPE *env,
	int *errorFlag, Tcl_DString *filesPtr)
{
    BODY *body = mail_newbody();
    char *type;
    char *encoding;
    char *parameter;
    char buf[1024];
    char *filename = NULL;
    char *value;
    char *children;
    int pgp_sign, pgp_encrypt;
    Tcl_Obj *oPtr;

    if (NULL == (type = Tcl_GetVar2(interp, handler, "type",
	    TCL_GLOBAL_ONLY))) {
	Tcl_SetResult(interp, "Internal error, no bhandler(type)", TCL_STATIC);
	(*errorFlag)++;
	return body;
    }
    if (!strcasecmp(type, "text"))		body->type = TYPETEXT;
    else if (!strcasecmp(type, "multipart"))	body->type = TYPEMULTIPART;
    else if (!strcasecmp(type, "message"))	body->type = TYPEMESSAGE;
    else if (!strcasecmp(type, "application"))	body->type = TYPEAPPLICATION;
    else if (!strcasecmp(type, "audio"))	body->type = TYPEAUDIO;
    else if (!strcasecmp(type, "image"))	body->type = TYPEIMAGE;
    else if (!strcasecmp(type, "video"))	body->type = TYPEVIDEO;
    else					body->type = TYPEOTHER;
    if (NULL == (encoding = Tcl_GetVar2(interp, handler, "encoding",
	    TCL_GLOBAL_ONLY))) {
	body->encoding = ENC7BIT;
    } else {
	if (!strcasecmp(encoding, "7bit"))	    body->encoding = ENC7BIT;
	else if (!strcasecmp(encoding, "8bit"))	    body->encoding = ENC8BIT;
	else if (!strcasecmp(encoding, "binary"))   body->encoding = ENCBINARY;
	else if (!strcasecmp(encoding, "base64"))   body->encoding = ENCBASE64;
	else if (!strcasecmp(encoding, "quoted-printable"))
					body->encoding = ENCQUOTEDPRINTABLE;
	else {
	    snprintf(buf, sizeof(buf), "Unkown encoding %s\n", encoding);
	    Tcl_SetResult(interp, buf, TCL_VOLATILE);
	    (*errorFlag)++;
	}
    }
    if (NULL == (body->subtype = Tcl_GetVar2(interp, handler, "subtype",
	    TCL_GLOBAL_ONLY))) {
	Tcl_SetResult(interp,"Internal error, no bhandler(subtype)",TCL_STATIC);
	(*errorFlag)++;
	return body;
    }
    body->subtype = cpystr(body->subtype);
    if ((parameter = Tcl_GetVar2(interp,handler,"parameter",TCL_GLOBAL_ONLY))) {
	(*errorFlag) += RatParseParameter(interp, parameter, &body->parameter);
    }
    if ((parameter =
	    Tcl_GetVar2(interp, handler, "disp_parm", TCL_GLOBAL_ONLY))) {
	(*errorFlag) += RatParseParameter(interp, parameter,
					  &body->disposition.parameter);
    }
    body->disposition.type = cpystr(Tcl_GetVar2(interp, handler,
	    "disp_type", TCL_GLOBAL_ONLY));
    if (body->disposition.type && !strlen(body->disposition.type)) {
	body->disposition.type = NULL;
    }
    body->id = cpystr(Tcl_GetVar2(interp, handler, "id", TCL_GLOBAL_ONLY));
    if (body->id && !strlen(body->id)) {
	body->id = NULL;
    }
    oPtr = Tcl_GetVar2Ex(interp, handler, "description", TCL_GLOBAL_ONLY);
    if (oPtr) {
	body->description = (char*)RatEncodeHeaderLine(interp, oPtr, 13);
    }
    if (body->description && !strlen(body->description)) {
	body->description = NULL;
    }

    if (TYPEMULTIPART == body->type && NULL != (children =
	    Tcl_GetVar2(interp, handler, "children", TCL_GLOBAL_ONLY))) {
	PART **partPtrPtr = &body->nested.part;
	int childrenArgc, i;
	char **childrenArgv;
	Tcl_SplitList(interp, children, &childrenArgc, &childrenArgv);
	for (i=0; i<childrenArgc; i++) {
	    *partPtrPtr = mail_newbody_part();
	    (*partPtrPtr)->body = *RatCreateBody(interp, childrenArgv[i],
		    env, errorFlag, filesPtr);
	    partPtrPtr = &(*partPtrPtr)->next;
	}
    } else if (TYPEMESSAGE == body->type) {
	struct stat sbuf;
	char *message;
	int fd;

	if (NULL == (filename = Tcl_GetVar2(interp, handler,"filename",
		TCL_GLOBAL_ONLY))) {
	    Tcl_SetResult(interp, "Internal error, no bhandler(filename)",
			  TCL_STATIC);
	    (*errorFlag)++;
	    return body;
	}
	if (-1 == (fd = open(filename, O_RDONLY))) {
	    snprintf(buf, sizeof(buf), "Failed to open file \"%s\": %s",
		    filename, Tcl_PosixError(interp));
	    Tcl_SetResult(interp, buf, TCL_VOLATILE);
	    (*errorFlag)++;
	    return body;
	}
	fstat(fd, &sbuf);
	message = (char*)ckalloc(sbuf.st_size+1);
	read(fd, message, sbuf.st_size);
	message[sbuf.st_size] = '\0';
	(void)close(fd);

	body->nested.msg = RatParseMsg(interp, message);
	body->contents.text.data = (unsigned char*)message;
	body->contents.text.size = sbuf.st_size;

    } else {
	FILE *fp;
	struct stat statbuf;
	int len = 0, allocated, ci;
	char *data;

	if (NULL == (filename = Tcl_GetVar2(interp, handler,"filename",
		TCL_GLOBAL_ONLY))) {
	    Tcl_SetResult(interp, "Internal error, no bhandler(filename)",
			  TCL_STATIC);
	    (*errorFlag)++;
	    return body;
	}
	if (NULL == (fp = fopen(filename, "r"))) {
	    snprintf(buf, sizeof(buf), "Failed to open file \"%s\": %s",
		    filename, Tcl_PosixError(interp));
	    Tcl_SetResult(interp, buf, TCL_VOLATILE);
	    (*errorFlag)++;
	    return body;
	}
	fstat(fileno(fp), &statbuf);
	allocated = (statbuf.st_size*105)/100;
	data = (char*)ckalloc(allocated);
	if (ENCBINARY != body->encoding) {
	    while (EOF != (ci = getc(fp))) {
		if (len >= allocated-2) {
		    allocated += 1024;
		    data = (char*)ckrealloc(data, allocated);
		}
		if (ci == '\n') {
		    data[len++] = '\r';
		}
		data[len++] = ci;
	    }
	} else {
	    fread(data, statbuf.st_size, 1, fp);
	    len = statbuf.st_size;
	}
	data[len] = '\0';
	fclose(fp);
	body->contents.text.data = (unsigned char*)data;
	body->size.bytes = body->contents.text.size = len;

	if (TYPEMULTIPART == body->type) {
	    body->type = TYPEAPPLICATION;
	    ckfree(body->subtype);
	    body->subtype = cpystr("octet-stream");
	}
    }
    if (filename) {
	int removeFile = 0;

	if ((value = Tcl_GetVar2(interp,handler,"removeFile",TCL_GLOBAL_ONLY))){
	    Tcl_GetBoolean(interp, value, &removeFile);
	}
	if (removeFile && filesPtr) {
	    Tcl_DStringAppendElement(filesPtr, filename);
	}
    }

    /*
     * Do PGP signing and/or encryption
     */
    if (!(value = Tcl_GetVar2(interp, handler, "pgp_sign", TCL_GLOBAL_ONLY)) ||
         TCL_OK != Tcl_GetBoolean(interp, value, &pgp_sign)) {
	pgp_sign = 0;
    }
    if (!(value = Tcl_GetVar2(interp, handler,"pgp_encrypt",TCL_GLOBAL_ONLY)) ||
         TCL_OK != Tcl_GetBoolean(interp, value, &pgp_encrypt)) {
	pgp_encrypt = 0;
    }
    if (pgp_sign || pgp_encrypt) {
	if (pgp_encrypt) {
	    body = RatPGPEncrypt(interp, env, body, pgp_sign);
	} else {
	    body = RatPGPSign(interp, env, body);
	}
	if (!body) {
	    Tcl_SetResult(interp, "Failed to do PGP operation", TCL_STATIC);
	    (*errorFlag)++;
	    return body;
	}
    }

    return body;
}

/*
 *----------------------------------------------------------------------
 *
 * RatParseParameter --
 *
 *	Splits a parameter list into elements
 *
 * Results:
 *      number of encountered errors
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatParseParameter(Tcl_Interp *interp, const char *src, PARAMETER **dstPtrPtr)
{
    int parmListArgc, parmArgc, i, errcount = 0;
    char **parmListArgv, **parmArgv;
    char buf[1024];

    Tcl_SplitList(interp, src, &parmListArgc, &parmListArgv);
    for (i=0; i<parmListArgc; i++) {
	if (TCL_ERROR == Tcl_SplitList(interp, parmListArgv[i], &parmArgc,
		&parmArgv) || 2 != parmArgc) {
	    snprintf(buf, sizeof(buf), "Illegal parameter: %s",
		    parmListArgv[i]);
	    Tcl_SetResult(interp, buf, TCL_VOLATILE);
	    errcount++;
	} else {
	    *dstPtrPtr = mail_newbody_parameter();
	    (*dstPtrPtr)->attribute = cpystr(parmArgv[0]);
	    (*dstPtrPtr)->value = cpystr(parmArgv[1]);
	    dstPtrPtr = &(*dstPtrPtr)->next;
	}
	ckfree(parmArgv);
    }
    ckfree(parmListArgv);
    return errcount;
}

/*
 *----------------------------------------------------------------------
 *
 * RatSendPGPCommand --
 *
 *	Sends a PGP command to the master and then waits for a reply.
 *
 * Results:
 *      A pointer to a static buffer containing the command.
 *
 * Side effects:
 *      Other commands that arrive while we are waiting are placed in cmdList
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatSendPGPCommand(char *cmd)
{
    static char buf[1024];
    CmdList **cmdPtrPtr;

    fwrite(cmd, strlen(cmd)+1, 1, stdout);
    fflush(stdout);
    for (cmdPtrPtr = &cmdList; *cmdPtrPtr; cmdPtrPtr = &(*cmdPtrPtr)->next);
    while(1) {
	fgets(buf, sizeof(buf), stdin);
	if (feof(stdin)) {
	    exit(0);
	}
	buf[strlen(buf)-1] = '\0';
	if (strncmp("PGP ", buf, 4)) {
	    *cmdPtrPtr = (CmdList*)ckalloc(sizeof(CmdList));
	    (*cmdPtrPtr)->cmd = cpystr(buf);
	    (*cmdPtrPtr)->next = NULL;
	    cmdPtrPtr = &(*cmdPtrPtr)->next;
	} else {
	    return buf+4;
	}
    }
}
