/*
 * ratFolder.c --
 *
 *      This file contains basic support code for the folder commands. Each
 *      folder type is created using an unique command. This command returns
 *      a folder handler, which when invoked calls the RatFolderCmd()
 *      procedure with a pointer to a RatFolderInfo structure as clientData.
 *
 * TkRat software and its included text is Copyright 1996-1999 by
 * Martin Forssn
 *
 * The full text of the legal notices is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "ratFolder.h"

/*
 * This structure is used to hold the data while sorting
 */
typedef struct SortData {
    char *msgid;
    char *in_reply_to;
    char *subject;
    char *sender;
    time_t date;
    struct SortData *nextPtr;
    int prev, next, child, parent;
    Tcl_Obj *tPtr;
} SortData;

/* 
 * Global list of folders
 */
RatFolderInfo *ratFolderList = NULL;

/*
 * The number of folders opened. This is used when making
 * the folder entities.
 */
static int numFolders = 0;

/*
 * Sort order names
 */
struct {
    SortOrder order;
    int reverse;
    char *name;
} sortNames[] = {
    {SORT_THREADED,	0, "threaded"},
    {SORT_SUBJDATE,	0, "subject"},
    {SORT_SENDERDATE,	0, "sender"},
    {SORT_SUBJECT,	0, "subjectonly"},
    {SORT_SENDER,	0, "senderonly"},
    {SORT_DATE,		0, "date"},
    {SORT_NONE,		0, "folder"},
    {SORT_NONE,		1, "reverseFolder"},
    {SORT_DATE,		1, "reverseDate"},
    {0,			0, NULL}
};

/*
 * Global variables used to controll the sorting functions
 */
static SortData *baseSortDataPtr;

static Tcl_ObjCmdProc RatFolderOpen;
static int RatFolderCmd(ClientData clientData,
	Tcl_Interp *interp, int argc, char *argv[]);
static void RatFolderSort(Tcl_Interp *interp, RatFolderInfo *infoPtr);
static int RatFolderSortCompareDate(const void *arg1, const void *arg2);
static int RatFolderSortCompareSubject(const void *arg1, const void *arg2);
static int RatFolderSortCompareSender(const void *arg1, const void *arg2);
static int IsChild(SortData *dataPtr, int child, int parent);
static int RatFolderSortLinearize(int *p, int n, SortData *dataPtr, int first,
	int depth);
static int RatUpdateFolder(Tcl_Interp *interp, RatFolderInfo *infoPtr,
	RatUpdateType mode);
static Tcl_TimerProc RatFolderUpdateTime;


/*
 *----------------------------------------------------------------------
 *
 * RatFolderInit --
 *
 *      Initializes the folder commands.
 *
 * Results:
 *      The return value is normally TCL_OK; if something goes wrong
 *	TCL_ERROR is returned and an error message will be left in
 *	the result area.
 *
 * Side effects:
 *	The folder creation commands are created in interp. And the
 *	Std folder is initialized.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatFolderInit(Tcl_Interp *interp)
{
    RatInitMessages();
    if (TCL_OK != RatStdFolderInit(interp)) {
	return TCL_ERROR;
    }
    if (TCL_OK != RatDbFolderInit(interp)) {
	return TCL_ERROR;
    }
    if (TCL_OK != RatDisFolderInit(interp)) {
	return TCL_ERROR;
    }
    Tcl_CreateObjCommand(interp, "RatOpenFolder", RatFolderOpen,
	    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "RatParseExp", RatParseExpCmd, NULL, NULL);
    Tcl_CreateCommand(interp, "RatGetExp", RatGetExpCmd, NULL, NULL);
    Tcl_CreateCommand(interp, "RatFreeExp", RatFreeExpCmd, NULL, NULL);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatGetOpenFolder --
 *
 *      Search the list of open folders for a folder matching the given
 *	definition.
 *
 * Results:
 *	If a match is found then that folders reference count is incremented
 *	and the infoPtr is returned. Otherwise NULL is returned.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

RatFolderInfo*
RatGetOpenFolder(char *def)
{
    RatFolderInfo *infoPtr;

    for (infoPtr = ratFolderList;
	    infoPtr && strcmp(infoPtr->definition, def);
	    infoPtr = infoPtr->nextPtr);
    if (infoPtr) {
	infoPtr->refCount++;
    }
    return infoPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * RatFolderOpen --
 *
 *      See the INTERFACE specification
 *
 * Results:
 *      The return value is normally TCL_OK and a foilder handle is left
 *	in the result area; if something goes wrong TCL_ERROR is returned
 *	and an error message will be left in the result area.
 *
 * Side effects:
 *	The folder creation commands are created in interp.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatFolderOpen(ClientData clientData, Tcl_Interp *interp, int objc,
	Tcl_Obj *CONST objv[])
{
    RatFolderInfo *infoPtr;
    int i, listc, lobjc;
    char *sortName = NULL, *def;
    Tcl_Obj **lobjv, *idobjv[16], *oPtr;

    if (objc < 3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " prot flags [args]\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Check open folders
     */
    idobjv[0] = objv[1];
    for (i=3; i<objc; i++) {
	idobjv[i-2] = objv[i];
    }
    oPtr = Tcl_NewListObj(i-2, idobjv);
    if ((infoPtr = RatGetOpenFolder(Tcl_GetString(oPtr)))) {
	Tcl_SetResult(interp, infoPtr->cmdName, TCL_VOLATILE);
	Tcl_DecrRefCount(oPtr);
	return TCL_OK;
    }
    def = cpystr(Tcl_GetString(oPtr));
    Tcl_DecrRefCount(oPtr);

    if (!strcmp(Tcl_GetString(objv[1]), "std")) {
	infoPtr = RatStdFolderCreate(interp, objc, objv);
    } else if (!strcmp(Tcl_GetString(objv[1]), "dbase")) {
	infoPtr = RatDbFolderCreate(interp, objc, objv);
    } else if (!strcmp(Tcl_GetString(objv[1]), "dis")) {
	infoPtr = RatDisFolderCreate(interp, objc, objv);
    } else {
	Tcl_SetResult(interp,
		"Wrong protocol. Only 'std', 'dbase' and 'dis' are supported",
		TCL_STATIC);
	return TCL_ERROR;
    }
    if (NULL == infoPtr) {
	return TCL_ERROR;
    }
    Tcl_ListObjLength(interp, objv[2], &listc);
    for (i=0; i < listc-1; i+=2) {
        Tcl_ListObjIndex(interp, objv[2], i, &oPtr);
	if (!strcmp("sort", Tcl_GetString(oPtr))) {
	    Tcl_ListObjIndex(interp, objv[2], i+1, &oPtr);
	    sortName = Tcl_GetString(oPtr);
	    break;
	}
    }
    infoPtr->definition = def;
    infoPtr->refCount = 1;
    infoPtr->watcherInterval = 0;
    Tcl_ListObjGetElements(interp,
	    Tcl_GetVar2Ex(interp, "option", "watcher_time", TCL_GLOBAL_ONLY),
	    &lobjc, &lobjv);
    for (i=0; i < lobjc; i++) {
        Tcl_ListObjIndex(interp, lobjv[i], 0, &oPtr);
        if (!strcmp(Tcl_GetString(objv[1]), Tcl_GetString(oPtr))) {
	    Tcl_ListObjIndex(interp, lobjv[i], 1, &oPtr);
	    Tcl_GetIntFromObj(interp, oPtr, &infoPtr->watcherInterval);
	    break;
        }
    }
    if (!sortName || !strcmp("default", sortName)) {
	sortName = Tcl_GetVar2(interp, "option", "folder_sort",TCL_GLOBAL_ONLY);
    }
    for (i=0; sortNames[i].name && strcmp(sortNames[i].name, sortName); i++);
    if (sortNames[i].name) {
	infoPtr->sortOrder = sortNames[i].order;
	infoPtr->reverse = sortNames[i].reverse;
    } else {
	infoPtr->sortOrder = SORT_NONE;
	infoPtr->reverse = 0;
    }
    infoPtr->sortOrderChanged = 0;
    infoPtr->cmdName = ckalloc(16);
    infoPtr->allocated = infoPtr->number;
    infoPtr->msgCmdPtr = (char **) ckalloc(infoPtr->allocated*sizeof(char*));
    infoPtr->privatePtr = (ClientData**)ckalloc(
	    infoPtr->allocated*sizeof(ClientData));
    for (i=0; i<infoPtr->allocated; i++) {
	infoPtr->msgCmdPtr[i] = (char *) NULL;
	infoPtr->privatePtr[i] = (ClientData*) NULL;
    }
    (*infoPtr->initProc)(infoPtr, interp, -1);
    infoPtr->presentationOrder = (int*) ckalloc(infoPtr->allocated*sizeof(int));
    infoPtr->hidden = (int*) ckalloc(infoPtr->allocated*sizeof(int));
    infoPtr->flagsChanged = 0;
    infoPtr->nextPtr = ratFolderList;
    ratFolderList = infoPtr;
    RatFolderSort(interp, infoPtr);
    sprintf(infoPtr->cmdName, "RatFolder%d", numFolders++);
    Tcl_CreateCommand(interp, infoPtr->cmdName, RatFolderCmd,
    	    (ClientData) infoPtr, (Tcl_CmdDeleteProc *) NULL);
    Tcl_SetVar2Ex(interp, "folderExists", infoPtr->cmdName,
	    Tcl_NewIntObj(infoPtr->number), TCL_GLOBAL_ONLY);
    Tcl_SetVar2Ex(interp, "folderRecent", infoPtr->cmdName,
	    Tcl_NewIntObj(infoPtr->recent), TCL_GLOBAL_ONLY);
    Tcl_SetVar2Ex(interp, "folderUnseen", infoPtr->cmdName,
	    Tcl_NewIntObj(infoPtr->unseen), TCL_GLOBAL_ONLY);
    Tcl_SetResult(interp, infoPtr->cmdName, TCL_VOLATILE);
    if (infoPtr->watcherInterval) {
	infoPtr->timerToken = Tcl_CreateTimerHandler(
		infoPtr->watcherInterval*1000, RatFolderUpdateTime,
		(ClientData)infoPtr);
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatFolderCmd --
 *
 *      Main folder entity procedure. This procedure implements the
 *	folder commands mentioned in ../doc/interface. In order to make
 *	this a tad easier it uses the procedures defined in the
 *	RatFolderInfo structure :-)
 *
 * Results:
 *      Depends on the input :-)
 *
 * Side effects:
 *	The specified folder may be modified.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatFolderCmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
    RatFolderInfo *infoPtr = (RatFolderInfo*) clientData;
    int length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);
    if (strncmp(argv[1], "update", length) == 0) {
	RatUpdateType mode;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " update, checkpoint or expunge\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (!strcmp("update", argv[2])) {
	    mode = RAT_UPDATE;
	} else if (!strcmp("checkpoint", argv[2])) {
	    if (!infoPtr->flagsChanged) {
		Tcl_SetResult(interp, "0", TCL_STATIC);
		return TCL_OK;
	    }
	    mode = RAT_CHECKPOINT;
	    infoPtr->flagsChanged = 0;
	} else if (!strcmp("sync", argv[2])) {
	    mode = RAT_SYNC;
	} else {
	    Tcl_AppendResult(interp, "Bad mode argument: should be "
		    "\"update\", \"check\" or \"sync\"", (char*)NULL);
	    return TCL_ERROR;
	}
	return RatUpdateFolder(interp, infoPtr, mode);

    } else if (strncmp(argv[1], "close", length) == 0) {
	int force = 0;

	if (argc != 2 &&
		(argc != 3 || TCL_OK != Tcl_GetBoolean(interp,argv[2],&force))){
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " close [force]\"", (char *) NULL);
	    return TCL_ERROR;
	}
	return RatFolderClose(interp, infoPtr, force);

    } else if (strncmp(argv[1], "setName", length) == 0) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " setName name\"", (char *) NULL);
	    return TCL_ERROR;
	}
	ckfree(infoPtr->name);
	infoPtr->name = (char *) ckalloc(strlen(argv[2])+1);
	strcpy(infoPtr->name, argv[2]);
	return TCL_OK;

    } else if (strncmp(argv[1], "info", length) == 0) {
	int infoArgc;
	char *infoArgv[3];
	char numberBuf[16], sizeBuf[16];
	char *list;

	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " info\"", (char *) NULL);
	    return TCL_ERROR;
	}
	sprintf(numberBuf, "%d", infoPtr->visible);
	sprintf(sizeBuf, "%d", infoPtr->size);
	infoArgv[0] = infoPtr->name;
	infoArgv[1] = numberBuf;
	infoArgv[2] = sizeBuf;
	infoArgc = 3;
	list = Tcl_Merge(infoArgc, infoArgv);
	Tcl_SetResult(interp, list, TCL_DYNAMIC);
	return TCL_OK;

    } else if (strncmp(argv[1], "list", length) == 0) {
	ListExpression *exprPtr;
	Tcl_DString result;
	Tcl_Obj *oPtr;
	int i;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " list format\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (NULL == (exprPtr = RatParseList(argv[2]))) {
	    Tcl_SetResult(interp, "Illegal list format", TCL_STATIC);
	    return TCL_ERROR;
	}

	Tcl_DStringInit(&result);
	for (i=0; i < infoPtr->visible; i++) {
	    oPtr = RatDoList(interp, exprPtr, infoPtr->infoProc,
		    (ClientData)infoPtr, infoPtr->presentationOrder[i]);
	    Tcl_DStringAppendElement(&result, Tcl_GetString(oPtr));
	    Tcl_DecrRefCount(oPtr);
	}
	RatFreeListExpression(exprPtr);
	Tcl_DStringResult(interp, &result);
	return TCL_OK;

    } else if (strcmp(argv[1], "get") == 0) {
	int index;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " get index\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetInt(interp, argv[2], &index)) {
	    return TCL_ERROR;
	}
	if (index < 0 || index >= infoPtr->visible) {
	    Tcl_SetResult(interp, "Index is out of bounds", TCL_STATIC);
	    return TCL_ERROR;
	}
	if (NULL == infoPtr->msgCmdPtr[infoPtr->presentationOrder[index]]) {
	    infoPtr->msgCmdPtr[infoPtr->presentationOrder[index]] =
		    (*infoPtr->createProc)(infoPtr, interp,
		    infoPtr->presentationOrder[index]);
	}
	Tcl_SetResult(interp,
		infoPtr->msgCmdPtr[infoPtr->presentationOrder[index]],
		TCL_VOLATILE);
	return TCL_OK;

    } else if (strncmp(argv[1], "setFlag", length) == 0) {
	int index, value, unseen, recent;
	RatFlag flag;

	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " setFlag index flag value\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetInt(interp, argv[2], &index)) {
	    return TCL_ERROR;
	}
	if (index < 0 || index >= infoPtr->visible) {
	    Tcl_SetResult(interp, "Index is out of bounds", TCL_STATIC);
	    return TCL_ERROR;
	}
	if (0 == strcmp(argv[3], "seen")) {
	    flag = RAT_SEEN;
	} else if (0 == strcmp(argv[3], "deleted")) {
	    flag = RAT_DELETED;
	} else if (0 == strcmp(argv[3], "flagged")) {
	    flag = RAT_FLAGGED;
	} else if (0 == strcmp(argv[3], "answered")) {
	    flag = RAT_ANSWERED;
	} else if (0 == strcmp(argv[3], "draft")) {
	    flag = RAT_DRAFT;
	} else if (0 == strcmp(argv[3], "recent")) {
	    flag = RAT_RECENT;
	} else {
	    Tcl_SetResult(interp, "Unkown flag; should be one of seen, deleted, flagged, answered",
		     TCL_STATIC);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetBoolean(interp, argv[4], &value)) {
	    return TCL_ERROR;
	}

	if (value != (*infoPtr->getFlagProc)(infoPtr, interp,
		infoPtr->presentationOrder[index], flag)) {
	    recent = infoPtr->recent;
	    unseen = infoPtr->unseen;
	    (*infoPtr->setFlagProc)(infoPtr, interp,
		    infoPtr->presentationOrder[index], flag, value);
	    infoPtr->flagsChanged = 1;
	    if (infoPtr->recent != recent) {
		Tcl_SetVar2Ex(timerInterp, "folderRecent", infoPtr->cmdName,
			Tcl_NewIntObj(infoPtr->recent), TCL_GLOBAL_ONLY);
	    }
	    if (infoPtr->unseen != unseen) {
		Tcl_SetVar2Ex(timerInterp, "folderUnseen", infoPtr->cmdName,
			Tcl_NewIntObj(infoPtr->unseen), TCL_GLOBAL_ONLY);
	    }
	}
	return TCL_OK;

    } else if (strncmp(argv[1], "getFlag", length) == 0) {
	int index;
	RatFlag flag;

	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " getFlag index flag\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetInt(interp, argv[2], &index)) {
	    return TCL_ERROR;
	}
	if (index < 0 || index >= infoPtr->visible) {
	    Tcl_SetResult(interp, "Index is out of bounds", TCL_STATIC);
	    return TCL_ERROR;
	}
	if (0 == strcmp(argv[3], "seen")) {
	    flag = RAT_SEEN;
	} else if (0 == strcmp(argv[3], "deleted")) {
	    flag = RAT_DELETED;
	} else if (0 == strcmp(argv[3], "flagged")) {
	    flag = RAT_FLAGGED;
	} else if (0 == strcmp(argv[3], "answered")) {
	    flag = RAT_ANSWERED;
	} else if (0 == strcmp(argv[3], "draft")) {
	    flag = RAT_DRAFT;
	} else if (0 == strcmp(argv[3], "recent")) {
	    flag = RAT_RECENT;
	} else {
	    Tcl_SetResult(interp, "Unkown flag; should be one of seen, deleted, flagged, answered",
		     TCL_STATIC);
	    return TCL_ERROR;
	}

	Tcl_SetObjResult(interp, Tcl_NewIntObj(
		(*infoPtr->getFlagProc)(infoPtr, interp,
		infoPtr->presentationOrder[index], flag)));
	return TCL_OK;

    } else if (strncmp(argv[1], "flagged", length) == 0) {
	RatFlag flag;
	char buf[64];
	int i;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " flagged flag\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (0 == strcmp(argv[2], "seen")) {
	    flag = RAT_SEEN;
	} else if (0 == strcmp(argv[2], "deleted")) {
	    flag = RAT_DELETED;
	} else if (0 == strcmp(argv[2], "flagged")) {
	    flag = RAT_FLAGGED;
	} else if (0 == strcmp(argv[2], "answered")) {
	    flag = RAT_ANSWERED;
	} else if (0 == strcmp(argv[2], "draft")) {
	    flag = RAT_DRAFT;
	} else if (0 == strcmp(argv[2], "recent")) {
	    flag = RAT_RECENT;
	} else {
	    Tcl_SetResult(interp, "Unkown flag; should be one of seen, deleted, flagged, answered",
		     TCL_STATIC);
	    return TCL_ERROR;
	}

	Tcl_ResetResult(interp);
	for (i=0; i<infoPtr->visible; i++) {
	    if ((*infoPtr->getFlagProc)(infoPtr, interp,
		    infoPtr->presentationOrder[i], flag)) {
		sprintf(buf, "%d", i);
		Tcl_AppendElement(interp, buf);
	    }
	}
	return TCL_OK;

    } else if (strncmp(argv[1], "insert", length) == 0) {
	Tcl_CmdInfo cmdInfo;
	int i, result;

	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " insert message ?message ...?\"", (char *) NULL);
	    return TCL_ERROR;
	}

	if (argc-2 > infoPtr->allocated-infoPtr->number) {
	    infoPtr->allocated += argc;
	    infoPtr->msgCmdPtr = (char**)ckrealloc(infoPtr->msgCmdPtr,
	    	    infoPtr->allocated*sizeof(char*));
	    infoPtr->privatePtr = (ClientData**)ckrealloc(infoPtr->privatePtr,
	    	    infoPtr->allocated*sizeof(ClientData*));
	    infoPtr->presentationOrder = (int*)ckrealloc(
		    infoPtr->presentationOrder, infoPtr->allocated*sizeof(int));
	    infoPtr->hidden = (int *) ckrealloc(infoPtr->hidden,
		    infoPtr->allocated*sizeof(int));
	}
	for(i=2; i<argc; i++) {
	    if (0 == Tcl_GetCommandInfo(interp, argv[i], &cmdInfo)
		    || NULL == cmdInfo.objClientData) {
		Tcl_AppendResult(interp, "error \"", argv[i],
			"\" is not a valid message command", (char *) NULL);
		return TCL_ERROR;
	    }
	}
	result = (*infoPtr->insertProc)(infoPtr, interp, argc-2, &argv[2]);
	RatFolderSort(interp, infoPtr);
	return result;

    } else if (strncmp(argv[1], "type", length) == 0) {
	Tcl_SetResult(interp, infoPtr->type, TCL_STATIC);
	return TCL_OK;

    } else if (strncmp(argv[1], "find", length) == 0) {
	int msgNo, i;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " find message\"", (char *) NULL);
	    return TCL_ERROR;
	}

	Tcl_SetObjResult(interp, Tcl_NewIntObj(-1));
	for (msgNo=0; msgNo < infoPtr->number; msgNo++) {
	    if (infoPtr->msgCmdPtr[msgNo]
	    	    && !strcmp(infoPtr->msgCmdPtr[msgNo], argv[2])) {
		for (i=0; msgNo != infoPtr->presentationOrder[i]; i++);
		Tcl_SetObjResult(interp, Tcl_NewIntObj(i));
		break;
	    }
	}
	return TCL_OK;

    } else if (strncmp(argv[1], "match", length) == 0) {
	Tcl_DString ds;
	int i, expId;
	char buf[8];

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " match expId\"", (char *) NULL);
	    return TCL_ERROR;
	}

	Tcl_DStringInit(&ds);
	expId = atoi(argv[2]);

	for (i=0; i<infoPtr->number; i++) {
	    if (RatExpMatch(interp, expId, infoPtr->infoProc,
		    (ClientData)infoPtr, infoPtr->presentationOrder[i])) {
		sprintf(buf, "%d", i);
		Tcl_DStringAppendElement(&ds, buf);
	    }
	}
	Tcl_DStringResult(interp, &ds);
	return TCL_OK;

    } else if (strncmp(argv[1], "setSortOrder", length) == 0) {
	char *name = argv[2];
	int i, j;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " setSortOrder order\"", (char *) NULL);
	    return TCL_ERROR;
	}

	if (!strcmp("default", name)) {
	    name = Tcl_GetVar2(interp, "option", "folder_sort",TCL_GLOBAL_ONLY);
	}

	for (i=0; sortNames[i].name && strcmp(sortNames[i].name, name); i++);
	if (sortNames[i].name) {
	    /*
	     * If the old sort order was threaded and the new one is not then
	     * we should clear the threading info
	     */
	    if (infoPtr->sortOrder == SORT_THREADED
		    && sortNames[i].order != SORT_THREADED) {
		for (j=0; j<infoPtr->number; j++) {
		    (*infoPtr->setInfoProc)(interp,(ClientData)infoPtr,
			    RAT_FOLDER_THREADING, j, NULL);
		}
	    }
	    infoPtr->sortOrder = sortNames[i].order;
	    infoPtr->reverse = sortNames[i].reverse;
	    infoPtr->sortOrderChanged = 1;
	    return TCL_OK;
	} else {
	    Tcl_SetResult(interp, "No such sort order", TCL_STATIC);
	    return TCL_ERROR;
	}

    } else if (strncmp(argv[1], "getSortOrder", length) == 0) {
	int i;

	for (i=0; sortNames[i].name; i++) {
	    if (infoPtr->sortOrder == sortNames[i].order
		    && infoPtr->reverse == sortNames[i].reverse) {
		Tcl_SetResult(interp, sortNames[i].name, TCL_STATIC);
	    }
	}
	return TCL_OK;

    } else if (strncmp(argv[1], "netsync", length) == 0) {
	if (!infoPtr->syncProc) {
	    Tcl_AppendResult(interp, "Operation unsupported on this folder",
		    (char *) NULL);
	    return TCL_ERROR;
	}
	return (*infoPtr->syncProc)(infoPtr, interp);

    } else if (strncmp(argv[1], "refcount", length) == 0) {
	char buf[64];

	sprintf(buf, "%d", infoPtr->refCount);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	return TCL_OK;

    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1], (char *) NULL);
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatFolderSort --
 *
 *      Sorts the folder according to the users wishes. The user may
 *	communicates their will via the folder_sort variable. Currently
 *	The following methods are implemented:
 *	  subjectonly		- Alphabetically on subject
 *	  sender		- Alphabetically on sender name
 *	  folder		- Sorts in native folder order
 *	  reverseFolder		- The reverse of the above
 *	  date			- By date sent
 *	  reverseDate		- By reverse date sent
 *	  subject		- Group messages with the same subject
 *				  and sort the groups by the earliest date
 *				  in each group.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The presentation order member of the RatFolderInfo structure
 *	is initialized. The size of the folder is updated.
 *
 * TODO, convert to Tcl_Obj
 *
 *----------------------------------------------------------------------
 */

static void
RatFolderSort(Tcl_Interp *interp, RatFolderInfo *infoPtr)
{
    int i, j, k, pi, pj, snarf, numParm, seen, *tmpPtr, first, last,
	*p=infoPtr->presentationOrder,
	needDate=0, needSubject=0, needSender=0, needIds = 0,
	*uniqList, uniqListUsed, *subList, *lengthList;
    SortData *dataPtr, *dPtr;
    Tcl_Obj *oPtr, *o2Ptr;

    if (0 == infoPtr->number) {
	infoPtr->visible = 0;
	return;
    }

    Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option", "dsn_snarf_reports",
	    TCL_GLOBAL_ONLY), &snarf);

    switch(infoPtr->sortOrder) {
        case SORT_NONE:
	    break;
        case SORT_SUBJECT:
	    needSubject = 1;
	    break;
        case SORT_THREADED:
	    needIds = 1;
	    needDate = 1;
	    needSubject = 1;
	    break;
        case SORT_SUBJDATE:
	    needDate = 1;
	    needSubject = 1;
	    break;
        case SORT_SENDER:
	    needSender = 1;
	    break;
        case SORT_SENDERDATE:
	    needDate = 1;
	    needSender = 1;
	    break;
        case SORT_DATE:
	    needDate = 1;
	    break;
    }

    dataPtr = (SortData*)ckalloc(infoPtr->number*sizeof(*dataPtr));
    infoPtr->size = 0;
    for (i=0; i<infoPtr->number; i++) {
	infoPtr->hidden[i] = 0;
	infoPtr->presentationOrder[i] = i;
	oPtr=(*infoPtr->infoProc)(interp,(ClientData)infoPtr,RAT_FOLDER_SIZE,i);
	if (oPtr) {
	    Tcl_GetIntFromObj(interp, oPtr, &j);
	    infoPtr->size += j;
	}
	oPtr=(*infoPtr->infoProc)(interp,(ClientData)infoPtr,RAT_FOLDER_TYPE,i);
	if (oPtr && !strcasecmp(Tcl_GetString(oPtr), "multipart/report")) {
	    oPtr = (*infoPtr->infoProc)(interp, (ClientData)infoPtr,
		RAT_FOLDER_PARAMETERS, i);
	    if (!oPtr) {
		continue;
	    }
	    Tcl_ListObjLength(interp, oPtr, &numParm);
	    for (j=0; j<numParm; j++) {
		Tcl_ListObjIndex(interp, oPtr, j, &o2Ptr);
		if (!strcasecmp(Tcl_GetString(o2Ptr),
			"report-type delivery-status")) {
		    /*
		     * This is a DSN; call the proper authorities
		     */
		    seen=(*infoPtr->getFlagProc)(infoPtr, interp, i, RAT_SEEN);
		    if (!infoPtr->msgCmdPtr[i]) {
			infoPtr->msgCmdPtr[i] =
				(*infoPtr->createProc)(infoPtr, interp, i);
		    }
		    if (RatDSNHandle(interp, infoPtr->msgCmdPtr[i]) && snarf) {
			if ((*infoPtr->getFlagProc)(
				infoPtr, interp, i, RAT_RECENT)){
			    infoPtr->recent--;
			}
			if (!seen) {
			   infoPtr->unseen--;
			}
			infoPtr->hidden[i] = 1;
			(*infoPtr->setFlagProc)(infoPtr,interp,i,RAT_DELETED,1);
		    } else {
			(*infoPtr->setFlagProc)(infoPtr,interp,i,RAT_SEEN,seen);
		    }
		    break;
		}
	    }
	}
	if (needSubject) {
	    oPtr = (*infoPtr->infoProc)(interp, (ClientData)infoPtr,
		    RAT_FOLDER_CANONSUBJECT, i);
	    dataPtr[i].subject = Tcl_GetString(oPtr);
	}
	if (needSender) {
	    oPtr = (*infoPtr->infoProc)(interp,(ClientData)infoPtr,
		    RAT_FOLDER_NAME, i);
	    dataPtr[i].sender = Tcl_GetString(oPtr);
	    dataPtr[i].sender = cpystr(dataPtr[i].sender);
	    lcase(dataPtr[i].sender);
	}
	if (needDate) {
	    oPtr = (*infoPtr->infoProc)(interp,(ClientData)infoPtr,
		    RAT_FOLDER_DATE_N, i);
	    Tcl_GetLongFromObj(interp, oPtr, &dataPtr[i].date);
	}
	if (needIds) {
	    oPtr = (*infoPtr->infoProc)(interp,(ClientData)infoPtr,
		    RAT_FOLDER_MSGID, i);
	    if (oPtr) {
		dataPtr[i].msgid = Tcl_GetString(oPtr);
	    } else {
		dataPtr[i].msgid = "";
	    }
	    oPtr = (*infoPtr->infoProc)(interp,(ClientData)infoPtr,
		    RAT_FOLDER_REF, i);
	    if (oPtr) {
		dataPtr[i].in_reply_to = Tcl_GetString(oPtr);
	    } else {
		dataPtr[i].in_reply_to = "";
	    }
	}
    }

    baseSortDataPtr = dataPtr;
    switch (infoPtr->sortOrder) {
    case SORT_NONE:
	for (i=0; i<infoPtr->number; i++) {
	    p[i] = i;
	}
	break;
    case SORT_THREADED:
	for (i=0; i<infoPtr->number; i++) {
	    p[i] = i;
	}
	qsort((void*)p, infoPtr->number, sizeof(int), RatFolderSortCompareDate);
	/*for (i=0; i<infoPtr->number; i++) {
	    printf("Msg: %d (really %d)\n", i, p[i]);
	    printf(" Subj: %s\n", dataPtr[p[i]].subject);
	    printf("MsgId: %s\n", dataPtr[p[i]].msgid);
	    printf("  Ref: %s\n", (dataPtr[p[i]].in_reply_to ?
		    dataPtr[p[i]].in_reply_to : "(NULL)"));
	}*/
	/*
	 * Start by sorting on hard references
	 */
	dataPtr[p[0]].prev = -1;
	dataPtr[p[0]].next = -1;
	dataPtr[p[0]].child = -1;
	dataPtr[p[0]].parent = -1;
	dataPtr[p[0]].tPtr = NULL;
	first = last = p[0];
	for (i=1; i<infoPtr->number; i++) {
	    pi = p[i];
	    dataPtr[pi].tPtr = NULL;
	    /* Find any replies to the current message */
	    for (j=0; j<i && *dataPtr[pi].msgid; j++) {
		pj = p[j];
		if (!strcmp(dataPtr[pj].in_reply_to, dataPtr[pi].msgid)){
		    /* Here 'j' is considered a reply to 'i' */
		    dataPtr[pi].next = dataPtr[pj].next;
		    dataPtr[pi].prev = dataPtr[pj].prev;
		    dataPtr[pi].parent = dataPtr[pj].parent;
		    dataPtr[pi].child = pj;
		    dataPtr[pj].next = -1;
		    dataPtr[pj].prev = -1;
		    dataPtr[pj].parent = pi;
		    if (-1!=dataPtr[pi].next) dataPtr[dataPtr[pi].next].prev=pi;
		    if (-1!=dataPtr[pi].prev) dataPtr[dataPtr[pi].prev].next=pi;
		    if (-1!=dataPtr[pi].parent)
			dataPtr[dataPtr[pi].parent].child = pi;
		    if (first == pj) first = pi;
		    if (last == pj) last = pi;
		    break;
		}
	    }
	    if (j == i || !*dataPtr[pi].msgid) {
		dataPtr[pi].prev = last;
		dataPtr[last].next = pi;
		dataPtr[pi].next = -1;
		dataPtr[pi].child = -1;
		dataPtr[pi].parent = -1;
		last = pi;
	    }
	    /* Find message which the current message is a reply to */
	    for (j=0; j<i
		    && -1 == dataPtr[pi].parent
		    && *dataPtr[pi].in_reply_to; j++) {
		pj = p[j];
		if (!strcmp(dataPtr[pj].msgid, dataPtr[pi].in_reply_to)
			 && !IsChild(dataPtr, pj, pi)) {
		    if (first == pi) {
			if (-1 == dataPtr[first].next) {
			    first = pj;
			} else {
			    first = dataPtr[first].next;
			}
		    }
		    if (last == pi) {
			if (-1 == dataPtr[last].prev) {
			    last = pj;
			} else {
			    last = dataPtr[last].prev;
			}
		    }
		    if (dataPtr[pi].next != -1) {
			dataPtr[dataPtr[pi].next].prev = dataPtr[pi].prev;
		    }
		    if (dataPtr[pi].prev != -1) {
			dataPtr[dataPtr[pi].prev].next = dataPtr[pi].next;
		    }
		    if (-1 != dataPtr[pj].child) {
			for (pj = dataPtr[pj].child; -1 != dataPtr[pj].next;
				pj = dataPtr[pj].next);
			dataPtr[pi].prev = pj;
			dataPtr[pj].next = pi;
		    } else {
			dataPtr[pj].child = pi;
			dataPtr[pi].parent = pj;
			dataPtr[pi].prev = -1;
		    }
		    dataPtr[pi].next = -1;
		    break;
		}
	    }
	}
	/*
	 * Now we have a number of 'trees' linked by hard references.
	 * Here we try to link the top nodes in all trees by subject.
	 */
	for (i=1; i<infoPtr->number; i++) {
	    pi = p[i];
	    if (-1 != dataPtr[pi].parent) continue;

	    for (j=i-1; j>=0; j--) {
		pj = p[j];
		if (!strcmp(dataPtr[pi].subject, dataPtr[pj].subject)
			&& !IsChild(dataPtr, pj, pi)) {
		    /*
		     * 'pi' is later in the same thread as 'pj'
		     * First we remove it from the list.
		     */
		    dataPtr[dataPtr[pi].prev].next = dataPtr[pi].next;
		    if (dataPtr[pi].next != -1) {
			dataPtr[dataPtr[pi].next].prev = dataPtr[pi].prev;
		    }

		    /*
		     * If the parent to 'pj' also has the same subject
		     * then we add this message after 'pj', otherwise
		     * we add it under 'pj'
		     */
		    for (k=pj; -1 != dataPtr[k].prev; k = dataPtr[k].prev);
		    if (-1 != dataPtr[k].parent && !strcmp(dataPtr[pi].subject,
			    dataPtr[dataPtr[k].parent].subject)) {
			dataPtr[pi].prev = pj;
			dataPtr[pi].next = dataPtr[pj].next;
			if (dataPtr[pj].next != -1) {
			    dataPtr[dataPtr[pj].next].prev = pi;
			}
			dataPtr[pj].next = pi;
		    } else {
			dataPtr[pi].parent = pj;
			dataPtr[pi].prev = -1;
			dataPtr[pi].next = dataPtr[pj].child;
			if (-1 != dataPtr[pj].child) {
			    dataPtr[dataPtr[pj].child].prev = pi;
			    dataPtr[dataPtr[pj].child].parent = -1;
			}
			dataPtr[pj].child = pi;
		    }
		    break;
		}
	    }
	}
	/*printf("First: %d\n", first);
	for (i=0; i<infoPtr->number; i++) {
	    printf("%d:\tprev: %2d  next: %2d  parent: %2d  child: %2d\n",
		    i, dataPtr[i].prev, dataPtr[i].next, dataPtr[i].parent,
		    dataPtr[i].child);
	}*/
	RatFolderSortLinearize(p, infoPtr->number, dataPtr, first, 0);
	for (i=0; i<infoPtr->number; i++) {
	    (*infoPtr->setInfoProc)(interp,(ClientData)infoPtr,
		    RAT_FOLDER_THREADING, i, dataPtr[i].tPtr);
	}
	break;
    case SORT_SUBJDATE:
    case SORT_SENDERDATE:
	/*
	 * This algorithm is complicated:
	 * - First we build a list of unique subjects in uniqList. Each entry
	 *   in this list contains the index of the first message with this
	 *   subject. The messages are linked with the nextPtr field in
	 *   the SortData structs.
	 * - Then we sort each found subject. This is done by placing the
	 *   indexes of the messages in subList. And sort that. When it
	 *   is sorted we rebuild the subject chains via the nextPtr;
	 * - After that we sort the first message in each subject. This is done
	 *   by reusing the uniqList. We replace each entry in it with a pointer
	 *   to the first entry in the set. Actually we do this in the preceding
	 *   step. Then we sort this list.
	 * - Finally we build to result array.
	 */
	uniqList = (int*)ckalloc(2*infoPtr->number*sizeof(*uniqList));
	subList = &uniqList[infoPtr->number];
	lengthList = &uniqList[2*infoPtr->number];
	for (i=uniqListUsed=0; i<infoPtr->number; i++) {
	    for (j=0; j<uniqListUsed; j++) {
		if (infoPtr->sortOrder == SORT_SUBJDATE ?
			!strcmp(dataPtr[i].subject,
			dataPtr[uniqList[j]].subject) : !strcmp(
			dataPtr[i].sender, dataPtr[uniqList[j]].sender)) {
		    dataPtr[i].nextPtr = dataPtr[uniqList[j]].nextPtr;
		    dataPtr[uniqList[j]].nextPtr = &dataPtr[i];
		    break;
		}
	    }
	    if (j == uniqListUsed) {
		dataPtr[i].nextPtr = NULL;
		uniqList[uniqListUsed++] = i;
	    }
	}
	for (i=0; i<uniqListUsed; i++) {
	    if (NULL != dataPtr[uniqList[i]].nextPtr) {
		for (j = 0, dPtr = &dataPtr[uniqList[i]]; dPtr;
			dPtr = dPtr->nextPtr) {
		    subList[j++] = dPtr-dataPtr;
		}
		qsort((void*)subList, j, sizeof(int), RatFolderSortCompareDate);
		for (k=0; k<j-1; k++) {
		    dataPtr[subList[k]].nextPtr = &dataPtr[subList[k+1]];
		}
		dataPtr[subList[k]].nextPtr = NULL;
		uniqList[i] = subList[0];
	    }
	}
	qsort((void*)uniqList, uniqListUsed, sizeof(int),
		RatFolderSortCompareDate);
	for (i=k=0; i<uniqListUsed; i++) {
	    for (dPtr = &dataPtr[uniqList[i]]; dPtr; dPtr = dPtr->nextPtr) {
		p[k++] = dPtr-baseSortDataPtr;
	    }
	}
	ckfree(uniqList);
	break;
    case SORT_SENDER:
	for (i=0; i<infoPtr->number; i++) {
	    p[i] = i;
	}
	qsort((void*)p, infoPtr->number,sizeof(int),RatFolderSortCompareSender);
	break;
    case SORT_SUBJECT:
	for (i=0; i<infoPtr->number; i++) {
	    p[i] = i;
	}
	qsort((void*)p,infoPtr->number,sizeof(int),RatFolderSortCompareSubject);
	break;
    case SORT_DATE:
	for (i=0; i<infoPtr->number; i++) {
	    p[i] = i;
	}
	qsort((void*)p, infoPtr->number, sizeof(int), RatFolderSortCompareDate);
	break;
    }

    if (infoPtr->reverse) {
	tmpPtr = (int*)ckalloc(infoPtr->number*sizeof(int));
	for (i=infoPtr->number-1, j=0; i >= 0; i--) {
	    if (!infoPtr->hidden[p[i]]) {
		tmpPtr[j++] = p[i];
	    }
	}
	memcpy(p, tmpPtr, j*sizeof(int));
	ckfree(tmpPtr);
    } else {
	for (i=j=0; i < infoPtr->number; i++) {
	    if (!infoPtr->hidden[p[i]]) {
		p[j++] = p[i];
	    }
	}
    }
    infoPtr->visible = j;

    /*
     * Cleanup dataPtr
     */
    for (i=0; i<infoPtr->number; i++) {
	if (needSender) {
	    ckfree(dataPtr[i].sender);
	}
    }

    ckfree(dataPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * RatFolderSortCompare* --
 *
 *	This is the comparison functions used by RatFolderSort. They
 *	expect to get pointers to integers as argumens. The integers
 *	pointed at are indexes into a list of SortData structs which can be
 *	found at the address in baseSortDataPtr.
 *
 * Results:
 *	An integers describing the order of the compared objects.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatFolderSortCompareDate(const void *arg1, const void *arg2)
{
    return baseSortDataPtr[*((int*)arg1)].date
	    - baseSortDataPtr[*((int*)arg2)].date;
}

static int
RatFolderSortCompareSubject(const void *arg1, const void *arg2)
{
    return strcmp(baseSortDataPtr[*((int*)arg1)].subject,
		  baseSortDataPtr[*((int*)arg2)].subject);
}

static int
RatFolderSortCompareSender(const void *arg1, const void *arg2)
{
    return strcmp(baseSortDataPtr[*((int*)arg1)].sender,
		  baseSortDataPtr[*((int*)arg2)].sender);
}


/*
 *----------------------------------------------------------------------
 *
 * RatFolderCanonalizeSubject --
 *
 * Copy a subject line and remove certain constructs (the re:).
 *
 * Results:
 *	A new object reference
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

Tcl_Obj*
RatFolderCanonalizeSubject (const char *s)
{
    const char *e;
    Tcl_Obj *nPtr;
    int len;

    if (s) {
	/*
	 * We first try to find the start of the actual text (i.e. without any
	 * leading Re:'s and whitespaces. Then we find how long the text is
	 * (ignore trailing whitespaces).
	 */
	len = strlen(s);
	e = s+len-1;
	while (*s) {
	    while (*s && s < e && isspace((unsigned char)*s)) s++, len--;
	    if (!strncasecmp(s, "re", 2) && (':' == s[2]
		    || isspace((unsigned char)s[2]))) {
		s += 2;
		len -= 2;
		if (*s == ':') {
		    s++;
		    len--;
		}
	    } else {
		break;
	    }
	}
	while (isspace((unsigned char)*e) && e > s) {
	    e--;
	    len--;
	}
	nPtr = Tcl_NewStringObj(s, len);
	len = Tcl_UtfToLower(Tcl_GetString(nPtr));
	Tcl_SetObjLength(nPtr, len);
	return nPtr;
    } else {
	return Tcl_NewStringObj("", 0);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * IsChild --
 *
 * 	See if one message has the other as one of its ancestors.
 *
 * Results:
 *	A non-zero value if the child is related to the parent
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */
static int
IsChild(SortData *dataPtr, int child, int parent)
{
    int i;

    for (i=child; i > -1 && i != parent; i = dataPtr[i].parent);
    return (i == parent);
}

/*
 *----------------------------------------------------------------------
 *
 * RatFolderSortLinearize --
 *
 * Linearizes the linked list of messages. The list is also sorted.
 *
 * Results:
 *	Number of elements added to p
 *
 * Side effects:
 *	Modifies the p array.
 *
 *
 *----------------------------------------------------------------------
 */
static int
RatFolderSortLinearize(int *p, int n, SortData *dataPtr, int first, int depth)
{
    int *s = (int*)ckalloc(sizeof(int)*n);
    int i, j, k, o, ns;
    char *c;

    for (i=first, ns=0; -1 != i; i = dataPtr[i].next) {
	s[ns++] = i;
    }
    qsort((void*)s, ns, sizeof(int), RatFolderSortCompareDate);
    for (i=j=0; i<ns; i++) {
	if (depth) {
	    dataPtr[s[i]].tPtr = Tcl_NewObj();
	    for (k=0; k<depth-1; k++) {
		Tcl_AppendToObj(dataPtr[s[i]].tPtr, " ", 1);
	    }
	    Tcl_AppendToObj(dataPtr[s[i]].tPtr, "+", 1);
	}
	p[j++] = s[i];
	if (-1 != dataPtr[s[i]].child) {
	    o = j;
	    j += RatFolderSortLinearize(&p[j], n-ns, dataPtr,
		    dataPtr[s[i]].child, depth+1);
	    if (i < ns-1 && depth) {
		while (o<j) {
		    c = Tcl_GetStringFromObj(dataPtr[p[o++]].tPtr, NULL);
		    c[depth-1] = '|';
		}
	    }
	}
    }

    ckfree(s);
    return j;
}

/*
 *----------------------------------------------------------------------
 *
 * RatGetMsgInfo --
 *
 * Gets info from message structure and formats it somewhat. None of the
 * informations items needs both the messagePtr and the eltPtr. The following
 * table describes which needs which:
 *
 *  Info			msgPtr	envPtr	bodyPtr	eltPtr	size
 *  RAT_FOLDER_SUBJECT		-	needed	-	-	-
 *  RAT_FOLDER_CANONSUBJECT	-	needed	-	-	-
 *  RAT_FOLDER_NAME		needed	needed	-	-	-
 *  RAT_FOLDER_MAIL_REAL	-	needed	-	-	-
 *  RAT_FOLDER_MAIL		needed	needed	-	-	-
 *  RAT_FOLDER_NAME_RECIPIENT	needed	needed	-	-	-
 *  RAT_FOLDER_MAIL_RECIPIENT	needed	needed	-	-	-
 *  RAT_FOLDER_SIZE		-	-	-	-	needed
 *  RAT_FOLDER_SIZE_F		-	-	-	-	needed
 *  RAT_FOLDER_DATE_F		-	needed	-	needed	-
 *  RAT_FOLDER_DATE_N		-	needed	-	needed	-
 *  RAT_FOLDER_STATUS		    [not supported ]
 *  RAT_FOLDER_TYPE		-	-	needed	-	-
 *  RAT_FOLDER_PARAMETERS	-	-	needed	-	-
 *  RAT_FOLDER_FLAGS		-	-	-	needed	-
 *  RAT_FOLDER_UNIXFLAGS	-	-	-	needed	-
 *  RAT_FOLDER_MSGID		-	needed	-	-	-
 *  RAT_FOLDER_REF		-	needed	-	-	-
 *  RAT_FOLDER_INDEX		    [not supported ]
 *  RAT_FOLDER_THREADING	    [not supported ]
 *
 * Results:
 *	A pointer to a string which is valid at least until the next call
 *	to this structure.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

Tcl_Obj*
RatGetMsgInfo(Tcl_Interp *interp, RatFolderInfoType type, MessageInfo *msgPtr,
	ENVELOPE *envPtr, BODY *bodyPtr, MESSAGECACHE *eltPtr, int size)
{
    Tcl_Obj *oPtr = NULL, *pPtr[2];
    time_t time, zonediff;
    MESSAGECACHE dateElt, *dateEltPtr;
    PARAMETER *parmPtr;
    ADDRESS *adrPtr;
    struct tm tm;
    char buf[1024], *s, *e;

    switch (type) {
	case RAT_FOLDER_SUBJECT:
	    oPtr = Tcl_NewStringObj(
		    RatDecodeHeader(interp,envPtr->subject,0), -1);
	    break;
	case RAT_FOLDER_CANONSUBJECT:
	    oPtr = RatFolderCanonalizeSubject(
		    RatDecodeHeader(interp,envPtr->subject,0));
	    break;
	case RAT_FOLDER_MAIL_REAL:
	    for (adrPtr = envPtr->from; adrPtr; adrPtr = adrPtr->next) {
		if (adrPtr->mailbox && adrPtr->host) {
		    break;
		}
	    }
	    if (!adrPtr) {
		oPtr = Tcl_NewStringObj(NULL, 0);
	    } else {
		oPtr = Tcl_NewStringObj(RatAddressMail(adrPtr), -1);
	    }
	    break;
	case RAT_FOLDER_NAME:
	    if (!envPtr->from) {
		oPtr = Tcl_NewObj();
		break;
	    }
	    if (RAT_ISME_YES == msgPtr->fromMe
		    || (RAT_ISME_UNKOWN == msgPtr->fromMe
			&& RatAddressIsMe(interp, envPtr->from, 1))) {
		msgPtr->fromMe = RAT_ISME_YES;
		if (envPtr->to && envPtr->to->personal) {
		    oPtr = Tcl_GetVar2Ex(interp, "t", "to", TCL_GLOBAL_ONLY);
		    if (Tcl_IsShared(oPtr)) {
			oPtr = Tcl_DuplicateObj(oPtr);
		    }
		    Tcl_AppendToObj(oPtr, ": ", 2);
		    Tcl_AppendToObj(oPtr,
			    RatDecodeHeader(interp, envPtr->to->personal,0),-1);
		    break;
		}
	    } else {
		msgPtr->fromMe = RAT_ISME_NO;
		if (envPtr->from->personal) {
		    oPtr = Tcl_NewStringObj(RatDecodeHeader(interp,
			    envPtr->from->personal, 0), -1);
		    break;
		}
	    }
	    /* fallthrough */
	case RAT_FOLDER_MAIL:
	    oPtr = Tcl_NewObj();
	    if (RAT_ISME_YES == msgPtr->fromMe
		    || (RAT_ISME_UNKOWN == msgPtr->fromMe
			&& RatAddressIsMe(interp, envPtr->from, 1))) {
		msgPtr->fromMe = RAT_ISME_YES;
		adrPtr = envPtr->to;
		Tcl_AppendObjToObj(oPtr, Tcl_GetVar2Ex(interp, "t", "to",
			TCL_GLOBAL_ONLY));
		Tcl_AppendToObj(oPtr, ": ", 2);
	    } else {
		msgPtr->fromMe = RAT_ISME_NO;
		adrPtr = envPtr->from;
	    }
	    for (; adrPtr; adrPtr = adrPtr->next) {
		if (adrPtr->mailbox && adrPtr->host) {
		    break;
		}
	    }
	    if (!adrPtr) {
		Tcl_DecrRefCount(oPtr);
		oPtr = Tcl_NewObj();
	    } else {
		Tcl_AppendToObj(oPtr, RatAddressMail(adrPtr), -1);
	    }
	    break;
	case RAT_FOLDER_NAME_RECIPIENT:
	    if (!envPtr->to) {
		oPtr = Tcl_NewObj();
		break;
	    }
	    if (RAT_ISME_YES == msgPtr->toMe
		    || (RAT_ISME_UNKOWN == msgPtr->toMe
			&& RatAddressIsMe(interp, envPtr->to, 1))) {
		msgPtr->toMe = RAT_ISME_YES;
		if (envPtr->from && envPtr->from->personal) {
		    oPtr = Tcl_GetVar2Ex(interp, "t", "to", TCL_GLOBAL_ONLY);
		    if (Tcl_IsShared(oPtr)) {
			oPtr = Tcl_DuplicateObj(oPtr);
		    }
		    Tcl_AppendToObj(oPtr, ": ", 2);
		    Tcl_AppendToObj(oPtr,
			    RatDecodeHeader(interp, envPtr->to->personal,0),-1);
		    break;
		}
	    } else {
		msgPtr->toMe = RAT_ISME_NO;
		if (envPtr->to->personal) {
		    oPtr = Tcl_NewStringObj(RatDecodeHeader(interp,
			    envPtr->to->personal, 0), -1);
		    break;
		}
	    }
	    /* fallthrough */
	case RAT_FOLDER_MAIL_RECIPIENT:
	    oPtr = Tcl_NewObj();
	    if (RAT_ISME_YES == msgPtr->toMe
		    || (RAT_ISME_UNKOWN == msgPtr->toMe
			&& RatAddressIsMe(interp, envPtr->to, 1))) {
		msgPtr->toMe = RAT_ISME_YES;
		adrPtr = envPtr->from;
		Tcl_AppendObjToObj(oPtr, Tcl_GetVar2Ex(interp, "t", "to",
			TCL_GLOBAL_ONLY));
		Tcl_AppendToObj(oPtr, ": ", 2);
	    } else {
		msgPtr->toMe = RAT_ISME_NO;
		adrPtr = envPtr->to;
	    }
	    for (; adrPtr; adrPtr = adrPtr->next) {
		if (adrPtr->mailbox && adrPtr->host) {
		    break;
		}
	    }
	    if (!adrPtr) {
		Tcl_DecrRefCount(oPtr);
		oPtr = Tcl_NewObj();
	    } else {
		Tcl_AppendToObj(oPtr, RatAddressMail(adrPtr), -1);
	    }
	    break;
	case RAT_FOLDER_SIZE:
	    oPtr = Tcl_NewIntObj(size);
	    break;
	case RAT_FOLDER_SIZE_F:
	    return RatMangleNumber(size);
	    break;
	case RAT_FOLDER_DATE_F:
	    if (envPtr->date && T == mail_parse_date(&dateElt, envPtr->date)) {
		dateEltPtr = &dateElt;
	    } else {
		dateEltPtr = eltPtr;
	    }
	    oPtr = RatFormatDate(interp, dateEltPtr->month-1, dateEltPtr->day);
	    break;
	case RAT_FOLDER_DATE_N:
	    if (envPtr->date && T == mail_parse_date(&dateElt, envPtr->date)) {
		dateEltPtr = &dateElt;
	    } else {
		dateEltPtr = eltPtr;
	    }
	    tm.tm_sec = dateEltPtr->seconds;
	    tm.tm_min = dateEltPtr->minutes;
	    tm.tm_hour = dateEltPtr->hours;
	    tm.tm_mday = dateEltPtr->day;
	    tm.tm_mon = dateEltPtr->month - 1;
	    tm.tm_year = dateEltPtr->year+70;
	    tm.tm_wday = 0;
	    tm.tm_yday = 0;
	    tm.tm_isdst = -1;
	    time = mktime(&tm);
	    zonediff = (dateEltPtr->zhours*60+dateEltPtr->zminutes)*60;
	    if (!dateEltPtr->zoccident) {
		zonediff *= -1;
	    }
	    time += zonediff;
	    oPtr = Tcl_NewObj();
	    Tcl_SetLongObj(oPtr, time);
	    break;
	case RAT_FOLDER_DATE_IMAP4:
	    if (envPtr->date && T == mail_parse_date(&dateElt, envPtr->date)) {
		dateEltPtr = &dateElt;
	    } else {
		dateEltPtr = eltPtr;
	    }
	    mail_date(buf, dateEltPtr); 
	    oPtr = Tcl_NewStringObj(buf, -1);
	    break;
	case RAT_FOLDER_TYPE:
	    oPtr = Tcl_NewObj();
	    Tcl_AppendStringsToObj(oPtr, body_types[bodyPtr->type], "/",
		    bodyPtr->subtype, NULL);
	    break;
	case RAT_FOLDER_PARAMETERS:
	    oPtr = Tcl_NewObj();
	    for (parmPtr = bodyPtr->parameter; parmPtr;
		    parmPtr = parmPtr->next) {
		pPtr[0] = Tcl_NewStringObj(parmPtr->attribute, -1);
		pPtr[1] = Tcl_NewStringObj(parmPtr->value, -1);
		Tcl_ListObjAppendElement(interp, oPtr, Tcl_NewListObj(2, pPtr));
	    }
	    break;
	case RAT_FOLDER_TO:
	    oPtr = Tcl_NewStringObj("", 0);
	    Tcl_SetObjLength(oPtr, RatAddressSize(envPtr->to, 1));
	    Tcl_GetString(oPtr)[0] = '\0';
	    rfc822_write_address(Tcl_GetString(oPtr), envPtr->to);
	    Tcl_SetObjLength(oPtr, strlen(Tcl_GetString(oPtr)));
	    break;
	case RAT_FOLDER_FROM:
	    oPtr = Tcl_NewStringObj("", 0);
	    Tcl_SetObjLength(oPtr, RatAddressSize(envPtr->from, 1));
	    Tcl_GetString(oPtr)[0] = '\0';
	    rfc822_write_address(Tcl_GetString(oPtr), envPtr->from);
	    Tcl_SetObjLength(oPtr, strlen(Tcl_GetString(oPtr)));
	    break;
	case RAT_FOLDER_SENDER:
	    oPtr = Tcl_NewStringObj("", 0);
	    Tcl_SetObjLength(oPtr, RatAddressSize(envPtr->sender, 1));
	    Tcl_GetString(oPtr)[0] = '\0';
	    rfc822_write_address(Tcl_GetString(oPtr), envPtr->sender);
	    Tcl_SetObjLength(oPtr, strlen(Tcl_GetString(oPtr)));
	    break;
	case RAT_FOLDER_CC:
	    oPtr = Tcl_NewStringObj("", 0);
	    Tcl_SetObjLength(oPtr, RatAddressSize(envPtr->cc, 1));
	    Tcl_GetString(oPtr)[0] = '\0';
	    rfc822_write_address(Tcl_GetString(oPtr), envPtr->cc);
	    Tcl_SetObjLength(oPtr, strlen(Tcl_GetString(oPtr)));
	    break;
	case RAT_FOLDER_REPLY_TO:
	    oPtr = Tcl_NewStringObj("", 0);
	    Tcl_SetObjLength(oPtr, RatAddressSize(envPtr->reply_to, 1));
	    Tcl_GetString(oPtr)[0] = '\0';
	    rfc822_write_address(Tcl_GetString(oPtr), envPtr->reply_to);
	    Tcl_SetObjLength(oPtr, strlen(Tcl_GetString(oPtr)));
	    break;
	case RAT_FOLDER_FLAGS:
	    oPtr = Tcl_NewStringObj(MsgFlags(eltPtr), -1);
	    break;
	case RAT_FOLDER_UNIXFLAGS:
	    s = buf;
	    if (eltPtr->seen)	  *s++ = 'R';
	    if (eltPtr->deleted)  *s++ = 'D';
	    if (eltPtr->flagged)  *s++ = 'F';
	    if (eltPtr->answered) *s++ = 'A';
	    oPtr = Tcl_NewStringObj(buf, s-buf);
	    break;
	case RAT_FOLDER_MSGID:
	    oPtr = Tcl_NewStringObj(envPtr->message_id, -1);
	    break;
	case RAT_FOLDER_REF:
	    if (envPtr->in_reply_to
		    && (s = strchr(envPtr->in_reply_to, '<'))
		    && (e = strchr(s, '>'))) {
		oPtr = Tcl_NewStringObj(s, e-s+1);
	    } else if (envPtr->references
		    && (s = strchr(envPtr->references, '<'))
		    && (e = strchr(s, '>'))) {
		oPtr = Tcl_NewStringObj(s, e-s+1);
	    } else {
		return NULL;
	    }
	    break;
	case RAT_FOLDER_STATUS:	   /*fallthrough */
	case RAT_FOLDER_INDEX:	   /*fallthrough */
	case RAT_FOLDER_THREADING: /*fallthrough */
	case RAT_FOLDER_END:
	    return NULL;
    }
    msgPtr->info[type] = oPtr;
    Tcl_IncrRefCount(oPtr);
    return oPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * MsgFlags --
 *
 *	Returns the flags of a message
 *
 * Results:
 *	A poiter to a static area containing the flags
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */
char*
MsgFlags(MESSAGECACHE *eltPtr)
{
    static Tcl_DString ds;
    static int initialized = 0;

    if (!initialized) {
	Tcl_DStringInit(&ds);
	initialized = 1;
    }

    Tcl_DStringSetLength(&ds, 0);
    if (eltPtr->seen) {
	Tcl_DStringAppend(&ds, RAT_SEEN_STR, -1);
    }
    if (eltPtr->deleted) {
	if (Tcl_DStringLength(&ds)) {
	    Tcl_DStringAppend(&ds, " ",1);
	}
	Tcl_DStringAppend(&ds, RAT_DELETED_STR, -1);
    }
    if (eltPtr->flagged) {
	if (Tcl_DStringLength(&ds)) {
	    Tcl_DStringAppend(&ds, " ",1);
	}
	Tcl_DStringAppend(&ds, RAT_FLAGGED_STR, -1);
    }
    if (eltPtr->answered) {
	if (Tcl_DStringLength(&ds)) {
	    Tcl_DStringAppend(&ds, " ",1);
	}
	Tcl_DStringAppend(&ds, RAT_ANSWERED_STR, -1);
    }
    if (eltPtr->draft) {
	if (Tcl_DStringLength(&ds)) {
	    Tcl_DStringAppend(&ds, " ",1);
	}
	Tcl_DStringAppend(&ds, RAT_DRAFT_STR, -1);
    }
    if (eltPtr->recent) {
	if (Tcl_DStringLength(&ds)) {
	    Tcl_DStringAppend(&ds, " ",1);
	}
	Tcl_DStringAppend(&ds, RAT_DRAFT_STR, -1);
    }

    return Tcl_DStringValue(&ds);
}


/*
 *----------------------------------------------------------------------
 *
 * RatParseFrom --
 *
 *	Parse the time in a 'From ' line. See ../imap/src/osdep/unix/unix.h
 *	for details on how this line may look like.
 *
 * Results:
 *	A poiter to a static area containing a MESSAGECACHE.
 *	The only valid fields in this are the time-fields
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */

MESSAGECACHE*
RatParseFrom(const char *from)
{
    static MESSAGECACHE elt;
    const char *cPtr;
    int i=0, found;

    /*
     * Start by finding the weekday name, if it is followed by one
     * space and a month-spec, then we assume we have found the date.
     */
    for (cPtr = from+5, found=0; cPtr && !found; cPtr = strchr(cPtr, ' ')) {
	for (i=0; i<7 && strncmp(cPtr+1, dayName[i], 3); i++);
	if (i < 7) {
	    for (i=0; i<12; i++) {
		if (!strncmp(cPtr+5, monthName[i], 3)) {
		    found = 1;
		    break;
		}
	    }
	}
    }
    if (!found) {
	return NULL;
    }
    elt.month = i+1;
    for (cPtr+=8; isspace(*cPtr) && *cPtr; cPtr++);
    if (!*cPtr) return NULL;
    elt.day = atoi(cPtr);
    for (cPtr++; !isspace(*cPtr) && *cPtr; cPtr++);
    for (cPtr++; isspace(*cPtr) && *cPtr; cPtr++);
    if (!*cPtr) return NULL;
    elt.hours = atoi(cPtr);
    for (cPtr++; ':' != *cPtr && *cPtr; cPtr++);
    elt.minutes = atoi(cPtr+1);
    for (cPtr++; isdigit(*cPtr) && *cPtr; cPtr++);
    if (!*cPtr) return NULL;
    if (':' == *cPtr) {
	elt.seconds = atoi(cPtr+1);
	for (cPtr++; isdigit(*cPtr) && *cPtr; cPtr++);
    } else {
	elt.seconds = 0;
    }
    while (1) {
	for (cPtr++; isspace(*cPtr) && *cPtr; cPtr++);
	if (isdigit(cPtr[0]) && isdigit(cPtr[1])
		&& isdigit(cPtr[2]) && isdigit(cPtr[3])){
	    elt.year = atoi(cPtr)-BASEYEAR;
	    break;
	} else {
	    for (cPtr++; !isspace(*cPtr) && *cPtr; cPtr++);
	}
	if (!*cPtr) return NULL;
    }
    elt.zoccident = 0;
    elt.zhours = 0;
    elt.zminutes = 0;
    return &elt;
}


/*
 *----------------------------------------------------------------------
 *
 * RatUpdateFolder --
 *
 *	Updates a folder
 *
 * Results:
 *	The number of new messages is left in the tcl result-buffer.
 *	A standard tcl-result is returned.
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatUpdateFolder(Tcl_Interp *interp, RatFolderInfo *infoPtr, RatUpdateType mode)
{
    int i, numNew, oldNumber, oldVisible, delta;

    oldVisible = infoPtr->visible;
    oldNumber = infoPtr->number;
    numNew = (*infoPtr->updateProc)(infoPtr, interp, mode);
    if (numNew < 0) {
	return TCL_ERROR;
    } else if (numNew != 0 || oldNumber != infoPtr->number
	    || infoPtr->sortOrderChanged) {
	if (infoPtr->number > infoPtr->allocated) {
	    infoPtr->allocated = infoPtr->number;
	    infoPtr->msgCmdPtr = (char **) ckrealloc(infoPtr->msgCmdPtr,
		    infoPtr->allocated*sizeof(char*));
	    infoPtr->privatePtr = (ClientData**)ckrealloc(infoPtr->privatePtr,
		    infoPtr->allocated*sizeof(ClientData*));
	    infoPtr->presentationOrder = (int *) ckrealloc(
		    infoPtr->presentationOrder,
		    infoPtr->allocated*sizeof(int));
	    infoPtr->hidden = (int *) ckrealloc(infoPtr->hidden,
		    infoPtr->allocated*sizeof(int));
	}
	for (i=infoPtr->number-numNew; i<infoPtr->number; i++) {
	    infoPtr->msgCmdPtr[i] = (char *) NULL;
	    infoPtr->privatePtr[i] = (ClientData*) NULL;
	    (*infoPtr->initProc)(infoPtr, interp, i);
	}
	RatFolderSort(interp, infoPtr);
	infoPtr->sortOrderChanged = 0;
    }
    delta = infoPtr->visible - oldVisible;
    Tcl_SetObjResult(interp, Tcl_NewIntObj((delta>0 ? delta : 0)));
    if (delta) {
	Tcl_SetVar2Ex(timerInterp, "folderExists", infoPtr->cmdName,
		Tcl_NewIntObj(infoPtr->number), TCL_GLOBAL_ONLY);
	Tcl_SetVar2Ex(timerInterp, "folderRecent", infoPtr->cmdName,
		Tcl_NewIntObj(infoPtr->recent), TCL_GLOBAL_ONLY);
	Tcl_SetVar2Ex(timerInterp, "folderUnseen", infoPtr->cmdName,
		Tcl_NewIntObj(infoPtr->unseen), TCL_GLOBAL_ONLY);
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatFolderUpdateTime --
 *
 *	Updates a folder
 *
 * Results:
 *	The number of new messages is left in the tcl result-buffer.
 *	A standard tcl-result is returned.
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */

static void
RatFolderUpdateTime(ClientData clientData)
{
    RatFolderInfo *infoPtr = (RatFolderInfo*)clientData;

    if (TCL_OK == RatUpdateFolder(timerInterp, infoPtr, RAT_UPDATE)) {
	infoPtr->timerToken = Tcl_CreateTimerHandler(
		infoPtr->watcherInterval*1000, RatFolderUpdateTime,
		(ClientData)infoPtr);
    }
}


/*
 *----------------------------------------------------------------------
 *
 * RatFolderClose --
 *
 *	Closes a folder
 *
 * Results:
 *	A standard tcl result
 *
 * Side effects:
 *	Many, all associated with cleaning up from the folder
 *
 *
 *----------------------------------------------------------------------
 */

int
RatFolderClose(Tcl_Interp *interp, RatFolderInfo *infoPtr, int force)
{
    RatFolderInfo **rfiPtrPtr;
    int i, ret, expunge;

    Tcl_GetBoolean(interp,
	   Tcl_GetVar2(interp, "option", "expunge_on_close", TCL_GLOBAL_ONLY),
	   &expunge);

    if (infoPtr->refCount-- != 1 && !force) {
	if (expunge) {
	    RatUpdateFolder(interp, infoPtr, RAT_SYNC);
	}
	return TCL_OK;
    }
    for (rfiPtrPtr = &ratFolderList; infoPtr != *rfiPtrPtr;
	    rfiPtrPtr = &(*rfiPtrPtr)->nextPtr);
    *rfiPtrPtr = infoPtr->nextPtr;
    ckfree(infoPtr->name);
    ckfree(infoPtr->definition);
    ret = (*infoPtr->closeProc)(infoPtr, interp, expunge);
    for(i=0; i < infoPtr->number; i++) {
	if (NULL != infoPtr->msgCmdPtr[i]) {
	    (void)RatMessageDelete(interp, infoPtr->msgCmdPtr[i]);
	    infoPtr->msgCmdPtr[i] = 0;
	}
    }
    if (infoPtr->watcherInterval) {
	Tcl_DeleteTimerHandler(infoPtr->timerToken);
    }
    Tcl_UnsetVar2(interp, "folderExists", infoPtr->cmdName,TCL_GLOBAL_ONLY);
    Tcl_UnsetVar2(interp, "folderUnseen", infoPtr->cmdName,TCL_GLOBAL_ONLY);
    (void)Tcl_DeleteCommand(interp, infoPtr->cmdName);
    ckfree(infoPtr->cmdName);
    ckfree(infoPtr->msgCmdPtr);
    ckfree(infoPtr->privatePtr);
    ckfree(infoPtr->presentationOrder);
    ckfree(infoPtr->hidden);
    ckfree(infoPtr);
    return ret;
}
