/*
 * Author:      Daniel Ratton Figueiredo <ratton@land.ufrj.br>
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-2000, William Chia-Wei Cheng.
 *
 * Permission limited to the use, copy, display, distribute without
 * charging for a fee, and produce derivative works of "tgif" and
 * its documentation for not-for-profit purpose is hereby granted by
 * the Author, provided that the above copyright notice appears in
 * all copies made of "tgif" and that both the copyright notice
 * and this permission notice appear in supporting documentation,
 * and that the name of the Author not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.  All other rights (including, but not limited to, the
 * right to sell "tgif", the right to sell or distribute derivative
 * works of "tgif", the right to distribute "tgif" for a fee, and
 * the right to include "tgif" or derivative works of "tgif" in a
 * for-sale product or service) are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * @(#)$Header: /mm/src/tgif/v4/RCS/tangram2.c,v 4.15 2000/02/21 17:02:51 william Exp $
 */

#define _INCLUDE_FROM_TANGRAM2_C_

#include "tgifdefs.h"

#include "attr.e"
#include "auxtext.e"
#include "box.e"
#include "cmd.e"
#include "dialog.e"
#include "exec.e"
#include "menu.e"
#include "msg.e"
#include "miniline.e"
#include "obj.e"
#include "pattern.e"
#include "select.e"
#include "setup.e"
#include "tangram2.e"
#include "util.e"

#ifdef _TANGRAM_II

#define DO_CMD(cmd) ExecACommandFromBuffer(cmd,NULL)

static TgMenuItemInfo tangram2MenuItemInfo[] = {
   { "About Tangram-II", NULL, "Information about Tangram-II", NULL,
     CMDID_ABOUT_TANGRAM2 },
   { "Generate Markov Chain", NULL, "Generate Markov Chain", NULL,
     CMDID_TANGRAM2_GENERATE_CHAIN },
   { NULL, NULL, NULL, NULL, INVALID }
};
TgMenuInfo tangram2MenuInfo={ TGMUTYPE_TEXT, tangram2MenuItemInfo,
                              CreateTangram2Menu };

/* ======================= Init & CleanUp ======================= */

void CleanUpTangram2()
{
   CleanUpTangram2ShortCut();
}

int InitTangram2()
{
   InitTangram2ShortCut();

   return TRUE;
}

/* ======================= Tangram2 Shortcuts ======================= */

static struct ShortCutRec tangram2ShortCutXlateTbl[] = {
   { '\0', 0, "AboutTangram2()", 0, CMDID_ABOUT_TANGRAM2 },
   { '\0', 0, "Tangram2GenerateChain()", 0, CMDID_TANGRAM2_GENERATE_CHAIN },
   { '\0', 0, "", 0, 0 }
};

static int tangram2ShortCutCmdIdIndex[MAXTANGRAM2CMDIDS-CMDID_TANGRAM2_BASE];

void CleanUpTangram2ShortCut()
{
}

int InitTangram2ShortCut()
{
   int i=0, num_cmdids=MAXTANGRAM2CMDIDS-CMDID_TANGRAM2_BASE;

   for (i=0; i < num_cmdids; i++) {
      tangram2ShortCutCmdIdIndex[i] = INVALID;
   }
   for (i=0; *(tangram2ShortCutXlateTbl[i].name) != '\0'; i++) {
      int cmdid=tangram2ShortCutXlateTbl[i].cmdid;

      if (cmdid != INVALID) {
         int index=cmdid-CMDID_TANGRAM2_BASE;

         if (tangram2ShortCutCmdIdIndex[index] == INVALID) {
            tangram2ShortCutCmdIdIndex[index] = i;
         } else {
            fprintf(stderr, "Warning:  duplicate cmdid %1d.\n", cmdid);
         }
      }
   }
   return TRUE;
}

int ValidTangram2CmdName(buf, len, pn_num_args)
   char *buf;
   int len, *pn_num_args;
   /*
    * returns 0 if no match
    * otherwise, returns (CMDID_TANGRAM2_BASE|index) where index is the
    *       index into tangram2ShortCutXlateTbl
    */
{
   int i;

   for (i=0; *(tangram2ShortCutXlateTbl[i].name) != '\0'; i++) {
      if (strncmp(tangram2ShortCutXlateTbl[i].name, buf, len) == 0) {
         *pn_num_args = tangram2ShortCutXlateTbl[i].num_args;

         return (i+CMDID_TANGRAM2_BASE);
      }
   }
   return 0;
}

int ValidTangram2CmdId(nCmdId)
   int nCmdId;
{
   int cmd_index=nCmdId-CMDID_TANGRAM2_BASE, xlate_index=INVALID;

   if (nCmdId >= MAXTANGRAM2CMDIDS || nCmdId < CMDID_TANGRAM2_BASE) {
      return FALSE;
   }
   xlate_index = tangram2ShortCutCmdIdIndex[cmd_index];
   if (xlate_index == INVALID ||
         tangram2ShortCutXlateTbl[xlate_index].num_args != 0) {
      return FALSE;
   }
   return TRUE;
}

int DoTangram2Cmd(nCmdId, args)
   int nCmdId;
   char *args;
   /* returns INVALID if the event can be caught by other windows */
{
   switch (nCmdId) {
   case CMDID_ABOUT_TANGRAM2: AboutTangram2(); break;
   case CMDID_TANGRAM2_GENERATE_CHAIN: Tangram2GenerateChain(); break;
   default: break;
   }
   return BAD;
}

int FetchTangram2ShortCutNumArgs(index, pn_num_args) 
   int index, *pn_num_args;
{  
   *pn_num_args = tangram2ShortCutXlateTbl[index].num_args;
   
   return TRUE;
}

int DoTangram2ShortCut(index, args)
   int index;
   char *args;
   /* return FALSE if cannot execute shortcut */
{
   if (index < 0 || index >= MAXTANGRAM2CMDIDS-CMDID_TANGRAM2_BASE) {
      return FALSE;
   }
   if (tangram2ShortCutXlateTbl[index].num_args == 0) {
      DoTangram2Cmd(tangram2ShortCutXlateTbl[index].cmdid, NULL);
   } else {
      DoTangram2Cmd(tangram2ShortCutXlateTbl[index].cmdid, args);
   }
   return TRUE;
}

/* ======================= Tangram2 Internal Commands ======================= */

void ExecStartSimulator ARGS_DECL((struct ObjRec *, char*));
void ExecSimulateStep ARGS_DECL((struct ObjRec *, char*));
void ExecEndSimulator ARGS_DECL((struct ObjRec *, char*));

static
ExecInfo gTangram2ExecInfo[] = {
   { (NLFN*)ExecStartSimulator,      "start_simulator",                   0, 0},
   { (NLFN*)ExecSimulateStep,        "simulate_step",                     0, 0},
   { (NLFN*)ExecEndSimulator,        "end_simulator",                     0, 0},
   { NULL, NULL, 0, 0 }
};

ExecInfo *Tangram2GetExecInfo(func_name)
   char *func_name;
{
   ExecInfo *pei=NULL;

   for (pei=gTangram2ExecInfo; pei->pfunc != NULL; pei++) {
      if (strcmp(pei->func_name, func_name) == 0) {
         return pei;
      }
   }
   return NULL;
}

/* --------------- Tangram2 Internal Commands Implementation --------------- */

#define MAXVARS              20
#define MAXVARNAME          250
#define MAXCONNECTTRIES      10
#define PORT               6743

#define END_OF_SIMULATION     1
#define STEP_SIMULATION       2

#define D printf ("Msg de Debug!\n");

typedef struct tagT_statevar {
  char name[MAXVARNAME];
  int  value;
} T_statevar;

typedef struct tagT_Packet {
  int code;
  int step;
  double elapsed_time;
  int varcount;
  T_statevar vars[MAXVARS];
} T_Packet;

static int simulator_socket=INVALID;

static
int BadAttr_Simulator(attr_name, cmd_name)
   char *attr_name, *cmd_name;
{
   char msg[MAXSTRING+1];

   sprintf(msg, "Can not find the '%s' %s '%s' command.",
         attr_name, "attribute while executing the", cmd_name);
   MsgBox(msg, TOOL_NAME, INFO_MB);
   return FALSE;
}

void ExecStartSimulator(obj_ptr, orig_cmd)
   char *orig_cmd;
   struct ObjRec *obj_ptr;
   /* start_simulator(); */
{
   char servidor[80];
   struct sockaddr_in endereco;
   struct hostent *hptr;
   int i;
   
   memset((char*)(&endereco), 0, sizeof(endereco)); 

   /* get the local host information - use localhost (127.0.0.1)  */
   /* gethostname(servidor, 80); */
   strcpy(servidor, "localhost");
   if ((hptr=gethostbyname(servidor)) != NULL) {
      memcpy((char*)&endereco.sin_addr, hptr->h_addr, hptr->h_length); 
   } else {
      perror("gethostbyname");
      return;
   }

   endereco.sin_family = AF_INET;
   endereco.sin_port = htons((u_short)PORT);

   if ((simulator_socket=socket(AF_INET, SOCK_STREAM, 0)) < 0) {
      perror("socket creation");
      return; 
   }

   /* connect to server. Try for MAXCONNECTTRIES times */
   i = 0;
   while (i < MAXCONNECTTRIES)
      if ((connect(simulator_socket, (struct sockaddr *)(&endereco),
            sizeof(endereco))) < 0) {
         perror("connect");
         printf("Still trying... (%d)\n",MAXCONNECTTRIES-i);
         close(simulator_socket);
         if ((simulator_socket=socket(AF_INET, SOCK_STREAM, 0)) < 0) {
           perror("socket creation");
           return; 
         }
         sleep(3);
         i++;
      } else {
         break;
      }
   if (i == MAXCONNECTTRIES) {
      perror("connect");
      printf("Bailing out... after %d tries\n", MAXCONNECTTRIES);
      return;
   }

   printf("Cliente Conectado.\n");
}

void ExecEndSimulator(obj_ptr, orig_cmd)
   char *orig_cmd;
   struct ObjRec *obj_ptr;
   /* end_simulator(); */
{
   T_Packet pack_out;
   int bywrite;

   if (simulator_socket == INVALID) {
      perror("invalid simulator_socket");
      return;
   }
   /* monta o pacote */
   memset((char *)&pack_out, 0, sizeof(T_Packet));
   pack_out.code = END_OF_SIMULATION;
    
   /* envia o pacote ao simulador */
   bywrite = write(simulator_socket, &pack_out, sizeof(T_Packet));
   if (bywrite < 0) {
      perror("write");
      return;
   }    
   close(simulator_socket);
   printf("TGIF: Simulation finished.\n");
}

static
void monta_buffer(char *buffer, T_Packet pack)
{
   int i;
   char temp[MAXSTRING+1];

   memset(buffer, 0, sizeof(buffer));
   for (i=0; i < pack.varcount; i++) {
      strcat(buffer, pack.vars[i].name);
      sprintf(temp, "=%d@", pack.vars[i].value);
      strcat(buffer, temp);
   }

   /* printf("monta: %s\n", buffer); */
}

static
int recv_packet(int soc, T_Packet *pack_in)
{
   int  byread, status;   /* bytes read from the socket */
   char buffer[sizeof(T_Packet)];

   /* clear and receive the packet */
   memset(buffer, 0, sizeof(T_Packet));
   byread = read(soc, buffer, sizeof(T_Packet));
   if (byread > 0) {
      while (byread < sizeof(T_Packet)) {
         status = read(soc, &(buffer[byread]), sizeof(T_Packet) - byread); 
         if (status > 0) {
            byread += status;
         } else {
            byread = status;
            break;
         }
      }      
   }
   memcpy((char*)pack_in, buffer, sizeof(T_Packet));

   if (byread <= 0) {
      perror("read");
      return(-1);
   } else {
      return(1);
   }
}

static
int send_packet(int soc, T_Packet pack_out)
{
   int  bywrite, status;
   char buffer[sizeof(T_Packet)];

   memcpy(buffer, (char*)&pack_out, sizeof(T_Packet));
   bywrite = write(soc, buffer, sizeof(T_Packet)); 
   if (bywrite > 0) {
      while (bywrite < sizeof(T_Packet)) {
         status = write(soc, &(buffer[bywrite]), sizeof(T_Packet) - bywrite); 
         if (status > 0) {
            bywrite += status;
         } else {
            bywrite = status;
            break;
         }
      }
   } 
   if (bywrite <= 0) {
      perror("write");
      return(-1);
   } else {
      return(1);
   }
}


static
void desmonta_buffer(char *buffer, T_Packet *pack)
{
   int i;
   char *token;

   /* printf("desmonta: %s\n", buffer); */

   i = 0;
   token = strtok(buffer, "=");
   strcpy(pack->vars[i].name, token);
   while (token != NULL) {
      token = strtok(NULL, "@");
      pack->vars[i].value = atoi(token);
      i++;
      token = strtok(NULL, "=");
      if (token == NULL) {
         break;
      }
      strcpy(pack->vars[i].name, token);
   }

   pack->varcount = i;
}

void ExecSimulateStep(obj_ptr, orig_cmd)
   char *orig_cmd;
   struct ObjRec *obj_ptr;
   /* simulate_step(); */
{
   char buffer[MAXVARS*sizeof(T_statevar)]; 

   struct AttrRec *attr_ptr;
   struct ObjRec *attr_owner_obj=NULL;
   static char execDummyStr[MAXSTRING+1];

   struct ObjRec *owner_obj=NULL, *named_obj;
   static char obj_name[MAXSTRING+1];

   T_Packet pack_in, pack_out;

   static int transitions = 0;
   static double tempo = 0;
   int step = 1;

   if (simulator_socket == INVALID) {
      perror("invalid simulator_socket");
      return;
   }
   /* pega o valor do atributo buffer */
   strcpy(execDummyStr, "buffer=");
   attr_ptr = FindAttrWithName(obj_ptr, execDummyStr, &attr_owner_obj);
   if (attr_ptr == NULL) {BadAttr_Simulator(execDummyStr, orig_cmd); return; }
   strcpy(buffer, attr_ptr->attr_value.s);

   /* pega o valor do atributo step */
   strcpy(execDummyStr, "Step=");
   attr_ptr = FindAttrWithName(obj_ptr, execDummyStr, &attr_owner_obj);
   if (attr_ptr == NULL) { BadAttr_Simulator(execDummyStr, orig_cmd); return; }
   step = atoi(attr_ptr->attr_value.s);

   /*
    * step nao pode ser menor do que zero (se for zero entao esta com play_mode)
    */
   if (step <= 0) {
      step = 1;
   }
   /* monta o pacote */
   memset((char *)&pack_out, 0, sizeof(T_Packet));
   pack_out.code = STEP_SIMULATION;
   pack_out.step = step;
   desmonta_buffer(buffer, &pack_out);

   /* envia o pacote ao simulador */
   if (send_packet(simulator_socket, pack_out) < 0) {
      return;
   }
   /* recebe pacote de resposta do simulador */
   if (recv_packet(simulator_socket, &pack_in) < 0) {
      return;
   }
   /* desmonta o pacote */
   tempo = pack_in.elapsed_time;
   transitions = step;
   monta_buffer(buffer, pack_in);

   /* seta o valor do atributo buffer  */
   strcpy(execDummyStr, "buffer=");
   attr_ptr = FindAttrWithName(obj_ptr, execDummyStr, &attr_owner_obj);
   if (attr_ptr == NULL) { BadAttr_Simulator(execDummyStr, orig_cmd); return; }
   ReplaceAttrFirstValue(attr_owner_obj, attr_ptr, buffer);

   /* set o valor do step_time (tempo de duracao deste step) */
   strcpy(execDummyStr, "step_time=");
   attr_ptr = FindAttrWithName(obj_ptr, execDummyStr, &attr_owner_obj);
   if (attr_ptr == NULL) { BadAttr_Simulator(execDummyStr, orig_cmd); return; }
   sprintf(execDummyStr, "%.3f", tempo);
   ReplaceAttrFirstValue(attr_owner_obj, attr_ptr, execDummyStr);

   /* atualiza o valor do atributo Time do objeto "__START_SIMULATOR__" */
   strcpy(obj_name, "__START_SIMULATOR__");
   named_obj = FindObjWithName(botObj, obj_ptr, obj_name, FALSE, FALSE,
         &owner_obj, NULL);

   strcpy(execDummyStr, "Time=");
   attr_ptr = FindAttrWithName(named_obj, execDummyStr, &attr_owner_obj);
   if (attr_ptr == NULL) { BadAttr_Simulator(execDummyStr, orig_cmd); return; }
   tempo += atof(attr_ptr->attr_value.s);
   sprintf(execDummyStr, "%.3f", tempo);
   ReplaceAttrFirstValue(attr_owner_obj, attr_ptr, execDummyStr);

   /* atualiza o valor do atributo Transitions do mesmo objeto */
   strcpy(execDummyStr, "Transitions=");
   attr_ptr = FindAttrWithName(named_obj, execDummyStr, &attr_owner_obj);
   if (attr_ptr == NULL) { BadAttr_Simulator(execDummyStr, orig_cmd); return; }
   transitions += atof(attr_ptr->attr_value.s);
   sprintf(execDummyStr, "%d", transitions);
   ReplaceAttrFirstValue(attr_owner_obj, attr_ptr, execDummyStr);
}

/* ======================= Tangram2 Menu ======================= */

int RefreshTangram2Menu(menu)
   TgMenu *menu;
{
   int ok=TRUE;

   return ok;
}

TgMenu *CreateTangram2Menu(parent_menu, x, y, menu_info, status_str_xlated)
   TgMenu *parent_menu;
   int x, y;
   TgMenuInfo *menu_info;
   int status_str_xlated; /* ignored, always 0 */
{
   TgMenu *menu=TgCreateMenuFromMenuInfo(parent_menu, x, y, menu_info, FALSE);

   if (menu != NULL) {
      if (!RefreshTangram2Menu(menu)) {
         return TgDestroyMenu(menu, TRUE);
      }
   }
   return menu;
}

int Tangram2Menu(X, Y, TrackMenubar)
   int X, Y, TrackMenubar;
{
   int rc=INVALID;
   TgMenu *menu=(tangram2MenuInfo.create_proc)(NULL, X, Y, &tangram2MenuInfo,
         INVALID);

   activeMenu = MENU_TANGRAM2;
   if (menu != NULL) {
      rc = TgMenuLoop(menu);
      TgDestroyMenu(menu, TRUE);
   }
   return rc;
}

/* ======================= Tangram2 Menu Functions ======================= */

void AboutTangram2()
{
   sprintf(gszMsgBox, "%s\n\n%s %s.",
         "Welcome to Tangram-II!",
         "Brought to you by the Tangram-II team at the",
         "Universidade Federal do Rio de Janeiro");
   MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
}

static char *GenerateChainExec[] = {
  "disable_undo();",
  "get_current_file(\"str\");",
  "read_file_into_attr(\"| basename $(str) .obj\", \"basename\");",
  "read_file_into_attr(\"| head -1 $(basename).setup\", \"dbglevel\");",
  "read_file_into_attr(\"| head -2 $(basename).setup | tail -1\", \"maxstates\");",
  "read_file_into_attr(\"| tail -2 $(basename).setup | head -1\", \"output\");",
  "read_file_into_attr(\"| tail -1 $(basename).setup\", \"processID\");",
  "strcpy(file_name, \"$(basename).parser\");",
  "strcpy(str, \" \");",
  "write_attr_into_file(str, $(file_name));",
  "exec(write_model);",
  "exec(write_global_rewards);",
  "exec(write_indep_chains);",
  "strcpy(command, \"gramatica $(file_name) $(basename) $(dbglevel) > $(output)\");",
  "launch(command);",
  "strcpy(command, \"generate_chain $(basename) $(maxstates) $(dbglevel) >> $(output)\");",
  "launch(command);",
  "enable_undo();",
   NULL
};

static char *GenerateChainWriteModel[] = {
  "find_obj_names(obj_list,\"\",\"Declaration=*\");",
  "get_line_in_attr(total_objs,obj_list,0);",
  "for_i(i,1,$(total_objs),1,write_obj);",
  NULL
};

static char *GenerateChainErrorParts[] = {
  "message_box(NULL,\"Objects must have these four parts: Variables_and_parameters, Events, Messages and Rewards. $(str)\",\"Mode Error\", \"stop\");",
  "exec(stop);",
  NULL
};


static char *GenerateChainStop[] = {
  "enable_undo();",
  "stop();",
  NULL
};


static char *GenerateChainAppendGlobalRewards[] = {
  "append_attr_into_file(__GLOBAL_REWARDS__.global_rewards, $(file_name));",
 NULL
};


static char *GenerateChainWriteIndependentChains[] = {
  "strcpy(str, \"Independent_Chains {\");",
  "append_attr_into_file(str,$(file_name));",
  "is_attr(result, __INDEP_CHAINS__.independent_chains);",
  "if($(result) != 0, append_indep_chains, NULL);",
  "strcpy(str, \"}\");",
  "append_attr_into_file(str,$(file_name));",
  NULL
};

static char *GenerateChainAppendIndepChains[] = {
  "append_attr_into_file(__INDEP_CHAINS__.independent_chains, $(file_name));",
  NULL
};


static char *GenerateChainWriteGlobalRewards[] = {
  "strcpy(str, \"Global_Rewards {\");",
  "append_attr_into_file(str,$(file_name));",
  "is_attr(result, __GLOBAL_REWARDS__.global_rewards);",
  "if($(result) != 0, append_global_rewards, NULL);",
  "strcpy(str, \"}\");",
  "append_attr_into_file(str,$(file_name));",
  NULL
};

static char *GenerateChainWriteMaxValues[] = {
  "strcpy(max_file_name, \"$(basename).maxvalues\");",
  "strcpy(str, \"$(__SETUP__.max_values)\");",
  "write_attr_into_file(__SETUP__.max_values, $(max_file_name));",
 NULL
};


static char *GenerateChainWriteObj[] = {
  "get_line_in_attr(obj_name,obj_list,$(i));",
  "strcpy(str, \"Object_Desc $(obj_name) (\");",
  "append_attr_into_file(str,$(file_name));",
  "is_attr(result,$(obj_name).Declaration);",
  "strcpy(str,\"Missing Declaration part in object: $(obj_name).\");",
  "if($(result) == 0,error_parts,NULL);",
  "strcpy(str, \"Declaration {\");",
  "append_attr_into_file(str,$(file_name));",
  "append_attr_into_file($(obj_name).Declaration,$(file_name));",
  "strcpy(str, \"}\");",
  "append_attr_into_file(str,$(file_name));",
  "is_attr(result,$(obj_name).Initialization);",
  "strcpy(str,\"Missing Initialization part in object: $(obj_name).\");",
  "if($(result) == 0,error_parts,NULL);",
  "strcpy(str, \"Initialization {\");",
  "append_attr_into_file(str,$(file_name));",
  "append_attr_into_file($(obj_name).Initialization,$(file_name));",
  "strcpy(str, \"}\");",
  "append_attr_into_file(str,$(file_name));",
  "is_attr(result,$(obj_name).Events);",
  "strcpy(str,\"Missing Events part in object: $(obj_name).\");",
  "if($(result) == 0,error_parts,NULL);",
  "strcpy(str, \"Events {\");",
  "append_attr_into_file(str,$(file_name));",
  "append_attr_into_file($(obj_name).Events,$(file_name));",
  "strcpy(str, \"}\");",
  "append_attr_into_file(str,$(file_name));",
  "is_attr(result,$(obj_name).Messages);",
  "strcpy(str,\"Missing Messages part in object: $(obj_name).\");",
  "if($(result) == 0,error_parts,NULL);",
  "strcpy(str, \"Messages {\");",
  "append_attr_into_file(str,$(file_name));",
  "append_attr_into_file($(obj_name).Messages,$(file_name));",
  "strcpy(str, \"}\");",
  "append_attr_into_file(str,$(file_name));",
  "is_attr(result,$(obj_name).Rewards);",
  "strcpy(str,\"Missing Rewards part in object: $(obj_name).\");",
  "if($(result) == 0,error_parts,NULL);",
  "strcpy(str, \"Rewards {\");",
  "append_attr_into_file(str,$(file_name));",
  "append_attr_into_file($(obj_name).Rewards,$(file_name));",
  "strcpy(str, \"}\");",
  "append_attr_into_file(str,$(file_name));",
  "strcpy(str, \")\");",
  "append_attr_into_file(str,$(file_name));",
  NULL
};

static
void AppendLinesToAttr(attr_ptr, ppsz_lines)
   struct AttrRec *attr_ptr;
   char **ppsz_lines;
{
   char **ppsz=NULL;
   struct TextRec *text_ptr=attr_ptr->obj->detail.t;

   for (ppsz=ppsz_lines; *ppsz != NULL; ppsz++) {
      MiniLineInfo *pFirstMiniLine=NULL, *pLastMiniLine=NULL;

      CreateMiniLineFromString(*ppsz, &pFirstMiniLine, &pLastMiniLine);
      text_ptr->minilines.last->next = pFirstMiniLine;
      pFirstMiniLine->prev = text_ptr->minilines.last;
      text_ptr->minilines.last = pLastMiniLine;
      text_ptr->lines++;
   }
   RecalcTextMetrics(text_ptr, attr_ptr->obj->x, text_ptr->baseline_y);
   UpdTextBBox(attr_ptr->obj);
}

void Tangram2GenerateChain()
{
   int saved_fill=objFill, saved_pen=penPat;
   struct ObjRec *tmp_box_obj=NULL;
   struct AttrRec *exec_attr=NULL;

   MakeQuiescent(); /* select nothing, edit nothing */

   objFill = penPat = NONEPAT;
   CreateBoxObj(0,0,10,10,TRUE);
   objFill = saved_fill;
   penPat = saved_pen;
   tmp_box_obj = topObj;

   /* Note: AddAttrByNameAndValue() adds the new object at tmp_box_obj->fattr */
   AddAttrByNameAndValue(tmp_box_obj, "i=", "");
   AddAttrByNameAndValue(tmp_box_obj, "dbglevel=", "");
   AddAttrByNameAndValue(tmp_box_obj, "maxstates=", "");
   AddAttrByNameAndValue(tmp_box_obj, "output=", "");
   AddAttrByNameAndValue(tmp_box_obj, "processID=", "");
   AddAttrByNameAndValue(tmp_box_obj, "name=", "__GENERATE_CHAIN__");
   AddAttrByNameAndValue(tmp_box_obj, "basename=", "");
   AddAttrByNameAndValue(tmp_box_obj, "total_objs=", "");
   AddAttrByNameAndValue(tmp_box_obj, "file_name=", "");
   AddAttrByNameAndValue(tmp_box_obj, "command=", "");
   AddAttrByNameAndValue(tmp_box_obj, "str=", "");
   AddAttrByNameAndValue(tmp_box_obj, "result=", "");
   AddAttrByNameAndValue(tmp_box_obj, "obj_name=", "");
   AddAttrByNameAndValue(tmp_box_obj, "max_file_name=", "");
   AddAttrByNameAndValue(tmp_box_obj, "obj_list=", "");

   AddAttrByNameAndValue(tmp_box_obj, "exec=", "");
   AppendLinesToAttr(tmp_box_obj->fattr, GenerateChainExec);

   AddAttrByNameAndValue(tmp_box_obj, "write_model=", "");
   AppendLinesToAttr(tmp_box_obj->fattr, GenerateChainWriteModel);

   AddAttrByNameAndValue(tmp_box_obj, "write_obj=", "");
   AppendLinesToAttr(tmp_box_obj->fattr, GenerateChainWriteObj);

   AddAttrByNameAndValue(tmp_box_obj, "stop=", "");
   AppendLinesToAttr(tmp_box_obj->fattr, GenerateChainStop);

   AddAttrByNameAndValue(tmp_box_obj, "error_parts=", "");
   AppendLinesToAttr(tmp_box_obj->fattr, GenerateChainErrorParts);

   AddAttrByNameAndValue(tmp_box_obj, "write_max_values=", "");
   AppendLinesToAttr(tmp_box_obj->fattr, GenerateChainWriteMaxValues);

   AddAttrByNameAndValue(tmp_box_obj, "write_global_rewards=", "");
   AppendLinesToAttr(tmp_box_obj->fattr, GenerateChainWriteGlobalRewards);

   AddAttrByNameAndValue(tmp_box_obj, "append_global_rewards=", "");
   AppendLinesToAttr(tmp_box_obj->fattr, GenerateChainAppendGlobalRewards);

   AddAttrByNameAndValue(tmp_box_obj, "write_indep_chains=", "");
   AppendLinesToAttr(tmp_box_obj->fattr, GenerateChainWriteIndependentChains);

   AddAttrByNameAndValue(tmp_box_obj, "append_indep_chains=", "");
   AppendLinesToAttr(tmp_box_obj->fattr, GenerateChainAppendIndepChains);

   StartCompositeCmd();

   AdjObjBBox(tmp_box_obj);
   RecordNewObjCmd();

   AddNewSelObj(tmp_box_obj);

   exec_attr = FindAttrWithName(topSel->obj, "exec=", NULL);
   if (exec_attr != NULL) {
      DoExecLoop(topSel->obj, exec_attr);
   }
   MakeQuiescent();
   AddNewSelObj(tmp_box_obj);
   PrepareToRecord(CMD_DELETE, topSel, botSel, numObjSelected);
   UnlinkObj(topSel->obj);
   FreeObj(topSel->obj);
   RemoveAllSel();
   RecordCmd(CMD_DELETE, NULL, NULL, NULL, 0);

   EndCompositeCmd();
}

#endif /* _TANGRAM_II */

