/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-2000, William Chia-Wei Cheng.
 *
 * Permission limited to the use, copy, display, distribute without
 * charging for a fee, and produce derivative works of "tgif" and
 * its documentation for not-for-profit purpose is hereby granted by
 * the Author, provided that the above copyright notice appears in
 * all copies made of "tgif" and that both the copyright notice
 * and this permission notice appear in supporting documentation,
 * and that the name of the Author not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.  All other rights (including, but not limited to, the
 * right to sell "tgif", the right to sell or distribute derivative
 * works of "tgif", the right to distribute "tgif" for a fee, and
 * the right to include "tgif" or derivative works of "tgif" in a
 * for-sale product or service) are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * @(#)$Header: /mm/src/tgif/v4/RCS/ruler.c,v 4.10 2000/01/15 22:24:09 william Exp $
 */

#define _INCLUDE_FROM_RULER_C_

#include "tgifdefs.h"

#include "choice.e"
#include "cursor.e"
#include "dialog.e"
#include "font.e"
#include "grid.e"
#include "mainmenu.e"
#include "msg.e"
#include "raster.e"
#include "ruler.e"
#include "setup.e"
#include "strtbl.e"
#include "util.e"

#define INCH_H (RULER_W-2)

#define HALF_INCH_H 10
#define QUARTER_INCH_H 6
#define MIN_INCH_H 3

#define HALF_CM_H 8
#define MIN_CM_H 4

int showMeasurement=FALSE;
int showCrossHair=FALSE;

static float gfPixelsPerUnit=(float)1.0;
static float gfNumUnits=(float)1.0;
static float gfNumFracUnits=(float)1.0;

static char numUnitStr[80], baseUnitStr[80], unitStr[80], formatUnitStr[80];

static GC rulerGC;

static int oldXOff=(-1), oldYOff=(-1);
static int rulerLen=0;
static int hRulerJustRedrawn=TRUE, justUnFrozen=FALSE;
static int freezeMarkRulerText=FALSE, frozenXOff=0, frozenYOff=0;

void GetUnitSpec(buf)
   char *buf;
{
   if (*formatUnitStr == '\0') {
      sprintf(buf, "%s %s/%s",
            (*numUnitStr)=='\0' ? "1" : numUnitStr,
            (*baseUnitStr)=='\0' ? "pixel" : baseUnitStr,
            (*unitStr)=='\0' ? "pixel" : unitStr);
   } else {
      char buf1[80];

      FormatFloat(&gfNumFracUnits, buf1);
      sprintf(buf, "%s %s/%s;%s;%s",
            (*numUnitStr)=='\0' ? "1" : numUnitStr,
            (*baseUnitStr)=='\0' ? "pixel" : baseUnitStr,
            (*unitStr)=='\0' ? "pixel" : unitStr, formatUnitStr, buf1);
   }
}

void ShowUnitMsg()
{
   sprintf(gszMsgBox, TgLoadString(STID_MEASUREMENT_SHOWN_IN_UNIT),
         (*unitStr)=='\0' ? "pixel" : unitStr,
         (*numUnitStr)=='\0' ? "1" : numUnitStr,
         (*baseUnitStr)=='\0' ? "pixel" : baseUnitStr);
   Msg(gszMsgBox);
}

static
int BadUnit(spec)
   char *spec;
{
   if (msgWindow != None) {
      sprintf(gszMsgBox, TgLoadString(STID_BAD_MEASUREMENT_UNIT_SPEC), spec);
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
   }
   return FALSE;
}

static
int VerifyFormatUnitStr(format_unit_str)
   char *format_unit_str;
{
   char *c_ptr=format_unit_str, *semi_ptr=NULL;
   int seek_percent=TRUE, count=0;

   if ((semi_ptr=strchr(format_unit_str, ';')) == NULL) {
      return FALSE;
   } else {
      float fval;

      *semi_ptr++ = '\0';
      UtilTrimBlanks(semi_ptr);
      if (sscanf(semi_ptr, "%f", &fval) != 1) {
         return FALSE;
      }
      gfNumFracUnits = fval;
   }
   for (c_ptr=format_unit_str; *c_ptr != '\0'; c_ptr++) {
      if (seek_percent) {
         if (*c_ptr == '%') {
            seek_percent = FALSE;
         }
      } else if (*c_ptr == 'f') {
         seek_percent = TRUE;
         count++;
      } else if (!(*c_ptr == '.' || (*c_ptr >= '0' && *c_ptr <= '9'))) {
         return FALSE;
      }
   }
   if (seek_percent && count==2) {
      strcpy(formatUnitStr, format_unit_str);
      return TRUE;
   }
   return FALSE;
}

int SetUnit(spec)
   char *spec;
{
   char *spec_copy=UtilStrDup(spec);
   char *num_ptr=NULL, *base_ptr=NULL, *unit_ptr=NULL, *semi_ptr=NULL;
   int ok=TRUE;

   if (spec_copy == NULL) {
      return FALSE;
   }
   if ((semi_ptr=strchr(spec_copy, ';')) != NULL) {
      *semi_ptr++ = '\0';
      if (!VerifyFormatUnitStr(semi_ptr)) {
         ok = BadUnit(spec);
      }
   } else {
      *formatUnitStr = '\0';
   }
   if ((num_ptr=strtok(spec_copy, " \t\n\r")) != NULL &&
         (base_ptr=strtok(NULL, "/ \t\n\r")) != NULL &&
         (unit_ptr=strtok(NULL, "/ \t\n\r")) != NULL) {
      float fval;

      if (sscanf(num_ptr, "%f", &fval) == 1 && fval > INT_TOL &&
            *base_ptr != '\0' && *unit_ptr != '\0') {
         gfNumUnits = fval;
         strcpy(numUnitStr, num_ptr);
         if (UtilStrICmp("pixel", unit_ptr) == 0) {
            *unitStr = '\0';
         } else {
            strcpy(unitStr, unit_ptr);
         }
         switch (*base_ptr) {
         case 'i':
         case 'I':
            gfPixelsPerUnit = gfNumUnits * ((float)ONE_INCH);
            strcpy(baseUnitStr, "in");
            break;
         case 'c':
         case 'C':
            gfPixelsPerUnit = gfNumUnits * ((float)ONE_CM);
            strcpy(baseUnitStr, "cm");
            break;
         case 'p':
         case 'P':
            gfPixelsPerUnit = gfNumUnits;
            *baseUnitStr = '\0';
            break;
         default: ok = BadUnit(spec); break;
         }
      } else {
         ok = BadUnit(spec);
      }
   } else {
      ok = BadUnit(spec);
   }
   if (!ok) {
      gfPixelsPerUnit = gfNumUnits = (float)1.0;
      *numUnitStr = *baseUnitStr = *unitStr = *formatUnitStr = '\0';
   }
   free(spec_copy);
   return ok;
}

void InitRuler()
{
   XGCValues values;
   char *c_ptr;

   values.foreground = myFgPixel;
   values.background = (threeDLook ? myLtGryPixel : myBgPixel);
   values.fill_style = FillSolid;
   values.font = rulerFontPtr->fid;

   rulerGC = XCreateGC(mainDisplay, mainWindow,
         GCForeground | GCBackground | GCFillStyle | GCFont, &values);

   showMeasurement = FALSE;
   if ((c_ptr=XGetDefault(mainDisplay, TOOL_NAME, "ShowMeasurement")) != NULL &&
         UtilStrICmp(c_ptr, "true") == 0) {
      showMeasurement = TRUE;
   }
   showCrossHair = FALSE;
/* if ((c_ptr=XGetDefault(mainDisplay, TOOL_NAME, "ShowCrossHair")) != NULL &&
         UtilStrICmp(c_ptr, "true") == 0) {
      showCrossHair = TRUE;
   } */

   gfPixelsPerUnit = gfNumUnits = (float)1.0;
   *numUnitStr = *baseUnitStr = *unitStr = *formatUnitStr = '\0';
   if ((c_ptr=XGetDefault(mainDisplay, TOOL_NAME, "ShowMeasurementUnit")) !=
         NULL) {
      char spec[80];

      if (strcmp("pixel", c_ptr) == 0) {
         strcpy(spec, "1 pixel/pixel");
      } else if (strcmp("inch", c_ptr) == 0) {
         sprintf(spec, "%1d pixel/in", ONE_INCH);
      } else if (strcmp("cm", c_ptr) == 0) {
         sprintf(spec, "%1d pixel/cm", ONE_CM);
      } else {
         strcpy(spec, c_ptr);
      }
      if (!SetUnit(spec)) {
         fprintf(stderr, TgLoadString(STID_INVALID_XDEF_USE_ALT_STR),
               TOOL_NAME, "ShowMeasurementUnit", spec, "pixel");
         fprintf(stderr, "\n");
      }
   }
   rulerLen = rulerW-1;
}

void CleanUpRuler()
{
   XFreeGC(mainDisplay, rulerGC);
}

void PixelToMeasurementUnit(Buf, NumPixels)
   char *Buf;
   int NumPixels;
{
   float fval;
   int ival;

   if (*unitStr == '\0') {
      sprintf(Buf, "%+1d", NumPixels);
   } else {
      fval = (float)((float)NumPixels*1000.0)/(gfPixelsPerUnit);
      ival = ((int)round(fval));
      fval = ((float)ival)/((float)1000.0);
      if (*formatUnitStr == '\0') {
         sprintf(Buf, "%+.3f%s", fval, unitStr);
      } else {
         float frac;

         ival = (int)fval;
         frac = ((ival >= 0) ? fval-((float)ival) : ((float)ival)-fval);
         fval = (float)ival;
         sprintf(Buf, formatUnitStr, fval, frac*gfNumFracUnits);
      }
   }
}

void RedrawHRuler(dpy, win)
   Display *dpy;
   Window win;
{
   register int i, pos, len, index;
   int inc, abs_inc, y=rulerW-(threeDLook ? (windowPadding>>1) : 0);
   char s[5];

   switch (gridSystem) {
   case ENGLISH_GRID:
      inc = (zoomedIn ? (xyEnglishGrid<<zoomScale) : xyEnglishGrid);
      abs_inc = ABS_SIZE(inc);

      if (drawOrigX % abs_inc == 0) {
         i = 0;
         pos = 0;
      } else {
         i = ((int)(drawOrigX / abs_inc) + 1) * abs_inc - drawOrigX;
         pos = ZOOMED_SIZE(i);
      }
      pos--; /* since drawWindow has no border */
      for ( ; i<drawWinW; i+=abs_inc, pos+=inc) {
         if ((GRID_ZOOMED_SIZE(i+drawOrigX)) % ONE_INCH == 0) {
            index = (int)((GRID_ZOOMED_SIZE(i+drawOrigX)) / ONE_INCH);
            sprintf(s, "%1d", GRID_ABS_SIZE(index));
            len = strlen(s);
            XDrawString(dpy, win, rulerGC, pos+2, y-INCH_H+rulerFontAsc,
                  s, len);
            XDrawLine(dpy, win, defaultGC, pos, y, pos, y-INCH_H);
         } else if ((GRID_ZOOMED_SIZE(i+drawOrigX)) % HALF_INCH == 0) {
            XDrawLine(dpy, win, defaultGC, pos, y, pos, y-HALF_INCH_H);
         } else if ((GRID_ZOOMED_SIZE(i+drawOrigX)) % QUARTER_INCH == 0) {
            XDrawLine(dpy, win, defaultGC, pos, y, pos, y-QUARTER_INCH_H);
         } else {
            XDrawLine(dpy, win, defaultGC, pos, y, pos, y-MIN_INCH_H);
         }
      }
      break;
   case METRIC_GRID:
      inc = (zoomedIn ? (xyMetricGrid<<zoomScale) : xyMetricGrid);
      abs_inc = ABS_SIZE(inc);

      if (drawOrigX % abs_inc == 0) {
         i = 0;
         pos = 0;
      } else {
         i = ((int)(drawOrigX / abs_inc) + 1) * abs_inc - drawOrigX;
         pos = ZOOMED_SIZE(i);
      }
      pos--; /* since drawWindow has no border */
      for ( ; i<drawWinW; i+=abs_inc, pos+=inc) {
         if ((GRID_ZOOMED_SIZE(i+drawOrigX)) % ONE_CM == 0) {
            index = (int)((GRID_ZOOMED_SIZE(i+drawOrigX)) / ONE_CM);
            sprintf(s, "%1d", GRID_ABS_SIZE(index));
            len = strlen(s);
            XDrawString(dpy, win, rulerGC, pos+2, y-INCH_H+rulerFontAsc,
                  s, len);
            XDrawLine(dpy, win, defaultGC, pos, y, pos, y-INCH_H);
         } else if ((GRID_ZOOMED_SIZE(i+drawOrigX)) % FIVE_MM == 0) {
            XDrawLine(dpy, win, defaultGC, pos, y, pos, y-HALF_CM_H);
         } else {
            XDrawLine(dpy, win, defaultGC, pos, y, pos, y-MIN_CM_H);
         }
      }
      break;
   }
}

static
void DrawHRuleTick(XOff)
   int XOff;
{
   XDrawLine(mainDisplay, hRuleWindow, revDefaultGC, XOff, 0, XOff, rulerLen);
}

static
void DrawVRuleTick(YOff)
{
   XDrawLine(mainDisplay, vRuleWindow, revDefaultGC, 0, YOff, rulerLen, YOff);
}

void RedrawHRulerWindow()
{
   XEvent ev;

   XClearWindow(mainDisplay, hRuleWindow);

   XSync(mainDisplay, False);
   while (XCheckWindowEvent(mainDisplay, hRuleWindow, ExposureMask, &ev)) ;

   RedrawHRuler(mainDisplay, hRuleWindow);

   oldXOff = (-1);
   DrawHRuleTick(oldXOff-1);
   hRulerJustRedrawn = TRUE;
   justUnFrozen = FALSE;
}

void RedrawVRuler(dpy, win)
   Display *dpy;
   Window win;
{
   register int i, pos, len, index;
   int inc, abs_inc, x=rulerW-(threeDLook ? (windowPadding>>1) : 0);
   char s[5];

   switch (gridSystem) {
   case ENGLISH_GRID:
      inc = (zoomedIn ? (xyEnglishGrid<<zoomScale) : xyEnglishGrid);
      abs_inc = ABS_SIZE(inc);

      if (drawOrigY % abs_inc == 0) {
         i = 0;
         pos = 0;
      } else {
         i = ((int)(drawOrigY / abs_inc) + 1) * abs_inc - drawOrigY;
         pos = ZOOMED_SIZE(i);
      }
      pos--; /* since drawWindow has no border */
      for ( ; i<drawWinH; i+=abs_inc, pos+=inc) {
         if ((GRID_ZOOMED_SIZE(i+drawOrigY)) % ONE_INCH == 0) {
            index = (int)((GRID_ZOOMED_SIZE(i+drawOrigY)) / ONE_INCH);
            sprintf(s, "%1d", GRID_ABS_SIZE(index));
            len = strlen(s);
            XDrawString(dpy, win, rulerGC, 1, pos+rulerFontAsc+2, s, len);
            XDrawLine(dpy, win, defaultGC, x, pos, x-INCH_H, pos);
         } else if ((GRID_ZOOMED_SIZE(i+drawOrigY)) % HALF_INCH == 0) {
            XDrawLine(dpy, win, defaultGC, x, pos, x-HALF_INCH_H, pos);
         } else if ((GRID_ZOOMED_SIZE(i+drawOrigY)) % QUARTER_INCH == 0) {
            XDrawLine(dpy, win, defaultGC, x, pos, x-QUARTER_INCH_H, pos);
         } else {
            XDrawLine(dpy, win, defaultGC, x, pos, x-MIN_INCH_H, pos);
         }
      }
      break;
   case METRIC_GRID:
      inc = (zoomedIn ? (xyMetricGrid<<zoomScale) : xyMetricGrid);
      abs_inc = ABS_SIZE(inc);

      if (drawOrigY % abs_inc == 0) {
         i = 0;
         pos = 0;
      } else {
         i = ((int)(drawOrigY / abs_inc) + 1) * abs_inc - drawOrigY;
         pos = ZOOMED_SIZE(i);
      }
      pos--; /* since drawWindow has no border */
      for ( ; i<drawWinH; i+=abs_inc, pos+=inc) {
         if ((GRID_ZOOMED_SIZE(i+drawOrigY)) % ONE_CM == 0) {
            index = (int)((GRID_ZOOMED_SIZE(i+drawOrigY)) / ONE_CM);
            sprintf(s, "%1d", GRID_ABS_SIZE(index));
            len = strlen(s);
            XDrawString(dpy, win, rulerGC, 1, pos+rulerFontAsc+2, s, len);
            XDrawLine(dpy, win, defaultGC, x, pos, x-INCH_H, pos);
         } else if ((GRID_ZOOMED_SIZE(i+drawOrigY)) % FIVE_MM == 0) {
            XDrawLine(dpy, win, defaultGC, x, pos, x-HALF_CM_H, pos);
         } else {
            XDrawLine(dpy, win, defaultGC, x, pos, x-MIN_CM_H, pos);
         }
      }
      break;
   }
}

void RedrawVRulerWindow()
{
   XEvent ev;

   XClearWindow(mainDisplay, vRuleWindow);

   XSync(mainDisplay, False);
   while (XCheckWindowEvent(mainDisplay, vRuleWindow, ExposureMask, &ev)) ;

   RedrawVRuler(mainDisplay, vRuleWindow);

   if (!freezeMarkRulerText && showMeasurement && !hRulerJustRedrawn &&
         oldYOff != 0) {
      char x_buf[80], y_buf[80], buf[80];

      PixelToMeasurementUnit(x_buf, ABS_X(oldXOff));
      PixelToMeasurementUnit(y_buf, ABS_Y(oldYOff));
      sprintf(buf, "[%s,%s]", x_buf, y_buf);
      XDrawString(mainDisplay, hRuleWindow, revDefaultGC, 8, 2+defaultFontAsc,
            buf, strlen(buf));
      hRulerJustRedrawn = TRUE;
   }
   oldYOff = (-1);
   DrawVRuleTick(oldYOff-1);
}

void MarkRulers(XOff, YOff)
   int XOff, YOff;
{
   char x_buf[80], y_buf[80], buf[80];

   DrawHRuleTick(oldXOff-1);
   DrawVRuleTick(oldYOff-1);
   if (showCrossHair) {
      XDrawLine(mainDisplay, drawWindow, revDefaultGC, oldXOff, 0,
            oldXOff, ZOOMED_SIZE(drawWinH));
      XDrawLine(mainDisplay, drawWindow, revDefaultGC, 0, oldYOff,
            ZOOMED_SIZE(drawWinW), oldYOff);
   }
   if (hRulerJustRedrawn) {
      hRulerJustRedrawn= FALSE;
   } else if (!freezeMarkRulerText && showMeasurement) {
      if (justUnFrozen) {
         justUnFrozen = FALSE;
         PixelToMeasurementUnit(x_buf, ABS_X(frozenXOff));
         PixelToMeasurementUnit(y_buf, ABS_Y(frozenYOff));
         sprintf(buf, "[%s,%s]", x_buf, y_buf);
      } else {
         PixelToMeasurementUnit(x_buf, ABS_X(oldXOff));
         PixelToMeasurementUnit(y_buf, ABS_Y(oldYOff));
         sprintf(buf, "[%s,%s]", x_buf, y_buf);
      }
      XDrawString(mainDisplay, hRuleWindow, revDefaultGC, 8, 2+defaultFontAsc,
            buf, strlen(buf));
   }
   DrawHRuleTick(XOff-1);
   DrawVRuleTick(YOff-1);

   if (showCrossHair) {
      XDrawLine(mainDisplay, drawWindow, revDefaultGC, XOff, 0,
            XOff, ZOOMED_SIZE(drawWinH));
      XDrawLine(mainDisplay, drawWindow, revDefaultGC, 0, YOff,
            ZOOMED_SIZE(drawWinW), YOff);
   }
   if (!freezeMarkRulerText && showMeasurement) {
      PixelToMeasurementUnit(x_buf, ABS_X(XOff));
      PixelToMeasurementUnit(y_buf, ABS_Y(YOff));
      sprintf(buf, "[%s,%s]", x_buf, y_buf);
      XDrawString(mainDisplay, hRuleWindow, revDefaultGC, 8, 2+defaultFontAsc,
            buf, strlen(buf));
   }
   oldXOff = XOff;
   oldYOff = YOff;
}

void RedrawRulers()
{
   RedrawHRulerWindow();
   RedrawVRulerWindow();
}

void GetCrossHairPosition(pnXOff, pnYOff, pnShown)
   int *pnXOff, *pnYOff, *pnShown;
{
   if (pnXOff != NULL) *pnXOff = oldXOff;
   if (pnYOff != NULL) *pnYOff = oldYOff;
   if (pnShown != NULL) *pnShown = showCrossHair;
}

void RedrawCrossHair()
{
   if (showCrossHair) {
      XDrawLine(mainDisplay, drawWindow, revDefaultGC, oldXOff, 0,
            oldXOff, ZOOMED_SIZE(drawWinH));
      XDrawLine(mainDisplay, drawWindow, revDefaultGC, 0, oldYOff,
            ZOOMED_SIZE(drawWinW), oldYOff);
   }
}

void ToggleShowCrossHair()
{
   /* the cross-hair stuff is disabled -- cannot get it to work right */
   if (!showCrossHair) return;

   RedrawCrossHair();

/* showCrossHair = !showCrossHair; */
   showCrossHair = FALSE;
   RedrawRulers();
   if (showCrossHair) {
      Msg(TgLoadString(STID_SHOW_CROSSHAIR_ENABLED));
      SetNullCursor(drawWindow);
   } else {
      SetDefaultCursor(drawWindow);
      ShowCursor();
      Msg(TgLoadString(STID_SHOW_CROSSHAIR_DISABLED));
   }
}

void RulersEventHandler(input)
   XEvent *input;
{
   if (input->type == ButtonPress) {
      XButtonEvent *button_ev=(&input->xbutton);

      if (button_ev->button == Button1) {
         IncGrid();
      } else if (button_ev->button == Button2) {
         GridMenu(button_ev->x_root, button_ev->y_root, FALSE);
      } else if (button_ev->button == Button3) {
         DecGrid();
      }
   } else if (input->xany.window == vRuleWindow) {
      if (input->type == Expose) {
         RedrawVRulerWindow();
      } else if (input->type == EnterNotify) {
         SetMouseStatus(TgLoadCachedString(CSTID_INC_GRID_SIZE),
               TgLoadCachedString(CSTID_GRID_MENU),
               TgLoadCachedString(CSTID_DEC_GRID_SIZE));
      }
   } else if (input->xany.window == hRuleWindow) {
      if (input->type == Expose) {
         RedrawHRulerWindow();
      } else if (input->type == EnterNotify) {
         SetMouseStatus(TgLoadCachedString(CSTID_INC_GRID_SIZE),
               TgLoadCachedString(CSTID_GRID_MENU),
               TgLoadCachedString(CSTID_DEC_GRID_SIZE));
      }
   }
}

void FreezeMarkRulerText()
{
   freezeMarkRulerText = TRUE;
   frozenXOff = oldXOff;
   frozenYOff = oldYOff;
}

void UnFreezeMarkRulerText()
{
   freezeMarkRulerText = FALSE;
   justUnFrozen = TRUE;
}

static int oldLtX, oldLtY, oldRbX, oldRbY, oldMdX, oldMdY;

static
void DoIntervalRulers()
{
   if (oldLtX == oldRbX) {
      XDrawLine(mainDisplay, hRuleWindow, revDefaultGC, oldLtX-1, 0, oldLtX-1,
            rulerLen);
   } else if (oldLtX==oldMdX || oldRbX==oldMdX) {
      XDrawLine(mainDisplay, hRuleWindow, revDefaultGC, oldLtX-1, 0, oldLtX-1,
            rulerLen);
      XDrawLine(mainDisplay, hRuleWindow, revDefaultGC, oldRbX-1, 0, oldRbX-1,
            rulerLen);
   } else {
      XDrawLine(mainDisplay, hRuleWindow, revDefaultGC, oldLtX-1, 0, oldLtX-1,
            rulerLen);
      XDrawLine(mainDisplay, hRuleWindow, revDefaultGC, oldMdX-1, 0, oldMdX-1,
            rulerLen);
      XDrawLine(mainDisplay, hRuleWindow, revDefaultGC, oldRbX-1, 0, oldRbX-1,
            rulerLen);
   }
   if (oldLtY == oldRbY) {
      XDrawLine(mainDisplay, vRuleWindow, revDefaultGC, 0, oldLtY-1, rulerLen-1,
            oldLtY);
   } else if (oldLtY==oldMdY || oldRbY==oldMdY) {
      XDrawLine(mainDisplay, vRuleWindow, revDefaultGC, 0, oldLtY-1, rulerLen-1,
            oldLtY);
      XDrawLine(mainDisplay, vRuleWindow, revDefaultGC, 0, oldRbY-1, rulerLen-1,
            oldRbY);
   } else {
      XDrawLine(mainDisplay, vRuleWindow, revDefaultGC, 0, oldLtY-1, rulerLen,
            oldLtY-1);
      XDrawLine(mainDisplay, vRuleWindow, revDefaultGC, 0, oldMdY-1, rulerLen,
            oldMdY-1);
      XDrawLine(mainDisplay, vRuleWindow, revDefaultGC, 0, oldRbY-1, rulerLen,
            oldRbY-1);
   }
   if (showCrossHair) {
      XDrawLine(mainDisplay, drawWindow, revDefaultGC, oldRbX, 0,
            oldRbX, ZOOMED_SIZE(drawWinH));
      XDrawLine(mainDisplay, drawWindow, revDefaultGC, 0, oldRbY,
            ZOOMED_SIZE(drawWinW), oldRbY);
   }
}

void BeginIntervalRulers(ltx, lty, rbx, rby)
   int ltx, lty, rbx, rby;
{
   DrawHRuleTick(oldXOff-1);
   DrawVRuleTick(oldYOff-1);
   if (showCrossHair) {
      XDrawLine(mainDisplay, drawWindow, revDefaultGC, oldXOff, 0,
            oldXOff, ZOOMED_SIZE(drawWinH));
      XDrawLine(mainDisplay, drawWindow, revDefaultGC, 0, oldYOff,
            ZOOMED_SIZE(drawWinW), oldYOff);
   }
   oldLtX = ltx; oldLtY = lty;
   oldRbX = rbx; oldRbY = rby;
   oldMdX = (oldLtX+oldRbX)>>1;
   oldMdY = (oldLtY+oldRbY)>>1;
   if (showMeasurement) FreezeMarkRulerText();
   DoIntervalRulers();
}

void DrawIntervalRulers(ltx, lty, rbx, rby)
   int ltx, lty, rbx, rby;
{
   DoIntervalRulers();
   oldLtX = ltx; oldLtY = lty;
   oldRbX = rbx; oldRbY = rby;
   oldMdX = (oldLtX+oldRbX)>>1; oldMdY = (oldLtY+oldRbY)>>1;
   DoIntervalRulers();
   if (showMeasurement) UnFreezeMarkRulerText();
}

void EndIntervalRulers(x, y)
   int x, y;
{
   DoIntervalRulers();
   oldXOff = x;
   oldYOff = y;
   DrawHRuleTick(oldXOff-1);
   DrawVRuleTick(oldYOff-1);
   if (showCrossHair) {
      XDrawLine(mainDisplay, drawWindow, revDefaultGC, oldXOff, 0,
            oldXOff, ZOOMED_SIZE(drawWinH));
      XDrawLine(mainDisplay, drawWindow, revDefaultGC, 0, oldYOff,
            ZOOMED_SIZE(drawWinW), oldYOff);
   }
   MarkRulers(x, y);
   frozenXOff = oldXOff;
   frozenYOff = oldYOff;
}

void StartShowMeasureCursor(XOff, YOff, Str, ExtraSpace)
   int XOff, YOff, ExtraSpace;
   char *Str;
{
   if (!showMeasurement) return;
   MarkRulers(XOff, YOff);
   FreezeMarkRulerText();
   if (Str != NULL && *Str != '\0') {
      int x=(ExtraSpace ? XOff+18 : XOff+4), y=YOff+defaultFontAsc;
      char *c_ptr=strchr(Str, '\n'), *line=Str;

      while (line != NULL) {
         if (c_ptr != NULL) *c_ptr = '\0';
         XDrawString(mainDisplay, drawWindow, revDefaultGC, x, y,
               line, strlen(line));
         if (c_ptr != NULL) {
            *c_ptr++ = '\n';
            line = c_ptr;
            c_ptr = strchr(line, '\n');
            y += defaultFontHeight;
         } else {
            break;
         }
      }
   }
}

void ShowMeasureCursor(XOff, YOff, Str, ExtraSpace)
   int XOff, YOff, ExtraSpace;
   char *Str;
{
   if (!showMeasurement) return;
   if (Str != NULL && *Str != '\0') {
      int x=(ExtraSpace ? XOff+18 : XOff+4), y=YOff+defaultFontAsc;
      char *c_ptr=strchr(Str, '\n'), *line=Str;

      while (line != NULL) {
         if (c_ptr != NULL) *c_ptr = '\0';
         XDrawString(mainDisplay, drawWindow, revDefaultGC, x, y,
               line, strlen(line));
         if (c_ptr != NULL) {
            *c_ptr++ = '\n';
            line = c_ptr;
            c_ptr = strchr(line, '\n');
            y += defaultFontHeight;
         } else {
            break;
         }
      }
   }
}

void EndShowMeasureCursor(XOff, YOff, Str, ExtraSpace)
   int XOff, YOff, ExtraSpace;
   char *Str;
{
   if (!showMeasurement) return;
   if (Str != NULL && *Str != '\0') {
      int x=(ExtraSpace ? XOff+18 : XOff+4), y=YOff+defaultFontAsc;
      char *c_ptr=strchr(Str, '\n'), *line=Str;

      while (line != NULL) {
         if (c_ptr != NULL) *c_ptr = '\0';
         XDrawString(mainDisplay, drawWindow, revDefaultGC, x, y,
               line, strlen(line));
         if (c_ptr != NULL) {
            *c_ptr++ = '\n';
            line = c_ptr;
            c_ptr = strchr(line, '\n');
            y += defaultFontHeight;
         } else {
            break;
         }
      }
   }
   UnFreezeMarkRulerText();
}

void ToggleShowMeasurement()
{
   showMeasurement = !showMeasurement;
   RedrawRulers();
   if (showMeasurement) {
      Msg(TgLoadString(STID_SHOW_MEASUREMENT_ENABLED));
   } else {
      Msg(TgLoadString(STID_SHOW_MEASUREMENT_DISABLED));
   }
   UpdatePinnedMenu(MENU_LAYOUT);
}

