/***************************************************************************
 *   copyright            : (C) 2003 by Pascal Brachet & Trolltech         *
 *                                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SPECIALICONVIEW_H
#define SPECIALICONVIEW_H

#ifndef QT_H
#include <qscrollview.h>
#include <qstring.h>
#include <qrect.h>
#include <qpoint.h>
#include <qsize.h>
#include <qfont.h>
#include <qbitmap.h>
#include <qpicture.h>
#endif // QT_H

class SpecialIconView;
class QPainter;
class QMimeSource;
class QMouseEvent;
class QKeyEvent;
class QFocusEvent;
class QShowEvent;
class SpecialIconViewItem;
class QStringList;

class SpecialIconViewItemPrivate;

class SpecialIconViewItem : public Qt
{
    friend class SpecialIconView;

public:
    SpecialIconViewItem( SpecialIconView *parent );
    SpecialIconViewItem( SpecialIconView *parent, SpecialIconViewItem *after );
    SpecialIconViewItem( SpecialIconView *parent, const QString &text );
    SpecialIconViewItem( SpecialIconView *parent, SpecialIconViewItem *after, const QString &text );
    SpecialIconViewItem( SpecialIconView *parent, const QString &text, const QPixmap &icon );
    SpecialIconViewItem( SpecialIconView *parent, SpecialIconViewItem *after, const QString &text, const QPixmap &icon );
#ifndef QT_NO_PICTURE
    SpecialIconViewItem( SpecialIconView *parent, const QString &text, const QPicture &picture );
    SpecialIconViewItem( SpecialIconView *parent, SpecialIconViewItem *after, const QString &text, const QPicture &picture );
#endif
    virtual ~SpecialIconViewItem();

    virtual void setRenameEnabled( bool allow );
    virtual void setDragEnabled( bool allow );
    virtual void setDropEnabled( bool allow );

    virtual QString text() const;
    virtual QPixmap *pixmap() const;
#ifndef QT_NO_PICTURE
    virtual QPicture *picture() const;
#endif
    virtual QString key() const;

    bool renameEnabled() const;
    bool dragEnabled() const;
    bool dropEnabled() const;

    SpecialIconView *iconView() const;
    SpecialIconViewItem *prevItem() const;
    SpecialIconViewItem *nextItem() const;

    int index() const;

    virtual void setSelected( bool s, bool cb );
    virtual void setSelected( bool s );
    virtual void setSelectable( bool s );

    bool isSelected() const;
    bool isSelectable() const;

    virtual void repaint();

    virtual bool move( int x, int y );
    virtual void moveBy( int dx, int dy );
    virtual bool move( const QPoint &pnt );
    virtual void moveBy( const QPoint &pnt );

    QRect rect() const;
    int x() const;
    int y() const;
    int width() const;
    int height() const;
    QSize size() const;
    QPoint pos() const;
    QRect textRect( bool relative = TRUE ) const;
    QRect pixmapRect( bool relative = TRUE ) const;
    bool contains( const QPoint& pnt ) const;
    bool intersects( const QRect& r ) const;

    virtual bool acceptDrop( const QMimeSource *mime ) const;

    virtual int compare( SpecialIconViewItem *i ) const;

    virtual void setText( const QString &text );
    virtual void setPixmap( const QPixmap &icon );
#ifndef QT_NO_PICTURE
    virtual void setPicture( const QPicture &icon );
#endif
    virtual void setText( const QString &text, bool recalc, bool redraw = TRUE );
    virtual void setPixmap( const QPixmap &icon, bool recalc, bool redraw = TRUE );
    virtual void setKey( const QString &k );

    virtual int rtti() const;
    static int RTTI;

protected:
    virtual void calcRect( const QString &text_ = QString::null );
    virtual void paintItem( QPainter *p, const QColorGroup &cg );
    virtual void paintFocus( QPainter *p, const QColorGroup &cg );
    virtual void dragEntered();
    virtual void dragLeft();
    void setItemRect( const QRect &r );
    void setTextRect( const QRect &r );
    void setPixmapRect( const QRect &r );
    void calcTmpText();

private:
    void init( SpecialIconViewItem *after = 0
#ifndef QT_NO_PICTURE
	       , QPicture *pic = 0
#endif
	       );
    void checkRect();

    SpecialIconView *view;
    QString itemText, itemKey;
    QString tmpText;
    QPixmap *itemIcon;
#ifndef QT_NO_PICTURE
    QPicture *itemPic;
#endif
    SpecialIconViewItem *prev, *next;
    uint allow_rename : 1;
    uint allow_drag : 1;
    uint allow_drop : 1;
    uint selected : 1;
    uint selectable : 1;
    uint dirty : 1;
    uint wordWrapDirty : 1;
    QRect itemRect, itemTextRect, itemIconRect;
    QRect oldRect;

    SpecialIconViewItemPrivate *d;private:
};

class SpecialIconViewPrivate;          /* don't touch */

class SpecialIconView : public QScrollView
{
    friend class SpecialIconViewItem;
    friend class SpecialIconViewPrivate;

    Q_OBJECT
    // #### sorting and sort direction do not work
    Q_ENUMS( SelectionMode ItemTextPos Arrangement ResizeMode )
    Q_PROPERTY( bool sorting READ sorting )
    Q_PROPERTY( bool sortDirection READ sortDirection )
    Q_PROPERTY( SelectionMode selectionMode READ selectionMode WRITE setSelectionMode )
    Q_PROPERTY( int gridX READ gridX WRITE setGridX )
    Q_PROPERTY( int gridY READ gridY WRITE setGridY )
    Q_PROPERTY( int spacing READ spacing WRITE setSpacing )
    Q_PROPERTY( ItemTextPos itemTextPos READ itemTextPos WRITE setItemTextPos )
    Q_PROPERTY( QBrush itemTextBackground READ itemTextBackground WRITE setItemTextBackground )
    Q_PROPERTY( Arrangement arrangement READ arrangement WRITE setArrangement )
    Q_PROPERTY( ResizeMode resizeMode READ resizeMode WRITE setResizeMode )
    Q_PROPERTY( int maxItemWidth READ maxItemWidth WRITE setMaxItemWidth )
    Q_PROPERTY( int maxItemTextLength READ maxItemTextLength WRITE setMaxItemTextLength )
    Q_PROPERTY( bool autoArrange READ autoArrange WRITE setAutoArrange )
    Q_PROPERTY( bool itemsMovable READ itemsMovable WRITE setItemsMovable )
    Q_PROPERTY( bool wordWrapIconText READ wordWrapIconText WRITE setWordWrapIconText )
    Q_PROPERTY( bool showToolTips READ showToolTips WRITE setShowToolTips )
    Q_PROPERTY( uint count READ count )

public:
    enum SelectionMode {
	Single = 0,
	Multi,
	Extended,
	NoSelection
    };
    enum Arrangement {
	LeftToRight = 0,
	TopToBottom
    };
    enum ResizeMode {
	Fixed = 0,
	Adjust
    };
    enum ItemTextPos {
	Bottom = 0,
	Right
    };

    SpecialIconView( QWidget* parent=0, const char* name=0, WFlags f = 0 );
    virtual ~SpecialIconView();

    virtual void insertItem( SpecialIconViewItem *item, SpecialIconViewItem *after = 0L );
    virtual void takeItem( SpecialIconViewItem *item );

    int index( const SpecialIconViewItem *item ) const;

    SpecialIconViewItem *firstItem() const;
    SpecialIconViewItem *lastItem() const;
    SpecialIconViewItem *currentItem() const;
    virtual void setCurrentItem( SpecialIconViewItem *item );
    virtual void setSelected( SpecialIconViewItem *item, bool s, bool cb = FALSE );

    uint count() const;

public:
    virtual void showEvent( QShowEvent * );

    virtual void setSelectionMode( SelectionMode m );
    SelectionMode selectionMode() const;

    SpecialIconViewItem *findItem( const QPoint &pos ) const;
    SpecialIconViewItem *findItem( const QString &text, ComparisonFlags compare = BeginsWith ) const;
    virtual void selectAll( bool select );
    virtual void clearSelection();
    virtual void invertSelection();

    virtual void repaintItem( SpecialIconViewItem *item );

    void ensureItemVisible( SpecialIconViewItem *item );
    SpecialIconViewItem* findFirstVisibleItem( const QRect &r ) const;
    SpecialIconViewItem* findLastVisibleItem( const QRect &r ) const;

    virtual void clear();

    virtual void setGridX( int rx );
    virtual void setGridY( int ry );
    int gridX() const;
    int gridY() const;
    virtual void setSpacing( int sp );
    int spacing() const;
    virtual void setItemTextPos( ItemTextPos pos );
    ItemTextPos itemTextPos() const;
    virtual void setItemTextBackground( const QBrush &b );
    QBrush itemTextBackground() const;
    virtual void setArrangement( Arrangement am );
    Arrangement arrangement() const;
    virtual void setResizeMode( ResizeMode am );
    ResizeMode resizeMode() const;
    virtual void setMaxItemWidth( int w );
    int maxItemWidth() const;
    virtual void setMaxItemTextLength( int w );
    int maxItemTextLength() const;
    virtual void setAutoArrange( bool b );
    bool autoArrange() const;
    virtual void setShowToolTips( bool b );
    bool showToolTips() const;

    void setSorting( bool sort, bool ascending = TRUE );
    bool sorting() const;
    bool sortDirection() const;

    virtual void setItemsMovable( bool b );
    bool itemsMovable() const;
    virtual void setWordWrapIconText( bool b );
    bool wordWrapIconText() const;

    bool eventFilter( QObject * o, QEvent * );

    QSize minimumSizeHint() const;
    QSize sizeHint() const;

    virtual void sort( bool ascending = TRUE );

    virtual void setFont( const QFont & );
    virtual void setPalette( const QPalette & );

    bool isRenaming() const;

public slots:
    virtual void arrangeItemsInGrid( const QSize &grid, bool update = TRUE );
    virtual void arrangeItemsInGrid( bool update = TRUE );
    virtual void setContentsPos( int x, int y );
    virtual void updateContents();

signals:
    void selectionChanged();
    void selectionChanged( SpecialIconViewItem *item );
    void currentChanged( SpecialIconViewItem *item );
    void clicked( SpecialIconViewItem * );
    void clicked( SpecialIconViewItem *, const QPoint & );
    void pressed( SpecialIconViewItem * );
    void pressed( SpecialIconViewItem *, const QPoint & );

    void doubleClicked( SpecialIconViewItem *item );
    void returnPressed( SpecialIconViewItem *item );
    void rightButtonClicked( SpecialIconViewItem* item, const QPoint& pos );
    void rightButtonPressed( SpecialIconViewItem* item, const QPoint& pos );
    void mouseButtonPressed( int button, SpecialIconViewItem* item, const QPoint& pos );
    void mouseButtonClicked( int button, SpecialIconViewItem* item, const QPoint& pos );
    void contextMenuRequested( SpecialIconViewItem* item, const QPoint &pos );
    void moved();
    void onItem( SpecialIconViewItem *item );
    void onViewport();
    void itemRenamed( SpecialIconViewItem *item, const QString & );
    void itemRenamed( SpecialIconViewItem *item );

protected slots:
    virtual void doAutoScroll();
    virtual void adjustItems();
    virtual void slotUpdate();

private slots:
    void clearInputString();
    void movedContents( int dx, int dy );

protected:
    void drawContents( QPainter *p, int cx, int cy, int cw, int ch );
    void contentsMousePressEvent( QMouseEvent *e );
    void contentsMouseReleaseEvent( QMouseEvent *e );
    void contentsMouseMoveEvent( QMouseEvent *e );
    void contentsMouseDoubleClickEvent( QMouseEvent *e );
    void contentsContextMenuEvent( QContextMenuEvent *e );

    void resizeEvent( QResizeEvent* e );
    void keyPressEvent( QKeyEvent *e );
    void focusInEvent( QFocusEvent *e );
    void focusOutEvent( QFocusEvent *e );
    void enterEvent( QEvent *e );

    virtual void drawRubber( QPainter *p );
    virtual void insertInGrid( SpecialIconViewItem *item );
    virtual void drawBackground( QPainter *p, const QRect &r );

    void emitSelectionChanged( SpecialIconViewItem * i = 0 );
    void emitRenamed( SpecialIconViewItem *item );

    SpecialIconViewItem *makeRowLayout( SpecialIconViewItem *begin, int &y, bool &changed );

    void styleChange( QStyle& );
    void windowActivationChange( bool );

private:
    void contentsMousePressEventEx( QMouseEvent *e );
    virtual void drawDragShapes( const QPoint &pnt );
    void drawContents( QPainter* );
    void findItemByName( const QString &text );
    void handleItemChange( SpecialIconViewItem *old, bool shift, bool control );

    int calcGridNum( int w, int x ) const;
    SpecialIconViewItem *rowBegin( SpecialIconViewItem *item ) const;
    void updateItemContainer( SpecialIconViewItem *item );
    void appendItemContainer();
    void rebuildContainers();
    QBitmap mask( QPixmap *pix ) const;

    SpecialIconViewPrivate *d;

private:	// Disabled copy constructor and operator=
#if defined(Q_DISABLE_COPY)
    SpecialIconView( const SpecialIconView & );
    SpecialIconView& operator=( const SpecialIconView & );
#endif
};

#endif
