///////////////////////////////////////////////////////////////////////
// File:        svmnode.cpp
// description_: ScrollView Menu Node
// Author:      Joern Wanke
// Created:     Thu Nov 29 2007
//
// (C) Copyright 2007, Google Inc.
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
///////////////////////////////////////////////////////////////////////
//
// A SVMenuNode is an entity which contains the mapping from a menu entry on
// the server side to the corresponding associated commands on the client.
// It is designed to be a tree structure with a root node, which can then be
// used to generate the appropriate messages to the server to display the
// menu structure there.
// A SVMenuNode can both be used in the context_ of popup menus as well as
// menu bars.
#include "svmnode.h"

#include <iostream>

#include "scrollview.h"

// Create the empty root menu node. with just a caption. All other nodes should
// be added to this or one of the submenus.
SVMenuNode::SVMenuNode() {
  cmd_event_ = -1;
  text_ = NULL;
  child_ = NULL;
  next_ = NULL;
  parent_ = NULL;
  toggle_value_ = false;
  is_check_box_entry_ = false;
  value_ = NULL;
  description_ = NULL;
}

SVMenuNode::~SVMenuNode() {
  delete[] text_;
//  delete[] description_;
}

// Create a new sub menu node with just a caption.  This is used to create
// nodes which act as parent nodes to other nodes (e.g. submenus).
SVMenuNode* SVMenuNode::AddChild(const char* txt) {
  SVMenuNode* s = new SVMenuNode(-1, txt, false, false, NULL, NULL);
  this->AddChild(s);
  return s;
}

// Create a "normal" menu node which is associated with a command event.
void SVMenuNode::AddChild(const char* txt, int command_event) {
  this->AddChild(new SVMenuNode(command_event, txt, false, false, NULL, NULL));
}

// Create a menu node with an associated value (which might be changed
// through the gui).
void SVMenuNode::AddChild(const char* txt, int command_event,
                          const char* val) {
  this->AddChild(new SVMenuNode(command_event, txt, false, false, val, NULL));
}

// Create a menu node with an associated value and description_.
void SVMenuNode::AddChild(const char* txt, int command_event, const char* val,
                          const char* desc) {
  this->AddChild(new SVMenuNode(command_event, txt, false, false, val, desc));
}

// Create a flag menu node.
void SVMenuNode::AddChild(const char* txt, int command_event, int tv) {
  this->AddChild(new SVMenuNode(command_event, txt, tv, true, NULL, NULL));
}

// Convenience function called from the different constructors to initialize
// the different values of the menu node.
SVMenuNode::SVMenuNode(int command_event, const char* txt,
                       int tv, bool check_box_entry, const char* val,
                       const char* desc) {
  cmd_event_ = command_event;

  text_ = new char[strlen(txt) + 1];
  strncpy(text_, txt, strlen(txt));
  text_[strlen(txt)] = '\0';

  value_ = val;
  description_ = desc;

  child_ = NULL;
  next_ = NULL;
  parent_ = NULL;
  toggle_value_ = tv != 0;
  is_check_box_entry_ = check_box_entry;
}

// Add a child node to this menu node.
void SVMenuNode::AddChild(SVMenuNode* svmn) {
  svmn->parent_ = this;
  // No children yet.
  if (child_ == NULL) {
    child_ = svmn;
  } else {
    SVMenuNode* cur = child_;
    while (cur->next_ != NULL) { cur = cur->next_; }
    cur->next_ = svmn;
  }
}

// Build a menu structure for the server and send the necessary messages.
// Should be called on the root node. If menu_bar is true, a menu_bar menu
// is built (e.g. on top of the window), if it is false a popup menu is
// built which gets shown by right clicking on the window.
// Deletes itself afterwards.
void SVMenuNode::BuildMenu(ScrollView* sv, bool menu_bar) {
  if ((parent_ != NULL) && (menu_bar)) {
    if (is_check_box_entry_) {
      sv->MenuItem(parent_->text_, text_, cmd_event_, toggle_value_);
    } else { sv->MenuItem(parent_->text_, text_, cmd_event_); }
  } else if ((parent_ != NULL) && (!menu_bar)) {
    if (description_ != NULL) { sv->PopupItem(parent_->text_, text_,
                                             cmd_event_, value_, description_);
      } else { sv->PopupItem(parent_->text_, text_); }
  }
  if (child_ != NULL) { child_->BuildMenu(sv, menu_bar); delete child_; }
  if (next_ != NULL) { next_->BuildMenu(sv, menu_bar); delete next_; }
}
