/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-dlg-handles-list.c:
 * Dialog to set a list of handles.
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-dlg-handles-list.h"
#include "ti-util.h"
#include "ti-config.h"
#include "ti-handle-list-editor.h"

#include <glade/glade.h>

G_DEFINE_TYPE (TIDlgHandlesList, ti_dlg_handles_list, G_TYPE_OBJECT);

/**
 * Instance private data.
 */
struct _TIDlgHandlesListPrivate {
    GtkWindow* parent;

    GladeXML* glade_xml;

    TIHandleListEditor* handle_list_editor;

    GtkWidget* dialog;
};
typedef struct _TIDlgHandlesListPrivate TIDlgHandlesListPrivate;

#define TI_DLG_HANDLES_LIST_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_DLG_HANDLES_LIST, TIDlgHandlesListPrivate))

/* Function prototypes */
static void _button_ok_clicked (GtkButton *button, gpointer user_data);
static void _button_cancel_clicked (GtkButton *button, gpointer user_data);
static void _ti_dlg_handles_list_setup (TIDlgHandlesList* dlg_handles_list);

/**
 * Drop all references to other objects.
 */
static void
ti_dlg_handles_list_dispose (GObject *object)
{
    TIDlgHandlesList *dlg_handles_list = TI_DLG_HANDLES_LIST (object);
    TIDlgHandlesListPrivate *priv = TI_DLG_HANDLES_LIST_GET_PRIVATE (dlg_handles_list);

    if (priv->glade_xml != NULL) {
        g_object_unref (priv->glade_xml);
        priv->glade_xml = NULL;
    }

    G_OBJECT_CLASS (ti_dlg_handles_list_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_dlg_handles_list_class_init (TIDlgHandlesListClass *ti_dlg_handles_list_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_dlg_handles_list_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_dlg_handles_list_dispose;
		
	/* Add private */
	g_type_class_add_private (ti_dlg_handles_list_class, sizeof (TIDlgHandlesListPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_dlg_handles_list_init (TIDlgHandlesList *ti_dlg_handles_list)
{
//    TIDlgHandlesListPrivate* priv = TI_DLG_HANDLES_LIST_GET_PRIVATE (ti_dlg_handles_list);
}

/**
 * Returns a new instance.
 */
TIDlgHandlesList*
ti_dlg_handles_list_new (GtkWindow* parent, TIHandleMapper* handle_mapper, const gchar* title) 
{
    TIDlgHandlesList* dialog = NULL;
    TIDlgHandlesListPrivate* priv = NULL;
    gchar* glade_file_path = NULL;

	dialog = g_object_new (TI_TYPE_DLG_HANDLES_LIST, NULL);

    priv = TI_DLG_HANDLES_LIST_GET_PRIVATE (dialog);
    priv->parent = parent;
    priv->handle_list_editor = TI_HANDLE_LIST_EDITOR (ti_handle_list_editor_new (handle_mapper));

    glade_file_path = g_strdup_printf ("%s%s", TI_DATA_DIR_PREFIX, "dlg-handles-list.xml");
    priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);
    if (priv->glade_xml == NULL)
    {
        g_critical ("Error loading glade file \"%s\".", glade_file_path);
        g_object_unref (dialog);
        dialog = NULL;
        goto CLEAN_UP;
    }

    _ti_dlg_handles_list_setup (dialog);

    gtk_window_set_title (GTK_WINDOW (priv->dialog), title);

    gtk_window_set_transient_for (GTK_WINDOW (priv->dialog), GTK_WINDOW (priv->parent));
    gtk_window_set_position (GTK_WINDOW (priv->dialog), GTK_WIN_POS_CENTER_ON_PARENT);

    glade_xml_signal_connect_data (priv->glade_xml, "button_ok_clicked", G_CALLBACK(_button_ok_clicked), priv->dialog);
    glade_xml_signal_connect_data (priv->glade_xml, "button_cancel_clicked", G_CALLBACK(_button_cancel_clicked), priv->dialog);

    CLEAN_UP:
    g_free (glade_file_path);

    return dialog;
}

/**
 * Run
 */
gboolean
ti_dlg_handles_list_run (TIDlgHandlesList* dlg_handles_list, GArray** handles)
{
    TIDlgHandlesListPrivate *priv = TI_DLG_HANDLES_LIST_GET_PRIVATE (dlg_handles_list);
    gint result;

    g_assert (handles != NULL);

    ti_handle_list_editor_set (priv->handle_list_editor, NULL);

    gtk_widget_show_all (priv->dialog);
    result = gtk_dialog_run (GTK_DIALOG (priv->dialog));
    gtk_widget_hide_all (priv->dialog);

    *handles = ti_handle_list_editor_get (priv->handle_list_editor);

    if (*handles == NULL)
    {
        return FALSE;
    }
    else
    {
        return result == GTK_RESPONSE_OK;
    }
}

/**
 * Called when OK button is clicked.
 */
static void
_button_ok_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_OK);
}

/**
 * Called when "Cancel" button is clicked.
 */
static void
_button_cancel_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_CANCEL);
}

/**
 * Setup
 */
static void
_ti_dlg_handles_list_setup (TIDlgHandlesList* dlg_handles_list)
{
    TIDlgHandlesListPrivate *priv = TI_DLG_HANDLES_LIST_GET_PRIVATE (dlg_handles_list);
    GtkWidget* widget;

    priv->dialog = glade_xml_get_widget (priv->glade_xml, "dialog_handles_list");
    g_assert (GTK_IS_DIALOG (priv->dialog));

    // Handle List Editor
    widget = glade_xml_get_widget (priv->glade_xml, "container_handle_list_editor");
    g_assert (widget != NULL && GTK_IS_CONTAINER (widget));
    gtk_container_add (GTK_CONTAINER (widget), GTK_WIDGET (priv->handle_list_editor));
}
