#   include	"config.h"

#   include	"bmintern.h"
#   include	<string.h>
#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Color reduction.							*/
/*									*/
/************************************************************************/

typedef struct HistogramEntry
    {
    int			heCount;
    RGB8Color		heColor;
    } HistogramEntry;

typedef struct	HashBucket
    {
    HistogramEntry	hbHistogramEntry;
    int			hbNumber;
    struct HashBucket *	hbNext;
    } HashBucket;

#   define	hbCount	hbHistogramEntry.heCount
#   define	hbRed	hbHistogramEntry.heColor.rgb8Red
#   define	hbGreen	hbHistogramEntry.heColor.rgb8Green
#   define	hbBlue	hbHistogramEntry.heColor.rgb8Blue

typedef struct ColorBox
    {
    int		cbFirst;
    int		cbColorCount;
    int		cbPixelCount;
    int		cbCutNumber;
    } ColorBox;

typedef struct CutNode
    {
    unsigned int	cnLeft;
    unsigned int	cnRight;
    unsigned char	cnValue;
    unsigned char	cnComponent;
    } CutNode;

#   define	CN_LEAF		0
#   define	CN_RED		1
#   define	CN_GREEN	2
#   define	CN_BLUE		3

/************************************************************************/
/*  Color hash varia.							*/
/************************************************************************/

#define HASH_SIZE 6553

#define ppm_hash(r,g,b)	((( (int) (r) * 33023 +    \
			    (int) (g) * 30013 +    \
			    (int) (b) * 27011 ) & 0x7fffffff) % HASH_SIZE )

static void bmFreeColorHash(	HashBucket **	hashTable	)
    {
    int			i;

    for ( i= 0; i < HASH_SIZE; i++ )
	{
	HashBucket *	hashBucket= hashTable[i];

	while( hashBucket )
	    {
	    HashBucket *	nx= hashBucket->hbNext;
	    free( (char *)hashBucket );
	    hashBucket= nx;
	    }
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Insert a new hash item: Convenience routine.			*/
/*									*/
/************************************************************************/

static int bmInsertHash(	HashBucket **	hashTable,
				int		hash,
				int		number,
				int		r,
				int		g,
				int		b )
    {
    HashBucket *	hashBucket= (HashBucket *)malloc( sizeof(HashBucket) );

    if  ( ! hashBucket )
	{ XDEB(hashBucket); return -1;	}

    hashBucket->hbNext= hashTable[hash];
    hashTable[hash]= hashBucket;

    hashBucket->hbCount= 1;
    hashBucket->hbNumber= number;

    hashBucket->hbRed= r;
    hashBucket->hbGreen= g;
    hashBucket->hbBlue= b;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Compare colors, used with qsort()					*/
/*									*/
/************************************************************************/

static int bmHistRedCompare( const void * v1, const void * v2 )
    {
    const	HistogramEntry *	he1= (HistogramEntry *)v1;
    const	HistogramEntry *	he2= (HistogramEntry *)v2;

    return he1->heColor.rgb8Red- he2->heColor.rgb8Red;
    }

static int bmHistGreenCompare( const void * v1, const void * v2 )
    {
    const	HistogramEntry *	he1= (HistogramEntry *)v1;
    const	HistogramEntry *	he2= (HistogramEntry *)v2;

    return he1->heColor.rgb8Green- he2->heColor.rgb8Green;
    }

static int bmHistBlueCompare( const void * v1, const void * v2 )
    {
    const	HistogramEntry *	he1= (HistogramEntry *)v1;
    const	HistogramEntry *	he2= (HistogramEntry *)v2;

    return he1->heColor.rgb8Blue- he2->heColor.rgb8Blue;
    }

/************************************************************************/
/*									*/
/*  Find the median for splitting a box.				*/
/*									*/
/************************************************************************/

static void bmSplitBox(	HistogramEntry *	histogram,
			int			colorCount,
			int			pixelCount,
			int *			pColorsLeft,
			int *			pPixelsLeft,
			int *			pComponent,
			int *			pValue		)
    {
    unsigned char	rMin, rMax;
    unsigned char	gMin, gMax;
    unsigned char	bMin, bMax;

    int			i;
    int			count;

    HistogramEntry *	he;

    rMin= rMax= histogram[0].heColor.rgb8Red;
    gMin= gMax= histogram[0].heColor.rgb8Green;
    bMin= bMax= histogram[0].heColor.rgb8Blue;

    he= histogram+ 1;
    for ( i= 1; i < colorCount; he++, i++ )
	{
	if  ( rMin > he->heColor.rgb8Red )
	    { rMin = he->heColor.rgb8Red;	}
	if  ( rMax < he->heColor.rgb8Red )
	    { rMax = he->heColor.rgb8Red;	}

	if  ( gMin > he->heColor.rgb8Green )
	    { gMin = he->heColor.rgb8Green;	}
	if  ( gMax < he->heColor.rgb8Green )
	    { gMax = he->heColor.rgb8Green;	}

	if  ( bMin > he->heColor.rgb8Blue )
	    { bMin = he->heColor.rgb8Blue;	}
	if  ( bMax < he->heColor.rgb8Blue )
	    { bMax = he->heColor.rgb8Blue;	}
	}

    if  ( 77* ( rMax- rMin ) > 150* ( gMax- gMin ) &&
	  77* ( rMax- rMin ) >  29* ( bMax- bMin ) )
	{
	qsort( histogram, colorCount, sizeof(HistogramEntry),
							bmHistRedCompare );
	*pComponent= CN_RED;
	}
    else{
	if  ( 150* ( gMax- gMin ) > 29* ( bMax- bMin ) )
	    {
	    qsort( histogram, colorCount, sizeof(HistogramEntry),
							bmHistGreenCompare );
	    *pComponent= CN_GREEN;
	    }
	else{
	    qsort( histogram, colorCount, sizeof(HistogramEntry),
							bmHistBlueCompare );
	    *pComponent= CN_BLUE;
	    }
	}

    he= histogram+ 1;
    count= histogram[0].heCount;
    for ( i= 1; i < colorCount- 1; he++, i++ )
	{
	if  ( count >= pixelCount/ 2 )
	    { break;	}

	count += he->heCount;
	}

    switch( *pComponent )
	{
	case CN_RED:	*pValue= he->heColor.rgb8Red;	break;
	case CN_GREEN:	*pValue= he->heColor.rgb8Green;	break;
	case CN_BLUE:	*pValue= he->heColor.rgb8Blue;	break;
	default:	LDEB(*pComponent);		break; /* shame */
	}

    *pColorsLeft= i;
    *pPixelsLeft= count;

    return;
    }

/************************************************************************/
/*									*/
/*  Find a collection of colors using 'median cut'.			*/
/*									*/
/************************************************************************/

static int bmFindColors(	HistogramEntry *	histogram,
				RGB8Color *		palette,
				CutNode *		cutNodes,
				int			colorCount,
				int			pixelCount,
				int			maxcolors )
    {
    ColorBox *			boxes;
    int				i;
    int				j;
    int				boxCount;
    int				nodeCount= 0;

    boxes= (ColorBox *)malloc( maxcolors* sizeof(ColorBox) );
    if  ( ! boxes )
	{ LLDEB(maxcolors,boxes); return -1; }

    boxes[0].cbFirst= 0;
    boxes[0].cbColorCount= colorCount;
    boxes[0].cbPixelCount= pixelCount;
    boxes[0].cbCutNumber= 0;
    boxCount= 1;

    cutNodes[0].cnComponent= CN_LEAF;
    cutNodes[0].cnValue= 0;		/*  Irrelevant !	*/
    cutNodes[0].cnLeft= 0;		/*  Both to this color.	*/
    cutNodes[0].cnRight= 0;		/*  Both to this color.	*/
    nodeCount= 1;

    while( boxCount < maxcolors )
	{
	int	biggestBox;
	int	colorsLeft;
	int	pixelsLeft;

	int	cutComponent;
	int	cutValue;

	for ( i= 0; i < boxCount; i++ )
	    {
	    if  ( boxes[i].cbColorCount > 1 )
		{ break;	}
	    }
	if  ( i >= boxCount )
	    { /* LLDEB(i,boxCount); */ break;	}
	biggestBox= i;
	for ( i= biggestBox+ 1; i < boxCount; i++ )
	    {
	    if  ( boxes[i].cbColorCount > 1				&&
		  boxes[i].cbPixelCount > boxes[biggestBox].cbPixelCount )
		{ biggestBox= i;	}
	    }

	bmSplitBox( histogram+ boxes[biggestBox].cbFirst,
				    boxes[biggestBox].cbColorCount,
				    boxes[biggestBox].cbPixelCount,
				    &colorsLeft, &pixelsLeft,
				    &cutComponent, &cutValue );

	/****************/
	/*  Cut Nodes	*/
	/****************/
	i= boxes[biggestBox].cbCutNumber;
	cutNodes[i].cnComponent= cutComponent;
	cutNodes[i].cnValue= cutValue;
	cutNodes[i].cnLeft= nodeCount;
	cutNodes[i].cnRight= nodeCount+ 1;

	cutNodes[nodeCount].cnComponent= CN_LEAF;
	cutNodes[nodeCount].cnValue= 0;		/*  Irrelevant !	*/
	cutNodes[nodeCount].cnLeft= biggestBox;	/*  Both to this color.	*/
	cutNodes[nodeCount].cnRight= biggestBox;/*  Both to this color.	*/

	cutNodes[nodeCount+1].cnComponent= CN_LEAF;
	cutNodes[nodeCount+1].cnValue= 0;	/*  Irrelevant !	*/
	cutNodes[nodeCount+1].cnLeft= boxCount;	/*  Both to this color.	*/
	cutNodes[nodeCount+1].cnRight= boxCount;/*  Both to this color.	*/

	/****************/
	/*  Boxes	*/
	/****************/
	boxes[boxCount].cbFirst= boxes[biggestBox].cbFirst+ colorsLeft;
	boxes[boxCount].cbColorCount=
				boxes[biggestBox].cbColorCount- colorsLeft;
	boxes[boxCount].cbPixelCount=
				boxes[biggestBox].cbPixelCount- pixelsLeft;
	boxes[boxCount].cbCutNumber= nodeCount+ 1;

	boxes[biggestBox].cbColorCount= colorsLeft;
	boxes[biggestBox].cbPixelCount= pixelsLeft;
	boxes[biggestBox].cbCutNumber= nodeCount;

	nodeCount += 2; boxCount++;
	}

    for ( i= 0; i < boxCount; i++ )
	{
	long			sR= 0;
	long			sG= 0;
	long			sB= 0;
	long			N= 0;
	HistogramEntry *	he= histogram+ boxes[i].cbFirst;

	for ( j= 0; j < boxes[i].cbColorCount; j++, he++ )
	    {
	    sR += he->heCount* he->heColor.rgb8Red;
	    sG += he->heCount* he->heColor.rgb8Green;
	    sB += he->heCount* he->heColor.rgb8Blue;
	    N  += he->heCount;
	    }
	palette[i].rgb8Red= sR/ N;
	palette[i].rgb8Green= sG/ N;
	palette[i].rgb8Blue= sB/ N;
	}

    free( (char *)boxes );

    return boxCount;
    }

/************************************************************************/
/*									*/
/*  Various translation routines.					*/
/*									*/
/************************************************************************/

static void bm24to8or4Cut(	int			pixelsHigh,
				int			pixelsWide,
				int			inBytesPerRow,
				int			outBytesPerRow,
				const unsigned char *	bufIn,
				unsigned char *		bufOut,
				CutNode *		cutNodes,
				int			bitsPerPixel )
    {
    int				row;
    int				col;
    int				i;

    unsigned char		r;
    unsigned char		g;
    unsigned char		b;

    const unsigned char *	from;
    unsigned char *		to;

    for ( row= 0; row < pixelsHigh; row++ )
	{
	from= bufIn+ row* inBytesPerRow;
	to= bufOut+ row* outBytesPerRow;

	for ( col= 0; col < pixelsWide; col++ )
	    {
	    r= *(from++); g= *(from++); b= *(from++);
	    i= 0;
	    for (;;)
		{
		switch( cutNodes[i].cnComponent )
		    {
		    case CN_RED:
			if  ( r < cutNodes[i].cnValue )
			    { i= cutNodes[i].cnLeft;	continue; }
			else{ i= cutNodes[i].cnRight;	continue; }
		    case CN_GREEN:
			if  ( g < cutNodes[i].cnValue )
			    { i= cutNodes[i].cnLeft;	continue; }
			else{ i= cutNodes[i].cnRight;	continue; }
		    case CN_BLUE:
			if  ( b < cutNodes[i].cnValue )
			    { i= cutNodes[i].cnLeft;	continue; }
			else{ i= cutNodes[i].cnRight;	continue; }
		    case CN_LEAF:
			break;
		    default:
			LDEB(cutNodes[i].cnComponent); return;
		    }
		break;
		}

	    switch( bitsPerPixel )
		{
		case 8:
		    *(to++)= cutNodes[i].cnLeft; break;
		case 4:
		    if  ( col % 2 )
			{ *(to++) |=  cutNodes[i].cnLeft;	 break; }
		    else{ *(to  )=  ( cutNodes[i].cnLeft << 4 ); break; }
		default:
		    LDEB(bitsPerPixel); return;
		}
	    }
	}
    }

static void bm24to8or4Hash(	int			pixelsHigh,
				int			pixelsWide,
				int			inBytesPerRow,
				int			outBytesPerRow,
				const unsigned char *	bufIn,
				unsigned char *		bufOut,
				HashBucket **		hashTable,
				int			bitsPerPixel )
    {
    int				row;
    int				col;

    unsigned char		r;
    unsigned char		g;
    unsigned char		b;

    const unsigned char *	from;
    unsigned char *		to;

    for ( row= 0; row < pixelsHigh; row++ )
	{
	from= bufIn+ row* inBytesPerRow;
	to= bufOut+ row* outBytesPerRow;

	for ( col= 0; col < pixelsWide; col++ )
	    {
	    int			hash;
	    HashBucket *	hashBucket;

	    r= *(from++); g= *(from++); b= *(from++);

	    hash= ppm_hash( r, g, b );
	    hashBucket= hashTable[hash];
	    while( hashBucket )
		{
		if  ( hashBucket->hbRed == r	&&
		      hashBucket->hbGreen == g	&&
		      hashBucket->hbBlue == b	)
		    { break;	}

		hashBucket= hashBucket->hbNext;
		}

	    if  ( ! hashBucket )
		{ XDEB(hashBucket); r= 0;	}
	    else{ r= hashBucket->hbNumber;	}

	    switch( bitsPerPixel )
		{
		case 8:
		    *(to++)= r; break;
		case 4:
		    if  ( col % 2 )
			{ *(to++) |=  r;        break; }
		    else{ *(to  )=  ( r << 4 ); break; }
		default:
		    LDEB(bitsPerPixel); return;
		}
	    }
	}
    }

/************************************************************************/
/*									*/
/*  Hash the colors in a 24 bit image. Do not return more than a	*/
/*  given number of colors.						*/
/*									*/
/*  1)  Make a hash table for the colors.				*/
/*									*/
/************************************************************************/

static int bmHashColors24(	HashBucket ***		pHashTable,
				int *			pColorCount,
				int			pixelsHigh,
				int			pixelsWide,
				int			bytesPerRow,
				const unsigned char *	bufIn,
				int			maxcolors,
				unsigned int		mask		)
    {
    unsigned int		r;
    unsigned int		g;
    unsigned int		b;

    int				row;
    int				col;

    HashBucket **		hashTable;
    HashBucket *		hashBucket;
    int				hash;

    int				colorCount= 0;

    const unsigned char *	from;

    /*  1  */
    hashTable= (HashBucket **)malloc( HASH_SIZE* sizeof(HashBucket *) );
    if  ( ! hashTable )
	{ XDEB(hashTable); return -1;	}
    for ( row= 0; row < HASH_SIZE; row++ )
	{ hashTable[row]= (HashBucket *)0;	}

    for ( row= 0; row < pixelsHigh; row++ )
	{
	from= bufIn+ row* bytesPerRow;

	for ( col= 0; col < pixelsWide; col++ )
	    {
	    r= *(from++) & mask; g= *(from++) & mask; b= *(from++) & mask;
	    hash= ppm_hash( r, g, b );

	    if  ( hash < 0 || hash >= HASH_SIZE )
		{ LLDEB(hash,HASH_SIZE); return -1; }

	    hashBucket= hashTable[hash];
	    while( hashBucket )
		{
		if  ( hashBucket->hbRed == r	&&
		      hashBucket->hbGreen == g	&&
		      hashBucket->hbBlue == b	)
		    { break;	}
		hashBucket= hashBucket->hbNext;
		}
	    if  ( hashBucket )
		{ hashBucket->hbCount++; continue;	}

	    if  ( bmInsertHash( hashTable, hash, colorCount++, r, g, b ) )
		{ LDEB(1); bmFreeColorHash( hashTable ); return -1; }

	    if  ( colorCount > maxcolors )
		{ bmFreeColorHash( hashTable ); return 1; }
	    }
	}

    *pColorCount= colorCount;
    *pHashTable= hashTable;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Convert a color hash table to a histogram.				*/
/*									*/
/************************************************************************/

static int bmHashToHistogram(	HistogramEntry **	pHistogram,
				HashBucket **		hashTable,
				int			colorCount )
    {
    HashBucket *		hashBucket;
    HistogramEntry *		histogram;

    int				row;
    int				i;

    /*  3  */
    histogram= (HistogramEntry *)malloc( colorCount* sizeof(HistogramEntry) );
    if  ( ! histogram )
	{ LLDEB(colorCount,histogram); return -1; }

    row= 0;
    for ( i= 0; i < HASH_SIZE; i++ )
	{
	hashBucket= hashTable[i];

	while( hashBucket )
	    {
	    histogram[row++]= hashBucket->hbHistogramEntry;
	    hashBucket= hashBucket->hbNext;
	    }
	}

    *pHistogram= histogram; return 0;
    }

static void bmHashToPalette(	RGB8Color *		palette,
				HashBucket **		hashTable )
    {
    HashBucket *		hashBucket;

    int				i;

    for ( i= 0; i < HASH_SIZE; i++ )
	{
	hashBucket= hashTable[i];

	while( hashBucket )
	    {
	    palette[hashBucket->hbNumber]=
					hashBucket->hbHistogramEntry.heColor;

	    hashBucket= hashBucket->hbNext;
	    }
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Reduce the number of colors in an image to at most 'maxcolors'.	*/
/*									*/
/*  0)  Allocate the palette.						*/
/*  1)  Make a hash table for the colors.				*/
/*  2)  Store and count all the colors in the hash table.		*/
/*  3)  If the number of colors is below the maximum, fill the palette	*/
/*	with the colors in the hash table.				*/
/*  4)  Else translate the hash table to a histogram.			*/

/*  4)  Allocate memory for the palette and for the tree that will be	*/
/*	used to classify the colors.					*/
/*  5)  Classify colors: Yields a palette and a tree.			*/
/*  6)  Allocate and initialise output.					*/
/*  7)  Translate to representative of box.				*/
/*									*/
/************************************************************************/

int bmColorReduce(	BitmapDescription *		bdOut,
			const BitmapDescription *	bdIn,
			unsigned char **		pBufOut,
			const unsigned char *		bufIn,
			int				maxcolors )
    {
    int				row;
    int				i;

    unsigned char		mask;

    HashBucket **		hashTable= (HashBucket **)0;
    HistogramEntry *		histogram= (HistogramEntry *)0;
    RGB8Color *			palette= (RGB8Color *)0;
    CutNode *			cutNodes= (CutNode *)0;

    int				colorCount= 0;
    int				colorsFound;

    int				bitsPerPixel;

    if  ( maxcolors > 256 )
	{ LDEB(maxcolors); return -1;	}

    if  ( maxcolors <= 16 )
	{ bitsPerPixel= 4;	}
    else{ bitsPerPixel= 8;	}

    /*  0  */
    palette= (RGB8Color *)malloc( maxcolors* sizeof(RGB8Color) );
    if  ( ! palette )
	{ LLDEB(maxcolors,palette); goto failure;	}

    memset( palette, 0, maxcolors* sizeof(RGB8Color) );

    /*  1, 2  */
    switch( bdIn->bdBitsPerPixel )
	{
	case 24:
	    for ( i= 0; i < 8; i++ )
		{
		mask= ~( 0xff >> ( 8- i ) );

		row= bmHashColors24( &hashTable, &colorCount,
				bdIn->bdPixelsHigh, bdIn->bdPixelsWide,
				bdIn->bdBytesPerRow, bufIn, 32768, mask );
		if  ( row < 0 )
		    { LDEB(row); return -1;	}
		if  ( row == 0 )
		    { break;	}
		}
	    break;
	default:
	    LDEB(bdIn->bdBitsPerPixel); return -1;
	}

    /*  3  */
    if  ( mask == 0xff && colorCount <= maxcolors )
	{
	bmHashToPalette( palette, hashTable );

	colorsFound= colorCount;
	}
    else{
	/*  4  */
	if  ( bmHashToHistogram( &histogram, hashTable, colorCount ) )
	    { LDEB(1); goto failure;	}

	cutNodes= (CutNode *)malloc( 2* maxcolors* sizeof(CutNode) );
	if  ( ! cutNodes )
	    { LLDEB(maxcolors,cutNodes); goto failure;	} 

	memset( cutNodes, 0, 2* maxcolors* sizeof(CutNode) );

	/*  5  */
	colorsFound= bmFindColors( histogram, palette, cutNodes, colorCount,
			    bdIn->bdPixelsHigh* bdIn->bdPixelsWide, maxcolors );

	if  ( colorsFound < 1 )
	    { LLDEB(maxcolors,colorsFound); goto failure; }
	}

    /*  6  */
    bdOut->bdPixelsWide= bdIn->bdPixelsWide;
    bdOut->bdPixelsHigh= bdIn->bdPixelsHigh;

    bdOut->bdBitsPerSample= bdIn->bdBitsPerSample;
    bdOut->bdSamplesPerPixel= 3;
    bdOut->bdBitsPerPixel= bitsPerPixel;
    bdOut->bdBytesPerRow= ( bdOut->bdBitsPerPixel* bdOut->bdPixelsWide+ 7 )/8;
    bdOut->bdBufferLength= bdOut->bdPixelsHigh* bdOut->bdBytesPerRow;

    bdOut->bdXResolution= bdIn->bdXResolution;
    bdOut->bdYResolution= bdIn->bdYResolution;
    bdOut->bdUnit= bdIn->bdUnit;

    bdOut->bdColorEncoding= BMcoRGB8PALETTE;
    bdOut->bdColorCount= colorsFound;

    bdOut->bdHasAlpha= 0;

    bdOut->bdRGB8Palette= palette;

    *pBufOut= malloc( bdOut->bdBufferLength );
    if  ( ! *pBufOut )
	{ LLDEB( bdOut->bdBufferLength,*pBufOut); goto failure; }

    /*  7  */
    switch( bdIn->bdBitsPerPixel )
	{
	case 24:
	    switch( bitsPerPixel )
		{
		case 8:
		case 4:
		    if  ( cutNodes )
			{
			bm24to8or4Cut( bdIn->bdPixelsHigh, bdIn->bdPixelsWide,
			    bdIn->bdBytesPerRow, bdOut->bdBytesPerRow,
			    bufIn, *pBufOut, cutNodes, bitsPerPixel );
			}
		    else{
			bm24to8or4Hash( bdIn->bdPixelsHigh, bdIn->bdPixelsWide,
			    bdIn->bdBytesPerRow, bdOut->bdBytesPerRow,
			    bufIn, *pBufOut, hashTable, bitsPerPixel );
			}
		    break;

		default:
		    LLDEB(bdIn->bdBitsPerPixel,bdOut->bdBitsPerPixel);
		    goto failure;
		}
	    break;
	default:
	    LLDEB(bdIn->bdBitsPerPixel,bdOut->bdBitsPerPixel);
	    goto failure;
	}

    if  ( histogram )
	{ free( histogram );	}
    if  ( cutNodes )
	{ free( cutNodes );	}
    if  ( hashTable )
	{ bmFreeColorHash( hashTable );	}

    return 0;

  failure:
    if  ( histogram )
	{ free( histogram );	}
    if  ( cutNodes )
	{ free( cutNodes );	}
    if  ( palette )
	{ free( palette );	}
    if  ( hashTable )
	{ bmFreeColorHash( hashTable );	}

    return -1;
    }

