#   include	"config.h"

#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>
#   include	<ctype.h>
#   include	<math.h>
#   include	<locale.h>

#   include	"docFont.h"
#   include	"psFont.h"
#   include	"appSystem.h"

#   include	<appDebugon.h>

#   ifndef	M_PI
#	define	M_PI	3.14159265358979323846
#   endif

/************************************************************************/
/*  A primitive table driven parser for AFM files.			*/
/************************************************************************/
#   define	AFMlenLINE	256

typedef struct AfmKeyword
    {
    char *	akString;
    int		(*akConsumer)(	FILE *		f,
				int		valPos,
				char *		input,
				AfmFontInfo *	afi );
    } AfmKeyword;

static int psAfmConsumeLines(	FILE *		f,
				char *		input,
				AfmFontInfo *   afi,
				AfmKeyword *	ak,
				int		keywordCount );

static int psSetEncoding(	AfmFontInfo *	afi,
				const char **	glyphNames,
				int		encoding,
				int		nameCount,
				int		complain );

static int psAfmIgnore(			FILE *, int, char *,AfmFontInfo *);
static int psAfmFontMetrics(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmFontName(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmFullName(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmFamilyName(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmWeight(			FILE *,	int, char *,AfmFontInfo *);
static int psAfmFontBBox(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmEncodingScheme(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmMappingScheme(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmCharacterSet(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmCharacters(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmIsFixedV(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmCapHeight(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmXHeight(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmAscender(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmDescender(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmStartDirection(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmUnderlinePosition(	FILE *,	int, char *,AfmFontInfo *);
static int psAfmUnderlineThickness(	FILE *,	int, char *,AfmFontInfo *);
static int psAfmItalicAngle(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmCharWidth(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmIsFixedPitch(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmStartCharMetrics(	FILE *,	int, char *,AfmFontInfo *);
static int psAfmStartKernData(		FILE *,	int, char *,AfmFontInfo *);
static int psAfmStartComposites(	FILE *,	int, char *,AfmFontInfo *);
static int psAfmStartTrackKern(		FILE *,	int, char *,AfmFontInfo *);

# define psAfmStdHW psAfmIgnore
# define psAfmStdVW psAfmIgnore

static AfmKeyword	psAfmFileKeywords[]=
    {
	{ "Comment",		psAfmIgnore,		},
	{ "StartFontMetrics",	psAfmFontMetrics,	},
    };

static AfmKeyword	psAfmMetricsKeywords[]=
    {
	{ "Comment",		psAfmIgnore,		},
	{ "FontName",		psAfmFontName,		},
	{ "FullName",		psAfmFullName,		},
	{ "FamilyName",		psAfmFamilyName,	},
	{ "Weight",		psAfmWeight,		},
	{ "FontBBox",		psAfmFontBBox,		},
	{ "Version",		psAfmIgnore,		},
	{ "Notice",		psAfmIgnore,		},
	{ "EncodingScheme",	psAfmEncodingScheme,	},
	{ "MappingScheme",	psAfmMappingScheme,	},
	{ "EscChar",		psAfmIgnore,		},
	{ "CharacterSet",	psAfmCharacterSet,	},
	{ "Characters",		psAfmCharacters,	},
	{ "IsBaseFont",		psAfmIgnore,		},
	{ "VVector",		psAfmIgnore,		},
	{ "IsFixedV",		psAfmIsFixedV,		},
	{ "CapHeight",		psAfmCapHeight,		},
	{ "XHeight",		psAfmXHeight,		},
	{ "Ascender",		psAfmAscender,		},
	{ "Descender",		psAfmDescender,		},
	{ "StartDirection",	psAfmStartDirection,	},
	{ "UnderlinePosition",	psAfmUnderlinePosition,	},
	{ "UnderlineThickness",	psAfmUnderlineThickness,},
	{ "ItalicAngle",	psAfmItalicAngle,	},
	{ "CharWidth",		psAfmCharWidth,		},
	{ "IsFixedPitch",	psAfmIsFixedPitch,	},
	{ "StartCharMetrics",	psAfmStartCharMetrics,	},
	{ "StartKernData",	psAfmStartKernData,	},
	{ "StartComposites",	psAfmStartComposites,	},
	{ "StdHW",		psAfmStdHW,		},
	{ "StdVW",		psAfmStdVW,		},
	{ "EndFontMetrics",	0,			},
    };

/************************************************************************/
/*  Initialise an AfmFontInfo						*/
/************************************************************************/
static void psInitAfmFontInfo(	AfmFontInfo *	afi )
    {
    afi->afiFontName= (char *)0;
    afi->afiFullName= (char *)0;
    afi->afiFamilyName= (char *)0;
    afi->afiWeight= (char *)0;

    afi->afiItalicAngle= 0.0;
    afi->afiTanItalicAngle= 0.0;

    afi->afiIsFixedPitch= 0;
    afi->afiFontBBox.abbLeft= 0;
    afi->afiFontBBox.abbBottom= 0;
    afi->afiFontBBox.abbRight= 0;
    afi->afiFontBBox.abbTop= 0;
    afi->afiUnderlinePosition= 0;
    afi->afiUnderlineThickness= 0;
    afi->afiEncodingScheme= (char *)0;
    afi->afiCapHeight= 0;
    afi->afiXHeight= 0;
    afi->afiAscender= 0;
    afi->afiDescender= 0;
    afi->afiCharacterSet= (char *)0;

    afi->afiMetricCount= 0;
    afi->afiMetrics= (AfmCharMetric *)0;

    return;
    }

static void psFreeAfmFontInfo(	AfmFontInfo *	afi )
    {
    if  ( afi->afiFontName )
	{ free( afi->afiFontName );	}
    if  ( afi->afiFullName )
	{ free( afi->afiFullName );	}
    if  ( afi->afiFamilyName )
	{ free( afi->afiFamilyName );	}
    if  ( afi->afiWeight )
	{ free( afi->afiWeight );	}
    if  ( afi->afiEncodingScheme )
	{ free( afi->afiEncodingScheme );}
    if  ( afi->afiCharacterSet )
	{ free( afi->afiCharacterSet );}

    if  ( afi->afiMetrics )
	{ free( afi->afiMetrics );}

    free( afi );

    return;
    }

static void psFontWidthFromName(	int *		pWidth,
					const char *	name )
    {
    int			width= FONTwidthNORMAL;
    const char *	s= name;

    while( s[0] == ' ' )
	{ s++;	}

    while( *s )
	{
	if  ( ! strncmp( s, "Narrow", 6 )	&&
	      ( s[6] == '\0' || s[6] == ' ' )	)
	    { width= FONTwidthNARROW;	}
	
	if  ( ! strncmp( s, "Condensed", 9 )	&&
	      ( s[9] == '\0' || s[9] == ' ' )	)
	    { width= FONTwidthCONDENSED;	}
	
	if  ( ! strncmp( s, "Extended", 8 )	&&
	      ( s[8] == '\0' || s[8] == ' ' )	)
	    { width= FONTwidthEXTENDED;	}

	while( s[0] && s[0] != ' ' )
	    { s++;	}
	while( s[0] == ' ' )
	    { s++;	}
	}

    *pWidth= width;
    }

/************************************************************************/
/*									*/
/*  Compare routine for sorting the faces of a font.			*/
/*									*/
/*  Serves two purposes:						*/
/*  1)  Present them in a predictable order in the font chooser.	*/
/*  2)  Makes it easier to present width varians as different families.	*/
/*									*/
/************************************************************************/

static int psFontCompareTypeFaces(	const void *	voitaft1,
					const void *	voitaft2 )
    {
    const AppFontTypeface *	aft1= (AppFontTypeface *)voitaft1;
    const AppFontTypeface *	aft2= (AppFontTypeface *)voitaft2;

    if  ( aft1->aftWidth > aft2->aftWidth )
	{ return  1;	}
    if  ( aft1->aftWidth < aft2->aftWidth )
	{ return -1;	}

    if  ( aft1->aftIsBold && aft1->aftIsSlanted		&&
	  ( ! aft2->aftIsBold || ! aft2->aftIsSlanted )	)
	{ return  1;	}

    if  ( ( ! aft1->aftIsBold || ! aft1->aftIsSlanted )	&&
	  aft2->aftIsBold && aft2->aftIsSlanted		)
	{ return -1;	}

    if  ( aft1->aftIsSlanted && ! aft2->aftIsSlanted )
	{ return  1;	}
    if  ( ! aft1->aftIsSlanted && aft2->aftIsSlanted )
	{ return -1;	}

    if  ( aft1->aftIsBold && ! aft2->aftIsBold )
	{ return  1;	}
    if  ( ! aft1->aftIsBold && aft2->aftIsBold )
	{ return -1;	}

    return 0;
    }

static int psFontFamilyText(	AppFontFamily *		aff )
    {
    AppFontTypeface *	aft;
    char *		text;
    int			l;
    AfmFontInfo *	afi;
    int			face;

    aft= aff->affFaces;
    afi= (AfmFontInfo *)aft->aftPrintingData;

    l= strlen( afi->afiFullName );
    text= (char *)malloc( l+ 1 );
    if  ( ! text )
	{ XDEB(text); return -1;	}
    strcpy( text, afi->afiFullName );

    for ( face= 0; face < aff->affFaceCount; aft++, face++ )
	{
	afi= (AfmFontInfo *)aft->aftPrintingData;

	while( strncmp( afi->afiFullName, text, l ) )
	    {
	    char *	past= strrchr( text, ' ' );

	    if  ( ! past )
		{ XDEB(past); return -1;	}
	    while( past > text && past[-1] == ' ' )
		{ past--;	}

	    *past= '\0'; l= past- text;
	    }

	if  ( afi->afiIsFixedPitch )
	    { aff->affHasFixedWidth= 1;		}
	else{ aff->affHasProportionalWidth= 1;	}
	}

    aff->affFontFamilyText= text;

    aft= aff->affFaces;;
    for ( face= 0; face < aff->affFaceCount; aft++, face++ )
	{
	afi= (AfmFontInfo *)aft->aftPrintingData;

	if  ( afi->afiFullName[l+ 1] )
	    { aft->aftFaceName= afi->afiFullName+ l+ 1;	}
	else{ aft->aftFaceName= afi->afiWeight;		}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Try to determine the character set of a font and the encoding.	*/
/*									*/
/************************************************************************/

typedef struct FontEncoding
    {
    const char *	feEncodingScheme;
    const char **	feGlyphNames;
    int			feGlyphCount;
    int			feEncoding;
    int			feMapToIso;
    } FontEncoding;

static FontEncoding PS_Encodings[]=
{
    /*  0  */	{ "ISOLatin1Encoding",		psIsoLatin1GlyphNames, 256,
						ENCODINGpsISO_8859_1,
						0,
						},
    /*  1  */	{ "AdobeStandardEncoding",	psIsoLatin1GlyphNames, 256,
						ENCODINGpsISO_8859_1,
						1,
						},
    /*  2  */	{ "FontSpecific",		psSymbolGlyphNames, 256,
						ENCODINGpsADOBE_SYMBOL,
						0,
						},
    /*  3  */	{ "FontSpecific",		psCyrillicGlyphNames, 256,
						ENCODINGpsADOBE_CYRILLIC,
						0,
						},
    /*  4  */	{ "FontSpecific",		psCyrillicGlyphNames,256,
						ENCODINGpsADOBE_CYRILLIC,
						0,
						},
    /*  5  */	{ "FontSpecific",		psIsoLatin2GlyphNames,256,
						ENCODINGpsISO_8859_2,
						0,
						},
    /*  6  */	{ "FontSpecific",		psDingbatGlyphNames, 256,
						ENCODINGpsADOBE_DINGBATS,
						0,
						},
};

static int psTryEncoding(	AfmFontInfo *	afi,
				FontEncoding *	fe )
    {
    const int complain= 0;

    if  ( ! psSetEncoding( afi, fe->feGlyphNames,
				fe->feEncoding, fe->feGlyphCount, complain ) )
	{ afi->afiMapToIso= fe->feMapToIso; return 0; }

    return -1;
    }

static int psGetFontEncoding(	AfmFontInfo *	afi )
    {
    int			i;
    FontEncoding *	fe;
    char *		encodingScheme= afi->afiEncodingScheme;

    if  ( ! encodingScheme )
	{ encodingScheme= "FontSpecific";	}

    fe= PS_Encodings;
    for ( i= 0; i < sizeof(PS_Encodings)/sizeof(FontEncoding); fe++, i++ )
	{
	if  ( ! strcmp( encodingScheme, fe->feEncodingScheme ) )
	    {
	    if  ( ! psTryEncoding( afi, fe ) )
		{ return 0;	}
	    }
	}

    fe= PS_Encodings;
    for ( i= 0; i < sizeof(PS_Encodings)/sizeof(FontEncoding); fe++, i++ )
	{
	if  ( ! psTryEncoding( afi, fe ) )
	    { return 0;	}
	}

    return -1;
    }

/************************************************************************/
/*									*/
/*  Get a line of input. (Essentially fgets())				*/
/*									*/
/************************************************************************/
static char *	psGetInputLine(	char *		s,
				int *		pLen,
				int		size,
				FILE *		stream )
    {
    s= fgets( s, size, stream );

    if  ( s )
	{
	int		lineLength= strlen( s );

	if  ( lineLength > 0 )
	    {
	    if  ( s[lineLength-1] == '\n' )
		{
		s[--lineLength]= '\0';

		if  ( lineLength > 0		&&
		      s[lineLength-1] == '\r'	)
		    { s[--lineLength]= '\0'; }
		}
	    }

	*pLen= lineLength;
	}

    return s;
    }

/************************************************************************/
/*									*/
/*  Make a catalog of postscript fonts for the font chooser.		*/
/*									*/
/*  The code assumes that afmDirectory is some kind of a resource, that	*/
/*  does not vary over different calls to the routine.			*/
/*									*/
/************************************************************************/
#   define	FILEL	400

static AppFontFamily *	EditFontPsFamilies= (AppFontFamily *)0;
static int		EditFontPsFamilyCount= 0;

static int PsFontSizes[]= { 8, 9, 10, 11, 12, 14, 16, 18, 24, 36, 48, 64 };

static int psGotAfmFile(	const char *	filename,
				void *		through )
    {
    int			fl;
    char		input[AFMlenLINE+1];
    FILE *		afmFile;
    int			res;

    char *		faceName;

    int			fam;
    int			face;

    AppFontFamily *	aff;
    AppFontTypeface *	aft;
    AfmFontInfo *	afi;

    int			len= strlen( filename );

    if  ( len == 0 )
	{ LDEB(len); return -1;	}

    afi= (AfmFontInfo *)malloc( sizeof(AfmFontInfo) );
    if  ( ! afi )
	{ XDEB(afi); return -1;	}

    psInitAfmFontInfo( afi );

    afmFile= fopen( filename, "r" );
    if  ( ! afmFile )
	{ SXDEB(filename,afmFile); psFreeAfmFontInfo( afi ); return 0; }

    res= psAfmConsumeLines( afmFile, input, afi, psAfmFileKeywords,
			sizeof(psAfmFileKeywords)/sizeof(AfmKeyword) );

    fclose( afmFile );

    if  ( res < 0 )
	{ SLDEB(filename,res); psFreeAfmFontInfo( afi ); return 0; }

    if  ( psGetFontEncoding( afi ) )
	{
	SSDEB(filename,afi->afiFullName);
	SDEB(afi->afiEncodingScheme);
	psFreeAfmFontInfo( afi ); return 0;
	}

    aff= EditFontPsFamilies;
    for ( fam= 0; fam < EditFontPsFamilyCount; aff++, fam++ )
	{
	if  ( ! strcmp( aff->affFontFamilyName, afi->afiFamilyName ) )
	    { break;	}
	}

    if  ( fam >= EditFontPsFamilyCount )
	{
	aff= (AppFontFamily *)
	    realloc( EditFontPsFamilies, (fam+1)* sizeof(AppFontFamily) );
	if  ( ! aff )
	    { LXDEB(fam,aff); return -1;	}
	EditFontPsFamilies= aff;

	aff += fam;

	docInitFontFamily( aff );

	aff->affFontFamilyName= afi->afiFamilyName; /*ASSIGN!*/

	EditFontPsFamilyCount= fam+ 1;
	}

    fl= strlen( afi->afiFamilyName );

    if  ( ! strncmp( afi->afiFamilyName, afi->afiFullName, fl )	&&
	  afi->afiFullName[fl] == '-'				)
	{ afi->afiFullName[fl]=   ' ';	}

    if  ( ! strncmp( afi->afiFamilyName, afi->afiFullName, fl )	&&
	  ( afi->afiFullName[fl] == ' '	||
	    afi->afiFullName[fl] == '\0'	)			)
	{
	faceName= afi->afiFullName+ fl;
	while( *faceName == ' ' )
	    { faceName++;	}
	if  ( ! * faceName )
	    { faceName= afi->afiWeight;	}
	}
    else{ SSDEB(afi->afiFamilyName,afi->afiFullName); return 0;	}

    aft= aff->affFaces;
    for ( face= 0; face < aff->affFaceCount; aft++, face++ )
	{
	if  ( ! strcmp( aft->aftFaceName, faceName )	)
	    { break;	}
	}

    if  ( face >= aff->affFaceCount )
	{
	aft= (AppFontTypeface *)
		realloc( aff->affFaces, (face+1)* sizeof(AppFontTypeface) );
	if  ( ! aft )
	    { LXDEB(face,aft); return -1;	}
	aff->affFaces= aft;

	aft += face;
	aft->aftFaceName= faceName;

	aft->aftIsScalable= 1;
	aft->aftSizes= PsFontSizes;
	aft->aftSizeCount= sizeof(PsFontSizes)/sizeof(int);
	aft->aftIsFixedWidth= afi->afiIsFixedPitch;
	aft->aftPrintingData= (void *)afi;
	aft->aftXQueryFormat= (char *)0;

	psFontWidthFromName( &(aft->aftWidth), afi->afiFullName+ fl );

	if  ( aft->aftIsFixedWidth )
	    {
	    if  ( aff->affHasProportionalWidth )
		{ SDEB(afi->afiFullName);	}
	    aff->affHasFixedWidth= 1;
	    }
	else{
	    if  ( aff->affHasFixedWidth )
		{ SDEB(afi->afiFullName);	}
	    aff->affHasProportionalWidth= 1;
	    }

	aft->aftIsBold= 0;
	if  ( ! strcmp( afi->afiWeight, "Bold" )	||
	      ! strcmp( afi->afiWeight, "Demi" )	)
	    { aft->aftIsBold= 1; }

	aft->aftIsSlanted= 0;
	if  ( afi->afiItalicAngle < -1.0 )
	    { aft->aftIsSlanted= 1; }

	aff->affFaceCount= face+ 1;
	}

    return 0;
    }

int psFontCatalog(	const char *		afmDirectory,
			AppFontFamily **	pFamilies,
			int *			pCount		)
    {
    FILE *			xfontDir;
    int				lineLength;

    int				j;

    int				fam;
    int				face;

    AppFontFamily *		aff;
    AppFontTypeface *		aft;
    AfmFontInfo *		afi;

    char			scratch[FILEL+ 1];

    if  ( EditFontPsFamilyCount > 0 )
	{
	*pFamilies= EditFontPsFamilies;
	*pCount= EditFontPsFamilyCount;
	return 0;
	}

    setlocale( LC_NUMERIC, "C" );

    j= appForAllFiles( afmDirectory, "afm", (void *)0, psGotAfmFile );

    setlocale( LC_NUMERIC, "" );

    if  ( j )
	{ SDEB(afmDirectory); return -1;	}

    aff= EditFontPsFamilies;
    for ( fam= 0; fam < EditFontPsFamilyCount; aff++, fam++ )
	{
	qsort( aff->affFaces, aff->affFaceCount,
			    sizeof(AppFontTypeface), psFontCompareTypeFaces );

	}

    fam= 0; aff= EditFontPsFamilies;
    while( fam < EditFontPsFamilyCount )
	{
	aft= aff->affFaces;
	for ( face= 0; face < aff->affFaceCount- 1; aft++, face++ )
	    {
	    if  ( aft[1].aftWidth != aft[0].aftWidth )
		{ break; }
	    }

	if  ( face < aff->affFaceCount- 1 )
	    {
	    AppFontFamily *	naff;
	    AppFontTypeface *	naft;

	    naff= (AppFontFamily *)realloc( EditFontPsFamilies,
			    (EditFontPsFamilyCount+1)* sizeof(AppFontFamily) );
	    if  ( ! naff )
		{ LXDEB(fam,naff); return -1;	}
	    EditFontPsFamilies= naff;
	    EditFontPsFamilyCount++;

	    naft= (AppFontTypeface *)malloc( (face+1)*sizeof(AppFontTypeface) );
	    if  ( ! naft )
		{ XDEB(naft); return -1;	}

	    for ( j= EditFontPsFamilyCount; j > fam; j-- )
		{ EditFontPsFamilies[j]= EditFontPsFamilies[j-1];	}

	    aff= EditFontPsFamilies+ fam;
	    docInitFontFamily( aff );

	    aff[0].affFaces= naft;
	    aft= aff[1].affFaces;
	    for ( j= 0; j < face+ 1; j++ )
		{ naft[j]= aft[j+face+1]; }
	    aff[0].affFaceCount= face+1;
	    aff[0].affWidth= naft[0].aftWidth;
	    aff[0].affFontFamilyName= aff[1].affFontFamilyName;

	    aft= aff[1].affFaces;
	    for ( j= face+ 1; j < aff->affFaceCount; j++ )
		{ aft[j]= aft[j+face+1];	}
	    aff[1].affFaceCount -= face+ 1;
	    aff[1].affWidth= aft[0].aftWidth;

	    if  ( psFontFamilyText( aff+ 0 ) )
		{ LDEB(1); return -1;	}
	    if  ( psFontFamilyText( aff+ 1 ) )
		{ LDEB(1); return -1;	}
	    }

	aff++; fam++;
	}

    sprintf( scratch, "%s/xfonts.dir", afmDirectory );
    xfontDir= fopen( scratch, "r" );
    if  ( xfontDir )
	{
	while( psGetInputLine( scratch, &lineLength, FILEL, xfontDir ) )
	    {
	    char *	s= scratch;

	    while( *s && ! isspace( *s ) )
		{ s++; }

	    if  ( ! *s )
		{ SDEB(scratch); continue;	}

	    *(s++)= '\0';
	    while( isspace( *s ) )
		{ s++;	}
	    if  ( ! *s )
		{ SDEB(scratch); continue;	}

	    aff= EditFontPsFamilies;
	    for ( fam= 0; fam < EditFontPsFamilyCount; aff++, fam++ )
		{
		aft= aff->affFaces;
		for ( face= 0; face < aff->affFaceCount; aft++, face++ )
		    {
		    afi= (AfmFontInfo *)aft->aftPrintingData;

		    if  ( ! strcmp( afi->afiFontName, scratch )	)
			{ break;	}
		    }

		if  ( face < aff->affFaceCount )
		    { break;	}
		}

	    if  ( fam < EditFontPsFamilyCount )
		{
		j= strlen( s );
		while( j > 1 && ( s[j-1] == '\n' || isspace( s[j-1] ) ) )
		    { j--; s[j]= '\0';	}

		aft->aftXQueryFormat= (char *)malloc( j+ 1 );
		if  ( ! aft->aftXQueryFormat )
		    { XDEB(aft->aftXQueryFormat); return -1;	}
		strcpy( aft->aftXQueryFormat, s );
		}
	    }

	fclose( xfontDir );
	}

    *pFamilies= EditFontPsFamilies;
    *pCount= EditFontPsFamilyCount;

    return 0;
    }

/************************************************************************/
/*  Consume a series of lines.						*/
/************************************************************************/
static int psAfmConsumeLines(	FILE *		f,
				char *		input,
				AfmFontInfo *	afi,
				AfmKeyword *	keywords,
				int		keywordCount )
    {
    for (;;)
	{
	int		k;
	int		b;
	int		p;
	AfmKeyword *	ak;

	if  ( ! psGetInputLine( input, &p, AFMlenLINE, f ) )
	    { return 1;	}

	b= 0;
	while( input[b] &&   isspace( input[b] ) )
	    { b++;	}

	if  ( ! input[b] )
	    { continue;	}

	p= b;
	while( input[p] && ! isspace( input[p] ) )
	    { p++;	}
	if  ( input[p] )
	    { input[p++]= '\0';	}

	while( isspace( input[p] ) )
	    { p++;	}

	ak= keywords;
	for ( k= 0; k < keywordCount; ak++, k++ )
	    {
	    if  ( ! strcmp( input+ b, ak->akString ) )
		{ break;	}
	    }

	if  ( k >= keywordCount )
	    { SDEB(input+ b); return -1;	}
	if  ( ! ak->akConsumer )
	    { return 0;	}

	if  ( (*ak->akConsumer)( f, p, input, afi ) )
	    { SDEB(input+ b); return -1;	}
	}
    }

/************************************************************************/
/*  Ignore an input line.						*/
/************************************************************************/
static int	psAfmIgnore(	FILE *		f,
				int		valPos,
				char *		input,
				AfmFontInfo *	afi )
    { return 0;	}

/************************************************************************/
/*  Consume font metrics.						*/
/*  1)  Not interested in the version.					*/
/************************************************************************/
static int psAfmFontMetrics(	FILE *		f,
				int		valPos,
				char *		input,
				AfmFontInfo *	afi )
    {
    return psAfmConsumeLines( f, input, afi, psAfmMetricsKeywords,
			    sizeof(psAfmMetricsKeywords)/sizeof(AfmKeyword) );
    }

/************************************************************************/
/*  Extract various types of values.					*/
/************************************************************************/

static int psAfmSaveString(	char **		pTarget,
				char *		source )
    {
    *pTarget= (char *)malloc( strlen( source )+ 1 );

    if  ( ! *pTarget )
	{ XDEB(*pTarget); return -1;	}

    strcpy( *pTarget, source );
    return 0;
    }

static int psAfmGetNumber(	double *	pNumber,
				char *		input,
				int		valPos )
    {
    char *	s= input+ valPos;
    char *	past;
    double	d;

    d= strtod( s, &past );
    if  ( past == s )
	{ SDEB(input+ valPos); return -1;	}
    s= past;

    while( isspace( *s ) )
	{ s++;	}

    if  ( *s )
	{ SDEB(input+ valPos); return -1;	}

    *pNumber= d;

    return 0;
    }

/************************************************************************/
/*  Extract various strings.						*/
/************************************************************************/
static int psAfmFontName(	FILE *		f,
				int		valPos,
				char *		input,
				AfmFontInfo *	afi )
    { return psAfmSaveString( &(afi->afiFontName), input+ valPos ); }

static int psAfmFullName(	FILE *		f,
				int		valPos,
				char *		input,
				AfmFontInfo *	afi )
    { return psAfmSaveString( &(afi->afiFullName), input+ valPos ); }

static int psAfmFamilyName(	FILE *		f,
				int		valPos,
				char *		input,
				AfmFontInfo *	afi )
    { return psAfmSaveString( &(afi->afiFamilyName), input+ valPos ); }

static int	psAfmWeight(	FILE *		f,
				int		valPos,
				char *		input,
				AfmFontInfo *	afi )
    { return psAfmSaveString( &(afi->afiWeight), input+ valPos ); }

static int	psAfmEncodingScheme(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    { return psAfmSaveString( &(afi->afiEncodingScheme), input+ valPos ); }

/************************************************************************/
/*  Unimplemented..							*/
/************************************************************************/

static int	psAfmBBox(	int		valPos,
				char *		input,
				AfmBBox *	abb )
    {
    char *	s= input+ valPos;
    char *	past;
    double	left, bottom, right, top;

    int		done= 0;

    left= strtod( s, &past );
    if  ( past == s )
	{ SSDEB(input,input+ valPos); return -1;	}
    done += past- s; s= past;

    bottom= strtod( s, &past );
    if  ( past == s )
	{ SSDEB(input,input+ valPos); return -1;	}
    done += past- s; s= past;

    right= strtod( s, &past );
    if  ( past == s )
	{ SSDEB(input,input+ valPos); return -1;	}
    done += past- s; s= past;

    top= strtod( s, &past );
    if  ( past == s )
	{ SSDEB(input,input+ valPos); return -1;	}
    done += past- s; s= past;

    abb->abbLeft= (int)left;
    abb->abbBottom= (int)bottom;
    abb->abbRight= (int)right;
    abb->abbTop= (int)top;

    return done;
    }

static int psAfmFontBBox(	FILE *		f,
				int		valPos,
				char *		input,
				AfmFontInfo *	afi )
    {
    AfmBBox	abb;
    int		res= psAfmBBox( valPos, input, &abb );
    char *	s;

    if  ( res < 0 )
	{ LDEB(res); return -1;	}

    s= input+ valPos+ res;
    while( isspace( *s ) )
	{ s++;	}
    if  ( *s )
	{ SSDEB(input,input+ valPos+ res); return -1;	}

    afi->afiFontBBox= abb;

    return 0;
    }

static int	psAfmMappingScheme(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    { LDEB(1); return -1; }

static int	psAfmCharacterSet(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    { return psAfmSaveString( &(afi->afiCharacterSet), input+ valPos ); }

static int	psAfmCharacters(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    {
    LDEB(1); return -1;
    }

static int	psAfmIsFixedV(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    {
    LDEB(1); return -1;
    }

static int	psAfmStartDirection(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    {
    LDEB(1); return -1;
    }

static int	psAfmCapHeight(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    {
    double	d;

    if  ( psAfmGetNumber( &d, input, valPos ) )
	{ LDEB(1); return -1;	}

    afi->afiCapHeight= (int)d; return 0;
    }

static int	psAfmXHeight(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    {
    double	d;

    if  ( psAfmGetNumber( &d, input, valPos ) )
	{ LDEB(1); return -1;	}

    afi->afiXHeight= (int)d; return 0;
    }

static int	psAfmAscender(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    {
    double	d;

    if  ( psAfmGetNumber( &d, input, valPos ) )
	{ LDEB(1); return -1;	}

    afi->afiAscender= (int)d; return 0;
    }

static int	psAfmDescender(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    {
    double	d;

    if  ( psAfmGetNumber( &d, input, valPos ) )
	{ LDEB(1); return -1;	}

    afi->afiDescender= (int)d; return 0;
    }

static int	psAfmUnderlinePosition(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    {
    double	d;

    if  ( psAfmGetNumber( &d, input, valPos ) )
	{ LDEB(1); return -1;	}

    afi->afiUnderlinePosition= (int)d; return 0;
    }

static int	psAfmUnderlineThickness(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    {
    double	d;

    if  ( psAfmGetNumber( &d, input, valPos ) )
	{ LDEB(1); return -1;	}

    afi->afiUnderlineThickness= (int)d; return 0;
    }

static int	psAfmItalicAngle(	FILE *		f,
					int		valPos,
					char *		input,
					AfmFontInfo *	afi )
    {
    if  ( psAfmGetNumber( &(afi->afiItalicAngle), input, valPos ) )
	{ return -1;	}

    afi->afiTanItalicAngle= tan( ( M_PI* afi->afiItalicAngle ) / 180  );

    return 0;
    }

static int	psAfmCharWidth(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    {
    LDEB(1); return -1;
    }

static int	psAfmIsFixedPitch(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    {
    if  ( ! strcmp( input+ valPos, "false" ) )
	{ afi->afiIsFixedPitch= 0; return 0;	}
    if  ( ! strcmp( input+ valPos, "true" ) )
	{ afi->afiIsFixedPitch= 1; return 0;	}

    SSDEB(input,input+ valPos); return -1;
    }

static int	psAfmStartCharMetrics(	FILE *		f,
					int		valPos,
					char *		input,
					AfmFontInfo *	afi )
    {
    int		lineLength;

    while( psGetInputLine( input, &lineLength, AFMlenLINE, f ) )
	{
	AfmCharMetric *	acm;
	char *		b;
	char *		p;

	int		C= -1;
	double		WX= 0.0;
	char *		N= NULL;
	AfmBBox		abb;

	abb.abbLeft= abb.abbRight= abb.abbTop= abb.abbBottom= 0;

	if  ( ! strncmp( input, "EndCharMetrics", 14 ) )
	    {
	    b= input+ 14;

	    while( isspace (*b ) )
		{ b++;	}

	    if  ( *b )
		{ SDEB(input); return -1;	}

	    return 0;
	    }

	b= input;
	while( isspace( *b ) )
	    { b++;	}

	while( *b )
	    {
	    p= b; while( *p && ! isspace( *p ) )
		{ p++;	}
	    if  ( isspace( *p ) )
		{ *(p++)= '\0';	}

	    if  ( ! strcmp( b, "C" ) )
		{
		b= p;
		C= strtol( b, &p, 10 );
		if  ( b == p )
		    { SDEB(b); return -1;	}
		goto nextToken;
		}

	    if  ( ! strcmp( b, "CX" ) )
		{
		b= p;
		C= strtol( b, &p, 16 );
		if  ( b == p )
		    { SDEB(b); return -1;	}
		while( isspace( *p ) )
		    { p++;	}
		if  ( *p != ';' )
		    { SDEB(b); return -1;	}
		b= p+ 1; continue;
		}

	    if  ( ! strcmp( b, "WX" ) )
		{
		b= p;
		WX= strtod( b, &p );
		if  ( b == p )
		    { SDEB(b); return -1;	}
		goto nextToken;
		}

	    if  ( ! strcmp( b, "N" ) )
		{
		b= p; while( isspace( *b ) )
		    { b++;	}
		p= b; while( *p && ! isspace( *p ) )
		    { p++;	}
		if  ( b == p )
		    { SDEB(b); return -1;	}
		N= (char *)malloc( p- b+ 1 );
		if  ( ! N )
		    { XDEB(N); return -1;	}
		strncpy( N, b, p- b )[p- b]= '\0';
		goto nextToken;
		}

	    if  ( ! strcmp( b, "L" ) )
		{
		b= p; while( isspace( *b ) )
		    { b++;	}
		p= b; while( *p && ! isspace( *p ) )
		    { p++;	}
		if  ( b == p )
		    { SDEB(b); return -1;	}
		b= p; while( isspace( *b ) )
		    { b++;	}
		p= b; while( *p && ! isspace( *p ) )
		    { p++;	}
		if  ( b == p )
		    { SDEB(b); return -1;	}
		goto nextToken;
		}

	    if  ( ! strcmp( b, "B" ) )
		{
		int	done;

		b= p;
		done= psAfmBBox( 0, b, &abb );

		if  ( done < 0 )
		    { SLDEB(b,done); return -1;	}

		p= b+ done; goto nextToken;
		}

	    SSDEB(b,N); return -1;

	    nextToken:
		while( isspace( *p ) )
		    { p++;	}
		if  ( *p != ';' )
		    { SDEB(b); return -1;	}
		b= p+ 1;
		while( isspace( *b ) )
		    { b++;	}
		continue;
	    }

	acm= (AfmCharMetric *)realloc( afi->afiMetrics,
		    ( afi->afiMetricCount+ 1 )* sizeof( AfmCharMetric ) );
	if  ( ! acm )
	    { LXDEB(C,acm); return -1;	}
	afi->afiMetrics= acm;

	acm += afi->afiMetricCount++;

	acm->acmC= C;
	acm->acmWX= (int)WX;
	acm->acmN= N;
	acm->acmKernPairs= (AfmKerningPair *)0;
	acm->acmKernPairCount= 0;
	acm->acmBBox= abb;
	}

    LDEB(AFMlenLINE); return -1;
    }

static int	psAfmStartKernPairs(	FILE *		f,
					int		valPos,
					char *		input,
					AfmFontInfo *	afi )
    {
    int		lineLength;

    while( psGetInputLine( input, &lineLength, AFMlenLINE, f ) )
	{
	AfmCharMetric *		acm;
	AfmKerningPair *	akp;
	char *			b;
	char *			p;

	char *	N1= (char *)0;
	char *	N2= (char *)0;
	double	XVec= 0.0;

	int	pos1;
	int	pos2;

	if  ( ! strncmp( input, "EndKernPairs", 12 ) )
	    {
	    b= input+ 12;

	    while( isspace (*b ) )
		{ b++;	}

	    if  ( *b )
		{ SDEB(input); return -1;	}

	    return 0;
	    }

	b= input;
	while( isspace( *b ) )
	    { b++;	}

	if  ( ! *b || *b == '\n' )
	    { continue;	}

	while( *b )
	    {
	    p= b; while( *p && ! isspace( *p ) )
		{ p++;	}
	    if  ( isspace( *p ) )
		{ *(p++)= '\0';	}

	    if  ( ! strcmp( b, "KPX" ) )
		{
		b= p; while( isspace( *b ) )
		    { b++;	}
		p= b; while( *p && ! isspace( *p ) )
		    { p++;	}
		if  ( b == p )
		    { SDEB(b); return -1;	}
		if  ( isspace( *p ) )
		    { *(p++)= '\0';	}
		N1= b;
		b= p; while( isspace( *b ) )
		    { b++;	}
		p= b; while( *p && ! isspace( *p ) )
		    { p++;	}
		if  ( b == p )
		    { SDEB(b); return -1;	}
		if  ( isspace( *p ) )
		    { *(p++)= '\0';	}
		N2= b;
		b= p;
		XVec= strtod( b, &p );
		if  ( b == p )
		    { SDEB(b); return -1;	}
		goto nextToken;
		}

	    SDEB(b); return -1;

	    nextToken:
		while( isspace( *p ) )
		    { p++;	}
		if  ( *p && *p != ';' )
		    { SDEB(b); return -1;	}
		if  ( *p == ';' )
		    { b= p+ 1; }
		else{ b= p   ; }

		while( isspace( *b ) )
		    { b++;	}
		continue;
	    }

	acm= afi->afiMetrics;
	for ( pos2= 0; pos2 < afi->afiMetricCount; acm++, pos2++ )
	    {
	    if  ( acm->acmN && ! strcmp( acm->acmN, N2 ) )
		{ break;	}
	    }

	acm= afi->afiMetrics;
	for ( pos1= 0; pos1 < afi->afiMetricCount; acm++, pos1++ )
	    {
	    if  ( acm->acmN && ! strcmp( acm->acmN, N1 ) )
		{ break;	}
	    }

	if  ( pos1 >= afi->afiMetricCount || pos2 >= afi->afiMetricCount )
	    { LLDEB(pos1,pos2); return -1;	}

	akp= (AfmKerningPair *)realloc( acm->acmKernPairs,
			( acm->acmKernPairCount+ 1 )* sizeof(AfmKerningPair) );
	if  ( ! akp )
	    { LXDEB(acm->acmKernPairCount,akp); return -1;	}
	acm->acmKernPairs= akp;

	akp += acm->acmKernPairCount++;
	akp->akpPosition= pos2;
	akp->akpXVec= (int)XVec;
	}

    LDEB(AFMlenLINE); return -1;
    }

static AfmKeyword	psAfmKernDataKeywords[]=
    {
	{ "Comment",		psAfmIgnore,		},
	{ "StartKernPairs",	psAfmStartKernPairs,	},
	{ "StartTrackKern",	psAfmStartTrackKern,	},
	{ "EndKernData",	0,			},
    };

static int	psAfmStartKernData(	FILE *	f,
					int	valPos,
					char *	input,
					AfmFontInfo *	afi )
    {
    return psAfmConsumeLines( f, input, afi, psAfmKernDataKeywords,
		    sizeof(psAfmKernDataKeywords)/sizeof(AfmKeyword) );
    }

static int	psAfmStartComposites(	FILE *		f,
					int		valPos,
					char *		input,
					AfmFontInfo *	afi )
    {
    int		lineLength;

    while( psGetInputLine( input, &lineLength, AFMlenLINE, f ) )
	{
	if  ( ! strncmp( input, "EndComposites", 13 ) )
	    { break;					}
	}

    return 0;
    }

static int	psAfmStartTrackKern(	FILE *		f,
					int		valPos,
					char *		input,
					AfmFontInfo *	afi )
    {
    int		lineLength;

    while( psGetInputLine( input, &lineLength, AFMlenLINE, f ) )
	{
	if  ( ! strncmp( input, "EndTrackKern", 13 ) )
	    { break;					}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Set the encoding of an afi according to glyph names.		*/
/*									*/
/************************************************************************/
static int psFindAlternate(	const AfmFontInfo *	afi,
				const char *		glyphName )
    {
    const AlternateGlyphName *	agn= PS_AlternateNames;

    while( agn->agnStandardName )
	{
	if  ( ! strcmp( glyphName, agn->agnStandardName ) )
	    {
	    int			j;
	    AfmCharMetric *	acm;

	    acm= afi->afiMetrics;
	    for ( j= 0; j < afi->afiMetricCount; acm++, j++ )
		{
		if  ( ! strcmp( acm->acmN, agn->agnAlternateName ) )
		    { break;	}
		}

	    if  ( j < afi->afiMetricCount )
		{ return j;	}
	    }

	agn++;
	}

    return -1;
    }

static int psSetEncoding(	AfmFontInfo *	afi,
				const char **	glyphNames,
				int		encoding,
				int		nameCount,
				int		complain )
    {
    int		i;
    int		rval= 0;

    if  ( nameCount > 256 )
	{ LLDEB(nameCount,256); return -1;	}

    for ( i= 0; i < nameCount; i++ )
	{
	int		j;
	AfmCharMetric *	acm;

	if  ( ! glyphNames[i] )
	    { afi->afiEncodingVector[i]= -1; continue;	}

	acm= afi->afiMetrics;
	for ( j= 0; j < afi->afiMetricCount; acm++, j++ )
	    {
	    if  ( ! strcmp( acm->acmN, glyphNames[i] ) )
		{ break;	}
	    }

	if  ( j >= afi->afiMetricCount )
	    { j= psFindAlternate( afi, glyphNames[i] );	}

	if  ( j < 0 )
	    {
	    if  ( complain )
		{ LSDEB(i,glyphNames[i]);	}

	    afi->afiEncodingVector[i]= -1;
	    rval= 1;
	    }
	else{ afi->afiEncodingVector[i]= j;	}
	}

    while( i < 256 )
	{ afi->afiEncodingVector[i++]= -1;	}

    if  ( rval == 0 )
	{ afi->afiEncoding= encoding; }

    return rval;
    }

/************************************************************************/
/*									*/
/*  Calculate the width of a string.					*/
/*									*/
/*  1)  Assume that space is encoded as space.				*/
/*  2)  Contrary to expectations, kerning is an aplication subject. The	*/
/*	printer by itself does not do kerning.				*/
/*									*/
/************************************************************************/

int psCalculateStringExtents(	AfmBBox *		abb,
				const unsigned char *	s,
				int			len,
				int			twipsSize,
				int			withKerning,
				const AfmFontInfo *	afi )
    {
    long	right= 0L;
    long	unitsWide= 0L;

    long	top= 0;
    long	bottom= 0;
    long	prevRight= 0;

    while( len > 0 )
	{
	int		c= afi->afiEncodingVector[*(s++)];
	AfmCharMetric *	acm;

	if  ( c < 0 )
	    {
	    c= afi->afiEncodingVector[' '];
	    if  ( c < 0 )
		{ LDEB(c); return -1;	}
	    }

	acm= afi->afiMetrics+ c;
	if  ( top < acm->acmBBox.abbTop )
	    { top= acm->acmBBox.abbTop;	}
	if  ( bottom < acm->acmBBox.abbBottom )
	    { bottom= acm->acmBBox.abbBottom;	}

	prevRight= unitsWide;
	unitsWide += acm->acmWX;

	/*  2  */
	if  ( withKerning && len > 1 )
	    {
	    int			cc= afi->afiEncodingVector[*s];

	    if  ( c >= 0 )
		{
		int			j;
		AfmKerningPair *	akp= acm->acmKernPairs;

		for ( j= 0; j < acm->acmKernPairCount; akp++, j++ )
		    {
		    if  ( akp->akpPosition == cc )
			{ unitsWide += akp->akpXVec; break; }
		    }
		}
	    }

	if  ( c != ' ' )
	    { right= prevRight+ acm->acmBBox.abbRight; }

	len--;
	}

    abb->abbLeft= 0;
    abb->abbTop= ( twipsSize* top+ 500 )/1000;
    abb->abbBottom= ( twipsSize* bottom+ 500 )/1000;
    abb->abbRight= ( twipsSize* right+ 500 )/1000;

    return ( twipsSize* unitsWide+ 500 )/1000;
    }
