/************************************************************************/
/*									*/
/*  Font administration for a document.					*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<string.h>
#   include	<appDebugon.h>

#   include	"docFont.h"

/************************************************************************/
/*									*/
/*  Default value for text attributes.					*/
/*									*/
/************************************************************************/

TextAttribute	DocDefaultAttributes=
				{ -1, 24, 0, };

/************************************************************************/
/*									*/
/*  Initialise a Font List.						*/
/*  Clean a Font List.							*/
/*  Copy a Font List.							*/
/*									*/
/************************************************************************/

void docInitFontList(	DocumentFontList *	dfl )
    {
    dfl->dflCount= 0;
    dfl->dflFonts= (DocumentFont *)0;
    }

void docCleanFontList(	DocumentFontList *	dfl )
    {
    int		i;

    for ( i = 0; i < dfl->dflCount; i++ )
	{ docCleanFont( dfl->dflFonts+ i );	}

    if  ( dfl->dflFonts )
	{ free( dfl->dflFonts );	}
    }

int docCopyFontList(	DocumentFontList *		to,
			const DocumentFontList *	from )
    {
    int			i;
    DocumentFontList	copy;

    docInitFontList( &copy );

    copy.dflFonts= malloc( from->dflCount* sizeof(DocumentFont) );
    if  ( ! copy.dflFonts )
	{ XDEB(copy.dflFonts); docCleanFontList( &copy ); return -1;	}

    for ( i = 0; i < from->dflCount; i++ )
	{ docInitFont( copy.dflFonts+ i );	}
    copy.dflCount= from->dflCount;

    for ( i = 0; i < from->dflCount; i++ )
	{
	if  ( docCopyFont( copy.dflFonts+ i, from->dflFonts+ i ) )
	    { LDEB(i); docCleanFontList( &copy ); return -1;	}
	}

    docCleanFontList( to );

    *to= copy; return 0;
    }

/************************************************************************/
/*									*/
/*  Initialise a font.							*/
/*  Clean a font.							*/
/*  Copy a font.							*/
/*									*/
/************************************************************************/

void docInitFont(	DocumentFont *	df	)
    {
    df->dfFamilyStyle= (char *)0;
    df->dfName= (char *)0;
    df->dfDocFamilyNumber= -1;
    df->dfPsFamilyNumber= -1;
    df->dfInstanceCount= 0;
    df->dfInstances= (DocumentFontInstance *)0;
    df->dfCharset= FONTcharsetDEFAULT;
    df->dfPitch= FONTpitchDEFAULT;
    }

void docCleanFont(	DocumentFont *	df	)
    {
    if  ( df->dfFamilyStyle )
	{ free( df->dfFamilyStyle );	}
    if  ( df->dfName )
	{ free( df->dfName );	}
    if  ( df->dfInstances )
	{ free( df->dfInstances );	}
    }

int docCopyFont(	DocumentFont *		to,
			const DocumentFont *	from )
    {
    DocumentFont	copy;

    copy= *from;

    copy.dfFamilyStyle= (char *)0;
    copy.dfName= (char *)0;
    copy.dfInstances= (DocumentFontInstance *)0;

    if  ( from->dfFamilyStyle )
	{
	copy.dfFamilyStyle= strdup( from->dfFamilyStyle );
	if  ( ! copy.dfFamilyStyle )
	    { XDEB(copy.dfFamilyStyle); docCleanFont( &copy ); return -1; }
	}

    if  ( from->dfName )
	{
	copy.dfName= strdup( from->dfName );
	if  ( ! copy.dfName )
	    { XDEB(copy.dfName); docCleanFont( &copy ); return -1; }
	}

    if  ( from->dfInstanceCount > 0 )
	{
	int	i;

	copy.dfInstances= (DocumentFontInstance *)
	    malloc( from->dfInstanceCount* sizeof( DocumentFontInstance ) );

	for ( i= 0; i < from->dfInstanceCount; i++ )
	    { copy.dfInstances[i]= from->dfInstances[i];	}
	}

    if  ( to->dfFamilyStyle )
	{ free( to->dfFamilyStyle );	}
    if  ( to->dfName )
	{ free( to->dfName );	}
    if  ( to->dfInstances )
	{ free( to->dfInstances );	}

    *to= copy; return 0;
    }

/************************************************************************/
/*									*/
/*  Insert a font.							*/
/*									*/
/************************************************************************/

DocumentFont *	docInsertFont(	DocumentFontList *	dfl,
				int			n,
				const char *		familyStyle,
				const char *		fontName )
    {
    DocumentFont *	df;
    char *		s;

    char *		familyStyle_s;
    char *		fontName_s;

    if  ( n > dfl->dflCount )
	{
	df= (DocumentFont *)realloc( dfl->dflFonts,
		    ( n + 1 ) * sizeof( DocumentFont ) );
	}
    else{
	df= (DocumentFont *)realloc( dfl->dflFonts,
		    ( dfl->dflCount + 1 ) * sizeof( DocumentFont ) );
	}
    if  ( ! df )
	{ LLDEB(dfl->dflCount,df); return df; }
    dfl->dflFonts= df;

    if  ( familyStyle && familyStyle[0] )
	{
	s= (char *)malloc( strlen( familyStyle )+ 1 );
	if  ( ! s )
	    { LDEB(s); return (DocumentFont *)0;	}
	strcpy( s, familyStyle );
	familyStyle_s= s;
	}
    else{ familyStyle_s= (char *)0;	}

    if  ( fontName && fontName[0] )
	{
	s= (char *)malloc( strlen( fontName )+ 1 );
	if  ( ! s )
	    {
	    LDEB(s);
	    if  ( familyStyle_s )
		{ free( familyStyle_s );	}
	    return (DocumentFont *)0;
	    }
	strcpy( s, fontName );
	fontName_s= s;
	}
    else{ fontName_s= (char *)0;	}

    if  ( n == -1 )
	{
	for ( n= 0; n < dfl->dflCount; n++ )
	    {
	    if  (  df[n].dfDocFamilyNumber < 0 )
		{ break;	}
	    }
	}
    else{
	if  ( n < dfl->dflCount && df[n].dfDocFamilyNumber >= 0 )
	    { LLDEB(n,df[n].dfDocFamilyNumber);	}
	/* No!!
	for ( i= dfl->dflCount; i > n; i-- )
	    { df[i]= df[i-1];	}
	*/
	}

    while( dfl->dflCount < n )
	{ docInitFont( df+ dfl->dflCount ); dfl->dflCount++;	}

    df += n;
    docInitFont( df );

    df->dfDocFamilyNumber= n;
    df->dfFamilyStyle= familyStyle_s;
    df->dfName= fontName_s;

    if  ( n == dfl->dflCount )
	{ dfl->dflCount++;	}

    return df;
    }

/************************************************************************/
/*									*/
/*  Set text attributes to a default value.				*/
/*									*/
/************************************************************************/

void docInitTextAttribute(	TextAttribute *	ta	)
    {
    ta->taFontNumber= -1;
    ta->taFontSizeHalfPoints= 0;
    ta->taFontIsBold= 0;
    ta->taFontIsSlanted= 0;
    ta->taIsUnderlined= 0;
    ta->taShowAsLink= 0;
    ta->taSuperSub= DOCfontREGULAR;
    ta->taSmallCaps= 0;
    ta->taHasStrikeThrough= 0;
    ta->taCapitals= 0;
    }

unsigned int docAttributeDifference(	TextAttribute		ta1,
					TextAttribute		ta2,
					unsigned int		updMask )
    {
    unsigned int	mask= 0;

    mask |= TAupdFONTFAMILY *
		    ( ta1.taFontNumber != ta2.taFontNumber );
    mask |= TAupdFONTSIZE *
		    ( ta1.taFontSizeHalfPoints != ta2.taFontSizeHalfPoints );
    mask |= TAupdFONTBOLD *
		    ( ta1.taFontIsBold != ta2.taFontIsBold );
    mask |= TAupdFONTSLANTED *
		    ( ta1.taFontIsSlanted != ta2.taFontIsSlanted );
    mask |= TAupdTEXTUNDERLINED *
		    ( ta1.taIsUnderlined != ta2.taIsUnderlined );
    mask |= TAupdSHOWASLINK *
		    ( ta1.taShowAsLink != ta2.taShowAsLink );
    mask |= TAupdSUPERSUB *
		    ( ta1.taSuperSub != ta2.taSuperSub );
    mask |= TAupdSMALLCAPS *
		    ( ta1.taSmallCaps != ta2.taSmallCaps );
    mask |= TAupdCAPITALS *
		    ( ta1.taCapitals != ta2.taCapitals );
    mask |= TAupdSTRIKETHROUGH *
		    ( ta1.taHasStrikeThrough != ta2.taHasStrikeThrough );

    return mask & updMask;
    }

void docUpdateTextAttribute(	unsigned int *		pChangeMask,
				unsigned int		updMask,
				TextAttribute *		taTo,
				TextAttribute		taFrom )
    {
    unsigned int	changeMask= *pChangeMask;

    if  ( updMask & TAupdFONTFAMILY					&&
	  taTo->taFontNumber != taFrom.taFontNumber			)
	{
	taTo->taFontNumber= taFrom.taFontNumber;
	changeMask |= TAupdFONTFAMILY;
	}

    if  ( updMask & TAupdFONTSIZE					&&
	  taTo->taFontSizeHalfPoints != taFrom.taFontSizeHalfPoints	)
	{
	taTo->taFontSizeHalfPoints= taFrom.taFontSizeHalfPoints;
	changeMask |= TAupdFONTSIZE;
	}

    if  ( updMask & TAupdFONTBOLD					&&
	  taTo->taFontIsBold != taFrom.taFontIsBold			)
	{
	taTo->taFontIsBold= taFrom.taFontIsBold;	
	changeMask |= TAupdFONTBOLD;
	}

    if  ( updMask & TAupdFONTSLANTED					&&
	  taTo->taFontIsSlanted != taFrom.taFontIsSlanted		)
	{
	taTo->taFontIsSlanted= taFrom.taFontIsSlanted;
	changeMask |= TAupdFONTSLANTED;
	}

    if  ( updMask & TAupdTEXTUNDERLINED				&&
	  taTo->taIsUnderlined != taFrom.taIsUnderlined			)
	{
	taTo->taIsUnderlined= taFrom.taIsUnderlined;
	changeMask |= TAupdTEXTUNDERLINED;
	}

    if  ( updMask & TAupdSHOWASLINK					&&
	  taTo->taShowAsLink != taFrom.taShowAsLink			)
	{
	taTo->taShowAsLink= taFrom.taShowAsLink;
	changeMask |= TAupdSHOWASLINK;
	}

    if  ( updMask & TAupdSUPERSUB					&&
	  taTo->taSuperSub != taFrom.taSuperSub				)
	{
	taTo->taSuperSub= taFrom.taSuperSub;
	changeMask |= TAupdSUPERSUB;
	}

    if  ( updMask & TAupdSMALLCAPS					&&
	  taTo->taSmallCaps != taFrom.taSmallCaps			)
	{
	taTo->taSmallCaps= taFrom.taSmallCaps;
	changeMask |= TAupdSMALLCAPS;
	}

    if  ( updMask & TAupdCAPITALS					&&
	  taTo->taCapitals != taFrom.taCapitals				)
	{
	taTo->taCapitals= taFrom.taCapitals;
	changeMask |= TAupdCAPITALS;
	}

    if  ( updMask & TAupdSTRIKETHROUGH					&&
	  taTo->taHasStrikeThrough != taFrom.taHasStrikeThrough		)
	{
	taTo->taHasStrikeThrough= taFrom.taHasStrikeThrough;
	changeMask |= TAupdSTRIKETHROUGH;
	}

    *pChangeMask= changeMask;
    return;
    }

/************************************************************************/
/*									*/
/*  An half hearted attempt to derive the font style from the family	*/
/*  name.								*/
/*									*/
/************************************************************************/

const char *	docFontFamilyStyle(	const char *	fontFamilyName )
    {
    if  ( ! strcmp( fontFamilyName, "Arial" ) )
	{ return "fswiss";	}
    if  ( ! strcmp( fontFamilyName, "Helvetica" ) )
	{ return "fswiss";	}

    if  ( ! strcmp( fontFamilyName, "Times" )		||
	  ! strcmp( fontFamilyName, "Times New Roman" )	)
	{ return "froman";	}

    if  ( ! strcmp( fontFamilyName, "Symbol" ) )
	{ return "ftech";	}

    if  ( ! strcmp( fontFamilyName, "Courier" ) )
	{ return "fmodern";	}

    if  ( ! strcmp( fontFamilyName, "Ohlfs" ) )
	{ return "fmodern";	}

    return "fnil";
    }

/************************************************************************/
/*									*/
/*  Compare two faces for the ordering in a fonts tool.			*/
/*									*/
/************************************************************************/

int docFontCompareFaces(	const void *	veft1,
				const void *	veft2 )
    {
    const AppFontTypeface *	eft1= (AppFontTypeface *)veft1;
    const AppFontTypeface *	eft2= (AppFontTypeface *)veft2;

    if  ( eft1->aftIsBold && ! eft2->aftIsBold )
	{ return  1;	}
    if  ( eft2->aftIsBold && ! eft1->aftIsBold )
	{ return -1;	}

    if  ( eft1->aftIsSlanted && ! eft2->aftIsSlanted )
	{ return  1;	}
    if  ( eft2->aftIsSlanted && ! eft1->aftIsSlanted )
	{ return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Initialise a font family struct.					*/
/*									*/
/************************************************************************/

void docInitFontFamily(	AppFontFamily *	aff )
    {
    aff->affFontFamilyName= (char *)0;
    aff->affFontFamilyText= (char *)0;
    aff->affFaces= (AppFontTypeface *)0;
    aff->affFaceCount= 0;
    aff->affHasFixedWidth= 0;
    aff->affHasProportionalWidth= 0;
    aff->affWidth= FONTwidthNORMAL;
    }

/************************************************************************/
/*									*/
/*  Find or insert a font in a document font list.			*/
/*									*/
/************************************************************************/

int docGetFontByName(	DocumentFontList *	dfl,
			const char *		fontFamilyName )
    {
    int			i;
    DocumentFont *	df;

    df= dfl->dflFonts;
    for ( i= 0; i < dfl->dflCount; df++, i++ )
	{
	if  ( df->dfDocFamilyNumber < 0	||
	      ! df->dfName		)
	    { continue;	}

	if  ( ! strcmp( df->dfName, fontFamilyName ) )
	    { break;	}
	}

    if  ( i < dfl->dflCount )
	{ return df->dfDocFamilyNumber;	}
    else{
	DocumentFont *	fresh;
	const char *	fontFamilyStyle;

	fontFamilyStyle= docFontFamilyStyle( fontFamilyName );

	fresh= docInsertFont( dfl, -1,
				fontFamilyStyle, fontFamilyName );
	if  ( ! fresh )
	    { LDEB(dfl->dflCount); return -1;	}

	return fresh->dfDocFamilyNumber;
	}
    }
