/************************************************************************/
/*									*/
/*  Document window application independent functionality.		*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>

#   include	<locale.h>

#   include	<appSystem.h>
#   include	<appPaper.h>
#   include	"appFrame.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Callback for the file menu.						*/
/*									*/
/************************************************************************/

int appDocSaveDocumentByName(		EditDocument *		ed,
					APP_WIDGET		option,
					int			interactive,
					int			format,
					const char *		filename )
    {
    EditApplication *		ea= ed->edApplication;

    if  ( (*ea->eaSaveDocument)( ed->edPrivateData, format,
			    ea->eaNameAndVersion, ed->edTitle, filename ) )
	{
	if  ( interactive )
	    {
	    appReportSaveFailure( ea, option,
				    ed->edToplevel.atTopWidget, filename );
	    }

	return -1;
	}

    return 0;
    }

void appDocFileSave(	APP_WIDGET	option,
			void *		voided,
			void *		call_data )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;

    int				interactive= 1;

    if  ( ! ea->eaSaveDocument )
	{ XDEB(ea->eaSaveDocument); return;	}

    if  ( ! ed->edFilename		||
	  ed->edFileReadOnly		||
	  ed->edFormat < 0		)
	{ appDocFileSaveAs( option, voided, (void *)0 ); return; }

    if  ( appDocSaveDocumentByName( ed, option, interactive,
					    ed->edFormat, ed->edFilename ) )
	{
	appReportSaveFailure( ea, option,
				ed->edToplevel.atTopWidget, ed->edFilename );
	}
    else{ appDocumentChanged( ed, 0 ); }

    return;
    }

/************************************************************************/
/*									*/
/*  'Close' callback for the file menu and for the window manager menu	*/
/*									*/
/*  1)  As 'mwm' sometimes sends a 'Save yourself' when 'Close' is	*/
/*	selected from the window manager menu, deactivate the protocol.	*/
/*	B.T.W. This does not help.					*/
/*									*/
/*									*/
/************************************************************************/

void appDocFileClose(	APP_WIDGET	option,
			void *		voided,
			void *		call_data )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    if  ( ed->edHasBeenChanged )
	{ appRunReallyCloseDialog( option, ed ); return; }

    appCloseDocument( ea, ed );
    }

APP_CLOSE_CALLBACK( appDocFileCloseCallback, w, voided )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    if  ( ed->edHasBeenChanged )
	{ appRunReallyCloseDialog( (APP_WIDGET)0, ed ); return; }

    appCloseDocument( ea, ed );
    }

void appDocFileNew(	APP_WIDGET	option,
			void *		voided,
			void *		call_data	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    appAppFileNew( option, (void *)ea, call_data );
    }

void appDocFileQuit(	APP_WIDGET	option,
			void *		voided,
			void *		call_data )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    appQuitApplication( option, ed->edToplevel.atTopWidget, ea );

    return;
    }

void appDocFileMini(	APP_WIDGET	option,
			void *		voided,
			void *		call_data )
    {
    EditDocument *		ed= (EditDocument *)voided;

    appIconifyShellWidget( ed->edToplevel.atTopWidget );

    return;
    }

void appDocFileHide(	APP_WIDGET	option,
			void *		voided,
			void *		call_data	 )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;

    if  ( ed->edIsVisible )
	{
	appHideShellWidget( ed->edToplevel.atTopWidget );

	ed->edIsVisible= 0;

	appDocVisible( ea, ed, ed->edIsVisible );
	}

    return;
    }

void appReportSaveFailure(	EditApplication *	ea,
				APP_WIDGET		option,
				APP_WIDGET		relative,
				const char *		filename )
    {
    AppFileMessageResources *	afmr= &(ea->eaFileMessageResources);

    if  ( appTestFileWritable( filename ) )
	{
	appQuestionRunSubjectErrorDialog( ea, relative, option,
					filename, afmr->afmrFileNotWritable );
	}
    else{
	appQuestionRunSubjectErrorDialog( ea, relative, option,
					filename, afmr->afmrFileNotWritable );
	}
    }

/************************************************************************/
/*									*/
/*  The user selected the 'about' option.				*/
/*									*/
/************************************************************************/

void appDocAbout(	APP_WIDGET	option,
			void *		voided,
			void *		call_data )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;

    ea->eaMainVisibleAsAbout= 1;

    appShowShellWidget( ea->eaToplevel.atTopWidget );

    return;
    }

/************************************************************************/
/*									*/
/*  Print a document.							*/
/*									*/
/*  1)  Avoid user oriented numeric formatting in a PostScript file,	*/
/*	that is meant to be read by computers.				*/
/*									*/
/************************************************************************/

void appCallPrintFunction(	FILE *				f,
				EditDocument *			ed,
				const DocumentGeometry *	dgPaper,
				int				nup,
				int				horizontal,
				int				firstPage,
				int				lastPage )
    {
    EditApplication *	ea= ed->edApplication;

    /*  1  */
    setlocale( LC_NUMERIC, "C" );

    (*ea->eaPrintDocument)( f, ed->edPrivateData, ed->edFormat,
				 &(ed->edDrawingData),
				ed->edTitle, ea, dgPaper,
				nup, horizontal, firstPage, lastPage );

    /*  1  */
    setlocale( LC_NUMERIC, "" );

    return;
    }

int appPrintDocument(	EditDocument *			ed,
			int				printer,
			const DocumentGeometry *	dgPaper,
			int				nup,
			int				horizontal,
			int				firstPage,
			int				lastPage )
    {
    EditApplication *	ea= ed->edApplication;
    PrintDestination *	pd= ea->eaPrintDestinations+ printer;
    FILE *		f;
    char		scratchName[L_tmpnam+1];
    char *		scratchCommand= (char *)0;
    int			fileLen;

    const char *	s;
    char *		to;

    switch( pd->pdPrintKind )
	{
	case APPprinterPIPE:
	    f= popen( pd->pdCommand, "w" );

	    if  ( ! f )
		{ SXDEB(pd->pdCommand,f); return -1;	}

	    appCallPrintFunction( f, ed, dgPaper,
					nup, horizontal, firstPage, lastPage );

	    pclose( f );
	    return 0;

	case APPprinterTMPFILE:
	    tmpnam( scratchName );
	    fileLen= strlen( scratchName );

	    to= scratchCommand= malloc( pd->pdCommandLength+ 
					    pd->pdPercentCount* fileLen+ 1 );
	    if  ( ! scratchCommand )
		{ LXDEB(pd->pdCommandLength,scratchCommand); return -1; }
	    s= pd->pdCommand;

	    while( *s )
		{
		if  ( s[0] == '%' && s[1] == 'f' )
		    {
		    strcpy( to, scratchName ); to += fileLen;
		    s += 2; continue;
		    }

		*(to++)= *(s++);
		}
	    *to= '\0';

	    f= fopen( scratchName, "w" );
	    if  ( ! f )
		{ SXDEB(scratchName,f); return -1;	}

	    appCallPrintFunction( f, ed, dgPaper,
					nup, horizontal, firstPage, lastPage );

	    fclose( f );
	    system( scratchCommand );
	    free( scratchCommand );
	    return 0;

	case APPprinterTOFILE:
	default:
	    LDEB(pd->pdPrintKind);
	    return -1;
	}
    }

static char *	appDocBuildFaxCommand(		const char *	faxCommand,
						int		commandLen,
						const char *	fileName,
						const char *	faxNumber,
						char *		title,
						int		fileCount,
						int		faxCount,
						int		titleCount )
    {
    char *		command;
    const char *	s;
    char *		to;

    int			fileLen;
    int			faxLen;
    int			titleLen;

    if  ( ! title )
	{ title= "";	}

    fileLen= strlen( fileName );
    faxLen= strlen( faxNumber );
    titleLen= strlen( title );

    to= command= (char *)malloc( commandLen+
					fileCount* fileLen+
					faxCount* faxLen+
					titleCount* titleLen+ 1 );
    if  ( ! command )
	{ XDEB(command); return (char *)0;	}

    s= faxCommand;
    while( *s )
	{
	if  ( s[0] == '%' && s[1] == 'f' )
	    {
	    strcpy( to, fileName ); to += fileLen;
	    s += 2; continue;
	    }

	if  ( s[0] == '%' && s[1] == 'n' )
	    {
	    strcpy( to, faxNumber ); to += faxLen;
	    s += 2; continue;
	    }

	if  ( s[0] == '%' && s[1] == 't' )
	    {
	    strcpy( to, title ); to += titleLen;
	    s += 2; continue;
	    }

	*(to++)= *(s++);
	}
    *to= '\0';

    return command;
    }

int appFaxDocument(	EditDocument *			ed,
			const char *			faxNumber,
			const DocumentGeometry *	dgPaper,
			int				nup,
			int				horizontal,
			int				firstPage,
			int				lastPage )
    {
    EditApplication *	ea= ed->edApplication;
    int			fileCount= 0;
    int			faxCount= 0;
    int			titleCount= 0;

    int			commandLen;

    char		scratchName[L_tmpnam+1];

    char *		s;
    char *		command;

    FILE *		f;

    if  ( ! ea->eaFaxCommand )
	{ XDEB(ea->eaFaxCommand); return -1;	}

    s= ea->eaFaxCommand;
    while( *s )
	{
	if  ( s[0] == '%' && s[1] == 'f' )
	    { fileCount++; s += 2; continue;	}

	if  ( s[0] == '%' && s[1] == 'n' )
	    { faxCount++; s += 2; continue;	}

	if  ( s[0] == '%' && s[1] == 't' )
	    { titleCount++; s += 2; continue;	}

	s++;
	}

    commandLen= s- ea->eaFaxCommand;

    if  ( faxCount == 0 )
	{ LDEB(faxCount); return -1;	}

    if  ( fileCount > 0 )
	{
	tmpnam( scratchName );
	command= appDocBuildFaxCommand( ea->eaFaxCommand, commandLen,
					scratchName, faxNumber, ed->edTitle,
					fileCount, faxCount, titleCount );

	if  ( ! command )
	    { XDEB(command); return -1;	}

	f= fopen( scratchName, "w" );
	if  ( ! f )
	    { SXDEB(scratchName,f); free( command ); return -1;	}

	appCallPrintFunction( f, ed,
			    dgPaper, nup, horizontal, firstPage, lastPage );

	fclose( f );

	system( command );
	}
    else{
	scratchName[0]= '\0';

	command= appDocBuildFaxCommand( ea->eaFaxCommand, commandLen,
					scratchName, faxNumber, ed->edTitle,
					fileCount, faxCount, titleCount );

	f= popen( command, "w" );
	if  ( ! f )
	    { SXDEB(command,f); free( command ); return -1;	}

	appCallPrintFunction( f, ed, dgPaper, nup, horizontal,
						    firstPage, lastPage );

	pclose( f );
	}

    free( command );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Callback for the item in the files menu.				*/
/*									*/
/************************************************************************/

void appMakeDocVisible(	EditApplication *	ea,
			EditDocument *		ed )
    {
    appShowShellWidget( ed->edToplevel.atTopWidget );

    if  ( ! ed->edIsVisible )
	{
	ed->edIsVisible= 1;

	appDocVisible( ea, ed, ed->edIsVisible );
	}
    }

void appDocToFront(	APP_WIDGET	option,
			void *		voided,
			void *		call_data )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;

    appMakeDocVisible( ea, ed );

    appGuiSetToggleItemState( option, ed->edHasBeenChanged );

    return;
    }

/************************************************************************/
/*									*/
/*  Callbacks for the edit menu.					*/
/*									*/
/************************************************************************/

void appDocEditCopy(	APP_WIDGET	option,
			void *		voided,
			void *		voidpbcs )
    {
    EditDocument *			ed= (EditDocument *)voided;
    EditApplication *			ea= ed->edApplication;

    if  ( ! ea->eaDocCopy )
	{ SXDEB(ed->edFilename,ea->eaDocCopy); return;	}

    (*ea->eaDocCopy)( ed ); return;
    }

void appDocEditCut(	APP_WIDGET	option,
			void *		voided,
			void *		voidpbcs )
    {
    EditDocument *			ed= (EditDocument *)voided;
    EditApplication *			ea= ed->edApplication;

    if  ( ! ea->eaDocCut )
	{ SXDEB(ed->edFilename,ea->eaDocCut); return;	}

    (*ea->eaDocCut)( ed ); return;
    }

void appDocEditPaste(	APP_WIDGET	option,
			void *		voided,
			void *		voidpbcs )
    {
    EditDocument *			ed= (EditDocument *)voided;

    appDocAskForPaste( ed, "PRIMARY" );
    }

void appDocEditSelAll(	APP_WIDGET	option,
			void *		voided,
			void *		voidpbcs )
    {
    EditDocument *			ed= (EditDocument *)voided;
    EditApplication *			ea= ed->edApplication;

    if  ( ! ea->eaDocSelAll )
	{ SXDEB(ed->edFilename,ea->eaDocSelAll); return;	}

    ea->eaDocSelAll( ed ); return;
    }

/************************************************************************/
/*									*/
/*  Close a a document.							*/
/*									*/
/************************************************************************/

void appCloseDocument(	EditApplication *	ea,
			EditDocument *		ed	)
    {
    appRemoveDocument( ea, ed );

    appDestroyShellWidget( ed->edToplevel.atTopWidget );

    return;
    }

/************************************************************************/
/*									*/
/*  Callbacks for the scrollbars, and the size admin  of a document.	*/
/*									*/
/************************************************************************/

static void appDocScrollVertically(	EditDocument *		ed,
					int			scrolledY )
    {
    EditApplication *		ea= ed->edApplication;
    AppDrawingData *		add= &(ed->edDrawingData);

    int				ox;
    int				oy;
    int				high;
    int				wide;

    DocumentRectangle		drClip;
    DocumentRectangle		drScreen;

    appDrawNoClipping( add );

    drClip= add->addBackRect;

    high= ed->edVisibleRect.drY1- ed->edVisibleRect.drY0+ 1;
    wide= ed->edVisibleRect.drX1- ed->edVisibleRect.drX0+ 1;

    if  ( scrolledY > 0 )
	{
	if  ( high > scrolledY )
	    {
	    appDrawMoveArea( add, 0, scrolledY, wide, high- scrolledY, 0, 0 );

	    drClip.drY0= ed->edVisibleRect.drY1- scrolledY;
	    drClip.drY1= ed->edVisibleRect.drY1;
	    }
	else{
	    drClip.drY0= ed->edVisibleRect.drY0;
	    drClip.drY1= ed->edVisibleRect.drY1;
	    }
	}
    else{
	if  ( high+ scrolledY > 0 )
	    {
	    appDrawMoveArea( add, 0, 0, wide, high+ scrolledY, 0, -scrolledY );

	    drClip.drY0= ed->edVisibleRect.drY0;
	    drClip.drY1= ed->edVisibleRect.drY0- scrolledY;
	    }
	else{
	    drClip.drY0= ed->edVisibleRect.drY0;
	    drClip.drY1= ed->edVisibleRect.drY1;
	    }
	}

    ox= ed->edVisibleRect.drX0;
    oy= ed->edVisibleRect.drY0;

    drScreen.drX0= drClip.drX0- ox;
    drScreen.drY0= drClip.drY0- oy;
    drScreen.drX1= drClip.drX1- ox;
    drScreen.drY1= drClip.drY1- oy;

    appDrawSetClipRect( add, &drScreen );

    (*ea->eaDrawRectangle)( ed->edDocumentWidget, ed, &drClip, ox, oy );

    if  ( ed->edLeftRuler && ea->eaScrollVerticalRuler )
	{
	(*ea->eaScrollVerticalRuler)
			( ed->edLeftRuler, ed->edLeftRulerWidget, scrolledY );
	}

    return;
    }

static void appDocScrollHorizontally(	EditDocument *		ed,
					int			scrolledX )
    {
    EditApplication *		ea= ed->edApplication;
    AppDrawingData *		add= &(ed->edDrawingData);

    int				ox;
    int				oy;
    int				high;
    int				wide;

    DocumentRectangle		drClip;
    DocumentRectangle		drScreen;

    appDrawNoClipping( add );

    drClip= add->addBackRect;

    high= ed->edVisibleRect.drY1- ed->edVisibleRect.drY0+ 1;
    wide= ed->edVisibleRect.drX1- ed->edVisibleRect.drX0+ 1;

    if  ( scrolledX > 0 )
	{
	if  ( wide > scrolledX )
	    {
	    appDrawMoveArea( add, scrolledX, 0, wide- scrolledX, high, 0, 0 );

	    drClip.drX0= ed->edVisibleRect.drX1- scrolledX;
	    drClip.drX1= ed->edVisibleRect.drX1;
	    }
	else{
	    drClip.drX0= ed->edVisibleRect.drX0;
	    drClip.drX1= ed->edVisibleRect.drX1;
	    }
	}
    else{
	if  ( wide+ scrolledX > 0 )
	    {
	    appDrawMoveArea( add, 0, 0, wide+ scrolledX, high, -scrolledX, 0 );

	    drClip.drX0= ed->edVisibleRect.drX0;
	    drClip.drX1= ed->edVisibleRect.drX0- scrolledX;
	    }
	else{
	    drClip.drX0= ed->edVisibleRect.drX0;
	    drClip.drX1= ed->edVisibleRect.drX1;
	    }
	}

    ox= ed->edVisibleRect.drX0;
    oy= ed->edVisibleRect.drY0;

    drScreen.drX0= drClip.drX0- ox;
    drScreen.drY0= drClip.drY0- oy;
    drScreen.drX1= drClip.drX1- ox;
    drScreen.drY1= drClip.drY1- oy;

    appDrawSetClipRect( add, &drScreen );

    (*ea->eaDrawRectangle)( ed->edDocumentWidget, ed, &drClip, ox, oy );

    if  ( ed->edTopRuler && ea->eaScrollHorizontalRuler )
	{
	(*ea->eaScrollHorizontalRuler)
			( ed->edTopRuler, ed->edTopRulerWidget, scrolledX );
	}

    return;
    }

void appDocVerticalScrollbarCallback(	APP_WIDGET	w,
					void *		voided,
					void *		voidscbs )
    {
    EditDocument *		ed= (EditDocument *)voided;

    int				scrolledY;

    scrolledY= appGuiGetScrollbarValueFromCallback( w, voidscbs )-
						    ed->edVisibleRect.drY0;

    ed->edVisibleRect.drY0 += scrolledY;
    ed->edVisibleRect.drY1 += scrolledY;

    if  ( scrolledY == 0 )
	{ return;	}

    appDocScrollVertically( ed, scrolledY );

    return;
    }


void appDocHorizontalScrollbarCallback(	APP_WIDGET	w,
					void *		voided,
					void *		voidscbs )
    {
    EditDocument *		ed= (EditDocument *)voided;

    int				scrolledX;

    scrolledX= appGuiGetScrollbarValueFromCallback( w, voidscbs )-
						    ed->edVisibleRect.drX0;


    ed->edVisibleRect.drX0 += scrolledX;
    ed->edVisibleRect.drX1 += scrolledX;

    if  ( scrolledX == 0 )
	{ return;	}

    appDocScrollHorizontally( ed, scrolledX );

    return;
    }

/************************************************************************/
/*									*/
/*  Scroll the selection into view.					*/
/*									*/
/*  NOTE: sliderSize is passed to XmScrollBarSetValues() because of a	*/
/*	bug in lesstif Release 0.87.0. (Jan 1999).			*/
/*									*/
/************************************************************************/

void appScrollToRectangle(	EditDocument *		ed,
				int			x0,
				int			y0,
				int			x1,
				int			y1,
				int *			pScrolledX,
				int *			pScrolledY )
    {
    const AppDrawingData *	add= &(ed->edDrawingData);

    int				sliderSize;
	
    int				oox= ed->edVisibleRect.drX0;
    int				ooy= ed->edVisibleRect.drY0;
    int				nox;
    int				noy;

    int				set;

    nox= oox;
    noy= ooy;

    appGuiGetScrollbarValues( &noy, &sliderSize, ed->edVerticalScrollbar );

    if  ( y1 > noy+ sliderSize )
	{
	set= y1- sliderSize;

	if  ( noy != set )
	    {
	    appGuiSetScrollbarValues( ed->edVerticalScrollbar,
							set, sliderSize );
	    noy= set;
	    }
	}

    if  ( y0 < noy )
	{
	set= y0;

	if  ( noy != set )
	    {
	    appGuiSetScrollbarValues( ed->edVerticalScrollbar,
							set, sliderSize );
	    noy= set;
	    }
	}

    if  ( noy+ sliderSize > add->addBackRect.drY1 )
	{
	set= add->addBackRect.drY1- sliderSize+ 1;
	if  ( set < 0 )
	    { set= 0;	}

	if  ( noy != set )
	    {
	    appGuiSetScrollbarValues( ed->edVerticalScrollbar,
							set, sliderSize );
	    noy= set;
	    }
	}

    appGuiGetScrollbarValues( &nox, &sliderSize, ed->edHorizontalScrollbar );

    if  ( x1 > nox+ sliderSize )
	{
	set= x1- sliderSize;
	if  ( nox != set )
	    {
	    appGuiSetScrollbarValues( ed->edHorizontalScrollbar,
							set, sliderSize );
	    nox= set;
	    }
	}

    if  ( x0 < nox )
	{
	set= x0;
	if  ( nox != set )
	    {
	    appGuiSetScrollbarValues( ed->edHorizontalScrollbar,
							set, sliderSize );
	    nox= set;
	    }
	}

    if  ( nox+ sliderSize > add->addBackRect.drX1 )
	{
	set= add->addBackRect.drX1- sliderSize+ 1;
	if  ( set < 0 )
	    { set= 0;	}

	if  ( nox != set )
	    {
	    appGuiSetScrollbarValues( ed->edHorizontalScrollbar,
							set, sliderSize );
	    nox= set;
	    }
	}

    *pScrolledX= nox- oox;
    *pScrolledY= noy- ooy;

    ed->edVisibleRect.drX0 += *pScrolledX;
    ed->edVisibleRect.drY0 += *pScrolledY;
    ed->edVisibleRect.drX1 += *pScrolledX;
    ed->edVisibleRect.drY1 += *pScrolledY;

    if  ( *pScrolledX != 0 )
	{ appDocScrollHorizontally( ed, *pScrolledX );	}
    if  ( *pScrolledY != 0 )
	{ appDocScrollVertically( ed, *pScrolledY );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Prevent the Shell that contains a document from being resized	*/
/*  beyond normal limits.						*/
/*									*/
/************************************************************************/

static void appFileAdaptHorizontalRulerRange(	EditDocument *	ed,
						int		width )
    {
    EditApplication *		ea= ed->edApplication;

    if  ( ed->edTopRuler && ea->eaSetHorizontalRulerRange )
	{
	int	leftRulerWide= 0;
	int	leftRulerHigh= 0;

	if  ( ed->edLeftRulerWidget )
	    {
	    appDrawGetSizeOfWidget( &leftRulerWide, &leftRulerHigh,
						    ed->edLeftRulerWidget );
	    }

	(*ea->eaSetHorizontalRulerRange)( ed->edTopRuler,
					ed->edTopRulerWidget,
					ed->edVisibleRect.drX0,
					ed->edVisibleRect.drX1,
					width+ leftRulerWide );
	}

    return;
    }

static void appFileAdaptVerticalRulerRange(	EditDocument *	ed,
						int		height )
    {
    EditApplication *		ea= ed->edApplication;

    if  ( ed->edLeftRuler && ea->eaSetVerticalRulerRange )
	{
	(*ea->eaSetVerticalRulerRange)( ed->edLeftRuler,
					ed->edLeftRulerWidget,
					ed->edVisibleRect.drY0,
					ed->edVisibleRect.drY1,
					height );
	}

    return;
    }

void appAdaptToDocumentSize(	EditDocument *	ed,
				int		width,
				int		height )
    {
    AppDrawingData *	add= &(ed->edDrawingData);
    int			changed= 0;
    int			d;

    if  ( ed->edVisibleRect.drY1 != ed->edVisibleRect.drY0+ height- 1 )
	{
	ed->edVisibleRect.drY1= ed->edVisibleRect.drY0+ height- 1;
	changed= 1;

	if  ( ed->edVisibleRect.drY1 > add->addBackRect.drY1 )
	    {
	    d= ed->edVisibleRect.drY1- add->addBackRect.drY1;

	    if  ( d > ed->edVisibleRect.drY0 )
		{ d= ed->edVisibleRect.drY0;	}

	    ed->edVisibleRect.drY0 -= d;
	    ed->edVisibleRect.drY1 -= d;
	    }

	appFileAdaptVerticalRulerRange( ed, height );

	appExposeRectangle( add, 0, 0, 0, 0 );
	}

    if  ( ed->edVisibleRect.drX1 != ed->edVisibleRect.drX0+ width+ 1 )
	{
	ed->edVisibleRect.drX1= ed->edVisibleRect.drX0+ width+ 1;
	changed= 1;

	if  ( ed->edVisibleRect.drX1 > add->addBackRect.drX1 )
	    {
	    d= ed->edVisibleRect.drX1- add->addBackRect.drX1;

	    if  ( d > ed->edVisibleRect.drX0 )
		{ d= ed->edVisibleRect.drX0;	}

	    ed->edVisibleRect.drX0 -= d;
	    ed->edVisibleRect.drX1 -= d;
	    }

	appFileAdaptHorizontalRulerRange( ed, width );

	appExposeRectangle( add, 0, 0, 0, 0 );
	}

    if  ( changed )
	{ appDocSetScrollbarValues( ed );	}

    return;
    }

int appSetDocumentFilename(	EditDocument *		ed,
				const char *		filename )
    {
    char *			s= (char *)0;

    if  ( filename )
	{
	s= strdup( filename );
	if  ( ! s )
	    { XDEB(s); return -1;	}
	}

    if  ( ed->edFilename )
	{ free( ed->edFilename );	}
    ed->edFilename= s;

    return 0;
    }

int appFormatDocumentTitle(	const char **		pWindowTitle,
				const char **		pIconTitle,
				EditApplication *	ea,
				const char *		title )
    {
    const char *		baseName;

    int				len;
    static char *		fullTitle;
    static char *		fullIconName;

    baseName= strrchr( title, '/' );
    if  ( baseName )
	{ baseName++;		}
    else{ baseName= title;	}

    len= strlen( ea->eaApplicationName )+ 2+ strlen( title )+ 1;
    fullTitle= (char *)realloc( fullTitle, len );
    if  ( ! fullTitle )
	{ LXDEB(len,fullTitle); return -1;	}

    len= strlen( ea->eaApplicationName )+ 2+ strlen( baseName )+ 1;
    fullIconName= (char *)realloc( fullIconName, len );
    if  ( ! fullIconName )
	{ LXDEB(len,fullIconName); return -1;	}

    sprintf( fullTitle, "%s: %s", ea->eaApplicationName, title );
    sprintf( fullIconName, "%s: %s", ea->eaApplicationName, baseName );

    *pWindowTitle= fullTitle; *pIconTitle= fullIconName; return 0;
    }

int appSetDocumentTitle(	EditDocument *		ed,
				const char *		title )
    {
    EditApplication *		ea= ed->edApplication;

    char *			s;

    const char *		fullTitle;
    const char *		fullIconName;

    s= strdup( title );
    if  ( ! s )
	{ XDEB(s); return -1;	}

    if  ( ed->edTitle )
	{ free( ed->edTitle );	}
    ed->edTitle= s;

    if  ( appFormatDocumentTitle( &fullTitle, &fullIconName, ea, ed->edTitle ) )
	{ SDEB(ed->edTitle); return -1;	}

    appGuiSetShellTitle( ed->edToplevel.atTopWidget, fullTitle );
    appGuiSetIconTitle( ed->edToplevel.atTopWidget, fullIconName );

    appRenameDocumentOptions( ed->edApplication, ed, ed->edTitle );

    return 0;
    }


void appDocumentRulerWidth(	EditApplication *	ea,
				EditDocument *		ed )
    {
    int			mult;

    double		horPixPerMM;
    double		verPixPerMM;
    double		xfac;
    double		yfac;

    appGetFactors( ea, &horPixPerMM, &verPixPerMM, &xfac, &yfac );

    /*  1  */
    ed->edLeftRulerWidthPixels=
			(int)( ea->eaLeftRulerWidthMM* horPixPerMM );
    ed->edTopRulerHeightPixels=
			(int)( ea->eaTopRulerHeightMM* verPixPerMM );
    ed->edRightRulerWidthPixels=
			(int)( ea->eaRightRulerWidthMM * horPixPerMM );
    ed->edBottomRulerHeightPixels=
			(int)( ea->eaBottomRulerHeightMM* verPixPerMM );

    mult= ea->eaLeftRulerWidthMultiple;
    if  ( mult )
	{
	ed->edLeftRulerWidthPixels=
		    mult* ( ( ed->edLeftRulerWidthPixels+ mult- 1 )/ mult );
	}

    mult= ea->eaTopRulerHeightMultiple;
    if  ( mult )
	{
	ed->edTopRulerHeightPixels=
		    mult* ( ( ed->edTopRulerHeightPixels+ mult- 1 )/ mult );
	}

    mult= ea->eaRightRulerWidthMultiple;
    if  ( mult )
	{
	ed->edRightRulerWidthPixels=
		    mult* ( ( ed->edRightRulerWidthPixels+ mult- 1 )/ mult );
	}

    mult= ea->eaBottomRulerHeightMultiple;
    if  ( mult )
	{
	ed->edBottomRulerHeightPixels=
		    mult* ( ( ed->edBottomRulerHeightPixels+ mult- 1 )/ mult );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Free all resources allocated for a document.			*/
/*									*/
/************************************************************************/

void appFreeDocument(		EditApplication *	ea,
				EditDocument *		ed )
    {
    if  ( ed->edLeftRuler )
	{ (*ea->eaFreeLeftRuler)( ed->edLeftRuler ); }
    if  ( ed->edTopRuler )
	{ (*ea->eaFreeTopRuler)( ed->edTopRuler ); }
    if  ( ed->edRightRuler )
	{ (*ea->eaFreeRightRuler)( ed->edRightRuler ); }
    if  ( ed->edBottomRuler )
	{ (*ea->eaFreeBottomRuler)( ed->edBottomRuler ); }

    if  ( ed->edFilename )
	{ free( ed->edFilename );	}
    if  ( ed->edTitle )
	{ free( ed->edTitle );	}

    if  ( ed->edCheckpointFilename )
	{
	if  ( appRemoveFile( ed->edCheckpointFilename ) )
	    { SDEB(ed->edCheckpointFilename);	}

	free( ed->edCheckpointFilename );
	}

#   ifdef USE_MOTIF
    if  ( ed->edInputContext )
	{ XDestroyIC( ed->edInputContext ); }
#   endif

    appCleanDrawingData( &(ed->edDrawingData) );
    appCleanColors( &(ed->edColors) );

    free( ed );

    return;
    }

void appInitEditDocument(	EditApplication *	ea,
				EditDocument *		ed )
    {
    ed->edApplication= ea;

    ed->edFilename= (char *)0;
    ed->edTitle= (char *)0;
    ed->edCheckpointFilename= (char *)0;
    ed->edFormat= -1;
    ed->edFileReadOnly= 0;

    ed->edMenuBar= (APP_WIDGET)0;
    ed->edMainWindow= (APP_WIDGET)0;
	ed->edFileMenu= (APP_WIDGET)0;
	ed->edFileMenuButton= (APP_WIDGET)0;
	ed->edEditMenu= (APP_WIDGET)0;
	ed->edEditMenuButton= (APP_WIDGET)0;
	ed->edWindowMenu= (APP_WIDGET)0;
	ed->edWindowMenuButton= (APP_WIDGET)0;
	ed->edHelpMenu= (APP_WIDGET)0;
	ed->edHelpMenuButton= (APP_WIDGET)0;

    ed->edToolbar= (APP_WIDGET)0;
    ed->edScrolledWindow= (APP_WIDGET)0;
	ed->edVerticalScrollbar= (APP_WIDGET)0;
	ed->edHorizontalScrollbar= (APP_WIDGET)0;
	ed->edDocumentWidget= (APP_WIDGET)0;

	ed->edLeftRulerWidget= (APP_WIDGET)0;
	ed->edTopRulerWidget= (APP_WIDGET)0;
	ed->edRightRulerWidget= (APP_WIDGET)0;
	ed->edBottomRulerWidget= (APP_WIDGET)0;

	ed->edLeftRuler= (void *)0;
	ed->edTopRuler= (void *)0;
	ed->edRightRuler= (void *)0;
	ed->edBottomRuler= (void *)0;

#   ifdef USE_MOTIF
    ed->edInputContext= (XIC)0;
#   endif

    ed->edLeftRulerWidthPixels= 0;
    ed->edTopRulerHeightPixels= 0;
    ed->edRightRulerWidthPixels= 0;
    ed->edBottomRulerHeightPixels= 0;

    ed->edHasBeenChanged= 0;
    ed->edIsReadonly= 0;
    ed->edIsVisible= 0;

    ed->edShellExtraWidth= -1;
    ed->edShellExtraHeight= -1;

    appInitColors( &(ed->edColors) );
    appInitDrawingData( &(ed->edDrawingData) );

    ed->edMapped= 0;

    return;
    }

int appMakeDocumentWindow(	EditDocument **		pEd,
				EditApplication *	ea,
				int			readonly,
				const char *		title,
				const char *		filename )
    {
    char			scratch[100];

    EditDocument *		ed;

    if  ( ! title )
	{
	static int			count= 1;

	AppFileMessageResources *	afmr= &(ea->eaFileMessageResources);

	sprintf( scratch, afmr->afmrNamelessTitleFormat, count++ );
	title= scratch;
	}

    /*  1  */
    ed= (EditDocument *)malloc( sizeof(EditDocument) );
    if  ( ! ed )
	{ XDEB(ed); return -1;	}

    appInitEditDocument( ea, ed );

    if  ( filename )
	{ ed->edFilename= strdup( filename );	}
    else{ ed->edFilename= (char *)0;		}

    ed->edIsReadonly= readonly;
    ed->edTitle= strdup( title );

    if  ( ea->eaMakePrivateData	)
	{
	ed->edPrivateData= (*ea->eaMakePrivateData)();
	if  ( ! ed->edPrivateData )
	    { XDEB(ed->edPrivateData); appFreeDocument( ea, ed ); return -1; }
	}

    if  ( appFinishDocumentWindow( ed ) )
	{ appFreeDocument( ea, ed ); return -1; }

    *pEd= ed; return 0;
    }

void appDocumentCalculateExtraSize(	EditDocument *	ed )
    {
    int	shellWide;
    int	shellHigh;

    int	docWide;
    int	docHigh;

    appDrawGetSizeOfWidget( &shellWide, &shellHigh,
					    ed->edToplevel.atTopWidget );

    appDrawGetSizeOfWidget( &docWide, &docHigh,
					    ed->edDocumentWidget );

    ed->edShellExtraWidth= shellWide- docWide;
    ed->edShellExtraHeight= shellHigh- docHigh;

    appSetShellConstraints( ed );

    appAdaptToDocumentSize( ed, docWide, docHigh );
    }

int appSetupDocument(	EditApplication *	ea,
			EditDocument *		ed	)
    {
    AppDrawingData *		add= &(ed->edDrawingData);

    double			screenPixelsPerMM;
    double			verPixPerMM;
    double			xfac;
    double			yfac;

    int				wide;
    int				high;

    appGetFactors( ea, &screenPixelsPerMM, &verPixPerMM, &xfac, &yfac );

    ed->edPixelsPerMM= screenPixelsPerMM;

    appSetDrawingEnvironment( add, ea->eaMagnification, xfac,
			screenPixelsPerMM,
			ea->eaAfmDirectory, ea->eaToplevel.atTopWidget );

    add->addPageGapPixels= (int)( ea->eaPageGapMM* verPixPerMM );

    if  ( (*ea->eaLayoutDocument)( ed->edPrivateData, ed->edFormat,
					&(ed->edDrawingData),
					&(ea->eaDefaultDocumentGeometry) ) )
	{ SDEB(ed->edFilename); return -1; }

    appSetShellConstraints( ed );

    wide= add->addPaperRect.drX1- add->addPaperRect.drX0;
    high= add->addPaperRect.drY1- add->addPaperRect.drY0;

    if  ( wide > ( 5* ea->eaScreenPixelsWide )/ 6 )
	{ wide=( 5* ea->eaScreenPixelsWide )/ 6; }
    if  ( high > ( 5* ea->eaScreenPixelsHigh )/ 6 )
	{ high=( 5* ea->eaScreenPixelsHigh )/ 6; }

#   ifdef USE_MOTIF
    XtVaSetValues( ed->edDocumentWidget,
			    XmNborderWidth,		0,
			    XmNshadowThickness,		0,
			    XmNwidth,			wide,
			    XmNheight,			high,
			    NULL );
#   endif

#   ifdef USE_GTK
    gtk_drawing_area_size( GTK_DRAWING_AREA( ed->edTopRulerWidget ),
					ed->edLeftRulerWidthPixels+
					wide+
					ed->edRightRulerWidthPixels,
					ed->edTopRulerHeightPixels );
    gtk_drawing_area_size( GTK_DRAWING_AREA( ed->edLeftRulerWidget ),
					ed->edLeftRulerWidthPixels, high );


    gtk_drawing_area_size( GTK_DRAWING_AREA( ed->edDocumentWidget ),
								wide, high );

    gtk_drawing_area_size( GTK_DRAWING_AREA( ed->edRightRulerWidget ),
					ed->edRightRulerWidthPixels, high );
    gtk_drawing_area_size( GTK_DRAWING_AREA( ed->edBottomRulerWidget ),
					ed->edLeftRulerWidthPixels+
					wide+
					ed->edRightRulerWidthPixels,
					ed->edBottomRulerHeightPixels );

    gtk_object_set_user_data( GTK_OBJECT( ed->edDocumentWidget ), (void *)ed );
#   endif

    ed->edVisibleRect.drX0= add->addPaperRect.drX0;
    ed->edVisibleRect.drY0= add->addPaperRect.drY0;
    ed->edVisibleRect.drX1= add->addPaperRect.drX1;
    ed->edVisibleRect.drY1= add->addPaperRect.drY1;

    if  ( ea->eaSetTopRuler		&&
	  (*ea->eaSetTopRuler)( ed )	)
	{ SDEB(ed->edFilename);	}

    if  ( ea->eaSetLeftRuler		&&
	  (*ea->eaSetLeftRuler)( ed )	)
	{ SDEB(ed->edFilename);	}

    appDocSetScrollbarValues( ed );

#   ifdef USE_MOTIF
    XtRealizeWidget( ed->edToplevel.atTopWidget );
#   endif

#   ifdef USE_GTK
    gtk_widget_realize( ed->edToplevel.atTopWidget );
#   endif

    if  ( ea->eaFinishDocumentSetup && (*ea->eaFinishDocumentSetup)( ea, ed ) )
	{ SDEB(ed->edFilename); return -1; }

    appShowShellWidget( ed->edToplevel.atTopWidget );

    ed->edIsVisible= 1;
    ed->edHasBeenChanged= 0;

    appDocVisible( ea, ed, ed->edIsVisible );

    return 0;
    }

APP_EVENT_HANDLER( appDocConfigure, w, voided, event )
    {
    EditDocument *	ed= (EditDocument *)voided;

    int			wide;
    int			high;

    if  ( appDrawGetSizeFromConfigureEvent( &wide, &high, w, event ) )
	{ return;	}

    appAdaptToDocumentSize( ed, wide, high );

    return;
    }

/************************************************************************/
/*									*/
/*  Fill the menu of a document window.					*/
/*									*/
/************************************************************************/

void appDocFillMenu(	EditDocument *		ed )
    {
    EditApplication *	ea= ed->edApplication;

    ed->edFileMenu= appMakeMenu( &(ed->edFileMenuButton),
		    &(ed->edToplevel), ea, ed->edMenuBar,
		    *(ea->eaDocFileMenuText), 0,
		    ea->eaDocFileMenuItems, ea->eaDocFileMenuItemCount,
		    (void *)ed );

    ed->edEditMenu= appMakeMenu( &(ed->edEditMenuButton),
		    &(ed->edToplevel), ea, ed->edMenuBar,
		    *(ea->eaDocEditMenuText), 0,
		    ea->eaDocEditMenuItems, ea->eaDocEditMenuItemCount,
		    (void *)ed );

    if  ( ea->eaMakePrivateDocumentMenus )
	{ (*ea->eaMakePrivateDocumentMenus)( ea, ed, ed->edMenuBar ); }

    ed->edWindowMenu= appMakeMenu( &(ed->edWindowMenuButton),
		    &(ed->edToplevel), ea, ed->edMenuBar,
		    *(ea->eaDocWindowMenuText), 0,
		    ea->eaDocWindowMenuItems, ea->eaDocWindowMenuItemCount,
		    (void *)ed );

    if  ( ea->eaDocHelpMenuItems )
	{
	ed->edHelpMenu= appMakeMenu( &(ed->edHelpMenuButton),
		    &(ed->edToplevel), ea, ed->edMenuBar,
		    *(ea->eaDocHelpMenuText), 1,
		    ea->eaDocHelpMenuItems, ea->eaDocHelpMenuItemCount,
		    (void *)ed );
	}

    }

/************************************************************************/
/*									*/
/*  Open a a document.							*/
/*									*/
/*  1)  Allocate a document.						*/
/*  3)  Make a shell to contain everything.				*/
/*  4)  Make the window pane.						*/
/*									*/
/************************************************************************/

EditDocument * appOpenDocument(	EditApplication *	ea,
				APP_WIDGET		relative,
				APP_WIDGET		option,
				int			readonly,
				const char *		filename )
    {
    EditDocument *		ed;
    int				fileReadonly= 0;
    AppFileMessageResources *	afmr= &(ea->eaFileMessageResources);

    if  ( readonly )
	{ fileReadonly= 1;	}
    else{
	if  ( appTestFileReadable( filename ) )
	    {
	    int	resp;

	    resp= appQuestionRunSubjectOkCancelDialog( ea,
					relative, option,
					filename, afmr->afmrFileReadOnly,
					(char *)0, (char *)0 );

	    if  ( resp != AQDrespOK )
		{ return (EditDocument *)0;	}

	    fileReadonly= 1;
	    }
	}

    /*  1,3,4  */
    if  ( appMakeDocumentWindow( &ed, ea, readonly, filename, filename ) )
	{ SDEB(filename); return (EditDocument *)0;	}

    ed->edFileReadOnly= fileReadonly;

    if  ( (*ea->eaOpenDocument)( ea, ed->edPrivateData, &(ed->edFormat),
						relative, option, filename ) )
	{ return (EditDocument *)0; }

    if  ( appSetupDocument( ea, ed ) )
	{ SDEB(filename); return (EditDocument *)0; }

    appSetDocument( ea, ed );

    return ed;
    }

int appNewDocument(	EditApplication *	ea,
			const char *		filename )
    {
    const int			readonly= 0;
    const char *		title= filename;

    EditDocument *		ed;

    /*  1,3,4  */
    if  ( appMakeDocumentWindow( &ed, ea, readonly, title, filename ) )
	{ SDEB(filename); return -1;	}

    if  ( (*ea->eaNewDocument)( ea, ed, filename ) )
	{ SDEB(title); return -1; }

    if  ( appSetupDocument( ea, ed ) )
	{ SDEB(title); return -1; }

    appSetDocument( ea, ed );

    return 0;
    }

/************************************************************************/
/*									*/
/*  File Conversion entry point. ea is supposed to be initialized.	*/
/*  Depending on the application this routine may or mar not work with	*/
/*  an EditApplication struct that is not initialized.			*/
/*									*/
/************************************************************************/

int appFileConvert(	EditApplication *	ea,
			const char *		fromName,
			const char *		toName )
    {
    int			interactive= ( ea->eaToplevel.atTopWidget != NULL );

    const char *	slash;
    const char *	dot;
    AppFileExtension *	afe;

    void *		privateData= (void *)0;
    int			fromFormat= -1;

    int			i;

    if  ( ! strcmp( fromName, toName ) )
	{ SSDEB(fromName,toName); return -1;	}

    if  ( ea->eaMakePrivateData	)
	{
	privateData= (*ea->eaMakePrivateData)();
	if  ( ! privateData )
	    { XDEB(privateData); return -1; }
	}

    if  ( (*ea->eaOpenDocument)( ea, privateData, &fromFormat,
						ea->eaToplevel.atTopWidget,
						(APP_WIDGET)0, fromName ) )
	{ SDEB(fromName); return -1; }

    slash= strrchr( toName, '/' );
    if  ( slash )
	{ dot= strrchr( slash+ 1, '.' );	}
    else{ dot= strrchr( toName, '.' );	}

    afe= ea->eaFileExtensions;
    for ( i= 0; i < ea->eaFileExtensionCount; afe++, i++ )
	{
	if  ( ! ( afe->afeUseFlags & APPFILE_CAN_SAVE ) )
	    { continue;	}

	if  ( ea->eaCanSaveDocument				&&
	      (*ea->eaCanSaveDocument)( privateData, i )	)
	    { continue;	}

	if  ( dot					&&
	      afe->afeExtension			&&
	      strcmp( dot+ 1, afe->afeExtension )	)
	    { continue;	}

	if  ( (*ea->eaSaveDocument)( privateData, i,
				ea->eaNameAndVersion, toName, toName ) )
	    {
	    if  ( interactive )
		{
		appReportSaveFailure( ea, (APP_WIDGET)0,
					    ea->eaToplevel.atTopWidget, toName );
		}

	    (*ea->eaFreeDocument)( privateData, fromFormat,
						    (AppDrawingData *)0 );
	    return -1;
	    }

	(*ea->eaFreeDocument)( privateData, fromFormat,
						    (AppDrawingData *)0 );
	return 0;
	}

    SSDEB(fromName,toName);
    (*ea->eaFreeDocument)( privateData, fromFormat, (AppDrawingData *)0 );
    return -1;
    }


/************************************************************************/
/*									*/
/*  Destroy callback on a document widget.				*/
/*									*/
/************************************************************************/

void appDestroyEditDocument(	APP_WIDGET		w,
				void *			voided,
				void *			callData )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;

    if  ( ed->edPrivateData )
	{
	(*ea->eaFreeDocument)( ed->edPrivateData, ed->edFormat,
						    &(ed->edDrawingData) );
	}

    appFreeDocument( ea, ed );

    return;
    }

/************************************************************************/
/*									*/
/*  Command line Print To File entrypoint.				*/
/*									*/
/************************************************************************/

int appPrintToFile(	EditApplication *	ea,
			const char *		fromName,
			const char *		toName,
			const char *		paperString )
    {
    int			rval= 0;

    void *		privateData= (void *)0;
    int			fromFormat= -1;

    AppDrawingData	add;

    double		screenPixelsPerMM;
    double		verPixPerMM;
    double		xfac;
    double		yfac;

    FILE *		f;

    DocumentGeometry	dg;

    dg= ea->eaDefaultDocumentGeometry;

    appInitDrawingData( &add );

    if  ( ! strcmp( fromName, toName ) )
	{ SSDEB(fromName,toName); return -1;	}

    if  ( ea->eaMakePrivateData	)
	{
	privateData= (*ea->eaMakePrivateData)();
	if  ( ! privateData )
	    { XDEB(privateData); return -1; }
	}

    if  ( paperString )
	{
	int	paperFormat;

	if  ( appPaperFormatFromString( &paperFormat,
					    &(dg.dgPageWideTwips),
					    &(dg.dgPageHighTwips),
					    ea->eaUnitInt, paperString ) )
	    { SDEB(paperString); return -1;	}
	}

    if  ( (*ea->eaOpenDocument)( ea, privateData, &fromFormat,
				ea->eaToplevel.atTopWidget, (APP_WIDGET)0, fromName ) )
	{ SDEB(fromName); return -1; }

    appGetFactors( ea, &screenPixelsPerMM, &verPixPerMM, &xfac, &yfac );

    appSetDrawingEnvironment( &add, ea->eaMagnification, xfac,
			    screenPixelsPerMM,
			    ea->eaAfmDirectory, ea->eaToplevel.atTopWidget );

    add.addPageGapPixels= (int)( ea->eaPageGapMM* verPixPerMM );

    if  ( (*ea->eaLayoutDocument)( privateData, fromFormat, &add, &dg ) )
	{
	SDEB(fromName);

	(*ea->eaFreeDocument)( privateData, fromFormat, &add );

	appCleanDrawingData( &add );

	return -1;
	}

    f= fopen( toName, "w" );
    if  ( ! f )
	{
	SXDEB(toName,f);

	(*ea->eaFreeDocument)( privateData, fromFormat, &add );

	appCleanDrawingData( &add );

	return -1;
	}

    if  ( (*ea->eaPrintDocument)( f, privateData, fromFormat, &add,
					fromName, ea, &dg, 1, 0, -1, -1 ) )
	{ SDEB(fromName); rval= -1;	}

    fclose( f );

    (*ea->eaFreeDocument)( privateData, fromFormat, &add );

    appCleanDrawingData( &add );

    return rval;
    }

void appDocSetScrollbarValues(	EditDocument *	ed )
    {
    AppDrawingData *	add= &(ed->edDrawingData);

    int			sliderSize;
    int			maximum;
    int			minimum;
    int			value;

    sliderSize= ed->edVisibleRect.drY1- ed->edVisibleRect.drY0+ 1;
    minimum= add->addBackRect.drY0;
    value= ed->edVisibleRect.drY0;

    if  ( sliderSize >= add->addBackRect.drY1+ 1 )
	{ maximum= ed->edVisibleRect.drY1+ 1;	}
    else{ maximum= add->addBackRect.drY1+ 1;	}

    if  ( value+ sliderSize > maximum )
	{ value= maximum- sliderSize; }

#   ifdef USE_MOTIF
    XtVaSetValues( ed->edVerticalScrollbar,
			XmNminimum,		minimum,
			XmNmaximum,		maximum,
			XmNvalue,		value,
			XmNsliderSize,		sliderSize,
			XmNpageIncrement,	( 9* sliderSize+ 9 )/10,
			XmNincrement,		( sliderSize+ 19 )/ 20,
			NULL );
#   endif

#   ifdef USE_GTK
    ed->edVerticalAdjustment->lower= minimum;
    ed->edVerticalAdjustment->upper= maximum;
    ed->edVerticalAdjustment->value= value;
    ed->edVerticalAdjustment->page_size= sliderSize;
    ed->edVerticalAdjustment->page_increment= ( 9* sliderSize+ 9 )/10;
    ed->edVerticalAdjustment->step_increment= ( sliderSize+ 19 )/ 20;

    gtk_adjustment_changed( ed->edVerticalAdjustment );
#   endif

    sliderSize= ed->edVisibleRect.drX1- ed->edVisibleRect.drX0+ 1;
    minimum= add->addBackRect.drX0;
    value= ed->edVisibleRect.drX0;

    if  ( sliderSize >= add->addBackRect.drX1+ 1 )
	{ maximum= ed->edVisibleRect.drX1+ 1;	}
    else{ maximum= add->addBackRect.drX1+ 1;	}

#   ifdef USE_MOTIF
    XtVaSetValues( ed->edHorizontalScrollbar,
			XmNminimum,	minimum,
			XmNmaximum,	maximum,
			XmNvalue,	value,
			XmNsliderSize,	sliderSize,
			NULL );
#   endif

#   ifdef USE_GTK
    ed->edHorizontalAdjustment->lower= minimum;
    ed->edHorizontalAdjustment->upper= maximum;
    ed->edHorizontalAdjustment->value= value;
    ed->edHorizontalAdjustment->page_size= sliderSize;

    gtk_adjustment_changed( ed->edHorizontalAdjustment );
#   endif

    return;
    }


/************************************************************************/
/*									*/
/*  Redraw a rectangle in the document widget.				*/
/*									*/
/************************************************************************/

APP_EVENT_HANDLER( appDocExposeHandler, w, voided, exposeEvent )
    {
    EditDocument *		ed= (EditDocument *)voided;
    AppDrawingData *		add= &(ed->edDrawingData);
    EditApplication *		ea= ed->edApplication;

    int				ox= ed->edVisibleRect.drX0;
    int				oy= ed->edVisibleRect.drY0;

    DocumentRectangle		drClip;

    /*  1  */
    appCollectExposures( &drClip, add, ox, oy, exposeEvent );

    /*  2,3,4  */
    (*ea->eaDrawRectangle)( w, ed, &drClip, ox, oy );

    appDrawNoClipping( add );

    return;
    }

