/************************************************************************/
/*									*/
/*  Ted: Handle user input.						*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"tedApp.h"
#   include	"tedRuler.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Defines for the key names. This keeps #ifdef's out of the actual	*/
/*  code.								*/
/*									*/
/************************************************************************/

# ifdef USE_MOTIF

#   include	<X11/keysym.h>

#   define KEY_CONTROL_MASK	ControlMask
#   define KEY_SHIFT_MASK	ShiftMask

#  ifdef XK_ISO_Left_Tab
#   define KEY_ISO_Left_Tab	XK_ISO_Left_Tab
#  endif
#   define KEY_i		XK_i
#   define KEY_Tab		XK_Tab
#   define KEY_j		XK_j
#   define KEY_m		XK_m
#   define KEY_KP_Enter		XK_KP_Enter
#   define KEY_Return		XK_Return
#   define KEY_KP_Delete	XK_KP_Delete
#   define KEY_Delete		XK_Delete
#   define KEY_BackSpace	XK_BackSpace
#   define KEY_KP_Home		XK_KP_Home
#   define KEY_Home		XK_Home
#   define KEY_KP_End		XK_KP_End
#   define KEY_End		XK_End
#   define KEY_KP_Left		XK_KP_Left
#   define KEY_Left		XK_Left
#   define KEY_KP_Right		XK_KP_Right
#   define KEY_Right		XK_Right
#   define KEY_KP_Up		XK_KP_Up
#   define KEY_Up		XK_Up
#   define KEY_KP_Down		XK_KP_Down
#   define KEY_Down		XK_Down
#   define KEY_KP_Prior		XK_KP_Prior
#   define KEY_Prior		XK_Prior
#   define KEY_KP_Next		XK_KP_Next
#   define KEY_Next		XK_Next
#   define KEY_c		XK_c
#   define KEY_x		XK_x
#   define KEY_v		XK_v
#   define KEY_Shift_L		XK_Shift_L
#   define KEY_Shift_R		XK_Shift_R
#   define KEY_Alt_L		XK_Alt_L
#   define KEY_Alt_R		XK_Alt_R
#   define KEY_Control_L	XK_Control_L
#   define KEY_Control_R	XK_Control_R
#   define KEY_Caps_Lock	XK_Caps_Lock
#   define KEY_Insert		XK_Insert
#   define KEY_KP_Insert	XK_KP_Insert
#   define KEY_Num_Lock		XK_Num_Lock
# endif

# ifdef USE_GTK

#   include	<gdk/gdkkeysyms.h>

#   define KEY_CONTROL_MASK	GDK_CONTROL_MASK
#   define KEY_SHIFT_MASK	GDK_SHIFT_MASK

#   define KEY_ISO_Left_Tab	GDK_ISO_Left_Tab
#   define KEY_i		GDK_i
#   define KEY_Tab		GDK_Tab
#   define KEY_j		GDK_j
#   define KEY_m		GDK_m
#   define KEY_KP_Enter		GDK_KP_Enter
#   define KEY_Return		GDK_Return
#   define KEY_KP_Delete	GDK_KP_Delete
#   define KEY_Delete		GDK_Delete
#   define KEY_BackSpace	GDK_BackSpace
#   define KEY_KP_Home		GDK_KP_Home
#   define KEY_Home		GDK_Home
#   define KEY_KP_End		GDK_KP_End
#   define KEY_End		GDK_End
#   define KEY_KP_Left		GDK_KP_Left
#   define KEY_Left		GDK_Left
#   define KEY_KP_Right		GDK_KP_Right
#   define KEY_Right		GDK_Right
#   define KEY_KP_Up		GDK_KP_Up
#   define KEY_Up		GDK_Up
#   define KEY_KP_Down		GDK_KP_Down
#   define KEY_Down		GDK_Down
#   define KEY_KP_Prior		GDK_KP_Prior
#   define KEY_Prior		GDK_Prior
#   define KEY_KP_Next		GDK_KP_Next
#   define KEY_Next		GDK_Next
#   define KEY_c		GDK_c
#   define KEY_x		GDK_x
#   define KEY_v		GDK_v
#   define KEY_Shift_L		GDK_Shift_L
#   define KEY_Shift_R		GDK_Shift_R
#   define KEY_Alt_L		GDK_Alt_L
#   define KEY_Alt_R		GDK_Alt_R
#   define KEY_Control_L	GDK_Control_L
#   define KEY_Control_R	GDK_Control_R
#   define KEY_Caps_Lock	GDK_Caps_Lock
#   define KEY_Insert		GDK_Insert
#   define KEY_KP_Insert	GDK_KP_Insert
#   define KEY_Num_Lock		GDK_Num_Lock
# endif

/************************************************************************/
/*									*/
/*  Just log events that pass by for debugging purposes.		*/
/*									*/
/*  NOTE the silly constuction to do away with the 'unused' compiler	*/
/*	 warning.							*/
/*									*/
/************************************************************************/

#   ifdef USE_MOTIF

static void tedLogEvent(	Widget		w,
				void *		voided,
				XEvent *	event,
				Boolean *	pRefused )
    {
    EditDocument *		ed= (EditDocument *)voided;

    appDebug( "EVENT \"%s\": %s\n",
			ed->edTitle, APP_X11EventNames[event->type] );

    *pRefused= 1;

    if  ( ! event )
	{ return;	}
    if  ( ! event )
	{ tedLogEvent( w, voided, event, pRefused );	}
    }

#   endif

/************************************************************************/
/*									*/
/*  Handle mouse clicks.						*/
/*									*/
/************************************************************************/
#   define	TED_DRAG_INTERVAL	(150L)

typedef struct DraggingContext
    {
    int				dcMouseX;
    int				dcMouseY;
    BufferPosition		dcAnchorPosition;
    BufferItem *		dcRootItem;
    SelectionScope		dcSelectionScope;
    EditDocument *		dcEd;
    } DraggingContext;

static APP_EVENT_HANDLER( tedInputDragMouseMove, w, vdc, event )
    {
    DraggingContext *	dc= (DraggingContext *)vdc;

    if  ( appGetCoordinatesFromMouseMoveEvent(
			    &(dc->dcMouseX), &(dc->dcMouseY), w, event ) )
	{ return;	}

    if  ( tedExtendSelectionToXY( dc->dcEd, dc->dcRootItem,
				    &(dc->dcAnchorPosition),
				    &(dc->dcSelectionScope),
				    dc->dcMouseX, dc->dcMouseY ) )
	{ LDEB(1); return;	}

    return;
    }

static APP_EVENT_HANDLER( tedInputDragMouseUp, w, vdc, event )
    {
    DraggingContext *		dc= (DraggingContext *)vdc;
    EditDocument *		ed= dc->dcEd;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    BufferSelection		bsNew= td->tdSelection;

    int				scrolledX= 0;
    int				scrolledY= 0;

    tedSetSelection( ed, &(dc->dcSelectionScope), &bsNew,
						    &scrolledX, &scrolledY );

    tedAdaptToolsToSelection( ed );

    return;
    }

static APP_TIMER_HANDLER( tedTick, voiddc )
    {
    DraggingContext *		dc= (DraggingContext *)voiddc;
    EditDocument *		ed= dc->dcEd;
    AppDrawingData *		add= &(ed->edDrawingData);

    int				ox= ed->edVisibleRect.drX0;
    int				oy= ed->edVisibleRect.drY0;

    int				mouse_x;
    int				mouse_y;

    int				scrolledX= 0;
    int				scrolledY= 0;

#   ifdef USE_MOTIF
    {
    Window		root;
    Window		child;
    int			root_x;
    int			root_y;
    int			win_x;
    int			win_y;
    unsigned int	mask;

    XQueryPointer( add->addDisplay, add->addDrawable,
						    &root, &child,
						    &root_x, &root_y,
						    &win_x, &win_y, &mask );
    mouse_x= win_x; mouse_y= win_y;
    }
#   endif

#   ifdef USE_GTK
    {
    gint		win_x;
    gint		win_y;
    GdkModifierType	mask;

    gdk_window_get_pointer( add->addDrawable, &win_x, &win_y, &mask );

    mouse_x= win_x; mouse_y= win_y;
    }
#   endif

    appScrollToRectangle( ed,
				mouse_x+ ox, mouse_y+ oy,
				mouse_x+ ox, mouse_y+ oy,
				&scrolledX, &scrolledY );

    if  ( scrolledX || scrolledY )
	{
	if  ( tedExtendSelectionToXY( ed, dc->dcRootItem,
					&(dc->dcAnchorPosition),
					&(dc->dcSelectionScope),
					dc->dcMouseX, dc->dcMouseY ) )
	    { LDEB(1);	}
	}

#   ifdef USE_MOTIF
    return;
#   endif
#   ifdef USE_GTK
    return 0;
#   endif
    }

static void tedButton1Pressed(	APP_WIDGET			w,
				EditDocument *			ed,
				APP_EVENT *			downEvent )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    AppDrawingData *		add= &(ed->edDrawingData);
    double			xfac= add->addMagnifiedPixelsPerTwip;
    BufferDocument *		bd= td->tdDocument;
    const DocumentProperties *	dp= &(bd->bdProperties);

    DraggingContext		dc;

    int				ox= ed->edVisibleRect.drX0;
    int				oy= ed->edVisibleRect.drY0;

    int				mouseX;
    int				mouseY;
    int				docX;
    int				docY;

    int				pageY;
    int				page;
    BufferItem *		sectBi= (BufferItem *)0;
    HeaderFooter *		hf= (HeaderFooter *)0;
    const DocumentGeometry *	dgSect;
    int				otherHeaderFooter= 0;

    int				scrolledX= 0;
    int				scrolledY= 0;

    int				sectNr;

    int				button;
    int				upDown;
    int				seq;

    if  ( appGetCoordinatesFromMouseButtonEvent( &mouseX, &mouseY,
						    &button, &upDown, &seq,
						    w, downEvent ) )
	{ LDEB(1); return;	}

    docX= mouseX+ ox;
    docY= mouseY+ oy;

    dc.dcRootItem= &(bd->bdItem);
    dc.dcSelectionScope.ssInHeaderFooter= DOCinBODY;
    dc.dcSelectionScope.ssHeaderFooterSection= -1;

    page= docY/ add->addPageStepPixels;
    pageY= docY- ( page* add->addPageStepPixels );
    for ( sectNr= 0; sectNr < bd->bdItem.biChildCount; sectNr++ )
	{
	sectBi= bd->bdItem.biChildren[sectNr];

	if  ( sectBi->biTopPosition.lpPage <= page	&&
	      sectBi->biBelowPosition.lpPage >= page	)
	    { break;	}
	}

    if  ( sectNr >= bd->bdItem.biChildCount )
	{ LLDEB(page,sectNr); return;	}

    dgSect= &(sectBi->biSectDocumentGeometry);
    if  ( pageY < TWIPStoPIXELS( xfac, dgSect->dgTopMarginTwips ) )
	{
	dc.dcSelectionScope.ssInHeaderFooter=
				docWhatPageHeader( &hf, sectBi, page, dp );
	if  ( ! hf || ! hf->hfItem )
	    { dc.dcSelectionScope.ssInHeaderFooter= DOCinBODY;	}
	}

    if  ( pageY > TWIPStoPIXELS( xfac,
		    dgSect->dgPageHighTwips- dgSect->dgBottomMarginTwips ) )
	{
	dc.dcSelectionScope.ssInHeaderFooter=
				docWhatPageFooter( &hf, sectBi, page, dp );
	if  ( ! hf || ! hf->hfItem )
	    { dc.dcSelectionScope.ssInHeaderFooter= DOCinBODY;	}
	}

    if  ( dc.dcSelectionScope.ssInHeaderFooter		!=
	  td->tdSelectionScope.ssInHeaderFooter		)
	{ otherHeaderFooter= 1;	}

    if  ( dc.dcSelectionScope.ssInHeaderFooter != DOCinBODY )
	{
	if  ( sectNr != td->tdSelectionScope.ssHeaderFooterSection )
	    { otherHeaderFooter= 1;	}
	if  ( page != td->tdSelectionScope.ssHeaderFooterPage )
	    { otherHeaderFooter= 1;	}
	}

    if  ( dc.dcSelectionScope.ssInHeaderFooter != DOCinBODY )
	{
	dc.dcRootItem= hf->hfItem;
	dc.dcSelectionScope.ssHeaderFooterSection= sectNr;
	dc.dcSelectionScope.ssHeaderFooterPage= page;
	}
    else{
	dc.dcSelectionScope.ssHeaderFooterSection= -1;
	dc.dcSelectionScope.ssHeaderFooterPage= -1;
	}

    if  ( otherHeaderFooter )
	{
	if  ( seq > 1 )
	    {
	    BufferPosition		bp;
	    BufferItem *		bi;
	    int				inHeadFoot;

	    int				done;
	    DocumentRectangle		drChanged;

	    docInitPosition( &bp );

	    inHeadFoot= dc.dcSelectionScope.ssInHeaderFooter;

	    if  ( dc.dcSelectionScope.ssInHeaderFooter == DOCinBODY )
		{ bi= sectBi;	}
	    else{
		if  ( tedRedoHeaderFooterLayout( &done, hf, page,
				    dc.dcSelectionScope.ssInHeaderFooter,
				    sectBi, bd, add, &drChanged ) )
		    { LDEB(page); return;	}

		bi= hf->hfItem;
		}

	    if  ( tedFindPosition( bi, add, docX, docY, &bp ) )
		{ LLDEB(docX,docY); return;	}

	    tedSetSelectedPosition( ed, &(dc.dcSelectionScope), &bp,
						    &scrolledX, &scrolledY );

	    tedAdaptToolsToPosition( ed );

	    appExposeRectangle( add, 0, 0, 0, 0 );
	    }

	return;
	}

    if  ( seq > 1 )
	{
	int			wasObject= 0;

	BufferSelection		bs;

	docInitSelection( &bs );

	if  ( tedFindPosition( dc.dcRootItem, add, docX, docY,
						    &(dc.dcAnchorPosition) ) )
	    { LLDEB(docX,docY); return; }

	bs.bsBegin= dc.dcAnchorPosition;
	bs.bsEnd= dc.dcAnchorPosition;

	if  ( seq > 2 )
	    {
	    docLinePositions( &(bs.bsBegin), &(bs.bsEnd),
			dc.dcAnchorPosition.bpBi, dc.dcAnchorPosition.bpLine );
	    }
	else{
	    TextParticule *	tp;

	    tp= dc.dcAnchorPosition.bpBi->biParaParticules+
					    dc.dcAnchorPosition.bpParticule;

	    bs.bsBegin.bpStroff= tp->tpStroff;
	    bs.bsEnd.bpStroff= tp->tpStroff+ tp->tpStrlen;

	    if  ( tp->tpKind == DOCkindOBJECT )
		{ wasObject= 1;	}
	    }

	bs.bsDirection= 1;
	bs.bsAnchor= bs.bsBegin;

	tedSetSelection( ed, &(dc.dcSelectionScope), &bs,
						    &scrolledX, &scrolledY );

	tedAdaptToolsToSelection( ed );

	if  ( wasObject )
	    {
	    tedPositionCoordinates( &(bs.bsBegin), add );
	    tedSetObjectWindows( ed, &(bs.bsBegin), ox, oy );
	    }

	return;
	}

    if  ( tedFindPosition( dc.dcRootItem, add, docX, docY,
						    &(dc.dcAnchorPosition) ) )
	{ LLDEB(docX,docY); return;	}

    tedSetSelectedPosition( ed, &(dc.dcSelectionScope), &(dc.dcAnchorPosition),
						    &scrolledX, &scrolledY );

    tedAdaptToolsToPosition( ed );

    if  ( ed->edFileReadOnly )
	{
	int		startPart;
	int		endPart;
	const char *	fileName;
	int		fileSize;
	const char *	markName;
	int		markSize;

	if  ( ! docGetHyperlinkForPosition( bd, &(dc.dcAnchorPosition),
				&startPart, &endPart,
				&fileName, &fileSize, &markName, &markSize ) )
	    {
	    tedFollowLink( (APP_WIDGET)0, (APP_WIDGET)0, ed,
				    fileName, fileSize, markName, markSize );

	    return;
	    }
	}

    dc.dcEd= ed;
    dc.dcMouseX= mouseX;
    dc.dcMouseY= mouseY;

    appRunDragLoop( w, ed->edApplication, downEvent,
				tedInputDragMouseUp,
				tedInputDragMouseMove,
				TED_DRAG_INTERVAL, tedTick,
				(void *)&dc );

    return;
    }

/************************************************************************/
/*									*/
/*  Handle keyboard input.						*/
/*									*/
/*  a)  Handle miscelaneous keysyms as keysyms, even if they have a	*/
/*	string representation.						*/
/*									*/
/************************************************************************/

static void tedInputSetSelectedPosition( EditDocument *		ed,
					const SelectionScope *	ss,
					const BufferPosition *	bp )
    {
    int			scrolledX= 0;
    int			scrolledY= 0;

    tedSetSelectedPosition( ed, ss, bp, &scrolledX, &scrolledY );

    tedAdaptToolsToPosition( ed );

    return;
    }

static void tedInputExtendSelection(	EditDocument *		ed,
					const SelectionScope *	ss,
					const BufferPosition *	bpSet )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    const AppDrawingData *	add= &(ed->edDrawingData);
    BufferPosition		bpNew;

    bpNew= *bpSet;

    tedPositionCoordinates( &bpNew, add );

    tedExtendSelectionToPosition( ed, ss, &(td->tdSelection.bsAnchor), &bpNew );

    tedSelectionRectangle( &(td->tdSelectedRectangle),
						 add, &(td->tdSelection) );

    tedAdaptToolsToSelection( ed );

    return;
    }

static void tedInputChangeSelection(	EditDocument *		ed,
					unsigned int		keyState,
					const SelectionScope *	ss,
					const BufferPosition *	bp )
    {
    if  ( keyState & KEY_SHIFT_MASK )
	{
	tedInputExtendSelection( ed, ss, bp );
	return;
	}

    tedInputSetSelectedPosition( ed, ss, bp );

    return;
    }

static int tedMoveRightOnKey(	BufferPosition *	bpNew,
				int			keyState,
				const AppDrawingData *	add )
    {
    if  ( keyState & KEY_CONTROL_MASK )
	{
	const int	lastOne= 1;

	if  ( docNextWord( bpNew, lastOne ) )
	    { return -1;	}

	tedPositionCoordinates( bpNew, add );
	}
    else{
	if  ( tedNextPosition( add, bpNew ) )
	    { return -1;	}
	}

    return 0;
    }

static int tedMoveLeftOnKey(	BufferPosition *	bpNew,
				int			keyState,
				const AppDrawingData *	add )
    {
    const int	lastOne= 1;

    if  ( keyState & KEY_CONTROL_MASK )
	{
	if  ( docPrevWord( bpNew, lastOne ) )
	    { return -1;	}

	tedPositionCoordinates( bpNew, add );
	}
    else{
	if  ( tedPrevPosition( add, bpNew, lastOne ) )
	    { return -1;	}
	}

    return 0;
    }

static void tedProcessKeyEvent(	EditDocument *		ed,
				APP_EVENT *		event,
				APP_KEY_VALUE		keySym,
				unsigned char		scratch[40],
				int			gotString,
				int			gotKey,
				unsigned int		state )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SelectionScope *		ss= &(td->tdSelectionScope);
    const AppDrawingData *	add= &(ed->edDrawingData);
    BufferSelection *		bs= &(td->tdSelection);

    if  ( gotString > 0 )
	{
	AppPhysicalFont *	apf;

	if  ( ! td->tdCanReplaceSelection )
	    { return;	}

	if  ( td->tdCurrentPhysicalFont < 0				||
	      td->tdCurrentPhysicalFont >=
				add->addPhysicalFontList.apflCount	)
	    {
	    LDEB(td->tdCurrentPhysicalFont);
	    LDEB(add->addPhysicalFontList.apflCount);
	    return;
	    }

	apf= add->addPhysicalFontList.apflFonts+ td->tdCurrentPhysicalFont;

	if  ( appCharExistsInFont( apf->apfFontStruct, scratch[0] ) )
	    {
	    tedAppReplaceSelection( ed, scratch, gotString );
	    return;
	    }
	}

    if  ( gotKey == 0 )
	{ return;	}

    switch( keySym )
	{
	BufferPosition	bpNew;

#	ifdef KEY_ISO_Left_Tab
	case  KEY_ISO_Left_Tab:
	    if  ( bs->bsBegin.bpBi			&&
		  bs->bsBegin.bpBi->biParaInTable	&&
		  ! ( state & KEY_CONTROL_MASK )	)
		{ goto shiftTab;	}
	    else{ return;		}
#	endif

	case KEY_i:
	    if  ( state != KEY_CONTROL_MASK )
		{ goto unknown; }
	    /*FALLTHROUGH*/
	case KEY_Tab:
	    if  ( bs->bsBegin.bpBi			&&
		  bs->bsBegin.bpBi->biParaInTable	&&
		  ! ( state & KEY_CONTROL_MASK )	)
		{
		if  ( state & KEY_SHIFT_MASK )
		    {
		  shiftTab:
		    if  ( docFirstPosition( bs->bsBegin.bpBi->biParent,
							    &bpNew )	||
			  tedPrevPosition( add, &bpNew, /*last=*/ 0 )	)
			{ return;	}
		    }
		else{
		    if  ( docLastPosition( bs->bsBegin.bpBi->biParent,
							    &bpNew )	||
			  tedNextPosition( add, &bpNew )		)
			{ return;	}
		    }

		tedInputSetSelectedPosition( ed, ss, &bpNew );

		return;
		}

	    if  ( state & KEY_SHIFT_MASK )
		{ return;	}

	    if  ( ! td->tdCanReplaceSelection )
		{ return;	}

	    tedAppReplaceSelection( ed, scratch, 0 );

	    tedSetTab( ed );

	    appDocumentChanged( ed, 1 );

	    return;

	case KEY_j: case KEY_m:
	    if  ( state != KEY_CONTROL_MASK )
		{ goto unknown; }
	    /*FALLTHROUGH*/
	case KEY_KP_Enter:
	case KEY_Return:
	    if  ( ! td->tdCanReplaceSelection )
		{ return;	}

	    tedAppReplaceSelection( ed, scratch, 0 );

	    tedSplitParagraph( ed,
			    state == KEY_CONTROL_MASK	&&
			    keySym != KEY_j		&&
			    keySym != KEY_m		);

	    appDocumentChanged( ed, 1 );

	    return;

	case KEY_KP_Delete:
	case KEY_Delete:
	    if  ( ! td->tdCanReplaceSelection )
		{ return;	}

	    if  ( tedHasIBarSelection( td ) )
		{
		bpNew= bs->bsBegin;

		if  ( tedMoveRightOnKey( &bpNew, state, add ) )
		    { return;	}

		if  ( ! docPositionsInsideCell( &(bs->bsBegin), &bpNew ) )
		    { return;	}

		if  ( tedMoveRightOnKey( &(bs->bsEnd), state, add ) )
		    { return;	}
		}

	    tedAppReplaceSelection( ed, scratch, 0 );

	    return;

	case KEY_BackSpace:
	    if  ( ! td->tdCanReplaceSelection )
		{ return;	}

	    if  ( tedHasIBarSelection( td ) )
		{
		bpNew= bs->bsBegin;

		if  ( tedMoveLeftOnKey( &bpNew, state, add ) )
		    { return;	}

		if  ( ! docPositionsInsideCell( &bpNew, &(bs->bsBegin) ) )
		    {
		    const int		keyState= 0;

		    tedInputChangeSelection( ed, keyState, ss, &bpNew );
		    return;
		    }

		if  ( tedMoveLeftOnKey( &(bs->bsBegin), state, add ) )
		    { return;	}
		}

	    tedAppReplaceSelection( ed, scratch, 0 );
	    return;

	case KEY_KP_Home:
	case KEY_Home:
	    if  ( ( state & KEY_SHIFT_MASK ) && bs->bsDirection >= 0 )
		{ bpNew= bs->bsEnd;	}
	    else{ bpNew= bs->bsBegin;	}

	    if  ( tedBeginOfLine( add, &bpNew ) )
		{ return;	}

	    tedInputChangeSelection( ed, state, ss, &bpNew );
	    return;

	case KEY_KP_End:
	case KEY_End:
	    if  ( ! ( state & KEY_SHIFT_MASK ) || bs->bsDirection >= 0 )
		{ bpNew= bs->bsEnd;	}
	    else{ bpNew= bs->bsBegin;	}

	    if  ( tedEndOfLine( add, &bpNew ) )
		{ return;	}

	    tedInputChangeSelection( ed, state, ss, &bpNew );
	    return;

	case KEY_KP_Left:
	case KEY_Left:
	    if  ( ( state & KEY_SHIFT_MASK ) && bs->bsDirection >= 0 )
		{ bpNew= bs->bsEnd;	}
	    else{ bpNew= bs->bsBegin;	}

	    if  ( tedMoveLeftOnKey( &bpNew, state, add ) )
		{ return;	}

	    tedInputChangeSelection( ed, state, ss, &bpNew );
	    return;

	case KEY_KP_Right:
	case KEY_Right:
	    if  ( ! ( state & KEY_SHIFT_MASK ) || bs->bsDirection >= 0 )
		{ bpNew= bs->bsEnd;	}
	    else{ bpNew= bs->bsBegin;	}

	    if  ( tedMoveRightOnKey( &bpNew, state, add ) )
		{ return;	}

	    tedInputChangeSelection( ed, state, ss, &bpNew );
	    return;

	case KEY_KP_Up:
	case KEY_Up:
	    if  ( ( state & KEY_SHIFT_MASK ) && bs->bsDirection >= 0 )
		{ bpNew= bs->bsEnd;	}
	    else{ bpNew= bs->bsBegin;	}

	    if  ( state & KEY_CONTROL_MASK )
		{
		const int	lastOne= 1;

		if  ( bpNew.bpStroff == 0 )
		    {
		    bpNew.bpBi= docPrevParagraph( bpNew.bpBi );
		    if  ( ! bpNew.bpBi )
			{ return;	}

		    if  ( docParaBegin( &bpNew, lastOne ) )
			{ return;	}
		    }
		else{
		    if  ( docParaBegin( &bpNew, lastOne ) )
			{ return;	}
		    }
		}
	    else{
		if  ( tedPrevLine( add, &bpNew ) )
		    { return;	}
		}

	    tedInputChangeSelection( ed, state, ss, &bpNew );
	    return;

	case KEY_KP_Down:
	case KEY_Down:
	    if  ( ! ( state & KEY_SHIFT_MASK ) || bs->bsDirection >= 0 )
		{ bpNew= bs->bsEnd;	}
	    else{ bpNew= bs->bsBegin;	}

	    if  ( state & KEY_CONTROL_MASK )
		{
		const int	lastOne= 1;

		if  ( bpNew.bpStroff < bpNew.bpBi->biParaStrlen		&&
		      ( state & KEY_SHIFT_MASK )		)
		    {
		    if  ( docParaEnd( &bpNew, lastOne ) )
			{ return;	}
		    }
		else{
		    bpNew.bpBi= docNextParagraph( bpNew.bpBi );
		    if  ( ! bpNew.bpBi )
			{ return;	}

		    if  ( docParaBegin( &bpNew, lastOne ) )
			{ return;	}
		    }
		}
	    else{
		if  ( tedNextLine( add, &bpNew ) )
		    { return;	}
		}

	    tedInputChangeSelection( ed, state, ss, &bpNew );
	    return;

	case KEY_KP_Prior:
	case KEY_Prior:
	    if  ( ( state & KEY_SHIFT_MASK ) && bs->bsDirection >= 0 )
		{ bpNew= bs->bsEnd;	}
	    else{ bpNew= bs->bsBegin;	}

	    if  ( tedPageUp( &bpNew, td->tdDocument, add,
					ed->edVisibleRect.drY1-
					ed->edVisibleRect.drY0 )	&&
		  tedFirstPosition( add, td->tdDocument, &bpNew )	)
		{ return;	}

	    tedInputChangeSelection( ed, state, ss, &bpNew );
	    return;

	case KEY_KP_Next:
	case KEY_Next:
	    if  ( ! ( state & KEY_SHIFT_MASK ) || bs->bsDirection >= 0 )
		{ bpNew= bs->bsEnd;	}
	    else{ bpNew= bs->bsBegin;	}

	    if  ( tedPageDown( &bpNew, td->tdDocument, add,
					add->addDocRect.drY1,
					ed->edVisibleRect.drY1-
					ed->edVisibleRect.drY0 )	&&
		  tedLastPosition( add, td->tdDocument, &bpNew )	)
		{ return;	}

	    tedInputChangeSelection( ed, state, ss, &bpNew );
	    return;

#	if 0
	case KEY_c:
	    /************************************************************/
	    /*  Usually this key is intercepted by the 'Edit' menu.	*/
	    /************************************************************/
	    if  ( state != KEY_CONTROL_MASK )
		{ goto defaultCase;	}

	    tedDocCopy( ed, (XEvent *)keyEvent );
	    return;
	case KEY_x:
	    /************************************************************/
	    /*  Usually this key is intercepted by the 'Edit' menu.	*/
	    /************************************************************/
	    if  ( state != KEY_CONTROL_MASK )
		{ goto defaultCase;	}

	    tedDocCut( ed, (XEvent *)keyEvent );
	    return;
	case KEY_v:
	    /************************************************************/
	    /*  Usually this key is intercepted by the 'Edit' menu.	*/
	    /************************************************************/
	    if  ( state != KEY_CONTROL_MASK )
		{ goto defaultCase;	}

	    appDocAskForPaste( ed, "PRIMARY", keyEvent->time );
	    return;
#	endif

	case KEY_Shift_L:
	case KEY_Shift_R:
	case KEY_Alt_L:
	case KEY_Alt_R:
	case KEY_Control_L:
	case KEY_Control_R:
	case KEY_Caps_Lock:
	case KEY_Insert:
	case KEY_KP_Insert:
	case KEY_Num_Lock:
	    return;
	default: unknown:
#	    ifdef USE_GTK
	    gtk_accel_group_activate( ed->edToplevel.atAccelGroup,
							    keySym, state );
#	    endif

#	    ifdef APP_DEBUG
#	    ifdef USE_MOTIF
	    appDebug( "INPUT \"%s\": %s (%s%s<Key>%s) \"%.*s\"\n",
			ed->edTitle, APP_X11EventNames[event->type],
			(state&KEY_SHIFT_MASK)?"Shift":"",
			(state&KEY_CONTROL_MASK)?"Ctrl":"",
			XKeysymToString( keySym ), gotString, scratch );
#	    endif
#	    endif
	}

    return;
    }

APP_EVENT_HANDLER( tedKeyPressed, w, voided, keyEvent )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferSelection *		bs= &(td->tdSelection);

    APP_KEY_VALUE		keySym;
    unsigned char		scratch[40];
    int				gotString;
    int				gotKey;
    unsigned int		state;

    if  ( ! bs->bsBegin.bpBi )
	{
#	ifdef USE_MOTIF
	const AppDrawingData *	add= &(ed->edDrawingData);

	XBell( add->addDisplay, 0 );
#	endif

#	ifdef USE_GTK
	gdk_beep();
#	endif

	return;
	}

    appGuiGetStringFromKeyboardEvent( ed->edInputContext, w, keyEvent,
				&gotString, &gotKey, &state,
				(char *)scratch, sizeof(scratch), &keySym );

    tedStopCursorBlink( ed );

    if  ( gotString > 0 || gotKey > 0 )
	{
	tedProcessKeyEvent( ed, keyEvent,
				keySym, scratch, gotString, gotKey, state );
	}

    if  ( tedHasIBarSelection( td ) )
	{ tedStartCursorBlink( ed );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Drag routine for inserted objects.					*/
/*									*/
/************************************************************************/

typedef struct TedObjectDrag
    {
    EditDocument *	todEd;

    InsertedObject *	todIo;
    BufferPosition	todBp;

    int			todMouseX;
    int			todMouseY;

    int			todRight;
    int			todBottom;
    } TedObjectDrag;

static void tedObjectHandleMove(	TedObjectDrag *		tod,
					int			mouseX,
					int			mouseY )
    {
    EditDocument *		ed= tod->todEd;
    AppDrawingData *		add= &(ed->edDrawingData);
    int				ox= ed->edVisibleRect.drX0;
    int				oy= ed->edVisibleRect.drY0;

    const BufferItem *		bi;
    const TextParticule *	tp;
    const TextLine *		tl;

    int				moved= 0;

    int				wide= tod->todIo->ioDragWide;
    int				high= tod->todIo->ioDragHigh;

    bi= tod->todBp.bpBi;
    tp= bi->biParaParticules+ tod->todBp.bpParticule;
    tl= bi->biParaLines+ tod->todBp.bpLine;

    if  ( tod->todRight && tod->todMouseX != mouseX )
	{ tod->todIo->ioDragWide += mouseX- tod->todMouseX; moved= 1; }

    if  ( tod->todBottom && tod->todMouseY != mouseY )
	{ tod->todIo->ioDragHigh += mouseY- tod->todMouseY; moved= 1; }

    if  ( wide < tod->todIo->ioDragWide )
	{ wide=  tod->todIo->ioDragWide;	}
    if  ( high < tod->todIo->ioDragHigh )
	{ high=  tod->todIo->ioDragHigh;	}

    if  ( moved )
	{
	int		baseline;

	baseline= tl->tlTopPosition.lpYPixels+ tl->tlLineAscentPixels;

	appExposeRectangle( add,
			tp->tpX0- ox,
			baseline- tod->todIo->ioPixelsHigh- oy, wide, high );
	}

    tod->todMouseX= mouseX;
    tod->todMouseY= mouseY;
    }

static APP_EVENT_HANDLER( tedObjectDragMouseMove, w, vtod, event )
    {
    TedObjectDrag *		tod= (TedObjectDrag *)vtod;

    int				x;
    int				y;

    if  ( appGetCoordinatesFromMouseMoveEvent( &x, &y, w, event ) )
	{ return;	}

    tedObjectHandleMove( tod, x, y );

    return;
    }

static APP_EVENT_HANDLER( tedObjectDragMouseUp, w, vtod, event )
    {
    TedObjectDrag *		tod= (TedObjectDrag *)vtod;

    int				x;
    int				y;

    int				button;
    int				upDown;
    int				seq;

    if  ( appGetCoordinatesFromMouseButtonEvent(
				&x, &y, &button, &upDown, &seq, w, event ) )
	{ return;	}

    tedObjectHandleMove( tod, x, y );

    return;
    }

static int tedObjectDrag(	APP_WIDGET	w,
				EditDocument *	ed,
				APP_EVENT *	downEvent )
    {
    EditApplication *	ea= ed->edApplication;
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    int			ox= ed->edVisibleRect.drX0;
    int			oy= ed->edVisibleRect.drY0;

    int			wide;
    int			high;

    int			button;
    int			upDown;
    int			seq;

    int			x;
    int			y;

    TedObjectDrag	tod;

    tod.todEd= ed;

    tod.todBottom= tod.todRight= 0;

    if  ( appGetCoordinatesFromMouseButtonEvent(
				    &(tod.todMouseX), &(tod.todMouseY),
				    &button, &upDown, &seq, w, downEvent ) )
	{ LDEB(1); return -1;	}

    docInitPosition( &(tod.todBp) );
    if  ( tedGetObjectSelection( td, &(tod.todBp), &(tod.todIo) ) )
	{ return 2;	}

    wide= tod.todIo->ioPixelsWide;
    high= tod.todIo->ioPixelsHigh;
    x= tod.todBp.bpXPixels;
    y= tod.todBp.bpBaselinePixels;

    if  ( seq > 1					||
	  upDown < 1					||
	  tod.todMouseX+ ox < x				||
	  tod.todMouseX+ ox > x+ wide			||
	  tod.todMouseY+ oy > y				||
	  tod.todMouseY+ oy < y- high			)
	{ return 2;	}

    if  ( tod.todMouseX+ ox >= x + wide/ 2- RESIZE_BLOCK/ 2	&&
	  tod.todMouseX+ ox <  x + wide/ 2+ RESIZE_BLOCK/ 2	&&
	  tod.todMouseY+ oy >= y- RESIZE_BLOCK		)
	{ tod.todBottom= 1;	}

    if  ( tod.todMouseX+ ox >= x + wide- RESIZE_BLOCK		&&
	  tod.todMouseY+ oy >= y - high/ 2- RESIZE_BLOCK/ 2 	&&
	  tod.todMouseY+ oy <  y - high/ 2+ RESIZE_BLOCK/ 2	)
	{ tod.todRight= 1;	}

    if  ( tod.todMouseX+ ox >= x + wide- RESIZE_BLOCK		&&
	  tod.todMouseY+ oy >= y- RESIZE_BLOCK			)
	{ tod.todRight= 1; tod.todBottom= 1;	}

    if  ( ! tod.todRight && ! tod.todBottom )
	{ return 1;	}

    tod.todIo->ioDragWide= tod.todIo->ioPixelsWide;
    tod.todIo->ioDragHigh= tod.todIo->ioPixelsHigh;

    appRunDragLoop( w, ea, downEvent,
				tedObjectDragMouseUp,
				tedObjectDragMouseMove,
				0, (APP_TIMER_CALLBACK)0,
				(void *)&tod );

    if  ( tod.todIo->ioDragWide != tod.todIo->ioPixelsWide	||
	  tod.todIo->ioDragHigh != tod.todIo->ioPixelsHigh	)
	{
	AppDrawingData *	add= &(ed->edDrawingData);
	double			xfac= add->addMagnifiedPixelsPerTwip;

	if  ( tod.todIo->ioDragWide != tod.todIo->ioPixelsWide )
	    {
	    TextParticule *	tp;

	    tod.todIo->ioScaleX= ( 100.0* tod.todIo->ioDragWide )/
			    TWIPStoPIXELS( xfac, tod.todIo->ioTwipsWide );

	    tod.todIo->ioPixelsWide= TWIPStoPIXELS( xfac,
		    ( tod.todIo->ioScaleX* tod.todIo->ioTwipsWide )/ 100 );

	    tp= tod.todBp.bpBi->biParaParticules+ tod.todBp.bpParticule;

	    tp->tpPixelsWide= tod.todIo->ioPixelsWide;
	    }

	if  ( tod.todIo->ioDragHigh != tod.todIo->ioPixelsHigh )
	    {
	    tod.todIo->ioScaleY= ( 100.0* tod.todIo->ioDragHigh )/
			    TWIPStoPIXELS( xfac, tod.todIo->ioTwipsHigh );

	    tod.todIo->ioPixelsHigh= TWIPStoPIXELS( xfac,
		    ( tod.todIo->ioScaleY* tod.todIo->ioTwipsHigh )/ 100 );
	    }

	if  ( tedResizeObject( ed, &(td->tdSelection.bsBegin) ) )
	    { LDEB(1);	}

	appDocumentChanged( ed, 1 );
	}

    tod.todIo->ioDragWide= 0;
    tod.todIo->ioDragHigh= 0;

    ox= ed->edVisibleRect.drX0;
    oy= ed->edVisibleRect.drY0;

    tedSetObjectWindows( ed, &(td->tdSelection.bsBegin), ox, oy );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Handle mouse button down events for the document widget.		*/
/*									*/
/************************************************************************/

APP_EVENT_HANDLER( tedMouseButtonPressed, w, voided, downEvent )
    {
    EditDocument *	ed= (EditDocument *)voided;
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    tedStopCursorBlink( ed );

#   ifdef USE_MOTIF
    switch( downEvent->xbutton.button )
	{
	case Button1:
	    if  ( downEvent->xbutton.subwindow )
		{
		tedObjectDrag( w, ed, downEvent );
		return;
		}

	    tedButton1Pressed( w, ed, downEvent );
	    break;
	case Button2:
	    appDocAskForPaste( ed, "PRIMARY" );
	    break;
	case Button3:
	    break;
	default:
	    LDEB(downEvent->xbutton.button);
	    break;
	}
#   endif

#   ifdef USE_GTK
    switch( downEvent->button.button )
	{
	int	res;

	case 1:
	    res= tedObjectDrag( w, ed, downEvent );
	    if  ( res < 1 )
		{ return;	}

	    tedButton1Pressed( w, ed, downEvent );
	    break;
	case 2:
	    appDocAskForPaste( ed, "PRIMARY" );
	    break;
	case 3:
	    break;
	default:
	    LDEB(downEvent->button.button);
	    break;
	}
#   endif

    if  ( tedHasIBarSelection( td ) )
	{ tedStartCursorBlink( ed );	}

    return;
    }

