/************************************************************************/
/*									*/
/*  Editor functionality.						*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"tedApp.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Set an I Bar selection and cause the I Bar to be drawn.		*/
/*									*/
/************************************************************************/

static int tedEditSetIBarSelection(	EditDocument *		ed,
					BufferItem *		bi,
					int *			pScrolledX,
					int *			pScrolledY,
					int			stroff )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    BufferPosition *	bp= &(td->tdSelection.bsBegin);
    SelectionScope	ss= td->tdSelectionScope;

    const int		lastOne= 1;

    td->tdVisibleSelectionCopied= 0;

    if  ( td->tdObjectSelected )
	{ tedHideObjectWindows( ed );	}

    if  ( docFindLineAndParticule( bi, stroff, bp, lastOne ) )
	{ LDEB(stroff); return -1;	}

    tedSetSelectedPosition( ed, &ss, bp, pScrolledX, pScrolledY );

    tedAdaptToolsToPosition( ed );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Do the last steps that set an I-Bar selection after an editing	*/
/*  action.								*/
/*									*/
/*  1)  The order in which the size of the window is adapted and the	*/
/*	window is scrolled to the selection depends on the direction	*/
/*	of the change, as the scrollbars complain when the corrent	*/
/*	position is not in range.					*/
/*									*/
/************************************************************************/

int tedEditFinishIBarSelection(	EditDocument *			ed,
				BufferItem *			bi,
				const DocumentRectangle *	drChanged,
				int				oldBackY1,
				int				stroff )
    {
    AppDrawingData *		add= &(ed->edDrawingData);

    int				scrolledX= 0;
    int				scrolledY= 0;

    /*  1  */
    if  ( add->addBackRect.drY1 > oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    if  ( bi )
	{ tedEditSetIBarSelection( ed, bi, &scrolledX, &scrolledY, stroff ); }
    else{ XDEB(bi);	}


    /*  1  */
    if  ( add->addBackRect.drY1 < oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    tedExposeRectangle( ed, drChanged, scrolledX, scrolledY );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Delete some buffer items but first close the objects in them.	*/
/*									*/
/************************************************************************/

void tedDeleteItems(	BufferDocument *	bd,
			BufferItem *		bi,
			int			first,
			int			count,
			AppDrawingData *	add )
    {
    int		i;
    int		f= first;

    for ( i= 0; i < count; f++, i++ )
	{
	docCloseItemObjects( bd, bi->biChildren[f],
					(void *)add, tedCloseObject );
	}

    docDeleteItems( bd, bi, first, count );
    }

/************************************************************************/
/*									*/
/*  Replace the selection in the document with new text.		*/
/*									*/
/*  b)  Replace the tail of the beginning paragraph with the new text.	*/
/*  c)  Merge the two paragraphs.					*/
/*  d)  Erase all paragraphs after the beginning of the selection	*/
/*	to, including the end.						*/
/*  e)  Update the paragraph buffer and the particule administration.	*/
/*									*/
/*  B)  The first particule of the line was split, also reformat the	*/
/*	previous line.							*/
/*	If paragraphs were merged, redraw the whole neighbourhood.	*/
/*  C)  - Recalculate the geometry from the start of the selection to	*/
/*	  the end of the paragraph.					*/
/*	- Redraw the affected part of the paragraph.			*/
/*  6)  Update the notion of current particule and current line.	*/
/*  7)  Redraw the I-Bar.						*/
/*									*/
/************************************************************************/

/*  B  */
static void tedAdjustRedrawBegin( TedDocument *			td,
				const DocumentRectangle *	drBack,
				DocumentRectangle *		drChanged,
				int *				pLine,
				int				part )
    {
    TextLine *	tl= td->tdSelection.bsBegin.bpBi->biParaLines+ *pLine;

    if  ( *pLine == 0 )
	{
	if  ( td->tdSelection.bsBegin.bpBi != td->tdSelection.bsEnd.bpBi )
	    {
	    drChanged->drX0= drBack->drX0;
	    drChanged->drX1= drBack->drX1;

	    if  ( drChanged->drY0 > tl->tlTopPosition.lpYPixels )
		{ drChanged->drY0=  tl->tlTopPosition.lpYPixels;	}
	    }

	return;
	}

    if  ( part == tl->tlFirstParticule					||
	  td->tdSelection.bsBegin.bpBi != td->tdSelection.bsEnd.bpBi	)
	{
	(*pLine)--; tl--;

	drChanged->drX0= drBack->drX0;
	drChanged->drX1= drBack->drX1;

	if  ( drChanged->drY0 > tl->tlTopPosition.lpYPixels )
	    { drChanged->drY0=  tl->tlTopPosition.lpYPixels;	}
	}

    return;
    }

int tedReplaceSelection(	DocumentRectangle *	drChanged,
				BufferSelection *	bsReplacement,
				EditDocument *		ed,
				const unsigned char *	addedText,
				int			addedLength )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SelectionScope *		ss= &(td->tdSelectionScope);
    BufferDocument *		bd= td->tdDocument;

    int				stroffShift= 0;
    int				partShift= 0;

    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    int				paragraphsMerged;
	
    int				line;
    int				part;

    int				stroff;

    TextLine *			tl;
    int				upto;

    LayoutPosition		lpBelow;

    BufferSelection		bsRep;

    docInitSelection( &bsRep );

    stroff= td->tdSelection.bsBegin.bpStroff;
    part= td->tdSelection.bsBegin.bpParticule;
    line= td->tdSelection.bsBegin.bpLine;

    tl= bi->biParaLines+ line;
    upto= tl->tlStroff+ tl->tlStrlen;

    paragraphsMerged= td->tdSelection.bsEnd.bpBi != bi;
    lpBelow= td->tdSelection.bsEnd.bpBi->biBelowPosition;

    /*  b,c,d,e  */
    if  ( docReplaceSelection( bd, &(td->tdSelection),
		    &partShift, &stroffShift,
		    addedText, addedLength, td->tdCurrentTextAttribute,
		    (void *)add, tedCloseObject ) )
	{ LDEB(addedLength); return -1;	}

    /*  B  */
    tedAdjustRedrawBegin( td, &(add->addBackRect), drChanged, &line, part );

    if  ( paragraphsMerged )
	{ bi->biBelowPosition= lpBelow; }

    /*  C  */
    if  ( tedAdjustParagraphLayout( bi, line, stroffShift, upto+ addedLength,
				ss->ssInHeaderFooter, add, bd, drChanged ) )
	{ LDEB(line); }

    if  ( paragraphsMerged )
	{
	BufferSelection		bsLayout;
	BufferItem *		endBi= docNextParagraph( bi );
	BufferItem *		layoutBi;
	const int		nearest= 1;

	if  ( ! endBi )
	    { endBi= bi;	}
	
	docInitSelection( &bsLayout );
	bsLayout.bsBegin.bpBi= bi;
	bsLayout.bsBegin.bpParticule= 0;
	bsLayout.bsBegin.bpStroff= 0;

	bsLayout.bsEnd.bpBi= endBi;
	bsLayout.bsEnd.bpParticule= endBi->biParaParticuleCount- 1;
	bsLayout.bsEnd.bpStroff= endBi->biParaStrlen;

	layoutBi= docGetSelectionRoot( bd, nearest, ss, &bsLayout );

	if  ( ! layoutBi )
	    { XDEB(layoutBi); return -1;	}


	if  ( tedLayoutItem( layoutBi, bd,
				    ss->ssInHeaderFooter, add, drChanged ) )
	    { LDEB(1);	}
	}

    if  ( tedOpenItemObjects( bi, &(ed->edColors), add ) )
	{ LDEB(1);	}

    bsRep.bsBegin.bpBi= bi;
    bsRep.bsBegin.bpStroff= stroff;

    bsRep.bsEnd.bpBi= bi;
    bsRep.bsEnd.bpStroff= stroff+ addedLength;

    docFindLineAndParticule( bsRep.bsBegin.bpBi, bsRep.bsBegin.bpStroff,
							&(bsRep.bsBegin), 1 );
    docFindLineAndParticule( bsRep.bsEnd.bpBi, bsRep.bsEnd.bpStroff,
							&(bsRep.bsEnd), 0 );

    bsRep.bsAnchor= bsRep.bsBegin;
    bsRep.bsCol1= bsRep.bsCol0;
    bsRep.bsDirection= bsRep.bsEnd.bpStroff > bsRep.bsBegin.bpStroff;

    *bsReplacement= bsRep;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Merge the selection into one paragraph.				*/
/*									*/
/************************************************************************/

static int tedIsIndentationParticule(	const BufferItem *	bi,
					const TextParticule *	tp )
    {
    const unsigned char *	s= bi->biParaString+ tp->tpStroff;
    int				i;

    if  ( tp->tpKind == DOCkindTAB )
	{ return 1;	}

    if  ( tp->tpKind != DOCkindTEXT )
	{ return 0;	}

    for ( i= 0; i < tp->tpStrlen; i++ )
	{
	if  ( *(s++) != ' ' )
	    { return 0;	}
	}

    return 1;
    }

static void tedMergeParagraphsInSelectionLow(	EditDocument *	ed,
						int *		fieldMap )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SelectionScope *		ss= &(td->tdSelectionScope);
    BufferDocument *		bd= td->tdDocument;
    AppDrawingData *		add= &(ed->edDrawingData);

    BufferItem *		biTo;
    BufferItem *		biFrom;

    int				oldBackY1= add->addBackRect.drY1;

    const DocumentRectangle *	drBack= &(add->addBackRect);
    DocumentRectangle		drChanged;
    
    int				endMoved= 0;

    int				scrolledX= 0;
    int				scrolledY= 0;

    unsigned int		fieldUpd= FIELDdoNOTHING;

    drChanged= td->tdSelectedRectangle;
    drChanged.drX1= drBack->drX1;

    if  ( td->tdSelection.bsBegin.bpBi == td->tdSelection.bsEnd.bpBi )
	{ LDEB(1); return; }

    biTo= td->tdSelection.bsBegin.bpBi;
    biFrom= docNextParagraph( biTo );
    if  ( ! biTo )
	{ XDEB(biTo); return;	}

    endMoved= biTo->biParaStrlen;

    for (;;)
	{
	int	partFrom= 0;

	int	particulesInserted= 0;
	int	charactersCopied= 0;


	while( partFrom < biFrom->biParaParticuleCount- 1		&&
	       tedIsIndentationParticule( biFrom,
				biFrom->biParaParticules+ partFrom )	)
	    { partFrom++; }

	if  ( partFrom < biFrom->biParaParticuleCount )
	    {
	    int		toCount= biTo->biParaParticuleCount;

	    if  ( toCount > 0						&&
		  biTo->biParaParticules[toCount-1].tpKind ==
							DOCkindTEXT	&&
		  biTo->biParaString[biTo->biParaStrlen-1] != ' '	)
		{
		if  ( docInflateTextString( biTo, 1 ) )
		    { LDEB(biTo->biParaStrlen); return; }

		biTo->biParaString[biTo->biParaStrlen++]= ' ';
		biTo->biParaString[biTo->biParaStrlen  ]= '\0';
		biTo->biParaParticules[toCount- 1].tpStrlen++;
		endMoved++;
		}

	    if  ( docCopyParticules( bd, bd, fieldMap, &fieldUpd, biTo, biFrom,
			biTo->biParaParticuleCount, partFrom,
			biFrom->biParaParticuleCount- partFrom,
			&particulesInserted, &charactersCopied, (char *)0 ) )
		{ LDEB(biFrom->biParaParticuleCount); return;	}

	    }

	if  ( biFrom == td->tdSelection.bsEnd.bpBi )
	    {
	    endMoved -= biFrom->biParaParticules[partFrom].tpStroff;
	    break;
	    }

	endMoved += charactersCopied;

	biFrom= docNextParagraph( biFrom );
	if  ( ! biFrom )
	    { XDEB(biFrom); return;	}
	}

    biTo= td->tdSelection.bsBegin.bpBi;
    biFrom= docNextParagraph( biTo );
    if  ( ! biTo )
	{ XDEB(biTo); return;	}

    for (;;)
	{
	BufferItem *	nextBi= docNextParagraph( biFrom );

	docDeleteItems( bd, biFrom->biParent, biFrom->biNumberInParent, 1 );

	if  ( biFrom == td->tdSelection.bsEnd.bpBi )
	    { break;	}

	if  ( ! nextBi )
	    { XDEB(nextBi); return;	}
	biFrom= nextBi;
	}

    if  ( tedLayoutItem( biTo, bd, ss->ssInHeaderFooter, add, &drChanged ) )
	{ LDEB(1);	}

    td->tdSelection.bsEnd.bpBi= biTo;
    td->tdSelection.bsEnd.bpStroff += endMoved;

    tedDelimitCurrentSelection( td, add );

    tedScrollToSelection( ed, &scrolledX, &scrolledY );

    tedExposeRectangle( ed, &drChanged, scrolledX, scrolledY );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    return;
    }

void tedMergeParagraphsInSelection(	EditDocument *	ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    int *			fieldMap;

    fieldMap= docAllocateFieldMap( bd );
    if  ( ! fieldMap )
	{ XDEB(fieldMap); return;	}

    tedMergeParagraphsInSelectionLow( ed, fieldMap );

    free( fieldMap );

    return;
    }

/************************************************************************/
/*									*/
/*  Replace the selection in a document with another document.		*/
/*  ( Used with 'paste', 'undo', 'redo'. )				*/
/*									*/
/*  B)  The first particule of the line was split, probably, part fits	*/
/*	on the previous line. Reformat from the previous particule.	*/
/*	If paragraphs were merged, redraw the whole neighbourhood.	*/
/*									*/
/*  1)  Replace the selection of the target with the text of those	*/
/*	particules at the beginning of the source that have the same	*/
/*	attributes.							*/
/*  2)  Insert the rest of the first paragraph of the source into the	*/
/*	target.								*/
/*  4)  If the insertion consists of more than one paragraph, split the	*/
/*	target paragraph.						*/
/*  5)  Insert the last particule of the insertion as text.		*/
/*									*/
/*  z)  Copy all paragraphs between the first and last (exclusive) of	*/
/*	the source to the target.					*/
/*									*/
/************************************************************************/

static int tedIncludeParagraphs(	const char *		filename,
					int			inHeaderFooter,
					int *			pEndedInTable,
					int			startWithTable,
					BufferItem *		biFrom,
					BufferItem *		biTo,
					BufferPosition *	bpEndFrom,
					BufferDocument *	bdTo,
					BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					AppDrawingData *	add,
					DocumentRectangle *	drChanged,
					AppColors *		ac  )
    {
    BufferItem *		rowBi= (BufferItem *)0;
    int				inTable= 0;

    if  ( ! startWithTable )
	{
	biFrom= docNextParagraph( biFrom );
	if  ( ! biFrom )
	    { XDEB(biFrom); return -1;	}

	if  ( biFrom == bpEndFrom->bpBi )
	    { return 0; }
	}

    for (;;)
	{
	BufferItem *	insBi;

	if  ( biFrom->biParaInTable )
	    {
	    BufferPosition	bp;

	    BufferItem *	biRowTo= biTo->biParent->biParent;
	    BufferItem *	biRowFrom= biFrom->biParent->biParent;

	    inTable= 1;

	    if  ( biTo->biNumberInParent !=
					biTo->biParent->biChildCount -1 )
		{
		if  ( docSplitGroupItem( bdTo, &insBi,
				biTo->biParent, biTo->biNumberInParent+ 1 ) )
		    { LDEB(1); return -1;	}

		}

	    if  ( biTo->biParent->biNumberInParent ==
					    biRowTo->biChildCount -1 )
		{
		tedLayoutItem( biTo->biParent, bdTo,
					    inHeaderFooter, add, drChanged );
		tedLayoutItem( insBi, bdTo,
					    inHeaderFooter, add, drChanged );
		}
	    else{
		if  ( docSplitGroupItem( bdTo, &insBi,
			    biRowTo, biTo->biParent->biNumberInParent+ 1 ) )
		    { LDEB(1); return -1;	}

		tedLayoutItem( biRowTo, bdTo,
					inHeaderFooter, add, drChanged );
		tedLayoutItem( insBi, bdTo,
					inHeaderFooter, add, drChanged );

		biRowTo= insBi;
		}

	    rowBi= docCopyRowItem( bdTo, bdFrom, fieldMap, pFieldUpd,
			biRowTo->biParent, biRowTo->biNumberInParent+ 1,
			biRowFrom, filename );

	    if  ( ! rowBi )
		{ XDEB(rowBi); return -1;	}

	    if  ( tedLayoutItem( rowBi, bdTo,
					inHeaderFooter, add, drChanged ) )
		{ LDEB(1); return -1;	}

	    if  ( tedOpenItemObjects( rowBi, ac, add ) )
		{ LDEB(1); return -1;	}

	    if  ( docLastPosition( biRowFrom, &bp ) )
		{ LDEB(1); break;	}

	    if  ( biFrom == bpEndFrom->bpBi )
		{ break;	}

	    biFrom= docNextParagraph( bp.bpBi );
	    if  ( ! biFrom )
		{ break;	}

	    if  ( biFrom == bpEndFrom->bpBi && ! biFrom->biParaInTable )
		{ break;	}

	    if  ( docLastPosition( rowBi, &bp ) )
		{ LDEB(1); break;	}

	    biTo= bp.bpBi;
	    }
	else{
	    if  ( rowBi )
		{
		BufferItem *	childBi;

		insBi= docInsertItem( bdTo, rowBi->biParent,
				    rowBi->biNumberInParent+ 1, DOClevROW );
		if  ( ! insBi )
		    { XDEB(insBi); return -1;	}

		childBi= docInsertItem( bdTo, insBi, 0, DOClevCELL );
		if  ( ! childBi )
		    { XDEB(childBi); return -1;	}

		childBi= docCopyParaItem( bdTo, bdFrom, fieldMap, pFieldUpd,
					    childBi, 0, biFrom, filename );

		if  ( ! childBi )
		    { XDEB(childBi); return -1;	}

		biTo= childBi;
		}
	    else{
		insBi= docCopyParaItem( bdTo, bdFrom, fieldMap, pFieldUpd,
				biTo->biParent,
				biTo->biNumberInParent+ 1, biFrom, filename );

		if  ( biTo->biParaInTable )
		    { insBi->biParaInTable= biTo->biParaInTable;	}

		if  ( ! insBi )
		    { XDEB(insBi); return -1;	}

		biTo= insBi;
		}

	    inTable= 0;

	    if  ( tedLayoutItem( insBi, bdTo,
					inHeaderFooter, add, drChanged ) )
		{ LDEB(1); return -1;	}

	    if  ( tedOpenItemObjects( insBi, ac, add ) )
		{ LDEB(1); return -1;	}

	    biFrom= docNextParagraph( biFrom );
	    if  ( ! biFrom )
		{ XDEB(biFrom); return -1;	}

	    if  ( biFrom == bpEndFrom->bpBi )
		{ break;	}

	    rowBi= (BufferItem *)0;
	    }
	}

    *pEndedInTable= inTable; return 0;
    }

/************************************************************************/
/*									*/
/*  Include the tail of a paragraph in another paragraph.		*/
/*									*/
/*  It is known by the caller that at the beginning if the series of	*/
/*  particules that is to be included, no particule merging is		*/
/*  possible/necessary.							*/
/*									*/
/*  1)  Inclusion at the end of the paragraph... No particule merging.	*/
/*  2)  If the last particule of the source, and the subsequent		*/
/*	particule of the target have the same attributes, let the	*/
/*	text replacement routine decide whether to merge them or not.	*/
/*  3)  Copy those particules that are to be copied 'as is.'		*/
/*  4)  Insert the last paricule as text if this is desirable.		*/
/*									*/
/************************************************************************/

static int tedIncludeTail(	BufferDocument *	bdTo,
				const BufferDocument *	bdFrom,
				int *			fieldMap,
				unsigned int *		pFieldUpd,
				BufferItem *            biTo,
				const BufferItem *	biFrom,
				int			partTo,
				int			partFrom,
				int *			pPartShift,
				int *			pStroffShift,
				const AppDrawingData *	add,
				const char *		refFileName )
    {
    int			copyLastAsText;
    int			particulesCopied;

    TextParticule *	tpTo;
    TextParticule *	tpFrom;

    int			particulesInserted= 0;
    int			charactersCopied= 0;

    /*  1  */
    if  ( biTo->biParaStrlen == 0		||
	  partTo == biTo->biParaParticuleCount	)
	{ copyLastAsText= 0;	}
    else{
	tpTo= biTo->biParaParticules+ partTo;
	tpFrom= biFrom->biParaParticules+ biFrom->biParaParticuleCount- 1;

	/*  2  */
	if  ( tpFrom->tpKind == DOCkindTEXT				&&
	      tpTo->tpKind == DOCkindTEXT				&&
	      docEqualTextAttributes( &(tpTo->tpTextAttribute),
					&(tpFrom->tpTextAttribute) )	)
	    { copyLastAsText= 1;	}
	else{ copyLastAsText= 0;	}
	}

    /*  3  */
    particulesCopied= biFrom->biParaParticuleCount- partFrom- copyLastAsText;
    if  ( particulesCopied > 0 )
	{
	if  ( docCopyParticules( bdTo, bdFrom, fieldMap, pFieldUpd,
				    biTo, biFrom, partTo, partFrom,
				    particulesCopied, &particulesInserted,
				    &charactersCopied, refFileName ) )
	    { LDEB(1); return -1;	}
	}
    else{
	if  ( particulesCopied < 0 )
	    { LDEB(particulesCopied); particulesCopied= 0;	}
	}

    /*  4  */
    if  ( copyLastAsText > 0 )
	{
	tpTo= biTo->biParaParticules+ partTo+ particulesCopied;

	tpFrom= biFrom->biParaParticules+
				biFrom->biParaParticuleCount- copyLastAsText;

	if  ( docParaReplaceText( bdTo, biTo, partTo+ particulesCopied,
		    tpTo->tpStroff,
		    &particulesInserted, &charactersCopied, tpTo->tpStroff,
		    biFrom->biParaString+ tpFrom->tpStroff,
		    tpFrom->tpStrlen,
		    tpFrom->tpTextAttribute,
		    (void *)add, tedCloseObject ) )
	    { LDEB(1); return -1;	}
	}

    *pPartShift += particulesInserted;
    *pStroffShift += charactersCopied;

    return 0;
    }

static int tedIncludeHead(	BufferItem *		biTo,
				const BufferItem *	biFrom,
				const int		part,
				const int		stroff,
				TedDocument *		tdTo,
				BufferDocument *	bdFrom,
				int *			fieldMap,
				unsigned int *		pFieldUpd,
				int *			pPartShift,
				int *			pStroffShift,
				int *			pLine,
				int *			pCharactersCopied,
				DocumentRectangle *	drChanged,
				const AppDrawingData *	add,
				const char *		refFileName )
    {
    const DocumentRectangle *	drBack= &(add->addBackRect);

    TextParticule *		tpTo;
    TextParticule *		tpFrom;

    int				partTo= part;
    unsigned int		stroffTo= stroff;

    int				charactersCopied= 0;
    int				particulesCopied;
    int				particuleSplit= 0;

    BufferDocument *		bdTo= tdTo->tdDocument;

    tpTo= biTo->biParaParticules+ tdTo->tdSelection.bsBegin.bpParticule;
    tpFrom= biFrom->biParaParticules;

    if  ( biTo->biParaStrlen == 0 )
	{
	int		toInTable= biTo->biParaInTable;

	if  ( docCopyParagraphRuler( &biTo->biParaProperties,
					    &biFrom->biParaProperties )	)
	    { LDEB(1); }

	biTo->biParaInTable= toInTable;
	tedParaScreenGeometry( biTo, add );
	}

    /*  B  */
    tedAdjustRedrawBegin( tdTo, drBack, drChanged, pLine, partTo );

    /*  1  */
    particulesCopied= 0;
    while( particulesCopied < biFrom->biParaParticuleCount		&&
	   tpTo->tpKind == DOCkindTEXT					&&
	   tpFrom[particulesCopied].tpKind == DOCkindTEXT		&&
	   docEqualTextAttributes( &(tpTo->tpTextAttribute),
			&(tpFrom[particulesCopied].tpTextAttribute) )	)
	{ particulesCopied++; }

    if  ( particulesCopied > 0 )
	{
	charactersCopied= tpFrom[particulesCopied-1].tpStroff+
				    tpFrom[particulesCopied-1].tpStrlen;
			    
	if  ( docReplaceSelection( bdTo, &(tdTo->tdSelection),
		    pPartShift, pStroffShift,
		    biFrom->biParaString+ tpFrom->tpStroff, charactersCopied,
		    tpFrom[particulesCopied-1].tpTextAttribute,
		    (void *)add, tedCloseObject ) )
	    { LDEB(tpFrom->tpStrlen); return -1;	}
	}
    else{
	if  ( ! tedHasIBarSelection( tdTo )				&&
	      docReplaceSelection( bdTo, &(tdTo->tdSelection),
		    pPartShift, pStroffShift,
		    (unsigned char *)0, 0, tdTo->tdCurrentTextAttribute,
		    (void *)add, tedCloseObject )		)
	    { LDEB(0); return -1;	}
	}

    stroffTo += charactersCopied;

    if  ( stroffTo >= biTo->biParaStrlen )
	{ partTo= biTo->biParaParticuleCount;	}
    else{
	partTo= 0; tpTo= biTo->biParaParticules;
	while( tpTo->tpStroff+ tpTo->tpStrlen < stroffTo )
	    { partTo++; tpTo++;	}

	if  ( tpTo->tpStroff != stroffTo				&&
	      tpTo->tpStroff+ tpTo->tpStrlen != stroffTo		)
	    { particuleSplit= 1;	}

	if  ( particulesCopied < biFrom->biParaParticuleCount	&&
	      particuleSplit					)
	    {
	    if  ( ! docCopyParticule( biTo, partTo+ 1, stroffTo,
		    tpTo->tpStroff+ tpTo->tpStrlen- stroffTo,
							DOCkindTEXT, tpTo ) )
		{ LDEB(1); return -1;	}

	    tpTo= biTo->biParaParticules+ partTo;
	    tpTo->tpStrlen= stroffTo- tpTo->tpStroff;

	    (*pPartShift)++; partTo++;
	    }
	}

    if  ( particulesCopied < biFrom->biParaParticuleCount )
	{
	int	partFrom= particulesCopied;

	if  ( tedIncludeTail( bdTo, bdFrom, fieldMap, pFieldUpd,
			    biTo, biFrom, partTo, partFrom,
			    pPartShift, pStroffShift, add, refFileName ) )
	    { LLDEB(partTo,partFrom); return -1;	}

	charactersCopied= biFrom->biParaStrlen;
	}

    *pCharactersCopied= charactersCopied;

    return 0;
    }

static int tedIncludeDocument(	EditDocument *		edTo,
				BufferDocument *	bdFrom,
				int *			fieldMap )
    {
    AppDrawingData *		add= &(edTo->edDrawingData);

    TedDocument *		tdTo= (TedDocument *)edTo->edPrivateData;
    SelectionScope *		ssTo= &(tdTo->tdSelectionScope);
    BufferDocument *		bdTo= tdTo->tdDocument;

    int				stroffShift= 0;
    int				partShift= 0;
    int				endedInTable= 0;

    BufferPosition		bpBeginFrom;
    BufferPosition		bpEndFrom;

    BufferItem *		biFrom;

    BufferItem *		biTo;

    int				line;
    int				part;
    int				stroff;
    int				stroffEnd;

    int				charactersCopied= 0;

    int				oldBackY1= add->addBackRect.drY1;

    unsigned int		fieldUpd= FIELDdoNOTHING;

    const DocumentRectangle *	drBack= &(add->addBackRect);
    DocumentRectangle		drChanged;

    if  ( docFirstPosition( &(bdFrom->bdItem), &bpBeginFrom ) )
	{ LDEB(1); return -1;	}
    if  ( docLastPosition( &(bdFrom->bdItem), &bpEndFrom ) )
	{ LDEB(1); return -1;	}

    biFrom= bpBeginFrom.bpBi;
    biTo= tdTo->tdSelection.bsBegin.bpBi;

    stroff= tdTo->tdSelection.bsBegin.bpStroff;
    part= tdTo->tdSelection.bsBegin.bpParticule;
    line= tdTo->tdSelection.bsBegin.bpLine;

    drChanged= tdTo->tdSelectedRectangle;
    drChanged.drX0= drBack->drX0;
    drChanged.drX1= drBack->drX1;

    if  ( ! biFrom->biParaInTable )
	{
	if  ( tedIncludeHead( biTo, biFrom, part, stroff,
		    tdTo, bdFrom, fieldMap, &fieldUpd,
		    &partShift, &stroffShift, &line,
		    &charactersCopied, &drChanged, add, edTo->edFilename ) )
	    { LDEB(1); return -1;	}
	}

    if  ( bpEndFrom.bpBi != biFrom )
	{
	BufferItem *	newBi;

	const int	particulesCopied= 0;
	TextParticule *	tpFrom;

	/*  4  */
	if  ( docSplitParaItem( bdTo, &newBi, biTo, stroff+ charactersCopied ) )
	    { LLDEB(part+ partShift,stroff+ charactersCopied); return -1; }

	/*  C  */
	if  ( tedLayoutItem( biTo, bdTo,
				ssTo->ssInHeaderFooter, add, &drChanged ) )
	    { LDEB(1); return -1;	}

	if  ( tedOpenItemObjects( biTo, &edTo->edColors, add ) )
	    { LDEB(1); return -1;	}

	if  ( tedIncludeParagraphs( edTo->edFilename,
				    ssTo->ssInHeaderFooter,
				    &endedInTable,
				    biFrom->biParaInTable,
				    biFrom, biTo, &bpEndFrom,
				    bdTo, bdFrom, fieldMap, &fieldUpd,
				    add, &drChanged, &(edTo->edColors) ) )
	    { LDEB(1); return -1;	}

	part= 0; stroff= 0; partShift= 0; stroffShift= 0;

	biFrom= bpEndFrom.bpBi;
	tpFrom= biFrom->biParaParticules;

	if  ( ! endedInTable && biFrom->biParaStrlen > 0 )
	    {
	    int		partTo= part;
	    int		partFrom= particulesCopied;
	    int		wasEmpty= newBi->biParaStrlen == 0;
	    int		savedInTable= newBi->biParaInTable;

	    if  ( tedIncludeTail( bdTo, bdFrom, fieldMap, &fieldUpd,
			newBi, biFrom, partTo, partFrom,
			&partShift, &stroffShift, add, edTo->edFilename ) )
		{ LLDEB(partTo,partFrom); return -1;	}

	    if  ( wasEmpty &&
		    docCopyParagraphProperties( &newBi->biParaProperties,
						&biFrom->biParaProperties ) )
		{ LDEB(wasEmpty);	}

	    newBi->biParaInTable= savedInTable;
	    }

	if  ( tedLayoutItem( newBi, bdTo,
				ssTo->ssInHeaderFooter, add, &drChanged ) )
	    { LDEB(1); return -1;	}

	if  ( tedOpenItemObjects( newBi, &edTo->edColors, add ) )
	    { LDEB(1); return -1;	}

	biTo= newBi;
	stroffEnd= 0+ stroffShift;
	}
    else{
	stroffEnd= tdTo->tdSelection.bsBegin.bpStroff+ charactersCopied;

	/*  C  */
	if  ( tedAdjustParagraphLayout( biTo, line, stroffShift, stroffEnd,
		ssTo->ssInHeaderFooter, add, bdTo, &drChanged ) )
	    { LDEB(1); }

	if  ( tedOpenItemObjects( biTo, &edTo->edColors, add ) )
	    { LDEB(1); return -1;	}
	}

    if  ( fieldUpd )
	{
	int		changed= 0;

	if  ( docRecalculateTextFields( &changed,
					bdTo, &(bdTo->bdItem), fieldUpd ) )
	    { LDEB(1);	}

	if  ( changed )
	    {
	    if  ( tedLayoutDocumentTree( tdTo, add ) )
		{ LDEB(1); return -1;	}

	    docUnionRectangle( &drChanged, &drChanged, &(add->addBackRect) );
	    }
	}

    tedEditFinishIBarSelection( edTo, biTo, &drChanged, oldBackY1, stroffEnd );

    return 0;
    }

int tedIncludePlainDocument(	APP_WIDGET		w,
				EditDocument *		ed,
				BufferDocument *	bdFrom )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferPosition		bpBeginTo;
    BufferPosition		bpBeginFrom;
    BufferItem *		bi;

    TextAttribute		ta;

    int				rval;
    int *			fieldMap;

    bpBeginTo= td->tdSelection.bsBegin;
    if  ( ! bpBeginTo.bpBi				||
	  bpBeginTo.bpBi->biParaParticuleCount == 0	)
	{ LDEB(1); return -1;	}

    ta= bpBeginTo.bpBi->biParaParticules[bpBeginTo.bpParticule].tpTextAttribute;

    if  ( docFirstPosition( &(bdFrom->bdItem), &bpBeginFrom ) )
	{ LDEB(1); return -1;	}
    bi= bpBeginFrom.bpBi;

    for (;;)
	{
	TextParticule *		tp;
	int			part;

	tp= bi->biParaParticules;

	for ( part= 0; part < bi->biParaParticuleCount; part++, tp++ )
	    { tp->tpTextAttribute= ta; }

	bi= docNextParagraph( bi );
	if  ( ! bi )
	    { break;	}
	}

    fieldMap= docAllocateFieldMap( bdFrom );
    if  ( ! fieldMap )
	{ XDEB(fieldMap); return -1;	}

    rval= tedIncludeDocument( ed, bdFrom, fieldMap );

    free( fieldMap );

    return rval;
    }

int tedIncludeRtfDocument(	APP_WIDGET		w,
				EditDocument *		ed,
				BufferDocument *	bdFrom )
    {
    TedDocument *		tdTo= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bdTo= tdTo->tdDocument;
    DocumentProperties *	dpTo= &(bdTo->bdProperties);

    int				rval;
    int *			fieldMap;

    if  ( bdFrom->bdProperties.dpContainsTables )
	{
	if  ( tdTo->tdSelection.bsCol0 >= 0 )
	    { return 0;	}
	if  ( tdTo->tdSelection.bsBegin.bpBi->biParaInTable )
	    { return 0;	}
	if  ( tdTo->tdSelection.bsEnd.bpBi->biParaInTable )
	    { return 0;	}
	}

    if  ( docMergeDocumentFontsIntoList( &(dpTo->dpFontList), bdFrom ) )
	{ LDEB(1); return -1;	}

    fieldMap= docAllocateFieldMap( bdFrom );
    if  ( ! fieldMap )
	{ XDEB(fieldMap); return -1;	}

    rval= tedIncludeDocument( ed, bdFrom, fieldMap );

    free( fieldMap );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Split a paragraph.							*/
/*									*/
/*  1)  Split in the buffer administration.				*/
/*  2)  Recalculate the geometry from the start of the selection to the	*/
/*	end of the paragraph.						*/
/*  3)  Redraw the affected part of the paragraph.			*/
/*  4)  Layout the new paragraph.					*/
/*									*/
/************************************************************************/

int tedSplitParaContents(	BufferItem **		pNewBi,
				BufferItem *		bi,
				DocumentRectangle *	drChanged,
				EditDocument *		ed,
				int			splitLevel,
				int			onNewPage )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SelectionScope *		ss= &(td->tdSelectionScope);
    BufferDocument *		bd= td->tdDocument;

    BufferItem *		newBi;

    AppDrawingData *		add= &(ed->edDrawingData);

    int				n;
    int				level;

    /*  1  */
    if  ( docSplitParaItem( bd, &newBi, bi, td->tdSelection.bsBegin.bpStroff ) )
	{ LDEB(td->tdSelection.bsBegin.bpStroff); return -1;	}

    newBi->biParaStartsOnNewPage= ! newBi->biParaInTable && onNewPage;
    
    tedParaScreenGeometry( newBi, add );

    /*  3  */
    *drChanged= td->tdSelectedRectangle;
    drChanged->drX0= add->addBackRect.drX0;
    drChanged->drX1= add->addBackRect.drX1; 

    if  ( bi->biParaBottomBorder.bpIsSet )
	{ bi->biParaBottomBorder.bpIsSet= 0;	}
    if  ( bi->biParaSpaceAfterTwips > 0 )
	{ bi->biParaSpaceAfterTwips= 0;		}

    if  ( newBi->biParaTopBorder.bpIsSet )
	{ newBi->biParaTopBorder.bpIsSet= 0;	}
    if  ( newBi->biParaSpaceBeforeTwips > 0 )
	{ newBi->biParaSpaceBeforeTwips= 0;	}

    /*  2  */
    if  ( tedLayoutItem( bi, bd, ss->ssInHeaderFooter, add, drChanged ) < 0 )
	{ LDEB(1); return -1;	}

    /*  4  */
    if  ( tedLayoutItem( newBi, bd,
				ss->ssInHeaderFooter, add, drChanged ) < 0 )
	{ LDEB(1); return -1;	}

    if  ( tedOpenItemObjects( newBi, &ed->edColors, add ) )
	{ LDEB(1); return -1;	}

    level= DOClevPARA;
    n= newBi->biNumberInParent;
    while( level > splitLevel )
	{
	BufferItem *	splitBi;

	if  ( docSplitGroupItem( bd, &splitBi, bi->biParent, n ) )
	    { LDEB(1); return -1;	}

	tedLayoutItem( bi->biParent, bd,
				    ss->ssInHeaderFooter, add, drChanged );
	tedLayoutItem( splitBi, bd, ss->ssInHeaderFooter, add, drChanged );

	n= splitBi->biNumberInParent+ 1;
	bi= splitBi; level--;
	}

    *pNewBi= newBi; return 0;
    }

void tedSplitParagraph(		EditDocument *		ed,
				int			onNewPage )
    {
    AppDrawingData *	add= &(ed->edDrawingData);
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferItem *	bi= td->tdSelection.bsBegin.bpBi;
    BufferItem *	newBi;

    int			oldBackY1= add->addBackRect.drY1;

    DocumentRectangle	drChanged;

    if  ( tedSplitParaContents( &newBi, bi, &drChanged,
						ed, DOClevPARA, onNewPage ) )
	{ LDEB(1); return;	}

    /*  7,8  */
    tedEditFinishIBarSelection( ed, newBi, &drChanged, oldBackY1, 0 );

    return;
    }

/************************************************************************/
/*									*/
/*  Insert a 'Tab'.							*/
/*									*/
/************************************************************************/

void tedSetTab(		EditDocument *		ed )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SelectionScope *		ss= &(td->tdSelectionScope);
    BufferDocument *		bd= td->tdDocument;
    TextParticule *		tp;

    int				stroffShift;
    int				partShift;

    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    int				oldX0;

    int				line;
    int				part;

    int				oldBackY1= add->addBackRect.drY1;

    DocumentRectangle		drChanged;
    const DocumentRectangle *	drBack= &(add->addBackRect);

    part= td->tdSelection.bsBegin.bpParticule;
    line= td->tdSelection.bsBegin.bpLine;

    /*  1  */
    oldX0= bi->biParaParticules[part].tpX0;

    tp= docParaSpecialParticule( bi, DOCkindTAB,
				    part, td->tdSelection.bsEnd.bpStroff,
				    &partShift, &stroffShift );
    if  ( ! tp )
	{ LXDEB(part,tp); return;	}

    drChanged= td->tdSelectedRectangle;
    drChanged.drX1= drBack->drX1;

    /*  3,4,5  */
    if  ( tedAdjustParagraphLayout( bi, line, stroffShift,
				td->tdSelection.bsBegin.bpStroff+ stroffShift,
					    ss->ssInHeaderFooter,
					    add, bd, &drChanged ) )
	{ LDEB(line); }

    /*  6,7  */
    tedEditFinishIBarSelection( ed, bi, &drChanged, oldBackY1,
			    td->tdSelection.bsBegin.bpStroff+ stroffShift );

    return;
    }

/************************************************************************/
/*									*/
/*  Insert an 'Object'.							*/
/*									*/
/************************************************************************/

int tedInsertNewObject(		APP_WIDGET		w,
				InsertedObject *	io,
				EditDocument *		ed )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SelectionScope *		ss= &(td->tdSelectionScope);
    BufferDocument *		bd= td->tdDocument;
    TextParticule *		tp;

    int				stroffShift;
    int				partShift;

    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    int				oldX0;

    int				line;
    int				part;

    int				oldBackY1= add->addBackRect.drY1;

    DocumentRectangle		drChanged;
    const DocumentRectangle *	drBack= &(add->addBackRect);

    part= td->tdSelection.bsBegin.bpParticule;
    line= td->tdSelection.bsBegin.bpLine;

    /*  1  */
    oldX0= bi->biParaParticules[part].tpX0;

    tp= docParaSpecialParticule( bi, DOCkindOBJECT,
				    part, td->tdSelection.bsEnd.bpStroff,
				    &partShift, &stroffShift );
    if  ( ! tp )
	{ LXDEB(part,tp); return -1;	}

    tp->tpObjectNumber= bi->biParaObjectCount++;

    drChanged= td->tdSelectedRectangle;
    drChanged.drX1= drBack->drX1;

    /*  3,4,5  */
    if  ( tedAdjustParagraphLayout( bi, line, stroffShift,
			    td->tdSelection.bsBegin.bpStroff+ stroffShift+ 1,
			    ss->ssInHeaderFooter, add, bd, &drChanged ) )
	{ LDEB(line); }

    /*  6,7  */
    tedEditFinishIBarSelection( ed, bi, &drChanged, oldBackY1,
			    td->tdSelection.bsBegin.bpStroff+ stroffShift );

    tp->tpPhysicalFont= 0;

    return 0;
    }

/************************************************************************/
/*									*/
/*  An object has been resized: Redo layout.				*/
/*									*/
/************************************************************************/

int tedResizeObject(		EditDocument *		ed,
				const BufferPosition *	bp )
    {
    BufferItem *		bi= bp->bpBi;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SelectionScope *		ss= &(td->tdSelectionScope);
    BufferDocument *		bd= td->tdDocument;
    DocumentProperties *	dp= &(bd->bdProperties);
    AppDrawingData *		add= &(ed->edDrawingData);
    const DocumentRectangle *	drBack= &(add->addBackRect);
    TextParticule *		tp= bi->biParaParticules+ bp->bpParticule;
    int				line= bp->bpLine;

    int				stroffShift= 0;

    int				oldBackY1= add->addBackRect.drY1;
    DocumentRectangle		drChanged;

    int				scrolledX= 0;
    int				scrolledY= 0;

    drChanged= td->tdSelectedRectangle;
    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;

    tedAdjustRedrawBegin( td, drBack, &drChanged, &line, bp->bpParticule );

    if  ( tedReopenObject( bd, bi, tp, &ed->edColors,
					    &(dp->dpFontList), add ) )
	{ LDEB(1); return -1;	}

    if  ( tedAdjustParagraphLayout( bi, line, stroffShift,
				td->tdSelection.bsEnd.bpStroff+ stroffShift,
				ss->ssInHeaderFooter, add, bd, &drChanged ) )
	{ LDEB(line); }

    tedDelimitCurrentSelection( td, add );

    tedScrollToSelection( ed, &scrolledX, &scrolledY );

    tedExposeRectangle( ed, &drChanged, scrolledX, scrolledY );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Replace the selection with something new.				*/
/*									*/
/*  1)  No selection... Just refuse.					*/
/*  2)  Replace nothing with nothing: Do nothing.			*/
/*									*/
/************************************************************************/

void tedAppReplaceSelection(	EditDocument *		ed,
				const unsigned char *	word,
				int			len	)
    {
    AppDrawingData *	add= &(ed->edDrawingData);
    int			oldBackY1= add->addBackRect.drY1;

    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferSelection	bsRep;

    DocumentRectangle	drChanged;

    drChanged= td->tdSelectedRectangle;
    drChanged.drX1= add->addBackRect.drX1;

    /*  1  */
    if  ( ! td->tdSelection.bsBegin.bpBi )
	{ XDEB(td->tdSelection.bsBegin.bpBi); return; }

    /*  2  */
    if  ( tedHasIBarSelection( td ) && len == 0	)
	{ return;	}

    if  ( tedReplaceSelection( &drChanged, &bsRep, ed, word, len ) )
	{ return;	}

    /*  6,7  */
    tedEditFinishIBarSelection( ed, bsRep.bsEnd.bpBi, &drChanged, oldBackY1,
							bsRep.bsEnd.bpStroff );

    appDocumentChanged( ed, 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Delete the current paragraph from a document.			*/
/*									*/
/************************************************************************/

static int tedEmptyParagraph(	BufferItem *		paraBi,
				BufferDocument *	bd,
				AppDrawingData *	add )
    {
    int			part;
    TextParticule *	tp= paraBi->biParaParticules;
    TextAttribute	ta= tp->tpTextAttribute;

    paraBi->biParaStrlen= 0;

    for ( part= 0; part < paraBi->biParaParticuleCount; tp++, part++ )
	{
	tedCloseObject( bd, paraBi, tp, (void *)add );
	docCleanParticuleObject( paraBi, tp );
	}

    docDeleteParticules( paraBi, 0, paraBi->biParaParticuleCount );
    docDeleteLines( paraBi, 0, paraBi->biParaLineCount );

    if  ( ! docInsertTextParticule( paraBi, 0, 0, 0, DOCkindTEXT, ta ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

int tedDeleteCurrentParagraph(	EditApplication *	ea )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SelectionScope *		ss= &(td->tdSelectionScope);
    BufferDocument *		bd= td->tdDocument;
    BufferItem *		paraBi= td->tdSelection.bsBegin.bpBi;
    AppDrawingData *		add= &(ed->edDrawingData);

    BufferPosition		bpNew;
    BufferItem *		parentBi;

    int				oldBackY1= add->addBackRect.drY1;
    DocumentRectangle		drChanged;

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;
    drChanged.drY0= paraBi->biTopPosition.lpYPixels- 1;
    drChanged.drY1= paraBi->biBelowPosition.lpYPixels- 1;

    docInitPosition( &bpNew );
    if  ( docLastPosition( paraBi, &bpNew ) 	||
	  docNextPosition( &bpNew )		)
	{
	const int	lastOne= 0;

	docInitPosition( &bpNew );
	if  ( docFirstPosition( paraBi, &bpNew )	||
	      docPrevPosition( &bpNew, lastOne )	)
	    { docInitPosition( &bpNew ); }
	}

    parentBi= paraBi->biParent;
    if  ( parentBi->biChildCount < 2 )
	{
	if  ( tedEmptyParagraph( paraBi, bd, add ) )
	    { LDEB(1); return -1;	}

	if  ( docFirstPosition( paraBi, &bpNew ) )
	    { LDEB(1); return -1;	}

	tedLayoutItem( paraBi, bd, ss->ssInHeaderFooter, add, &drChanged );

	bpNew.bpBi= paraBi; bpNew.bpStroff= 0;
	}
    else{
	tedDeleteItems( bd, parentBi, paraBi->biNumberInParent, 1, add );

	tedLayoutItem( parentBi, bd, ss->ssInHeaderFooter, add, &drChanged );
	}

    appDocumentChanged( ed, 1 );

    tedEditFinishIBarSelection( ed, bpNew.bpBi, &drChanged, oldBackY1,
							    bpNew.bpStroff );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Make sure that a Section contains an empty paragraph.		*/
/*									*/
/************************************************************************/

static int tedSectionParagraph(		BufferDocument *		bd,
					BufferItem **			pParaBi,
					BufferItem *			sectBi,
					const ParagraphProperties *	pp,
					TextAttribute			ta )
    {
    BufferItem *	bi;

    bi= docInsertEmptyParagraph( bd, sectBi, ta );
    if  ( ! bi )
	{ XDEB(bi); return -1;	}

    if  ( docCopyParagraphProperties( &(bi->biParaProperties), pp ) )
	{ LDEB(1);	}

    *pParaBi= bi; return 0;
    }

/************************************************************************/
/*									*/
/*  Delete the current section.						*/
/*									*/
/************************************************************************/

int tedDeleteCurrentSection(	EditApplication *	ea )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SelectionScope *		ss= &(td->tdSelectionScope);
    BufferDocument *		bd= td->tdDocument;
    BufferItem *		paraBi= td->tdSelection.bsBegin.bpBi;
    BufferItem *		sectBi;
    AppDrawingData *		add= &(ed->edDrawingData);

    BufferPosition		bpNew;
    BufferItem *		parentBi;

    int				oldBackY1= add->addBackRect.drY1;
    DocumentRectangle		drChanged;

    sectBi= paraBi->biParent->biParent->biParent;

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;
    drChanged.drY0= sectBi->biTopPosition.lpYPixels- 1;
    drChanged.drY1= sectBi->biBelowPosition.lpYPixels- 1;

    docInitPosition( &bpNew );
    if  ( docLastPosition( sectBi, &bpNew ) 	||
	  docNextPosition( &bpNew )		)
	{
	const int	lastOne= 0;

	docInitPosition( &bpNew );
	if  ( docFirstPosition( sectBi, &bpNew )	||
	      docPrevPosition( &bpNew, lastOne )	)
	    { docInitPosition( &bpNew ); }
	}

    parentBi= sectBi->biParent;
    if  ( parentBi->biChildCount < 2 )
	{
	ParagraphProperties	pp;
	TextParticule *		tp= paraBi->biParaParticules;
	TextAttribute		ta= tp->tpTextAttribute;

	docInitParagraphProperties( &pp );

	if  ( docCopyParagraphProperties( &pp, &(paraBi->biParaProperties) ) )
	    { LDEB(1); return -1;	}

	tedDeleteItems( bd, sectBi, 0, sectBi->biChildCount, add );

	if  ( tedSectionParagraph( bd, &paraBi, sectBi, &pp, ta ) )
	    { LDEB(1); return -1;	}

	docCleanParagraphProperties( &pp );

	bpNew.bpBi= paraBi; bpNew.bpStroff= 0;
	}
    else{
	tedDeleteItems( bd, parentBi, paraBi->biNumberInParent, 1, add );
	}

    tedLayoutItem( parentBi, bd, ss->ssInHeaderFooter, add, &drChanged );

    appDocumentChanged( ed, 1 );

    tedEditFinishIBarSelection( ed, bpNew.bpBi, &drChanged, oldBackY1,
							    bpNew.bpStroff );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert a paragraph Before/After the selection, depending on the	*/
/*  value of the 'after' argument.					*/
/*									*/
/************************************************************************/

int tedInsertParagraph(	EditApplication *	ea,
			int			after )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SelectionScope *		ss= &(td->tdSelectionScope);
    BufferDocument *		bd= td->tdDocument;
    AppDrawingData *		add= &(ed->edDrawingData);

    BufferItem *		paraBi;
    BufferItem *		newBi;

    TextAttribute		ta;
    BufferItem *		parentBi;
    int				pos;

    int				oldBackY1= add->addBackRect.drY1;
    DocumentRectangle		drChanged;

    if  ( after )
	{
	paraBi= td->tdSelection.bsEnd.bpBi;
	if  ( ! paraBi )
	    { XDEB(paraBi); return -1;	}

	pos= paraBi->biNumberInParent+ 1;
	ta= paraBi->biParaParticules[
			    paraBi->biParaParticuleCount-1].tpTextAttribute;
	}
    else{
	paraBi= td->tdSelection.bsBegin.bpBi;
	if  ( ! paraBi )
	    { XDEB(paraBi); return -1;	}

	pos= paraBi->biNumberInParent;
	ta= paraBi->biParaParticules[0].tpTextAttribute;
	}

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;
    drChanged.drY0= paraBi->biTopPosition.lpYPixels- 1;
    drChanged.drY1= paraBi->biBelowPosition.lpYPixels- 1;

    parentBi= paraBi->biParent;

    newBi= docInsertItem( bd, parentBi, pos, DOClevPARA );
    if  ( ! newBi )
	{ XDEB(newBi); return -1;	}

    if  ( docCopyParagraphProperties( &(newBi->biParaProperties),
						&(paraBi->biParaProperties) ) )
	{ LDEB(1);	}

    if  ( ! docInsertTextParticule( newBi, 0, 0, 0, DOCkindTEXT, ta ) )
	{ LDEB(1); return -1;	}

    tedLayoutItem( paraBi, bd, ss->ssInHeaderFooter, add, &drChanged );

    tedLayoutItem( newBi, bd, ss->ssInHeaderFooter, add, &drChanged );

    appDocumentChanged( ed, 1 );

    tedEditFinishIBarSelection( ed, newBi, &drChanged, oldBackY1, 0 );

    return 0;
    }

int tedInsertSection(	EditApplication *	ea,
			int			after )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SelectionScope *		ss= &(td->tdSelectionScope);
    BufferDocument *		bd= td->tdDocument;
    AppDrawingData *		add= &(ed->edDrawingData);

    BufferItem *		paraBi;
    BufferItem *		sectBi;
    BufferItem *		newBi;
    BufferItem *		bi;

    TextAttribute		ta;
    BufferItem *		parentBi;
    int				pos;

    int				oldBackY1= add->addBackRect.drY1;
    DocumentRectangle		drChanged;

    if  ( after )
	{
	paraBi= td->tdSelection.bsEnd.bpBi;
	if  ( ! paraBi )
	    { XDEB(paraBi); return -1;	}
	sectBi= paraBi->biParent->biParent->biParent;

	pos= sectBi->biNumberInParent+ 1;
	ta= paraBi->biParaParticules[
			    paraBi->biParaParticuleCount-1].tpTextAttribute;
	}
    else{
	paraBi= td->tdSelection.bsBegin.bpBi;
	if  ( ! paraBi )
	    { XDEB(paraBi); return -1;	}
	sectBi= paraBi->biParent->biParent->biParent;

	pos= sectBi->biNumberInParent;
	ta= paraBi->biParaParticules[0].tpTextAttribute;
	}

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;
    drChanged.drY0= sectBi->biTopPosition.lpYPixels- 1;
    drChanged.drY1= sectBi->biBelowPosition.lpYPixels- 1;

    parentBi= sectBi->biParent;

    newBi= docInsertItem( bd, parentBi, pos, DOClevSECT );
    if  ( ! newBi )
	{ XDEB(newBi); return -1;	}

    if  ( docCopySectionProperties( &(newBi->biSectProperties),
						&(sectBi->biSectProperties) ) )
	{ LDEB(1);	}

    if  ( tedSectionParagraph( bd, &bi, newBi,
				    &(paraBi->biParaProperties), ta ) )
	{ LDEB(1); return -1;	}

    tedLayoutItem( sectBi, bd, ss->ssInHeaderFooter, add, &drChanged );

    tedLayoutItem( newBi, bd, ss->ssInHeaderFooter, add, &drChanged );

    appDocumentChanged( ed, 1 );

    tedEditFinishIBarSelection( ed, bi, &drChanged, oldBackY1, 0 );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert a (line/page) break particule in the document.		*/
/*									*/
/*  TABs could have been inserted with this routine, but against	*/
/*  flashing, a more subtle approach is ised for TABs.			*/
/*									*/
/************************************************************************/

static void tedEditInsertBreakParticule(	EditDocument *	ed,
						int		kind )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    SelectionScope *		ss= &(td->tdSelectionScope);
    BufferDocument *		bd= td->tdDocument;
    TextParticule *		tp;

    int				stroffShift;
    int				partShift;

    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    int				oldX0;

    int				line;
    int				part;

    int				oldBackY1= add->addBackRect.drY1;

    DocumentRectangle		drChanged;
    const DocumentRectangle *	drBack= &(add->addBackRect);

    part= td->tdSelection.bsBegin.bpParticule;
    line= td->tdSelection.bsBegin.bpLine;

    /*  1  */
    oldX0= bi->biParaParticules[part].tpX0;

    tp= docParaSpecialParticule( bi, kind,
				    part, td->tdSelection.bsEnd.bpStroff,
				    &partShift, &stroffShift );
    if  ( ! tp )
	{ LXDEB(part,tp); return;	}

    drChanged= td->tdSelectedRectangle;
    drChanged.drX0= drBack->drX0;
    drChanged.drX1= drBack->drX1;

    appDocumentChanged( ed, 1 );

    /*  3,4,5  */
    if  ( tedLayoutItem( bi, bd, ss->ssInHeaderFooter, add, &drChanged ) )
	{ LDEB(1); }

    /*  6,7  */
    tedEditFinishIBarSelection( ed, bi, &drChanged, oldBackY1,
			    td->tdSelection.bsBegin.bpStroff+ stroffShift );

    return;
    }

/************************************************************************/
/*									*/
/*  Insert a line/page break in the document.				*/
/*									*/
/************************************************************************/

void tedDocInsertLineBreak(	APP_WIDGET	option,
				void *		voided,
				void *		voidpbcs )
    {
    tedEditInsertBreakParticule( (EditDocument *)voided, DOCkindLINEBREAK );
    return;
    }

void tedDocInsertPageBreak(	APP_WIDGET	option,
				void *		voided,
				void *		voidpbcs )
    {
    tedEditInsertBreakParticule( (EditDocument *)voided, DOCkindPAGEBREAK );
    return;
    }

/************************************************************************/
/*									*/
/*  Insert a section break in the document.				*/
/*									*/
/************************************************************************/

void tedDocInsertSectBreak(	APP_WIDGET	option,
				void *		voided,
				void *		voidpbcs )
    {
    EditDocument *	ed= (EditDocument *)voided;
    const int		onNewPage= 0;

    tedSplitParagraph( ed, onNewPage );
    return;
    }

