/************************************************************************/
/*									*/
/*  Ted, Screen drawing and forcing drawing through			*/
/*  appExposeRectangle().						*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>

#   include	"tedApp.h"
#   include	"docDraw.h"
#   include	<X11/cursorfont.h>

#   include	<appDebugon.h>

# define DRDEB(dr) APP_DEB(appDebug( "%s(%3d) %s=[%4d..%4d,%4d..%4d]\n", \
			     __FILE__, __LINE__, #dr, \
			     (dr)->drX0, (dr)->drX1, (dr)->drY0, (dr)->drY1 ))

# define LOG_REDRAWS	0

typedef struct ScreenDrawingData
    {
    int			sddForegroundSet;
#			define	SDDpixelUNDEF	0
#			define	SDDpixelFORE	1
#			define	SDDpixelLINK	2
#			define	SDDpixelBACK	3
#			define	SDDpixelGRID	4
#			define	SDDpixelPAGE	5

    APP_COLOR_RGB	sddForeColor;
    APP_COLOR_RGB	sddLinkColor;
    APP_COLOR_RGB	sddBackColor;
    APP_COLOR_RGB	sddGridColor;
    APP_COLOR_RGB	sddPageColor;

    int			sddOx;
    int			sddOy;
    } ScreenDrawingData;

/************************************************************************/
/*									*/
/*  Optimise GC changes.						*/
/*									*/
/************************************************************************/

static void tedDrawSetForegroundFun(	AppDrawingData *	add,
					ScreenDrawingData *	sdd,
					int			which )
    {
    APP_COLOR_RGB *	pix;

    if  ( sdd->sddForegroundSet == which )
	{ return;	}

    switch( which )
	{
	case SDDpixelFORE:	pix= &(sdd->sddForeColor);	break;
	case SDDpixelLINK:	pix= &(sdd->sddLinkColor);	break;
	case SDDpixelBACK:	pix= &(sdd->sddBackColor);	break;
	case SDDpixelGRID:	pix= &(sdd->sddGridColor);	break;
	case SDDpixelPAGE:	pix= &(sdd->sddPageColor);	break;

	case SDDpixelUNDEF:
	    sdd->sddForegroundSet= which;
	    return;
	default:
	    LDEB(which); return;
	}

    appDrawSetForegroundColor( add, pix );

    sdd->sddForegroundSet= which;
    }

#   define tedDrawSetForeground( add, sdd, which )		\
    if  ( (sdd)->sddForegroundSet != (which) )			\
	{ tedDrawSetForegroundFun( (add), (sdd), (which) );	}

/************************************************************************/
/*									*/
/*  Various Border drawing routines.					*/
/*									*/
/************************************************************************/

static void tedDrawHorizontalBorder(	const BorderProperties *	bp,
					int				asGrid,
					const DrawingContext *		dc,
					ScreenDrawingData *		sdd,
					int				above,
					int				x0,
					int				x1,
					int				y )
    {
    AppDrawingData *	add= dc->dcDrawingData;
    int			wide;
    int			thick= tedBorderThick( &wide, bp, add );

    if  ( asGrid && thick == 0 )
	{ thick= 1;	}

    if  ( thick > 0 )
	{
	const DocumentRectangle *	drClip= dc->dcClipRect;
	DocumentRectangle		drBorder;

	if  ( above )
	    { y -= thick;	}

	drBorder.drX0= x0;
	drBorder.drX1= x1;
	drBorder.drY0= y;
	drBorder.drY1= y+ thick- 1;

	if  ( ! drClip							  ||
	      docIntersectRectangle( &drBorder, &drBorder, dc->dcClipRect ) )
	    {
	    appDrawFillRectangle( add,
			drBorder.drX0- sdd->sddOx,
			drBorder.drY0- sdd->sddOy,
			drBorder.drX1- drBorder.drX0+ 1,
			drBorder.drY1- drBorder.drY0+ 1 );
	    }
	}

    return;
    }

static void tedDrawVerticalBorder(	const BorderProperties *	bp,
					int				asGrid,
					const DrawingContext *		dc,
					ScreenDrawingData *		sdd,
					int				x,
					int				y0,
					int				y1 )
    {
    AppDrawingData *	add= dc->dcDrawingData;
    int			wide;
    int			thick= tedBorderThick( &wide, bp, add );

    if  ( asGrid && thick == 0 )
	{ thick= 1;	}

    if  ( thick > 0 )
	{
	const DocumentRectangle *	drClip= dc->dcClipRect;
	DocumentRectangle		drBorder;

	drBorder.drX0= x;
	drBorder.drX1= x+ thick- 1;
	drBorder.drY0= y0- 1;
	drBorder.drY1= y1;

	if  ( ! drClip							  ||
	      docIntersectRectangle( &drBorder, &drBorder, dc->dcClipRect ) )
	    {
	    appDrawFillRectangle( add,
			drBorder.drX0- sdd->sddOx,
			drBorder.drY0- sdd->sddOy,
			drBorder.drX1- drBorder.drX0+ 1,
			drBorder.drY1- drBorder.drY0+ 1 );
	    }
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Draw a text item.							*/
/*									*/
/*  1)  Tabs need not to be drawn.					*/
/*									*/
/************************************************************************/

static int tedDrawTab(	const BufferItem *	bi,
			const TextParticule *	tp,
			int			baseLine,
			ScreenDrawingData *	sdd,
			DrawingContext *	dc )
    {
    AppDrawingData *	add= dc->dcDrawingData;
    TabStop *		ts= bi->biParaTabStops+ tp->tpObjectNumber;

    int			x0;
    int			x1;

    AppPhysicalFont *	apf;

    switch( ts->tsLeader )
	{
	case DOCtlNONE:
	    break;

	case DOCtlDOTS:
	    if  ( tp->tpPhysicalFont < 0 )
		{ LDEB(tp->tpPhysicalFont); return -1;	}
	    apf= add->addPhysicalFontList.apflFonts+ tp->tpPhysicalFont;

	    x0= tp->tpX0+ apf->apfFullSizePixels/ 4;
	    x1= tp->tpX0+ tp->tpPixelsWide- apf->apfFullSizePixels/ 2;

	    x0= 3* ( ( x0+ 2 )/ 3 );

	    if  ( x1 <= x0 )
		{ return 0;	}

	    if  ( tp->tpTextAttribute.taFontIsBold )
		{
		static char	boldDot[]= { 2, 1 };

#		ifdef USE_MOTIF
		XSetLineAttributes( add->addDisplay, add->addGc,
				    2, LineOnOffDash, CapButt, JoinMiter );

		XSetDashes( add->addDisplay, add->addGc,
					    0, boldDot, sizeof( boldDot ) );
#		endif

#		ifdef USE_GTK
		gdk_gc_set_line_attributes( add->addGc,
			2, GDK_LINE_ON_OFF_DASH, GDK_CAP_BUTT, GDK_JOIN_MITER );

		gdk_gc_set_dashes( add->addGc, 0, boldDot, sizeof( boldDot ) );
#		endif

		}
	    else{
		static char	dot[]= { 1, 2 };

#		ifdef USE_MOTIF
		XSetLineAttributes( add->addDisplay, add->addGc,
				    1, LineOnOffDash, CapButt, JoinMiter );

		XSetDashes( add->addDisplay, add->addGc,
						0, dot, sizeof( dot ) );
#		endif

#		ifdef USE_GTK
		gdk_gc_set_line_attributes( add->addGc,
			1, GDK_LINE_ON_OFF_DASH, GDK_CAP_BUTT, GDK_JOIN_MITER );

		gdk_gc_set_dashes( add->addGc, 0, dot, sizeof( dot ) );
#		endif
		}

	    appDrawDrawLine( add,
				    x0- sdd->sddOx, baseLine- sdd->sddOy,
				    x1- sdd->sddOx, baseLine- sdd->sddOy );

	    break;

	case DOCtlHYPH:
	    LDEB(ts->tsLeader);
	    break;

	case DOCtlUNDERLINE:
	    if  ( tp->tpPhysicalFont < 0 )
		{ LDEB(tp->tpPhysicalFont); return -1;	}
	    apf= add->addPhysicalFontList.apflFonts+ tp->tpPhysicalFont;

	    x0= tp->tpX0+ apf->apfFullSizePixels/ 4;
	    x1= tp->tpX0+ tp->tpPixelsWide- apf->apfFullSizePixels/ 2;

	    if  ( x1 <= x0 )
		{ return 0;	}

	    if  ( tp->tpTextAttribute.taFontIsBold )
		{
#		ifdef USE_MOTIF
		XSetLineAttributes( add->addDisplay, add->addGc,
				    2, LineSolid, CapButt, JoinMiter );
#		endif

#		ifdef USE_GTK
		gdk_gc_set_line_attributes( add->addGc,
			2, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER );
#		endif
		}
	    else{
#		ifdef USE_MOTIF
		XSetLineAttributes( add->addDisplay, add->addGc,
					1, LineSolid, CapButt, JoinMiter );
#		endif

#		ifdef USE_GTK
		gdk_gc_set_line_attributes( add->addGc,
			1, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER );
#		endif
		}

	    appDrawDrawLine( add,
				    x0- sdd->sddOx, baseLine- sdd->sddOy,
				    x1- sdd->sddOx, baseLine- sdd->sddOy );

	    break;

	case DOCtlTHICK:
	    LDEB(ts->tsLeader);
	    break;
	case DOCtlEQUAL:
	    LDEB(ts->tsLeader);
	    break;
	default:
	    LDEB(ts->tsLeader);
	    break;
	}

    return 0;
    }

static int tedDrawParticules(	const BufferItem *	bi,
				const TextParticule *	tp,
				int			count,
				int			baseLine,
				ScreenDrawingData *	sdd,
				DrawingContext *	dc )
    {
    AppDrawingData *	add= dc->dcDrawingData;
    unsigned char *	paraString= bi->biParaString;

    int			drawn;
    int			i;
    int			len;
    int			y;

    AppPhysicalFont *	apf;

    /*  1  */
    switch( tp->tpKind )
	{ 
	case DOCkindTAB:
	    if  ( tp->tpObjectNumber >= 0			&&
		  tp->tpObjectNumber < bi->biParaTabCount	)
		{
		if  ( tedDrawTab( bi, tp, baseLine, sdd, dc ) )
		    { LDEB(1);	}
		}

	    return drawn= 1;

	case DOCkindTEXT:
	    break;

	case DOCkindFIELDSTART:
	case DOCkindFIELDEND:
	case DOCkindXE:
	case DOCkindTC:
	case DOCkindLINEBREAK:
	case DOCkindPAGEBREAK:
	    return drawn= 1;

	case DOCkindOBJECT:
	    if  ( tedDrawObject( bi, tp, baseLine,
					    sdd->sddOx, sdd->sddOy, add ) )
		{ LDEB(1); return -1;	}

	    return drawn= 1;

	default:
	    LDEB(tp->tpKind); return -1;
	}

    drawn= 1;
    len= tp[drawn-1].tpStroff+ tp[drawn-1].tpStrlen- tp->tpStroff;

    if  ( tp->tpPhysicalFont < 0 )
	{ LDEB(tp->tpPhysicalFont); return -1;	}

    apf= add->addPhysicalFontList.apflFonts+ tp->tpPhysicalFont;
    y= baseLine;

    if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUPERSCRIPT )
	{ y -= ( 3* apf->apfFullSizePixels )/ 10; }

    if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUBSCRIPT )
	{ y += ( 3* apf->apfFullSizePixels )/ 10; }

    while( len > 0 && paraString[tp->tpStroff+ len- 1] == ' ' )
	{ len--;	}

    if  ( len > 0 )
	{
	if  ( tp->tpPhysicalFont != dc->dcCurrentPhysicalFont )
	    {
	    appDrawSetFont( add, apf->apfFontStruct );

	    dc->dcCurrentPhysicalFont= tp->tpPhysicalFont;
	    }

	appDrawDrawString( add,
				tp->tpX0- sdd->sddOx, y- sdd->sddOy,
				(char *)paraString+ tp->tpStroff, len );

	/*
	appDebug( "[%3d,%3d] -- %.*s\n",
		y, tp->tpX0,
		len, (char *)paraString+ tp->tpStroff );
	*/
	}

    i= 0;
    while( i < drawn )
	{
	int	x0;
	int	x1;
	int	y0;
	int	h;

	if  ( ! tp[i].tpTextAttribute.taIsUnderlined )
	    { i++; continue;	}

	x1= x0= tp[i].tpX0;
	y0= baseLine+ apf->apfUnderLinePositionPixels;
	h= apf->apfUnderlineThicknessPixels;

	while( i < drawn && tp[i].tpTextAttribute.taIsUnderlined )
	    { x1= tp[i].tpX0+ tp[i].tpPixelsWide; i++;	}

	appDrawFillRectangle( add, x0- sdd->sddOx, y0- sdd->sddOy, x1- x0, h );
	}

    return drawn;
    }

static int tedDrawTextLine(	const BufferItem *		bi,
				int				line,
				const FormattingFrame *		ign_ffTwips,
				const FormattingFrame *		ffPixels,
				void *				vsdd,
				DrawingContext *		dc )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;

    const TextLine *		tl= bi->biParaLines+ line;
    const TextParticule *	tp= bi->biParaParticules+ tl->tlFirstParticule;

    AppDrawingData *	add= dc->dcDrawingData;
    int			done;

    done= 0;
    while( done < tl->tlParticuleCount )
	{
	int		drawn;
	int		baseline;

	if  ( tp->tpTextAttribute.taShowAsLink )
	    { tedDrawSetForeground( add, sdd, SDDpixelLINK );	}
	else{ tedDrawSetForeground( add, sdd, SDDpixelFORE );	}

	baseline= tl->tlTopPosition.lpYPixels+ tl->tlLineAscentPixels;

	drawn= tedDrawParticules( bi, tp, tl->tlParticuleCount- done,
							baseline, sdd, dc );
	if  ( drawn < 1 )
	    { LDEB(drawn); return -1;	}

	done += drawn; tp += drawn;
	}

    return done;
    }

static int tedLineRectangle(	const BufferItem *		bi,
				AppDrawingData *		add,
				int				line,
				const BufferSelection *		bs,
				const DocumentRectangle *	drClip,
				const FormattingFrame *		ffPixels,
				DocumentRectangle *		drRedraw )
    {
    double		xfac= add->addMagnifiedPixelsPerTwip;
    BufferPosition	bpLineBegin;
    BufferPosition	bpLineEnd;

    const TextLine *	tl= bi->biParaLines+ line;

    DocumentRectangle	drLine;

    const int		mindLine= 1;

    docLinePositions( &bpLineBegin, &bpLineEnd, bi, line );

    drLine.drY0= tl->tlTopPosition.lpYPixels;
    drLine.drY1= tl->tlTopPosition.lpYPixels+
				TWIPStoPIXELS( xfac, tl->tlLineSpacingTwips );

    if  ( docComparePositions( &(bs->bsBegin), &bpLineBegin, mindLine ) < 0 )
	{
	drLine.drX0= ffPixels->ffX0TextLines;

	if  ( bi->biParaFirstIndentTwips < 0 )
	    { drLine.drX0= ffPixels->ffX0FirstLine;	}
	}
    else{ drLine.drX0= bs->bsBegin.bpXPixels;	}

    if  ( docComparePositions( &(bs->bsEnd), &bpLineEnd, mindLine ) > 0 )
	{ drLine.drX1= ffPixels->ffX1TextLines;	}
    else{ drLine.drX1= bs->bsEnd.bpXPixels;		}

    if  ( ! drClip || docIntersectRectangle( drRedraw, drClip, &drLine ) )
	{ return 1;	}
    else{ return 0;	}
    }

static int tedDrawTextReverse(	const BufferItem *		bi,
				int				line,
				const FormattingFrame *		ign_ffTwips,
				const FormattingFrame *		ffPixels,
				void *				vsdd,
				DrawingContext *		dc )
    {
    const BufferSelection *	bs= dc->dcSelection;
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;

    DocumentRectangle		drRedraw;
    int				done;

    if  ( tedLineRectangle( bi, add, line, bs,
					dc->dcClipRect, ffPixels, &drRedraw ) )
	{
	drRedraw.drX0 -= sdd->sddOx;
	drRedraw.drX1 -= sdd->sddOx;
	drRedraw.drY0 -= sdd->sddOy;
	drRedraw.drY1 -= sdd->sddOy;

	appDrawSetClipRect( add, &drRedraw );

	done= tedDrawTextLine( bi, line, ign_ffTwips, ffPixels, vsdd, dc );
	}
    else{
	TextLine *	tl= bi->biParaLines+ line;

	done= tl->tlParticuleCount;
	}

    return done;
    }

static int tedHighlightBack(	const BufferItem *		bi,
				int				line,
				const FormattingFrame *		ign_ffTwips,
				const FormattingFrame *		ffPixels,
				void *				vsdd,
				DrawingContext *		dc )
    {
    const BufferSelection *	bs= dc->dcSelection;
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;

    TextLine *			tl= bi->biParaLines+ line;

    DocumentRectangle		drRedraw;

    if  ( tedLineRectangle( bi, add, line, bs,
					dc->dcClipRect, ffPixels, &drRedraw ) )
	{
	tedDrawSetForeground( add, sdd, SDDpixelBACK );

	appDrawFillRectangle( add,
		drRedraw.drX0- sdd->sddOx,
		drRedraw.drY0- sdd->sddOy,
		drRedraw.drX1- drRedraw.drX0+ 1,
		drRedraw.drY1- drRedraw.drY0+ 1 );
	}

    return tl->tlParticuleCount;
    }

/************************************************************************/
/*									*/
/*  Draw a single paragraph. The Graphical context has been prepared	*/
/*  by the caller.							*/
/*									*/
/*  7)	Do not print below the bottom of fixed height cells. ( Word	*/
/*	does not do so either ).					*/
/*									*/
/************************************************************************/

static int tedDrawParaBottom(	const BufferItem *		bi,
				const BorderProperties *	bp,
				const FormattingFrame *		ign_ffTwips,
				const FormattingFrame *		ffPixels,
				void *				vsdd,
				DrawingContext *		dc )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;
    double			xfac= add->addMagnifiedPixelsPerTwip;
    int				spaceAfterPixels;

    const int			asGrid= 0;

    int				x0;

    spaceAfterPixels= TWIPStoPIXELS( xfac, bi->biParaSpaceAfterTwips );

    x0= ffPixels->ffX0TextLines;
    if  ( bi->biParaFirstIndentTwips < 0 )
	{ x0= ffPixels->ffX0FirstLine;	}

    tedDrawSetForeground( add, sdd, SDDpixelFORE );

    tedDrawHorizontalBorder( bp, asGrid, dc, sdd, 1,
				x0, ffPixels->ffX1TextLines,
				bi->biBelowPosition.lpYPixels-
				spaceAfterPixels -1 );

    return 0;
    }


static int tedDrawParaTop(	const BufferItem *		bi,
				const BorderProperties *	bp,
				const FormattingFrame *		ign_ffTwips,
				const FormattingFrame *		ffPixels,
				void *				vsdd,
				DrawingContext *		dc )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;
    double			xfac= add->addMagnifiedPixelsPerTwip;
    int				spaceBeforePixels;

    const int			asGrid= 0;

    int				x0;

    spaceBeforePixels= TWIPStoPIXELS( xfac, bi->biParaSpaceBeforeTwips );

    x0= ffPixels->ffX0TextLines;
    if  ( bi->biParaFirstIndentTwips < 0 )
	{ x0= ffPixels->ffX0FirstLine;	}

    tedDrawSetForeground( add, sdd, SDDpixelFORE );

    tedDrawHorizontalBorder( bp, asGrid, dc, sdd, 0,
					    x0, ffPixels->ffX1TextLines,
					    bi->biTopPosition.lpYPixels+
					    spaceBeforePixels );
    return 0;
    }

static int tedDrawCellTop(	const BorderProperties *	bp,
				int				asGrid,
				int				ign_x0Twips,
				int				x0Pixels,
				int				ign_x1Twips,
				int				x1Pixels,
				void *				vsdd,
				DrawingContext *		dc,
				const LayoutPosition *		lpTop )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;

    if  ( asGrid )
	{ tedDrawSetForeground( add, sdd, SDDpixelGRID );	}
    else{ tedDrawSetForeground( add, sdd, SDDpixelFORE );	}

    tedDrawHorizontalBorder( bp, asGrid, dc, sdd, 1,
				    x0Pixels, x1Pixels, lpTop->lpYPixels );

    return 0;
    }

static int tedDrawCellLeft(	const BorderProperties *	bp,
				int				asGrid,
				void *				vsdd,
				DrawingContext *		dc,
				int				x0Twips,
				int				x0Pixels,
				const LayoutPosition *		lpTop,
				const LayoutPosition *		lpBelow )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;

    if  ( asGrid )
	{ tedDrawSetForeground( add, sdd, SDDpixelGRID );	}
    else{ tedDrawSetForeground( add, sdd, SDDpixelFORE );	}

    tedDrawVerticalBorder( bp, asGrid, dc, sdd, x0Pixels,
				    lpTop->lpYPixels, lpBelow->lpYPixels );

    return 0;
    }

static int tedDrawCellRight(	const BorderProperties *	bp,
				int				asGrid,
				void *				vsdd,
				DrawingContext *		dc,
				int				x1Twips,
				int				x1Pixels,
				const LayoutPosition *		lpTop,
				const LayoutPosition *		lpBelow )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;

    if  ( asGrid )
	{ tedDrawSetForeground( add, sdd, SDDpixelGRID );	}
    else{ tedDrawSetForeground( add, sdd, SDDpixelFORE );	}

    tedDrawVerticalBorder( bp, asGrid, dc, sdd, x1Pixels,
				    lpTop->lpYPixels, lpBelow->lpYPixels );

    return 0;
    }

static int tedDrawCellBottom(	const BorderProperties *	bp,
				int				asGrid,
				int				ign_x0Twips,
				int				x0Pixels,
				int				ign_x1Twips,
				int				x1Pixels,
				void *				vsdd,
				DrawingContext *		dc,
				const LayoutPosition *		lpBottom )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;

    if  ( asGrid )
	{ tedDrawSetForeground( add, sdd, SDDpixelGRID );	}
    else{ tedDrawSetForeground( add, sdd, SDDpixelFORE );	}

    tedDrawHorizontalBorder( bp, asGrid, dc, sdd, 0,
				    x0Pixels, x1Pixels, lpBottom->lpYPixels );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Skip to the next page.						*/
/*									*/
/************************************************************************/

static void tedDrawPageGap(	ScreenDrawingData *		sdd,
				DrawingContext *		dc,
				int				page )
    {
    AppDrawingData *		add= dc->dcDrawingData;

    DocumentRectangle		drBetween;

    drBetween.drX0= add->addBackRect.drX0;
    drBetween.drX1= add->addBackRect.drX1;
    drBetween.drY0= page* add->addPageStepPixels- add->addPageGapPixels;
    drBetween.drY1= page* add->addPageStepPixels;

    if  ( dc->dcClipRect						   &&
	  ! docIntersectRectangle( &drBetween, &drBetween, dc->dcClipRect ) )
	{ return;	}

    tedDrawSetForeground( add, sdd, SDDpixelPAGE );

    appDrawFillRectangle( add,
					drBetween.drX0- sdd->sddOx,
					drBetween.drY0- sdd->sddOy,
					drBetween.drX1- drBetween.drX0+ 1,
					drBetween.drY1- drBetween.drY0+ 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Draw an I Bar.							*/
/*									*/
/************************************************************************/
#   define	IW	5

static int tedDrawIBar(	const BufferPosition *	bp,
			int			ox,
			int			oy,
			AppDrawingData *	add )
    {
    int		y0= bp->bpTopPosition.lpYPixels;

    appDrawFillRectangle( add, bp->bpXPixels- ox, y0- oy, 1, bp->bpY1Pixels- y0 );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw the eight blocks around a selected object.			*/
/*									*/
/************************************************************************/

static void tedDrawObjectBlocks(	int			x0,
					int			y,
					int			wide,
					int			high,
					const DrawingContext *	dc,
					ScreenDrawingData *	sdd )
    {
    AppDrawingData *	add= dc->dcDrawingData;

    APP_POINT		xp[8];
    int			i;

    /*  bottom  */
    xp[0].x= x0- sdd->sddOx;
    xp[0].y= y- RESIZE_BLOCK- sdd->sddOy;

    xp[1].x= x0 +wide/2- RESIZE_BLOCK/2- sdd->sddOx;
    xp[1].y= y- RESIZE_BLOCK- sdd->sddOy;

    xp[2].x= x0 +wide- RESIZE_BLOCK- sdd->sddOx;
    xp[2].y= y- RESIZE_BLOCK- sdd->sddOy;

    /*  middle */
    xp[3].x= x0- sdd->sddOx;
    xp[3].y= y- high/2- RESIZE_BLOCK/2 -sdd->sddOy;

    xp[4].x= x0 +wide- RESIZE_BLOCK- sdd->sddOx;
    xp[4].y= y- high/2- RESIZE_BLOCK/2 -sdd->sddOy;

    /*  top  */
    xp[5].x= x0- sdd->sddOx;
    xp[5].y= y- high- sdd->sddOy;

    xp[6].x= x0 +wide/2- RESIZE_BLOCK/2- sdd->sddOx;
    xp[6].y= y- high- sdd->sddOy;

    xp[7].x= x0 +wide- RESIZE_BLOCK- sdd->sddOx;
    xp[7].y= y- high- sdd->sddOy;

    tedDrawSetForeground( add, sdd, SDDpixelBACK );

    for ( i= 0; i < 8; i++ )
	{
	appDrawFillRectangle( add, xp[i].x, xp[i].y, RESIZE_BLOCK, RESIZE_BLOCK );
	}

    tedDrawSetForeground( add, sdd, SDDpixelFORE );

    for ( i= 0; i < 8; i++ )
	{
	appDrawDrawRectangle( add,
			    xp[i].x, xp[i].y, RESIZE_BLOCK- 1, RESIZE_BLOCK- 1 );
	}

    appDrawDrawRectangle( add,
			x0- sdd->sddOx, y- high- sdd->sddOy, wide- 1, high- 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Draw the box around an active header or footer.			*/
/*									*/
/************************************************************************/

static void tedDrawHeaderFooterBox(	BufferItem *			sectBi,
					const DocumentProperties *	dp,
					ScreenDrawingData *		sdd,
					DrawingContext *		dc,
					int				page,
					int				inHdFt )
    {
    AppDrawingData *		add= dc->dcDrawingData;
    
    DocumentRectangle		drBox;

    static char			dot[]= { 1, 2 };

    if  ( tedGetHeaderFooterBox( &drBox, sectBi, dp, add, page, inHdFt ) )
	{ LDEB(inHdFt); return;	}

#   ifdef USE_MOTIF
    XSetLineAttributes( add->addDisplay, add->addGc,
				    1, LineOnOffDash, CapButt, JoinMiter );

    XSetDashes( add->addDisplay, add->addGc, 0, dot, sizeof( dot ) );
#   endif

#   ifdef USE_GTK
    gdk_gc_set_line_attributes( add->addGc,
		    1, GDK_LINE_ON_OFF_DASH, GDK_CAP_BUTT, GDK_JOIN_MITER );

    gdk_gc_set_dashes( add->addGc, 0, dot, sizeof( dot ) );
#   endif

    appDrawDrawLine( add,
			    drBox.drX0- sdd->sddOx, drBox.drY0- sdd->sddOy,
			    drBox.drX1- sdd->sddOx, drBox.drY0- sdd->sddOy );
    appDrawDrawLine( add,
			    drBox.drX0- sdd->sddOx, drBox.drY1- sdd->sddOy,
			    drBox.drX1- sdd->sddOx, drBox.drY1- sdd->sddOy );
    appDrawDrawLine( add,
			    drBox.drX0- sdd->sddOx, drBox.drY0- sdd->sddOy,
			    drBox.drX0- sdd->sddOx, drBox.drY1- sdd->sddOy );
    appDrawDrawLine( add,
			    drBox.drX1- sdd->sddOx, drBox.drY0- sdd->sddOy,
			    drBox.drX1- sdd->sddOx, drBox.drY1- sdd->sddOy );

    return;
    }

/************************************************************************/
/*									*/
/*  Expose handler for documents.					*/
/*									*/
/*  2)  Clear background.						*/
/*  2b) If the selected area overlaps with the exposed region, draw the	*/
/*	selection background.						*/
/*  3)  Draw text.							*/
/*  4)	Draw page separators.						*/
/*  5)  Draw I bar if necessary.					*/
/*									*/
/************************************************************************/

void tedRedrawRectangle(	APP_WIDGET		w,
				TedDocument *		td,
				DocumentRectangle *	drClip,
				AppDrawingData *	add,
				AppColors *		acSys,
				int			ox,
				int			oy )
    {
    BufferDocument *		bd= td->tdDocument;
    BufferItem *		docBi= &(bd->bdItem);
    DocumentProperties *	dp= &(bd->bdProperties);

    DrawingContext		dc;
    ScreenDrawingData		sdd;

    BufferItem *		selBi= (BufferItem *)0;

    APP_COLOR_RGB		selColor;
    APP_COLOR_RGB		linkFore;

    SelectionScope		ss;
    BufferSelection		bs;

    if  ( tedGetSelection( &ss, &bs, td ) )
	{ LDEB(1); return;	}

    if  ( td->tdVisibleSelectionCopied )
	{ selColor= td->tdXSelColor;		}
    else{ selColor= td->tdSelColor;		}

    linkFore= td->tdFieldColor;

    docInitDrawingContext( &dc );

    dc.dcDrawingData= add;
    dc.dcDocument= bd;
    dc.dcDrawTableGrid= td->tdDrawTableGrid;
    dc.dcClipRect= drClip;

    sdd.sddOx= ox;
    sdd.sddOy= oy;

    if  ( appDrawBlackColor( add, &(sdd.sddForeColor) ) )
	{ LDEB(1); return;	}
    if  ( appDrawWhiteColor( add, &(sdd.sddBackColor) ) )
	{ LDEB(1); return;	}
    sdd.sddLinkColor= linkFore;
    sdd.sddGridColor= td->tdTableColor;
    sdd.sddPageColor= add->addBackColor;

#   if LOG_REDRAWS
    docLogRectangle( "REDRAW", drClip );
#   endif

    /*  2  */
    tedDrawSetForeground( add, &sdd, SDDpixelUNDEF );
    tedDrawSetForeground( add, &sdd, SDDpixelBACK );

    appDrawFillRectangle( add,
			    drClip->drX0- ox, drClip->drY0-oy,
			    drClip->drX1- drClip->drX0+ 1,
			    drClip->drY1- drClip->drY0+ 1 );

    /*  2a  */
    if  ( tedHasSelection( td ) && ! tedHasIBarSelection( td ) )
	{
	const int	nearest= 1;

	selBi= docGetSelectionRoot( bd, nearest, &ss, &bs );

	if  ( ! selBi )
	    { XDEB(selBi);	}
	}

    if  ( selBi )
	{
	tedDrawSetForeground( add, &sdd, SDDpixelUNDEF );
	sdd.sddBackColor= selColor;

	dc.dcDrawTextLine= tedHighlightBack;
	dc.dcDrawParaTop= (DRAW_PARA_TOP)0;
	dc.dcDrawParaBottom= (DRAW_PARA_BOTTOM)0;
	dc.dcDrawCellTop= (DRAW_CELL_TOP)0;
	dc.dcDrawCellBottom= (DRAW_CELL_BOTTOM)0;
	dc.dcDrawCellLeft= (DRAW_CELL_LEFT)0;
	dc.dcDrawCellRight= (DRAW_CELL_RIGHT)0;
	dc.dcParaSidesPixels= tedParagraphFramePixels;
	dc.dcFinishPage= (FINISH_PAGE)0;
	dc.dcStartPage= (START_PAGE)0;

	dc.dcLayoutHeader= (LAYOUT_EXTERNAL)0;
	dc.dcLayoutFooter= (LAYOUT_EXTERNAL)0;
	dc.dcDrawHeadersFooters= 0;

	dc.dcSelection= &bs;

	if  ( docDrawItem( selBi, (void *)&sdd, &dc ) )
	    { LDEB(1);	}
	}

    /*  3  */
    if  ( appDrawBlackColor( add, &(sdd.sddForeColor) ) )
	{ LDEB(1); return;	}
    if  ( appDrawWhiteColor( add, &(sdd.sddBackColor) ) )
	{ LDEB(1); return;	}

    dc.dcDrawTextLine= tedDrawTextLine;
    dc.dcDrawParaTop= tedDrawParaTop;
    dc.dcDrawParaBottom= tedDrawParaBottom;
    dc.dcDrawCellTop= tedDrawCellTop;
    dc.dcDrawCellBottom= tedDrawCellBottom;
    dc.dcDrawCellLeft= tedDrawCellLeft;
    dc.dcDrawCellRight= tedDrawCellRight;
    dc.dcParaSidesPixels= tedParagraphFramePixels;

    dc.dcLayoutHeader= tedLayoutHeaderItem;
    dc.dcLayoutFooter= tedLayoutFooterItem;
    dc.dcDrawHeadersFooters= 1;

    dc.dcSelection= (BufferSelection *)0;

    appDrawSetBackgroundWhite( add );

    tedDrawSetForeground( add, &sdd, SDDpixelUNDEF );
    tedDrawSetForeground( add, &sdd, SDDpixelFORE );
    dc.dcCurrentPhysicalFont= -1;

    if  ( docDrawItem( docBi, (void *)&sdd, &dc ) )
	{ LDEB(1);	}

    {
    int		firstPage;
    int		lastPage;

    int		page;
    int		sect;

    int		y0= drClip->drY0;
    int		y1= drClip->drY1;

    y0= drClip->drY0- add->addPageGapPixels;
    if  ( y0 < 0 )
	{ y0= 0;	}

    y1= drClip->drY1+ add->addPageStepPixels- 1;
    if  ( y1 > docBi->biBelowPosition.lpYPixels )
	{ y1=  docBi->biBelowPosition.lpYPixels;	}

    firstPage= y0/ add->addPageStepPixels;
    lastPage=  y1/ add->addPageStepPixels;

    if  ( lastPage > docBi->biBelowPosition.lpPage )
	{ lastPage=  docBi->biBelowPosition.lpPage;	}

    for ( page= firstPage; page <= lastPage; page++ )
	{ tedDrawPageGap( &sdd, &dc, page );	}

    for ( sect= 0; sect < docBi->biChildCount; sect++ )
	{
	BufferItem *	sectBi= docBi->biChildren[sect];
	BufferItem *	nextSectBi= (BufferItem *)0;
	BufferItem *	prevSectBi= (BufferItem *)0;
	int		first= sectBi->biTopPosition.lpPage;
	int		last= sectBi->biBelowPosition.lpPage;

	if  ( first < firstPage )
	    { first=  firstPage;	}
	if  ( last > lastPage )
	    { last= lastPage;	}

	if  ( sect > 0 )
	    { prevSectBi= docBi->biChildren[sect- 1];	}
	if  ( sect < docBi->biChildCount- 1 )
	    { nextSectBi= docBi->biChildren[sect+ 1];	}

	for ( page= first; page <= last; page++ )
	    {
	    if  ( ! prevSectBi					||
		  prevSectBi->biBelowPosition.lpPage < page	)
		{ docDrawPageHeader( sectBi, (void *)&sdd, &dc, page ); }

	    if  ( ! nextSectBi					||
		  nextSectBi->biTopPosition.lpPage > page	)
		{ docDrawPageFooter( sectBi, (void *)&sdd, &dc, page ); }

	    if  ( ss.ssInHeaderFooter != DOCinBODY		&&
		  page == ss.ssHeaderFooterPage			&&
		  sect == ss.ssHeaderFooterSection		)
		{
		tedDrawSetForeground( add, &sdd, SDDpixelFORE );

		tedDrawHeaderFooterBox( sectBi, dp, &sdd, &dc,
						page, ss.ssInHeaderFooter );
		}
	    }
	}

    }

    if  ( selBi && acSys->acAllocator.caDepth < 4 )
	{
	if  ( appDrawWhiteColor( add, &(sdd.sddForeColor) ) )
	    { LDEB(1); return;	}
	if  ( appDrawWhiteColor( add, &(sdd.sddLinkColor) ) )
	    { LDEB(1); return;	}

	tedDrawSetForeground( add, &sdd, SDDpixelUNDEF );
	tedDrawSetForeground( add, &sdd, SDDpixelFORE );

	dc.dcDrawTextLine= tedDrawTextReverse;
	dc.dcDrawParaTop= (DRAW_PARA_TOP)0;
	dc.dcDrawParaBottom= (DRAW_PARA_BOTTOM)0;
	dc.dcDrawCellTop= (DRAW_CELL_TOP)0;
	dc.dcDrawCellBottom= (DRAW_CELL_BOTTOM)0;
	dc.dcDrawCellLeft= (DRAW_CELL_LEFT)0;
	dc.dcDrawCellRight= (DRAW_CELL_RIGHT)0;
	dc.dcParaSidesPixels= tedParagraphFramePixels;
	dc.dcFinishPage= (FINISH_PAGE)0;
	dc.dcStartPage= (START_PAGE)0;

	dc.dcLayoutHeader= (LAYOUT_EXTERNAL)0;
	dc.dcLayoutFooter= (LAYOUT_EXTERNAL)0;
	dc.dcDrawHeadersFooters= 0;

	dc.dcSelection= &bs;

	if  ( docDrawItem( selBi, (void *)&sdd, &dc ) )
	    { LDEB(1);	}
	}

    tedDrawSetForeground( add, &sdd, SDDpixelFORE );

    {
    /*  5  */
    if  ( tedHasIBarSelection( td ) )
	{
	const BufferPosition *	bpBegin= &(bs.bsBegin);

	if  ( drClip->drY0 <= bpBegin->bpY1Pixels		&&
	      drClip->drY1 >= bpBegin->bpTopPosition.lpYPixels	&&
	      ! td->tdShowIBarId				)
	    { tedDrawIBar( bpBegin, ox, oy, add ); }
	}
    else{
	InsertedObject *	io;
	BufferPosition		bp;

	docInitPosition( &bp );

	if  ( td->tdObjectSelected			&&
	      ! tedGetObjectSelection( td, &bp, &io  )	)
	    {
	    tedDrawSetForeground( add, &sdd, SDDpixelUNDEF );
	    if  ( appDrawBlackColor( add, &(sdd.sddForeColor) ) )
		{ LDEB(1); return;	}
	    sdd.sddBackColor= selColor;

	    tedPositionCoordinates( &bp, add );

	    if  ( drClip->drY0 <= bp.bpY1Pixels			&&
		  drClip->drY1 >= bp.bpTopPosition.lpYPixels	)
		{
		if  ( io->ioDragWide > 0 )
		    {
		    tedDrawObjectBlocks( bp.bpXPixels,
			bp.bpBaselinePixels+ io->ioDragHigh- io->ioPixelsHigh,
			io->ioDragWide, io->ioDragHigh, &dc, &sdd );
		    }
		else{
		    tedDrawObjectBlocks( bp.bpXPixels, bp.bpBaselinePixels,
			io->ioPixelsWide, io->ioPixelsHigh, &dc, &sdd );
		    }
		}
	    }
	}
    }
	  
    return;
    }

/************************************************************************/
/*									*/
/*  Cause a rectangle to be redrawn.					*/
/*									*/
/*  Cause the smallest rectangle that contains the selection to be	*/
/*  redrawn.								*/
/*									*/
/************************************************************************/

void tedExposeRectangle(	const EditDocument *		ed,
				const DocumentRectangle *	drChanged,
				int				scrolledX,
				int				scrolledY )
    {
    DocumentRectangle		drExpose;
    DocumentRectangle		drScrolled;

    const AppDrawingData *	add= &(ed->edDrawingData);
    const DocumentRectangle *	drVisible= &(ed->edVisibleRect);

    if  ( ! drChanged )
	{ appExposeRectangle( add, 0, 0, 0, 0 ); return; }

    drScrolled= *drChanged;
    drScrolled.drX0 -= scrolledX;
    drScrolled.drX1 -= scrolledX;
    drScrolled.drY0 -= scrolledY;
    drScrolled.drY1 -= scrolledY;

    if  ( scrolledX != 0 )
	{
	drScrolled.drY0= drVisible->drY0;
	drScrolled.drY1= drVisible->drY1;
	}

    if  ( scrolledY != 0 )
	{
	drScrolled.drX0= drVisible->drX0;
	drScrolled.drX1= drVisible->drX1;
	}

    docUnionRectangle( &drScrolled, &drScrolled, drChanged );

    drExpose.drX0= drVisible->drX0;
    drExpose.drY0= drVisible->drY0;
    drExpose.drX1= drVisible->drX1;
    drExpose.drY1= drVisible->drY1;

    if  ( docIntersectRectangle( &drExpose, &drExpose, &drScrolled ) )
	{
	int	ox= drVisible->drX0;
	int	oy= drVisible->drY0;

#	if LOG_REDRAWS
	docLogRectangle( "CLEAR!", &drExpose );
#	endif

	appExposeRectangle( add,
				    drExpose.drX0- ox,
				    drExpose.drY0- oy,
				    drExpose.drX1- drExpose.drX0+ 1,
				    drExpose.drY1- drExpose.drY0+ 1 );
	}
    }

void tedExposeSelection(	const EditDocument *	ed,
				const BufferSelection *	bs,
				int			scrolledX,
				int			scrolledY )
    {
    const AppDrawingData *	add= &(ed->edDrawingData);
    DocumentRectangle		drClip;

    tedSelectionRectangle( &drClip, add, bs );

    tedExposeRectangle( ed, &drClip, scrolledX, scrolledY );

    return;
    }

/************************************************************************/
/*									*/
/*  Blinking cursor.							*/
/*									*/
/*  1)  Turned off when we are debugging exposures and redraws.		*/
/*									*/
/************************************************************************/

void tedUndrawIBar(	const EditDocument *	ed )
    {
#   if LOG_REDRAWS
    /*  1  */
#   else
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    const BufferPosition *	bp= &(td->tdSelection.bsBegin);

    DocumentRectangle		drIBar;

    drIBar.drX0= bp->bpXPixels- IW;
    drIBar.drX1= bp->bpXPixels+ IW;
    drIBar.drY0= bp->bpTopPosition.lpYPixels;
    drIBar.drY1= bp->bpY1Pixels;

    tedExposeRectangle( ed, &drIBar, /*scrolledX,Y*/ 0,0 );
#   endif

    return;
    }

void tedRedrawIBar(	TedDocument *		td,
			int			ox,
			int			oy,
			AppDrawingData *	add )
    {
    tedDrawIBar( &(td->tdSelection.bsBegin), ox, oy, add );

    return;
    }

