/************************************************************************/
/*									*/
/*  Buffer manipulation routines.					*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	"docBuf.h"

/************************************************************************/
/*									*/
/*  Initialise a BufferPosition/BufferSelection.			*/
/*									*/
/************************************************************************/

void docInitPosition(	BufferPosition *	bp )
    {
    bp->bpBi= (BufferItem *)0;
    bp->bpLine= 0;
    bp->bpParticule= -1;
    bp->bpStroff= -1;

    bp->bpXPixels= -1;
    docInitLayoutPosition( &bp->bpTopPosition );
    bp->bpY1Pixels= -1;
    bp->bpBaselinePixels= -1;

    return;
    }

void docInitSelectionScope(	SelectionScope *	ss )
    {
    ss->ssInHeaderFooter= DOCinBODY;
    ss->ssHeaderFooterSection= -1;
    ss->ssHeaderFooterPage= -1;

    return;
    }

void docInitSelection(	BufferSelection *	bs )
    {
    docInitPosition( &(bs->bsBegin) );
    docInitPosition( &(bs->bsEnd) );
    docInitPosition( &(bs->bsAnchor) );

    bs->bsCol0= -1;
    bs->bsCol1= -1;

    bs->bsDirection= 0;

    return;
    }

void docInitTableRectangle(	TableRectangle *	tr )
    {
    tr->trSectionNumber= -1;

    tr->trCol0= -1;
    tr->trCol1= -1;
    tr->trCol11= -1;

    tr->trRow00= -1;
    tr->trRow0= -1;
    tr->trRow1= -1;
    tr->trRow11= -1;

    return;
    }

/************************************************************************/
/*									*/
/*  Reduce a selection to one paragraph.				*/
/*									*/
/************************************************************************/

void docConstrainSelectionToOneParagraph( int *			pBeginMoved,
					int *			pEndMoved,
					BufferSelection *	bs )
    {
    int		lenBegin;
    int		lenEnd;

    if  ( bs->bsBegin.bpBi == bs->bsEnd.bpBi )
	{ return;	}

    lenBegin= bs->bsBegin.bpBi->biParaStrlen- bs->bsBegin.bpStroff;
    lenEnd= bs->bsEnd.bpStroff;

    if  ( lenEnd > lenBegin )
	{
	bs->bsBegin.bpBi= bs->bsEnd.bpBi;
	bs->bsBegin.bpParticule= 0;
	bs->bsBegin.bpStroff= 0;
	bs->bsCol0= bs->bsCol1;

	if  ( bs->bsAnchor.bpBi != bs->bsBegin.bpBi )
	    { bs->bsAnchor= bs->bsBegin;	}

	*pBeginMoved= 1; *pEndMoved= 0; return;
	}
    else{
	bs->bsEnd.bpBi= bs->bsBegin.bpBi;
	bs->bsEnd.bpParticule= bs->bsEnd.bpBi->biParaParticuleCount -1;
	bs->bsEnd.bpStroff= bs->bsEnd.bpBi->biParaStrlen;
	bs->bsCol1= bs->bsCol0;

	if  ( bs->bsAnchor.bpBi != bs->bsEnd.bpBi )
	    { bs->bsAnchor= bs->bsEnd;	}

	*pBeginMoved= 0; *pEndMoved= 1; return;
	}

    return;
    }


/************************************************************************/
/*									*/
/*  Return the union of two BufferSelections in the same paragraph.	*/
/*									*/
/************************************************************************/

void docUnionParaSelections(		BufferSelection *	bs,
					const BufferSelection *	bs1,
					const BufferSelection *	bs2 )
    {
    BufferSelection	bsNew;

    if  ( bs1->bsBegin.bpBi !=  bs1->bsEnd.bpBi )
	{ XXDEB(bs1->bsBegin.bpBi,bs1->bsEnd.bpBi); return;	}
    if  ( bs2->bsBegin.bpBi !=  bs2->bsEnd.bpBi )
	{ XXDEB(bs2->bsBegin.bpBi,bs2->bsEnd.bpBi); return;	}
    if  ( bs1->bsBegin.bpBi !=  bs2->bsEnd.bpBi )
	{ XXDEB(bs1->bsBegin.bpBi,bs2->bsEnd.bpBi); return;	}

    bsNew= *bs1;

    if  ( bs2->bsBegin.bpStroff < bs1->bsBegin.bpStroff )
	{ bsNew.bsBegin.bpStroff= bs2->bsBegin.bpStroff;	}
    else{ bsNew.bsBegin.bpStroff= bs1->bsBegin.bpStroff;	}

    if  ( bs2->bsBegin.bpParticule < bs1->bsBegin.bpParticule )
	{ bsNew.bsBegin.bpParticule= bs2->bsBegin.bpParticule;	}
    else{ bsNew.bsBegin.bpParticule= bs1->bsBegin.bpParticule;	}

    if  ( bs2->bsBegin.bpLine < bs1->bsBegin.bpLine )
	{ bsNew.bsBegin.bpLine= bs2->bsBegin.bpLine;	}
    else{ bsNew.bsBegin.bpLine= bs1->bsBegin.bpLine;	}

    /**/

    if  ( bs2->bsEnd.bpStroff > bs1->bsEnd.bpStroff )
	{ bsNew.bsEnd.bpStroff= bs2->bsEnd.bpStroff;	}
    else{ bsNew.bsEnd.bpStroff= bs1->bsEnd.bpStroff;	}

    if  ( bs2->bsEnd.bpParticule > bs1->bsEnd.bpParticule )
	{ bsNew.bsEnd.bpParticule= bs2->bsEnd.bpParticule;	}
    else{ bsNew.bsEnd.bpParticule= bs1->bsEnd.bpParticule;	}

    if  ( bs2->bsEnd.bpLine > bs1->bsEnd.bpLine )
	{ bsNew.bsEnd.bpLine= bs2->bsEnd.bpLine;	}
    else{ bsNew.bsEnd.bpLine= bs1->bsEnd.bpLine;	}

    /**/

    *bs= bsNew; return;
    }

/************************************************************************/
/*									*/
/*  Intersect two TableRectangles.					*/
/*									*/
/************************************************************************/

int docIntersectTableRectangle(	TableRectangle *		tr,
				const TableRectangle *		tr1,
				const TableRectangle *		tr2 )
    {
    TableRectangle	res= *tr1;

    if  ( tr1->trSectionNumber != tr2->trSectionNumber )
	{ return 0;	}

    if  ( tr1->trRow00 != tr2->trRow00 )
	{ LLDEB(tr1->trRow00,tr2->trRow00); return 0;	}
    if  ( tr1->trRow11 != tr2->trRow11 )
	{ LLDEB(tr1->trRow11,tr2->trRow11); return 0;	}
    if  ( tr1->trCol11 != tr2->trCol11 )
	{ LLDEB(tr1->trCol11,tr2->trCol11); return 0;	}

    if  ( tr1->trCol1 < tr2->trCol0 )
	{ return 0;	}

    if  ( tr2->trCol1 < tr1->trCol0 )
	{ return 0;	}

    if  ( tr1->trRow1 < tr2->trRow0 )
	{ return 0;	}

    if  ( tr2->trRow1 < tr1->trRow0 )
	{ return 0;	}

    if  ( ! tr )
	{ return 1;	}

    if  ( tr1->trCol0 < tr2->trCol0 )
	{ res.trCol0= tr2->trCol0;	}
    else{ res.trCol0= tr1->trCol0;	}

    if  ( tr1->trRow0 < tr2->trRow0 )
	{ res.trRow0= tr2->trRow0;	}
    else{ res.trRow0= tr1->trRow0;	}

    if  ( tr1->trCol1 < tr2->trCol1 )
	{ res.trCol1= tr1->trCol1;	}
    else{ res.trCol1= tr2->trCol1;	}

    if  ( tr1->trRow1 < tr2->trRow1 )
	{ res.trRow1= tr1->trRow1;	}
    else{ res.trRow1= tr2->trRow1;	}

    *tr= res; return 1;
    }

void docExpandTableRectangleToWholeTable(	TableRectangle *	tr )
    {
    tr->trRow0= tr->trRow00;
    tr->trRow1= tr->trRow11;
    tr->trCol0= 0;
    tr->trCol1= tr->trCol11;

    return;
    }

void docExpandTableRectangleToWholeRows(	TableRectangle *	tr )
    {
    tr->trCol0= 0;
    tr->trCol1= tr->trCol11;

    return;
    }

void docExpandTableRectangleToWholeColumns(	TableRectangle *	tr )
    {
    tr->trRow0= tr->trRow00;
    tr->trRow1= tr->trRow11;

    return;
    }

int docShiftTableRectangleByRows(	TableRectangle *	tr,
					int			rows )
    {
    TableRectangle	trShifted;
    TableRectangle	trTable;

    trShifted= *tr;
    trTable= *tr;

    trShifted.trRow0 += rows;
    trShifted.trRow1 += rows;

    docExpandTableRectangleToWholeTable( &trTable );

    if  ( ! docIntersectTableRectangle( &trShifted, &trShifted, &trTable ) )
	{ return -1;	}

    *tr= trShifted; return 0;
    }

int docShiftTableRectangleByColumns(	TableRectangle *	tr,
					int			columns )
    {
    TableRectangle	trShifted;
    TableRectangle	trTable;

    trShifted= *tr;
    trTable= *tr;

    trShifted.trCol0 += columns;
    trShifted.trCol1 += columns;

    docExpandTableRectangleToWholeTable( &trTable );

    if  ( ! docIntersectTableRectangle( &trShifted, &trShifted, &trTable ) )
	{ return -1;	}

    *tr= trShifted; return 0;
    }

/************************************************************************/
/*									*/
/*  Move to the first/last position.					*/
/*									*/
/************************************************************************/

int docFirstPosition(	BufferItem *		bi,
			BufferPosition *	bp )
    {
    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA )
	    {
	    bp->bpBi= bi;
	    bp->bpStroff= 0;
	    bp->bpParticule= 0;
	    bp->bpLine= 0;

	    return 0;
	    }

	if  ( bi->biChildCount == 0 )
	    { LDEB(bi->biChildCount); return -1;	}

	bi= bi->biChildren[0];
	}

    XDEB(bi); return -1;
    }

int docLastPosition(	BufferItem *		bi,
			BufferPosition *	bp )
    {
    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA )
	    {
	    bp->bpBi= bi;
	    bp->bpStroff= bi->biParaStrlen;
	    bp->bpParticule= bi->biParaParticuleCount- 1;
	    bp->bpLine= bi->biParaLineCount- 1;

	    return 0;
	    }

	if  ( bi->biChildCount == 0 )
	    { LDEB(bi->biChildCount); return -1;	}

	bi= bi->biChildren[bi->biChildCount- 1];
	}

    XDEB(bi); return -1;
    }

/************************************************************************/
/*									*/
/*  Move to the next/previous position.					*/
/*									*/
/************************************************************************/

static BufferItem * docNextItem(	BufferItem *	bi,
					int		level )
    {
    for (;;)
	{
	if  ( ! bi->biParent )
	    { return (BufferItem *)0;	}

	if  ( bi->biNumberInParent < bi->biParent->biChildCount- 1 )
	    {
	    bi= bi->biParent->biChildren[bi->biNumberInParent+ 1];

	    while( bi->biLevel < level	&&
		   bi->biChildCount > 0	)
		{ bi= bi->biChildren[0]; }

	    if  ( bi->biLevel == level )
		{ return bi;	}
	    }
	else{ bi= bi->biParent;	}
	}
    }

BufferItem *	docNextParagraph(	BufferItem *	bi )
    { return docNextItem( bi, DOClevPARA );	}

BufferItem *	docNextSection(	BufferItem *	bi )
    { return docNextItem( bi, DOClevSECT );	}

static BufferItem * docPrevItem(	BufferItem *	bi,
					int		level )
    {
    for (;;)
	{
	if  ( ! bi->biParent )
	    { return (BufferItem *)0;	}

	if  ( bi->biNumberInParent > 0 )
	    {
	    bi= bi->biParent->biChildren[bi->biNumberInParent- 1];

	    while( bi->biLevel < level	&&
		   bi->biChildCount > 0	)
		{ bi= bi->biChildren[bi->biChildCount- 1]; }

	    if  ( bi->biLevel == level )
		{ return bi;	}
	    }
	else{ bi= bi->biParent;	}
	}
    }

BufferItem *	docPrevParagraph(	BufferItem *	bi )
    { return docPrevItem( bi, DOClevPARA );	}

BufferItem *	docPrevSection(		BufferItem *	bi )
    { return docPrevItem( bi, DOClevSECT );	}

int docNextPosition(	BufferPosition *	bp )
    {
    BufferItem *	bi= bp->bpBi;

    int			stroff= bp->bpStroff+ 1;

    int			lastOne= 1;

    while( bi )
	{
	if  ( stroff <= bi->biParaStrlen )
	    {
	    docFindLineAndParticule( bi, stroff, bp, lastOne );
	    return 0;
	    }

	bi= docNextParagraph( bi );
	stroff= 0;

	if  ( bi && bi->biParaStrlen == 0 )
	    {
	    docFindLineAndParticule( bi, stroff, bp, lastOne );
	    return 0;
	    }
	}

    return -1;
    }

int docPrevPosition(	BufferPosition *	bp,
			int			lastOne )
    {
    BufferItem *	bi= bp->bpBi;

    int			stroff= bp->bpStroff- 1;

    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA		&&
	      stroff >= 0			)
	    {
	    docFindLineAndParticule( bi, stroff, bp, lastOne );
	    return 0;
	    }

	bi= docPrevParagraph( bi );
	if  ( bi )
	    { stroff= bi->biParaStrlen; }
	}

    return -1;
    }

int docParaBegin(	BufferPosition *	bp,
			int			lastOne )
    {
    BufferItem *	bi= bp->bpBi;
    const int		stroff= 0;

    docFindLineAndParticule( bi, stroff, bp, lastOne );
    return 0;
    }

int docParaEnd(		BufferPosition *	bp,
			int			lastOne )
    {
    BufferItem *	bi= bp->bpBi;

    docFindLineAndParticule( bi, bi->biParaStrlen, bp, lastOne );
    return 0;
    }

/************************************************************************/
/*									*/
/*  Move to the Next/Previous word.					*/
/*									*/
/*  1)  Words start after a particule ending in a space or at the	*/
/*	beginning of a paragraph.					*/
/*									*/
/************************************************************************/

int docNextWord(	BufferPosition *	bp,
			int			lastOne )
    {
    BufferItem *	bi= bp->bpBi;
    int			stroff= bp->bpStroff;

    if  ( stroff < bi->biParaStrlen )
	{
	int		part= bp->bpParticule;
	TextParticule *	tp= bi->biParaParticules+ part;
	unsigned char *	s= bi->biParaString;

	if  ( stroff == tp->tpStroff+ tp->tpStrlen )
	    { part++; tp++; }

	while( part < bi->biParaParticuleCount )
	    {
	    /*  1  */
	    if  ( tp->tpKind == DOCkindTEXT				&&
		  tp->tpStrlen > 0					&&
		  ( s[tp->tpStroff+ tp->tpStrlen- 1] == ' '	||
		    tp->tpStroff+ tp->tpStrlen == bi->biParaStrlen )	)
		{ stroff= tp->tpStroff+ tp->tpStrlen; break;	}

	    part++; tp++;
	    }

	if  ( part == bi->biParaParticuleCount )
	    { stroff= bi->biParaStrlen; }
	}
    else{
	bi= docNextParagraph( bi );

	if  ( ! bi )
	    { return -1;	}

	stroff= 0;
	}

    docFindLineAndParticule( bi, stroff, bp, lastOne );
    return 0;
    }

int docPrevWord(	BufferPosition *	bp,
			int			lastOne )
    {
    BufferItem *	bi= bp->bpBi;
    int			stroff= bp->bpStroff;

    if  ( stroff > 0 )
	{
	int		part= bp->bpParticule;
	TextParticule *	tp= bi->biParaParticules+ part;
	unsigned char *	s= bi->biParaString;

	if  ( stroff == tp->tpStroff )
	    { part--; tp--; }

	while( part >= 0 )
	    {
	    /*  1  */
	    if  ( tp->tpKind == DOCkindTEXT				&&
		  tp->tpStrlen > 0					&&
		  ( s[tp->tpStroff+ tp->tpStrlen- 1] == ' '	||
		    tp->tpStroff+ tp->tpStrlen == bi->biParaStrlen )	)
		{ break;	}

	    part--; tp--;
	    }

	part--; tp--;

	while( part >= 0 )
	    {
	    /*  1  */
	    if  ( tp->tpKind == DOCkindTEXT				&&
		  tp->tpStrlen > 0					&&
		  ( s[tp->tpStroff+ tp->tpStrlen- 1] == ' '	||
		    tp->tpStroff+ tp->tpStrlen == bi->biParaStrlen )	)
		{ stroff= tp->tpStroff+ tp->tpStrlen; break;	}

	    part--; tp--;
	    }

	if  ( part < 0 )
	    { stroff= 0;	}
	}
    else{
	bi= docPrevParagraph( bi );

	if  ( ! bi )
	    { return -1;	}

	stroff= bi->biParaStrlen;
	}

    docFindLineAndParticule( bi, stroff, bp, lastOne );
    return 0;
    }


int docPrevLine(	TextParticule **	pTp,
			TextLine **		pTl,
			BufferPosition *	bp	)
    {
    TextLine *		tl;
    TextParticule *	tp;
    BufferItem *	bi= bp->bpBi;

    int			part= bp->bpParticule;
    int			line= bp->bpLine- 1;

    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA		&&
	      line >= 0				)
	    {
	    tl= bi->biParaLines+ line;
	    tp= bi->biParaParticules+ part;

	    while( part > tl->tlFirstParticule		&&
		  part > 0				)
		{ tp--; part--;	}
	    
	    bp->bpBi= bi;
	    bp->bpStroff= tl->tlStroff;
	    bp->bpParticule= part;
	    bp->bpLine= line;

	    if  ( pTp )
		{ *pTp= tp;	}
	    if  ( pTl )
		{ *pTl= tl;	}

	    return 0;
	    }

	bi= docPrevParagraph( bi );
	if  ( bi )
	    {
	    part= bi->biParaParticuleCount- 1;
	    line= bi->biParaLineCount- 1;
	    }
	}

    return -1;
    }

int docNextLine(	TextParticule **	pTp,
			TextLine **		pTl,
			BufferPosition *	bp	)
    {
    TextLine *		tl;
    TextParticule *	tp;
    BufferItem *	bi= bp->bpBi;

    int			part= bp->bpParticule;
    int			line= bp->bpLine+ 1;

    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA		&&
	      line < bi->biParaLineCount	)
	    {
	    tl= bi->biParaLines+ line;
	    tp= bi->biParaParticules+ part;

	    while( part < tl->tlFirstParticule		&&
		  part < bi->biParaParticuleCount	)
		{ tp++; part++;	}
	    
	    bp->bpBi= bi;
	    bp->bpStroff= tl->tlStroff;
	    bp->bpParticule= part;
	    bp->bpLine= line;
	    *pTp= tp;
	    *pTl= tl;

	    return 0;
	    }

	bi= docNextParagraph( bi );
	part= 0; line= 0;
	}

    return -1;
    }

/************************************************************************/
/*									*/
/*  Compare two positions in a document.				*/
/*									*/
/*  Comparison is in terms of the reading order in the document.	*/
/*									*/
/************************************************************************/

int docCompareItemPositions(	const BufferItem *	bi1,
				const BufferItem *	bi2 )
    {
    while( bi1->biParent && bi1->biLevel > bi2->biLevel )
	{ bi1= bi1->biParent;	}

    while( bi2->biParent && bi2->biLevel > bi1->biLevel )
	{ bi2= bi2->biParent;	}

    if  ( bi1 == bi2 )
	{ return 0;	}

    while( bi1->biParent			&&
	   bi2->biParent			&&
	   ( bi1->biParent != bi2->biParent )	)
	{ bi1= bi1->biParent; bi2= bi2->biParent; }

    if  ( bi1->biParent && bi1->biParent == bi2->biParent )
	{
	if  ( bi1->biNumberInParent > bi2->biNumberInParent )
	    { return  1;	}

	if  ( bi1->biNumberInParent < bi2->biNumberInParent )
	    { return -1;	}

	return 0;
	}

    XXDEB(bi1->biParent,bi2->biParent);

    return 0;
    }

int docComparePositions(	const BufferPosition *	bp1,
				const BufferPosition *	bp2,
				int			mindLine )
    {
    const BufferItem *	bi1= bp1->bpBi;
    const BufferItem *	bi2= bp2->bpBi;

    if  ( bi1 == bi2 )
	{
	if  ( bp1->bpStroff > bp2->bpStroff )
	    { return  1;	}

	if  ( bp1->bpStroff < bp2->bpStroff )
	    { return -1;	}

	if  ( mindLine )
	    {
	    if  ( bp1->bpLine > bp2->bpLine )
		{ return  1;	}

	    if  ( bp1->bpLine < bp2->bpLine )
		{ return -1;	}
	    }

	return 0;
	}

    return docCompareItemPositions( bi1, bi2 );
    }

/************************************************************************/
/*									*/
/*  Translate the result of a search in a paragraph to a selection.	*/
/*									*/
/************************************************************************/

void docSetParaSelection(	BufferSelection *	bs,
				BufferItem *		bi,
				int			direction,
				int			stroff,
				int			length	)
    {
    docFindLineAndParticule( bi, stroff, &(bs->bsBegin), 1 );
    docFindLineAndParticule( bi, stroff+ length, &(bs->bsEnd), 0 );

    bs->bsDirection= direction;

    bs->bsCol0= bs->bsCol1= -1;

    if  ( direction >= 0 )
	{ bs->bsAnchor= bs->bsBegin;	}
    else{ bs->bsAnchor= bs->bsEnd;	}

    return;
    }

/************************************************************************/
/*									*/
/*  Find the range of rows for a 'table'				*/
/*									*/
/************************************************************************/

int docDelimitTable(	BufferItem *		paraBi,
			BufferItem **		pSectBi,
			int *			pCol,
			int *			pRow0,
			int *			pRow,
			int *			pRow1 )
    {
    BufferItem *	rowBi;
    BufferItem *	sectBi;
    int			col;

    int			row;
    int			row0;
    int			row1;

    if  ( paraBi->biLevel != DOClevPARA		||
	  ! paraBi->biParaInTable		)
	{ /* LLDEB(paraBi->biLevel,paraBi->biParaInTable); */ return -1; }

    rowBi= paraBi->biParent->biParent;
    sectBi= rowBi->biParent;
    col= paraBi->biParent->biNumberInParent;

    row= rowBi->biNumberInParent;
    row0= rowBi->biNumberInParent;
    row1= rowBi->biNumberInParent;

    while( row0 > 0							&&
	   docAlignedColumns( &(rowBi->biRowProperties),
		&(sectBi->biChildren[row0-1]->biRowProperties) )	)
	{ row0--;	}

    while( row1 < sectBi->biChildCount- 1				&&
	   docAlignedColumns( &(rowBi->biRowProperties),
		&(sectBi->biChildren[row1+1]->biRowProperties) )	)
	{ row1++;	}

    *pSectBi= sectBi; *pCol= col;
    *pRow0= row0; *pRow= row; *pRow1= row1;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Are we in a something.						*/
/*									*/
/************************************************************************/

int docParticuleInField(	BufferItem *	bi,
				int		part )
    {
    TextParticule *	tp;

    part--;

    tp= bi->biParaParticules+ part;
    while( part >= 0 )
	{
	if  ( tp->tpKind == DOCkindFIELDEND )
	    { return 0;	}
	if  ( tp->tpKind == DOCkindFIELDSTART )
	    { return 1;	}

	tp--; part--;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Get buffer positions for a text line.				*/
/*									*/
/************************************************************************/

void docLinePositions(	BufferPosition *	bpLineBegin,
			BufferPosition *	bpLineEnd,
			const BufferItem *	bi,
			int			line )
    {
    const TextLine *	tl= bi->biParaLines+ line;

    docInitPosition( bpLineBegin );
    docInitPosition( bpLineEnd );

    bpLineBegin->bpBi= bpLineEnd->bpBi= (BufferItem *)bi;
    bpLineBegin->bpLine= bpLineEnd->bpLine= line;
    bpLineBegin->bpParticule= tl->tlFirstParticule;
    bpLineEnd->bpParticule= tl->tlFirstParticule+ tl->tlParticuleCount- 1;
    bpLineBegin->bpStroff= tl->tlStroff;
    bpLineEnd->bpStroff= tl->tlStroff+ tl->tlStrlen;

    return;
    }

/************************************************************************/
/*									*/
/*  Determine what header/footer applies for a certain page.		*/
/*									*/
/************************************************************************/

int docWhatPageHeader(	HeaderFooter **			pHf,
			BufferItem *			sectBi,
			int				page,
			const DocumentProperties *	dp )
    {
    const SectionProperties *	sp= &(sectBi->biSectProperties);

    if  ( page == sectBi->biTopPosition.lpPage && sp->spHasTitlePage )
	{
	*pHf= &(sectBi->biSectFirstPageHeader);
	return DOCinFIRST_HEADER;
	}

    if  ( ! dp->dpHasFacingPages )
	{
	*pHf= &(sectBi->biSectHeader);
	return DOCinSECT_HEADER;
	}

    if  ( page % 2 )
	{
	*pHf= &(sectBi->biSectLeftPageHeader);
	return DOCinLEFT_HEADER;
	}
    else{
	*pHf= &(sectBi->biSectRightPageHeader);
	return DOCinRIGHT_HEADER;
	}
    }

int docWhatPageFooter(	HeaderFooter **			pHf,
			BufferItem *			sectBi,
			int				page,
			const DocumentProperties *	dp )
    {
    const SectionProperties *	sp= &(sectBi->biSectProperties);

    if  ( page == sectBi->biTopPosition.lpPage && sp->spHasTitlePage )
	{
	*pHf= &(sectBi->biSectFirstPageFooter);
	return DOCinFIRST_FOOTER;
	}

    if  ( ! dp->dpHasFacingPages )
	{
	*pHf= &(sectBi->biSectFooter);
	return DOCinSECT_FOOTER;
	}

    if  ( page % 2 )
	{
	*pHf= &(sectBi->biSectLeftPageFooter);
	return DOCinLEFT_FOOTER;
	}
    else{
	*pHf= &(sectBi->biSectRightPageFooter);
	return DOCinRIGHT_FOOTER;
	}
    }

/************************************************************************/
/*									*/
/*  Return a particular header or footer by scope.			*/
/*									*/
/************************************************************************/

static HeaderFooter *	docSectionHeaderFooter(	BufferItem *	sectBi,
						int		which )
    {
    switch( which )
	{
	case DOCinBODY:
	    LDEB(which); return (HeaderFooter *)0;

	case DOCinSECT_HEADER:
	    return &(sectBi->biSectHeader);

	case DOCinFIRST_HEADER:
	    return &(sectBi->biSectFirstPageHeader);

	case DOCinLEFT_HEADER:
	    return &(sectBi->biSectLeftPageHeader);

	case DOCinRIGHT_HEADER:
	    return &(sectBi->biSectRightPageHeader);

	case DOCinSECT_FOOTER:
	    return &(sectBi->biSectFooter);

	case DOCinFIRST_FOOTER:
	    return &(sectBi->biSectFirstPageFooter);

	case DOCinLEFT_FOOTER:
	    return &(sectBi->biSectLeftPageFooter);

	case DOCinRIGHT_FOOTER:
	    return &(sectBi->biSectRightPageFooter);

	default:
	    LDEB(which); return (HeaderFooter *)0;
	}
    }

/************************************************************************/
/*									*/
/*  What BufferItem is the common root of the selection.		*/
/*									*/
/************************************************************************/

BufferItem *	docGetSelectionRoot(	BufferDocument *	bd,
					int			nearest,
					const SelectionScope *	ss,
					const BufferSelection *	bs )
    {
    BufferItem *	docBi= &(bd->bdItem);
    BufferItem *	sectBi= (BufferItem *)0;

    BufferItem *	bi1= bs->bsBegin.bpBi;
    BufferItem *	bi2= bs->bsEnd.bpBi;

    if  ( nearest )
	{
	while( bi1->biParent && bi1->biLevel > bi2->biLevel )
	    { bi1= bi1->biParent;	}

	while( bi2->biParent && bi2->biLevel > bi1->biLevel )
	    { bi2= bi2->biParent;	}

	if  ( bi1 == bi2 )
	    { return bi1;	}

	while( bi1->biParent			&&
	       bi2->biParent			&&
	       ( bi1->biParent != bi2->biParent )	)
	    { bi1= bi1->biParent; bi2= bi2->biParent; }

	if  ( bi1->biParent && bi1->biParent == bi2->biParent )
	    { return bi1->biParent;	}
	}

    if  ( ss->ssHeaderFooterSection >= 0 )
	{ sectBi= docBi->biChildren[ss->ssHeaderFooterSection]; }

    if  ( ss->ssInHeaderFooter == DOCinBODY )
	{ return docBi;	}
    else{
	HeaderFooter *	hf;

	hf= docSectionHeaderFooter( sectBi, ss->ssInHeaderFooter );
	if  ( ! hf )
	    { LXDEB(ss->ssInHeaderFooter,hf); return (BufferItem *)0;	}

	return hf->hfItem;
	}
    }

/************************************************************************/
/*									*/
/*  Return the Header/Footer corresponding to a certain Selection	*/
/*  Scope.								*/
/*									*/
/************************************************************************/

int docGetHeaderFooter(		HeaderFooter **		pHf,
				BufferItem **		pSectBi,
				const SelectionScope *	ss,
				const BufferSelection *	bs,
				const BufferDocument *	bd,
				int			which )
    {
    HeaderFooter *	hf;
    BufferItem *	sectBi;

    if  ( ss->ssInHeaderFooter == DOCinBODY )
	{
	sectBi= bs->bsBegin.bpBi;
	while( sectBi && sectBi->biLevel > DOClevSECT )
	    { sectBi= sectBi->biParent;	}

	if  ( ! sectBi || sectBi->biLevel != DOClevSECT )
	    { XDEB(sectBi); return -1;	}
	}
    else{
	const BufferItem *	docBi= &(bd->bdItem);

	if  ( ss->ssHeaderFooterSection < 0			||
	      ss->ssHeaderFooterSection >= docBi->biChildCount	)
	    {
	    LLDEB(ss->ssHeaderFooterSection,docBi->biChildCount);
	    return -1;
	    }

	sectBi= docBi->biChildren[ss->ssHeaderFooterSection];
	if  ( ! sectBi || sectBi->biLevel != DOClevSECT )
	    { XDEB(sectBi); return -1;	}
	}

    hf= docSectionHeaderFooter( sectBi, which );
    if  ( ! hf )
	{ XDEB(hf); return -1;	}

    *pHf= hf; *pSectBi= sectBi; return 0;
    }

/************************************************************************/
/*									*/
/*  Translate a TableRectangle to a BufferSelection.			*/
/*									*/
/************************************************************************/

int docTableRectangleSelection(	BufferSelection *	bs,
				BufferItem **		pSectBi,
				BufferDocument *	bd,
				const SelectionScope *	ss,
				const TableRectangle *	tr )
    {
    BufferSelection	bsNew;

    BufferItem *	docBi= &(bd->bdItem);
    BufferItem *	sectBi;
    BufferItem *	rowBi;
    BufferItem *	cellBi;

    docInitSelection( &bsNew );

    /*******/

    if  ( ss->ssInHeaderFooter == DOCinBODY )
	{
	if  ( tr->trSectionNumber < 0			||
	      tr->trSectionNumber >= docBi->biChildCount	)
	    { LLDEB(tr->trSectionNumber,docBi->biChildCount); return -1; }

	sectBi= docBi->biChildren[tr->trSectionNumber];
	}
    else{
	HeaderFooter *	hf;

	if  ( ss->ssHeaderFooterSection < 0			||
	      ss->ssHeaderFooterSection >= docBi->biChildCount	)
	    { LLDEB(ss->ssHeaderFooterSection,docBi->biChildCount); return -1; }

	sectBi= docBi->biChildren[ss->ssHeaderFooterSection];

	hf= docSectionHeaderFooter( sectBi, ss->ssInHeaderFooter );
	if  ( ! hf )
	    { XDEB(hf); return -1;	}
	if  ( ! hf->hfItem )
	    { XDEB(hf->hfItem); return -1;	}

	sectBi= hf->hfItem;
	}

    /*******/

    if  ( tr->trRow0 < 0			||
	  tr->trRow0 >= sectBi->biChildCount	)
	{ LLDEB(tr->trRow0,sectBi->biChildCount); return -1;	}
    rowBi= sectBi->biChildren[tr->trRow0];

    if  ( tr->trCol0 < 0			||
	  tr->trCol0 >= rowBi->biChildCount	)
	{ LLDEB(tr->trCol0,rowBi->biChildCount); return -1;	}
    cellBi= rowBi->biChildren[tr->trCol0];

    if  ( docFirstPosition( cellBi, &bsNew.bsBegin ) )
	{ LDEB(0); return -1;	}

    /*******/

    if  ( tr->trRow1 < 0			||
	  tr->trRow1 >= sectBi->biChildCount	)
	{ LLDEB(tr->trRow0,sectBi->biChildCount); return -1;	}
    rowBi= sectBi->biChildren[tr->trRow1];

    if  ( tr->trCol1 < 0			||
	  tr->trCol1 >= rowBi->biChildCount	)
	{ LLDEB(tr->trCol1,rowBi->biChildCount); return -1;	}
    cellBi= rowBi->biChildren[tr->trCol1];

    if  ( docLastPosition( cellBi, &bsNew.bsEnd ) )
	{ LDEB(0); return -1;	}

    /*******/

    bsNew.bsCol0= tr->trCol0; bsNew.bsCol1= tr->trCol1;
    bsNew.bsDirection= 1; bsNew.bsAnchor= bsNew.bsBegin;

    if  ( tr->trRow1 < tr->trRow0					||
	  ( tr->trRow1 == tr->trRow0 && tr->trCol1 < tr->trCol0 )	)
	{ bsNew.bsDirection= -1; bsNew.bsAnchor= bsNew.bsEnd; }


    *bs= bsNew;
    *pSectBi= sectBi;

    return 0;
    }

