/************************************************************************/
/*									*/
/*  Save a BufferDocument into an RTF file.				*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appSystem.h>

#   include	<bitmap.h>

#   include	<appDebugon.h>

#   include	<charnames.h>
#   include	<psFont.h>
#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Forward declaration..						*/
/*									*/
/************************************************************************/

static int docRtfSaveItem(	SimpleOutputStream *	sos,
				const BufferItem *	bi,
				const BufferDocument *	bd,
				const BufferSelection *	bs,
				RtfWritingContext *	rwc );

/************************************************************************/
/*									*/
/*  Save a tag with an argument.					*/
/*									*/
/************************************************************************/

static void docRtfWriteArgTag(	const char *		tag,
				int			arg,
				SimpleOutputStream *	sos )
    {
    char	scratch[20];

    sioOutPutString( tag, sos );

    sprintf( scratch, "%d", arg );
    sioOutPutString( scratch, sos );

    return;
    }

static void docRtfEscapeString(	const unsigned char *	s,
				const unsigned char *	outputMapping,
				int *			pCol,
				int			n,
				SimpleOutputStream *	sos )
    {
    while( n && *s )
	{
	int		c= *s;

	if  ( outputMapping )
	    { c= outputMapping[c];	}

	switch( c )
	    {
	    case '{': case '\\': case '}':
		sioOutPutCharacter( '\\', sos );
		sioOutPutCharacter( c, sos );
		*pCol += 2;
		break;
	    default:
		if  ( c > 127 )
		    {
		    static char	hexdigits[]= "0123456789abcdef";

		    sioOutPutCharacter( '\\', sos );
		    sioOutPutCharacter( '\'', sos );
		    sioOutPutCharacter( hexdigits[ ( c >> 4 ) & 0x0f ], sos );
		    sioOutPutCharacter( hexdigits[ ( c >> 0 ) & 0x0f ], sos );
		    *pCol += 4;
		    }
		else{
		    sioOutPutCharacter( c, sos );
		    *pCol += 1;
		    }
		break;
	    }

	n--; s++;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Save border definition.						*/
/*									*/
/************************************************************************/
static void docRtfSaveBorder(	char *				tag,
				const BorderProperties *	bp,
				SimpleOutputStream *		sos )
    {
    if  ( ! bp->bpIsSet )
	{ return;	}

    sioOutPutString( tag, sos );

    switch( bp->bpThickness )
	{
	case 0:
	    break;
	case 1:	sioOutPutString( "\\brdrs", sos );	break;
	case 2:	sioOutPutString( "\\brdrth", sos );	break;
	default:
	    LDEB(bp->bpThickness);
	    break;
	}

    switch( bp->bpStyle )
	{
	case DOCbsSH:		sioOutPutString( "\\brdrsh", sos ); break;
	case DOCbsDB:		sioOutPutString( "\\brdrdb", sos ); break;
	case DOCbsDOT:		sioOutPutString( "\\brdrdot", sos ); break;
	case DOCbsDASH:		sioOutPutString( "\\brdrdash", sos ); break;
	case DOCbsHAIR:		sioOutPutString( "\\brdrhair", sos ); break;
	case DOCbsDASHSM:	sioOutPutString( "\\brdrdashsm", sos ); break;
	case DOCbsDASHD:	sioOutPutString( "\\brdrdashd", sos ); break;
	case DOCbsDASHDD:	sioOutPutString( "\\brdrdashdd", sos ); break;
	case DOCbsTRIPLE:	sioOutPutString( "\\brdrtriple", sos ); break;
	case DOCbsTNTHSG:	sioOutPutString( "\\brdrtnthsg", sos ); break;
	case DOCbsTHTNSG:	sioOutPutString( "\\brdrthtnsg", sos ); break;
	case DOCbsTNTHTNSG:	sioOutPutString( "\\brdrtnthtnsg", sos ); break;
	case DOCbsTNTHMG:	sioOutPutString( "\\brdrtnthmg", sos ); break;
	case DOCbsTHTNMG:	sioOutPutString( "\\brdrthtnmg", sos ); break;
	case DOCbsTNTHTNMG:	sioOutPutString( "\\brdrtnthtnmg", sos ); break;
	case DOCbsTNTHLG:	sioOutPutString( "\\brdrtnthlg", sos ); break;
	case DOCbsTHTNLG:	sioOutPutString( "\\brdrthtnlg", sos ); break;
	case DOCbsTNTHTNLG:	sioOutPutString( "\\brdrtnthtnlg", sos ); break;
	case DOCbsWAVY:		sioOutPutString( "\\brdrwavy", sos ); break;
	case DOCbsWAVYDB:	sioOutPutString( "\\brdrwavydb", sos ); break;
	case DOCbsDASHDOTSTR:	sioOutPutString( "\\brdrdashdotstr", sos );
				break;
	case DOCbsEMBOSS:	sioOutPutString( "\\brdremboss", sos ); break;
	case DOCbsENGRAVE:	sioOutPutString( "\\brdrengrave", sos ); break;
	default:
	    LDEB(bp->bpStyle);
	    break;
	}

    if  ( bp->bpWidthTwips != 0 )
	{ docRtfWriteArgTag( "\\brdrw", bp->bpWidthTwips, sos );	}

    if  ( bp->bpColor != 0 )
	{ docRtfWriteArgTag( "\\brdrcf", bp->bpColor, sos );	}

    if  ( bp->bpSpacingTwips != 0 )
	{ docRtfWriteArgTag( "\\brsp", bp->bpSpacingTwips, sos );	}

    sioOutPutString( "\r\n", sos );

    return;
    }

/************************************************************************/
/*									*/
/*  Save possible attribute changes.					*/
/*									*/
/************************************************************************/
static int docRtfSaveAttributeNeg(	SimpleOutputStream *	sos,
					int *			pAttChange,
					int *			pCol,
					TextAttribute		taOld,
					TextAttribute		taNew	)
    {
    if  ( taOld.taSuperSub != DOCfontREGULAR	&&
	  taNew.taSuperSub != taOld.taSuperSub	)
	{
	sioOutPutString( "\\nosupersub", sos );
	*pCol += 11; *pAttChange= 1;
	}

    if  ( ! taNew.taHasStrikeThrough && taOld.taHasStrikeThrough )
	{
	sioOutPutString( "\\strike0", sos );
	*pCol += 8; *pAttChange= 1;
	}

    if  ( ! taNew.taIsUnderlined && taOld.taIsUnderlined )
	{
	sioOutPutString( "\\ul0", sos );
	*pCol += 4; *pAttChange= 1;
	}

    if  ( ! taNew.taFontIsSlanted && taOld.taFontIsSlanted )
	{
	sioOutPutString( "\\i0", sos );
	*pCol += 3; *pAttChange= 1;
	}

    if  ( ! taNew.taFontIsBold && taOld.taFontIsBold )
	{
	sioOutPutString( "\\b0", sos );
	*pCol += 3; *pAttChange= 1;
	}

    return 0;
    }

static int docRtfSaveAttributePos(	SimpleOutputStream *	sos,
					int *			pAttChange,
					int *			pCol,
					TextAttribute		taOld,
					TextAttribute		taNew	)
    {
    if  ( taNew.taFontNumber != taOld.taFontNumber )
	{
	docRtfWriteArgTag( "\\f", taNew.taFontNumber, sos );
	*pCol += 3; *pAttChange= 1;
	}

    if  ( taNew.taFontSizeHalfPoints != taOld.taFontSizeHalfPoints )
	{
	docRtfWriteArgTag( "\\fs", taNew.taFontSizeHalfPoints, sos );
	*pCol += 4; *pAttChange= 1;
	}

    if  ( taNew.taFontIsBold && ! taOld.taFontIsBold )
	{
	sioOutPutString( "\\b", sos );
	*pCol += 2; *pAttChange= 1;
	}

    if  ( taNew.taFontIsSlanted && ! taOld.taFontIsSlanted )
	{
	sioOutPutString( "\\i", sos );
	*pCol += 2; *pAttChange= 1;
	}

    if  ( taNew.taIsUnderlined && ! taOld.taIsUnderlined )
	{
	sioOutPutString( "\\ul", sos );
	*pCol += 3; *pAttChange= 1;
	}

    if  ( taNew.taHasStrikeThrough && ! taOld.taHasStrikeThrough )
	{
	sioOutPutString( "\\strike", sos );
	*pCol += 7; *pAttChange= 1;
	}

    if  ( taNew.taSuperSub != DOCfontREGULAR		&&
	  taNew.taSuperSub != taOld.taSuperSub	)
	{
	switch( taNew.taSuperSub )
	    {
	    case DOCfontSUPERSCRIPT:
		sioOutPutString( "\\super", sos );
		*pCol += 6; *pAttChange= 1;
		break;
	    case DOCfontSUBSCRIPT:
		sioOutPutString( "\\sub", sos );
		*pCol += 4; *pAttChange= 1;
		break;
	    default:
		LDEB(taNew.taSuperSub); break;
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a paragraph in RTF format.					*/
/*									*/
/*  Column counting is approximate as it is only for cosmetic reasons.	*/
/*									*/
/************************************************************************/

static void docRtfParaSaveProperties( SimpleOutputStream *	sos,
				int				saveIntbl,
				int *				pPropertyChange,
				int *				pCol,
				const ParagraphProperties *	newPP,
				const ParagraphProperties *	prevPP )
    {
    int			startFromDefault= 0;
    int			i;

    ParagraphProperties	ppp;

    docInitParagraphProperties( &ppp );
    docCopyParagraphProperties( &ppp, prevPP );

    if  ( newPP->ppTabCount != ppp.ppTabCount )
	{ startFromDefault= 1;	}
    else{
	for ( i= 0; i < newPP->ppTabCount; i++ )
	    {
	    if  ( newPP->ppTabStops[i].tsTwips	!=
		  ppp.ppTabStops[i].tsTwips	)
		{ startFromDefault= 1; break; }
	    }
	}

    if  ( ! docEqualBorder( &(ppp.ppTopBorder),
					    &(newPP->ppTopBorder) )	||
	  ! docEqualBorder( &(ppp.ppBottomBorder),
					    &(newPP->ppBottomBorder) )	)
	{ startFromDefault= 1;	}

    if  ( ppp.ppLineSpacingTwips != newPP->ppLineSpacingTwips		||
	  ppp.ppLineSpacingIsMultiple != newPP->ppLineSpacingIsMultiple )
	{ startFromDefault= 1;	}

    if  ( startFromDefault )
	{
	docCleanParagraphProperties( &ppp );
	docInitParagraphProperties( &ppp );

	/*  1  */
	sioOutPutString( "\\pard", sos );
	*pCol += 5;

	if  ( saveIntbl && newPP->ppInTable )
	    { sioOutPutString( "\\intbl", sos ); *pCol += 6;	}

	*pPropertyChange= 1;
	}

    if  ( newPP->ppStartsOnNewPage != ppp.ppStartsOnNewPage )
	{
	if  ( newPP->ppStartsOnNewPage )
	    { sioOutPutString( "\\pagebb",  sos ); *pCol += 7; }
	else{ sioOutPutString( "\\pagebb0", sos ); *pCol += 8; }
	}

    if  ( *pCol >= 72 )
	{ sioOutPutString( "\r\n", sos ); *pCol= 0;	}

    if  ( newPP->ppFirstIndentTwips != ppp.ppFirstIndentTwips )
	{
	docRtfWriteArgTag( "\\fi", newPP->ppFirstIndentTwips, sos );
	*pCol += 7; *pPropertyChange= 1;
	}
    if  ( newPP->ppLeftIndentTwips != ppp.ppLeftIndentTwips )
	{
	docRtfWriteArgTag( "\\li", newPP->ppLeftIndentTwips, sos );
	*pCol += 7; *pPropertyChange= 1;
	}
    if  ( newPP->ppRightIndentTwips != ppp.ppRightIndentTwips )
	{
	docRtfWriteArgTag( "\\ri", newPP->ppRightIndentTwips, sos );
	*pCol += 7; *pPropertyChange= 1;
	}
    if  ( newPP->ppSpaceBeforeTwips != ppp.ppSpaceBeforeTwips )
	{
	docRtfWriteArgTag( "\\sb", newPP->ppSpaceBeforeTwips, sos );
	*pCol += 7; *pPropertyChange= 1;
	}
    if  ( newPP->ppSpaceAfterTwips != ppp.ppSpaceAfterTwips )
	{
	docRtfWriteArgTag( "\\sa", newPP->ppSpaceAfterTwips, sos );
	*pCol += 7; *pPropertyChange= 1;
	}
    if  ( newPP->ppLineSpacingTwips != ppp.ppLineSpacingTwips )
	{
	docRtfWriteArgTag( "\\sl", newPP->ppSpaceAfterTwips, sos );
	docRtfWriteArgTag( "\\slmult", newPP->ppLineSpacingIsMultiple, sos );
	*pCol += 14; *pPropertyChange= 1;
	}

    if  ( newPP->ppAlignment != ppp.ppAlignment )
	{
	switch( newPP->ppAlignment )
	    {
	    case DOCiaLEFT:	sioOutPutString( "\\ql", sos ); break;
	    case DOCiaRIGHT:	sioOutPutString( "\\qr", sos ); break;
	    case DOCiaCENTERED:	sioOutPutString( "\\qc", sos ); break;
	    case DOCiaJUSTIFIED:sioOutPutString( "\\qj", sos ); break;
	    default:
		LDEB(newPP->ppAlignment); break;
	    }

	*pCol += 3; *pPropertyChange= 1;
	}

    if  ( startFromDefault )
	{
	TabStop *	ts= newPP->ppTabStops;

	for ( i= 0; i < newPP->ppTabCount; ts++, i++ )
	    {
	    if  ( *pCol >= 65 )
		{ sioOutPutString( "\r\n", sos ); *pCol= 0;	}

	    switch( ts->tsKind )
		{
		case DOCtkLEFT:
		    break;
		case DOCtkRIGHT:
		    sioOutPutString( "\\tqr", sos ); *pCol += 4;
		    break;
		case DOCtkCENTRE:
		    sioOutPutString( "\\tqc", sos ); *pCol += 4;
		    break;
		case DOCtkDECIMAL:
		    sioOutPutString( "\\tqdec", sos ); *pCol += 6;
		    break;
		default:
		    LDEB(ts->tsKind); break;
		}

	    switch( ts->tsLeader )
		{
		case DOCtlNONE:
		    break;
		case DOCtlDOTS:
		    sioOutPutString( "\\tldot", sos ); *pCol += 6;
		    break;
		case DOCtlHYPH:
		    sioOutPutString( "\\tlhyph", sos ); *pCol += 7;
		    break;
		case DOCtlUNDERLINE:
		    sioOutPutString( "\\tlul", sos ); *pCol += 5;
		    break;
		case DOCtlTHICK:
		    sioOutPutString( "\\tlth", sos ); *pCol += 5;
		    break;
		case DOCtlEQUAL:
		    sioOutPutString( "\\tleq", sos ); *pCol += 5;
		    break;
		default:
		    LDEB(ts->tsLeader); break;
		}

	    docRtfWriteArgTag( "\\tx", ts->tsTwips, sos );
	    *pCol += 7;
	    }
	}

    if  ( ! docEqualBorder( &(ppp.ppTopBorder),
					    &(newPP->ppTopBorder) )	)
	{ docRtfSaveBorder( "\\brdrt", &(newPP->ppTopBorder), sos ); }

    if  ( ! docEqualBorder( &(ppp.ppBottomBorder),
					    &(newPP->ppBottomBorder) )	)
	{ docRtfSaveBorder( "\\brdrb", &(newPP->ppBottomBorder), sos ); }

    docCleanParagraphProperties( &ppp );

    return;
    }

int docRtfSaveRuler(	SimpleOutputStream *		sos,
			const ParagraphProperties *	pp )
    {
    int				col= 0;
    int				precedingTags= 1;
    int				noIntbl= 1;

    ParagraphProperties		refPP;

    docInitParagraphProperties( &refPP );

    sioOutPutString( "{\\ruler\\pard", sos );

    docRtfParaSaveProperties( sos, noIntbl, &precedingTags, &col, pp, &refPP );

    sioOutPutString( "}", sos );

    docCleanParagraphProperties( &refPP );

    return 0;
    }

static int docRtfSaveObjectData(	const ObjectData *		od,
					SimpleOutputStream *		sos )
    {
    const unsigned char *	s;
    int				i;

    s= od->odBytes;
    for ( i= 0; i < od->odSize; s++, i++ )
	{
	if  ( ! ( i % 78 ) )
	    { sioOutPutString( "\r\n", sos ); }

	sioOutPutCharacter( *s, sos );
	}

    return 0;
    }


static int docRtfSavePictureTags(	InsertedObject *		io,
					SimpleOutputStream *		sos )
    {
    int		xExt= io->ioXExtent;
    int		yExt= io->ioYExtent;

    if  ( xExt == 0 )
	{ xExt= (int)( 1000.0* io->ioTwipsWide )/ ( 20* POINTS_PER_CM ); }
    if  ( yExt == 0 )
	{ yExt= (int)( 1000.0* io->ioTwipsHigh )/ ( 20* POINTS_PER_CM ); }

    docRtfWriteArgTag( "\\picw", xExt, sos );
    docRtfWriteArgTag( "\\pich", yExt, sos );

    if  ( io->ioScaleX != 100 )
	{ docRtfWriteArgTag( "\\picscalex", io->ioScaleX, sos ); }
    if  ( io->ioScaleY != 100 )
	{ docRtfWriteArgTag( "\\picscaley", io->ioScaleY, sos ); }

    docRtfWriteArgTag( "\\picwgoal", io->ioTwipsWide, sos );
    docRtfWriteArgTag( "\\pichgoal", io->ioTwipsHigh, sos );

    if  ( io->ioTopCropTwips != 0 )
	{ docRtfWriteArgTag( "\\piccropt", io->ioTopCropTwips, sos );	}
    if  ( io->ioBottomCropTwips != 0 )
	{ docRtfWriteArgTag( "\\piccropb", io->ioBottomCropTwips, sos ); }
    if  ( io->ioLeftCropTwips != 0 )
	{ docRtfWriteArgTag( "\\piccropl", io->ioLeftCropTwips, sos );	}
    if  ( io->ioRightCropTwips != 0 )
	{ docRtfWriteArgTag( "\\piccropr", io->ioRightCropTwips, sos );	}

    if  ( io->ioBliptag == 0 )
	{ io->ioBliptag= appGetTimestamp(); 	}

    if  ( io->ioBliptag != 0 )
	{ docRtfWriteArgTag( "\\bliptag", io->ioBliptag, sos ); }

    return 0;
    }

static int docRtfSaveObject(	SimpleOutputStream *		sos,
				const BufferItem *		bi,
				const TextParticule *		tp )
    {
    InsertedObject *	io= bi->biParaObjects+ tp->tpObjectNumber;


    switch( io->ioKind )
	{
	case DOCokPICTWMETAFILE:
	    sioOutPutString( "\r\n{\\pict\\wmetafile8", sos );

	    docRtfSavePictureTags( io, sos );

	    docRtfSaveObjectData( &io->ioObjectData, sos );

	    sioOutPutString( "\r\n}", sos );
	    return 0;

	case DOCokMACPICT:
	    sioOutPutString( "\r\n{\\pict\\macpict", sos );

	    docRtfSavePictureTags( io, sos );

	    docRtfSaveObjectData( &io->ioObjectData, sos );

	    sioOutPutString( "\r\n}", sos );
	    return 0;

	case DOCokPICTPNGBLIP:
	    sioOutPutString( "\r\n{\\pict\\pngblip", sos );

	    docRtfSavePictureTags( io, sos );

	    docRtfSaveObjectData( &io->ioObjectData, sos );

	    sioOutPutString( "\r\n}", sos );
	    return 0;

	case DOCokPICTJPEGBLIP:
	    sioOutPutString( "\r\n{\\pict\\jpegblip", sos );

	    docRtfSavePictureTags( io, sos );

	    docRtfSaveObjectData( &io->ioObjectData, sos );

	    sioOutPutString( "\r\n}", sos );
	    return 0;

	case DOCokPICTEMFBLIP:
	    sioOutPutString( "\r\n{\\pict\\emfblip", sos );

	    docRtfSavePictureTags( io, sos );

	    docRtfSaveObjectData( &io->ioObjectData, sos );

	    sioOutPutString( "\r\n}", sos );
	    return 0;

	case DOCokOLEOBJECT:
	    sioOutPutString( "\r\n{\\object\\objemb", sos );

	    if  ( io->ioObjectClass )
		{
		sioOutPutString( "\r\n{\\*\\objclass ", sos );
		sioOutPutString( (char *)io->ioObjectClass, sos );
		sioOutPutString( "}", sos );
		}

	    if  ( io->ioObjectName )
		{
		sioOutPutString( "\r\n{\\*\\objname ", sos );
		sioOutPutString( (char *)io->ioObjectName, sos );
		sioOutPutString( "}", sos );
		}

	    docRtfWriteArgTag( "\\objw", io->ioTwipsWide, sos );
	    docRtfWriteArgTag( "\\objh", io->ioTwipsHigh, sos );
	    if  ( io->ioScaleX != 100 )
		{ docRtfWriteArgTag( "\\objscalex", io->ioScaleX, sos ); }
	    if  ( io->ioScaleY != 100 )
		{ docRtfWriteArgTag( "\\objscaley", io->ioScaleY, sos ); }

	    sioOutPutString( "\r\n{\\*\\objdata ", sos );
	    docRtfSaveObjectData( &io->ioObjectData, sos );
	    sioOutPutString( "\r\n}", sos );

	    if  ( io->ioResultKind == DOCokPICTWMETAFILE )
		{
		sioOutPutString( "{\\result {\\pict\\wmetafile8", sos );

		docRtfSavePictureTags( io, sos );

		docRtfSaveObjectData( &io->ioResultData, sos );

		sioOutPutString( "\r\n}}", sos );
		}

	    sioOutPutString( "\r\n}", sos );

	    return 0;
	default:
	    LDEB(io->ioKind); return -1;
	}

    /*  Not reached ..
    return 0;
    */
    }

/************************************************************************/
/*									*/
/*  Reserve a number of columns in the output file.			*/
/*									*/
/************************************************************************/
static void docRtfReserveColumns(	SimpleOutputStream *	sos,
					int			cols,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    int				col= *pCol;

    if  ( col > 0 && col+ cols > 72 )
	{
	sioOutPutString( "\r\n", sos );
	rwc->rwcHasPrecedingTags= 0; col= 0;
	}

    *pCol= col; return;
    }

/************************************************************************/
/*									*/
/*  Switch text attributes.						*/
/*									*/
/************************************************************************/

static void docRtfSaveAttributes(	SimpleOutputStream *	sos,
					int			cols,
					TextAttribute		ta,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    int				col= *pCol;

    docRtfSaveAttributeNeg( sos, &(rwc->rwcHasPrecedingTags), &col,
						rwc->rwcTextAttribute, ta );

    docRtfReserveColumns( sos, cols, &col, rwc );

    docRtfSaveAttributePos( sos, &(rwc->rwcHasPrecedingTags), &col,
						rwc->rwcTextAttribute, ta );
    *pCol= col; return;
    }

/************************************************************************/
/*									*/
/*  Finish/Begin writing a hyperlink.					*/
/*									*/
/************************************************************************/

static void docRtfFinishFldrslt(	SimpleOutputStream *	sos,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    int				col= *pCol;

    sioOutPutString( "}}}", sos );
    col += 3;
    rwc->rwcHasPrecedingTags= 0;
    rwc->rwcInFldrslt--;
    rwc->rwcTextAttribute= rwc->rwcOutsideLinkAttribute;

    *pCol= col; return;
    }

static void docRtfStartField(		SimpleOutputStream *	sos,
					const DocumentField *	df,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    const unsigned char *	bytes= df->dfInstructions.odBytes;
    int				byteCount= df->dfInstructions.odSize;

    int				col= *pCol;

    docRtfReserveColumns( sos, 19, &col, rwc );

    rwc->rwcOutsideLinkAttribute= rwc->rwcTextAttribute;

    sioOutPutString( "{\\field{\\*\\fldinst{", sos );
    col += 19;

    docRtfReserveColumns( sos, byteCount, &col, rwc );

    while( byteCount > 1 && bytes[byteCount- 1] == ' ' )
	{ byteCount--;	}

    docRtfEscapeString( bytes, (const unsigned char *)0, &col, byteCount, sos );

    sioOutPutString( " }}{\\fldrslt{", sos ); col += 13;

    rwc->rwcInFldrslt++;

    *pCol= col; return;
    }

static void docRtfStartBookmark(	SimpleOutputStream *	sos,
					const DocumentField *	df,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    int				col= *pCol;

    const char *		markName= (char *)0;
    int				markSize;

    if  ( ! docFieldGetBookmark( df, &markName, &markSize ) )
	{
	sioOutPutString( "{\\*\\bkmkstart ", sos ); col += 14;
	docRtfEscapeString( (const unsigned char *)markName, 
						    (const unsigned char *)0,
						    &col, markSize, sos );
	sioOutPutString( "}", sos ); col += 1;
	}

    *pCol= col; return;
    }

static void docRtfFinishBookmark(	SimpleOutputStream *	sos,
					const DocumentField *	df,
					int *			pCol,
					RtfWritingContext *	rwc )
    {
    int				col= *pCol;

    const char *		markName= (char *)0;
    int				markSize;

    if  ( ! docFieldGetBookmark( df, &markName, &markSize ) )
	{
	sioOutPutString( "{\\*\\bkmkend ", sos ); col += 12;
	docRtfEscapeString( (const unsigned char *)markName, 
						    (const unsigned char *)0,
						    &col, markSize, sos );
	sioOutPutString( "}", sos ); col += 1;
	}

    *pCol= col; return;
    }

/************************************************************************/
/*									*/
/*  Save Cell properties.						*/
/*									*/
/************************************************************************/

static void docRtfSaveCellProperties(	const CellProperties *	cp,
					int			shiftLeft,
					SimpleOutputStream *	sos )
    {
    int		didAnything= 0;

    if  ( cp->cpLeftInMergedRow )
	{ sioOutPutString( "\\clmgf", sos ); didAnything= 1;	}
    if  ( cp->cpMergedWithLeft )
	{ sioOutPutString( "\\clmrg", sos ); didAnything= 1;	}
    if  ( cp->cpTopInMergedColumn )
	{ sioOutPutString( "\\clvmgf", sos ); didAnything= 1;	}
    if  ( cp->cpMergedWithAbove )
	{ sioOutPutString( "\\clvmrg", sos ); didAnything= 1;	}

    if  ( didAnything )
	{ sioOutPutString( "\r\n", sos );	}

    docRtfSaveBorder( "\\clbrdrt", &(cp->cpTopBorder), sos );
    docRtfSaveBorder( "\\clbrdrl", &(cp->cpLeftBorder), sos );
    docRtfSaveBorder( "\\clbrdrr", &(cp->cpRightBorder), sos );
    docRtfSaveBorder( "\\clbrdrb", &(cp->cpBottomBorder), sos );

    switch( cp->cpShadingPattern )
	{
	case DOCspSOLID:
	    break;

	case DOCspBGBDIAG:
	    sioOutPutString( "\\clbgbdiag", sos );	break;
	case DOCspBGCROSS:
	    sioOutPutString( "\\clbgcross", sos );	break;
	case DOCspBGDCROSS:
	    sioOutPutString( "\\clbgdcross", sos );	break;
	case DOCspBGDKBDIAG:
	    sioOutPutString( "\\clbgdkbdiag", sos );	break;
	case DOCspBGDKCROSS:
	    sioOutPutString( "\\clbgdkcross", sos );	break;
	case DOCspBGDKDCROSS:
	    sioOutPutString( "\\clbgdkdcross", sos );	break;
	case DOCspBGDKFDIAG:
	    sioOutPutString( "\\clbgdkfdiag", sos );	break;
	case DOCspBGDKHORIZ:
	    sioOutPutString( "\\clbgdkhor", sos );	break;
	case DOCspBGDKVERT:
	    sioOutPutString( "\\clbgdkvert", sos );	break;
	case DOCspBGFDIAG:
	    sioOutPutString( "\\clbgfdiag", sos );	break;
	case DOCspBGHORIZ:
	    sioOutPutString( "\\clbghoriz", sos );	break;
	case DOCspBGVERT:
	    sioOutPutString( "\\clbgvert", sos );	break;

	default:
	    LDEB(cp->cpShadingPattern);
	    break;
	}

    docRtfWriteArgTag( "\\cellx", cp->cpRightBoundaryTwips- shiftLeft, sos );
    sioOutPutString( "\r\n", sos );
    }

static void docRtfSaveRowProperties(	const RowProperties *	rp,
					int			col0,
					int			col1,
					SimpleOutputStream *	sos )
    {
    const CellProperties *	cp= rp->rpCells;

    int				col;
    int				shiftLeft= 0;

    sioOutPutString( "\\trowd", sos );

    if  ( rp->rpHalfGapWidthTwips != 0 )
	{ docRtfWriteArgTag( "\\trgaph", rp->rpHalfGapWidthTwips, sos ); }

    if  ( rp->rpLeftIndentTwips != 0 )
	{ docRtfWriteArgTag( "\\trleft", rp->rpLeftIndentTwips, sos ); }

    if  ( rp->rpHeightTwips != 0 )
	{ docRtfWriteArgTag( "\\trrh", rp->rpHeightTwips, sos ); }

    if  ( rp->rpIsTableHeader )
	{ sioOutPutString( "\\trhdr", sos ); }
    if  ( rp->rpKeepOnPage )
	{ sioOutPutString( "\\trkeep", sos ); }

    sioOutPutString( "\r\n", sos );

    docRtfSaveBorder( "\\trbrdrt", &(rp->rpTopBorder), sos );
    docRtfSaveBorder( "\\trbrdrl", &(rp->rpLeftBorder), sos );
    docRtfSaveBorder( "\\trbrdrr", &(rp->rpRightBorder), sos );
    docRtfSaveBorder( "\\trbrdrb", &(rp->rpBottomBorder), sos );

    if  ( rp->rpHasVerticalBorders )
	{ docRtfSaveBorder( "\\trbrdrv", &(rp->rpVerticalBorder), sos ); }
    if  ( rp->rpHasHorizontalBorders )
	{ docRtfSaveBorder( "\\trbrdrh", &(rp->rpHorizontalBorder), sos ); }

    if  ( col0 > 0 )
	{ shiftLeft= cp[col0-1].cpRightBoundaryTwips; }

    for ( col= 0; col < rp->rpCellCount; cp++, col++ )
	{
	if  ( ( col0 < 0 || col >= col0 )	&&
	      ( col1 < 0 || col <= col1 )	)
	    { docRtfSaveCellProperties( cp, shiftLeft, sos );	}
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Enter/Leave a table.						*/
/*									*/
/************************************************************************/

static int docRtfPushTable(	RtfWritingContext *	rwc,
				SimpleOutputStream *	sos,
				const BufferItem *	rowBi,
				const BufferSelection *	bs )
    {
    int			col0= -1;
    int			col1= -1;

    sioOutPutString( "\r\n{", sos );

    rwc->rwcInTable= 1;

    rwc->rwcOutsideTableTextAttribute= rwc->rwcTextAttribute;

    if  ( docCopyParagraphProperties(
			    &(rwc->rwcOutsideTableParagraphProperties),
			    &(rwc->rwcParagraphProperties) ) )
	{ LDEB(1); return -1;	}

    if  ( bs )
	{ col0= bs->bsCol0; col1= bs->bsCol1; 	}

    docRtfSaveRowProperties( &(rowBi->biRowProperties), col0, col1, sos );

    if  ( docCopyRowProperties( &(rwc->rwcRowProperties),
					&(rowBi->biRowProperties) ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

static int docRtfPopTable(	RtfWritingContext *	rwc,
				SimpleOutputStream *	sos )
    {
    sioOutPutString( "}\r\n", sos );

    rwc->rwcInTable= 0;

    docCleanRowProperties( &(rwc->rwcRowProperties) );
    docInitRowProperties( &(rwc->rwcRowProperties) );

    rwc->rwcTextAttribute= rwc->rwcOutsideTableTextAttribute;

    if  ( docCopyParagraphProperties(
			    &(rwc->rwcParagraphProperties),
			    &(rwc->rwcOutsideTableParagraphProperties) ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save an inidividual header, footer, or footnote.			*/
/*									*/
/************************************************************************/

static int docRtfSaveExternalItem(	SimpleOutputStream *	sos,
					const BufferItem *	bi,
					const BufferDocument *	bd,
					const char *		tag,
					RtfWritingContext *	rwc )
    {
    int			i;
    int			col= 0;

    int			savedInTable= rwc->rwcInTable;

    if  ( ! bi )
	{ return 0;	}

    if  ( bi->biLevel != DOClevSECT )
	{ SDEB(docLevelStr(bi->biLevel)); return -1;	}

    sioOutPutString( tag, sos );

    docCleanParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitTextAttribute( &(rwc->rwcTextAttribute) );

    rwc->rwcInTable= 0;

    sioOutPutString( "\\pard\\plain ", sos );

    for ( i= 0; i < bi->biChildCount; i++ )
	{
	if  ( docRtfSaveItem( sos, bi->biChildren[i], bd,
						(BufferSelection *)0, rwc ) )
	    { LDEB(i); return -1;	}
	}

    docCleanParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitTextAttribute( &(rwc->rwcTextAttribute) );

    while( rwc->rwcInFldrslt > 0 )
	{ docRtfFinishFldrslt( sos, &col, rwc );	}

    if  ( rwc->rwcInTable && docRtfPopTable( rwc, sos ) )
	{ LDEB(1);	}

    sioOutPutString( "}\r\n", sos );

    rwc->rwcInTable= savedInTable;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a paragraph.							*/
/*									*/
/*  1)  When we are saving a selection, and the selection is inside a	*/
/*	table cell, do not set the \intbl flag.				*/
/*  2)  To make WP 8 happy, always save 'intbl' for the first paragraph	*/
/*	in a table row.							*/
/*  3)  The 'intbl0' branch of this code is never activated because of	*/
/*	the pushing and popping of properties around tables.		*/
/*									*/
/************************************************************************/

static int docRtfSaveParaItem(	SimpleOutputStream *	sos,
				const BufferDocument *	bd,
				const BufferItem *	bi,
				const BufferSelection *	bs,
				RtfWritingContext *	rwc )
    {
    TextParticule *			tp;
    unsigned char *			s;

    int					col= 0;

    int					stroffUpto;
    int					part= 0;
    int					stroff= 0;

    int					saveIntbl;

    const DocumentField *		df;
    const FieldKindInformation *	fki;

    saveIntbl= ! bs || ! docSelectionInsideCell( bs );

    if  ( saveIntbl )
	{
	/*  2  */
	if  ( bi->biParaInTable != rwc->rwcParagraphProperties.ppInTable ||
	      ( bi->biParaInTable			&&
		bi->biNumberInParent == 0		&&
		bi->biParent				&&
		bi->biParent->biNumberInParent == 0	)		 )
	    {
	    if  ( bi->biParaInTable )
		{ sioOutPutString( "\\intbl",  sos ); col += 6; }
	    else{ sioOutPutString( "\\intbl0", sos ); col += 7; } /*  3  */
	    }
	}

    docRtfParaSaveProperties( sos, saveIntbl, &(rwc->rwcHasPrecedingTags), &col,
					&(bi->biParaProperties),
					&(rwc->rwcParagraphProperties) );

    /*  1  */
    docInitTextAttribute( &(rwc->rwcTextAttribute) );
    docInitTextAttribute( &(rwc->rwcOutsideLinkAttribute) );
    rwc->rwcTextAttribute.taFontSizeHalfPoints= 24;
    rwc->rwcOutsideLinkAttribute.taFontSizeHalfPoints= 24;

    /*  2  */
    sioOutPutString( "\\plain", sos ); col += 6;
    rwc->rwcHasPrecedingTags= 1;

    if  ( bi->biParaParticuleCount == 0 )
	{ LDEB(bi->biParaParticuleCount);	}

    if  ( bs && bs->bsBegin.bpBi == bi )
	{
	part= bs->bsBegin.bpParticule;
	stroff= bs->bsBegin.bpStroff;
	}

    tp= bi->biParaParticules+ part;
    s= bi->biParaString+ stroff;

    if  ( bs && bs->bsEnd.bpBi == bi )
	{ stroffUpto= bs->bsEnd.bpStroff;	}
    else{ stroffUpto= -1;			}

    for ( ; part < bi->biParaParticuleCount; part++, tp++ )
	{
	if  ( tp->tpTextAttribute.taFontNumber >= 0 )
	    {
	    docRtfSaveAttributes( sos, tp->tpStrlen,
					    tp->tpTextAttribute, &col, rwc );
	    rwc->rwcTextAttribute= tp->tpTextAttribute;
	    }

	if  ( rwc->rwcSaveAsLink )
	    {
	    docRtfStartField( sos, &(rwc->rwcSaveAsHyperlinkField), &col, rwc );

	    rwc->rwcSaveAsLink= 0;

	    if  ( rwc->rwcSaveAsLinkAsRef )
		{
		docRtfStartField( sos, &(rwc->rwcSaveAsRefField), &col, rwc );
		}
	    else{
		if  ( rwc->rwcSaveAsLinkAsPageref )
		    {
		    docRtfStartField( sos,
				    &(rwc->rwcSaveAsPagerefField), &col, rwc );
		    }
		}
	    }

	switch( tp->tpKind )
	    {
	    int		n;

	    case DOCkindTAB:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		sioOutPutString( "\\tab", sos ); s++; stroff++; col += 4;
		rwc->rwcHasPrecedingTags= 1;
		continue;

	    case DOCkindLINEBREAK:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		sioOutPutString( "\\line\r\n", sos ); col= 0;
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 1 */
		rwc->rwcHasPrecedingTags= 0;
		continue;

	    case DOCkindPAGEBREAK:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		sioOutPutString( "\\page\r\n", sos ); col= 0;
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 1 */
		rwc->rwcHasPrecedingTags= 0;
		continue;

	    case DOCkindTEXT:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		if  ( rwc->rwcHasPrecedingTags )
		    {
		    sioOutPutString( " ", sos ); col++;
		    rwc->rwcHasPrecedingTags= 0;
		    }

		if  ( stroffUpto >= 0				&&
		      tp->tpStroff+ tp->tpStrlen > stroffUpto	)
		    { n= stroffUpto- stroff;			}
		else{ n= tp->tpStroff+ tp->tpStrlen- stroff;	}

		docRtfEscapeString( s, rwc->rwcOutputMapping, &col, n, sos );
		stroff += n; s += n;

		continue;
	    case DOCkindOBJECT:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		if  ( docRtfSaveObject( sos, bi, tp ) )
		    { LDEB(tp->tpKind); }
		s += tp->tpStrlen; stroff += tp->tpStrlen;
		rwc->rwcHasPrecedingTags= 0; col= 1;
		continue;

	    case DOCkindFIELDSTART:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
		fki= DOC_FieldKinds+ df->dfKind;

		if  ( fki->fkiIsFieldInRtf		&&
		      fki->fkiLevel == DOClevTEXT	)
		    {
		    docRtfStartField( sos, df, &col, rwc );
		    rwc->rwcHasPrecedingTags= 0;
		    }

		if  ( df->dfKind == DOCfkBOOKMARK	&&
		      rwc->rwcSaveBookmarks		)
		    {
		    docRtfStartBookmark( sos, df, &col, rwc );
		    rwc->rwcHasPrecedingTags= 0;
		    }

		if  ( df->dfKind == DOCfkFOOTNOTE )
		    {
		    if  ( docRtfSaveExternalItem( sos,
					df->dfReferencedItem, bd,
					"{\\footnote ", rwc )	)
			{ LDEB(1); return -1;	}
		    }

		if  ( df->dfKind == DOCfkCHFTN )
		    {
		    sioOutPutString( "\\chftn ", sos ); col += 7;

		    /*HACK*/
		    s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */

		    while( part < bi->biParaParticuleCount- 1	&&
			   tp[1].tpKind != DOCkindFIELDEND	)
			{
			s += tp[1].tpStrlen; stroff += tp[1].tpStrlen;
			part++; tp++;
			}

		    continue;
		    }

		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		continue;

	    case DOCkindFIELDEND:
		df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
		fki= DOC_FieldKinds+ df->dfKind;

		if  ( df->dfKind == DOCfkBOOKMARK	&&
		      rwc->rwcSaveBookmarks		)
		    {
		    docRtfFinishBookmark( sos, df, &col, rwc );
		    rwc->rwcHasPrecedingTags= 0;
		    }

		if  ( fki->fkiIsFieldInRtf		&&
		      fki->fkiLevel == DOClevTEXT	)
		    {
		    if  ( rwc->rwcInFldrslt )
			{ docRtfFinishFldrslt( sos, &col, rwc );	}
		    }

		if  ( df->dfKind == DOCfkFOOTNOTE	||
		      df->dfKind == DOCfkCHFTN		)
		    { /* nothing */	}

		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		rwc->rwcHasPrecedingTags= 0;
		continue;

	    case DOCkindXE:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		if  (  rwc->rwcSaveBookmarks )
		    {
		    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

		    sioOutPutString( "{\\xe {", sos ); col += 5;
		    docRtfEscapeString( df->dfInstructions.odBytes,
				    (const unsigned char *)0,
				    &col, df->dfInstructions.odSize, sos );
		    sioOutPutString( "}}", sos ); col += 1;

		    rwc->rwcHasPrecedingTags= 0;
		    }
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		continue;

	    case DOCkindTC:
		if  ( stroffUpto >= 0 && stroff >= stroffUpto )
		    { break;	}

		if  (  rwc->rwcSaveBookmarks )
		    {
		    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

		    sioOutPutString( "{\\tc {", sos ); col += 5;
		    docRtfEscapeString( df->dfInstructions.odBytes,
				    (const unsigned char *)0,
				    &col, df->dfInstructions.odSize, sos );
		    sioOutPutString( "}}", sos ); col += 1;

		    rwc->rwcHasPrecedingTags= 0;
		    }
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		continue;

	    default:
		LDEB(tp->tpKind);
		s += tp->tpStrlen; stroff += tp->tpStrlen;
		continue;
	    }

	break;
	}

    if  ( stroff == bi->biParaStrlen )
	{
	if  ( rwc->rwcSaveAsLinkAsRef		&&
	      rwc->rwcSaveAsLinkAsPageref	)
	    {
	    if  ( rwc->rwcInFldrslt > 0 )
		{ docRtfFinishFldrslt( sos, &col, rwc );	}

	    sioOutPutString( "\\tab", sos ); col += 4;

	    docRtfStartField( sos, &(rwc->rwcSaveAsPagerefField), &col, rwc );

	    sioOutPutString( "?", sos ); col += 1;

	    rwc->rwcSaveAsLinkAsRef= 0;
	    rwc->rwcSaveAsLinkAsPageref= 0;
	    }

	while( rwc->rwcInFldrslt > 0 )
	    { docRtfFinishFldrslt( sos, &col, rwc ); }

	if  ( ! bi->biParaInTable					||
	      bi->biNumberInParent < bi->biParent->biChildCount- 1	)
	    { sioOutPutString( "\\par\r\n", sos );	}
	/*
	else{ sioOutPutString( "\\par", sos );		}
	*/
	}

    if  ( docCopyParagraphProperties( &(rwc->rwcParagraphProperties),
						&(bi->biParaProperties) ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a Row level item.						*/
/*									*/
/************************************************************************/

static int docRtfSaveRowItem(	SimpleOutputStream *	sos,
				const BufferDocument *	bd,
				const BufferItem *	rowBi,
				const BufferSelection *	bs,
				RtfWritingContext *	rwc )
    {
    int			col;

    if  ( ! bs								||
	  bs->bsBegin.bpBi->biParent != bs->bsEnd.bpBi->biParent	)
	{
	if  ( ! docEqualRows( &(rowBi->biRowProperties),
						&(rwc->rwcRowProperties) ) )
	    {
	    if  ( rwc->rwcInTable && docRtfPopTable( rwc, sos ) )
		{ LDEB(1);	}
	    }

	if  ( ! rwc->rwcInTable && docRtfPushTable( rwc, sos, rowBi, bs ) )
	    { LDEB(1);	}
	}

    for ( col= 0; col < rowBi->biChildCount; col++ )
	{
	int		par;
	BufferItem *	cellBi= rowBi->biChildren[col];

	if  ( bs && docCompareItemPositions( cellBi, bs->bsBegin.bpBi ) < 0 )
	    { continue;	}

	if  ( bs && docCompareItemPositions( cellBi, bs->bsEnd.bpBi ) > 0 )
	    { continue;	}

	if  ( ! bs						||
	      ( ( bs->bsCol0 < 0 || col >= bs->bsCol0 )	&&
		( bs->bsCol1 < 0 || col <= bs->bsCol1 )	)	)
	    {
	    for ( par= 0; par < cellBi->biChildCount; par++ )
		{
		BufferItem *	paraBi= cellBi->biChildren[par];

		if  ( bs && docCompareItemPositions( paraBi,
						    bs->bsBegin.bpBi ) < 0 )
		    { continue;	}

		if  ( bs && docCompareItemPositions( paraBi,
						    bs->bsEnd.bpBi ) > 0 )
		    { continue;	}

		if  ( docRtfSaveParaItem( sos, bd, paraBi, bs, rwc ) )
		    { LLDEB(col,par); return -1;	}
		}

	    if  ( col < rowBi->biChildCount- 1 )
		{ sioOutPutString( "\\cell\r\n", sos );	}
	    else{ sioOutPutString( "\\cell", sos );		}
	    }
	}

    sioOutPutString( "\\row\r\n", sos );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save the headers and footers of a Section.				*/
/*									*/
/************************************************************************/

static int docRtfSaveHeadersFooters(	SimpleOutputStream *	sos,
					const BufferItem *	bi,
					const BufferDocument *	bd,
					RtfWritingContext *	rwc )
    {
    if  ( docRtfSaveExternalItem( sos, bi->biSectHeader.hfItem, bd,
					    "{\\header\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, bi->biSectFirstPageHeader.hfItem, bd,
					    "{\\headerf\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, bi->biSectLeftPageHeader.hfItem, bd,
					    "{\\headerl\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, bi->biSectRightPageHeader.hfItem, bd,
					    "{\\headerr\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, bi->biSectFooter.hfItem, bd,
					    "{\\footer\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, bi->biSectFirstPageFooter.hfItem, bd,
					    "{\\footerf\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, bi->biSectLeftPageFooter.hfItem, bd,
					    "{\\footerl\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveExternalItem( sos, bi->biSectRightPageFooter.hfItem, bd,
					    "{\\footerr\r\n", rwc )	)
	{ LDEB(1); return -1;	}

    return 0;
    }

static int docSaveSectionProperties(	SimpleOutputStream *		sos,
					const DocumentProperties *	dp,
					const BufferSelection *		bs,
					const SectionProperties *	sp,
					RtfWritingContext *		rwc )
    {
    const DocumentGeometry *	dgDoc= &(dp->dpGeometry);
    const DocumentGeometry *	dgSect= &(sp->spDocumentGeometry);

    int				i;
    ParagraphNumber *		pn;

    sioOutPutString( "\\sectd", sos );

    if  ( dgSect->dgPageWideTwips != dgDoc->dgPageWideTwips		||
	  dgSect->dgPageHighTwips != dgDoc->dgPageHighTwips		||
	  dgSect->dgLeftMarginTwips != dgDoc->dgLeftMarginTwips		||
	  dgSect->dgTopMarginTwips != dgDoc->dgTopMarginTwips		||
	  dgSect->dgRightMarginTwips != dgDoc->dgRightMarginTwips	||
	  dgSect->dgBottomMarginTwips != dgDoc->dgBottomMarginTwips	||
	  dgSect->dgGutterTwips != dgDoc->dgGutterTwips			)
	{
	docRtfWriteArgTag( "\\pgwsxn",	dgSect->dgPageWideTwips, sos );
	docRtfWriteArgTag( "\\pghsxn",	dgSect->dgPageHighTwips, sos );
	docRtfWriteArgTag( "\\marglsxn",dgSect->dgLeftMarginTwips, sos );
	docRtfWriteArgTag( "\\margtsxn",dgSect->dgTopMarginTwips, sos );
	docRtfWriteArgTag( "\\margrsxn",dgSect->dgRightMarginTwips, sos );
	docRtfWriteArgTag( "\\margbsxn",dgSect->dgBottomMarginTwips, sos );
	docRtfWriteArgTag( "\\guttersxn",dgSect->dgGutterTwips, sos );

	sioOutPutString( "\r\n", sos );
	}

    if  ( dgSect->dgHeaderPositionTwips != 720	||
	  dgSect->dgFooterPositionTwips != 720	)
	{
	docRtfWriteArgTag( "\\headery", dgSect->dgHeaderPositionTwips, sos );
	docRtfWriteArgTag( "\\footery", dgSect->dgFooterPositionTwips, sos );
	}

    if  ( ! bs && sp->spHasTitlePage )
	{ sioOutPutString( "\\titlepg", sos );	}

    if  ( sp->spStartPageNumber != 0 )
	{ docRtfWriteArgTag( "\\pgnstarts", sp->spStartPageNumber+ 1, sos ); }
    if  ( sp->spRestartPageNumbers )
	{ sioOutPutString( "\\pgnrestart", sos );	}

    if  ( sp->spBreakKind != DOCsbkPAGE )
	{
	switch( sp->spBreakKind )
	    {
	    case DOCsbkNONE:	sioOutPutString( "\\sbknone", sos );	break;
	    case DOCsbkCOL:	sioOutPutString( "\\sbkcol", sos );	break;
	    case DOCsbkPAGE:	sioOutPutString( "\\sbkpage", sos );	break;
	    case DOCsbkEVEN:	sioOutPutString( "\\sbkeven", sos );	break;
	    case DOCsbkODD:	sioOutPutString( "\\sbkodd", sos );	break;
	    default:
		LDEB(sp->spBreakKind);
	    }
	}

    if  ( sp->spPageNumberStyle != DOCpgnDEC )
	{
	switch( sp->spPageNumberStyle )
	    {
	    case DOCpgnDEC:	sioOutPutString( "\\pgndec", sos );	break;
	    case DOCpgnUCRM:	sioOutPutString( "\\pgnucrm", sos );	break;
	    case DOCpgnLCRM:	sioOutPutString( "\\pgnlcrm", sos );	break;
	    case DOCpgnUCLTR:	sioOutPutString( "\\pgnucltr", sos );	break;
	    case DOCpgnLCLTR:	sioOutPutString( "\\pgnlcltr", sos );	break;
	    default:
		LDEB(sp->spPageNumberStyle);
	    }
	}

    if  ( sp->spColumnCount != 1 )
	{ docRtfWriteArgTag( "\\cols", sp->spColumnCount, sos );	}
    if  ( sp->spColumnSpacingTwips != 720 )
	{ docRtfWriteArgTag( "\\colsx", sp->spColumnSpacingTwips, sos ); }
    if  ( sp->spColumnWidthTwips != 0 )
	{ docRtfWriteArgTag( "\\colw", sp->spColumnWidthTwips, sos ); }

    sioOutPutString( "\r\n", sos );

    pn= sp->spParagraphNumbers;
    for ( i= 0; i < sp->spParagraphNumberCount; pn++, i++ )
	{
	if  ( pn->pnLevel < 0 )
	    { continue;	}

	docRtfWriteArgTag( "{\\*\\pnseclvl", pn->pnLevel+ 1, sos );

	switch( pn->pnStyle )
	    {
	    case DOCpnCARD:	sioOutPutString( "\\pncard", sos );	break;
	    case DOCpnDEC:	sioOutPutString( "\\pndec", sos );	break;
	    case DOCpnUCLTR:	sioOutPutString( "\\pnucltr", sos );	break;
	    case DOCpnUCRM:	sioOutPutString( "\\pnucrm", sos );	break;
	    case DOCpnLCLTR:	sioOutPutString( "\\pnlcltr", sos );	break;
	    case DOCpnLCRM:	sioOutPutString( "\\pnlcrm", sos );	break;
	    case DOCpnORD:	sioOutPutString( "\\pnord", sos );	break;
	    case DOCpnORDT:	sioOutPutString( "\\pnordt", sos );	break;
	    default:
		LDEB(pn->pnStyle); break;
	    }

	docRtfWriteArgTag( "\\pnstart", pn->pnStartNumber+ 1, sos );

	if  ( pn->pnUseHangingIndent )
	    { sioOutPutString( "\\pnhang", sos );	}

	if  ( pn->pnIndentTwips > 0 )
	    { docRtfWriteArgTag( "\\pnindent", pn->pnIndentTwips, sos ); }

	if  ( pn->pnTextBefore[0] )
	    {
	    int	col= 0;

	    sioOutPutString( "{\\pntxtb ", sos );

	    docRtfEscapeString( pn->pnTextBefore, rwc->rwcOutputMapping,
			&col, strlen( (char *)pn->pnTextBefore ), sos );

	    sioOutPutString( "}", sos );
	    }

	if  ( pn->pnTextAfter[0] )
	    {
	    int	col= 0;

	    sioOutPutString( "{\\pntxta ", sos );

	    docRtfEscapeString( pn->pnTextAfter, rwc->rwcOutputMapping,
			&col, strlen( (char *)pn->pnTextAfter ), sos );

	    sioOutPutString( "}", sos );
	    }

	sioOutPutString( "}\r\n", sos );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a buffer item, hierarchically descending to the leaves of the	*/
/*  document tree.							*/
/*									*/
/************************************************************************/

static int docRtfSaveItem(	SimpleOutputStream *	sos,
				const BufferItem *	bi,
				const BufferDocument *	bd,
				const BufferSelection *	bs,
				RtfWritingContext *	rwc )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);
    int				col= 0;

    int				i;

    if  ( bs )
	{
	if  ( docCompareItemPositions( bi, bs->bsBegin.bpBi ) < 0	)
	    { return 0;	}

	if  ( docCompareItemPositions( bi, bs->bsEnd.bpBi ) > 0		)
	    { return 0;	}
	}

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevCELL:
	rowAsGroup:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docRtfSaveItem( sos, bi->biChildren[i], bd, bs, rwc ) )
		    { LDEB(i); return -1;	}
		}
	    break;
	case DOClevSECT:

	    if  ( docSaveSectionProperties( sos, dp, bs,
					    &(bi->biSectProperties), rwc ) )
		{ LDEB(1); return -1;	}

	    if  ( ! bs						&&
		  docRtfSaveHeadersFooters( sos, bi, bd, rwc )	)
		{ LDEB(1); return -1;	}

	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docRtfSaveItem( sos, bi->biChildren[i], bd, bs, rwc ) )
		    { LDEB(i); return -1;	}
		}

	    if  ( rwc->rwcInTable && docRtfPopTable( rwc, sos ) )
		{ LDEB(1);	}

	    while( rwc->rwcInFldrslt > 0 )
		{ docRtfFinishFldrslt( sos, &col, rwc );	}

	    sioOutPutString( "\\sect\r\n", sos );
	    break;
	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{
		if  ( rwc->rwcInTable && docRtfPopTable( rwc, sos ) )
		    { LDEB(1);	}
		goto rowAsGroup;
		}

	    if  ( docRtfSaveRowItem( sos, bd, bi, bs, rwc ) )
		{ LDEB(1); return -1;	}

	    break;

	case DOClevPARA:
	    if  ( docRtfSaveParaItem( sos, bd, bi, bs, rwc ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }

static void docRtfSaveInfo(	const char *		tag,
				const unsigned char *	info,
				RtfWritingContext *	rwc,
				SimpleOutputStream *	sos )
    {
    int		col= 0;

    if  ( ! info )
	{ return;	}

    sioOutPutString( tag, sos );
    docRtfEscapeString( info, rwc->rwcOutputMapping,
					&col, strlen( (char *)info ), sos );
    sioOutPutString( "}\r\n", sos );

    return;
    }

static void docRtfSaveDate(	const char *		tag,
				const struct tm *	tm,
				RtfWritingContext *	rwc,
				SimpleOutputStream *	sos )
    {
    if  ( tm->tm_mday == 0 )
	{ return;	}

    sioOutPutString( tag, sos );

    docRtfWriteArgTag( "\\yr",	tm->tm_year+ 1900, sos );
    docRtfWriteArgTag( "\\mo",	tm->tm_mon+ 1, sos );
    docRtfWriteArgTag( "\\dy",	tm->tm_mday, sos );
    docRtfWriteArgTag( "\\hr",	tm->tm_hour, sos );
    docRtfWriteArgTag( "\\min",	tm->tm_min, sos );

    sioOutPutString( "}\r\n", sos );
    }

static int docRtfSaveDocumentProperties(	SimpleOutputStream *	sos,
						RtfWritingContext *	rwc,
						const BufferDocument *	bd )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);
    const DocumentGeometry *	dg= &(dp->dpGeometry);

    int				i;

    int				codePage= bd->bdProperties.dpAnsiCodePage;

    if  ( codePage >= 0 )
	{
	switch( codePage )
	    {
	    case 1250:
		docRtfWriteArgTag( "\\ansicpg", codePage, sos );
		sioOutPutString( "\r\n", sos );
		memcpy( rwc->rwcOutputMapping, docISO2_to_WIN1250, 256 );
		break;

	    case 1251:
		docRtfWriteArgTag( "\\ansicpg", codePage, sos );
		sioOutPutString( "\r\n", sos );
		memcpy( rwc->rwcOutputMapping, docKOI8R_to_WIN1251, 256 );
		break;

	    case 1252:
		docRtfWriteArgTag( "\\ansicpg", codePage, sos );
		sioOutPutString( "\r\n", sos );
		break;

	    default:
		LDEB(codePage); break;
	    }
	}

    if  ( dp->dpFontList.dflCount > 0 )
	{
	DocumentFont *	df=  dp->dpFontList.dflFonts;

	sioOutPutString( "{\\fonttbl\r\n", sos );

	for ( i= 0; i < dp->dpFontList.dflCount; df++, i++ )
	    {
	    if  ( ! df->dfFamilyStyle && ! df->dfName )
		{ continue;	}

	    docRtfWriteArgTag( "{\\f", df->dfDocFamilyNumber, sos );
	    sioOutPutString( "\\", sos );
	    if  ( df->dfFamilyStyle )
		{ sioOutPutString( df->dfFamilyStyle, sos );	}
	    else{ sioOutPutString( "fnil", sos );		}

	    if  ( df->dfCharset != FONTcharsetDEFAULT )
		{ docRtfWriteArgTag( "\\fcharset", df->dfCharset, sos ); }

	    if  ( df->dfPitch != FONTpitchDEFAULT )
		{ docRtfWriteArgTag( "\\fprq", df->dfPitch, sos ); }

	    if  ( df->dfName )
		{
		sioOutPutString( " ", sos );
		sioOutPutString( df->dfName, sos );
		}
	    sioOutPutString( ";}\r\n", sos );
	    }

	sioOutPutString( "}\r\n", sos );
	}

    if  ( dp->dpColorCount > 0 )
	{
	RGB8Color *			rgb8= dp->dpColors;

	sioOutPutString( "{\\colortbl\r\n", sos );

	for ( i= 0; i < dp->dpColorCount; rgb8++, i++ )
	    {
	    if  ( i != dp->dpDefaultColor )
		{
		docRtfWriteArgTag( "\\red", rgb8->rgb8Red, sos );
		docRtfWriteArgTag( "\\green", rgb8->rgb8Green, sos );
		docRtfWriteArgTag( "\\blue", rgb8->rgb8Blue, sos );
		}
	    sioOutPutString( ";\r\n", sos );
	    }

	sioOutPutString( "}\r\n", sos );
	}

    if  ( dp->dpAuthorCount > 0 )
	{
	sioOutPutString( "{\\*\\revtbl\r\n", sos );

	for ( i= 0; i < dp->dpAuthorCount; i++ )
	    {
	    int		col= 0;

	    sioOutPutCharacter( '{', sos );

	    docRtfEscapeString( dp->dpAuthors[i], rwc->rwcOutputMapping, &col,
				    strlen( (char *)dp->dpAuthors[i] ), sos );

	    sioOutPutString("}\r\n", sos );
	    }

	sioOutPutString("}\r\n", sos );
	}

    if  ( docHasDocumentInfo( dp ) )
	{
	sioOutPutString( "{\\info\r\n", sos );

	docRtfSaveInfo( "{\\title ",		dp->dpTitle, rwc, sos );
	docRtfSaveInfo( "{\\author ",		dp->dpAuthor, rwc, sos );
	docRtfSaveInfo( "{\\subject ",		dp->dpSubject, rwc, sos );
	docRtfSaveInfo( "{\\keywords ",		dp->dpKeywords, rwc, sos );
	docRtfSaveInfo( "{\\comment ",		dp->dpComment, rwc, sos );
	docRtfSaveInfo( "{\\hlinkbase ",	dp->dpHlinkbase, rwc, sos );

	docRtfSaveDate( "{\\creatim ",		&(dp->dpCreatim), rwc, sos );
	docRtfSaveDate( "{\\revtim ",		&(dp->dpRevtim), rwc, sos );
	docRtfSaveDate( "{\\printim ",		&(dp->dpPrintim), rwc, sos );

	sioOutPutString( "}\r\n", sos );
	}

    docRtfWriteArgTag( "\\paperw", dg->dgPageWideTwips, sos );
    docRtfWriteArgTag( "\\paperh", dg->dgPageHighTwips, sos );

    docRtfWriteArgTag( "\\margl", dg->dgLeftMarginTwips, sos );
    docRtfWriteArgTag( "\\margr", dg->dgRightMarginTwips, sos );
    docRtfWriteArgTag( "\\margt", dg->dgTopMarginTwips, sos );
    docRtfWriteArgTag( "\\margb", dg->dgBottomMarginTwips, sos );

    if  ( dg->dgGutterTwips != 0 )
	{ docRtfWriteArgTag( "\\gutter", dg->dgGutterTwips, sos );	}

    if  ( dp->dpHasFacingPages )
	{ sioOutPutString( "\\facingp", sos );	}

    if  ( dp->dpTabIntervalTwips > 0 && dp->dpTabIntervalTwips != 720 )
	{ docRtfWriteArgTag( "\\deftab", dp->dpTabIntervalTwips, sos );	}

    sioOutPutString( "\r\n", sos );

    switch( dp->dpFootnotePosition )
	{
	case DPftnPOS_SECT_END:
	    sioOutPutString( "\\endnotes", sos );
	    break;
	case DPftnPOS_DOC_END:
	    sioOutPutString( "\\enddoc", sos );
	    break;
	case DPftnPOS_BELOW_TEXT:
	    sioOutPutString( "\\ftntj", sos );
	    break;
	case DPftnPOS_PAGE_BOTTOM:
	    sioOutPutString( "\\ftnbj", sos );
	    break;
	default:
	    LDEB(dp->dpFootnotePosition); break;
	}

    switch( dp->dpEndnotePosition )
	{
	case DPftnPOS_SECT_END:
	    sioOutPutString( "\\aendnotes", sos );
	    break;
	case DPftnPOS_DOC_END:
	    sioOutPutString( "\\aenddoc", sos );
	    break;
	case DPftnPOS_BELOW_TEXT:
	    sioOutPutString( "\\aftntj", sos );
	    break;
	case DPftnPOS_PAGE_BOTTOM:
	    sioOutPutString( "\\aftnbj", sos );
	    break;
	default:
	    LDEB(dp->dpEndnotePosition); break;
	}

    if  ( dp->dpStartFootnoteNumber != 1 )
	{ docRtfWriteArgTag( "\\ftnstart", dp->dpStartFootnoteNumber, sos ); }
    if  ( dp->dpStartEndnoteNumber != 1 )
	{ docRtfWriteArgTag( "\\aftnstart", dp->dpStartEndnoteNumber, sos ); }

    switch( dp->dpFootnotesRestart )
	{
	case DPftnRST_CONTINUOUS:
	    sioOutPutString( "\\ftnrstcont", sos );
	    break;
	case DPftnRST_PER_SECTION:
	    sioOutPutString( "\\ftnrestart", sos );
	    break;
	case DPftnRST_PER_PAGE:
	    sioOutPutString( "\\ftnrstpg", sos );
	    break;
	default:
	    LDEB(dp->dpFootnotesRestart); break;
	}

    switch( dp->dpEndnotesRestart )
	{
	case DPftnRST_CONTINUOUS:
	    sioOutPutString( "\\aftnrstcont", sos );
	    break;
	case DPftnRST_PER_SECTION:
	    sioOutPutString( "\\aftnrestart", sos );
	    break;
	case DPftnRST_PER_PAGE:
	    sioOutPutString( "\\aftnrstpg", sos );
	    break;
	default:
	    LDEB(dp->dpEndnotesRestart); break;
	}


    switch( dp->dpFootnoteNumberStyle )
	{
	case DPftnNAR:
	    sioOutPutString( "\\ftnnar", sos );
	    break;
	case DPftnNALC:
	    sioOutPutString( "\\ftnnalc", sos );
	    break;
	case DPftnNAUC:
	    sioOutPutString( "\\ftnnauc", sos );
	    break;
	case DPftnNRLC:
	    sioOutPutString( "\\ftnnrlc", sos );
	    break;
	case DPftnNRUC:
	    sioOutPutString( "\\ftnnruc", sos );
	    break;
	case DPftnNCHI:
	    sioOutPutString( "\\ftnnchi", sos );
	    break;
	default:
	    LDEB(dp->dpFootnoteNumberStyle); break;
	}

    switch( dp->dpEndnoteNumberStyle )
	{
	case DPftnNAR:
	    sioOutPutString( "\\aftnnar", sos );
	    break;
	case DPftnNALC:
	    sioOutPutString( "\\aftnnalc", sos );
	    break;
	case DPftnNAUC:
	    sioOutPutString( "\\aftnnauc", sos );
	    break;
	case DPftnNRLC:
	    sioOutPutString( "\\aftnnrlc", sos );
	    break;
	case DPftnNRUC:
	    sioOutPutString( "\\aftnnruc", sos );
	    break;
	case DPftnNCHI:
	    sioOutPutString( "\\aftnnchi", sos );
	    break;
	default:
	    LDEB(dp->dpEndnoteNumberStyle); break;
	}

    sioOutPutString( "\r\n", sos );

    return 0;
    }

static void docRtfInitWritingContext(	RtfWritingContext *	rwc )
    {
    int		i;

    docInitTextAttribute( &(rwc->rwcTextAttribute) );
    docInitTextAttribute( &(rwc->rwcOutsideLinkAttribute) );
    docInitParagraphProperties( &(rwc->rwcParagraphProperties) );
    docInitRowProperties( &(rwc->rwcRowProperties) );

    docInitTextAttribute( &(rwc->rwcOutsideTableTextAttribute) );
    docInitParagraphProperties( &(rwc->rwcOutsideTableParagraphProperties) );

    rwc->rwcInFldrslt= 0;
    rwc->rwcInTable= 0;

    rwc->rwcSaveBookmarks= 1;

    rwc->rwcSaveAsLink= 0;
    rwc->rwcSaveAsLinkAsRef= 0;
    rwc->rwcSaveAsLinkAsPageref= 0;
    docInitField( &(rwc->rwcSaveAsHyperlinkField) );
    docInitField( &(rwc->rwcSaveAsRefField) );
    docInitField( &(rwc->rwcSaveAsPagerefField) );

    for ( i= 0; i < 256; i++ )
	{ rwc->rwcOutputMapping[i]= i;	}
    }

static void docRtfCleanWritingContext(	RtfWritingContext *	rwc )
    {
    docCleanParagraphProperties( &rwc->rwcParagraphProperties );
    docCleanRowProperties( &rwc->rwcRowProperties );

    docCleanParagraphProperties( &rwc->rwcOutsideTableParagraphProperties );

    docCleanField( (BufferDocument *)0, &(rwc->rwcSaveAsHyperlinkField) );
    docCleanField( (BufferDocument *)0, &(rwc->rwcSaveAsRefField) );
    docCleanField( (BufferDocument *)0, &(rwc->rwcSaveAsPagerefField) );
    }

int docRtfSaveDocument(		SimpleOutputStream *	sos,
				BufferDocument *	bd,
				const SelectionScope *	ss,
				const BufferSelection *	bs,
				int			saveBookmarks )
    {
    const BufferItem *		bi= &bd->bdItem;
    int				col= 0;

    RtfWritingContext		rwc;

    if  ( bs )
	{
	const int	nearest= 1;

	bi= docGetSelectionRoot( bd, nearest, ss, bs );
	if  ( ! bi )
	    { XDEB(bi); return -1;	}
	}

    docRtfInitWritingContext( &rwc );
    rwc.rwcSaveBookmarks= saveBookmarks;

    sioOutPutString( "{\\rtf0\\ansi\r\n", sos );

    if  ( docRtfSaveDocumentProperties( sos, &rwc, bd ) )
	{ LDEB(1); return -1;	}

    if  ( docRtfSaveItem( sos, bi, bd, bs, &rwc ) )
	{ LDEB(bi->biLevel); return -1; }

    while( rwc.rwcInFldrslt > 0 )
	{ docRtfFinishFldrslt( sos, &col, &rwc ); }

    sioOutPutString( "}\r\n", sos );

    docRtfCleanWritingContext( &rwc );

    return 0;
    }

int docRtfSaveSelectionAsLink(	SimpleOutputStream *	sos,
				const BufferDocument *	bd,
				const SelectionScope *	ss,
				const BufferSelection *	bs,
				int			asRef,
				int			asPageref,
				const unsigned char *	fileName,
				int			fileSize,
				const unsigned char *	markName,
				int			markSize )
    {
    const BufferItem *		bi= &bd->bdItem;
    int				col= 0;

    RtfWritingContext		rwc;

    docRtfInitWritingContext( &rwc );

    rwc.rwcSaveBookmarks= 0;
    rwc.rwcSaveAsLink= 1;
    rwc.rwcSaveAsLinkAsRef= asRef;
    rwc.rwcSaveAsLinkAsPageref= asPageref;

    if  ( docFieldSetHyperlink( &(rwc.rwcSaveAsHyperlinkField),
						    fileName, fileSize,
						    markName, markSize ) )
	{ LDEB(1); docRtfCleanWritingContext( &rwc ); return -1;	}

    if  ( asRef && docFieldSetRef( &(rwc.rwcSaveAsRefField),
						    markName, markSize ) )
	{ LDEB(asRef); docRtfCleanWritingContext( &rwc ); return -1;	}

    if  ( asPageref && docFieldSetPageref( &(rwc.rwcSaveAsPagerefField),
						    markName, markSize ) )
	{ LDEB(asPageref); docRtfCleanWritingContext( &rwc ); return -1; }

    sioOutPutString( "{\\rtf0\\ansi\r\n", sos );

    if  ( docRtfSaveDocumentProperties( sos, &rwc, bd ) )
	{ LDEB(1); docRtfCleanWritingContext( &rwc ); return -1;	}

    sioOutPutString( "\r\n", sos );

    if  ( docRtfSaveItem( sos, bi, bd, bs, &rwc ) )
	{ LDEB(bi->biLevel); docRtfCleanWritingContext( &rwc ); return -1; }

    while( rwc.rwcInFldrslt > 0 )
	{ docRtfFinishFldrslt( sos, &col, &rwc ); }

    sioOutPutString( "}\r\n", sos );

    docRtfCleanWritingContext( &rwc );

    return 0;
    }
