/************************************************************************/
/*									*/
/*  Read the various document tables of an RTF text file into a		*/
/*  BufferDocument.							*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>
#   include	<psFont.h>
#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Consume a font table.						*/
/*									*/
/************************************************************************/

static int docRtfFontName(	RtfReadingContext *	rrc,
				const unsigned char *	name,
				int			len )
    {
    DocumentFont *		df;
    DocumentProperties *	dp= &(rrc->rrcBd->bdProperties);

    static char *		scratch;
    char *			fresh;

    fresh= (char *)realloc( scratch, len+ 1 );
    if  ( ! fresh )
	{ XDEB(fresh); return -1;	}
    scratch= fresh;
    strncpy( scratch, (const char *)name, len )[len]= '\0';

    if  ( len > 0 && scratch[len-1] == ';' )
	{ scratch[len-1]= '\0';	}

    if  ( ! scratch[0] )
	{ return 0;	}

    df= docInsertFont( &(dp->dpFontList),
		    rrc->rrcCurrentFont.dfDocFamilyNumber,
		    rrc->rrcCurrentFont.dfFamilyStyle, scratch );

    if  ( ! df )
	{ SSDEB(rrc->rrcCurrentFont.dfFamilyStyle,scratch); return -1; }

    df->dfCharset= rrc->rrcCurrentFont.dfCharset;
    df->dfPitch= rrc->rrcCurrentFont.dfPitch;

    return 0;
    }

static int docRtfFontNumber(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    docCleanFont( &(rrc->rrcCurrentFont) );
    docInitFont( &(rrc->rrcCurrentFont) );

    rrc->rrcCurrentFont.dfDocFamilyNumber= arg;

    return 0;
    }

static int docRtfFontFamily(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    rrc->rrcCurrentFont.dfFamilyStyle= strdup( rcw->rcwWord );

    if  ( ! rrc->rrcCurrentFont.dfFamilyStyle )
	{ XDEB(rrc->rrcCurrentFont.dfFamilyStyle); return -1;	}

    return 0;
    }

static int docRtfFontProperty(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case RTFidFCHARSET:
	    rrc->rrcCurrentFont.dfCharset= arg;
	    break;

	case RTFidFPRQ:
	    rrc->rrcCurrentFont.dfPitch= arg;
	    break;

	default:
	    SLDEB(rcw->rcwWord,arg);
	}

    return 0;
    }

static RtfControlWord	docRtfFontGroupWords[]=
    {
	{ "fnil",	RTFidFNIL,	DOClevANY, docRtfFontFamily, },
	{ "froman",	RTFidFROMAN,	DOClevANY, docRtfFontFamily, },
	{ "fswiss",	RTFidFSWISS,	DOClevANY, docRtfFontFamily, },
	{ "fmodern",	RTFidFMODERN,	DOClevANY, docRtfFontFamily, },
	{ "fscript",	RTFidFSCRIPT,	DOClevANY, docRtfFontFamily, },
	{ "fdecor",	RTFidFDECOR,	DOClevANY, docRtfFontFamily, },
	{ "ftech",	RTFidFTECH,	DOClevANY, docRtfFontFamily, },

	{ "fcharset",	RTFidFCHARSET,	DOClevDOC, docRtfFontProperty, },
	{ "fprq",	RTFidFPRQ,	DOClevDOC, docRtfFontProperty, },

	{ 0, 0, 0 }
    };

static int docRtfFontGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    docCleanFont( &(rrc->rrcCurrentFont) );
    docInitFont( &(rrc->rrcCurrentFont) );

    rrc->rrcCurrentFont.dfDocFamilyNumber= arg;

    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfFontGroupWords, docRtfEmptyTable,
				    docRtfFontName ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    docCleanFont( &(rrc->rrcCurrentFont) );
    docInitFont( &(rrc->rrcCurrentFont) );

    return 0;
    }

static RtfControlWord	docRtfFontTableWords[]=
    {
	{ "f",		RTFidF,		DOClevDOC, docRtfFontNumber, },

	{ "fnil",	RTFidFNIL,	DOClevANY, docRtfFontFamily, },
	{ "froman",	RTFidFROMAN,	DOClevANY, docRtfFontFamily, },
	{ "fswiss",	RTFidFSWISS,	DOClevANY, docRtfFontFamily, },
	{ "fmodern",	RTFidFMODERN,	DOClevANY, docRtfFontFamily, },
	{ "fscript",	RTFidFSCRIPT,	DOClevANY, docRtfFontFamily, },
	{ "fdecor",	RTFidFDECOR,	DOClevANY, docRtfFontFamily, },
	{ "ftech",	RTFidFTECH,	DOClevANY, docRtfFontFamily, },

	{ "fcharset",	RTFidFCHARSET,	DOClevDOC, docRtfFontProperty, },
	{ "fprq",	RTFidFPRQ,	DOClevDOC, docRtfFontProperty, },

	{ 0, 0, 0 }
    };

static RtfControlWord	docRtfFontTableGroups[]=
    {
	{ "f",		RTFidF,		DOClevDOC,  docRtfFontGroup, },

	{ 0, 0, 0 }
    };

int docRtfFontTable(	SimpleInputStream *	sis,
			const RtfControlWord *	rcw,
			int			arg,
			RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfFontTableWords, docRtfFontTableGroups,
				docRtfFontName ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Read a color table.							*/
/*									*/
/************************************************************************/

static int docRtfSaveColor(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len )
    {
    DocumentProperties *	dp= &(rrc->rrcBd->bdProperties);
    RGB8Color *			fresh;

    fresh= (RGB8Color *)realloc( dp->dpColors,
			    ( dp->dpColorCount+ 1) * sizeof( RGB8Color ) );
    if  ( ! fresh )
	{ LXDEB(dp->dpColorCount,fresh); return -1;	}

    dp->dpColors= fresh;
    fresh[dp->dpColorCount++]= rrc->rrcColor;

    if  ( ! rrc->rrcGotComponent )
	{
	if  ( dp->dpDefaultColor < 0 )
	    { dp->dpDefaultColor= dp->dpColorCount;	}
	}

    rrc->rrcGotComponent= 0;
    return 0;
    }

static int docRtfColorComp(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case	RTFidRED:
	    rrc->rrcGotComponent= 1;
	    rrc->rrcColor.rgb8Red= arg;
	    break;
	case	RTFidGREEN:
	    rrc->rrcGotComponent= 1;
	    rrc->rrcColor.rgb8Green= arg;
	    break;
	case	RTFidBLUE:
	    rrc->rrcGotComponent= 1;
	    rrc->rrcColor.rgb8Blue= arg;
	    break;
	default:
	    /* SLDEB(rcw->rcwWord,arg); */
	    break;
	}

    return 0;
    }

static RtfControlWord	docRtfColorTableWords[]=
    {
	{ "red",	RTFidRED,	DOClevDOC, docRtfColorComp, },
	{ "green",	RTFidGREEN,	DOClevDOC, docRtfColorComp, },
	{ "blue",	RTFidBLUE,	DOClevDOC, docRtfColorComp, },
	{ 0, 0, 0 }
    };

int docRtfColorTable(		SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    rrc->rrcGotComponent= 0;

    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfColorTableWords, docRtfEmptyTable,
				    docRtfSaveColor ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

/************************************************************************/
/*                                                                      */
/*  Consume the 'revision table' of a document                          */
/*    added by davina 6-2-00                                            */
/*                                                                      */
/************************************************************************/

static int docRtfSaveRevisionTable(	RtfReadingContext *	rrc,
					const unsigned char *	text,
					int			len )
    {
    unsigned char **		newAuthorArray;
    unsigned char *		newAuthor;
    DocumentProperties *	dp= &(rrc->rrcBd->bdProperties);

    newAuthorArray = (unsigned char**)realloc( dp->dpAuthors, 
			   (dp->dpAuthorCount+1) * sizeof(unsigned char *));
    if  ( ! newAuthorArray )
	{ XDEB(newAuthorArray); return -1;	}

    dp->dpAuthors= newAuthorArray;
    newAuthor= (unsigned char*)strdup( (char *)text );

    if  ( ! newAuthor )
	{ XDEB(newAuthor); return -1; }
    dp->dpAuthors[dp->dpAuthorCount++]= newAuthor;

    return 0;
    }

int docRtfRevisionTable(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    if  ( docRtfReadGroup(sis, rcw->rcwLevel, (RtfControlWord*)0, 0, 0, rrc,
                          docRtfDocumentWords, docRtfDocumentGroups,
                          docRtfSaveRevisionTable))
	{ SLDEB(rcw->rcwWord, arg); return -1; }

    return 0;
    }

/************************************************************************/
/*									*/
/*  Consume the 'stylesheet' of a document				*/
/*									*/
/************************************************************************/

static int docRtfCharacterStyle(	SimpleInputStream *	sis,
					const RtfControlWord *	rcw,
					int			arg,
					RtfReadingContext *	rrc	);

static int docRtfStyleName(	RtfReadingContext *	rrc,
				const unsigned char *	name,
				int			len	)
    {
    DocumentStyle *	ds;

    static char *	scratch;
    char *		fresh;

    fresh= (char *)realloc( scratch, len+ 1 );
    if  ( ! fresh )
	{ XDEB(fresh); return -1;	}
    scratch= fresh;
    strncpy( scratch, (const char *)name, len )[len]= '\0';

    if  ( len > 0 && scratch[len-1] == ';' )
	{ scratch[len-1]= '\0';	}

    ds= docInsertStyle( rrc->rrcBd, rrc->rrcStyleNumber, scratch );
    if  ( ! ds )
	{ XDEB(ds); return -1;	}

    docCopyParagraphProperties( &(ds->dsParagraphProperties),
					    &(rrc->rrcParagraphProperties) );
    ds->dsTextAttribute= rrc->rrcCurrentTextAttribute;

    rrc->rrcStyleNumber= 0;
    rrc->rrcStyleBasedOn= 0;
    rrc->rrcStyleNext= 0;

    docCleanParagraphProperties( &(rrc->rrcParagraphProperties) );
    docInitParagraphProperties( &(rrc->rrcParagraphProperties) );
    docInitTextAttribute( &(rrc->rrcCurrentTextAttribute) );
    rrc->rrcCurrentTextAttribute.taFontSizeHalfPoints= 24;
    rrc->rrcInDeletedText= 0;

    return 0;
    }

static int docRtfStyleProperty(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case RTFidSBASEDON:
	    rrc->rrcStyleBasedOn= arg;
	    break;
	case RTFidSNEXT:
	    rrc->rrcStyleNext= arg;
	    break;
	case RTFidS:
	    rrc->rrcStyleNumber= arg;
	    break;
	case RTFidSAUTOUPD:
	    break;
	case RTFidCS:
	    break;
	case RTFidADDITIVE:
	    break;

	default:
	    SDEB(rcw->rcwWord); break;
	}

    return 0;
    }

static RtfControlWord	docRtfStylesheetGroups[]=
    {
	{ "cs",		RTFidCS,	DOClevDOC, docRtfCharacterStyle, },

	{ 0, 0, 0 }
    };

static RtfControlWord	docRtfStylesheetWords[]=
    {
	{ "sbasedon",	RTFidSBASEDON,	DOClevDOC, docRtfStyleProperty, },
	{ "snext",	RTFidSNEXT,	DOClevDOC, docRtfStyleProperty, },
	{ "s",		RTFidS,		DOClevDOC, docRtfStyleProperty, },
	{ "sautoupd",	RTFidSAUTOUPD,	DOClevDOC, docRtfStyleProperty, },
	{ "additive",	RTFidADDITIVE,	DOClevDOC, docRtfStyleProperty, },
	{ "cs",		RTFidCS,	DOClevDOC, docRtfStyleProperty, },

	{ 0, 0, 0 }
    };

static int docRtfCharacterStyle(	SimpleInputStream *	sis,
					const RtfControlWord *	rcw,
					int			arg,
					RtfReadingContext *	rrc	)
    {
    rrc->rrcStyleNumber= arg;

    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfStylesheetWords, docRtfEmptyTable,
				    docRtfStyleName ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

int docRtfStylesheet(		SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    rrc->rrcStyleNumber= 0;
    rrc->rrcStyleBasedOn= 0;
    rrc->rrcStyleNext= 0;

    docCleanParagraphProperties( &(rrc->rrcParagraphProperties) );
    docInitParagraphProperties( &(rrc->rrcParagraphProperties) );
    docInitTextAttribute( &(rrc->rrcCurrentTextAttribute) );
    rrc->rrcCurrentTextAttribute.taFontSizeHalfPoints= 24;
    rrc->rrcInDeletedText= 0;

    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfStylesheetWords, docRtfStylesheetGroups,
				docRtfStyleName ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Read the info group.						*/
/*									*/
/************************************************************************/

static int docRtfInfoText(	RtfReadingContext *	rrc,
				unsigned char **	pText,
				const unsigned char *	text,
				int			len	)
    {
    char *	saved= (char *)malloc( len+ 1 );
    char *	to;

    if  ( ! saved )
	{ XDEB(saved); return -1;	}

    to= saved;
    while( len > 0 )
	{ *(to++)= rrc->rrcInputMapping[*(text++)]; len--;	}
    *to= '\0';

    if  ( *pText )
	{ free( *pText );	}

    *pText= (unsigned char *)saved;

    return 0;
    }

static int docRtfTitleText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    {
    return docRtfInfoText( rrc, &(rrc->rrcBd->bdProperties.dpTitle),
								text, len );
    }

static int docRtfAuthorText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    {
    return docRtfInfoText( rrc, &(rrc->rrcBd->bdProperties.dpAuthor),
								text, len );
    }

static int docRtfSubjectText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    {
    return docRtfInfoText( rrc, &(rrc->rrcBd->bdProperties.dpSubject),
								text, len );
    }

static int docRtfKeywordsText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    {
    return docRtfInfoText( rrc, &(rrc->rrcBd->bdProperties.dpKeywords),
								text, len );
    }

static int docRtfCommentText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    {
    return docRtfInfoText( rrc, &(rrc->rrcBd->bdProperties.dpComment),
								text, len );
    }

static int docRtfHlinkbaseText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    {
    return docRtfInfoText( rrc, &(rrc->rrcBd->bdProperties.dpHlinkbase),
								text, len );
    }

static int docRtfTitleGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfTitleText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfAuthorGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfAuthorText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfSubjectGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfSubjectText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfKeywordsGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfKeywordsText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfCommentGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfCommentText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfHlinkbaseGroup( SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfHlinkbaseText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfCreatimGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    appInvalidateTime( &(rrc->rrcTm) );

    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfRefuseText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    rrc->rrcBd->bdProperties.dpCreatim= rrc->rrcTm;

    return 0;
    }

static int docRtfRevtimGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    appInvalidateTime( &(rrc->rrcTm) );

    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfRefuseText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    rrc->rrcBd->bdProperties.dpRevtim= rrc->rrcTm;

    return 0;
    }

static int docRtfPrintimGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    appInvalidateTime( &(rrc->rrcTm) );

    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfRefuseText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    rrc->rrcBd->bdProperties.dpPrintim= rrc->rrcTm;

    return 0;
    }

static RtfControlWord	docRtfInfoGroups[]=
    {
	{ "title",	RTFidTITLE,	DOClevDOC, docRtfTitleGroup, },
	{ "author",	RTFidAUTHOR,	DOClevDOC, docRtfAuthorGroup, },
	{ "subject",	RTFidSUBJECT,	DOClevDOC, docRtfSubjectGroup, },
	{ "keywords",	RTFidKEYWORDS,	DOClevDOC, docRtfKeywordsGroup, },
	{ "comment",	RTFidCOMMENT,	DOClevDOC, docRtfCommentGroup, },
	{ "hlinkbase",	RTFidHLINKBASE,	DOClevDOC, docRtfHlinkbaseGroup, },

	{ "creatim",	RTFidCREATIM,	DOClevDOC, docRtfCreatimGroup, },
	{ "revtim",	RTFidREVTIM,	DOClevDOC, docRtfRevtimGroup, },
	{ "printim",	RTFidPRINTIM,	DOClevDOC, docRtfPrintimGroup, },

	{ "operator",	RTFidOPERATOR,	DOClevDOC, docRtfSkipGroup, },
	{ "version",	RTFidVERSION,	DOClevDOC, docRtfSkipGroup, },
	{ "edmins",	RTFidEDMINS,	DOClevDOC, docRtfSkipGroup, },
	{ "nofpages",	RTFidNOFPAGES,	DOClevDOC, docRtfSkipGroup, },
	{ "nofwords",	RTFidNOFWORDS,	DOClevDOC, docRtfSkipGroup, },
	{ "nofchars",	RTFidNOFCHARS,	DOClevDOC, docRtfSkipGroup, },
	{ "nofcharsws",	RTFidNOFCHARSWS,DOClevDOC, docRtfSkipGroup, },
	{ "vern",	RTFidVERN,	DOClevDOC, docRtfSkipGroup, },

	{ 0, 0, 0 }
    };

int docRtfReadInfo(		SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfEmptyTable, docRtfInfoGroups,
				docRtfIgnoreText );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

/************************************************************************/
/*									*/
/*  Consume the 'listtable' of a document				*/
/*									*/
/************************************************************************/

static int docRtfListNameText(	RtfReadingContext *	rrc,
				const unsigned char *	name,
				int			len )
    {
    char *		fresh;

    fresh= (char *)realloc( rrc->rrcDocumentList.dlListName, len+ 1 );
    if  ( ! fresh )
	{ XDEB(fresh); return -1;	}
    rrc->rrcDocumentList.dlListName= fresh;
    strncpy( rrc->rrcDocumentList.dlListName,
				    (const char *)name, len )[len]= '\0';

    if  ( len > 0 && rrc->rrcDocumentList.dlListName[len-1] == ';' )
	{ rrc->rrcDocumentList.dlListName[len-1]= '\0';	}

    return 0;
    }

static int docRtfListName(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfEmptyTable, docRtfEmptyTable,
				docRtfListNameText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfListProperty(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    switch( rcw->rcwId )
	{
	case RTFidLISTID:
	    rrc->rrcDocumentList.dlListId= arg;
	    break;
	case RTFidLISTTEMPLATEID:
	    rrc->rrcDocumentList.dlListTemplateId= arg;
	    break;
	case RTFidLISTSIMPLE:
	    rrc->rrcDocumentList.dlListIsSimple= arg != 0;
	    break;
	case RTFidLISTRESTARTHDN:
	    rrc->rrcDocumentList.dlRestartForEverySection= arg != 0;
	    break;
	default:
	    break;
	}

    return 0;
    }

static RtfControlWord	docRtfListGroups[]=
    {
	{ "listname",	RTFidLISTNAME,	DOClevDOC, docRtfListName, },

	{ 0, 0, 0 }
    };

static RtfControlWord	docRtfListWords[]=
    {
	{ "listid",	RTFidLISTID,	DOClevDOC,	docRtfListProperty, },
	{ "listsimple",	RTFidLISTSIMPLE, DOClevDOC,	docRtfListProperty, },
	{ "listrestarthdn",
			RTFidLISTRESTARTHDN,
					DOClevDOC,	docRtfListProperty, },
	{ "listtemplateid",
			RTFidLISTTEMPLATEID,
					DOClevDOC,	docRtfListProperty, },

	{ 0, 0, 0 }
    };

static int docRtfList(		SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfListWords, docRtfListGroups,
				docRtfIgnoreText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static RtfControlWord	docRtfListtableGroups[]=
    {
	{ "list",	RTFidLIST,	DOClevDOC, docRtfList, },

	{ 0, 0, 0 }
    };

int docRtfListtable(		SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfEmptyTable, docRtfListtableGroups,
				docRtfIgnoreText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }
