/************************************************************************/
/*									*/
/*  Read an RTF text file into a BufferDocument				*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	<psFont.h>
#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Forward declarations.						*/
/*									*/
/************************************************************************/

static int docRtfReadPntext(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc );
static int docRtfSkipPn(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc );
static int docRtfReadListtext(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc );
static int docRtfReadUpr(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc );
static int docRtfReadPnseclvl(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc );

/************************************************************************/
/*									*/
/*  Special actions for section properties.				*/
/*									*/
/************************************************************************/

static int docRtfTextSpecial(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case	RTFidU:
	    break;
	case	RTFidCHFTN:
	    break;
	default:
	    SLDEB(rcw->rcwWord,rcw->rcwId);
	    break;
	}

    return 0;
    }

static int docRtfBeginBorder(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    docInitBorderProperties( &(rrc->rrcBorderProperties) );
    rrc->rrcBorderProperties.bpIsSet= 1;
    return 0;
    }

static int docRtfBrdrProperty(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case RTFidBRDRS:
	    rrc->rrcBorderProperties.bpThickness= 1;
	    break;
	case RTFidBRDRTH:
	    rrc->rrcBorderProperties.bpThickness= 2;
	    break;
	case RTFidBRDRW:
	    if  ( arg > 255 )
		{ LDEB(arg); arg= 255;	}

	    rrc->rrcBorderProperties.bpWidthTwips= arg;
	    break;
	case RTFidBRDRCF:
	    if  ( arg > 255 )
		{ LDEB(arg); arg= 255;	}

	    rrc->rrcBorderProperties.bpColor= arg;
	    break;
	case RTFidBRSP:
	    rrc->rrcBorderProperties.bpSpacingTwips= arg;
	    break;
	case RTFidBRDRSH:
	    rrc->rrcBorderProperties.bpStyle= DOCbsSH;
	    break;
	case RTFidBRDRDB:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDB;
	    break;
	case RTFidBRDRDOT:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDOT;
	    break;
	case RTFidBRDRDASH:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDASH;
	    break;
	case RTFidBRDRHAIR:
	    rrc->rrcBorderProperties.bpStyle= DOCbsHAIR;
	    break;
	case RTFidBRDRDASHSM:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDASHSM;
	    break;
	case RTFidBRDRDASHD:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDASHD;
	    break;
	case RTFidBRDRDASHDD:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDASHDD;
	    break;
	case RTFidBRDRTRIPLE:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTRIPLE;
	    break;
	case RTFidBRDRTNTHSG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTNTHSG;
	    break;
	case RTFidBRDRTHTNSG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTHTNSG;
	    break;
	case RTFidBRDRTNTHTNSG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTNTHTNSG;
	    break;
	case RTFidBRDRTNTHMG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTNTHMG;
	    break;
	case RTFidBRDRTHTNMG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTHTNMG;
	    break;
	case RTFidBRDRTNTHTNMG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTNTHTNMG;
	    break;
	case RTFidBRDRTNTHLG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTNTHLG;
	    break;
	case RTFidBRDRTHTNLG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTHTNLG;
	    break;
	case RTFidBRDRTNTHTNLG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTNTHTNLG;
	    break;
	case RTFidBRDRWAVY:
	    rrc->rrcBorderProperties.bpStyle= DOCbsWAVY;
	    break;
	case RTFidBRDRWAVYDB:
	    rrc->rrcBorderProperties.bpStyle= DOCbsWAVYDB;
	    break;
	case RTFidBRDRDASHDOTSTR:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDASHDOTSTR;
	    break;
	case RTFidBRDREMBOSS:
	    rrc->rrcBorderProperties.bpStyle= DOCbsEMBOSS;
	    break;
	case RTFidBRDRENGRAVE:
	    rrc->rrcBorderProperties.bpStyle= DOCbsENGRAVE;
	    break;
	case RTFidBRDRNONE:
	    /* rrc->rrcBorderProperties.bpStyle= DOCbsNONE; */
	    rrc->rrcBorderProperties.bpIsSet= 0;
	    break;

	default:
	    SLDEB(rcw->rcwWord,rcw->rcwId);
	    break;
	}

    return 0;
    }

static int docRtfHierarchy(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {

    switch( rcw->rcwId )
	{
	case	RTFidSECT:
	    if  ( rrc->rrcSplitLevel < DOClevSECT )
		{
		if  ( docRtfAdjustLevel( rrc, DOClevPARA, DOClevTEXT ) )
		    { SDEB(rcw->rcwWord); return -1; }
		}

	    if  ( docRtfAdjustLevel( rrc, DOClevSECT- 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }
	    rrc->rrcSplitLevel= DOClevSECT- 1;
	    break;

	case	RTFidROW:
	    if  ( rrc->rrcSplitLevel < DOClevROW )
		{
		if  ( docRtfAdjustLevel( rrc, DOClevPARA, DOClevTEXT ) )
		    { SDEB(rcw->rcwWord); return -1; }
		}

	    if  ( docRtfAdjustLevel( rrc, DOClevROW- 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }
	    rrc->rrcSplitLevel= DOClevROW- 1;
	    break;

	case	RTFidCELL:
	    if  ( rrc->rrcSplitLevel <= DOClevCELL )
		{
		if  ( docRtfAdjustLevel( rrc, DOClevPARA, DOClevTEXT ) )
		    { SDEB(rcw->rcwWord); return -1; }
		}

	    if  ( docRtfAdjustLevel( rrc, DOClevCELL- 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }
	    rrc->rrcSplitLevel= DOClevCELL- 1;

	    break;

	case	RTFidPAR:
	    if  ( docRtfAdjustLevel( rrc, DOClevPARA, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }

	    if  ( docRtfAdjustLevel( rrc, DOClevPARA- 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }
	    break;

	case	RTFidTAB:
	    if  ( docSaveSpecialParticule( rrc->rrcBi,
			    rrc->rrcCurrentTextAttribute, DOCkindTAB ) )
		{ LDEB(1); return -1;	}
	    break;

	case	RTFidLINE:
	    if  ( docSaveSpecialParticule( rrc->rrcBi,
			    rrc->rrcCurrentTextAttribute, DOCkindLINEBREAK ) )
		{ LDEB(1); return -1;	}
	    break;

	case	RTFidPAGE:
	    if  ( rrc->rrcJustAfterPntext )
		{ rrc->rrcPageAfterPntext= 1; return 0; }

	    if  ( docSaveSpecialParticule( rrc->rrcBi,
			    rrc->rrcCurrentTextAttribute, DOCkindPAGEBREAK ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    SLDEB(rcw->rcwWord,rcw->rcwId); break;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Control words to remember border properties.			*/
/*									*/
/************************************************************************/

static RtfControlWord	docRtfBorderWords[]=
    {
	{ "brdrth",	RTFidBRDRTH,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrs",	RTFidBRDRS,	DOClevPARA, docRtfBrdrProperty, },
	/**/
	{ "brdrsh",	RTFidBRDRSH,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrdb",	RTFidBRDRDB,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrdot",	RTFidBRDRDOT,	DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrdash",	RTFidBRDRDASH ,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrhair",	RTFidBRDRHAIR,	DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrdashsm",	RTFidBRDRDASHSM,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrdashd",	RTFidBRDRDASHD,	DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrdashdd",	RTFidBRDRDASHDD,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtriple",	RTFidBRDRTRIPLE,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtnthsg",	RTFidBRDRTNTHSG,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrthtnsg",	RTFidBRDRTHTNSG,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtnthtnsg",
			RTFidBRDRTNTHTNSG,
					DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtnthmg",	RTFidBRDRTNTHMG,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrthtnmg",	RTFidBRDRTHTNMG,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtnthtnmg",
			RTFidBRDRTNTHTNMG,
					DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtnthlg",	RTFidBRDRTNTHLG,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrthtnlg",	RTFidBRDRTHTNLG,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtnthtnlg",
			RTFidBRDRTNTHTNLG,
					DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrwavy",	RTFidBRDRWAVY,	DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrwavydb",	RTFidBRDRWAVYDB,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrdashdotstr",
			RTFidBRDRDASHDOTSTR,
					DOClevPARA, docRtfBrdrProperty,	},
	{ "brdremboss",	RTFidBRDREMBOSS,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrengrave",RTFidBRDRENGRAVE,
					DOClevPARA, docRtfBrdrProperty,	},
	/**/
	{ "brdrw",	RTFidBRDRW,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrcf",	RTFidBRDRCF,	DOClevPARA, docRtfBrdrProperty, },
	{ "brsp",	RTFidBRSP,	DOClevPARA, docRtfBrdrProperty, },
	/**/
	{ "brdroutset",	RTFidBRDROUTSET,
					DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrnone",	RTFidBRDRNONE,
					DOClevPARA, docRtfBrdrProperty,	},
	{ 0, 0, 0 }
    };

RtfControlWord	docRtfDocumentWords[]=
    {
	{ "fc",		RTFidFC,	DOClevANY,  docRtfRememberProperty, },
				/****************************************/
				/*  Special characters.			*/
				/****************************************/
	{ "bullet",	RTFidBULLET,	DOClevTEXT, docRtfTextSpecial, },
	{ "chatn",	RTFidCHATN,	DOClevTEXT, docRtfTextSpecial, },
	{ "chdate",	RTFidCHDATE,	DOClevTEXT, docRtfTextSpecial, },
	{ "chdpa",	RTFidCHDPA,	DOClevTEXT, docRtfTextSpecial, },
	{ "chdpl",	RTFidCHDPL,	DOClevTEXT, docRtfTextSpecial, },
	{ "chftn",	RTFidCHFTN,	DOClevTEXT, docRtfChftn,	},
	{ "chftnsep",	RTFidCHFTNSEP,	DOClevTEXT, docRtfTextSpecial, },
	{ "chftnsepc",	RTFidCHFTNSEPC,	DOClevTEXT, docRtfTextSpecial, },
	{ "chpgn",	RTFidCHPGN,	DOClevTEXT, docRtfSpecialToField, },
	{ "chtime",	RTFidCHTIME,	DOClevTEXT, docRtfTextSpecial, },
	{ "column",	RTFidCOLUMN,	DOClevTEXT, docRtfTextSpecial, },
	{ "emdash",	RTFidEMDASH,	DOClevTEXT, docRtfTextSpecial, },
	{ "emspace",	RTFidEMSPACE,	DOClevTEXT, docRtfTextSpecial, },
	{ "endash",	RTFidENDASH,	DOClevTEXT, docRtfTextSpecial, },
	{ "enspace",	RTFidENSPACE,	DOClevTEXT, docRtfTextSpecial, },
	{ "ldblquote",	RTFidLDBLQUOTE,	DOClevTEXT, docRtfTextSpecial, },
	{ "lquote",	RTFidLQUOTE,	DOClevTEXT, docRtfTextSpecial, },
	{ "ltrmark",	RTFidLTRMARK,	DOClevTEXT, docRtfTextSpecial, },
	{ "rdblquote",	RTFidRDBLQUOTE,	DOClevTEXT, docRtfTextSpecial, },
	{ "rquote",	RTFidRQUOTE,	DOClevTEXT, docRtfTextSpecial, },
	{ "rtlmark",	RTFidRTLMARK,	DOClevTEXT, docRtfTextSpecial, },
	{ "sectnum",	RTFidSECTNUM,	DOClevTEXT, docRtfTextSpecial, },
	{ "softcol",	RTFidSOFTCOL,	DOClevTEXT, docRtfTextSpecial, },
	{ "softlheight",RTFidSOFTLHEIGHT, DOClevTEXT, docRtfTextSpecial, },
	{ "softline",	RTFidSOFTLINE,	DOClevTEXT, docRtfTextSpecial, },
	{ "softpage",	RTFidSOFTPAGE,	DOClevTEXT, docRtfTextSpecial, },
	{ "zwj",	RTFidZWJ,	DOClevTEXT, docRtfTextSpecial, },
	{ "zwnj",	RTFidZWNJ,	DOClevTEXT, docRtfTextSpecial, },

	{ "u",		RTFidU,		DOClevTEXT, docRtfTextSpecial, },
				/****************************************/
				/*  Hierarchy/Structure tags.		*/
				/****************************************/
	{ "tab",	RTFidTAB,	DOClevTEXT, docRtfHierarchy, },
	{ "line",	RTFidLINE,	DOClevTEXT, docRtfHierarchy, },
	{ "page",	RTFidPAGE,	DOClevTEXT, docRtfHierarchy, },

	{ "par",	RTFidPAR,	DOClevANY,  docRtfHierarchy, },
	{ "cell",	RTFidCELL,	DOClevANY,  docRtfHierarchy, },
	{ "row",	RTFidROW,	DOClevANY,  docRtfHierarchy, },
	{ "sect",	RTFidSECT,	DOClevANY,  docRtfHierarchy, },

	{ 0, 0, 0 }
    };

RtfControlWord	docRtfDocumentGroups[]=
    {
	{ "upr",	RTFidUPR,	DOClevANY,  docRtfReadUpr, },

	{ "fonttbl",	RTFidFONTTBL,	DOClevDOC,  docRtfFontTable,  },
	{ "colortbl",	RTFidCOLORTBL,	DOClevDOC,  docRtfColorTable, },
	{ "stylesheet",	RTFidSTYLESHEET,DOClevDOC,  docRtfStylesheet,  },
	/*
	{ "listtable",	RTFidLISTTABLE,	DOClevDOC,  docRtfListtable,  },
	*/
	{ "info",	RTFidINFO,	DOClevDOC,  docRtfReadInfo,  },
	{ "revtbl",	RTFidREVTBL,	DOClevDOC,  docRtfRevisionTable,  },
	{ "pn",		RTFidPN,	DOClevANY,  docRtfSkipPn,   },
	{ "pntext",	RTFidPNTEXT,	DOClevPARA, docRtfReadPntext, },
	{ "pict",	RTFidPICT,	DOClevPARA, docRtfReadPict, },
	{ "nonshppict",	RTFidNONSHPPICT,DOClevPARA, docRtfReadNonshppict, },
	{ "shppict",	RTFidSHPPICT,	DOClevPARA, docRtfReadShppict, },
	{ "object",	RTFidOBJECT,	DOClevPARA, docRtfReadObject, },
	{ "field",	RTFidFIELD,	DOClevPARA, docRtfReadField, },
	{ "shp",	RTFidSHP,	DOClevPARA, docRtfReadShape, },
	{ "shpgrp",	RTFidSHPGRP,	DOClevPARA, docRtfReadShape, },
	{ "xe",		RTFidXE,	DOClevPARA, docRtfReadLookupEntry, },
	{ "tc",		RTFidTC,	DOClevPARA, docRtfReadLookupEntry, },
	{ "tcn",	RTFidTCN,	DOClevPARA, docRtfReadLookupEntry, },
				/****************************************/
				/*  Headers and Footers.		*/
				/****************************************/
	{ "footer",	RTFidFOOTER,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "footerf",	RTFidFOOTERF,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "footerl",	RTFidFOOTERL,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "footerr",	RTFidFOOTERR,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "header",	RTFidHEADER,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "headerf",	RTFidHEADERF,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "headerl",	RTFidHEADERL,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "headerr",	RTFidHEADERR,	DOClevSECT, docRtfReadHeaderFooter, },

	{ "footnote",	RTFidFOOTNOTE,	DOClevPARA, docRtfReadFootnote, },

				/****************************************/
				/*  Multilevel Lists.			*/
				/****************************************/
	{ "pnseclvl",	RTFidPNSECLVL,	DOClevANY, docRtfReadPnseclvl,	},
				/****************************************/
				/*  Lists.				*/
				/****************************************/
	{ "listtext",	RTFidLISTTEXT,	DOClevANY, docRtfReadListtext, },
				/****************************************/
				/*  Annotation.				*/
				/****************************************/
	{ "chatn",	RTFidCHATN,	DOClevPARA, docRtfSkipGroup, },
				/****************************************/
				/*  Bookmarks				*/
				/****************************************/
	{ "bkmkstart",	RTFidBKMKSTART,	DOClevPARA, docRtfBkmkStart,	},
	{ "bkmkend",	RTFidBKMKEND,	DOClevPARA, docRtfBkmkEnd,	},

	{ 0, 0, 0 }
    };

/************************************************************************/
/*									*/
/*  Read a document as a whole.						*/
/*									*/
/************************************************************************/

int docRtfTextParticule(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    {
    if  ( rrc->rrcInDeletedText )
	{ return 0;	}

    if  ( rrc->rrcJustAfterPntext )
	{
	if  ( docCopyParagraphProperties( &(rrc->rrcBi->biParaProperties),
					 &(rrc->rrcParagraphProperties) ) )
	    { LDEB(1); return -1;	}

	if  ( rrc->rrcPageAfterPntext )
	    { rrc->rrcBi->biParaProperties.ppStartsOnNewPage= 1;	}

	rrc->rrcJustAfterPntext= 0;
	rrc->rrcPageAfterPntext= 0;
	}

    return docSaveParticules( rrc->rrcBi,
	    rrc->rrcCurrentTextAttribute, rrc->rrcInputMapping, text, len );
    }

static int docRtfReadDoc(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfTextParticule );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static int docRtfReadUpr(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfTextParticule );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static int docRtfReadListtext(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfTextParticule );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

/************************************************************************/
/*									*/
/*  Consume the description of a Paragraph Numbering level for a	*/
/*  section.								*/
/*									*/
/************************************************************************/

static int docRtfPndescText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    {
    unsigned char *	target;
    int			olen;

    switch( rrc->rrcWhichPnText )
	{
	case RTFidPNTXTA:
	    target= rrc->rrcParagraphNumber.pnTextAfter;
	    break;

	case RTFidPNTXTB:
	    target= rrc->rrcParagraphNumber.pnTextBefore;
	    break;

	default:
	    LDEB(rrc->rrcWhichPnText); return -1;
	}

    olen= strlen( (char *)target );
    if  ( olen+ len >= 32 )
	{ SLDEB((char *)target,len); return -1;	}

    memcpy( target+ olen, text, len );
    target[olen+ len]= '\0';

    return 0;
    }

static int docRtfPndescGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int			res;

    rrc->rrcWhichPnText= rcw->rcwId;

    res= docRtfReadGroup( sis, DOClevANY,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfPndescText );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    rrc->rrcWhichPnText= -1;

    return res;
    }


RtfControlWord	docRtfPndescGroups[]=
    {
	{ "pntxta",	RTFidPNTXTA,	DOClevANY, docRtfPndescGroup, },
	{ "pntxtb",	RTFidPNTXTB,	DOClevANY, docRtfPndescGroup, },

	{ 0, 0, 0 }
    };

static int docRtfReadPnseclvl(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int			res;
    int			level= arg- 1;

    if  ( level < 0 )
	{ LDEB(level);	}

    docInitParagraphNumber( &(rrc->rrcParagraphNumber) );

    res= docRtfReadGroup( sis, DOClevANY,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfPndescGroups,
				docRtfRefuseText );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res); return res;	}

    rrc->rrcParagraphNumber.pnLevel= level;

    if  ( level >= rrc->rrcParagraphNumberCount )
	{
	ParagraphNumber *	fresh;

	fresh= realloc( rrc->rrcParagraphNumbers,
				( level+ 1 )* sizeof( ParagraphNumber ) );
	if  ( ! fresh )
	    { LXDEB(level,fresh); return -1;	}

	rrc->rrcParagraphNumbers= fresh;

	while( rrc->rrcParagraphNumberCount <= level )
	    {
	    docInitParagraphNumber( rrc->rrcParagraphNumbers+ 
					    rrc->rrcParagraphNumberCount++ );
	    }
	}

    rrc->rrcParagraphNumbers[level]= rrc->rrcParagraphNumber;

    return res;
    }

static RtfControlWord	docRtfPnWords[]=
    {
				/****************************************/
				/*  Bullets and Numbering.		*/
				/*  (No destinations)			*/
				/****************************************/
	{ "pnacross",	RTFidPNACROSS,	DOClevANY, docRtfIgnoreWord, },
	{ "pncf",	RTFidPNCF,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlvl",	RTFidPNLVL,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlvlblt",	RTFidPNLVLBLT,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlvlbody",	RTFidPNLVLBODY,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlvlcont",	RTFidPNLVLCONT,	DOClevANY, docRtfIgnoreWord, },
	{ "pnnumonce",	RTFidPNNUMONCE,	DOClevANY, docRtfIgnoreWord, },
	{ "pnprev",	RTFidPNPREV,	DOClevANY, docRtfIgnoreWord, },
	{ "pnqc",	RTFidPNQC,	DOClevANY, docRtfIgnoreWord, },
	{ "pnql",	RTFidPNQL,	DOClevANY, docRtfIgnoreWord, },
	{ "pnqr",	RTFidPNQR,	DOClevANY, docRtfIgnoreWord, },
	{ "pnrestart",	RTFidPNRESTART,	DOClevANY, docRtfIgnoreWord, },
	{ "pnsp",	RTFidPNSP,	DOClevANY, docRtfIgnoreWord, },

	{ "ls",		RTFidLS,	DOClevANY, docRtfIgnoreWord, },
	{ "ilvl",	RTFidILVL,	DOClevANY, docRtfIgnoreWord, },
	{ "pnrnot",	RTFidPNRNOT,	DOClevANY, docRtfIgnoreWord, },

	{ 0, 0, 0 }
    };

static int docRtfSkipPn(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int		res;

    docInitParagraphNumber( &(rrc->rrcParagraphNumber) );

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfPnWords, docRtfPndescGroups,
				docRtfRefuseText );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

/************************************************************************/
/*									*/
/*  Read the result of a bullet.					*/
/*									*/
/*  For one reason or another, Word resets the paragraph properties	*/
/*  inside the result. So save and restore them.			*/
/*									*/
/************************************************************************/

static int docRtfReadPntext(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int				res;
    ParagraphProperties		pp;

    docInitParagraphProperties( &pp );
    docCopyParagraphProperties( &pp, &(rrc->rrcBi->biParaProperties) );

    res= docRtfReadGroup(	sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfEmptyTable,
				docRtfTextParticule );

    docCopyParagraphProperties( &(rrc->rrcBi->biParaProperties), &pp );
    docCleanParagraphProperties( &pp );

    rrc->rrcJustAfterPntext= 1;
    rrc->rrcPageAfterPntext= 0;

    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

/************************************************************************/
/*									*/
/*  Read a whole document.						*/
/*									*/
/************************************************************************/

static RtfControlWord	docRtfOutsideGroups[]=
    {
	{ "rtf",	RTFidRTF,	DOClevDOC, docRtfReadDoc,	},
	{ 0, 0, 0 }
    };

BufferDocument * docRtfReadFile(	SimpleInputStream *	sis )
    {
    BufferDocument *		bd;
    RtfReadingContext		rrc;

    int				res;
    const RtfControlWord *	rcw;

    char			controlWord[TEDszRTFCONTROL+1];
    int				gotArg;
    int				arg= -1;
    int				c;

    bd= (BufferDocument *)malloc( sizeof(BufferDocument) );
    if  ( ! bd )
	{ XDEB(bd); return bd;	}

    docInitDocument( bd );

    docRtfInitReadingContext( &rrc );

    rrc.rrcBd= bd;
    rrc.rrcBi= &(bd->bdItem);
    rrc.rrcLevel= DOClevDOC;
    rrc.rrcSplitLevel= DOClevDOC;

    res= docRtfFindControl( sis, &rrc, &c, controlWord, &gotArg, &arg );

    if  ( res != RTFfiCTRLGROUP )
	{ docFreeDocument( bd ); return (BufferDocument *)0; }

    rcw= docRtfFindWord( controlWord, docRtfOutsideGroups, 1 );
    if  ( ! rcw )
	{ LDEB(1); docFreeDocument( bd ); return (BufferDocument *)0; }

    res= docRtfApplyControlWord( sis, rcw, gotArg, arg, &rrc );
    if  ( res )
	{ LDEB(1); docFreeDocument( bd ); return (BufferDocument *)0; }

    docRtfCleanReadingContext( &rrc );

    return bd;
    }

/************************************************************************/
/*									*/
/*  Just read a ruler from RTF.						*/
/*									*/
/************************************************************************/
static int docRtfReadRulerGroup(	SimpleInputStream *	sis,
					const RtfControlWord *	rcw,
					int			arg,
					RtfReadingContext *	rrc	)
    {
    int		res;

    res= docRtfConsumeGroup( sis, DOClevPARA, rrc,
					    docRtfEmptyTable, NULL, NULL );

    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static RtfControlWord	docRtfRulerOutsideGroups[]=
    {
	{ "ruler",	RTFidRULER,	DOClevSECT, docRtfReadRulerGroup, },
	{ 0, 0, 0 }
    };

int docRtfReadRuler(	SimpleInputStream *	sis,
			ParagraphProperties *	pp )
    {
    BufferItem			bi;
    RtfReadingContext		rrc;

    int				res;
    const RtfControlWord *	rcw;

    char			controlWord[TEDszRTFCONTROL+1];
    int				gotArg;
    int				arg= -1;
    int				c;

    docInitItem( &bi, (BufferItem *)0, (BufferDocument *)0, 0, DOClevCELL );
    docRtfInitReadingContext( &rrc );

    rrc.rrcBi= &bi;
    rrc.rrcLevel= bi.biLevel;
    rrc.rrcSplitLevel= bi.biLevel;

    res= docRtfFindControl( sis, &rrc, &c, controlWord, &gotArg, &arg );

    if  ( res != RTFfiCTRLGROUP )
	{ LDEB(1); docCleanItem( (BufferDocument *)0, &bi ); return -1; }

    rcw= docRtfFindWord( controlWord, docRtfRulerOutsideGroups, 1 );
    if  ( ! rcw )
	{ LDEB(1); docCleanItem( (BufferDocument *)0, &bi ); return -1; }

    res= docRtfApplyControlWord( sis, rcw, gotArg, arg, &rrc );
    if  ( res )
	{ LDEB(1); docCleanItem( (BufferDocument *)0, &bi ); return -1; }

    docCopyParagraphProperties( pp, &(rrc.rrcParagraphProperties) );

    docCleanItem( (BufferDocument *)0, &bi );
    docRtfCleanReadingContext( &rrc );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Remember Document Properties.					*/
/*									*/
/************************************************************************/

int docRtfRememberDocProperty(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    DocumentProperties *	dp= &(rrc->rrcBd->bdProperties);
    SectionProperties *		sp= &(rrc->rrcSectionProperties);

    switch( rcw->rcwId )
	{
	case	RTFidMARGL:
	    dp->dpGeometry.dgLeftMarginTwips= arg;
	    sp->spDocumentGeometry.dgLeftMarginTwips= arg;
	    break;
	case	RTFidMARGT:
	    dp->dpGeometry.dgTopMarginTwips= arg;
	    sp->spDocumentGeometry.dgTopMarginTwips= arg;
	    break;
	case	RTFidMARGR:
	    dp->dpGeometry.dgRightMarginTwips= arg;
	    sp->spDocumentGeometry.dgRightMarginTwips= arg;
	    break;
	case	RTFidMARGB:
	    dp->dpGeometry.dgBottomMarginTwips= arg;
	    sp->spDocumentGeometry.dgBottomMarginTwips= arg;
	    break;
	case	RTFidPAPERW:
	    dp->dpGeometry.dgPageWideTwips= arg;
	    sp->spDocumentGeometry.dgPageWideTwips= arg;
	    break;
	case	RTFidPAPERH:
	    dp->dpGeometry.dgPageHighTwips= arg;
	    sp->spDocumentGeometry.dgPageHighTwips= arg;
	    break;
	case	RTFidGUTTER:
	    dp->dpGeometry.dgGutterTwips= arg;
	    sp->spDocumentGeometry.dgGutterTwips= arg;
	    break;

	case	RTFidFACINGP:
	    dp->dpHasFacingPages= 1;
	    break;
	case RTFidPGNSTART:
	    dp->dpStartPageNumber= arg;
	    break;
	case	RTFidDEFTAB:
	    dp->dpTabIntervalTwips= arg;
	    break;
	case	RTFidDEFF:
	    dp->dpDefaultFont= arg;
	    rrc->rrcCurrentTextAttribute.taFontNumber= arg;
	    break;

	case RTFidFET:
	    if  ( arg < 0 || arg > 2 )
		{ LDEB(arg); break;	}
	    dp->dpFootEndNoteType= arg;
	    break;
	case RTFidENDNOTES:
	    dp->dpFootnotePosition= DPftnPOS_SECT_END;
	    break;
	case RTFidENDDOC:
	    dp->dpFootnotePosition= DPftnPOS_DOC_END;
	    break;
	case RTFidFTNTJ:
	    dp->dpFootnotePosition= DPftnPOS_BELOW_TEXT;
	    break;
	case RTFidFTNBJ:
	    dp->dpFootnotePosition= DPftnPOS_PAGE_BOTTOM;
	    break;
	case RTFidAENDNOTES:
	    dp->dpEndnotePosition= DPftnPOS_SECT_END;
	    break;
	case RTFidAENDDOC:
	    dp->dpEndnotePosition= DPftnPOS_DOC_END;
	    break;
	case RTFidAFTNTJ:
	    dp->dpEndnotePosition= DPftnPOS_BELOW_TEXT;
	    break;
	case RTFidAFTNBJ:
	    dp->dpEndnotePosition= DPftnPOS_PAGE_BOTTOM;
	    break;
	case RTFidFTNSTART:
	    dp->dpStartFootnoteNumber= arg;
	    break;
	case RTFidAFTNSTART:
	    dp->dpStartEndnoteNumber= arg;
	    break;
	case RTFidFTNRSTCONT:
	    dp->dpFootnotesRestart= DPftnRST_CONTINUOUS;
	    break;
	case RTFidFTNRESTART:
	    dp->dpFootnotesRestart= DPftnRST_PER_SECTION;
	    break;
	case RTFidFTNRSTPG:
	    dp->dpFootnotesRestart= DPftnRST_PER_PAGE;
	    break;
	case RTFidAFTNRSTCONT:
	    dp->dpEndnotesRestart= DPftnRST_CONTINUOUS;
	    break;
	case RTFidAFTNRESTART:
	    dp->dpEndnotesRestart= DPftnRST_PER_SECTION;
	    break;
	case RTFidFTNNAR:
	    dp->dpFootnoteNumberStyle= DPftnNAR;
	    break;
	case RTFidFTNNALC:
	    dp->dpFootnoteNumberStyle= DPftnNALC;
	    break;
	case RTFidFTNNAUC:
	    dp->dpFootnoteNumberStyle= DPftnNAUC;
	    break;
	case RTFidFTNNRLC:
	    dp->dpFootnoteNumberStyle= DPftnNRLC;
	    break;
	case RTFidFTNNRUC:
	    dp->dpFootnoteNumberStyle= DPftnNRUC;
	    break;
	case RTFidFTNNCHI:
	    dp->dpFootnoteNumberStyle= DPftnNCHI;
	    break;
	case RTFidAFTNNAR:
	    dp->dpEndnoteNumberStyle= DPftnNAR;
	    break;
	case RTFidAFTNNALC:
	    dp->dpEndnoteNumberStyle= DPftnNALC;
	    break;
	case RTFidAFTNNAUC:
	    dp->dpEndnoteNumberStyle= DPftnNAUC;
	    break;
	case RTFidAFTNNRLC:
	    dp->dpEndnoteNumberStyle= DPftnNRLC;
	    break;
	case RTFidAFTNNRUC:
	    dp->dpEndnoteNumberStyle= DPftnNRUC;
	    break;
	case RTFidAFTNNCHI:
	    dp->dpEndnoteNumberStyle= DPftnNCHI;
	    break;

	case RTFidANSICPG:
	    switch( arg )
		{
		case 1250:
		    memcpy( rrc->rrcInputMapping, docWIN1250_to_ISO2, 256 );
		    dp->dpAnsiCodePage= arg;
		    break;

		case 1251:
		    memcpy( rrc->rrcInputMapping, docWIN1251_to_KOI8R, 256 );
		    dp->dpAnsiCodePage= arg;
		    break;

		case 1252:
		    dp->dpAnsiCodePage= arg;
		    break;

		default:
		    SLDEB(rcw->rcwWord,arg); break;
		}
	    break;

	default:
	    break;
	}
    
    return 0;
    }

/************************************************************************/
/*									*/
/*  Remember properties to be used subsequently.			*/
/*									*/
/*  The level should be DOClevANY.					*/
/*									*/
/*  As this is the way Word parses RTF, and this makes an RTF reader	*/
/*  insensitive to dubious RTF, this is the way to go in further	*/
/*  extensions.								*/
/*									*/
/************************************************************************/

/* Not practical
# define rrcBd do not use
*/
# define rrcBi do not use
# define rrcInsertedObject do not use
# define rrcFieldNumber do not use

# define	RTF_REMEMBER(s,id) \
    { s,      id,         DOClevANY,  docRtfRememberProperty, }

# define	RTF_DOCPROP(s,id) \
    { s,      id,         DOClevANY,  docRtfRememberDocProperty, }

# define	RTF_BORDER(s,id) \
    { s,	id,	DOClevANY,  docRtfRememberProperty, \
				    docRtfBeginBorder, docRtfBorderWords, }

RtfControlWord	docRtfPropertyWords[]=
{
				/****************************************/
				/*  Text properties.			*/
				/*  Character Formatting Properties	*/
				/****************************************/
    RTF_REMEMBER( "b",			RTFidB ),
    RTF_REMEMBER( "caps",		RTFidCAPS ),
    RTF_REMEMBER( "cf",			RTFidCF ),
    RTF_REMEMBER( "f",			RTFidF ),
    RTF_REMEMBER( "fs",			RTFidFS ),
    RTF_REMEMBER( "i",			RTFidI ),
    RTF_REMEMBER( "lang",		RTFidLANG ),
    RTF_REMEMBER( "noproof",		RTFidNOPROOF ),
    RTF_REMEMBER( "plain",		RTFidPLAIN ),
    RTF_REMEMBER( "ul",			RTFidUL ),
    RTF_REMEMBER( "ulnone",		RTFidULNONE ),
    RTF_REMEMBER( "up",			RTFidUP ),
    RTF_REMEMBER( "cgrid",		RTFidCGRID ),

    RTF_REMEMBER( "cb",			RTFidCB ),
    RTF_REMEMBER( "cchs",		RTFidCCHS ),
    RTF_REMEMBER( "cs",			RTFidCS ),
    RTF_REMEMBER( "deleted",		RTFidDELETED ),
    RTF_REMEMBER( "dn",			RTFidDN ),
    RTF_REMEMBER( "ltrch",		RTFidLTRCH ),
    RTF_REMEMBER( "nosupersub",		RTFidNOSUPERSUB ),
    RTF_REMEMBER( "outl",		RTFidOUTL ),
    RTF_REMEMBER( "revauth",		RTFidREVAUTH ),
    RTF_REMEMBER( "revauthdel",		RTFidREVAUTHDEL ),
    RTF_REMEMBER( "revdttm",		RTFidREVDTTM ),
    RTF_REMEMBER( "revdttmdel",		RTFidREVDTTMDEL ),
    RTF_REMEMBER( "revised",		RTFidREVISED ),
    RTF_REMEMBER( "rtlch",		RTFidRTLCH ),
    RTF_REMEMBER( "shad",		RTFidSHAD ),

    RTF_REMEMBER( "scaps",		RTFidSCAPS ),
    RTF_REMEMBER( "strike",		RTFidSTRIKE ),
    RTF_REMEMBER( "sub",		RTFidSUB ),
    RTF_REMEMBER( "super",		RTFidSUPER ),
    RTF_REMEMBER( "uld",		RTFidULD ),
    RTF_REMEMBER( "uldb",		RTFidULDB ),
    RTF_REMEMBER( "ulw",		RTFidULW ),
    RTF_REMEMBER( "v",			RTFidV ),
    RTF_REMEMBER( "kerning",		RTFidKERNING ),
    RTF_REMEMBER( "expnd",		RTFidEXPND ),
    RTF_REMEMBER( "expndtw",		RTFidEXPNDTW ),
    RTF_REMEMBER( "charscalex",		RTFidCHARSCALEX ),
    RTF_REMEMBER( "highlight",		RTFidHIGHLIGHT ),

    RTF_REMEMBER( "chbghoriz",		RTFidCHBGHORIZ ),
    RTF_REMEMBER( "chbgvert",		RTFidCHBGVERT ),
    RTF_REMEMBER( "chbgfdiag",		RTFidCHBGFDIAG ),
    RTF_REMEMBER( "chbgbdiag",		RTFidCHBGBDIAG ),
    RTF_REMEMBER( "chbgcross",		RTFidCHBGCROSS ),
    RTF_REMEMBER( "chbgdcross",		RTFidCHBGDCROSS ),
    RTF_REMEMBER( "chbgdkhoriz",	RTFidCHBGDKHORIZ ),
    RTF_REMEMBER( "chbgdkvert",		RTFidCHBGDKVERT ),
    RTF_REMEMBER( "chbgdkfdiag",	RTFidCHBGDKFDIAG ),
    RTF_REMEMBER( "chbgdkbdiag",	RTFidCHBGDKBDIAG ),
    RTF_REMEMBER( "chbgdkcross",	RTFidCHBGDKCROSS ),
    RTF_REMEMBER( "chbgdkdcross",	RTFidCHBGDKDCROSS ),

    RTF_BORDER(   "chbrdr",		RTFidCHBRDR ),

    RTF_REMEMBER( "chcbpat",		RTFidCHCBPAT ),
    RTF_REMEMBER( "chcfpat",		RTFidCHCFPAT ),
    RTF_REMEMBER( "chshdng",		RTFidCHSHDNG ),

    RTF_REMEMBER( "faauto",		RTFidFAAUTO ),

				/****************************************/
				/*  Paragraph properties.		*/
				/****************************************/
    RTF_REMEMBER( "pard",		RTFidPARD ),
    RTF_REMEMBER( "s",			RTFidS ),
    RTF_REMEMBER( "fi",			RTFidFI ),
    RTF_REMEMBER( "li",			RTFidLI ),
    RTF_REMEMBER( "ri",			RTFidRI ),
    RTF_REMEMBER( "ql",			RTFidQL ),
    RTF_REMEMBER( "qr",			RTFidQR ),
    RTF_REMEMBER( "qj",			RTFidQJ ),
    RTF_REMEMBER( "qc",			RTFidQC ),
    RTF_REMEMBER( "sa",			RTFidSA ),
    RTF_REMEMBER( "sb",			RTFidSB ),
    RTF_REMEMBER( "intbl",		RTFidINTBL ),
    RTF_REMEMBER( "hyphpar",		RTFidHYPHPAR ),

    RTF_REMEMBER( "pagebb",		RTFidPAGEBB ),

    RTF_REMEMBER( "keep",		RTFidKEEP ),
    RTF_REMEMBER( "keepn",		RTFidKEEPN ),
    RTF_REMEMBER( "outlinelevel",	RTFidOUTLINELEVEL ),

    RTF_REMEMBER( "sl",			RTFidSL ),
    RTF_REMEMBER( "slmult",		RTFidSLMULT ),
    RTF_REMEMBER( "nowidctlpar",	RTFidNOWIDCTLPAR ),
    RTF_REMEMBER( "widctlpar",		RTFidWIDCTLPAR ),
    RTF_REMEMBER( "adjustright",	RTFidADJUSTRIGHT ),
    RTF_REMEMBER( "adjustleft",		RTFidADJUSTLEFT ),

    RTF_REMEMBER( "sbauto",		RTFidSBAUTO ),
    RTF_REMEMBER( "saauto",		RTFidSAAUTO ),

    RTF_REMEMBER( "tql",		RTFidTQL ), /*?*/

    RTF_REMEMBER( "jclisttab",		RTFidJCLISTTAB ),
    RTF_REMEMBER( "nooverflow",		RTFidNOOVERFLOW ),
    RTF_REMEMBER( "faroman",		RTFidFAROMAN ),

    RTF_REMEMBER( "aspalpha",		RTFidASPALPHA ),
    RTF_REMEMBER( "aspnum",		RTFidASPNUM ),

				/****************************************/
				/*  Paragraph Shading.			*/
				/****************************************/
    RTF_REMEMBER( "bgbdiag",		RTFidBGBDIAG ),
    RTF_REMEMBER( "bgcross",		RTFidBGCROSS ),
    RTF_REMEMBER( "bgdcross",		RTFidBGDCROSS ),
    RTF_REMEMBER( "bgdkbdiag",		RTFidBGDKBDIAG ),
    RTF_REMEMBER( "bgdkcross",		RTFidBGDKCROSS ),
    RTF_REMEMBER( "bgdkdcross",		RTFidBGDKDCROSS ),
    RTF_REMEMBER( "bgdkfdiag",		RTFidBGDKFDIAG ),
    RTF_REMEMBER( "bgdkhoriz",		RTFidBGDKHORIZ ),
    RTF_REMEMBER( "bgdkvert",		RTFidBGDKVERT ),
    RTF_REMEMBER( "bgfdiag",		RTFidBGFDIAG ),
    RTF_REMEMBER( "bghoriz",		RTFidBGHORIZ ),
    RTF_REMEMBER( "bgvert",		RTFidBGVERT ),
    RTF_REMEMBER( "cbpat",		RTFidCBPAT ),
    RTF_REMEMBER( "cfpat",		RTFidCFPAT ),
    RTF_REMEMBER( "shading",		RTFidSHADING ),
				/****************************************/
				/*  Paragraph Borders.			*/
				/****************************************/
    RTF_BORDER( "brdrt",		RTFidBRDRT ),
    RTF_BORDER( "brdrb",		RTFidBRDRB ),
    RTF_BORDER( "brdrl",		RTFidBRDRL ),
    RTF_BORDER( "brdrr",		RTFidBRDRR ),
    RTF_BORDER( "brdrbtw",		RTFidBRDRBTW ),
    RTF_BORDER( "brdrbar",		RTFidBRDRBAR ),
    RTF_BORDER( "box",			RTFidBOX ),
				/****************************************/
				/*  Positioned Objects and Frames:	*/
				/*  ( Paragraph properties. )		*/
				/****************************************/
    RTF_REMEMBER( "absh",		RTFidABSH ),
    RTF_REMEMBER( "abslock",		RTFidABSLOCK ),
    RTF_REMEMBER( "absw",		RTFidABSW ),
    RTF_REMEMBER( "dfrmtxtx",		RTFidDFRMTXTX ),
    RTF_REMEMBER( "dfrmtxty",		RTFidDFRMTXTY ),
    RTF_REMEMBER( "dropcapli",		RTFidDROPCAPLI ),
    RTF_REMEMBER( "dropcapt",		RTFidDROPCAPT ),
    RTF_REMEMBER( "dxfrtext",		RTFidDXFRTEXT ),
    RTF_REMEMBER( "nowrap",		RTFidNOWRAP ),
    RTF_REMEMBER( "phcol",		RTFidPHCOL ),
    RTF_REMEMBER( "phmrg",		RTFidPHMRG ),
    RTF_REMEMBER( "phpg",		RTFidPHPG ),
    RTF_REMEMBER( "posnegx",		RTFidPOSNEGX ),
    RTF_REMEMBER( "posnegy",		RTFidPOSNEGY ),
    RTF_REMEMBER( "posx",		RTFidPOSX ),
    RTF_REMEMBER( "posxc",		RTFidPOSXC ),
    RTF_REMEMBER( "posxi",		RTFidPOSXI ),
    RTF_REMEMBER( "posxl",		RTFidPOSXL ),
    RTF_REMEMBER( "posxo",		RTFidPOSXO ),
    RTF_REMEMBER( "posxr",		RTFidPOSXR ),
    RTF_REMEMBER( "posy",		RTFidPOSY ),
    RTF_REMEMBER( "posyb",		RTFidPOSYB ),
    RTF_REMEMBER( "posyc",		RTFidPOSYC ),
    RTF_REMEMBER( "posyil",		RTFidPOSYIL ),
    RTF_REMEMBER( "posyt",		RTFidPOSYT ),
    RTF_REMEMBER( "pvmrg",		RTFidPVMRG ),
    RTF_REMEMBER( "pvpara",		RTFidPVPARA ),
    RTF_REMEMBER( "pvpg",		RTFidPVPG ),
				/****************************************/
				/*  Bullets and Numbering.		*/
				/*  (No destinations)			*/
				/****************************************/
    RTF_REMEMBER( "ls",			RTFidLS ),
    RTF_REMEMBER( "ilvl",		RTFidILVL ),
				/****************************************/
				/*  Revision marks for paragraph	*/
				/*  numbers.				*/
				/****************************************/
    RTF_REMEMBER( "pnrauth",		RTFidPNRAUTH ),
    RTF_REMEMBER( "pnrdate",		RTFidPNRDATE ),
    RTF_REMEMBER( "pnrnfc",		RTFidPNRNFC ),
    RTF_REMEMBER( "pnrpnbr",		RTFidPNRPNBR ),
    RTF_REMEMBER( "pnrrgb",		RTFidPNRRGB ),
    RTF_REMEMBER( "pnrstart",		RTFidPNRSTART ),
    RTF_REMEMBER( "pnrstop",		RTFidPNRSTOP ),
    RTF_REMEMBER( "pnrxst",		RTFidPNRXST ),
				/****************************************/
				/*  Tabs.				*/
				/****************************************/
    RTF_REMEMBER( "tx",			RTFidTX ),
    RTF_REMEMBER( "tb",			RTFidTB ),
    RTF_REMEMBER( "tldot",		RTFidTLDOT ),
    RTF_REMEMBER( "tleq",		RTFidTLEQ ),
    RTF_REMEMBER( "tlhyph",		RTFidTLHYPH ),
    RTF_REMEMBER( "tlth",		RTFidTLTH ),
    RTF_REMEMBER( "tlul",		RTFidTLUL ),
    RTF_REMEMBER( "tqc",		RTFidTQC ),
    RTF_REMEMBER( "tqdec",		RTFidTQDEC ),
    RTF_REMEMBER( "tqr",		RTFidTQR ),
				/****************************************/
				/*  Associated Character Properties	*/
				/****************************************/
    RTF_REMEMBER( "ab",			RTFidAB ),
    RTF_REMEMBER( "acaps",		RTFidACAPS ),
    RTF_REMEMBER( "acf",		RTFidACF ),
    RTF_REMEMBER( "adn",		RTFidADN ),
    RTF_REMEMBER( "aexpnd",		RTFidAEXPND ),
    RTF_REMEMBER( "af",			RTFidAF ),
    RTF_REMEMBER( "afs",		RTFidAFS ),
    RTF_REMEMBER( "ai",			RTFidAI ),
    RTF_REMEMBER( "alang",		RTFidALANG ),
    RTF_REMEMBER( "aoutl",		RTFidAOUTL ),
    RTF_REMEMBER( "ascaps",		RTFidASCAPS ),
    RTF_REMEMBER( "ashad",		RTFidASHAD ),
    RTF_REMEMBER( "astrike",		RTFidASTRIKE ),
    RTF_REMEMBER( "aul",		RTFidAUL ),
    RTF_REMEMBER( "auld",		RTFidAULD ),
    RTF_REMEMBER( "auldb",		RTFidAULDB ),
    RTF_REMEMBER( "aulnone",		RTFidAULNONE ),
    RTF_REMEMBER( "aulw",		RTFidAULW ),
    RTF_REMEMBER( "aup",		RTFidAUP ),
    RTF_REMEMBER( "loch",		RTFidLOCH ),
    RTF_REMEMBER( "hich",		RTFidHICH ),
    RTF_REMEMBER( "dbch",		RTFidDBCH ),
				/****************************************/
				/*  Row, Cell/Column properties.	*/
				/****************************************/
    RTF_REMEMBER( "trowd",		RTFidTROWD ),

    RTF_REMEMBER( "ltrrow",		RTFidLTRROW ),
    RTF_REMEMBER( "rtlrow",		RTFidRTLROW ),

    RTF_REMEMBER( "trgaph",		RTFidTRGAPH ),
    RTF_REMEMBER( "trhdr",		RTFidTRHDR ),
    RTF_REMEMBER( "trkeep",		RTFidTRKEEP ),
    RTF_REMEMBER( "trleft",		RTFidTRLEFT ),
    RTF_REMEMBER( "trqc",		RTFidTRQC ),
    RTF_REMEMBER( "trql",		RTFidTRQL ),
    RTF_REMEMBER( "trqr",		RTFidTRQR ),
    RTF_REMEMBER( "trrh",		RTFidTRRH ),

    RTF_REMEMBER( "trpaddl",		RTFidTRPADDL ),
    RTF_REMEMBER( "trpaddr",		RTFidTRPADDR ),
    RTF_REMEMBER( "trpaddfl",		RTFidTRPADDFL ),
    RTF_REMEMBER( "trpaddfr",		RTFidTRPADDFR ),
    RTF_REMEMBER( "trwWidth",		RTFidTRWWIDTH ),
    RTF_REMEMBER( "trftsWidth",		RTFidTRFTSWIDTH ),

    RTF_REMEMBER( "cellx",		RTFidCELLX ),

    RTF_REMEMBER( "celld",		RTFidCELLD ),
    RTF_REMEMBER( "cltxlrtb",		RTFidCLTXLRTB ),
    RTF_REMEMBER( "cltxtbrl",		RTFidCLTXTBRL ),
    RTF_REMEMBER( "clvertalt",		RTFidCLVERTALT ),
    RTF_REMEMBER( "clvertalc",		RTFidCLVERTALC ),
    RTF_REMEMBER( "clvertalb",		RTFidCLVERTALB ),
    RTF_REMEMBER( "clbgbdiag",		RTFidCLBGBDIAG ),
    RTF_REMEMBER( "clbgcross",		RTFidCLBGCROSS ),
    RTF_REMEMBER( "clbgdcross",		RTFidCLBGDCROSS ),
    RTF_REMEMBER( "clbgdkbdiag",	RTFidCLBGDKBDIAG ),
    RTF_REMEMBER( "clbgdkcross",	RTFidCLBGDKCROSS ),
    RTF_REMEMBER( "clbgdkdcross",	RTFidCLBGDKDCROSS ),
    RTF_REMEMBER( "clbgdkfdiag",	RTFidCLBGDKFDIAG ),
    RTF_REMEMBER( "clbgdkhor",		RTFidCLBGDKHOR ),
    RTF_REMEMBER( "clbgdkvert",		RTFidCLBGDKVERT ),
    RTF_REMEMBER( "clbgfdiag",		RTFidCLBGFDIAG ),
    RTF_REMEMBER( "clbghoriz",		RTFidCLBGHORIZ ),
    RTF_REMEMBER( "clbgvert",		RTFidCLBGVERT ),
    RTF_REMEMBER( "clmgf",		RTFidCLMGF ),
    RTF_REMEMBER( "clmrg",		RTFidCLMRG ),
    RTF_REMEMBER( "clvmgf",		RTFidCLVMGF ),
    RTF_REMEMBER( "clvmrg",		RTFidCLVMRG ),
    RTF_REMEMBER( "clcbpat",		RTFidCLCBPAT ),
    RTF_REMEMBER( "clcfpat",		RTFidCLCFPAT ),
    RTF_REMEMBER( "clshdng",		RTFidCLSHDNG ),

    RTF_REMEMBER( "clwWidth",		RTFidCLWWIDTH ),
    RTF_REMEMBER( "clftsWidth",		RTFidCLFTSWIDTH ),
				/****************************************/
				/*  Row/Cell borders.			*/
				/****************************************/
    RTF_BORDER( "trbrdrb",		RTFidTRBRDRB ),
    RTF_BORDER( "trbrdrh",		RTFidTRBRDRH ),
    RTF_BORDER( "trbrdrl",		RTFidTRBRDRL ),
    RTF_BORDER( "trbrdrr",		RTFidTRBRDRR ),
    RTF_BORDER( "trbrdrt",		RTFidTRBRDRT ),
    RTF_BORDER( "trbrdrv",		RTFidTRBRDRV ),

    RTF_BORDER( "clbrdrb",		RTFidCLBRDRB ),
    RTF_BORDER( "clbrdrl",		RTFidCLBRDRL ),
    RTF_BORDER( "clbrdrr",		RTFidCLBRDRR ),
    RTF_BORDER( "clbrdrt",		RTFidCLBRDRT ),
				/****************************************/
				/*  Document formatting.		*/
				/****************************************/
    RTF_REMEMBER( "pgbrdrhead",		RTFidPGBRDRHEAD ),
    RTF_REMEMBER( "pgbrdrfoot",		RTFidPGBRDRFOOT ),
    RTF_REMEMBER( "pgbrdrt",		RTFidPGBRDRT ),
    RTF_REMEMBER( "pgbrdrb",		RTFidPGBRDRB ),
    RTF_REMEMBER( "pgbrdrl",		RTFidPGBRDRL ),
    RTF_REMEMBER( "pgbrdrr",		RTFidPGBRDRR ),
    RTF_REMEMBER( "brdrart",		RTFidBRDRART ),
    RTF_REMEMBER( "pgbrdropt",		RTFidPGBRDROPT ),
    RTF_REMEMBER( "pgbrdrsnap",		RTFidPGBRDRSNAP ),

    RTF_DOCPROP( "margl",		RTFidMARGL ),
    RTF_DOCPROP( "margr",		RTFidMARGR ),
    RTF_DOCPROP( "margt",		RTFidMARGT ),
    RTF_DOCPROP( "margb",		RTFidMARGB ),
    RTF_DOCPROP( "paperh",		RTFidPAPERH ),
    RTF_DOCPROP( "paperw",		RTFidPAPERW ),
    RTF_DOCPROP( "gutter",		RTFidGUTTER ),

    RTF_REMEMBER( "psz",		RTFidPSZ ),

    RTF_REMEMBER( "jcompress",		RTFidJCOMPRESS),
    RTF_REMEMBER( "jexpand",		RTFidJEXPAND),

    RTF_REMEMBER( "horzdoc",		RTFidHORZDOC),
    RTF_REMEMBER( "dghspace",		RTFidDGHSPACE),
    RTF_REMEMBER( "dgvspace",		RTFidDGVSPACE),
    RTF_REMEMBER( "dghorigin",		RTFidDGHORIGIN),
    RTF_REMEMBER( "dgvorigin",		RTFidDGVORIGIN),
    RTF_REMEMBER( "dghshow",		RTFidDGHSHOW),
    RTF_REMEMBER( "dgvshow",		RTFidDGVSHOW),
    RTF_REMEMBER( "dgmargin",		RTFidDGMARGIN),

    RTF_DOCPROP( "pgnstart",		RTFidPGNSTART ),
    RTF_DOCPROP( "deff",		RTFidDEFF ),
    RTF_DOCPROP( "facingp",		RTFidFACINGP ),
    RTF_DOCPROP( "deftab",		RTFidDEFTAB ),

    RTF_REMEMBER( "truncex",		RTFidTRUNCEX ),
    RTF_REMEMBER( "nolead",		RTFidNOLEAD ),
    RTF_REMEMBER( "msmcap",		RTFidMSMCAP ),
    RTF_REMEMBER( "splytwnine",		RTFidSPLYTWNINE ),
    RTF_REMEMBER( "ftnlytwnine",	RTFidFTNLYTWNINE ),
    RTF_REMEMBER( "htmautsp",		RTFidHTMAUTSP ),
    RTF_REMEMBER( "useltbaln",		RTFidUSELTBALN ),
    RTF_REMEMBER( "alntblind",		RTFidALNTBLIND ),
    RTF_REMEMBER( "lytcalctblwd",	RTFidLYTCALCTBLWD ),
    RTF_REMEMBER( "lyttblrtgr",		RTFidLYTTBLRTGR ),
    RTF_REMEMBER( "lnbrkrule",		RTFidLNBRKRULE ),

    RTF_REMEMBER( "bdrrlswsix",		RTFidBDRRLSWSIX ),
    RTF_REMEMBER( "oldas",		RTFidOLDAS ),

    RTF_REMEMBER( "sprstsm",		RTFidSPRSTSM ), /* ignore */

				/****************************************/
				/*  Document properties.		*/
				/****************************************/
    RTF_REMEMBER( "noxlattoyen",	RTFidNOXLATTOYEN ),
    RTF_REMEMBER( "expshrtn",		RTFidEXPSHRTN ),
    RTF_REMEMBER( "noultrlspc",		RTFidNOULTRLSPC ),
    RTF_REMEMBER( "dntblnsbdb",		RTFidDNTBLNSBDB ),
    RTF_REMEMBER( "nospaceforul",	RTFidNOSPACEFORUL ),

				/****************************************/
				/*  Paragraph Numbering.		*/
				/****************************************/
    RTF_REMEMBER( "pncard",		RTFidPNCARD ),
    RTF_REMEMBER( "pndec",		RTFidPNDEC ),
    RTF_REMEMBER( "pnucltr",		RTFidPNUCLTR ),
    RTF_REMEMBER( "pnucrm",		RTFidPNUCRM ),
    RTF_REMEMBER( "pnlcltr",		RTFidPNLCLTR ),
    RTF_REMEMBER( "pnlcrm",		RTFidPNLCRM ),
    RTF_REMEMBER( "pnord",		RTFidPNORD ),
    RTF_REMEMBER( "pnordt",		RTFidPNORDT ),

    RTF_REMEMBER( "pnstart",		RTFidPNSTART ),
    RTF_REMEMBER( "pnindent",		RTFidPNINDENT ),
    RTF_REMEMBER( "pnhang",		RTFidPNHANG ),

    RTF_REMEMBER( "pnf",		RTFidPNF ),
    RTF_REMEMBER( "pnfs",		RTFidPNFS ),
    RTF_REMEMBER( "pnb",		RTFidPNB ),
    RTF_REMEMBER( "pni",		RTFidPNI ),
    RTF_REMEMBER( "pncaps",		RTFidPNCAPS ),
    RTF_REMEMBER( "pnscaps",		RTFidPNSCAPS ),
    RTF_REMEMBER( "pnul",		RTFidPNUL ),
    RTF_REMEMBER( "pnuld",		RTFidPNULD ),
    RTF_REMEMBER( "pnuldb",		RTFidPNULDB ),
    RTF_REMEMBER( "pnulnone",		RTFidPNULNONE ),
    RTF_REMEMBER( "pnstrike",		RTFidPNSTRIKE ),

				/****************************************/
				/*  Section properties.			*/
				/****************************************/
    RTF_REMEMBER( "sectdefaultcl",	RTFidSECTDEFAULTCL ),
    RTF_REMEMBER( "sectd",		RTFidSECTD ),

    RTF_REMEMBER( "headery",		RTFidHEADERY ),
    RTF_REMEMBER( "footery",		RTFidFOOTERY ),
    RTF_REMEMBER( "pgwsxn",		RTFidPGWSXN ),
    RTF_REMEMBER( "pghsxn",		RTFidPGHSXN ),
    RTF_REMEMBER( "marglsxn",		RTFidMARGLSXN ),
    RTF_REMEMBER( "margtsxn",		RTFidMARGTSXN ),
    RTF_REMEMBER( "margrsxn",		RTFidMARGRSXN ),
    RTF_REMEMBER( "margbsxn",		RTFidMARGBSXN ),
    RTF_REMEMBER( "guttersxn",		RTFidGUTTERSXN ),

    RTF_REMEMBER( "titlepg",		RTFidTITLEPG ),

    RTF_REMEMBER( "sbknone",		RTFidSBKNONE ),
    RTF_REMEMBER( "sbkcol",		RTFidSBKCOL ),
    RTF_REMEMBER( "sbkpage",		RTFidSBKPAGE ),
    RTF_REMEMBER( "sbkeven",		RTFidSBKEVEN ),
    RTF_REMEMBER( "sbkodd",		RTFidSBKODD ),

    RTF_REMEMBER( "pgndec",		RTFidPGNDEC ),
    RTF_REMEMBER( "pgnucrm",		RTFidPGNUCRM ),
    RTF_REMEMBER( "pgnlcrm",		RTFidPGNLCRM ),
    RTF_REMEMBER( "pgnucltr",		RTFidPGNUCLTR ),
    RTF_REMEMBER( "pgnlcltr",		RTFidPGNLCLTR ),

    RTF_REMEMBER( "pgnstarts",		RTFidPGNSTARTS ),
    RTF_REMEMBER( "pgncont",		RTFidPGNCONT ),
    RTF_REMEMBER( "pgnrestart",		RTFidPGNRESTART ),

    RTF_REMEMBER( "pgnhn",		RTFidPGNHN ),
    RTF_REMEMBER( "pgnhnsh",		RTFidPGNHNSH ),
    RTF_REMEMBER( "pgnhnsp",		RTFidPGNHNSP ),
    RTF_REMEMBER( "pgnhnsc",		RTFidPGNHNSC ),
    RTF_REMEMBER( "pgnhnsm",		RTFidPGNHNSM ),
    RTF_REMEMBER( "pgnhnsn",		RTFidPGNHNSN ),

    RTF_REMEMBER( "pgnx",		RTFidPGNX ),
    RTF_REMEMBER( "pgny",		RTFidPGNY ),

    RTF_REMEMBER( "linex",		RTFidLINEX ),

    RTF_REMEMBER( "cols",		RTFidCOLS ),
    RTF_REMEMBER( "colsx",		RTFidCOLSX ),
    RTF_REMEMBER( "colno",		RTFidCOLNO ),
    RTF_REMEMBER( "colsr",		RTFidCOLSR ),
    RTF_REMEMBER( "colw",		RTFidCOLW ),

    RTF_REMEMBER( "vertalb",		RTFidVERTALB ),
    RTF_REMEMBER( "vertalc",		RTFidVERTALC ),
    RTF_REMEMBER( "vertalj",		RTFidVERTALJ ),
    RTF_REMEMBER( "vertalt",		RTFidVERTALT ),

    RTF_REMEMBER( "sectlinegrid",	RTFidSECTLINEGRID ),

    RTF_REMEMBER( "endnhere",		RTFidENDNHERE ),
    RTF_REMEMBER( "binfsxn",		RTFidBINFSXN ),
    RTF_REMEMBER( "binsxn",		RTFidBINSXN ),
    RTF_REMEMBER( "ds",			RTFidDS ),
    RTF_REMEMBER( "linebetcol",		RTFidLINEBETCOL ),
    RTF_REMEMBER( "linecont",		RTFidLINECONT ),
    RTF_REMEMBER( "linemod",		RTFidLINEMOD ),
    RTF_REMEMBER( "lineppage",		RTFidLINEPPAGE ),
    RTF_REMEMBER( "linerestart",	RTFidLINERESTART ),
    RTF_REMEMBER( "linestarts",		RTFidLINESTARTS ),
    RTF_REMEMBER( "lndscpsxn",		RTFidLNDSCPSXN ),
    RTF_REMEMBER( "ltrsect",		RTFidLTRSECT ),
    RTF_REMEMBER( "rtlsect",		RTFidRTLSECT ),
    RTF_REMEMBER( "sectunlocked",	RTFidSECTUNLOCKED ),

				/****************************************/
				/*  Document properyies: (Footnotes)	*/
				/****************************************/
    RTF_DOCPROP( "fet",		RTFidFET ),

    RTF_DOCPROP( "endnotes",		RTFidENDNOTES ),
    RTF_DOCPROP( "enddoc",		RTFidENDDOC ),
    RTF_DOCPROP( "ftntj",		RTFidFTNTJ ),
    RTF_DOCPROP( "ftnbj",		RTFidFTNBJ ),
    RTF_DOCPROP( "ftnstart",		RTFidFTNSTART ),
    RTF_DOCPROP( "ftnrstcont",		RTFidFTNRSTCONT ),
    RTF_DOCPROP( "ftnrestart",		RTFidFTNRESTART ),
    RTF_DOCPROP( "ftnrstpg",		RTFidFTNRSTPG ),

    RTF_DOCPROP( "ftnnar",		RTFidFTNNAR ),
    RTF_DOCPROP( "ftnnalc",		RTFidFTNNALC ),
    RTF_DOCPROP( "ftnnauc",		RTFidFTNNAUC ),
    RTF_DOCPROP( "ftnnrlc",		RTFidFTNNRLC ),
    RTF_DOCPROP( "ftnnruc",		RTFidFTNNRUC ),
    RTF_DOCPROP( "ftnnchi",		RTFidFTNNCHI ),

    RTF_DOCPROP( "aenddoc",		RTFidAENDDOC ),
    RTF_DOCPROP( "aendnotes",		RTFidAENDNOTES ),
    RTF_DOCPROP( "aftntj",		RTFidAFTNTJ ),
    RTF_DOCPROP( "aftnbj",		RTFidAFTNBJ ),
    RTF_DOCPROP( "aftnstart",		RTFidAFTNSTART ),
    RTF_DOCPROP( "aftnrstcont",	RTFidAFTNRSTCONT ),
    RTF_DOCPROP( "aftnrestart",	RTFidAFTNRESTART ),
    /* RTF_DOCPROP( "aftnrstpg",	RTFidAFTNRSTPG ), insane */

    RTF_DOCPROP( "aftnnar",		RTFidAFTNNAR ),
    RTF_DOCPROP( "aftnnalc",		RTFidAFTNNALC ),
    RTF_DOCPROP( "aftnnauc",		RTFidAFTNNAUC ),
    RTF_DOCPROP( "aftnnrlc",		RTFidAFTNNRLC ),
    RTF_DOCPROP( "aftnnruc",		RTFidAFTNNRUC ),
    RTF_DOCPROP( "aftnnchi",		RTFidAFTNNCHI ),

    RTF_DOCPROP( "ftnalt",		RTFidFTNALT ),
				/****************************************/
				/*  Encoding.				*/
				/****************************************/
    RTF_REMEMBER( "ansi",		RTFidANSI ),
    RTF_REMEMBER( "mac",		RTFidMAC ),
    RTF_REMEMBER( "pc",			RTFidPC ),
    RTF_REMEMBER( "pca",		RTFidPCA ),
    RTF_DOCPROP( "ansicpg",		RTFidANSICPG ),

				/****************************************/
				/*  Unknown				*/
				/****************************************/
    RTF_REMEMBER( "itap",		RTFidITAP ),
    RTF_REMEMBER( "lin",		RTFidLIN ),
    RTF_REMEMBER( "rin",		RTFidRIN ),
    RTF_REMEMBER( "langnp",		RTFidLANGNP ),
    RTF_REMEMBER( "langfe",		RTFidLANGFE ),
    RTF_REMEMBER( "langfenp",		RTFidLANGFENP ),
    RTF_REMEMBER( "nolnhtadjtbl",	RTFidNOLNHTADJTBL ),

				/****************************************/
				/*  Document Formatting properties.	*/
				/****************************************/
    RTF_REMEMBER( "allprot",		RTFidALLPROT ),
    RTF_REMEMBER( "annotprot",		RTFidANNOTPROT ),
    RTF_REMEMBER( "brkfrm",		RTFidBRKFRM ),
    RTF_REMEMBER( "cvmme",		RTFidCVMME ),
    RTF_REMEMBER( "defformat",		RTFidDEFFORMAT ),
    RTF_REMEMBER( "deflang",		RTFidDEFLANG ),
    RTF_REMEMBER( "doctemp",		RTFidDOCTEMP ),
    RTF_REMEMBER( "fldalt",		RTFidFLDALT ),
    RTF_REMEMBER( "formdisp",		RTFidFORMDISP ),
    RTF_REMEMBER( "formprot",		RTFidFORMPROT ),
    RTF_REMEMBER( "formshade",		RTFidFORMSHADE ),
    RTF_REMEMBER( "fracwidth",		RTFidFRACWIDTH ),
    RTF_REMEMBER( "hyphauto",		RTFidHYPHAUTO ),
    RTF_REMEMBER( "hyphcaps",		RTFidHYPHCAPS ),
    RTF_REMEMBER( "hyphconsec",		RTFidHYPHCONSEC ),
    RTF_REMEMBER( "hyphhotz",		RTFidHYPHHOTZ ),
    RTF_REMEMBER( "landscape",		RTFidLANDSCAPE ),
    RTF_REMEMBER( "linestart",		RTFidLINESTART ),
    RTF_REMEMBER( "linkstyles",		RTFidLINKSTYLES ),
    RTF_REMEMBER( "ltrdoc",		RTFidLTRDOC ),
    RTF_REMEMBER( "makebackup",		RTFidMAKEBACKUP ),
    RTF_REMEMBER( "margmirror",		RTFidMARGMIRROR ),
    RTF_REMEMBER( "nocolbal",		RTFidNOCOLBAL ),
    RTF_REMEMBER( "noextrasprl",	RTFidNOEXTRASPRL ),
    RTF_REMEMBER( "notabind",		RTFidNOTABIND ),
    RTF_REMEMBER( "otblrul",		RTFidOTBLRUL ),
    RTF_REMEMBER( "prcolbl",		RTFidPRCOLBL ),
    RTF_REMEMBER( "printdata",		RTFidPRINTDATA ),
    RTF_REMEMBER( "psover",		RTFidPSOVER ),
    RTF_REMEMBER( "revbar",		RTFidREVBAR ),
    RTF_REMEMBER( "revisions",		RTFidREVISIONS ),
    RTF_REMEMBER( "revprop",		RTFidREVPROP ),
    RTF_REMEMBER( "revprot",		RTFidREVPROT ),
    RTF_REMEMBER( "rtldoc",		RTFidRTLDOC ),
    RTF_REMEMBER( "sprslnsp",		RTFidSPRSLNSP ),
    RTF_REMEMBER( "sprsspbf",		RTFidSPRSSPBF ),
    RTF_REMEMBER( "sprstsp",		RTFidSPRSTSP ),
    RTF_REMEMBER( "subfontbysize",	RTFidSUBFONTBYSIZE ),
    RTF_REMEMBER( "swpbdr",		RTFidSWPBDR ),
    RTF_REMEMBER( "transmf",		RTFidTRANSMF ),
    RTF_REMEMBER( "ight",		RTFidIGHT ),
    RTF_REMEMBER( "widowctrl",		RTFidWIDOWCTRL ),
    RTF_REMEMBER( "wraptrsp",		RTFidWRAPTRSP ),
    RTF_REMEMBER( "viewzk",		RTFidVIEWZK ),
    RTF_REMEMBER( "viewkind",		RTFidVIEWKIND ),
    RTF_REMEMBER( "viewscale",		RTFidVIEWSCALE ),
    RTF_REMEMBER( "pgbrdrhead",		RTFidPGBRDRHEAD ),
    RTF_REMEMBER( "pgbrdrfoot",		RTFidPGBRDRFOOT ),
    RTF_REMEMBER( "lytprtmet",		RTFidLYTPRTMET ),
    RTF_REMEMBER( "ogutter",		RTFidOGUTTER ),
    RTF_REMEMBER( "deflangfe",		RTFidDEFLANGFE ),
    RTF_REMEMBER( "uc",			RTFidUC ),
    RTF_REMEMBER( "level",		RTFidLEVEL ),
    RTF_REMEMBER( "ltrpar",		RTFidLTRPAR ),
    RTF_REMEMBER( "noline",		RTFidNOLINE ),
    RTF_REMEMBER( "rtlpar",		RTFidRTLPAR ),
    RTF_REMEMBER( "sbys",		RTFidSBYS ),
    RTF_REMEMBER( "subdocument",	RTFidSUBDOCUMENT ),

				/****************************************/
				/*  Fields in dates and times.		*/
				/****************************************/
    RTF_REMEMBER( "dy",			RTFidDY ),
    RTF_REMEMBER( "hr",			RTFidHR ),
    RTF_REMEMBER( "min",		RTFidMIN ),
    RTF_REMEMBER( "mo",			RTFidMO ),
    RTF_REMEMBER( "yr",			RTFidYR ),

    { 0, 0, 0 }
};

int docRtfRememberProperty(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case	RTFidDEFLANG:
	case	RTFidDEFLANGFE:
	case	RTFidUC:
	    break;

	case	RTFidSECTD:
	    {
	    const DocumentGeometry *	dgDoc;
	    DocumentGeometry *		dgSect;

	    if  ( ! rrc->rrcBd )
		{ XDEB(rrc->rrcBd); return -1;	}

	    dgDoc= &(rrc->rrcBd->bdProperties.dpGeometry);
	    dgSect= &(rrc->rrcSectionProperties.spDocumentGeometry);

	    docCleanSectionProperties( &(rrc->rrcSectionProperties) );
	    docInitSectionProperties(  &(rrc->rrcSectionProperties) );

	    *dgSect= *dgDoc;
	    }
	    break;

	case	RTFidCOLS:
	    rrc->rrcSectionProperties.spColumnCount= arg;
	case	RTFidCOLSX:
	    rrc->rrcSectionProperties.spColumnSpacingTwips= arg;
	case	RTFidCOLW:
	    rrc->rrcSectionProperties.spColumnWidthTwips= arg;

	case	RTFidTITLEPG:
	    rrc->rrcSectionProperties.spHasTitlePage= 1;
	    break;

	case	RTFidHEADERY:
	    if  ( arg != 0 )
		{
		rrc->rrcSectionProperties.spDocumentGeometry.
						dgHeaderPositionTwips= arg;
		}
	    break;
	case	RTFidFOOTERY:
	    if  ( arg != 0 )
		{
		rrc->rrcSectionProperties.spDocumentGeometry.
						dgFooterPositionTwips= arg;
		}
	    break;
	case	RTFidPGWSXN:
	    rrc->rrcSectionProperties.spDocumentGeometry.
						dgPageWideTwips= arg;
	    break;
	case	RTFidPGHSXN:
	    rrc->rrcSectionProperties.spDocumentGeometry.
						dgPageHighTwips= arg;
	    break;
	case	RTFidMARGLSXN:
	    rrc->rrcSectionProperties.spDocumentGeometry.
						dgLeftMarginTwips= arg;
	    break;
	case	RTFidMARGTSXN:
	    rrc->rrcSectionProperties.spDocumentGeometry.
						dgTopMarginTwips= arg;
	    break;
	case	RTFidMARGRSXN:
	    rrc->rrcSectionProperties.spDocumentGeometry.
						dgRightMarginTwips= arg;
	    break;
	case	RTFidMARGBSXN:
	    rrc->rrcSectionProperties.spDocumentGeometry.
						dgBottomMarginTwips= arg;
	    break;
	case	RTFidGUTTERSXN:
	    rrc->rrcSectionProperties.spDocumentGeometry.
						dgGutterTwips= arg;
	    break;

	case	RTFidSBKNONE:
	    rrc->rrcSectionProperties.spBreakKind= DOCsbkNONE;
	    break;
	case	RTFidSBKCOL:
	    rrc->rrcSectionProperties.spBreakKind= DOCsbkCOL;
	    break;
	case	RTFidSBKPAGE:
	    rrc->rrcSectionProperties.spBreakKind= DOCsbkPAGE;
	    break;
	case	RTFidSBKEVEN:
	    rrc->rrcSectionProperties.spBreakKind= DOCsbkEVEN;
	    break;
	case	RTFidSBKODD:
	    rrc->rrcSectionProperties.spBreakKind= DOCsbkODD;
	    break;

	case	RTFidPGNDEC:
	    rrc->rrcSectionProperties.spPageNumberStyle= DOCpgnDEC;
	    break;
	case	RTFidPGNUCRM:
	    rrc->rrcSectionProperties.spPageNumberStyle= DOCpgnUCRM;
	    break;
	case	RTFidPGNLCRM:
	    rrc->rrcSectionProperties.spPageNumberStyle= DOCpgnLCRM;
	    break;
	case	RTFidPGNUCLTR:
	    rrc->rrcSectionProperties.spPageNumberStyle= DOCpgnUCLTR;
	    break;
	case	RTFidPGNLCLTR:
	    rrc->rrcSectionProperties.spPageNumberStyle= DOCpgnLCLTR;
	    break;

	case	RTFidPGNSTARTS:
	    rrc->rrcSectionProperties.spStartPageNumber= arg- 1;
	    break;
	case	RTFidPGNCONT:
	    rrc->rrcSectionProperties.spRestartPageNumbers= 0;
	    break;
	case	RTFidPGNRESTART:
	    rrc->rrcSectionProperties.spRestartPageNumbers= 1;
	    break;

	case	RTFidDELETED:
	    rrc->rrcInDeletedText= 1;
	    break;

	case RTFidFI:
	    rrc->rrcParagraphProperties.ppFirstIndentTwips= arg;
	    break;
	case RTFidLI:
	    rrc->rrcParagraphProperties.ppLeftIndentTwips= arg;
	    break;
	case RTFidRI:
	    rrc->rrcParagraphProperties.ppRightIndentTwips= arg;
	    break;
	case RTFidSB:
	    rrc->rrcParagraphProperties.ppSpaceBeforeTwips= arg;
	    break;
	case RTFidSA:
	    rrc->rrcParagraphProperties.ppSpaceAfterTwips= arg;
	    break;
	case RTFidS:
	    rrc->rrcParagraphProperties.ppStyle= arg;
	    break;

	case RTFidSL:
	    if  ( arg > -1000 && arg < 1000 )
		{ rrc->rrcParagraphProperties.ppLineSpacingTwips= arg;	}
	    else{ rrc->rrcParagraphProperties.ppLineSpacingTwips= 0;	}
	    break;

	case RTFidSLMULT:
	    rrc->rrcParagraphProperties.ppLineSpacingIsMultiple= ( arg != 0 );
	    break;

	case RTFidTX:
	    rrc->rrcTabStop.tsTwips= arg;
	    if  ( docParaAddTab( &(rrc->rrcParagraphProperties),
						    &(rrc->rrcTabStop) ) )
		{ LDEB(arg); return -1;	}
	    docInitTabStop( &(rrc->rrcTabStop) );
	    break;

	case RTFidTQR:		rrc->rrcTabStop.tsKind=   DOCtkRIGHT;	break;
	case RTFidTQC:		rrc->rrcTabStop.tsKind=   DOCtkCENTRE;	break;
	case RTFidTQDEC:	rrc->rrcTabStop.tsKind=   DOCtkDECIMAL;	break;
	case RTFidTLDOT:	rrc->rrcTabStop.tsLeader= DOCtlDOTS;	break;
	case RTFidTLHYPH:	rrc->rrcTabStop.tsLeader= DOCtlHYPH;	break;
	case RTFidTLUL:		rrc->rrcTabStop.tsLeader= DOCtlUNDERLINE; break;
	case RTFidTLTH:		rrc->rrcTabStop.tsLeader= DOCtlTHICK;	break;
	case RTFidTLEQ:		rrc->rrcTabStop.tsLeader= DOCtlEQUAL;	break;

	case	RTFidPARD:
	    docCleanParagraphProperties( &(rrc->rrcParagraphProperties) );
	    docInitParagraphProperties(  &(rrc->rrcParagraphProperties) );
	    break;

	case RTFidQL:
	    rrc->rrcParagraphProperties.ppAlignment= DOCiaLEFT;
	    break;
	case RTFidQR:
	    rrc->rrcParagraphProperties.ppAlignment= DOCiaRIGHT;
	    break;
	case RTFidQC:
	    rrc->rrcParagraphProperties.ppAlignment= DOCiaCENTERED;
	    break;
	case RTFidQJ:
	    rrc->rrcParagraphProperties.ppAlignment= DOCiaJUSTIFIED;
	    break;

	case RTFidKEEP:
	    rrc->rrcParagraphProperties.ppKeepOnPage= arg != 0;
	    break;
	case RTFidKEEPN:
	    rrc->rrcParagraphProperties.ppKeepWithNext= arg != 0;
	    break;

	case RTFidPAGEBB:
	    rrc->rrcParagraphProperties.ppStartsOnNewPage= arg != 0;
	    break;

	case RTFidSHADING:
	    rrc->rrcParagraphProperties.ppShadingLevel= arg != 0;
	    break;

	case RTFidOUTLINELEVEL:
	    rrc->rrcParagraphProperties.ppOutlineLevel= arg != 0;
	    break;

	case RTFidINTBL:
	    rrc->rrcParagraphProperties.ppInTable= arg != 0;
	    rrc->rrcBd->bdProperties.dpContainsTables= 1;
	    break;

	case RTFidBRDRT:
	    rrc->rrcParagraphProperties.ppTopBorder=
						    rrc->rrcBorderProperties;
	    break;
	case RTFidBRDRL:
	    rrc->rrcParagraphProperties.ppLeftBorder=
						    rrc->rrcBorderProperties;
	    break;
	case RTFidBRDRR:
	    rrc->rrcParagraphProperties.ppRightBorder=
						    rrc->rrcBorderProperties;
	    break;
	case RTFidBRDRB:
	    rrc->rrcParagraphProperties.ppBottomBorder=
						    rrc->rrcBorderProperties;
	    break;
	case RTFidBRDRBTW:
	    rrc->rrcParagraphProperties.ppBetweenBorder=
						    rrc->rrcBorderProperties;
	    break;

	case RTFidBOX:
	    rrc->rrcParagraphProperties.ppBoxBorder= rrc->rrcBorderProperties;
	    break;

	case RTFidBRDRBAR:
	    rrc->rrcParagraphProperties.ppBar= rrc->rrcBorderProperties;
	    break;

	case RTFidNOWIDCTLPAR:
	case RTFidWIDCTLPAR:
	    break;

	case RTFidTRGAPH:
	    rrc->rrcRowProperties.rpHalfGapWidthTwips= arg;
	    break;

	case RTFidTRRH:
	    rrc->rrcRowProperties.rpHeightTwips= arg;
	    break;

	case RTFidTRLEFT:
	    rrc->rrcRowProperties.rpLeftIndentTwips= arg;
	    break;

	case RTFidTRHDR:
	    rrc->rrcRowProperties.rpIsTableHeader= ( arg != 0 );
	    break;

	case RTFidTRKEEP:
	    rrc->rrcRowProperties.rpKeepOnPage= ( arg != 0 );
	    break;

	case RTFidTRBRDRT:
	    rrc->rrcRowProperties.rpTopBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRL:
	    rrc->rrcRowProperties.rpLeftBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRB:
	    rrc->rrcRowProperties.rpBottomBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRR:
	    rrc->rrcRowProperties.rpRightBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRH:
	    rrc->rrcRowProperties.rpHasHorizontalBorders= ( arg != 0 );
	    rrc->rrcRowProperties.rpHorizontalBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRV:
	    rrc->rrcRowProperties.rpHasVerticalBorders= ( arg != 0 );
	    rrc->rrcRowProperties.rpVerticalBorder= rrc->rrcBorderProperties;
	    break;

	case RTFidTRQL:
	    rrc->rrcRowProperties.rpAlignment= DOCiaLEFT;
	    break;
	case RTFidTRQC:
	    rrc->rrcRowProperties.rpAlignment= DOCiaCENTERED;
	    break;
	case RTFidTRQR:
	    rrc->rrcRowProperties.rpAlignment= DOCiaRIGHT;
	    break;

	case RTFidCLMGF:
	    rrc->rrcCellProperties.cpLeftInMergedRow= ( arg != 0 );
	    break;
	case RTFidCLMRG:
	    rrc->rrcCellProperties.cpMergedWithLeft= ( arg != 0 );
	    break;
	case RTFidCLVMGF:
	    rrc->rrcCellProperties.cpTopInMergedColumn= ( arg != 0 );
	    break;
	case RTFidCLVMRG:
	    rrc->rrcCellProperties.cpMergedWithAbove= ( arg != 0 );
	    break;

	case RTFidCLBRDRT:
	    rrc->rrcCellProperties.cpTopBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidCLBRDRL:
	    rrc->rrcCellProperties.cpLeftBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidCLBRDRR:
	    rrc->rrcCellProperties.cpRightBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidCLBRDRB:
	    rrc->rrcCellProperties.cpBottomBorder= rrc->rrcBorderProperties;
	    break;

	case RTFidCELLD:
	    docCleanCellProperties( &(rrc->rrcCellProperties) );
	    docInitCellProperties( &(rrc->rrcCellProperties) );
	    break;
	case RTFidCLVERTALT:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaTOP;
	    break;
	case RTFidCLVERTALC:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaCENTERED;
	    break;
	case RTFidCLVERTALB:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaBOTTOM;
	    break;
	case RTFidCLTXLRTB:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaVERT_LEFT;
	    break;
	case RTFidCLTXTBRL:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaVERT_RIGHT;
	    break;

	case RTFidCLCFPAT:
	    rrc->rrcCellProperties.cpForegroundColor= arg;
	    break;
	case RTFidCLCBPAT:
	    rrc->rrcCellProperties.cpBackgroundColor= arg;
	    break;

	case RTFidCLSHDNG:
	    rrc->rrcCellProperties.cpShadingLevel= arg;
	    break;

	case RTFidCLBGHORIZ:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGHORIZ;
	    break;
	case RTFidCLBGVERT:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGVERT;
	    break;
	case RTFidCLBGFDIAG:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGFDIAG;
	    break;
	case RTFidCLBGBDIAG:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGBDIAG;
	    break;
	case RTFidCLBGCROSS:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGCROSS;
	    break;
	case RTFidCLBGDCROSS:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDCROSS;
	    break;
	case RTFidCLBGDKHOR:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDKHORIZ;
	    break;
	case RTFidCLBGDKVERT:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDKVERT;
	    break;
	case RTFidCLBGDKFDIAG:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDKFDIAG;
	    break;
	case RTFidCLBGDKBDIAG:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDKBDIAG;
	    break;
	case RTFidCLBGDKCROSS:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDKCROSS;
	    break;
	case RTFidCLBGDKDCROSS:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDKDCROSS;
	    break;

	case RTFidADJUSTRIGHT:
	case RTFidADJUSTLEFT:
	    break;
	case	RTFidCF:
	    /*  foreground color: index in color table or 0	*/
	    break;
	case	RTFidFC:
	    /*  Does not appear in the RTF spec.		*/
	    /*  It can however be found in NextStep files.	*/
	    break;
	case	RTFidUP:
	    break;
				/****************************************/
				/*  Set/Unset Text Attributes.		*/
				/****************************************/
	case	RTFidPLAIN:
	    docInitTextAttribute( &(rrc->rrcCurrentTextAttribute) );
	    if  ( rrc->rrcBd )
		{
		rrc->rrcCurrentTextAttribute.taFontNumber=
				    rrc->rrcBd->bdProperties.dpDefaultFont;
		}
	    rrc->rrcCurrentTextAttribute.taFontSizeHalfPoints= 24;
	    rrc->rrcInDeletedText= 0;
	    break;

	case	RTFidF:
	    rrc->rrcCurrentTextAttribute.taFontNumber= arg;
	    break;
	case	RTFidFS:
	    rrc->rrcCurrentTextAttribute.taFontSizeHalfPoints= arg;
	    break;
	case	RTFidB:
	    rrc->rrcCurrentTextAttribute.taFontIsBold= arg != 0;
	    break;
	case	RTFidI:
	    rrc->rrcCurrentTextAttribute.taFontIsSlanted= arg != 0;
	    break;
	case	RTFidUL:
	    rrc->rrcCurrentTextAttribute.taIsUnderlined= arg != 0;
	    break;
	case	RTFidULNONE:
	    rrc->rrcCurrentTextAttribute.taIsUnderlined= 0;
	    break;
	case	RTFidSUB:
	    rrc->rrcCurrentTextAttribute.taSuperSub= DOCfontSUBSCRIPT;
	    break;
	case	RTFidSUPER:
	    rrc->rrcCurrentTextAttribute.taSuperSub= DOCfontSUPERSCRIPT;
	    break;
	case	RTFidNOSUPERSUB:
	    rrc->rrcCurrentTextAttribute.taSuperSub= DOCfontREGULAR;
	    break;
	case	RTFidCAPS:
	    rrc->rrcCurrentTextAttribute.taCapitals= arg != 0;
	    break;
	case	RTFidSCAPS:
	    rrc->rrcCurrentTextAttribute.taSmallCaps= arg != 0;
	    break;
	case	RTFidSTRIKE:
	    rrc->rrcCurrentTextAttribute.taHasStrikeThrough= arg != 0;
	    break;
				/****************************************/
				/*  Set/Unset Text Attributes.		*/
				/*  In Paragraph Numbering.		*/
				/****************************************/
	case	RTFidPNF:
	    rrc->rrcParagraphNumber.pnTextAttribute.taFontNumber= arg;
	    break;
	case	RTFidPNFS:
	    rrc->rrcParagraphNumber.pnTextAttribute.taFontSizeHalfPoints= arg;
	    break;
	case	RTFidPNB:
	    rrc->rrcParagraphNumber.pnTextAttribute.taFontIsBold= arg != 0;
	    break;
	case	RTFidPNI:
	    rrc->rrcParagraphNumber.pnTextAttribute.taFontIsSlanted= arg != 0;
	    break;
	case	RTFidPNCAPS:
	    rrc->rrcParagraphNumber.pnTextAttribute.taCapitals= arg != 0;
	    break;
	case	RTFidPNSCAPS:
	    rrc->rrcParagraphNumber.pnTextAttribute.taSmallCaps= arg != 0;
	    break;
	case	RTFidPNUL:
	    rrc->rrcParagraphNumber.pnTextAttribute.taIsUnderlined= arg != 0;
	    break;
	case	RTFidPNULNONE:
	    rrc->rrcParagraphNumber.pnTextAttribute.taIsUnderlined= 0;
	    break;
	case	RTFidPNSTRIKE:
	    rrc->rrcParagraphNumber.pnTextAttribute.taHasStrikeThrough=
								    arg != 0;
	    break;

				/****************************************/
	case RTFidCHCFPAT:
	    rrc->rrcTextForeground= arg;
	    break;
	case RTFidCHCBPAT:
	    rrc->rrcTextBackground= arg;
	    break;

	case RTFidCHSHDNG:
	    rrc->rrcTextShadingLevel= arg;
	    break;

	case RTFidCHBGHORIZ:
	    rrc->rrcTextShadingPattern= DOCspBGHORIZ;
	    break;
	case RTFidCHBGVERT:
	    rrc->rrcTextShadingPattern= DOCspBGVERT;
	    break;
	case RTFidCHBGFDIAG:
	    rrc->rrcTextShadingPattern= DOCspBGFDIAG;
	    break;
	case RTFidCHBGBDIAG:
	    rrc->rrcTextShadingPattern= DOCspBGBDIAG;
	    break;
	case RTFidCHBGCROSS:
	    rrc->rrcTextShadingPattern= DOCspBGCROSS;
	    break;
	case RTFidCHBGDCROSS:
	    rrc->rrcTextShadingPattern= DOCspBGDCROSS;
	    break;
	case RTFidCHBGDKHORIZ:
	    rrc->rrcTextShadingPattern= DOCspBGDKHORIZ;
	    break;
	case RTFidCHBGDKVERT:
	    rrc->rrcTextShadingPattern= DOCspBGDKVERT;
	    break;
	case RTFidCHBGDKFDIAG:
	    rrc->rrcTextShadingPattern= DOCspBGDKFDIAG;
	    break;
	case RTFidCHBGDKBDIAG:
	    rrc->rrcTextShadingPattern= DOCspBGDKBDIAG;
	    break;
	case RTFidCHBGDKCROSS:
	    rrc->rrcTextShadingPattern= DOCspBGDKCROSS;
	    break;
	case RTFidCHBGDKDCROSS:
	    rrc->rrcTextShadingPattern= DOCspBGDKDCROSS;
	    break;

	case RTFidCHBRDR:
	    rrc->rrcTextBorderProperties= rrc->rrcBorderProperties;
	    break;

	case	RTFidTROWD:
	    docCleanRowProperties( &(rrc->rrcRowProperties) );
	    docInitRowProperties(  &(rrc->rrcRowProperties) );

	    docInitCellProperties( &(rrc->rrcCellProperties) );
	    break;

	case	RTFidCELLX:
	    rrc->rrcCellProperties.cpRightBoundaryTwips= arg;

	    if  ( docInsertRowColumn( &(rrc->rrcRowProperties), -1,
						&rrc->rrcCellProperties ) )
		{ LDEB(rrc->rrcRowProperties.rpCellCount); return -1; }

	    docInitBorderProperties( &(rrc->rrcCellProperties.cpTopBorder) );
	    docInitBorderProperties( &(rrc->rrcCellProperties.cpLeftBorder) );
	    docInitBorderProperties( &(rrc->rrcCellProperties.cpRightBorder) );
	    docInitBorderProperties( &(rrc->rrcCellProperties.cpBottomBorder) );

	    rrc->rrcCellProperties.cpLeftInMergedRow= 0;
	    rrc->rrcCellProperties.cpMergedWithLeft= 0;
	    rrc->rrcCellProperties.cpTopInMergedColumn= 0;
	    rrc->rrcCellProperties.cpMergedWithAbove= 0;

	    break;

				/****************************************/
				/*  Paragraph Numbering.		*/
				/****************************************/
	case RTFidPNCARD:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnCARD;
	    break;
	case RTFidPNDEC:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnDEC;
	    break;
	case RTFidPNUCLTR:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnUCLTR;
	    break;
	case RTFidPNUCRM:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnUCRM;
	    break;
	case RTFidPNLCLTR:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnLCLTR;
	    break;
	case RTFidPNLCRM:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnLCRM;
	    break;
	case RTFidPNORD:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnORD;
	    break;
	case RTFidPNORDT:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnORDT;
	    break;
	case RTFidPNSTART:
	    rrc->rrcParagraphNumber.pnStartNumber= arg- 1;
	    break;
	case RTFidPNINDENT:
	    rrc->rrcParagraphNumber.pnIndentTwips= arg;
	    break;
	case RTFidPNHANG:
	    rrc->rrcParagraphNumber.pnUseHangingIndent= 1;
	    break;
				/****************************************/
				/*  Encoding.				*/
				/****************************************/
	case RTFidPC:
	    memcpy( rrc->rrcInputMapping, docDOS437_to_ISO1, 256 );
	    break;
	case RTFidPCA:
	    memcpy( rrc->rrcInputMapping, docDOS850_to_ISO1, 256 );
	    break;
	case RTFidMAC:
	    memcpy( rrc->rrcInputMapping, docMAC_to_ISO1, 256 );
	    break;

	case RTFidCGRID:
	case RTFidLANG:
	    break;
				/****************************************/
				/*  Fields in dates and times.		*/
				/****************************************/
	case RTFidDY:
	    rrc->rrcTm.tm_mday= arg;
	    break;
	case RTFidHR:
	    rrc->rrcTm.tm_hour= arg;
	    break;
	case RTFidMIN:
	    rrc->rrcTm.tm_min= arg;
	    break;
	case RTFidMO:
	    rrc->rrcTm.tm_mon= arg- 1;
	    break;
	case RTFidYR:
	    rrc->rrcTm.tm_year= arg- 1900;
	    break;
				/****************************************/
				/*  Unknown				*/
				/****************************************/
	default:
	    /*SLDEB(rcw->rcwWord,rcw->rcwId);*/
	    break;
	}

    return 0;
    }
