/************************************************************************/
/*									*/
/*  Layout of a document.						*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   define	y0	math_y0
#   define	y1	math_y1
#   include	<math.h>
#   undef	y0
#   undef	y1

#   include	<appImage.h>
#   include	"docPs.h"
#   include	<sioHex.h>
#   include	<sioMemory.h>
#   include	<appWinMeta.h>
#   include	<appMacPict.h>

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Used for the layout of cells.					*/
/*									*/
/************************************************************************/

#   define LINEDISTFAC	5

typedef struct CellLayoutState
    {
    int			clsAdvanced;
    int			clsPara;
    int			clsPart;
    int			clsLinesDone;

    LayoutPosition	clsPosition;

    FormattingFrame	clsFormattingFrameTwips;
    FormattingFrame	clsFormattingFramePixels;
    } CellLayoutState;

typedef int (*LAYOUT_LINES)(	DocumentRectangle *		drChanged,
				BufferItem *			bi,
				int				inHeaderFooter,
				int				part,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				const FormattingFrame *		ffTwips,
				const FormattingFrame *		ffPixels,
				int *				pLinesDone,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom,
				ScreenLayout *			sl );

/************************************************************************/
/*									*/
/*  Translate text attributes to AfmFontInfo.				*/
/*									*/
/************************************************************************/

AfmFontInfo * docPsPrintGetAfi(	const AppPhysicalFontList *	apfl,
				int				physf )
    {
    AppFontFamily *			aff;
    int					affCount;
    AppFontTypeface *			aft;

    AppPhysicalFont *			apf;

    if  ( psFontCatalog( apfl->apflAfmDirectory, &aff, &affCount ) )
	{ SDEB(apfl->apflAfmDirectory); return (AfmFontInfo *)0;	}

    if  ( physf < 0 || physf >= apfl->apflCount )
	{ LLDEB(physf,apfl->apflCount); return (AfmFontInfo *)0;	}

    apf= apfl->apflFonts+ physf;

    if  ( apf->apfPsFamilyNumber < 0		||
	  apf->apfPsFamilyNumber >= affCount	)
	{ LLDEB(apf->apfPsFamilyNumber,affCount); return (AfmFontInfo *)0; }

    aff += apf->apfPsFamilyNumber;
    aft= aff->affFaces;

    if  ( apf->apfPsFaceNumber < 0			||
	  apf->apfPsFaceNumber >=  aff->affFaceCount	)
	{
	LLDEB(apf->apfPsFaceNumber,aff->affFaceCount);
	return (AfmFontInfo *)0;
	}

    aft += apf->apfPsFaceNumber;

    if  ( ! aft->aftPrintingData )
	{ XDEB(aft->aftPrintingData);	}

    return (AfmFontInfo *)aft->aftPrintingData;
    }

/************************************************************************/
/*									*/
/*  Continue to lay out the text on a subsequent page.			*/
/*									*/
/************************************************************************/

void docPsStartPageLayout(	int				pageNumber,
				AppDrawingData *		add,
				const SectionProperties *	sp,
				LayoutPosition *		lpTop )
    {
    double			xfac= add->addMagnifiedPixelsPerTwip;
    const DocumentGeometry *	dg= &(sp->spDocumentGeometry);

    lpTop->lpPage= pageNumber;
    lpTop->lpPageYTwips= dg->dgTopMarginTwips;

    lpTop->lpYPixels= pageNumber* add->addPageStepPixels+
				TWIPStoPIXELS( xfac, dg->dgTopMarginTwips );
    }

static void docPsToNextPage(	BufferItem *			bi,
				AppDrawingData *		add,
				LayoutPosition *		lpTop )
    {
    while ( bi				&&
	    bi->biLevel != DOClevSECT	)
	{ bi= bi->biParent;	}

    if  ( ! bi )
	{ XDEB(bi); return;	}

    docPsStartPageLayout( lpTop->lpPage+ 1,
					add, &(bi->biSectProperties), lpTop );

    return;
    }

/************************************************************************/
/*									*/
/*  Calculate the height of a series of lines in a paragraph.		*/
/*									*/
/************************************************************************/

static void docPsAboveParagraph(	BufferItem *			bi )
    {
    int				above= 0;
    const BorderProperties *	bp= (const BorderProperties *)0;

    above= bi->biParaSpaceBeforeTwips;

    if  ( bi->biParaInTable			&&
	  bi->biNumberInParent == 0		)
	{
	const BufferItem *	rowBi= bi->biParent->biParent;
	const CellProperties *	cp;
	int			col= bi->biParent->biNumberInParent;

	if  ( col >= rowBi->biRowCellCount )
	    { LLDEB(col,rowBi->biRowCellCount);	}
	else{
	    cp= &(rowBi->biRowCells[col]);

	    if  ( cp->cpTopBorder.bpIsSet )
		{
		above += cp->cpTopBorder.bpSpacingTwips;
		/*  Word seems not to allocate this
		above += cp->cpTopBorder.bpWidthTwips;
		*/
		}
	    }

	/*  Nicer but Word does not do this.
	above += ( bi->biParaLeadingTwips+ 1 )/ 2;
	*/
	}

    if  ( bi->biParaTopBorder.bpIsSet )
	{
	/*  Word seems not to allocate this
	above += bi->biParaTopBorder.bpWidthTwips;
	*/
	above += bi->biParaTopBorder.bpSpacingTwips;
	bp= &(bi->biParaTopBorder);
	}
    else{
	if  ( bi->biParaBoxBorder.bpIsSet )
	    {
	    BufferItem *	prevBi= (BufferItem *)0;

	    if  ( bi->biNumberInParent > 0 )
		{
		prevBi= bi->biParent->biChildren[
					    bi->biNumberInParent- 1];
		}
	    if  ( ! prevBi				||
		  ! prevBi->biParaBoxBorder.bpIsSet	)
		{
		/*  Word seems not to allocate this
		above += bi->biParaBoxBorder.bpWidthTwips;
		*/
		above += bi->biParaBoxBorder.bpSpacingTwips;
		bp= &(bi->biParaBoxBorder);
		}
	    }
	}

    bi->biParaSpaceAboveLinesTwips= above;
    bi->biParaBorderAboveParagraph= bp;

    return;
    }

static void docPsBelowParagraph(	BufferItem *			bi )
    {
    int				below= 0;
    const BorderProperties *	bp= (const BorderProperties *)0;

    if  ( bi->biParaBottomBorder.bpIsSet )
	{
	below += bi->biParaLeadingTwips;
	below += bi->biParaBottomBorder.bpSpacingTwips;
	/*  Word seems not to allocate this
	below += bi->biParaBottomBorder.bpWidthTwips;
	*/
	bp= &(bi->biParaBottomBorder);
	}
    else{
	if  ( bi->biParaBoxBorder.bpIsSet )
	    {
	    BufferItem *	nextBi= (BufferItem *)0;

	    if  ( bi->biNumberInParent <
				bi->biParent->biChildCount- 1 )
		{
		nextBi= bi->biParent->biChildren[
					    bi->biNumberInParent+ 1];
		}

	    if  ( ! nextBi				||
		  ! nextBi->biParaBoxBorder.bpIsSet	)
		{
		below += bi->biParaLeadingTwips;
		below += bi->biParaBoxBorder.bpSpacingTwips;
		/*  Word seems not to allocate this
		below += bi->biParaBoxBorder.bpWidthTwips;
		*/
		bp= &(bi->biParaBoxBorder);
		}
	    }
	}

    if  ( bi->biParaInTable					&&
	  bi->biNumberInParent == bi->biParent->biChildCount- 1	)
	{
	const BufferItem *	rowBi= bi->biParent->biParent;
	const CellProperties *	cp;
	int			col= bi->biParent->biNumberInParent;

	/*  Nicer but Word does not do this.
	below += bi->biParaLeadingTwips/ 2;
	*/

	if  ( col >= rowBi->biRowCellCount )
	    { LLDEB(col,rowBi->biRowCellCount);	}
	else{
	    cp= &(rowBi->biRowCells[col]);

	    if  ( cp->cpBottomBorder.bpIsSet )
		{
		below += cp->cpBottomBorder.bpSpacingTwips;
		/*  Word seems not to allocate this
		below += cp->cpBottomBorder.bpWidthTwips;
		*/
		}
	    }
	}

    below += bi->biParaSpaceAfterTwips;

    bi->biParaSpaceBelowLinesTwips= below;
    bi->biParaBorderBelowParagraph= bp;

    return;
    }

/************************************************************************/
/*									*/
/*  Find the position of the next tab.					*/
/*									*/
/************************************************************************/

static int docPsTabStop(	const BufferItem *		bi,
				int				x0Geometry,
				int				x0TextLines,
				int				tabInterval,
				int *				pTab,
				int *				pX,
				int				xPosition )
    {
    int				tab= -1;
    const ParagraphProperties *	pp= &(bi->biParaProperties);

    if  ( pp->ppTabCount == 0 )
	{
	if  ( xPosition < x0TextLines )
	    { *pX= x0TextLines; *pTab= tab; return 0; }

	if  ( tabInterval == 0 )
	    {
	    LLDEB(pp->ppTabCount,tabInterval);
	    *pX= xPosition; *pTab= tab; return 0;
	    }
	else{
	    *pTab= tab;

	    xPosition -= x0Geometry;
	    *pX= ( xPosition+ 1 )/ tabInterval;
	    *pX *= tabInterval;
	    *pX += tabInterval;
	    *pX += x0Geometry;
	    return 0;
	    }
	}
    else{
	TabStop *	ts= pp->ppTabStops;

	for ( tab= 0; tab < pp->ppTabCount; ts++, tab++ )
	    {
	    if  ( ts->tsTwips > xPosition- x0Geometry )
		{ break;	}
	    }

	if  ( tab >= pp->ppTabCount )
	    {
	    if  ( tabInterval > 0 )
		{
		xPosition -= x0Geometry;
		*pX= ( xPosition+ 1 )/ tabInterval;
		*pX *= tabInterval;
		*pX += tabInterval;
		*pX += x0Geometry;
		return 0;
		}
	    else{ LDEB(tabInterval); return 1;	}
	    }

	*pTab= tab;
	*pX= ts->tsTwips+ x0Geometry;
	return 0;
	}
    }

/************************************************************************/
/*									*/
/*  Calculate the layout of the next stretch of text upto the point	*/
/*  where it can be folded.						*/
/*									*/
/************************************************************************/

static int docPsLayoutWord(	const BufferItem *		bi,
				int *				pX1,
				int *				pVisibleX1,
				int *				pWordAscent,
				int *				pWordDescent,
				const DocumentFontList *	dfl,
				AppPhysicalFontList *		apfl,
				const TextParticule *		tp,
				ParticuleData *			pd,
				int				particuleCount,
				int				x0 )
    {
    int				particuleAscent;
    int				particuleDescent;
    int				fontAscent;
    int				fontDescent;
    int				wordAscent= 0;
    int				wordDescent= 0;

    int				accepted;
    int				width;
    int				decWidth;

    int				len= 0;
    int				xFrom= x0;
    const unsigned char *	from= bi->biParaString+ tp->tpStroff;

    int				x1= x0;
    int				visibleX1= x0;

    int				physicalFont= tp->tpPhysicalFont;

    accepted= 0;
    while( accepted < particuleCount && tp->tpKind == DOCkindTEXT )
	{
	AfmFontInfo *	afi;
	TextAttribute	ta;

	ta= tp->tpTextAttribute;
	afi= docPsPrintGetAfi( apfl, tp->tpPhysicalFont );
	if  ( ! afi )
	    { XDEB(afi); return -1;	}

	while( accepted < particuleCount		&&
	       tp->tpPhysicalFont == physicalFont	&&
	       tp->tpKind == DOCkindTEXT		)
	    {
	    AfmBBox		abb;
	    const int		withKerning= 0;
	    int			size;

	    int			dec;
	    int			d;

	    size= 10* tp->tpTextAttribute.taFontSizeHalfPoints;

	    if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUPERSCRIPT ||
		  tp->tpTextAttribute.taSuperSub == DOCfontSUBSCRIPT   )
		{ size= ( 6* size )/ 10; }

	    len += tp->tpStrlen;

	    width= psCalculateStringExtents( &abb, from, len,
						    size, withKerning, afi );

	    dec= -1;
	    for ( d= 0; d < len; d++ )
		{
		if  ( from[d] == '.' || from[d] == ','	)
		    { dec= d;	}
		}

	    if  ( dec >= 0 )
		{
		AfmBBox		abbDec;

		decWidth= psCalculateStringExtents( &abbDec, from, dec,
						    size, withKerning, afi );
		}
	    else{ decWidth= width;	}

	    x1= xFrom+ width;
	    visibleX1= xFrom+ ( abb.abbRight- abb.abbLeft );

	    pd->pdAfi= afi;
	    pd->pdX0= x0;
	    pd->pdWidth= xFrom+ width- x0;
	    pd->pdDecWidth= xFrom+ decWidth- x0;
	    pd->pdVisibleWidth= xFrom+ ( abb.abbRight- abb.abbLeft )- x0;
	    pd->pdTabNumber= -1;
	    pd->pdTabKind= -1;
	    pd->pdTabPosition= 0;
	    pd->pdVisiblePixels= 0;
	    pd->pdWhiteUnits= 0;
	    pd->pdCorrectBy= 0;
	    x0= x1;

	    particuleAscent= abb.abbTop;
	    particuleDescent= abb.abbBottom;
	    fontAscent= ( size* afi->afiFontBBox.abbTop+ 500 ) / 1000;
	    fontDescent= fontAscent- size;

	    if  ( wordAscent < particuleAscent	&& 
		  wordAscent < fontAscent	)
		{ wordAscent=  fontAscent;	}

	    if  ( wordDescent > particuleDescent	&&
		  wordDescent > fontDescent		)
		{ wordDescent=  fontDescent; }

	    accepted++; tp++; pd++;

	    if  ( ( len > 0 && from[len- 1] == ' ' )	||
		  accepted >= particuleCount		)
		{ break;	}

	    if  ( ! tp->tpTextAttribute.taFontIsSlanted	&&
		  ta.taFontIsSlanted			)
		{
		int		extra;

		if  ( ta.taFontSizeHalfPoints <
				    tp->tpTextAttribute.taFontSizeHalfPoints )
		    { extra= ta.taFontSizeHalfPoints;			}
		else{ extra= tp->tpTextAttribute.taFontSizeHalfPoints;	}

		extra = (int)( -afi->afiTanItalicAngle* 10* extra );

		x0 += extra;
		}
	    }

	if  ( ( len > 0 && from[len- 1] == ' ' )	||
	      accepted >= particuleCount		)
	    { break;	}

	ta= tp->tpTextAttribute;
	physicalFont= tp->tpPhysicalFont;
	from= bi->biParaString+ tp->tpStroff;
	xFrom= x0;
	len= 0;
	}

    if  ( accepted < particuleCount			&&
	  ! tp->tpTextAttribute.taFontIsSlanted		&&
	  tp[-1].tpTextAttribute.taFontIsSlanted	)
	{
	int		extra;
	AfmFontInfo *	afi;

	if  ( tp[-1].tpTextAttribute.taFontSizeHalfPoints	<
	      tp->tpTextAttribute.taFontSizeHalfPoints	)
	    { extra= tp[-1].tpTextAttribute.taFontSizeHalfPoints; }
	else{ extra= tp->tpTextAttribute.taFontSizeHalfPoints; }

	afi= docPsPrintGetAfi( apfl, tp[-1].tpPhysicalFont );
	if  ( ! afi )
	    { XDEB(afi); return -1;	}

	extra= (int) ( -afi->afiTanItalicAngle* 10* extra );

	x1 += extra;
	}

    *pWordAscent= wordAscent;
    *pWordDescent= wordDescent;
    *pX1= x1;
    *pVisibleX1= visibleX1;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Place as many particules on a line as possible, but stop on a tab.	*/
/*									*/
/************************************************************************/

#   define	PSfoundNOTHING		0
#   define	PSfoundTAB		1
#   define	PSfoundLINEFULL		2
#   define	PSfoundLINEBREAK	3
#   define	PSfoundPAGEBREAK	4

static int docPsLayoutText(	const BufferItem *		bi,
				int *				pFound,
				int *				pWordCount,
				int *				pX1,
				int *				pTextAscent,
				int *				pTextDescent,
				int				x1TextLines,
				const DocumentFontList *	dfl,
				AppPhysicalFontList *		apfl,
				const TextParticule *		tp,
				ParticuleData *			pd,
				int				particuleCount,
				int				acceptAtLeast,
				int				x0 )
    {
    int				textAscent= 0;
    int				textDescent= 0;

    int				accepted;
    int				found= PSfoundNOTHING;
    int				wordCount= 0;

    accepted= 0;
    while( accepted < particuleCount && found == PSfoundNOTHING )
	{
	int		done;

	switch( tp->tpKind )
	    {
	    int			x1;
	    int			visibleX1;

	    int			wordAscent;
	    int			wordDescent;

	    case DOCkindTEXT:
	    textCase:
		done= docPsLayoutWord( bi, &x1, &visibleX1,
						&wordAscent, &wordDescent,
						dfl, apfl, tp, pd,
						particuleCount- accepted, x0 );
		if  ( done < 1 )
		    { LDEB(done); return -1;	}
			    
		if  ( visibleX1 >= x1TextLines		&&
		      accepted >= acceptAtLeast		)
		    {
		    *pFound= PSfoundLINEFULL;
		    *pWordCount= wordCount;
		    *pX1= x0;
		    *pTextAscent= textAscent;
		    *pTextDescent= textDescent;
		    return accepted;
		    }

		if  ( textAscent < wordAscent )
		    { textAscent=  wordAscent;		}
		if  ( textDescent > wordDescent )
		    { textDescent=  wordDescent;	}

		wordCount++; accepted += done; tp += done; pd += done;
		x0= x1;
		break;

	    case DOCkindFIELDSTART:
		done= 0;
		while( accepted < particuleCount	&&
		       tp->tpKind == DOCkindFIELDSTART	)
		    {
		    pd->pdAfi= (AfmFontInfo *)0;
		    pd->pdX0= x0;
		    pd->pdWidth= 0;
		    pd->pdDecWidth= 0;
		    pd->pdVisibleWidth= 0;
		    pd->pdTabNumber= -1;
		    pd->pdTabKind= -1;
		    pd->pdTabPosition= 0;
		    pd->pdVisiblePixels= 0;
		    pd->pdWhiteUnits= 0;
		    pd->pdCorrectBy= 0;

		    if  ( accepted < acceptAtLeast )
			{ acceptAtLeast++;	}

		    accepted++; tp++; pd++;
		    }

		if  ( accepted < particuleCount		&&
		      tp->tpKind == DOCkindTEXT		)
		    { goto textCase;	}

		break;

	    case DOCkindTAB:
		found= PSfoundTAB;
		break;

	    case DOCkindFIELDEND:
	    case DOCkindXE:
	    case DOCkindTC:
		pd->pdAfi= (AfmFontInfo *)0;
		pd->pdX0= x0;
		pd->pdWidth= 0;
		pd->pdDecWidth= 0;
		pd->pdVisibleWidth= 0;
		pd->pdTabNumber= -1;
		pd->pdTabKind= -1;
		pd->pdTabPosition= 0;
		pd->pdVisiblePixels= 0;
		pd->pdWhiteUnits= 0;
		pd->pdCorrectBy= 0;

		accepted++; tp++; pd++; break;

	    case DOCkindOBJECT:
		{
		InsertedObject *	io;
		int			width;

		io= bi->biParaObjects+ tp->tpObjectNumber;

		width= ( io->ioScaleX* io->ioTwipsWide )/ 100;

		if  ( x0+ width >= x1TextLines		&&
		      accepted >= acceptAtLeast		)
		    {
		    *pFound= PSfoundLINEFULL;
		    *pWordCount= wordCount;
		    *pX1= x0;
		    *pTextAscent= textAscent;
		    *pTextDescent= textDescent;
		    return accepted;
		    }

		wordAscent= ( io->ioScaleY* io->ioTwipsHigh )/ 100;
		if  ( textAscent < wordAscent )
		    { textAscent=  wordAscent;	}

		pd->pdAfi= (AfmFontInfo *)0;
		pd->pdX0= x0;
		pd->pdWidth= width;
		pd->pdDecWidth= width;
		pd->pdVisibleWidth= width;
		pd->pdTabNumber= -1;
		pd->pdTabKind= -1;
		pd->pdTabPosition= 0;
		pd->pdVisiblePixels= 0;
		pd->pdWhiteUnits= 0;
		pd->pdCorrectBy= 0;
		x0 += width;

		accepted++; tp++; pd++; break;
		}

	    case DOCkindLINEBREAK:
	    case DOCkindPAGEBREAK:
		pd->pdAfi= (AfmFontInfo *)0;
		pd->pdX0= x0;
		pd->pdWidth= 0;
		pd->pdDecWidth= 0;
		pd->pdVisibleWidth= 0;
		pd->pdTabNumber= -1;
		pd->pdTabKind= -1;
		pd->pdTabPosition= 0;
		pd->pdVisiblePixels= 0;
		pd->pdWhiteUnits= 0;
		pd->pdCorrectBy= 0;

		if  ( tp->tpKind == DOCkindPAGEBREAK	&&
		      ! bi->biParaInTable		)
		    { found= PSfoundPAGEBREAK;	}
		else{ found= PSfoundLINEBREAK;	}

		accepted++; tp++; pd++; break;

	    default:
		LDEB(tp->tpKind); return -1;
	    }
	}

    *pFound= found;
    *pWordCount= wordCount;
    *pX1= x0;
    *pTextAscent= textAscent;
    *pTextDescent= textDescent;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Place as many particules on a line as possible. The caller can	*/
/*  require that a certain number of particules are accepted even if	*/
/*  these do not fit on the line. This is to prevent loops. A formatter	*/
/*  typically wants to be sure that at least one particule is accepted,	*/
/*  such that it can know that it actually advances.			*/
/*									*/
/************************************************************************/

static int docPsLayoutParticules( const BufferItem *		bi,
				int *				pFound,
				int *				pWordCount,
				int *				pLineAscent,
				int *				pLineDescent,
				int				tabInterval,
				const FormattingFrame *		ff,
				const DocumentFontList *	dfl,
				AppPhysicalFontList *		apfl,
				const TextParticule *		tp,
				ParticuleData *			pd,
				int				particuleCount,
				int				acceptAtLeast,
				int				x0 )
    {
    int				lineAscent= 0;
    int				lineDescent= 0;

    int				accepted;
    int				wordCount= 0;

    int				tabKind= DOCtkLEFT;
    int				tabX0= 0;
    int				tabX1= 0;
    int				tabParticule= -1;

    const TextParticule *	tpp= tp;
    ParticuleData *		pdd= pd;

    accepted= 0;
    while( accepted < particuleCount )
	{
	int		done;
	int		found;

	int		textAscent;
	int		textDescent;
	int		textWordCount;

	int		x1;
	AfmFontInfo *	afi;
	int		tab;

	int		visibleSinceTab;
	int		shift= 0;

	if  ( accepted >= acceptAtLeast )
	    { acceptAtLeast= 0;	}

	done= docPsLayoutText( bi, &found, &textWordCount, &x1,
				    &textAscent, &textDescent,
				    ff->ffX1TextLines, dfl, apfl, tpp, pdd,
				    particuleCount- accepted,
				    acceptAtLeast, x0 );
	if  ( done < 0 )
	    { LDEB(done); return -1;	}

	if  ( lineAscent < textAscent )
	    { lineAscent=  textAscent;	}
	if  ( lineDescent > textDescent )
	    { lineDescent=  textDescent;	}

	switch( tabKind )
	    {
	    case DOCtkLEFT:
		shift= 0; break;

	    case DOCtkDECIMAL:
		if  ( done > 0 && tabParticule >= 0 )
		    {
		    int		d;

		    visibleSinceTab=
			pdd[done-1].pdX0+ pdd[done-1].pdVisibleWidth- tabX0;

		    for ( d= done+ accepted- 1; d >= accepted; d-- )
			{
			if  ( pd[d].pdDecWidth <  pd[d].pdWidth )
			    {
			    visibleSinceTab=
					pd[d].pdX0+ pd[d].pdDecWidth- tabX0;
			    break;
			    }
			}

		    shift= tabX1- visibleSinceTab- tabX0;
		    if  ( shift < 0 )
			{ LDEB(shift); shift= 0;	}
		    else{ x1= tabX1;			}
		    pd[tabParticule].pdWidth= shift;
		    pd[tabParticule].pdVisibleWidth= 0;
		    }
		else{
		    if  ( tabParticule < 0 )
			{ LLDEB(done,tabParticule);	}
		    else{
			x1= tabX1;
			pd[tabParticule].pdWidth= tabX1- tabX0;
			pd[tabParticule].pdVisibleWidth= 0;
			}
		    }
		break;

	    case DOCtkRIGHT:
		if  ( done > 0 && tabParticule >= 0 )
		    {
		    visibleSinceTab=
			pdd[done-1].pdX0+ pdd[done-1].pdVisibleWidth- tabX0;

		    shift= tabX1- visibleSinceTab- tabX0;
		    if  ( shift < 0 )
			{ LDEB(shift); shift= 0;	}
		    else{ x1= tabX1;			}
		    pd[tabParticule].pdWidth= shift;
		    pd[tabParticule].pdVisibleWidth= 0;
		    }
		else{
		    if  ( tabParticule < 0 )
			{ LLDEB(done,tabParticule);	}
		    else{
			x1= tabX1;
			pd[tabParticule].pdWidth= tabX1- tabX0;
			pd[tabParticule].pdVisibleWidth= 0;
			}
		    }
		break;

	    case DOCtkCENTRE:
		if  ( done > 0 && tabParticule >= 0 )
		    {
		    visibleSinceTab=
			pdd[done-1].pdX0+ pdd[done-1].pdVisibleWidth- tabX0;

		    shift= tabX1- visibleSinceTab/ 2- tabX0;
		    if  ( shift < 0 )
			{ LDEB(shift); shift= 0;	}
		    else{ x1 += shift;			}
		    pd[tabParticule].pdWidth= shift;
		    pd[tabParticule].pdVisibleWidth= 0;
		    }
		else{
		    if  ( tabParticule >= 0 )
			{
			x1= tabX1;
			pd[tabParticule].pdWidth= tabX1- tabX0;
			pd[tabParticule].pdVisibleWidth= 0;
			}
		    }
		break;

	    default:
		LDEB(tabKind);
		shift= 0; break;
	    }

	if  ( shift > 0 )
	    {
	    int		i;

	    for ( i= 0; i < done; i++ )
		{ pdd[i].pdX0 += shift; }
	    }

	wordCount += textWordCount;
	accepted += done; tpp += done; pdd += done;
	x0= x1;

	switch( found )
	    {
	    case PSfoundTAB:
		break;

	    case PSfoundNOTHING:
	    case PSfoundLINEFULL:
		*pFound= found;
		*pWordCount= wordCount;
		*pLineAscent= lineAscent; *pLineDescent= lineDescent;
		return accepted;

	    case PSfoundLINEBREAK:
		*pFound= found;
		*pWordCount= wordCount;
		*pLineAscent= lineAscent; *pLineDescent= lineDescent;
		return accepted;

	    case PSfoundPAGEBREAK:
		*pFound= found;
		*pWordCount= wordCount;
		*pLineAscent= lineAscent; *pLineDescent= lineDescent;
		return accepted;

	    default:
		LDEB(found); return -1;
	    }

	afi= docPsPrintGetAfi( apfl, tpp->tpPhysicalFont );
	if  ( ! afi )
	    { XDEB(afi); return -1;	}

	tab= -1;
	if  ( docPsTabStop( bi, ff->ffX0Geometry, ff->ffX0TextLines,
						tabInterval, &tab, &x1, x0 ) )
	    {
	    if  ( accepted < 1 )
		{ LLDEB(particuleCount,accepted);	}

	    *pFound= PSfoundLINEFULL;
	    *pWordCount= wordCount;
	    *pLineAscent= lineAscent; *pLineDescent= lineDescent;
	    return accepted;
	    }

	if  ( tab >= 0 )
	    { tabKind= bi->biParaTabStops[tab].tsKind;	}
	else{ tabKind= DOCtkLEFT;			}
	tabX0= x0;
	tabX1= x1;
	tabParticule= accepted;

	pdd->pdAfi= afi;
	pdd->pdX0= x0;
	pdd->pdWidth= x1- x0;
	pdd->pdDecWidth= x1- x0;
	pdd->pdVisibleWidth= 0;
	pdd->pdTabNumber= tab;
	pdd->pdTabKind= tabKind;
	pdd->pdTabPosition= x1;
	pdd->pdVisiblePixels= 0;
	pdd->pdWhiteUnits= 0;
	pdd->pdCorrectBy= 0;

	if  ( tabKind == DOCtkLEFT )
	    { x0= x1;	}

	accepted++; tpp++; pdd++;

	}

    if  ( accepted < 1 )
	{ LLDEB(particuleCount,accepted);	}

    *pFound= PSfoundNOTHING;
    *pWordCount= wordCount;
    *pLineAscent= lineAscent; *pLineDescent= lineDescent;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Justify the particules in a line of text.				*/
/*									*/
/*  1)  Start justification after the last tab of the line. Justifying	*/
/*	after anything else than a left tab is ridiculous. Simply	*/
/*	refuse to.							*/
/*  2)  Ignore organisational particules such as the delimitation of	*/
/*	links and bookmarks at the end of the line.			*/
/*									*/
/************************************************************************/

static void docPsJustifyLine(	const BufferItem *	bi,
				const TextParticule *	tp,
				ParticuleData *		pd,
				int			accepted,
				int			x1TestLines )
    {
    int				i;
    int				extra;
    int				totalWeight;
    int				step;
    int				left;

    ParticuleData *		pdd;
    const TextParticule *	tpp;

    /*  1  */
    left= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( tpp->tpKind == DOCkindTAB )
	    { left= i+ 1;	}
	}

    if  ( left > 0 )
	{
	if  ( pd[left-1].pdTabKind != DOCtkLEFT )
	    { LDEB(pd[left-1].pdTabKind); return;	}

	tp += left; pd += left; accepted -= left;
	}

    /*  2  */
    while( accepted > 0 && tp[accepted- 1].tpStrlen == 0 )
	{ accepted--;	}

    if  ( accepted < 2 )
	{ /* LDEB(accepted); */ return;	}

    extra= x1TestLines- pd[accepted- 1].pdX0- pd[accepted- 1].pdVisibleWidth;

    if  ( extra < 0 )
	{
	LLDEB(extra,x1TestLines);
	tpp= tp; pdd= pd;
	for ( i= 0; i < accepted; tpp++, pdd++, i++ )
	    {
	    appDebug( "%-4s: %5d..%5d \"%*.*s\"\n",
		    docKindStr( tpp->tpKind ),
		    pdd->pdX0, pdd->pdX0+ pdd->pdVisibleWidth,
		    tpp->tpStrlen, tpp->tpStrlen,
		    bi->biParaString+ tpp->tpStroff );
	    }

	extra= 0;
	}

    totalWeight= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	pdd->pdWhiteUnits= 0;
	pdd->pdCorrectBy= 0;

	if  ( bi->biParaString[tpp->tpStroff+tpp->tpStrlen-1] == ' ' )
	    {
	    pdd->pdWhiteUnits=
		sqrt( (double)pdd[0].pdWidth+ (double)pdd[1].pdVisibleWidth );

	    totalWeight += pdd->pdWhiteUnits;
	    }
	}

    left= extra;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    step= ( extra* pdd->pdWhiteUnits )/ totalWeight;
	    if  ( step > left )
		{ step= left;	}

	    pdd->pdCorrectBy += step;

	    left -= step;
	    }
	}

    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    step= 1;
	    if  ( step > left )
		{ step= left;	}

	    pdd->pdCorrectBy += step;

	    left -= step;
	    }
	}

    step= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	pdd->pdX0 += step;
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    pdd->pdWidth += pdd->pdCorrectBy;
	    step += pdd->pdCorrectBy;
	    }
	}

    pdd->pdX0 += step;

    return;
    }

/************************************************************************/
/*									*/
/*  Place as many particules on a line as possible.			*/
/*									*/
/*  1)  Accept at least one particule.					*/
/*									*/
/*  A)  It looks nicer when fields do not start at the end of the line.	*/
/*	For the formatting it self this is irrelevant, but when a field	*/
/*	opens on the line before its contents, the shading of the	*/
/*	selection looks silly.						*/
/*									*/
/************************************************************************/

int docPsLineBox(	TextLine *			tl,
			const BufferItem *		bi,
			int				part,
			int				tabInterval,
			const DocumentFontList *	dfl,
			AppPhysicalFontList *		apfl,
			const TextParticule *		tp,
			ParticuleData *			pd,
			const FormattingFrame *		ff )
    {
    int			lineAscent;
    int			lineDescent;
    int			ascent;
    int			descent;
    int			xShift;

    int			accepted;
    int			found;
    int			wordCount;

    int			x0;
    int			visibleX1Twips;

    unsigned char *	from;

    if  ( part == 0 )
	{ x0= ff->ffX0FirstLine;	}
    else{ x0= ff->ffX0TextLines;	}

    from= bi->biParaString+ tp->tpStroff;

    /*  1  */
    accepted= docPsLayoutParticules( bi, &found, &wordCount,
				    &lineAscent, &lineDescent,
				    tabInterval, ff, dfl, apfl, tp, pd,
				    bi->biParaParticuleCount- part, 1, x0 );

    if  ( accepted <= 0 )
	{ LDEB(accepted); return -1;	}

    /*  A  */
    while( accepted > 1							&&
	   bi->biParaParticules[part+accepted-1].tpKind
					    == DOCkindFIELDSTART	)
	{ accepted--;	}

    tl->tlWordCount= wordCount;

    ascent= bi->biParaAscentTwips;
    descent= -bi->biParaDescentTwips;

    if  ( lineAscent > bi->biParaAscentTwips )
	{ ascent= lineAscent;	}
    if  ( lineDescent < bi->biParaDescentTwips )
	{ descent= -lineDescent; }

    tl->tlLineAscentTwips= ascent;
    tl->tlLineHeightTwips= ascent+ descent;
    tl->tlLineSpacingTwips= ascent+ descent+ bi->biParaLeadingTwips;

    if  ( bi->biParaLineSpacingTwips != 0			&&
	  ( part+ accepted < bi->biParaParticuleCount	||
	    bi->biParaLineSpacingIsMultiple		)	)
	{
	if  ( bi->biParaLineSpacingTwips > 0 )
	    {
	    if  ( bi->biParaLineSpacingTwips > tl->tlLineSpacingTwips )
		{ tl->tlLineSpacingTwips= bi->biParaLineSpacingTwips; }
	    }
	else{ tl->tlLineSpacingTwips= -bi->biParaLineSpacingTwips; }
	}

    if  ( accepted == 0 )
	{ visibleX1Twips= x0;	}
    else{
	ParticuleData *	pdd= pd+ accepted- 1;

	visibleX1Twips= pdd->pdX0+ pdd->pdVisibleWidth;
	}

    switch( bi->biParaAlignment )
	{
	case DOCiaLEFT:
	    xShift= 0; break;
	case DOCiaRIGHT:
	    xShift= ff->ffX1TextLines- visibleX1Twips;
	    break;
	case DOCiaCENTERED:
	    xShift= ( ff->ffX1TextLines- visibleX1Twips )/ 2;
	    break;
	case DOCiaJUSTIFIED:
	    xShift= 0;
	    if  ( found == PSfoundLINEFULL )
		{ docPsJustifyLine( bi, tp, pd, accepted, ff->ffX1TextLines ); }
	    break;
	default:
	    LDEB(bi->biParaAlignment); xShift= 0;
	    break;
	}

    if  ( xShift != 0 )
	{
	int		i;
	ParticuleData *	pdd= pd;

	for ( i= 0; i < accepted; pdd++, i++ )
	    { pdd->pdX0 += xShift;	}
	}

    if  ( found == PSfoundPAGEBREAK )
	{ tl->tlHasPageBreak= 1;	}
    else{ tl->tlHasPageBreak= 0;	}

    return accepted;
    }

int docPsClaimParticuleData(	const BufferItem *	bi,
				ParticuleData **	pParticuleData )
    {
    static ParticuleData *	PSPrintGeometry;
    static int			PSPrintGeometryCount;

    if  ( bi->biParaParticuleCount > PSPrintGeometryCount )
	{
	ParticuleData *	fresh;
	unsigned int	size;

	size= ( bi->biParaParticuleCount+ 1 )* sizeof( ParticuleData );

	fresh= (ParticuleData *)realloc( PSPrintGeometry,
			bi->biParaParticuleCount* sizeof( ParticuleData ) );
	if  ( ! fresh )
	    { LXDEB(bi->biParaParticuleCount,fresh); return -1;	}

	PSPrintGeometry= fresh;
	}

    *pParticuleData= PSPrintGeometry; return 0;
    }


/************************************************************************/
/*									*/
/*  Determine the 'majority' font of a paragraph, and get the font	*/
/*  extents for that font.						*/
/*									*/
/*  1)  Note that subscript/superscript is NOT taken into account.	*/
/*									*/
/************************************************************************/

int docPsParagraphLineExtents(	const AppPhysicalFontList *	apfl,
				BufferItem *			bi )
    {
    const TextParticule *	tp= bi->biParaParticules;

    int				size;
    int				paraAscent= 0;
    int				paraDescent= 0;
    int				i;

    static int *		counts;
    int *			fresh;

    int				found;
    int				foundCount;

    fresh= (int *)realloc( counts, apfl->apflCount* sizeof(int) );
    if  ( ! fresh )
	{ LXDEB(apfl->apflCount,fresh); return -1;	}
    counts= fresh;

    for ( i= 0; i < apfl->apflCount; i++ )
	{ counts[i]= 0;	}

    for ( i= 0; i < bi->biParaParticuleCount; tp++, i++ )
	{
	if  ( tp->tpKind != DOCkindTEXT		&&
	      tp->tpKind != DOCkindTAB		&&
	      tp->tpKind != DOCkindOBJECT	)
	    { continue;	}

	if  ( tp->tpPhysicalFont < 0			||
	      tp->tpPhysicalFont >= apfl->apflCount	)
	    { LLDEB(tp->tpPhysicalFont,apfl->apflCount); continue;	}

	counts[tp->tpPhysicalFont] += tp->tpStrlen+ 1;
	}

    found= -1;
    foundCount= 0;
    for ( i= 0; i < apfl->apflCount; i++ )
	{
	if  ( counts[i] > foundCount )
	    { found= i; foundCount= counts[i];	}
	}

    if  ( found >= 0 )
	{
	AfmFontInfo *	afi;

	afi= docPsPrintGetAfi( apfl, found );

	size= 10* apfl->apflFonts[found].apfAttribute.taFontSizeHalfPoints;

	paraAscent= ( size* afi->afiFontBBox.abbTop+ 500 ) / 1000;
	paraDescent= paraAscent- size;
	}
    else{ /* LDEB(found); */ size= 200;	}

    bi->biParaAscentTwips= paraAscent;
    bi->biParaDescentTwips= paraDescent;

    bi->biParaLeadingTwips= size/ LINEDISTFAC;

    docPsAboveParagraph( bi );
    docPsBelowParagraph( bi );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find the fonts of an image.						*/
/*									*/
/************************************************************************/

typedef int (*LIST_FONTS_PS)(	SimpleInputStream *	sis,
				const char *		afmDirectory,
				const char *		prefix,
				PostScriptFont **	pFontList,
				int *			pCount,
				int			xExt,
				int			yExt,
				int			twipsWide,
				int			twipsHigh );

int docPsListObjectFonts(	const InsertedObject *	io,
				const char *		afmDirectory,
				const char *		prefix,
				PostScriptFont **	pFontList,
				int *			pCount )
    {
    LIST_FONTS_PS	listFontsPs;

    SimpleInputStream *	sisMem;
    SimpleInputStream *	sisMeta;

    const ObjectData *	od;
    MemoryBuffer	mb;

    switch( io->ioKind )
	{
	case DOCokPICTWMETAFILE:
	    od= &(io->ioObjectData);
	    listFontsPs= appMetaListFontsPs;
	    break;

	case DOCokMACPICT:
	    od= &(io->ioObjectData);
	    listFontsPs= appMacPictListFontsPs;
	    break;

	case DOCokPICTJPEGBLIP:
	case DOCokPICTPNGBLIP:
	    return 0;

	case DOCokOLEOBJECT:
	    od= &(io->ioResultData);
	    listFontsPs= appMetaListFontsPs;
	    break;

	default:
	    LDEB(io->ioKind); return 0;
	}

    mb.mbCapacity= od->odSize;
    mb.mbBytes= od->odBytes;

    sisMem= sioInMemoryOpen( &mb );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisMeta= sioInHexOpen( sisMem );
    if  ( ! sisMeta )
	{ XDEB(sisMem); return -1;	}

    if  ( (*listFontsPs)( sisMeta, afmDirectory, prefix,
					pFontList, pCount,
					io->ioXExtent, io->ioYExtent,
					io->ioTwipsWide, io->ioTwipsHigh ) )
	{ SDEB(afmDirectory); return -1;	}

    sioInClose( sisMeta );
    sioInClose( sisMem );

    return 0;
    }


/************************************************************************/
/*									*/
/*  Recursively retrieve the list of PostScriptFonts that is used in	*/
/*  a document.								*/
/*									*/
/************************************************************************/

static int docPsPrintGetHeaderFooterFonts( const HeaderFooter *	hf,
				DocumentFontList *		dfl,
				const AppPhysicalFontList *	apfl,
				PostScriptFont **		pFontList,
				int *				pCount )
    {
    if  ( ! hf->hfItem )
	{ return 0;	}

    return docPsPrintGetItemFonts( hf->hfItem, dfl, apfl, pFontList, pCount );
    }

int docPsPrintGetItemFonts(	const BufferItem *		bi,
				DocumentFontList *		dfl,
				const AppPhysicalFontList *	apfl,
				PostScriptFont **		pFontList,
				int *				pCount )
    {
    int			part= 0;
    TextParticule *	tp;
    int			privateFont;
    AfmFontInfo *	afi;

    InsertedObject *	io;

    int			i;

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevROW:
	case DOClevCELL:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docPsPrintGetItemFonts( bi->biChildren[i],
					    dfl, apfl, pFontList, pCount ) )
		    { LDEB(1); return -1;	}
		}
	    break;

	case DOClevSECT:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docPsPrintGetItemFonts( bi->biChildren[i],
					dfl, apfl, pFontList, pCount ) )
		    { LDEB(1); return -1;	}
		}

	    if  ( docPsPrintGetHeaderFooterFonts( &(bi->biSectHeader),
					dfl, apfl, pFontList, pCount )	)
		{ LDEB(1); return -1;	}

	    if  ( docPsPrintGetHeaderFooterFonts( &(bi->biSectFirstPageHeader),
					dfl, apfl, pFontList, pCount )	)
		{ LDEB(1); return -1;	}

	    if  ( docPsPrintGetHeaderFooterFonts( &(bi->biSectLeftPageHeader),
					dfl, apfl, pFontList, pCount )	)
		{ LDEB(1); return -1;	}

	    if  ( docPsPrintGetHeaderFooterFonts( &(bi->biSectRightPageHeader),
					dfl, apfl, pFontList, pCount )	)
		{ LDEB(1); return -1;	}

	    if  ( docPsPrintGetHeaderFooterFonts( &(bi->biSectFooter),
					dfl, apfl, pFontList, pCount )	)
		{ LDEB(1); return -1;	}

	    if  ( docPsPrintGetHeaderFooterFonts( &(bi->biSectFirstPageFooter),
					dfl, apfl, pFontList, pCount )	)
		{ LDEB(1); return -1;	}

	    if  ( docPsPrintGetHeaderFooterFonts( &(bi->biSectLeftPageFooter),
					dfl, apfl, pFontList, pCount )	)
		{ LDEB(1); return -1;	}

	    if  ( docPsPrintGetHeaderFooterFonts( &(bi->biSectRightPageFooter),
					dfl, apfl, pFontList, pCount )	)
		{ LDEB(1); return -1;	}

	    break;

	case DOClevPARA:
	    tp= bi->biParaParticules; part= 0;

	    while( part < bi->biParaParticuleCount )
		{
		const int	appearsInText= 1;

		switch( tp->tpKind )
		    {
		    case DOCkindTEXT:
		    case DOCkindTAB:
			privateFont= tp->tpPhysicalFont;
			if  ( tp->tpPhysicalFont >= 0 )
			    {
			    afi= docPsPrintGetAfi( apfl, tp->tpPhysicalFont );
			    if  ( ! afi )
				{ XDEB(afi); return -1;	}

			    if  ( appPsRememberAfi( afi, tp->tpTextAttribute,
				    "f", appearsInText, pFontList, pCount ) )
				{ LDEB(1); return -1;	}
			    }

			while( part+ 1 < bi->biParaParticuleCount )
			    {
			    if  ( tp[1].tpKind != DOCkindTEXT	&&
				  tp[1].tpKind != DOCkindTAB	)
				{ break;	}
			    if  ( tp[1].tpPhysicalFont != privateFont	)
				{ break;	}

			    part++; tp++;
			    }
			part++; tp++; break;
		    case DOCkindOBJECT:
			io= bi->biParaObjects+ tp->tpObjectNumber;

			if  ( docPsListObjectFonts( io,
						apfl->apflAfmDirectory,
						"f", pFontList, pCount ) )
			    { LDEB(tp->tpKind); return -1;	}

			part++; tp++; break;
		    case DOCkindFIELDSTART:
		    case DOCkindFIELDEND:
		    case DOCkindXE:
		    case DOCkindTC:
		    case DOCkindLINEBREAK:
		    case DOCkindPAGEBREAK:
			part++; tp++; break;
		    default:
			LDEB(tp->tpKind); return -1;
		    }
		}
	    break;
	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Initialisation for paragraph layout.				*/
/*									*/
/************************************************************************/

static int docPsOpenParaFonts(	BufferItem *			bi,
				AppDrawingData *		add,
				const DocumentProperties *	dp )
    {
    int				part;
    TextParticule *		tp;

    const DocumentFontList *	dfl= &(dp->dpFontList);

    tp= bi->biParaParticules;
    for ( part= 0; part < bi->biParaParticuleCount; tp++, part++ )
	{
	if  ( tp->tpPhysicalFont < 0 )
	    {
	    tp->tpPhysicalFont= appOpenDocumentFont( add, dfl,
							tp->tpTextAttribute );

	    if  ( tp->tpPhysicalFont < 0 )
		{
		APP_DEB(appDebug( "\"%.*s\"\n",
			(int)tp->tpStrlen, bi->biParaString+ tp->tpStroff ));
		LLDEB(part,tp->tpPhysicalFont);
		return -1;
		}
	    }
	}

    return 0;
    }

static int docPsLayoutParaInit(	FormattingFrame *		ff,
				int				bottomTwips,
				int				stripHigh,
				BufferItem *			bi,
				AppDrawingData *		add,
				const DocumentProperties *	dp )
    {
    if  ( docPsOpenParaFonts( bi, add, dp ) )
	{ LDEB(1); return -1;	}

    docParagraphFrameTwips( ff, bottomTwips, stripHigh, bi );

    if  ( docPsParagraphLineExtents( &(add->addPhysicalFontList), bi ) )
	{ LDEB(1); return -1;	}

    bi->biParaLineCount= 0;

    return 0;
    }

static int docPsPlaceParaInit(	FormattingFrame *		ff,
				int				bottomTwips,
				int				stripHigh,
				BufferItem *			bi,
				AppDrawingData *		add,
				const DocumentProperties *	dp )
    {
    docParagraphFrameTwips( ff, bottomTwips, stripHigh, bi );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*									*/
/*  1)  Cope with the output of jade: Ignore enormous space before's	*/
/*	in footers.							*/
/*									*/
/************************************************************************/

static void docPsAboveLine(	LayoutPosition *		lp,
				int				fromLinePos,
				const BufferItem *		bi,
				int				inHeaderFooter,
				int				part,
				const DocumentGeometry *	dg,
				AppDrawingData *		add )
    {
    double		xfac= add->addMagnifiedPixelsPerTwip;

    int			spaceAboveLineTwips= 0;
    int			spaceAboveLinePixels= 0;

    /*  1  */
    if  ( part == 0 && ! fromLinePos && inHeaderFooter == DOCinBODY )
	{
	int		isFirst= 0;

	if  ( bi->biParaInTable )
	    {
	    if  ( bi->biNumberInParent == 0 )
		{ isFirst= 1;	}
	    }
	else{
	    if  ( lp->lpPageYTwips == dg->dgTopMarginTwips )
		{ isFirst= 1;	}
	    }

	spaceAboveLineTwips= bi->biParaSpaceAboveLinesTwips;

	if  ( ! isFirst )
	    { spaceAboveLineTwips += bi->biParaSpaceBeforeTwips;	}

	spaceAboveLinePixels= TWIPStoPIXELS( xfac, spaceAboveLineTwips );
	}

    lp->lpPageYTwips += spaceAboveLineTwips;
    lp->lpYPixels += spaceAboveLinePixels;

    return;
    }

static int docPsBelowLine(	const LayoutPosition *		lpTop,
				LayoutPosition *		lp,
				const TextLine *		tl,
				const BufferItem *		bi,
				int				inHeaderFooter,
				int				past,
				const FormattingFrame *		ffTwips,
				AppDrawingData *		add )
    {
    double		xfac= add->addMagnifiedPixelsPerTwip;

    int			spaceBelowLineTwips= 0;
    int			lineBottom;
    int			lineHeight;

    int			isLast= 0;

    if  ( past == bi->biParaParticuleCount )
	{
	if  ( bi->biParaInTable )
	    {
	    if  ( bi->biNumberInParent == bi->biParent->biChildCount- 1 )
		{ isLast= 1;	}
	    }

	if  ( ! isLast )
	    {
	    spaceBelowLineTwips= bi->biParaSpaceBelowLinesTwips;
	    spaceBelowLineTwips += bi->biParaSpaceAfterTwips;
	    }
	}

    lineBottom= lp->lpPageYTwips+ tl->tlLineHeightTwips+ spaceBelowLineTwips;
    lineHeight= lineBottom- lpTop->lpPageYTwips;

    if  ( inHeaderFooter == DOCinBODY				&&
	  lineBottom > ffTwips->ffY1				&&
	  lineHeight < ffTwips->ffPageHigh			&&
	  ( ffTwips->ffStripHigh < 0		||
	    lineHeight < ffTwips->ffStripHigh	)		)
	{ lp->lpPageYTwips= ffTwips->ffY1+ 1; return 1; }

    lp->lpPageYTwips += tl->tlLineSpacingTwips;
    lp->lpYPixels += TWIPStoPIXELS( xfac, tl->tlLineSpacingTwips );

    if  ( past == bi->biParaParticuleCount /* && ! isLast */ )
	{
	spaceBelowLineTwips= bi->biParaSpaceBelowLinesTwips;

	lp->lpPageYTwips += spaceBelowLineTwips;
	lp->lpYPixels += TWIPStoPIXELS( xfac, spaceBelowLineTwips );
	}

    return 0;
    }

static int docPsPlace_Line(	TextLine *			resTl,
				BufferItem *			bi,
				int				inHeaderFooter,
				int				part,
				TextParticule *			tp,
				ParticuleData *			pd,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				const FormattingFrame *		ffTwips,
				const FormattingFrame *		ffPixels,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom,
				ScreenLayout *			sl )
    {
    int				accepted;
    int				res;
    int				i;

    TextLine			workTl;
    TextLine *			tl= &(workTl);

    LayoutPosition		lp;
    const SectionProperties *	sp;
    const DocumentGeometry *	dg;

    const int			fromLinePos= 0;

    lp= *lpTop;
    sp= &(bi->biParent->biParent->biParent)->biSectProperties;
    dg= &(sp->spDocumentGeometry);

    workTl= *resTl;
    /*
    tl->tlStroff= tp->tpStroff;
    tl->tlFirstParticule= part;
    */

    /*  1  */
    docPsAboveLine( &lp, fromLinePos, bi, inHeaderFooter, part, dg, add );

    tl->tlTopPosition= lp;
    accepted= tl->tlParticuleCount;

    if  ( accepted < 1 )
	{ LDEB(accepted); docListItem(0,bi); return -1; }

    res= docPsBelowLine( lpTop, &lp, tl, bi, inHeaderFooter, part+ accepted,
							    ffTwips, add );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}
    if  ( res > 0 )
	{ *lpBottom= lp; return 0; }

    if  ( sl && sl->slLayoutLine					&&
	  (*sl->slLayoutLine)( tl, bi, part, accepted, add,
						ffPixels, tp, pd ) < 0	)
	{ LDEB(accepted); return -1;	}

    for ( i= 0; i < accepted; tp++, i++ )
	{
	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    DocumentField *	df;

	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

	    df->dfPage= lp.lpPage;
	    }
	}

    *lpBottom= lp;
    *resTl= *tl;

    return accepted;
    }

static int docPsPlaceLines(	DocumentRectangle *		drChanged,
				BufferItem *			bi,
				int				inHeaderFooter,
				int				part,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				const FormattingFrame *		ffTwips,
				const FormattingFrame *		ffPixels,
				int *				pLinesDone,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom,
				ScreenLayout *			sl )
    {
    int				done= 0;
    int				line= *pLinesDone;

    TextParticule *		tp= bi->biParaParticules+ part;
    TextLine *			tl= bi->biParaLines+ line;

    ParticuleData *		pd= (ParticuleData *)0;

    LayoutPosition		lp;

    lp= *lpTop;

    while( part < bi->biParaParticuleCount )
	{
	int		accepted;
	TextLine	boxLine;

	if  ( line >= bi->biParaLineCount )
	    { done= bi->biParaParticuleCount- part; break;	}

	boxLine= *tl;

	accepted= docPsPlace_Line( &boxLine,
				    bi, inHeaderFooter, part, tp, pd, add, bd,
				    ffTwips, ffPixels, &lp, &lp, sl );
	
	if  ( drChanged							&&
	      boxLine.tlTopPosition.lpYPixels !=
					tl->tlTopPosition.lpYPixels	)
	    {
	    double	xfac= add->addMagnifiedPixelsPerTwip;
	    int		oldY1Pixels;

	    if  ( drChanged->drY0 > boxLine.tlTopPosition.lpYPixels )
		{ drChanged->drY0=  boxLine.tlTopPosition.lpYPixels;	}
	    if  ( drChanged->drY0 > tl->tlTopPosition.lpYPixels )
		{ drChanged->drY0=  tl->tlTopPosition.lpYPixels;	}

	    if  ( drChanged->drY1 < lp.lpYPixels )
		{ drChanged->drY1=  lp.lpYPixels; }

	    oldY1Pixels= tl->tlTopPosition.lpYPixels+
				TWIPStoPIXELS( xfac, tl->tlLineSpacingTwips );

	    if  ( drChanged->drY1 < oldY1Pixels )
		{ drChanged->drY1=  oldY1Pixels; }
	    }

	if  ( accepted == 0 )
	    { break;	}

	*tl= boxLine;

	if  ( accepted < 0 )
	    { LLDEB(line,accepted); return -1; }

	part += accepted; tp += accepted; done += accepted; line++; tl++;

	if  ( boxLine.tlHasPageBreak )
	    { lp.lpPageYTwips= ffTwips->ffY1+ 1; break; }
	}

    *pLinesDone= line;
    *lpBottom= lp;

    return done;
    }

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*									*/
/*  1)  Cope with the output of sgmls: Ignore enormous space before's	*/
/*	in footers.							*/
/*									*/
/************************************************************************/

static int docPsLayout_Line(	TextLine *			resTl,
				int				fromLinePos,
				BufferItem *			bi,
				int				inHeaderFooter,
				int				part,
				TextParticule *			tp,
				ParticuleData *			pd,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				const FormattingFrame *		ffTwips,
				const FormattingFrame *		ffPixels,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom,
				ScreenLayout *			sl )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);
    int				accepted;
    int				res;
    int				i;

    TextLine			workTl;
    TextLine *			tl= &(workTl);

    LayoutPosition		lp;
    const SectionProperties *	sp;
    const DocumentGeometry *	dg;

    lp= *lpTop;
    sp= &(bi->biParent->biParent->biParent)->biSectProperties;
    dg= &(sp->spDocumentGeometry);

    docInitTextLine( tl );
    tl->tlStroff= tp->tpStroff;
    tl->tlFirstParticule= part;

    /*  1  */
    docPsAboveLine( &lp, fromLinePos, bi, inHeaderFooter, part, dg, add );

    tl->tlTopPosition= lp;
    accepted= docPsLineBox( tl, bi, part,
			    dp->dpTabIntervalTwips, &dp->dpFontList,
			    &(add->addPhysicalFontList), tp, pd, ffTwips );

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    res= docPsBelowLine( lpTop, &lp, tl, bi, inHeaderFooter, part+ accepted,
							    ffTwips, add );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}
    if  ( res > 0 )
	{ *lpBottom= lp; return 0; }

    tl->tlStroff= tp->tpStroff;
    tl->tlFirstParticule= part;
    tl->tlStrlen=
	    tp[accepted-1].tpStroff+ tp[accepted-1].tpStrlen- tp->tpStroff;
    tl->tlParticuleCount= accepted;

    if  ( sl && sl->slLayoutLine					&&
	  (*sl->slLayoutLine)( tl, bi, part, accepted, add,
						ffPixels, tp, pd ) < 0	)
	{ LDEB(accepted); return -1;	}

    for ( i= 0; i < accepted; tp++, i++ )
	{
	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    DocumentField *	df;

	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

	    df->dfPage= lp.lpPage;
	    }
	}

    *lpBottom= lp;
    *resTl= *tl;

    return accepted;
    }

static int docPsLayoutLines(	DocumentRectangle *		drChanged,
				BufferItem *			bi,
				int				inHeaderFooter,
				int				part,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				const FormattingFrame *		ffTwips,
				const FormattingFrame *		ffPixels,
				int *				pLinesDone,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom,
				ScreenLayout *			sl )
    {
    int				done= 0;
    int				line= *pLinesDone;

    TextParticule *		tp= bi->biParaParticules+ part;

    ParticuleData *		pd;

    LayoutPosition		lp;

    lp= *lpTop;

    if  ( docPsClaimParticuleData( bi, &pd ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    while( part < bi->biParaParticuleCount )
	{
	int		accepted;
	TextLine	boxLine;
	TextLine *	tl;
	const int	fromLinePos= 0;

	accepted= docPsLayout_Line( &boxLine, fromLinePos,
				    bi, inHeaderFooter, part, tp, pd, add, bd,
				    ffTwips, ffPixels, &lp, &lp, sl );

	if  ( accepted < 0 )
	    { LDEB(accepted); return -1;	}

	if  ( accepted == 0 )
	    { break;	}

	tl= docInsertTextLine( bi, -1 );
	if  ( ! tl )
	    { XDEB(tl); return -1;		}

	*tl= boxLine;

	part += accepted; tp += accepted; done += accepted; line++;

	if  ( boxLine.tlHasPageBreak )
	    { lp.lpPageYTwips= ffTwips->ffY1+ 1; break; }
	}

    *pLinesDone= line;
    *lpBottom= lp;

    return done;
    }

/************************************************************************/
/*									*/
/*  Layout as much of a table cell as fits on the current page.		*/
/*									*/
/************************************************************************/

static int docPsLayoutCellPageStrip( DocumentRectangle *	drChanged,
				int *				pAdvanced,
				BufferItem *			cellBi,
				int				inHeaderFooter,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				int				bottomTwips,
				int				stripHigh,
				START_PARAGRAPH			startParagraph,
				LAYOUT_LINES			layoutLines,
				ScreenLayout *			sl,
				const CellProperties *		cp,
				CellLayoutState *		cls )
				
    {
    const DocumentProperties *	dp= &(bd->bdProperties);
    int				advanced= 0;

    while( cls->clsPara < cellBi->biChildCount )
	{
	BufferItem *		paraBi= cellBi->biChildren[cls->clsPara];
	int			accepted;

	if  ( cls->clsPart == 0 )
	    {
	    if  ( (*startParagraph)( &(cls->clsFormattingFrameTwips),
				    bottomTwips, stripHigh, paraBi, add, dp ) )
		{ LDEB(1); return -1;	}

	    if  ( sl && sl->slStartParagraph				&&
		  (*sl->slStartParagraph)( &(cls->clsFormattingFramePixels),
						    paraBi, add, dp )	)
		{ LDEB(1); return -1;	}

	    paraBi->biTopPosition= cls->clsPosition;
	    }

	accepted= (*layoutLines)( drChanged,
		paraBi, inHeaderFooter, cls->clsPart, add, bd,
		&(cls->clsFormattingFrameTwips),
		&(cls->clsFormattingFramePixels),
		&cls->clsLinesDone,
		&(cls->clsPosition), &(cls->clsPosition), sl );

	if  ( accepted < 0 )
	    { LDEB(accepted); return -1;	}

	if  ( accepted > 0 )
	    { cls->clsAdvanced= advanced= 1;	}

	cls->clsPart += accepted;

	if  ( cls->clsPart < paraBi->biParaParticuleCount )
	    { break;	}

	paraBi->biBelowPosition= cls->clsPosition;

	cls->clsLinesDone= 0;
	cls->clsPart= 0;
	cls->clsPara++;
	}

    if  ( advanced )
	{ *pAdvanced= 1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Layout as much of a table row as fits on the current page.		*/
/*									*/
/************************************************************************/

static int docPsLayoutRowPageStrip( BufferItem *		rowBi,
				int *				pToNextPage,
				int *				pAtTop,
				int *				pRowHeight,
				CellLayoutState *		cls,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				START_PARAGRAPH			startParagraph,
				LAYOUT_LINES			layoutLines,
				ScreenLayout *			sl,
				LayoutJob *			lj )
    {
    int				col;

    int				toNextPage= 0;

    int				advanced;
    int				atTop= 1;

    LayoutPosition		lpBottom;
    int				bottomTwips= -1;
    int				stripHigh= -1;
    const CellProperties *	cp;

    lpBottom= lj->ljPosition;

    toNextPage= 0;

    if  ( rowBi->biRowHeightTwips < 0 )
	{
	bottomTwips= lj->ljPosition.lpPageYTwips-
				    rowBi->biRowHeightTwips- *pRowHeight;
	stripHigh= -rowBi->biRowHeightTwips;
	}

    advanced= 0;
    cp= rowBi->biRowCells;
    for ( col= 0; col < rowBi->biChildCount; cp++, cls++, col++ )
	{
	BufferItem *	cellBi= rowBi->biChildren[col];

	if  ( cp->cpMergedWithLeft )
	    { continue;	}

	cls->clsPosition= lj->ljPosition;

	if  ( docPsLayoutCellPageStrip( lj->ljChangedRectanglePixels,
			    &advanced, cellBi, lj->ljInHeaderFooter,
			    add, bd, bottomTwips, stripHigh, startParagraph,
			    layoutLines, sl, cp, cls ) )
	    { LDEB(1); return -1;	}

	if  ( cls->clsAdvanced )
	    { atTop= 0;	}

	if  ( lpBottom.lpPageYTwips < cls->clsPosition.lpPageYTwips )
	    { lpBottom.lpPageYTwips=  cls->clsPosition.lpPageYTwips; }

	if  ( lpBottom.lpYPixels < cls->clsPosition.lpYPixels )
	    { lpBottom.lpYPixels=  cls->clsPosition.lpYPixels;	}

	if  ( lj->ljInHeaderFooter == DOCinBODY				&&
	      cls->clsPara < cellBi->biChildCount		  	&&
	      ( bottomTwips < 0					||
	        cls->clsPosition.lpPageYTwips < bottomTwips	)	)
	    { toNextPage= 1;	}
	}

    *pRowHeight += lpBottom.lpPageYTwips- lj->ljPosition.lpPageYTwips;

    lj->ljPosition= lpBottom;

    *pToNextPage= toNextPage;
    *pAtTop= atTop;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Calculate the layout of a row in the document.			*/
/*									*/
/************************************************************************/

static void docPsInitRowLayout(	BufferItem *		rowBi,
				const LayoutPosition *	lp,
				CellLayoutState *	cls )
    {
    CellProperties *		cp;

    int				col;

    cp= rowBi->biRowCells;
    for ( col= 0; col < rowBi->biChildCount; cp++, cls++, col++ )
	{
	cls->clsAdvanced= 0;

	cls->clsPara= 0;
	cls->clsPart= 0;
	cls->clsLinesDone= 0;

	cls->clsPosition= *lp;

	rowBi->biChildren[col]->biTopPosition= *lp;
	}

    return;
    }

static int docPsLayoutRowItem(	BufferItem *			rowBi,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				LayoutJob *			lj,
				START_PARAGRAPH			startParagraph,
				LAYOUT_LINES			layoutLines,
				ScreenLayout *			sl )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);

    int				col;

    CellProperties *		cp;
    BufferItem *		cellBi;

    int				rowHeightTwips;
    int				toNextPage;
    int				atTop;

    static CellLayoutState *	staticCls;
    CellLayoutState *		cls;

    int				bottom= -1;

    double			xfac= add->addMagnifiedPixelsPerTwip;

    if  ( rowBi->biRowCellCount < rowBi->biChildCount )
	{
	LLDEB(rowBi->biRowCellCount,rowBi->biChildCount);
	docListItem( 0, rowBi );
	rowBi->biChildCount= rowBi->biRowCellCount;
	}

    cls= (CellLayoutState *)realloc( staticCls,
			    rowBi->biRowCellCount* sizeof(CellLayoutState) );
    if  ( ! cls )
	{ LXDEB(rowBi->biRowCellCount,cls); return -1;	}
    staticCls= cls;

    if  ( sl && sl->slStartRow			&&
	  (*sl->slStartRow)( rowBi, add, dp )	)
	{ LDEB(1); return -1;	}

    toNextPage= 0;
    rowHeightTwips= 0;
    if  ( rowBi->biRowHeightTwips < 0 )
	{ bottom= lj->ljPosition.lpPageYTwips- rowBi->biRowHeightTwips; }

    docPsInitRowLayout( rowBi, &(lj->ljPosition), cls );

    if  ( docPsLayoutRowPageStrip( rowBi, &toNextPage, &atTop,
			    &rowHeightTwips, cls,
			    add, bd, startParagraph, layoutLines, sl, lj ) )
	{ LDEB(1); return -1;	}

    if  ( toNextPage && rowBi->biRowKeepOnPage )
	{ atTop= 1;	}

    while( toNextPage )
	{
	docPsToNextPage( rowBi, add, &(lj->ljPosition) );

	if  ( atTop )
	    {
	    if  ( rowBi->biRowKeepOnPage )
		{ docPsInitRowLayout( rowBi, &(lj->ljPosition), cls ); }

	    rowHeightTwips= 0;
	    if  ( rowBi->biRowHeightTwips < 0 )
		{ bottom= lj->ljPosition.lpPageYTwips- rowBi->biRowHeightTwips;}

	    rowBi->biTopPosition= lj->ljPosition;
	    }

	if  ( docPsLayoutRowPageStrip( rowBi, &toNextPage, &atTop,
			    &rowHeightTwips, cls,
			    add, bd, startParagraph, layoutLines, sl, lj ) )
	    { LDEB(1); return -1;	}

	atTop= 0;
	}

    cp= rowBi->biRowCells;
    for ( col= 0; col < rowBi->biChildCount; cp++, col++ )
	{
	cellBi= rowBi->biChildren[col];

	if  ( cp->cpMergedWithLeft )
	    { continue;	}

	if  ( cp->cpBottomBorder.bpIsSet )
	    {
	    int		twips;
	    int		pixels;

	    twips= cp->cpBottomBorder.bpSpacingTwips;
	    /*  Word seems not to allocate this
	    twips += cp->cpBottomBorder.bpWidthTwips;
	    */
	    pixels= TWIPStoPIXELS( xfac, twips );

	    if  ( pixels == 0 )
		{ pixels= 1;	}

	    cls[col].clsPosition.lpPageYTwips += twips;
	    cls[col].clsPosition.lpYPixels += pixels;
	    }

	cellBi->biBelowPosition= cls[col].clsPosition;

	if  ( lj->ljPosition.lpPageYTwips < cls[col].clsPosition.lpPageYTwips )
	    { lj->ljPosition.lpPageYTwips=  cls[col].clsPosition.lpPageYTwips; }

	if  ( lj->ljPosition.lpYPixels < cls[col].clsPosition.lpYPixels )
	    { lj->ljPosition.lpYPixels=  cls[col].clsPosition.lpYPixels; }
	}

    if  ( rowHeightTwips < rowBi->biRowHeightTwips )
	{
	lj->ljPosition.lpPageYTwips +=
				rowBi->biRowHeightTwips- rowHeightTwips;

	lj->ljPosition.lpYPixels=
			lj->ljPosition.lpPage* add->addPageStepPixels+
			TWIPStoPIXELS( xfac, lj->ljPosition.lpPageYTwips );
	}

    if  ( rowHeightTwips < -rowBi->biRowHeightTwips )
	{
	lj->ljPosition.lpPageYTwips +=
				-rowBi->biRowHeightTwips- rowHeightTwips;

	lj->ljPosition.lpYPixels=
			lj->ljPosition.lpPage* add->addPageStepPixels+
			TWIPStoPIXELS( xfac, lj->ljPosition.lpPageYTwips );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Format the contents of a paragraph.					*/
/*									*/
/*  1)  As the layout process does not stop at the bottom of the page	*/
/*	in a header or footer, we will not continue on a new page here.	*/
/*									*/
/************************************************************************/

static int docPsLayoutParaContents( DocumentRectangle *		drChanged,
				BufferItem *			bi,
				int				inHeaderFooter,
				int				part,
				int *				pLinesDone,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				const FormattingFrame *		ffTwips,
				const FormattingFrame *		ffPixels,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom,
				START_PARAGRAPH			startParagraph,
				LAYOUT_LINES			layoutLines,
				ScreenLayout *			sl )
    {
    LayoutPosition		lp;
    int				line= *pLinesDone;
    int				done= 0;

    lp= *lpTop;

    while( part < bi->biParaParticuleCount )
	{
	int	accepted;

	accepted= (*layoutLines)( drChanged, bi, inHeaderFooter, part, add, bd,
				    ffTwips, ffPixels, &line, &lp, &lp, sl );

	if  ( accepted < 0 )
	    { LDEB(accepted); return -1;	}

	part += accepted;
	done += accepted;

	/*  1  */
	if  ( part < bi->biParaParticuleCount )
	    {
	    BufferItem *	rowBi= bi->biParent->biParent;

	    docPsToNextPage( rowBi, add, &lp );

	    if  ( part == 0 )
		{ bi->biTopPosition= lp; }
	    }
	}

    *lpBottom= lp;
    *pLinesDone= line;

    return done;
    }

static int docPsLayoutParaItem(	BufferItem *			bi,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				LayoutJob *			lj,
				START_PARAGRAPH			startParagraph,
				LAYOUT_LINES			layoutLines,
				ScreenLayout *			sl )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);

    FormattingFrame		ffTwips;
    FormattingFrame		ffPixels;

    int				accepted;
    int				part;
    int				line= 0;

    const int			bottomTwips= -1;
    const int			stripHigh= -1;

    /*  1  */
    if  ( lj->ljInHeaderFooter == DOCinBODY	&&
	  ! bi->biParaInTable			&&
	  bi->biParaStartsOnNewPage		)
	{
	BufferItem *	rowBi= bi->biParent->biParent;

	docPsToNextPage( rowBi, add, &(lj->ljPosition) );

	bi->biTopPosition= lj->ljPosition;
	}

    if  ( (*startParagraph)( &ffTwips, bottomTwips, stripHigh, bi, add, dp ) )
	{ LDEB(1); return -1;	}

    if  ( sl && sl->slStartParagraph				&&
	  (*sl->slStartParagraph)( &ffPixels, bi, add, dp )	)
	{ LDEB(1); return -1;	}

    part= 0;
    line= 0;

    accepted= docPsLayoutParaContents( lj->ljChangedRectanglePixels,
					bi, lj->ljInHeaderFooter, part, &line,
					add, bd, &ffTwips, &ffPixels,
					&(lj->ljPosition), &(lj->ljPosition), 
					startParagraph, layoutLines, sl );

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    if  ( bi->biParaLineCount < 1 )
	{ LDEB(bi->biParaLineCount); docListItem(0,bi); return -1; }

    return 0;
    }

/************************************************************************/
/*									*/
/*  Adjust the position of the top of a section, depending on the kind	*/
/*  of break								*/
/*									*/
/*  1)  Note that even and odd are swapped as from the users point of	*/
/*	view page numbers count from one.				*/
/*  2)  As this routine is only called for the body of the document,	*/
/*	page wraps can be performed unconditionally.			*/
/*									*/
/************************************************************************/

static void docPsPlaceSectTop(		BufferItem *			sectBi,
					AppDrawingData *		add,
					LayoutJob *			lj )
    {
    const SectionProperties *	sp= &(sectBi->biSectProperties);

    if  ( sectBi->biSectBreakKind != DOCsbkNONE	||
	  ( sectBi->biParent		&&
	    sectBi->biNumberInParent == 0	)	)
	{
	if  ( sectBi->biNumberInParent == 0 )
	    { docPsStartPageLayout( 0, add, sp, &(lj->ljPosition) );	}
	else{ docPsToNextPage( sectBi, add, &(lj->ljPosition) );	}

	/*  1  */
	if  ( ( sectBi->biSectBreakKind == DOCsbkEVEN	&&
		! ( lj->ljPosition.lpPage % 2 )		)	||
	      ( sectBi->biSectBreakKind == DOCsbkODD	&&
		( lj->ljPosition.lpPage % 2 )		)	)
	    { docPsToNextPage( sectBi, add, &(lj->ljPosition) );	}

	sectBi->biTopPosition= lj->ljPosition;

	if  ( sectBi->biNumberInParent == 0	&&
	      sectBi->biParent			)
	    { sectBi->biParent->biTopPosition= lj->ljPosition; }
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Adjust the bottom of an item to changes inside.			*/
/*									*/
/************************************************************************/

static void docPsLayoutAdjustBottom(	int *			pChanged,
					BufferItem *		bi,
					const LayoutPosition *	lp,
					DocumentRectangle *	drChanged )
    {
    int	changed= 0;

    if  ( bi->biBelowPosition.lpPageYTwips != lp->lpPageYTwips )
	{ bi->biBelowPosition.lpPageYTwips= lp->lpPageYTwips; changed= 1; }

    if  ( bi->biBelowPosition.lpPage != lp->lpPage )
	{ bi->biBelowPosition.lpPage= lp->lpPage; changed= 1; }

    if  ( bi->biBelowPosition.lpYPixels	<
	  lp->lpYPixels			)
	{
	if  ( drChanged					&&
	      drChanged->drY1 < lp->lpYPixels -1	)
	    { drChanged->drY1=  lp->lpYPixels -1;	}

	bi->biBelowPosition.lpYPixels= lp->lpYPixels; changed= 1;
	}

    if  ( bi->biBelowPosition.lpYPixels	> lp->lpYPixels )
	{
	if  ( drChanged							&&
	      drChanged->drY1 < bi->biBelowPosition.lpYPixels- 1	)
	    { drChanged->drY1=  bi->biBelowPosition.lpYPixels- 1;	}

	bi->biBelowPosition.lpYPixels= lp->lpYPixels; changed= 1;
	}

    if  ( changed )
	{ *pChanged= changed;	}

    return;
    }

/************************************************************************/
/*									*/
/*  Place successive items, after the predecessor.			*/
/*									*/
/************************************************************************/

static int docPsPlaceItems(	BufferItem *			biParent,
				int				from,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				LayoutJob *			lj )
    {
    int				i;
    int				changed;
    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;

    for ( i= from; i < biParent->biChildCount; i++ )
	{
	BufferItem *	child= biParent->biChildren[i];

	child->biTopPosition= lj->ljPosition;

	switch( child->biLevel )
	    {
	    case DOClevSECT:
		docPsPlaceSectTop( child, add, lj );

		if  ( docPsPlaceItems( child, 0, add, bd, lj ) )
		    { LDEB(1); return -1;	}
		break;

	    case DOClevDOC:
	    case DOClevCELL:
	    rowAsGroup:
		if  ( docPsPlaceItems( child, 0, add, bd, lj ) )
		    { LDEB(1); return -1;	}
		break;

	    case DOClevROW:
		if  ( ! child->biRowHasTableParagraphs )
		    { goto rowAsGroup;	}

		if  ( docPsLayoutRowItem( child, add, bd, lj,
						    docPsPlaceParaInit,
						    docPsPlaceLines,
						    &(lj->ljPlaceScreen) ) )
		    { LDEB(1); return -1;	}

		break;

	    case DOClevPARA:
		if  ( child->biParaLineCount == 0 )
		    { break;	}

		if  ( docPsLayoutParaItem( child, add, bd, lj,
						    docPsPlaceParaInit,
						    docPsPlaceLines,
						    &(lj->ljPlaceScreen) ) )
		    { LDEB(1); return -1;	}

		break;

	    default:
		LDEB(biParent->biLevel); return -1;
	    }

	docPsLayoutAdjustBottom( &changed, child,
						&(lj->ljPosition), drChanged );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Adjust the geometry of a parent item to changes in a child.		*/
/*									*/
/************************************************************************/

static int docPsAdjustParentGeometry(	BufferItem *		bi,
					AppDrawingData *	add,
					const BufferDocument *	bd,
					LayoutJob *		lj )
    {
    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;
    BufferItem *		biParent;
    int				from;

    biParent= bi->biParent;
    from= bi->biNumberInParent+ 1;

    while( biParent )
	{
	int		changed= 0;

	switch( biParent->biLevel )
	    {
	    case DOClevDOC:
	    case DOClevSECT:
	    case DOClevCELL:
	    parentRowAsGroup:
		if  ( bi->biNumberInParent == 0 )
		    {
		    LayoutPosition *	lpBi= &(bi->biTopPosition);
		    LayoutPosition *	lpPa= &(biParent->biTopPosition);

		    if  ( lpPa->lpPage != lpBi->lpPage			||
			  lpPa->lpPageYTwips != lpBi->lpPageYTwips	||
			  lpPa->lpYPixels != lpBi->lpYPixels		)
			{
			SDEB(docLevelStr(biParent->biLevel));
			SDEB(docLevelStr(bi->biLevel));
			LLDEB(lpPa->lpPage,lpBi->lpPage);
			LLDEB(lpPa->lpPageYTwips,lpBi->lpPageYTwips);
			LLDEB(lpPa->lpYPixels,lpBi->lpYPixels);

			biParent->biTopPosition= *lpBi;

			from= 0;
			}
		    }

		if  ( from <= biParent->biChildCount- 1 )
		    { docPsPlaceItems( biParent, from, add, bd, lj );	}
		else{ lj->ljPosition= bi->biBelowPosition;		}

		break;

	    case DOClevROW:
		if  ( ! biParent->biRowHasTableParagraphs )
		    { goto parentRowAsGroup;	}

		lj->ljPosition= biParent->biTopPosition;

		if  ( docPsLayoutRowItem( biParent, add, bd, lj,
						    docPsPlaceParaInit,
						    docPsPlaceLines,
						    &(lj->ljPlaceScreen) ) )
		    { LDEB(1); return -1;	}

		break;

	    default:
		LDEB(biParent->biLevel); return -1;
	    }

	docPsLayoutAdjustBottom( &changed, biParent,
						&(lj->ljPosition), drChanged );

	if  ( ! changed )
	    { break; }

	bi= biParent; biParent= bi->biParent;
	from= bi->biNumberInParent+ 1;
	}

    if  ( lj->ljInHeaderFooter == DOCinBODY && ! biParent )
	{
	int	extendRectangle;

	int	y1= lj->ljPosition.lpYPixels;

	y1= lj->ljPosition.lpPage* add->addPageStepPixels+
			    add->addPaperRect.drY1- add->addPaperRect.drY0;

	extendRectangle= add->addBackRect.drY1 != y1;

	if  ( drChanged					&&
	      extendRectangle				&&
	      drChanged->drY1 < add->addBackRect.drY1 )
	    { drChanged->drY1=  add->addBackRect.drY1; }

	add->addBackRect.drY1= y1;
	add->addDocRect.drY1= y1- add->addBottomMarginPixels;

	if  ( drChanged					&&
	      extendRectangle				&&
	      drChanged->drY1 < add->addBackRect.drY1 )
	    { drChanged->drY1=  add->addBackRect.drY1; }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Do the layout of a section.						*/
/*									*/
/************************************************************************/

static int docPsLayoutHeaderFooter(	int *				pY,
					HeaderFooter *			hf,
					int				yTwips,
					int				inHdFt,
					const SectionProperties *	sp,
					AppDrawingData *		add,
					const BufferDocument *		bd,
					LayoutJob *			lj )
    {
    double			xfac= add->addMagnifiedPixelsPerTwip;

    LayoutJob			headerLj;
    LayoutPosition		headerPosition;

    hf->hfPageUsedFor= -1;

    if  ( ! hf->hfItem )
	{ return 0;	}

    headerPosition.lpPage= 0;
    headerPosition.lpPageYTwips= yTwips;
    headerPosition.lpYPixels= TWIPStoPIXELS( xfac,
						headerPosition.lpPageYTwips );

    headerLj= *lj;
    headerLj.ljInHeaderFooter= inHdFt;
    headerLj.ljChangedRectanglePixels= (DocumentRectangle *)0;
    headerLj.ljPosition= headerPosition;

    if  ( docPsLayoutItemAndParents( hf->hfItem, add, bd, &headerLj, 0 ) )
	{ LDEB(1); return -1;	}

    if  ( headerPosition.lpPageYTwips > *pY )
	{ *pY= headerPosition.lpPageYTwips;	}

    return 0;
    }

static int docPsLayoutSectionItem(	BufferItem *		sectBi,
					AppDrawingData *	add,
					const BufferDocument *	bd,
					LayoutJob *		lj )
    {
    SectionProperties *		sp= &(sectBi->biSectProperties);
    DocumentGeometry *		dgSect= &(sp->spDocumentGeometry);

    int				belowHeadersTwips= dgSect->dgTopMarginTwips;
    int				belowFootersTwips= dgSect->dgPageHighTwips;

    int				i;

    if  ( docPsLayoutHeaderFooter( &belowHeadersTwips,
				&(sectBi->biSectHeader),
				dgSect->dgHeaderPositionTwips,
				DOCinSECT_HEADER, sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docPsLayoutHeaderFooter( &belowHeadersTwips,
				&(sectBi->biSectFirstPageHeader),
				dgSect->dgHeaderPositionTwips,
				DOCinFIRST_HEADER, sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docPsLayoutHeaderFooter( &belowHeadersTwips,
				&(sectBi->biSectLeftPageHeader),
				dgSect->dgHeaderPositionTwips,
				DOCinLEFT_HEADER, sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docPsLayoutHeaderFooter( &belowHeadersTwips,
				&(sectBi->biSectRightPageHeader),
				dgSect->dgHeaderPositionTwips,
				DOCinRIGHT_HEADER, sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( dgSect->dgTopMarginTwips < belowHeadersTwips )
	{ dgSect->dgTopMarginTwips= belowHeadersTwips+ 240;	}

    if  ( docPsLayoutHeaderFooter( &belowFootersTwips,
				&(sectBi->biSectFooter),
				dgSect->dgFooterPositionTwips,
				DOCinSECT_FOOTER, sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docPsLayoutHeaderFooter( &belowFootersTwips,
				&(sectBi->biSectFirstPageFooter),
				dgSect->dgFooterPositionTwips,
				DOCinFIRST_FOOTER, sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docPsLayoutHeaderFooter( &belowFootersTwips,
				&(sectBi->biSectLeftPageFooter),
				dgSect->dgFooterPositionTwips,
				DOCinLEFT_FOOTER, sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docPsLayoutHeaderFooter( &belowFootersTwips,
				&(sectBi->biSectRightPageFooter),
				dgSect->dgFooterPositionTwips,
				DOCinRIGHT_FOOTER, sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( belowFootersTwips > dgSect->dgPageHighTwips )
	{
	dgSect->dgFooterPositionTwips +=
			belowFootersTwips- dgSect->dgPageHighTwips;
	}

    if  ( ( sectBi->biSectFooter.hfItem			||
	    sectBi->biSectFirstPageFooter.hfItem	||
	    sectBi->biSectLeftPageFooter.hfItem		||
	    sectBi->biSectRightPageFooter.hfItem	)		&&
	  dgSect->dgBottomMarginTwips < dgSect->dgFooterPositionTwips	)
	{ dgSect->dgBottomMarginTwips= dgSect->dgFooterPositionTwips+ 240; }

    docPsPlaceSectTop( sectBi, add, lj );

    for ( i= 0; i < sectBi->biChildCount; i++ )
	{
	const int	andparents= 0;

	if  ( docPsLayoutItemAndParents( sectBi->biChildren[i],
						    add, bd, lj, andparents ) )
	    { LDEB(i); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Do the layout of a document item.					*/
/*									*/
/*  This is the main entry poin of the formatter.			*/
/*									*/
/************************************************************************/

int docPsLayoutItemAndParents(	BufferItem *		bi,
				AppDrawingData *	add,
				const BufferDocument *	bd,
				LayoutJob *		lj,
				int			andParents )
    {
    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;
    int				i;
    int				y1;

    y1= bi->biBelowPosition.lpYPixels- 1;

    i= bi->biTopPosition.lpYPixels;

    if  ( drChanged				&&
	  drChanged->drY0 > i	)
	{ drChanged->drY0=  i;	}

    bi->biTopPosition= lj->ljPosition;

    if  ( drChanged )
	{
	i= bi->biTopPosition.lpYPixels;

	if  ( drChanged->drY0 > i )
	    { drChanged->drY0=  i;			}
	if  ( drChanged->drY1 < y1 )
	    { drChanged->drY1=  y1;	}
	}

    switch( bi->biLevel )
	{
	case DOClevDOC:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docPsLayoutItemAndParents( bi->biChildren[i],
							    add, bd, lj, 0 ) )
		    { LDEB(1); return -1;	}
		}
	    break;

	case DOClevCELL:
	rowAsGroup:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		const int	andGrandparents= 0;

		if  ( docPsLayoutItemAndParents( bi->biChildren[i],
					    add, bd, lj, andGrandparents ) )
		    { LDEB(i); return -1;	}
		}
	    break;

	case DOClevSECT:
	    if  ( docPsLayoutSectionItem( bi, add, bd, lj ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{ goto rowAsGroup;	}

	    docFieldRefreshFlags( bi, bd );

	    if  ( docPsLayoutRowItem( bi, add, bd, lj,
					    docPsLayoutParaInit,
					    docPsLayoutLines,
					    &(lj->ljLayoutScreen) ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevPARA:
	    docFieldRefreshFlags( bi, bd );

	    if  ( docPsLayoutParaItem( bi, add, bd, lj,
					    docPsLayoutParaInit,
					    docPsLayoutLines,
					    &(lj->ljLayoutScreen) ) )
		{ LDEB(1); return -1;	}

	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    bi->biBelowPosition= lj->ljPosition;

    y1= bi->biBelowPosition.lpYPixels- 1;

    if  ( drChanged				&&
	  drChanged->drY1 < y1	)
	{ drChanged->drY1=  y1;	}

    if  ( andParents					&&
	  docPsAdjustParentGeometry( bi, add, bd, lj )	)
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Adjust the layout of a paragraph after a modification.		*/
/*									*/
/*  1)  Initialisation for paragraph formatting.			*/
/*  2)  Recalculate the layout of the first line that was changed.	*/
/*  3)  Did not fit on this page.. Try the next one.			*/
/*  4)  Layout subsequent lines of the paragraph, until the layout	*/
/*	process reaches the point outside the modified region where it	*/
/*	starts to cut lines at the same point where they were		*/
/*	originally cut.							*/
/*  5)  More lines than the paragraph originally contained are needed.	*/
/*	Just layout the rest.						*/
/*  6)  We are beyond the end of the modified region, and the new line	*/
/*	is to start where the old line starts. Just place the		*/
/*	subsequent lines from here.					*/
/*  7)  Recalculate the layout of a subsequent line that was changed.	*/
/*  8)  Remove superfluous lines from the administration.		*/
/*  9)  If the paragraph has changed height.. Adjust the redraw region.	*/
/*  10) Adjust the geometry of the rest of the document to the changes.	*/
/*									*/
/************************************************************************/

/*  3  */
static int docPsLayoutOneLine(	TextLine *			tl,
				BufferItem *			bi,
				int				inHeaderFooter,
				int				part,
				TextParticule *			tp,
				ParticuleData *			pd,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				const FormattingFrame *		ffTwips,
				const FormattingFrame *		ffPixels,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom,
				ScreenLayout *			sl,
				DocumentRectangle *		drChanged )
    {
    double			xfac= add->addMagnifiedPixelsPerTwip;

    const int			fromLinePos= 1;
    int				accepted;

    TextLine			boxLine;

    LayoutPosition		lp;

    lp= *lpTop;

    if  ( drChanged )
	{
	int	height= TWIPStoPIXELS( xfac, tl->tlLineSpacingTwips );

	if  ( drChanged->drY0 > lp.lpYPixels	)
	    { drChanged->drY0=  lp.lpYPixels;	}

	if  ( drChanged->drY1 < tl->tlTopPosition.lpYPixels+ height	)
	    { drChanged->drY1=  tl->tlTopPosition.lpYPixels+ height;	}
	}

    /*  2,7  */
    accepted= docPsLayout_Line( &boxLine, fromLinePos,
				    bi, inHeaderFooter, part, tp, pd, add, bd,
				    ffTwips, ffPixels, &lp, &lp, sl );

    if  ( accepted < 0 )
	{ LDEB(accepted); return -1;	}

    /*  3  */
    if  ( accepted == 0 )
	{
	if  ( inHeaderFooter != DOCinBODY )
	    { LDEB(inHeaderFooter);	}

	docPsToNextPage( bi, add, &lp );

	accepted= docPsLayout_Line( &boxLine, fromLinePos,
				    bi, inHeaderFooter, part, tp, pd, add, bd,
				    ffTwips, ffPixels, &lp, &lp, sl );

	if  ( accepted < 1 )
	    { LDEB(accepted); return -1;	}
	}

    if  ( drChanged )
	{
	if  ( drChanged->drY1 < lp.lpYPixels	)
	    { drChanged->drY1=  lp.lpYPixels;	}
	}

    *lpBottom= lp;
    *tl= boxLine;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Redo the layout of a paragraph after an editing action.		*/
/*									*/
/*  To avoid screen flashing, a minimalistic approach is used: As	*/
/*  little of the layout is redone, and an administration is kept of	*/
/*  what has been changed, in order to redraw a minimal screen		*/
/*  rectangle.								*/
/*									*/
/************************************************************************/

int docPsAdjustParaLayout(	BufferItem *		bi,
				int			line,
				int			stroffShift,
				int			upto,
				AppDrawingData *	add,
				const BufferDocument *	bd,
				LayoutJob *		lj )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);

    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;

    TextLine			boxLine;
    TextLine *			tl= bi->biParaLines+ line;

    int				part= tl->tlFirstParticule;
    TextParticule *		tp= bi->biParaParticules+ part;

    ParticuleData *		pd;

    LayoutPosition		lp;

    int				accepted;
    int				off;

    FormattingFrame		ffPixels;
    FormattingFrame		ffTwips;

    const int			bottomTwips= -1;
    const int			stripHigh= -1;

    int				heightChanged;

    if  ( docPsClaimParticuleData( bi, &pd ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    /*  1  */
    if  ( docPsOpenParaFonts( bi, add, dp ) )
	{ LDEB(1); return -1;	}

    docParagraphFrameTwips( &ffTwips, bottomTwips, stripHigh, bi );

    if  ( docPsParagraphLineExtents( &(add->addPhysicalFontList), bi ) )
	{ LDEB(1); return -1;	}

    if  ( lj->ljLayoutScreen.slStartParagraph				&&
	  (*lj->ljLayoutScreen.slStartParagraph)(
					&ffPixels, bi, add, dp )	)
	{ LDEB(1); return -1;	}

    lp= tl->tlTopPosition;
    lj->ljPosition= tl->tlTopPosition;

    if  ( drChanged )
	{
	if  ( drChanged->drX0 > add->addBackRect.drX0 )
	    { drChanged->drX0=  add->addBackRect.drX0;	}
	if  ( drChanged->drX1 < add->addBackRect.drX1 )
	    { drChanged->drX1=  add->addBackRect.drX1;	}

	if  ( drChanged->drY0 > tl->tlTopPosition.lpYPixels )
	    { drChanged->drY0=  tl->tlTopPosition.lpYPixels;	}
	}

    /*  2,3  */
    boxLine= *tl;
    accepted= docPsLayoutOneLine( &boxLine, bi, lj->ljInHeaderFooter, part,
					    tp, pd, add, bd,
					    &ffTwips, &ffPixels, &lp, &lp,
					    &(lj->ljLayoutScreen), drChanged );

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    *tl= boxLine;

    if  ( tl->tlHasPageBreak )
	{
	if  ( lj->ljInHeaderFooter != DOCinBODY )
	    { LDEB(lj->ljInHeaderFooter);	}

	docPsToNextPage( bi, add, &lp );
	}

    off= tl->tlStroff+ tl->tlStrlen;
    line++; tl++;
    part += accepted; tp += accepted; pd += accepted;

    /*  4  */
    while( part < bi->biParaParticuleCount )
	{
	/*  5  */
	if  ( line >= bi->biParaLineCount )
	    {
	    accepted= docPsLayoutParaContents( drChanged,
				bi, lj->ljInHeaderFooter,
				part, &line,
				add, bd, &ffTwips, &ffPixels, &lp, &lp, 
				docPsLayoutParaInit, docPsLayoutLines,
				&(lj->ljLayoutScreen) );

	    if  ( accepted < 1 )
		{ LDEB(accepted); return -1;	}

	    part += accepted; break;
	    }

	/*  6  */
	if  ( tl->tlStroff + stroffShift == off && off >= upto )
	    {
	    int		ll;
	    int		partShift= part- tl->tlFirstParticule;

	    for ( ll= line; ll < bi->biParaLineCount; ll++ )
		{
		bi->biParaLines[ll].tlFirstParticule += partShift;
		bi->biParaLines[ll].tlStroff += stroffShift;
		}

	    accepted= docPsLayoutParaContents( drChanged,
				bi, lj->ljInHeaderFooter, part, &line,
				add, bd, &ffTwips, &ffPixels, &lp, &lp, 
				docPsPlaceParaInit, docPsPlaceLines,
				&(lj->ljPlaceScreen) );

	    if  ( accepted < 1 )
		{ LDEB(accepted); return -1;	}

	    part += accepted; break;
	    }

	/*  7,3  */
	boxLine= *tl;
	accepted= docPsLayoutOneLine( &boxLine, bi, lj->ljInHeaderFooter,
					    part, tp, pd, add, bd,
					    &ffTwips, &ffPixels, &lp, &lp,
					    &(lj->ljLayoutScreen), drChanged );

	if  ( accepted < 1 )
	    { LDEB(accepted); return -1;	}

	*tl= boxLine;

	if  ( tl->tlHasPageBreak )
	    {
	    if  ( lj->ljInHeaderFooter != DOCinBODY )
		{ LDEB(lj->ljInHeaderFooter);	}

	    docPsToNextPage( bi, add, &lp );
	    }

	off= tl->tlStroff+ tl->tlStrlen;
	line++; tl++;
	part += accepted; tp += accepted; pd += accepted;
	}

    /*  8  */
    if  ( line < bi->biParaLineCount )
	{ docDeleteLines( bi, line, bi->biParaLineCount- line ); }

    if  ( part != bi->biParaParticuleCount )
	{ LLDEB(part,bi->biParaParticuleCount); return -1;	}
    if  ( line !=  bi->biParaLineCount )
	{ LLDEB(line,bi->biParaLineCount); return -1;	}

    /*  9  */
    heightChanged= 0;

    docPsLayoutAdjustBottom( &heightChanged, bi, &lp, drChanged );

    lj->ljPosition= lp;

    if  ( heightChanged )
	{
	/*  10  */
	if  ( docPsAdjustParentGeometry( bi, add, bd, lj ) )
	    { LDEB(1); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert the correct page number in a page header/footer.		*/
/*									*/
/************************************************************************/

int docLayoutHeaderFooter(	HeaderFooter *			hf,
				int				pageNumber,
				int				inHeaderFooter,
				BufferDocument *		bd,
				const BufferItem *		sectBi,
				AppDrawingData *		add,
				LAYOUT_EXTERNAL			layoutExternal,
				DocumentRectangle *		drChanged )
    {
    if  ( pageNumber != hf->hfPageUsedFor )
	{
	int		changed= 0;

	hf->hfItem->biSectHeaderFooterUseForSectBi= sectBi;
	hf->hfItem->biSectHeaderFooterUseForPage= pageNumber;

	if  ( docRecalculateTextFields( &changed, bd, hf->hfItem,
						FIELDdoDOC_FORMATTED	|
						FIELDdoDOC_COMPLETE	|
						FIELDdoPAGE_NUMBER	) )
	    { LDEB(pageNumber); return -1;	}

	if  ( layoutExternal 					&&
	      (*layoutExternal)( hf->hfItem, pageNumber, inHeaderFooter,
				    &(sectBi->biSectProperties), bd,
				    add, drChanged )	)
	    { LDEB(1); return -1;	}

	hf->hfPageUsedFor= pageNumber;
	}

    return 0;
    }

